﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nn/nn_Log.h>

#include <nnt/gfx/testGfx_GraphicsFramework.h>

namespace
{
    size_t g_GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nnt::gfx::GraphicsFramework g_GfxFramework;

    static const int        g_ExcuteFrame = 10000000 * 60;
    static const size_t     g_MemoryPoolSize =
        1 * 1024 * 1024 * 1024 +
        512 * 1024 * 1024;


    // メモリ 確保・破棄 関数
    void* PoolMemoryAllocateFunction(size_t size, size_t alignment)
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
        return aligned_alloc(alignment, size);
#else
        return _aligned_malloc(size, alignment);
#endif
    }

    void PoolMemoryFreeFunction(void* ptr)
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
        free(ptr);
#else
        _aligned_free(ptr);
#endif
    }
}

extern "C" void nnMain()
{
    // フレームワークを初期化
    {
        nnt::gfx::GraphicsFramework::InitializeGraphicsSystem(g_GraphicsSystemMemorySize);
        nnt::gfx::GraphicsFramework::FrameworkInfo info;
        info.SetDefault();
        g_GfxFramework.Initialize(info);
    }

    for (int i = 0; i < g_ExcuteFrame; i++)
    {
        //-----------------------------------------------------
        // 広大なメモリプールを作成する。
        //-----------------------------------------------------
        static nn::gfx::MemoryPool memoryPool;
        void* pAlignedMemory = NULL;
        {
            nn::gfx::MemoryPool::InfoType info;
            info.SetDefault();
            info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached);

            size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(g_GfxFramework.GetDevice(), info);
            pAlignedMemory = PoolMemoryAllocateFunction(g_MemoryPoolSize, alignment);

            info.SetPoolMemory(
                pAlignedMemory,
                nn::util::align_down(
                    g_MemoryPoolSize,
                    nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(g_GfxFramework.GetDevice(),info)
                )
            );
            memoryPool.Initialize(g_GfxFramework.GetDevice(), info);
        }


        // テクスチャ取得と vsync 待ち
        g_GfxFramework.AcquireTexture(0);
        g_GfxFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        g_GfxFramework.BeginFrame(0);
        nn::gfx::CommandBuffer* pCommandBuffer = g_GfxFramework.GetRootCommandBuffer(0);
        nn::gfx::ColorTargetView* pScanBuffer = g_GfxFramework.GetColorTargetView();

        pCommandBuffer->ClearColor(pScanBuffer, 1.f, .5f, .5f, 1.0f, nullptr);
        g_GfxFramework.EndFrame(0);
        g_GfxFramework.ExecuteCommand(0);
        g_GfxFramework.QueuePresentTexture(1);
        g_GfxFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));


        //-----------------------------------------------------
        // 広大なメモリプールを破棄する。
        //-----------------------------------------------------
        memoryPool.Finalize(g_GfxFramework.GetDevice());
        PoolMemoryFreeFunction(pAlignedMemory);
    }

    NN_LOG("FinalizeGraphicsFramework\n");
    g_GfxFramework.Finalize();
}
