﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/util/util_Base64.h>

#include "gfxUtilGpuBenchmark_JsonStreamer.h"
#include "gfxUtilGpuBenchmark_Property.h"

#include <rapidjson/document.h>
#include <rapidjson/writer.h>
#include <rapidjson/prettywriter.h>

namespace nnt { namespace gfx { namespace util {

namespace {

#define VersionKey "Version"
#define BenchmarkConfigurationKey "BenchmarkConfiguration"
#define TestIdKey "UniqueTestId"
#define WarmUpCountKey "WarmUpCount"
#define RepeatCountKey "RepeatCount"
#define NameKey "Name"
#define ValueKey "Value"
#define TypeKey "Type"
#define PropertylistKey "Propertylist"
#define TestCasesKey "TestCases"
#define ExpectedResultAverageKey "ExpectedResultAverage"
#define ExpectedResultStandardDeviationKey "ExpectedResultStandardDeviation"
#define OutputBufferHashKey "OutputBufferHash"


static const int MinSupportedVersion = 3;
static const int CurrentVersion = 3;


template <typename Writer>
void SerializeBenchmarkConfiguration(Writer& writer, const GpuBenchmark* pBenchmark)
{
    const GpuBenchmarkPropertyHolder** pDestinationArray = nullptr;
    int propertyCount = pBenchmark->GetPropertyCount();
    pDestinationArray = static_cast<const GpuBenchmarkPropertyHolder**>(alloca(propertyCount * sizeof(GpuBenchmarkPropertyHolder*)));
    NN_ASSERT(pDestinationArray != nullptr);
    pBenchmark->FillPropertyList(pDestinationArray, propertyCount);

    writer.StartObject();

    const char* benchmarkName = pBenchmark->GetName();
    writer.String(NameKey);
    writer.String(benchmarkName);


    writer.String(PropertylistKey);
    writer.StartArray();
    for (int i = 0; i < propertyCount; ++i)
    {
        writer.StartObject();

        const GpuBenchmarkPropertyHolder* pPropertyHolder = pDestinationArray[i];

        PropertyType type = pPropertyHolder->GetType();
        NN_ASSERT(type < PropertyType_Max);
        const char* typeName = GetPropertyTypeName(type);
        writer.String(TypeKey);
        writer.String(typeName);

        const char* propertyName = pPropertyHolder->GetName();
        writer.String(NameKey);
        writer.String(propertyName);


        writer.String(ValueKey);
        switch (type)
        {
        case PropertyType_IntegerRange:
            {
                writer.Int(pPropertyHolder->Get());
            }
            break;

        case PropertyType_Enumeration:
            {
                int value = pPropertyHolder->Get();
                const char* elementName =  pPropertyHolder->ToEnum()->GetElementNameAt(value);
                writer.String(elementName);
            }
            break;

        case PropertyType_Invalid:
        default:
            {
                NN_UNEXPECTED_DEFAULT;
            }
            break;
        }
        writer.EndObject();
    }

    writer.EndArray();

    writer.EndObject();
}

} // anonymous namespace

namespace json {

void* CrtAlloc(size_t size)
{
    return std::malloc(size);
}

void* CrtRealloc(void* originalPtr, size_t originalSize, size_t newSize)
{
    NN_UNUSED(originalSize);
    return std::realloc(originalPtr, newSize);
}

void CrtFree(void *ptr)
{
    std::free(ptr);
}


// Freeの関数は静的にしないとコンピレーションのエラーが発生します。
// だから、全部のコールバックを静的にしました。
// スレッドを使えば、同時実行の問題が行うかもしれません。
class CustomAllocator
{
public:
    static AllocatorCallbacks callbacks;

    static void* Malloc(size_t size)
    {
        if (size > 0)
        {
            return callbacks.pAllocFunc(size);
        }
        else
        {
            return nullptr; // standardize to returning NULL.
        }
    }

    static void* Realloc(void* originalPtr, size_t originalSize, size_t newSize)
    {
        if (newSize == 0) {
            callbacks.pFreeFunc(originalPtr);
            return nullptr;
        }
        return callbacks.pReallocFunc(originalPtr, originalSize, newSize);
    }

    static void Free(void *ptr)
    {
        callbacks.pFreeFunc(ptr);
    }
};

class NnFileOutputStream
{
    nn::Result          result;
    nn::fs::FileHandle  hFile;
    int64_t             fileOffset;

    char*               pBuffer;
    char*               pBufferEnd;
    char*               pCurrent;

public:

    void Open(const char* filepath, void* pCacheBuffer, size_t cacheBufferSize)
    {
        result = nn::fs::DeleteFile(filepath);
        result = nn::fs::CreateFile(filepath, 0);

        if (result.IsSuccess())
        {
            result = nn::fs::OpenFile(&hFile, filepath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        }

        fileOffset = 0;

        pBuffer = static_cast<char*>(pCacheBuffer);
        pBufferEnd = pBuffer + cacheBufferSize;
        pCurrent = pBuffer;
    }

    void Close()
    {
        result = nn::fs::FlushFile(hFile);
        nn::fs::CloseFile(hFile);
    }

    nn::Result GetResult()
    {
        return result;
    }

    void Put(char c)
    {
        if (pCurrent >= pBufferEnd)
            Flush();

        *pCurrent++ = c;
    }

    void PutN(char c, size_t n)
    {
        size_t available = static_cast<size_t>(pBufferEnd - pCurrent);

        while (n > available)
        {
            std::memset(pCurrent, c, available);
            pCurrent += available;
            Flush();
            n -= available;
            available = static_cast<size_t>(pBufferEnd - pCurrent);
        }

        if (n > 0)
        {
            std::memset(pCurrent, c, n);
            pCurrent += n;
        }
    }

    void Flush()
    {
        if (pCurrent != pBuffer)
        {
            size_t available = static_cast<size_t>(pCurrent - pBuffer);

            if (result.IsSuccess())
            {
                result = nn::fs::WriteFile(hFile, fileOffset, pBuffer, available, nn::fs::WriteOption::MakeValue(0));
                fileOffset += available;
            }

            pCurrent = pBuffer;
        }
    }

    // Not implemented
    char Peek() const
    {
        NN_ASSERT(false);
        return 0;
    }

    // Not implemented
    char Take() {
        NN_ASSERT(false);
        return 0;
    }

    // Not implemented
    size_t Tell() const
    {
        NN_ASSERT(false);
        return 0;
    }

    // Not implemented
    char* PutBegin()
    {
        NN_ASSERT(false);
        return 0;
    }

    // Not implemented
    size_t PutEnd(char*)
    {
        NN_ASSERT(false);
        return 0;
    }
};

AllocatorCallbacks CustomAllocator::callbacks = { nullptr, nullptr, nullptr };


typedef rapidjson::GenericDocument<rapidjson::UTF8<>, rapidjson::MemoryPoolAllocator<CustomAllocator>, CustomAllocator > JsonDocument;
typedef rapidjson::GenericValue<rapidjson::UTF8<>, rapidjson::MemoryPoolAllocator<CustomAllocator>> JsonValue;


bool MatchFilterPattern(const char* filterPattern, const JsonDocument::ConstValueIterator& iterator)
{
    if ((filterPattern == nullptr) || (strlen(filterPattern) == 0))
        return true;

    const JsonValue& testIdValue = (*iterator)[TestIdKey];
    const char* testCaseName = testIdValue.GetString();
    const char* result = strstr(testCaseName, filterPattern);
    return result != nullptr;
}

void SetBenchmarkConfiguration(JsonValue& jsonObject, JsonDocument::AllocatorType& jsonAllocator, const GpuBenchmark* pBenchmark)
{
    const GpuBenchmarkPropertyHolder** pDestinationArray = nullptr;
    int propertyCount = pBenchmark->GetPropertyCount();
    pDestinationArray = static_cast<const GpuBenchmarkPropertyHolder**>(alloca(propertyCount * sizeof(GpuBenchmarkPropertyHolder*)));
    NN_ASSERT(pDestinationArray != nullptr);
    pBenchmark->FillPropertyList(pDestinationArray, propertyCount);

    jsonObject.SetObject();

    JsonValue benchmarkNameValue(pBenchmark->GetName(), jsonAllocator);
    jsonObject.AddMember(NameKey, benchmarkNameValue, jsonAllocator);

    JsonValue propertylistValue;
    propertylistValue.SetArray();

    for (int i = 0; i < propertyCount; ++i)
    {
        const GpuBenchmarkPropertyHolder* pPropertyHolder = pDestinationArray[i];
        JsonValue jsonPropertyObject;
        jsonPropertyObject.SetObject();

        PropertyType type = pPropertyHolder->GetType();
        NN_ASSERT(type < PropertyType_Max);

        const char* typeName = GetPropertyTypeName(type);
        JsonValue typeNameValue(typeName, jsonAllocator);
        jsonPropertyObject.AddMember(TypeKey, typeNameValue, jsonAllocator);

        const char* propertyName = pPropertyHolder->GetName();
        JsonValue propertyNameValue(propertyName, jsonAllocator);
        jsonPropertyObject.AddMember(NameKey, propertyNameValue, jsonAllocator);

        switch (type)
        {
        case PropertyType_IntegerRange:
        {
            JsonValue propertyValue(pPropertyHolder->Get());
            jsonPropertyObject.AddMember(ValueKey, propertyValue, jsonAllocator);
        }
        break;

        case PropertyType_Enumeration:
        {
            int value = pPropertyHolder->Get();
            const char* elementName = pPropertyHolder->ToEnum()->GetElementNameAt(value);
            JsonValue propertyValue(elementName, jsonAllocator);
            jsonPropertyObject.AddMember(ValueKey, propertyValue, jsonAllocator);
        }
        break;

        case PropertyType_Invalid:
        default:
        {
            NN_UNEXPECTED_DEFAULT;
        }
        break;
        }

        propertylistValue.PushBack(jsonPropertyObject, jsonAllocator);
    }

    jsonObject.AddMember(PropertylistKey, propertylistValue, jsonAllocator);

}


void SetExpectedResults(JsonValue& jsonObject, JsonDocument::AllocatorType& jsonAllocator, uint64_t cpuTime, uint64_t gpuTime)
{
    jsonObject.SetArray();
    jsonObject.Reserve(2, jsonAllocator);
    jsonObject.PushBack(cpuTime, jsonAllocator);
    jsonObject.PushBack(gpuTime, jsonAllocator);
}


struct Document
{
    static const int        DefaultChunkCapacity = 64 * 1024;
    static const size_t     DefaultStackCapacity = 1024;

    AllocatorCallbacks allocatorCallbacks;
    CustomAllocator stackAllocator;
    JsonDocument::AllocatorType allocator;
    JsonDocument doc;

    Document()
    : allocatorCallbacks()
    , stackAllocator()
    , allocator(DefaultChunkCapacity, &stackAllocator)
    , doc(&allocator, DefaultStackCapacity, &stackAllocator)
    {
    }

    ~Document()
    {

    }
};

struct TestCaseIterator
{
    Document* pDocument;
    JsonDocument::ConstValueIterator currentIterator;
};

inline const JsonValue& GetIteratorValue(const TestCaseIterator* pIterator)
{
    NN_ASSERT(pIterator->currentIterator != pIterator->pDocument->doc[TestCasesKey].Begin());
    return *(pIterator->currentIterator - 1);
}

void Create(Document** ppDocument, AllocatorCallbacks* pAllocator)
{
    NN_ASSERT_NOT_NULL(ppDocument);

    AllocatorCallbacks allocatorCallbacks;
    if (pAllocator == nullptr)
    {
        allocatorCallbacks.pAllocFunc = &CrtAlloc;
        allocatorCallbacks.pReallocFunc = &CrtRealloc;
        allocatorCallbacks.pFreeFunc = &CrtFree;
    }
    else
    {
        allocatorCallbacks = *pAllocator;
    }

    CustomAllocator::callbacks = allocatorCallbacks;

    Document* pDocument = new (allocatorCallbacks.pAllocFunc(sizeof(Document))) Document();
    pDocument->allocatorCallbacks = allocatorCallbacks;

    JsonDocument &doc = pDocument->doc;


    doc.SetObject();

    doc.AddMember(VersionKey, CurrentVersion, doc.GetAllocator());

    JsonValue testCasesValue;
    testCasesValue.SetArray();
    doc.AddMember(TestCasesKey, testCasesValue, doc.GetAllocator());

    *ppDocument = pDocument;
}

void Finalize(Document* pDocument)
{
    NN_ASSERT_NOT_NULL(pDocument);

    CustomAllocator::callbacks = pDocument->allocatorCallbacks;

    pDocument->~Document();
    pDocument->allocatorCallbacks.pFreeFunc(pDocument);
}

nn::Result LoadFromFile(Document* pDocument, const char* filepath)
{
    NN_ASSERT_NOT_NULL(pDocument);
    NN_ASSERT_NOT_NULL(filepath);

    CustomAllocator::callbacks = pDocument->allocatorCallbacks;

    nn::Result result;
    nn::fs::FileHandle hFile;
    int64_t fileSize = 0;

    result = nn::fs::OpenFile(&hFile, filepath, nn::fs::OpenMode_Read);
    if (result.IsFailure())
        return result;

    result = nn::fs::GetFileSize(&fileSize, hFile);
    if (result.IsSuccess())
    {
        // +1
        char* pReadBuffer = static_cast<char*>(pDocument->allocatorCallbacks.pAllocFunc(fileSize + 1));
        memset(pReadBuffer, 0, fileSize + 1);
        size_t bytesRead = 0;
        result = nn::fs::ReadFile(&bytesRead, hFile, 0, pReadBuffer, fileSize, nn::fs::ReadOption::MakeValue(0));
        if (result.IsSuccess())
        {
            if (bytesRead == static_cast<size_t>(fileSize))
            {
                pDocument->doc.Parse(pReadBuffer);
                if (pDocument->doc.HasParseError())
                {
                    rapidjson::ParseErrorCode errorCode = pDocument->doc.GetParseError();
                    size_t errorOffset = pDocument->doc.GetErrorOffset();
                    NN_LOG("Parse erorr %d at offset %d\n", errorCode, errorOffset);
                    result = nn::fs::ResultDataCorrupted();
                }
                else
                {
                    int documentVersion = GetDocumentVersion(pDocument);
                    if (documentVersion < MinSupportedVersion)
                    {
                        result = nn::fs::ResultDataCorrupted();
                    }
                    else
                    {
                        const JsonValue& testCasesArray = pDocument->doc[TestCasesKey];
                        NN_LOG("Found %d test cases\n", testCasesArray.Size());
                    }
                }
            }
            else
            {
                result = nn::fs::ResultDataCorrupted();
            }
        }

        pDocument->allocatorCallbacks.pFreeFunc(pReadBuffer);
    }

    nn::fs::CloseFile(hFile);

    return result;
}

nn::Result SaveToFile(const Document* pDocument, const char* filepath)
{
    NN_ASSERT_NOT_NULL(pDocument);
    NN_ASSERT_NOT_NULL(filepath);

    CustomAllocator::callbacks = pDocument->allocatorCallbacks;

    NnFileOutputStream outputStream;

    size_t cacheBufferSize = 1024;
    void* pCacheBuffer = pDocument->allocatorCallbacks.pAllocFunc(cacheBufferSize);
    outputStream.Open(filepath, pCacheBuffer, cacheBufferSize);

    if (outputStream.GetResult().IsFailure())
    {
        NN_LOG("Cannot open destination file %s\n", filepath);
        return outputStream.GetResult();
    }

    rapidjson::PrettyWriter<NnFileOutputStream, rapidjson::UTF8<>, rapidjson::UTF8<>, CustomAllocator> writer(outputStream);
    pDocument->doc.Accept(writer);

    nn::Result result = outputStream.GetResult();

    outputStream.Close();
    pDocument->allocatorCallbacks.pFreeFunc(pCacheBuffer);

    return result;
}

int GetDocumentVersion(Document* pDocument)
{
    NN_ASSERT(pDocument->doc.HasMember(VersionKey));
    return pDocument->doc[VersionKey].GetInt();
}

void InitializeTestIterator(TestCaseIterator** ppOutIterator, Document* pDocument)
{
    NN_ASSERT_NOT_NULL(pDocument);
    NN_ASSERT_NOT_NULL(ppOutIterator);

    *ppOutIterator = new (pDocument->allocatorCallbacks.pAllocFunc(sizeof(TestCaseIterator))) TestCaseIterator();

    const JsonValue& testCasesArray = pDocument->doc[TestCasesKey];
    JsonDocument::ConstValueIterator iter = testCasesArray.Begin();

    (*ppOutIterator)->pDocument = pDocument;
    (*ppOutIterator)->currentIterator = iter;
}

void FinalizeTestIterator(TestCaseIterator* pIterator)
{
    NN_ASSERT_NOT_NULL(pIterator);

    Document* pDocument = pIterator->pDocument;

    pIterator->~TestCaseIterator();
    pDocument->allocatorCallbacks.pFreeFunc(pIterator);
}

void CopyTestIterator(TestCaseIterator** ppOutIterator, TestCaseIterator* pSourceIterator)
{
    NN_ASSERT_NOT_NULL(ppOutIterator);
    NN_ASSERT_NOT_NULL(pSourceIterator);

    TestCaseIterator* pIteratorCopy = new (pSourceIterator->pDocument->allocatorCallbacks.pAllocFunc(sizeof(TestCaseIterator))) TestCaseIterator();
    *pIteratorCopy = *pSourceIterator;

    *ppOutIterator = pIteratorCopy;
}

bool HasMoreTestCase(const char* filterPattern, const TestCaseIterator* pIterator)
{
    NN_ASSERT_NOT_NULL(pIterator);

    TestCaseIterator iterator = *pIterator;
    return MoveToNextTestCase(filterPattern, &iterator);
}

bool MoveToNextTestCase(const char* filterPattern, TestCaseIterator* pIterator)
{
    NN_ASSERT_NOT_NULL(pIterator);

    const JsonValue& testCasesArray = pIterator->pDocument->doc[TestCasesKey];
    JsonDocument::ConstValueIterator iter = pIterator->currentIterator;

    while (true)
    {
        if (iter == testCasesArray.End())
        {
            break;
        }

        if (MatchFilterPattern(filterPattern, iter))
        {
            break;
        }

        iter++;
    }

    if (iter != testCasesArray.End())
    {
        pIterator->currentIterator = iter + 1;
        return true;
    }

    return false;
}



const char* GetTestCaseName(const TestCaseIterator* pIterator)
{
    NN_ASSERT_NOT_NULL(pIterator);

    const JsonValue& value = GetIteratorValue(pIterator);
    NN_ASSERT(value.HasMember(BenchmarkConfigurationKey));
    NN_ASSERT(value[BenchmarkConfigurationKey].HasMember(NameKey));
    return value[BenchmarkConfigurationKey][NameKey].GetString();
}

const char* GetTestCaseId(const TestCaseIterator* pIterator)
{
    NN_ASSERT_NOT_NULL(pIterator);

    const JsonValue& value = GetIteratorValue(pIterator);
    NN_ASSERT(value.HasMember(TestIdKey));
    return value[TestIdKey].GetString();
}

int GetTestCaseWarmUpCount(const TestCaseIterator* pIterator)
{
    NN_ASSERT_NOT_NULL(pIterator);

    const JsonValue& value = GetIteratorValue(pIterator);
    int documentVersion = pIterator->pDocument->doc[VersionKey].GetInt();

    if (documentVersion < 2)
    {
        NN_ASSERT(!value.HasMember(WarmUpCountKey));
        return 4;
    }
    else
    {

        NN_ASSERT(value.HasMember(WarmUpCountKey));
        return value[WarmUpCountKey].GetInt();
    }
}

int GetTestCaseRepeatCount(const TestCaseIterator* pIterator)
{
    NN_ASSERT_NOT_NULL(pIterator);

    const JsonValue& value = GetIteratorValue(pIterator);
    NN_ASSERT(value.HasMember(RepeatCountKey));
    return value[RepeatCountKey].GetInt();
}

int GetTestCaseOutputBufferHash(const TestCaseIterator* pIterator, void* pOutputBufferHash, int outputBufferHashSize)
{
    NN_ASSERT_NOT_NULL(pIterator);

    const JsonValue& value = GetIteratorValue(pIterator);
    NN_ASSERT(value.HasMember(OutputBufferHashKey));
    const char* hastDataAsBase64 = value[OutputBufferHashKey].GetString();

    size_t count = 0;
    nn::util::Base64::Status resultStatus = nn::util::Base64::FromBase64String(
        &count, pOutputBufferHash, outputBufferHashSize,
        hastDataAsBase64, nn::util::Base64::Mode_NormalNoLinefeed);
    NN_ASSERT(resultStatus == nn::util::Base64::Status_Success);

    return static_cast<int>(count);
}

void ApplyTestCaseConfiguration(const TestCaseIterator* pIterator, GpuBenchmark* pBenchmark)
{
    NN_ASSERT_NOT_NULL(pIterator);
    NN_ASSERT_NOT_NULL(pBenchmark);

    const JsonValue& value = GetIteratorValue(pIterator);
    NN_ASSERT(value.HasMember(BenchmarkConfigurationKey));
    NN_ASSERT(value[BenchmarkConfigurationKey].HasMember(PropertylistKey));
    const JsonValue& propertyList = value[BenchmarkConfigurationKey][PropertylistKey];

    for (JsonValue::ConstValueIterator it = propertyList.Begin(), itEnd = propertyList.End(); it != itEnd; ++it)
    {
        NN_ASSERT(it->HasMember(NameKey));
        NN_ASSERT(it->HasMember(TypeKey));
        NN_ASSERT(it->HasMember(ValueKey));

        const char* properyName = (*it)[NameKey].GetString();
        const char* properyType = (*it)[TypeKey].GetString();

        GpuBenchmarkPropertyHolder* pGpuBenchmarkPropertyHolder = pBenchmark->FindPropertyByName(properyName);
        if (pGpuBenchmarkPropertyHolder == nullptr)
        {
            NN_LOG("Unknown property name: %s\n", properyName);
        }
        else
        {
            if (strcmp(properyType, GetPropertyTypeName(PropertyType_Enumeration)) == 0)
            {
                if (pGpuBenchmarkPropertyHolder->GetType() == PropertyType_Enumeration)
                {
                    const char* propertyValue = (*it)[ValueKey].GetString();
                    int elementIndex = pGpuBenchmarkPropertyHolder->ToEnum()->FindElementIndexFromName(propertyValue);
                    if (elementIndex >= 0)
                    {
                        pGpuBenchmarkPropertyHolder->Set(elementIndex);
                    }
                    else
                    {
                        NN_LOG("Property %s unknown enumeration value %s\n",
                            properyName, propertyValue);
                    }
                }
                else
                {
                    NN_LOG("Property %s type mismatch: expected %s, received %s\n",
                        properyName,
                        GetPropertyTypeName(PropertyType_Enumeration),
                        GetPropertyTypeName(pGpuBenchmarkPropertyHolder->GetType()));
                }
            }
            else if (strcmp(properyType, GetPropertyTypeName(PropertyType_IntegerRange)) == 0)
            {
                if (pGpuBenchmarkPropertyHolder->GetType() == PropertyType_IntegerRange)
                {
                    int propertyValue = (*it)[ValueKey].GetInt();
                    if (pGpuBenchmarkPropertyHolder->ToIntegerRange()->IsValidValue(propertyValue))
                    {
                        pGpuBenchmarkPropertyHolder->Set(propertyValue);
                    }
                    else
                    {
                        NN_LOG("Property %s invalid integer range value %d\n",
                            properyName, propertyValue);
                    }
                }
                else
                {
                    NN_LOG("Property %s type mismatch: expected %s, received %s",
                        properyName,
                        GetPropertyTypeName(PropertyType_Enumeration),
                        GetPropertyTypeName(pGpuBenchmarkPropertyHolder->GetType()));
                }
            }
            else
            {
                NN_LOG("Unknown property type: %s\n", properyType);
                NN_ABORT();
            }
        }
    }
}

void GetTestCaseExpectedResult(const TestCaseIterator* pIterator, BenchmarkTestResult* pBenchmarkTestResult)
{
    NN_ASSERT(pIterator != nullptr);

    const JsonValue& value = GetIteratorValue(pIterator);

    const char* memberNameAverage = ExpectedResultAverageKey;
    NN_ASSERT(value.HasMember(memberNameAverage));
    const JsonValue& expectedResultAverage = value[memberNameAverage];
    NN_ASSERT(expectedResultAverage.IsArray());
    NN_ASSERT(expectedResultAverage.Size() == 2);
    pBenchmarkTestResult->cpuTimeAverage = expectedResultAverage[0].GetUint64();
    pBenchmarkTestResult->gpuTimeAverage = expectedResultAverage[1].GetUint64();

    const char* memberNameStandardDeviation = ExpectedResultStandardDeviationKey;
    NN_ASSERT(value.HasMember(memberNameStandardDeviation));
    const JsonValue& expectedResultStandardDeviation = value[memberNameStandardDeviation];
    NN_ASSERT(expectedResultStandardDeviation.IsArray());
    NN_ASSERT(expectedResultStandardDeviation.Size() == 2);
    pBenchmarkTestResult->cpuTimeStandardDeviation = expectedResultStandardDeviation[0].GetUint64();
    pBenchmarkTestResult->gpuTimeStandardDeviation = expectedResultStandardDeviation[1].GetUint64();
}

void SetBenchmarkTestCaseObject(
    JsonValue& jsonTestCaseObject, JsonDocument::AllocatorType& jsonAllocator,
    const GpuBenchmark* pBenchmark,
    const BenchmarkTestResult* pBenchmarkTestResult,
    const void* pOutputHashContent, int outputHashContentSize,
    int warmUpCount, int repeatCount,
    const char* uniqueTestId)
{
    JsonDocument::MemberIterator itMember;
    JsonValue jsonBenchmarkConfigurationValue;
    SetBenchmarkConfiguration(jsonBenchmarkConfigurationValue, jsonAllocator, pBenchmark);
    jsonTestCaseObject.AddMember(BenchmarkConfigurationKey, jsonBenchmarkConfigurationValue, jsonAllocator);

    JsonValue warmUpCountConfigurationValue;
    warmUpCountConfigurationValue.SetInt(warmUpCount);
    jsonTestCaseObject.AddMember(WarmUpCountKey, warmUpCountConfigurationValue, jsonAllocator);

    JsonValue repeatCountConfigurationValue;
    repeatCountConfigurationValue.SetInt(repeatCount);
    jsonTestCaseObject.AddMember(RepeatCountKey, repeatCountConfigurationValue, jsonAllocator);

    JsonValue jsonExpectedResultMinValue;
    SetExpectedResults(
        jsonExpectedResultMinValue, jsonAllocator,
        pBenchmarkTestResult->cpuTimeAverage, pBenchmarkTestResult->gpuTimeAverage);
    jsonTestCaseObject.AddMember(ExpectedResultAverageKey, jsonExpectedResultMinValue, jsonAllocator);

    JsonValue jsonExpectedResultMaxValue;
    SetExpectedResults(
        jsonExpectedResultMaxValue, jsonAllocator,
        pBenchmarkTestResult->cpuTimeStandardDeviation, pBenchmarkTestResult->gpuTimeStandardDeviation);
    jsonTestCaseObject.AddMember(ExpectedResultStandardDeviationKey, jsonExpectedResultMaxValue, jsonAllocator);

    const int outputBufferHashBase64Size = 256;
    char outputBufferHashBase64[outputBufferHashBase64Size] = {};
    nn::util::Base64::Status resultStatus = nn::util::Base64::ToBase64String(
        outputBufferHashBase64, outputBufferHashBase64Size,
        pOutputHashContent, outputHashContentSize,
        nn::util::Base64::Mode_NormalNoLinefeed);
    NN_ASSERT(resultStatus == nn::util::Base64::Status_Success);
    JsonValue jsonOutputBufferHashValue;

    rapidjson::SizeType hashLength = static_cast<rapidjson::SizeType>(strlen(outputBufferHashBase64));
    jsonOutputBufferHashValue.SetString(outputBufferHashBase64, hashLength, jsonAllocator);
    jsonTestCaseObject.AddMember(OutputBufferHashKey, jsonOutputBufferHashValue, jsonAllocator);

    JsonValue jsonPlatformIdValue;
    jsonPlatformIdValue.SetString(uniqueTestId, jsonAllocator);
    jsonTestCaseObject.AddMember(TestIdKey, jsonPlatformIdValue, jsonAllocator);
}

bool HasTestWithId(Document* pDocument, const char* uniqueTestId)
{
    NN_ASSERT_NOT_NULL(pDocument);
    NN_ASSERT_NOT_NULL(uniqueTestId);

    CustomAllocator::callbacks = pDocument->allocatorCallbacks;

    JsonDocument &jsonDocument = pDocument->doc;

    JsonValue& jsonTestCasesArray = jsonDocument[TestCasesKey];
    JsonDocument::ValueIterator it = jsonTestCasesArray.Begin();
    JsonDocument::ValueIterator itEnd = jsonTestCasesArray.End();
    while (it != itEnd)
    {
        if (strcmp(it->FindMember(TestIdKey)->value.GetString(), uniqueTestId) == 0)
        {
            break;
        }
        ++it;
    }

    return (it != itEnd);
}


void CreateOrUpdateBenchmarkTestCase(
    Document* pDocument,
    const GpuBenchmark* pBenchmark,
    const BenchmarkTestResult* pBenchmarkTestResult,
    const void* pOutputBufferHash, int outputBufferHashSize,
    int warmUpCount, int repeatCount,
    const char* uniqueTestId)
{
    NN_ASSERT_NOT_NULL(pDocument);
    NN_ASSERT_NOT_NULL(pBenchmark);
    NN_ASSERT(warmUpCount >= 0);
    NN_ASSERT(repeatCount > 0);
    NN_ASSERT_NOT_NULL(uniqueTestId);

    CustomAllocator::callbacks = pDocument->allocatorCallbacks;

    JsonDocument &jsonDocument = pDocument->doc;

    JsonValue jsonTestCaseObject;
    jsonTestCaseObject.SetObject();
    SetBenchmarkTestCaseObject(
        jsonTestCaseObject, jsonDocument.GetAllocator(),
        pBenchmark, pBenchmarkTestResult,
        pOutputBufferHash, outputBufferHashSize,
        warmUpCount, repeatCount, uniqueTestId);

    JsonValue& jsonTestCasesArray = jsonDocument[TestCasesKey];
    JsonDocument::ValueIterator it = jsonTestCasesArray.Begin();
    JsonDocument::ValueIterator itEnd = jsonTestCasesArray.End();
    while (it != itEnd)
    {
        if (strcmp(it->FindMember(TestIdKey)->value.GetString(), uniqueTestId) == 0)
        {
            break;
        }
        ++it;
    }

    if (it != itEnd)
    {
        (*it) = jsonTestCaseObject;
    }
    else
    {
        jsonTestCasesArray.PushBack(jsonTestCaseObject, jsonDocument.GetAllocator());
    }
}

} // namespace json

} } } // namespace nnt { namespace gfx { namespace util {
