﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include "gfxUtilGpuBenchmark_GpuBenchmarkDepthTest.h"

#include "gfxUtilGpuBenchmark_ResHelpers.h"
#include "gfxUtilGpuBenchmark_PropertyMacros.h"
#include "gfxUtilGpuBenchmark_ComputeDepthTestShaderVariationIndex.h"

#include "gfxUtilGpuBenchmark_BuiltinDepthTestShader.h"


namespace nnt { namespace gfx { namespace util {

const char* GpuBenchmarkDepthTest::ClassName = "DepthTest";

GpuBenchmarkDepthTest::GpuBenchmarkDepthTest()
: m_OverdrawCount(16)
, m_DepthBufferDirection(DepthBufferDirection_Less)
{
}

GpuBenchmarkDepthTest::~GpuBenchmarkDepthTest()
{
}

void GpuBenchmarkDepthTest::Initialize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    // DepthBufferDirection
    GpuBenchmarkPropertyHolder* pPropertyDepthBufferDirection = m_PropertyArray.Get(Property_DepthBufferDirection);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyDepthBufferDirection, "DepthBufferDirection",
        DepthBufferDirection, pResourceAllocator,
        GpuBenchmarkDepthTest::GetDepthBufferDirection,
        GpuBenchmarkDepthTest::SetDepthBufferDirection,
        "Less", DepthBufferDirection_Less,
        "Greater", DepthBufferDirection_Greater);

    // EarlyZ
    GpuBenchmarkPropertyHolder* pPropertyEarlyZ = m_PropertyArray.Get(Property_EarlyZ);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyEarlyZ, "EarlyZ",
        EarlyZ, pResourceAllocator,
        GpuBenchmarkDepthTest::GetEarlyZ,
        GpuBenchmarkDepthTest::SetEarlyZ,
        "On", EarlyZ_On,
        "Off", EarlyZ_Off);
}

void GpuBenchmarkDepthTest::Finalize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    for (int i = 0; i < m_PropertyArray.GetCount(); ++i)
    {
        m_PropertyArray.Get(i)->Finalize();
    }
}

void GpuBenchmarkDepthTest::InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    GpuBenchmark::InitializeGfxObjects(pResourceAllocator, pDevice);

    m_OutputCopyBufferSize = InitializeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer,
        &m_RenderTextureColorTargetView, &m_ViewportScissorState,
        m_RenderSize, m_RenderSize, m_RenderFormat, m_TileMode,
        pResourceAllocator, pDevice);

    nn::gfx::Texture::InfoType depthTextureInfo;
    {
        depthTextureInfo.SetDefault();
        depthTextureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_DepthStencil);
        depthTextureInfo.SetWidth(m_RenderSize);
        depthTextureInfo.SetHeight(m_RenderSize);
        depthTextureInfo.SetDepth(1);
        depthTextureInfo.SetImageFormat(m_DepthBufferFormat);
        depthTextureInfo.SetMipCount(1);
        depthTextureInfo.SetArrayLength(1);
        depthTextureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        depthTextureInfo.SetTileMode(m_TileMode);
    }

    InitializeTexture(
        &m_DepthStencilTexture, depthTextureInfo,
        pResourceAllocator, MemoryPoolType_RenderTarget, pDevice);

    nn::gfx::DepthStencilView::InfoType depthStencilViewInfo;
    {
        depthStencilViewInfo.SetDefault();
        depthStencilViewInfo.SetTexturePtr(&m_DepthStencilTexture);
        depthStencilViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    }

    m_DepthStencilView.Initialize(pDevice, depthStencilViewInfo);

    nn::gfx::ComparisonFunction comparisonFunction = GetComparisonFunction();
    nn::gfx::DepthStencilState::InfoType depthStencilInfo;
    {
        depthStencilInfo.SetDefault();
        depthStencilInfo.SetDepthTestEnabled(true);
        depthStencilInfo.SetDepthComparisonFunction(comparisonFunction);
        depthStencilInfo.SetDepthWriteEnabled(true);
    }
    m_DepthStencilState.Initialize(pDevice, depthStencilInfo);

    InitializeResShader(&m_ResShader, g_DepthTestShaderData, sizeof(g_DepthTestShaderData), pResourceAllocator, pDevice);
}

void GpuBenchmarkDepthTest::FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    FinalizeResShader(&m_ResShader, pResourceAllocator, pDevice);

    m_DepthStencilState.Finalize(pDevice);

    m_DepthStencilView.Finalize(pDevice);
    FinalizeTexture(&m_DepthStencilTexture, pResourceAllocator, pDevice);

    FinalizeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer,
        &m_RenderTextureColorTargetView, &m_ViewportScissorState,
        pResourceAllocator, pDevice);

    GpuBenchmark::FinalizeGfxObjects(pResourceAllocator, pDevice);
}

void GpuBenchmarkDepthTest::PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
    pTestCommandBuffer->SetDepthStencilState(&m_DepthStencilState);

    const int shaderVariationIndex = ComputeDepthTestShaderVariationIndex(m_EarlyZ);
    nn::gfx::Shader* pShader = m_ResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex)->GetResShaderProgram(m_ResShader.codeType)->GetShader();
    pTestCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);
}

void GpuBenchmarkDepthTest::DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount)
{
    float depthClearValue = GetDepthClearValue();

    for (int runIndex = 0; runIndex < runCount; ++runIndex)
    {
        nn::gfx::ColorTargetView* pTestTarget = &m_RenderTextureColorTargetView;
        pTestCommandBuffer->ClearColor(pTestTarget, 0.5f, 0.5f, 0.5f, 0.5f, nullptr);
        pTestCommandBuffer->ClearDepthStencil(&m_DepthStencilView, depthClearValue, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        pTestCommandBuffer->SetRenderTargets(1, &pTestTarget, &m_DepthStencilView);
        pTestCommandBuffer->SetViewportScissorState(&m_ViewportScissorState);

        for (int overdrawIndex = 0; overdrawIndex < m_OverdrawCount; ++overdrawIndex)
        {
            int vertexOffset = ((m_OverdrawCount - 1) << 12) | (overdrawIndex << 2);
            pTestCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleList, 4, vertexOffset);
        }
    }
}

void GpuBenchmarkDepthTest::PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter)
{
    NN_ASSERT(runCount > 0);

    uint64_t gpuTimeElapsedValueInNs = static_cast<uint64_t>(gpuTimeElapsed.GetNanoSeconds());
    uint64_t cpuTimeElapsedValueInNs = static_cast<uint64_t>(cpuTimeElapsed.GetNanoSeconds());

    uint64_t gpuTimeElapsedAvgValueInNs = gpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);
    uint64_t cpuTimeElapsedAvgValueInNs = cpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);

    pDebugFontTextWriter->Print("gpu time (ns): %8lu\n", gpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("cpu time (ns): %8lu\n", cpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("total gpu time (ns): %12lu\n", gpuTimeElapsedValueInNs);
    pDebugFontTextWriter->Print("total cpu time (ns): %12lu\n", cpuTimeElapsedValueInNs);
}

void GpuBenchmarkDepthTest::CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer)
{
    int renderSize = m_RenderSize;

    nn::gfx::BufferTextureCopyRegion bufferTextureCopyRegion;
    bufferTextureCopyRegion.SetDefault();
    bufferTextureCopyRegion.SetBufferImageHeight(renderSize);
    bufferTextureCopyRegion.SetBufferImageWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetDefault();
    bufferTextureCopyRegion.EditTextureCopyRegion().SetWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetHeight(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().EditSubresource().SetDefault();

    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);
    pCommandBuffer->CopyImageToBuffer(&m_OutputCopyBuffer, &m_RenderTexture, bufferTextureCopyRegion);
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Write);
}

void GpuBenchmarkDepthTest::MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize)
{
    *pOutBuffer = m_OutputCopyBuffer.Map();
    *pOutBufferSize = m_OutputCopyBufferSize;
}

void GpuBenchmarkDepthTest::UnmapResultBuffer()
{
    m_OutputCopyBuffer.Unmap();
}

const char* GpuBenchmarkDepthTest::GetName() const
{
    return ClassName;
}

BenchmarkType GpuBenchmarkDepthTest::GetType() const
{
    return BenchmarkType_DepthTest;
}

int GpuBenchmarkDepthTest::GetPropertyCount() const
{
    return m_PropertyArray.GetCount();
}

int GpuBenchmarkDepthTest::FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

int GpuBenchmarkDepthTest::FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize)
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkDepthTest::FindPropertyByName(const char* propertyName)
{
    return m_PropertyArray.FindPropertyByName(propertyName);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkDepthTest::GetPropertyByIndex(int index)
{
    return m_PropertyArray.Get(index);
}

nn::gfx::ComparisonFunction GpuBenchmarkDepthTest::GetComparisonFunction() const
{
    switch (m_DepthBufferDirection)
    {
    case DepthBufferDirection_Less:
        return nn::gfx::ComparisonFunction_LessEqual;
    case DepthBufferDirection_Greater:
        return nn::gfx::ComparisonFunction_GreaterEqual;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

float GpuBenchmarkDepthTest::GetDepthClearValue() const
{
    switch (m_DepthBufferDirection)
    {
    case DepthBufferDirection_Less:
        return 1.0f;
    case DepthBufferDirection_Greater:
        return 0.0f;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}


} } } // namespace nnt { namespace gfx { namespace util {
