﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx/gfx_Types.h>
#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_Sampler.h>
#include <nn/gfx/gfx_CommandBuffer.h>
#include <nn/gfx/gfx_DescriptorPool.h>
#include <nn/gfx/gfx_MemoryPool.h>
#include <nn/gfx/gfx_Buffer.h>
#include <nn/gfx/gfx_Sync.h>
#include <nn/gfx/gfx_State.h>

#include "gfxUtilGpuBenchmark_Types.h"


namespace nn { namespace  gfx { namespace util {

class DebugFontTextWriter;

} } } // namespace nn { namespace  gfx { namespace util {



namespace nnt { namespace gfx { namespace util {

class ResourceAllocator;
class GpuBenchmarkPropertyHolder;

struct CommandBufferData
{
    nn::gfx::CommandBuffer      commandBuffer;
    nn::gfx::MemoryPool*        pCommandMemoryPool;
    ptrdiff_t                   commandMemoryOffset;
    size_t                      commandMemorySize;

    void*                       pControlMemory;
    size_t                      controlMemorySize;
};

struct RuntimeGfxObjects
{
    CommandBufferData           benchmarkCommandBuffer;

    nn::gfx::Buffer             timestampBuffer;
    size_t                      timestampBufferSize;

    nn::gfx::Fence              commandFence;

    nn::gfx::DescriptorPool*    pBufferViewDescriptorPool;
    nn::gfx::DescriptorPool*    pTextureViewDescriptorPool;
    nn::gfx::DescriptorPool*    pSamplerDescriptorPool;

    nn::gfx::VertexState        defaultVertexState;
    nn::gfx::BlendState         defaultBlendState;
    nn::gfx::RasterizerState    defaultRasterizerState;
    nn::gfx::DepthStencilState  defaultDepthStencilState;
};

class GpuBenchmark
{
private:
    bool                                        m_GetGfxObjectsInitialized;

public:
                                                GpuBenchmark();
    virtual                                     ~GpuBenchmark();

    virtual void                                Initialize(ResourceAllocator* pResourceAllocator);
    virtual void                                Finalize(ResourceAllocator* pResourceAllocator);

    virtual void                                InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);
    virtual void                                FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);
    bool                                        GetGfxObjectsInitialized() const;


    virtual void                                PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer);
    virtual void                                DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount);
    virtual void                                RenderDebug(nn::gfx::CommandBuffer* pTestCommandBuffer);

    virtual void                                PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter);

    virtual void                                CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer);
    virtual void                                MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize);
    virtual void                                UnmapResultBuffer();

    template<typename HashType>
    void                                        HashResultBuffer(uint8_t* pHashBuffer, int hashBufferSize);



    virtual const char*                         GetName() const;
    virtual BenchmarkType                       GetType() const;
    virtual int                                 GetPropertyCount() const;
    virtual int                                 FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const;
    virtual int                                 FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize);
    virtual GpuBenchmarkPropertyHolder*         FindPropertyByName(const char* propertyName);
    virtual GpuBenchmarkPropertyHolder*         GetPropertyByIndex(int index);
};

inline bool GpuBenchmark::GetGfxObjectsInitialized() const
{
    return m_GetGfxObjectsInitialized;
}

template<typename HashType>
void GpuBenchmark::HashResultBuffer(uint8_t* pHashBuffer, int hashBufferSize)
{
    NN_ASSERT(hashBufferSize >= HashType::HashSize);
    memset(pHashBuffer, 0, hashBufferSize);

    void* pBufferContent = 0;
    size_t bufferSize = 0;
    MapResultBuffer(&pBufferContent, &bufferSize);

    HashType hashGenerator;
    hashGenerator.Initialize();
    hashGenerator.Update(pBufferContent, bufferSize);
    hashGenerator.GetHash(pHashBuffer, hashBufferSize);

    UnmapResultBuffer();
}


void InitializeRuntimeGfxObjects(
    RuntimeGfxObjects* pRuntimeGfxObjects, nn::gfx::Device* pDevice,
    ResourceAllocator* pResourceAllocator);
void FinalizeRuntimeGfxObjects(
    RuntimeGfxObjects* pRuntimeGfxObjects, nn::gfx::Device* pDevice,
    ResourceAllocator* pResourceAllocator);

void RecordGpuBenchmarkCommandList(
    GpuBenchmark* pGpuBenchmark, RuntimeGfxObjects* pRuntimeGfxObjects,
    int warmupRunCount, int measureRunCount);

void RunGpuBenchmarkCommandList(
    nn::TimeSpan* pOutCpuDuration, nn::TimeSpan* pOutGpuDuration,
    nn::gfx::Queue* pQueue, RuntimeGfxObjects* pRuntimeGfxObjects);


// 注意： デスクリプタープールを変更します
void RenderGpuBenchmarkDebug(
    GpuBenchmark* pGpuBenchmark, RuntimeGfxObjects* pRuntimeGfxObjects,
    nn::gfx::CommandBuffer* pCommandBuffer);

enum ValidationResult
{
    ValidationResult_Pass,
    ValidationResult_MissBelow,
    ValidationResult_MissOver,
};

ValidationResult ValidateResultMinMax(
    nn::TimeSpan timeElapsed,
    uint64_t referenceResultMin,
    uint64_t referenceResultMax);

ValidationResult ValidateResultStandardDeviation(
    nn::TimeSpan timeElapsed,
    uint64_t referenceResultMean,
    uint64_t referenceResultStandardDeviation,
    int factor);


} } } // namespace nnt { namespace gfx { namespace util {
