﻿
#r "System.Web"
#r "System.Web.Extensions"
#r "System.Xml"

using System.Collections.Generic;
using System.Web.Script.Serialization;
using System.Xml;

const string nintendoHeader = @"/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>

";

public class VariationConfiguration
{
    public VariationConfiguration() { Preprocessor = new List<PreprocessorConfiguration>(); }
    public string IndexFunctionName { get; set; }
    public List<PreprocessorConfiguration> Preprocessor { get; set; }
}

public class PreprocessorConfiguration
{
    public PreprocessorConfiguration() { Values = new List<string>(); }
    public string ParameterName { get; set; }
    public string VertexShaderName { get; set; }
    public string PixelShaderName { get; set; }
    public List<string> Values { get; set; }
}


void WriteXmlPreprocessorDefinitionValue(XmlWriter writer, int index, string value)
{
    writer.WriteStartElement("PreprocessorDefinitionValue");
    writer.WriteStartAttribute("index");
    writer.WriteValue(index);
    writer.WriteEndAttribute();
    writer.WriteValue(value);
    writer.WriteEndElement();
}

void WriteXmlPreprocessorDefinitionDefinition(XmlWriter writer, string name, int index)
{
    writer.WriteStartElement("PreprocessorDefinitionDefinition");
    writer.WriteStartAttribute("index");
    writer.WriteValue(index);
    writer.WriteEndAttribute();
    writer.WriteStartAttribute("name");
    writer.WriteValue(name);
    writer.WriteEndAttribute();
    writer.WriteEndElement();
}


void WriteShaderVariationDefinition(XmlWriter writer, VariationConfiguration configuration)
{
    int vsDefinitionListCount = configuration.Preprocessor.Count(p => p.VertexShaderName != null);
    int psDefinitionListCount = configuration.Preprocessor.Count(p => p.PixelShaderName != null);

    writer.WriteStartElement("ShaderVariationDefinition");

    writer.WriteStartElement("VertexShaderVariationDefinition");

    writer.WriteStartElement("PreprocessorDefinitionDefinitionArray");
    writer.WriteStartAttribute("length");
    writer.WriteValue(vsDefinitionListCount + 2);
    writer.WriteEndAttribute();

    int vsDefinitionIndex = 0;
    foreach (var preprocessorConfiguration in configuration.Preprocessor)
    {
        string nameVs = preprocessorConfiguration.VertexShaderName;
        if (nameVs != null)
        {
            WriteXmlPreprocessorDefinitionDefinition(writer, nameVs, vsDefinitionIndex);
            vsDefinitionIndex++;
        }
    }

    WriteXmlPreprocessorDefinitionDefinition(writer, "VERTEX_SHADER", vsDefinitionIndex);
    WriteXmlPreprocessorDefinitionDefinition(writer, "INOUT", vsDefinitionIndex + 1);

    writer.WriteEndElement(); // VertexShaderVariationDefinition
    writer.WriteEndElement(); // PreprocessorDefinitionDefinitionArray

    writer.WriteStartElement("PixelShaderVariationDefinition");

    writer.WriteStartElement("PreprocessorDefinitionDefinitionArray");
    writer.WriteStartAttribute("length");
    writer.WriteValue(psDefinitionListCount + 2);
    writer.WriteEndAttribute();

    int psDefinitionIndex = 0;
    foreach (var preprocessorConfiguration in configuration.Preprocessor)
    {
        string namePs = preprocessorConfiguration.PixelShaderName;
        if (namePs != null)
        {
            WriteXmlPreprocessorDefinitionDefinition(writer, namePs, psDefinitionIndex);
            psDefinitionIndex++;
        }
    }

    WriteXmlPreprocessorDefinitionDefinition(writer, "PIXEL_SHADER", psDefinitionIndex);
    WriteXmlPreprocessorDefinitionDefinition(writer, "INOUT", psDefinitionIndex + 1);

    writer.WriteEndElement(); // VertexShaderVariationDefinition
    writer.WriteEndElement(); // PreprocessorDefinitionDefinitionArray

    writer.WriteEndElement();
}


static List<string> foo(int a, List<Array> x)
{
    List<string> retval = new List<string>();
    if (a == x.Count)
    {
        retval.Add("");
        return retval;
    }
    foreach (Object y in x[a])
    {
        foreach (string x2 in foo(a + 1, x))
        {
            retval.Add(y.ToString() + " " + x2.ToString());
        }

    }
    return retval;
}

void IterateOnCombination(List<List<string>> output, List<List<string>> list, int listIndex, List<string> currentResult)
{
    if (listIndex == list.Count)
    {
        output.Add(new List<string>(currentResult));
        return;
    }

    foreach (string s in list[listIndex])
    {
        currentResult.Add(s);
        IterateOnCombination(output, list, listIndex + 1, currentResult);
        currentResult.RemoveAt(currentResult.Count - 1);
    }
}

void WriteShaderVariationArray(
    XmlWriter writer, VariationConfiguration configuration)
{
    var combinationValuesList = new List<List<string>>();
    var definitionValuesList = new List<List<string>>();
    var vsDefinitionIndexList = new List<int>();
    var psDefinitionIndexList = new List<int>();

    int combinationCount = 1;
    int index = 0;

    foreach (PreprocessorConfiguration preprocessorConfiguration in configuration.Preprocessor)
    {
        if (preprocessorConfiguration.VertexShaderName != null)
        {
            vsDefinitionIndexList.Add(index);
        }

        if (preprocessorConfiguration.PixelShaderName != null)
        {
            psDefinitionIndexList.Add(index);
        }

        index = index + 1;
        definitionValuesList.Add(preprocessorConfiguration.Values);

        combinationCount = combinationCount * preprocessorConfiguration.Values.Count;
    }

    IterateOnCombination(combinationValuesList, definitionValuesList, 0, new List<string>());

    writer.WriteStartElement("ShaderVariationValueArray");
    writer.WriteStartAttribute("length");
    writer.WriteValue(combinationCount);
    writer.WriteEndAttribute();

    int variationIndex = 0;

    foreach (var combination in combinationValuesList)
    {
        writer.WriteStartElement("ShaderVariationValue");
        writer.WriteStartAttribute("index");
        writer.WriteValue(variationIndex);
        writer.WriteEndAttribute();

        writer.WriteStartElement("VertexShaderVariationValue");
        writer.WriteStartElement("PreprocessorDefinitionValueArray");
        writer.WriteStartAttribute("length");
        writer.WriteValue(vsDefinitionIndexList.Count + 2);
        writer.WriteEndAttribute();

        int vsCount = 0;
        foreach (int vsIndex in vsDefinitionIndexList)
        {
            WriteXmlPreprocessorDefinitionValue(
                writer,
                vsCount,
                combination[vsIndex]);

            vsCount = vsCount + 1;
        }

        WriteXmlPreprocessorDefinitionValue(writer, vsDefinitionIndexList.Count, "(1)");
        WriteXmlPreprocessorDefinitionValue(writer, vsDefinitionIndexList.Count + 1, "out");

        writer.WriteEndElement(); // PreprocessorDefinitionValueArray
        writer.WriteEndElement(); // VertexShaderVariationValue


        writer.WriteStartElement("PixelShaderVariationValue");
        writer.WriteStartElement("PreprocessorDefinitionValueArray");
        writer.WriteStartAttribute("length");
        writer.WriteValue(psDefinitionIndexList.Count + 2);
        writer.WriteEndAttribute();

        int psCount = 0;
        foreach (int psIndex in psDefinitionIndexList)
        {
            WriteXmlPreprocessorDefinitionValue(
                writer,
                psCount,
                combination[psIndex]);

            psCount = psCount + 1;
        }

        WriteXmlPreprocessorDefinitionValue(writer, psDefinitionIndexList.Count, "(1)");
        WriteXmlPreprocessorDefinitionValue(writer, psDefinitionIndexList.Count + 1, "in");

        writer.WriteEndElement(); // PreprocessorDefinitionValueArray
        writer.WriteEndElement(); // PixelShaderVariationValue

        writer.WriteEndElement(); // ShaderVariationValue

        variationIndex = variationIndex + 1;
    }


    writer.WriteEndElement(); // ShaderVariationValueArray
}


void OutputXmlVariationFile(string outputFilePath, VariationConfiguration configuration)
{
    var fileOutputStream = new FileStream(outputFilePath, FileMode.Create, FileAccess.Write);

    XmlWriterSettings settings = new XmlWriterSettings();
    settings.Encoding = new UTF8Encoding(true);
    settings.Indent = true;
    settings.IndentChars = "    ";
    var writer = XmlWriter.Create(fileOutputStream, settings);

    writer.WriteStartDocument();
    writer.WriteStartElement("GfxShaderVariation");
    writer.WriteAttributeString("version", "0.0.0");

    WriteShaderVariationDefinition(writer, configuration);

    WriteShaderVariationArray(writer, configuration);

    writer.WriteEndElement(); // GfxShaderVariation
    writer.WriteEndDocument();

    writer.Flush();
    fileOutputStream.Close();
}

void OutputComputeIndexFunction(string outputFilePath, VariationConfiguration configuration)
{
    List<int> multiplicationFactorList = new List<int>();
    foreach (PreprocessorConfiguration pc in configuration.Preprocessor)
    {
        for (int i = 0; i < multiplicationFactorList.Count; ++i)
        {
            multiplicationFactorList[i] = multiplicationFactorList[i] * pc.Values.Count;
        }

        multiplicationFactorList.Add(1);
    }


    var fileOutputStream = new FileStream(outputFilePath, FileMode.Create, FileAccess.Write);
    var writer = new StreamWriter(fileOutputStream, new UTF8Encoding(true));

    writer.Write(nintendoHeader);
    string indexFunctionName = configuration.IndexFunctionName;
    writer.Write(String.Format("inline int Compute{0}ShaderVariationIndex(", indexFunctionName));

    if (configuration.Preprocessor.Count > 0)
    {
        writer.Write(String.Format("int {0}", configuration.Preprocessor[0].ParameterName));

        foreach (PreprocessorConfiguration pc in configuration.Preprocessor.Skip(1))
        {
            writer.Write(String.Format(", int {0}", pc.ParameterName));
        }
    }

    writer.Write(")");
    writer.Write(Environment.NewLine);
    writer.Write("{");
    writer.Write(Environment.NewLine);

    foreach (PreprocessorConfiguration pc in configuration.Preprocessor)
    {
        writer.Write(String.Format("    NN_ASSERT({0} < {1});", pc.ParameterName, pc.Values.Count));
        writer.Write(Environment.NewLine);
    }

    writer.Write("    return ");
    if (configuration.Preprocessor.Count > 0)
    {
        for (int i = 0; i < configuration.Preprocessor.Count; ++i)
        {
            PreprocessorConfiguration pc = configuration.Preprocessor[i];
            int multiplicationFactor = multiplicationFactorList[i];
            if (i > 0)
                writer.Write(" + ");
            writer.Write(String.Format("({0} * {1})", pc.ParameterName, multiplicationFactor));
        }
    }
    else
    {
        writer.Write("0");
    }


    writer.Write(";");
    writer.Write(Environment.NewLine);
    writer.Write("}");
    writer.Write(Environment.NewLine);
    writer.Write("");
    writer.Write(Environment.NewLine);

    writer.Flush();
    writer.Close();
}


string inputFileName = Args[0];
string outputFileNameXml = Args[1];
string outputFileNameIndexFunction = Args[2];

string inputFileContent = File.ReadAllText(inputFileName);

JavaScriptSerializer serializer = new JavaScriptSerializer();
VariationConfiguration configuration = serializer.Deserialize<VariationConfiguration>(inputFileContent);

OutputXmlVariationFile(outputFileNameXml, configuration);
OutputComputeIndexFunction(outputFileNameIndexFunction, configuration);
