﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt/gfx/util/gfxUtil_AgingTestApi.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

#include "gfxUtilAgingTest_FrameRateTracker.h"
#include "gfxUtilAgingTest_FreeList.h"
#include "gfxUtilAgingTest_ValidationResultData.h"

#include "gfxUtilGpuBenchmark_GpuBenchmark.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"

namespace nnt { namespace gfx { namespace util {

class GpuBenchmark;
struct GpuBenchmarkResult;

namespace json {
struct Document;
struct TestCaseIterator;
struct BenchmarkTestResult;
}

namespace agingtest {


struct TestSuite
{
    bool                    active;

    json::Document*         pJsonDocument;
    json::TestCaseIterator* pTestCaseIterator;

    static const int        FilterPatternMaxLength = 128;
    char                    filterPattern[FilterPatternMaxLength];
    int                     startIndex;
    int                     maxCount;
    int                     iterationIndex;

    GpuBenchmark*           pGpuBenchmark;

    nn::TimeSpan            previousCpuDuration;
    nn::TimeSpan            previousGpuDuration;


    ValidationResultData    cpuResult;
    ValidationResultData    gpuResult;

    int                     testCount;
};


struct BmpData
{
    nn::gfx::Texture        texture;
    nn::gfx::TextureView    textureView;
    nn::gfx::TextureInfo    textureInfo;
};

struct TextureResourceData
{
    void*                       pBuffer;
    nn::gfx::ResTextureFile*    pResTextureFile;
    BmpData*                    pBmpData;

    int                         textureBaseSlotIndex;
    nn::gfx::DescriptorSlot     textureDescriptorSlot;
};

struct DebugFontWriterData
{
    nn::gfx::util::DebugFontTextWriter      debugFontWriter;
    void*                                   pDebugFontWriterHeap;
    size_t                                  debugFontWriterHeapSize;
    int                                     debugFontWriterTextureSlotIndex;
    int                                     debugFontWriterTextureDescriptor;
    int                                     debugFontWriterSamplerSlotIndex;
    int                                     debugFontWriterSamplerDescriptor;
};

class LibraryState
{
public:
    static const int FrameBufferWidth = 1280;
    static const int FrameBufferHeight = 768;
    static const int FrameBufferBpp = 4;
    static const int FrameBufferCount = 2;

private:
    static const int                                m_ActiveTestSuiteMaxCount = 8;
    typedef FreeList<TestSuite, m_ActiveTestSuiteMaxCount> TestSuiteArray;

    static const int                                m_ActiveTextureMaxCount = 8;
    typedef FreeList<TextureResourceData, m_ActiveTestSuiteMaxCount> TextureResourceDataArray;

    bool                                    m_Initialized;
    bool                                    m_FrameStarted;
    nns::gfx::GraphicsFramework             m_Gfw;

    nnt::gfx::util::ResourceAllocator       m_ResourceAllocator;
    nn::gfx::Queue                          m_GpuBenchmarkQueue;
    nn::gfx::Semaphore                      m_GpuBenchmarkQueueSemaphore;

    TestSuiteArray                          m_TestSuiteArray;
    TextureResourceDataArray                m_TextureResourceDataArray;

    int                                     m_SamplerDescriptorSlotIndex;
    nn::gfx::DescriptorSlot                 m_SamplerDescriptorSlot;

    nn::gfx::BlendState                     m_Draw2dRectTextureBlendState;

    DebugFontWriterData                     m_DebugFontWriterData;
    nns::gfx::PrimitiveRenderer::Renderer*  m_pPrimitiveRenderer;
    RuntimeGfxObjects                       m_RuntimeGfxObjects;

    FrameRateTracker                        m_FrameRateTracker;

    struct ResultHistoryElement
    {
        nn::TimeSpan        cpuDuration;
        nn::TimeSpan        gpuDuration;
        int                 testResultFlags;
    };

    static const int                    m_ResultHistoryLength = 256;
    int                                 m_ResultHistoryIndex;
    ResultHistoryElement                m_ResultHistory[m_ResultHistoryLength];
    bool                                m_HashComparisonFailed;

#if defined(NN_SDK_BUILD_DEBUG)
    nn::os::ThreadId                        m_OwnerThreadId;
#endif

public:
                            LibraryState();
                            ~LibraryState();

    void                    Initialize();
    void                    Finalize();


    void                    GetTestPlatformId(char* buffer, int bufferSize);

    nn::Result              LoadTestSuiteFromFile(TestSuiteHandle* pOutHandle, const char* szFilePath);
    void                    UnloadTestSuite(TestSuiteHandle handle);

    int                     InitializeTestSuite(
                                TestSuiteHandle handle, const char* filterPattern, int startIndex, int maxCount);
    void                    FinalizeTestSuite(TestSuiteHandle handle);

    bool                    HasMoreTests(TestSuiteHandle handle) const;
    bool                    BeginNextTest(TestSuiteHandle handle);
    const char*             GetTestName(TestSuiteHandle handle) const;
    const char*             GetTestCaseId(TestSuiteHandle handle) const;
    bool                    RunTest(
                                int* pOutTestResultFlags,
                                HashComparisonMode hashComparisonMode,
                                TestSuiteHandle handle);



    nn::Result              CreateTextureFromResourceFile(TextureHandle* pOutHandle, const char* szFilePath);
    nn::Result              CreateTextureFromBmpBuffer(TextureHandle* pOutHandle, const void* bmpBuffer, size_t bmpBufferSizeInBytes);
    nn::Result              CreateTextureFromBmpFile(TextureHandle* pOutHandle, const char* szFilePath);
    nn::Result              CreateTextureFromPixelBuffer(
                                TextureHandle* pOutHandle, const void* pPixelBuffer, size_t pixelBufferSizeInBytes,
                                int width, int height, ptrdiff_t pitch, PixelFormat pixelFormat);

    void                    FreeTextureResource(TextureHandle handle);

    int                     GetTextureWidth(TextureHandle handle);
    int                     GetTextureHeight(TextureHandle handle);

    void                    StartFrame();
    void                    EndFrame();

    void                    DrawTestResults(
                                TestSuiteHandle handle, int positionX, int positionY, float scale, const Color& color);
    void                    DrawFont(const char* text, int positionX, int positionY, float scale, const Color& color);
    void                    Draw2DFrame(int positionX, int positionY, int width, int height, const Color& color);
    void                    Draw2DRect(int positionX, int positionY, int width, int height, const Color& color);
    void                    Draw2DRectTexture(
                                int positionX, int positionY, int width, int height, TextureHandle handle);

    void                    DrawTestDebugInformation(TestSuiteHandle handle);

    nn::gfx::Device*                        GetGfxDevice();
    nn::gfx::CommandBuffer*                 GetGfxCommandBuffer();
    nn::gfx::util::DebugFontTextWriter*     GetDebugFontTextWriter();
    nns::gfx::PrimitiveRenderer::Renderer*  GetPrimitiveRenderer();

    int                                     GetFrameBufferWidth() const;
    int                                     GetFrameBufferHeight() const;

    void                    GetFps(float* pMinFps, float* pMaxFps, float* pAverageFps);

private:
    bool                    AllocateTestSuite(TestSuiteHandle* pOutTestSuiteHandle);
    void                    FreeTestSuite(TestSuiteHandle handle);

    void                    InitializeDebugFontTextWriter(int bufferCount);
    void                    FinalizeDebugFontTextWriter();

    void                    InitializePrimitiveRenderer(int screenWidth, int screenHeight, int bufferCount);
    void                    FinalizePrimitiveRenderer();


    void                    InitializeTestGfxObjects(int screenWidth, int screenHeight, int bufferCount);
    void                    FinalizeTestGfxObjects();

    GpuBenchmark*           InitializeBenchmark(const char* testName, json::TestCaseIterator* pTestCaseIterator);
    void                    FinalizeBenchmark(GpuBenchmark* pGpuBenchmark);

    bool                    ValidateAndUpdateTestResult(
                                int* pOutTestResultFlags,
                                HashComparisonMode hashComparisonMode,
                                TestSuite* pTestSuite);

    void                    ResetResultHistory();
    void                    AppendToResultHistory(
                                nn::TimeSpan cpuDuration, nn::TimeSpan gpuDuration, int testResultFlags);

    void                    ComputeStandardDeviationFromHistory(
                                uint64_t* pOutCpuMean, uint64_t* pOutCpuStandardDeviation,
                                uint64_t* pOutGpuMean, uint64_t* pOutGpuStandardDeviation) const;

    template<bool IsCpu>
    nn::TimeSpan            GetDuration(const ResultHistoryElement* pResultHistoryElement) const;
    template<bool IsCpu>
    uint64_t                GetDurationInNs(const ResultHistoryElement* pResultHistoryElement) const;
    template<bool IsCpu>
    void                    ComputeStandardDeviationFromHistory(uint64_t* pOutMean, uint64_t* pOutStandardDeviation) const;
    template<bool IsCpu>
    int                     FindFailedCountFromHistory(uint64_t mean, uint64_t standardDeviation, int factor) const;


    void                    LogTestResults(const TestSuite* pTestSuite) const;

    void                    InitializeTestCaseData(TestSuite* pTestSuite);
    void                    FinalizeTestCaseData(TestSuite* pTestSuite);

    void                    GetTestCaseAdjustedExpectedResult(
                                nnt::gfx::util::json::TestCaseIterator* pTestCaseIterator,
                                nnt::gfx::util::json::BenchmarkTestResult* pAdjustedExpectedResult) const;

    bool                    HashComparisonRequested(
                                HashComparisonMode hashComparisonMode,
                                int testCount, int testCountMinForValidation);
};

inline int LibraryState::GetFrameBufferWidth() const
{
    return FrameBufferWidth;
}

inline int LibraryState::GetFrameBufferHeight() const
{
    return FrameBufferHeight;
}

} } } } // namespace nnt { namespace gfx { namespace util { namespace agingtest {
