﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfxUtilAgingTest_FrameRateTracker.h"

#include <cstring>

namespace nnt { namespace gfx { namespace util { namespace agingtest {


FrameRateTracker::FrameRateTracker()
: m_FrameCounter(0)
, m_PreviousFrameTick(0)
, m_MaxTimeSpan()
, m_MinTimeSpan()
, m_AverageTimeSpan()
, m_FrameTimeSpanHistory()
{
}

FrameRateTracker::~FrameRateTracker()
{

}

void FrameRateTracker::Initialize()
{
    m_FrameCounter = 0;
    m_PreviousFrameTick = nn::os::Tick(0);
    m_MaxTimeSpan = std::numeric_limits<int64_t>::min();
    m_MinTimeSpan = std::numeric_limits<int64_t>::max();

    memset(m_FrameTimeSpanHistory, 0, sizeof(m_FrameTimeSpanHistory));
}

void FrameRateTracker::Finalize()
{

}

void FrameRateTracker::UpdateFps()
{
    nn::os::Tick endFrameTick = nn::os::GetSystemTick();

    if (m_FrameCounter == 0)
    {
        m_FrameCounter = 1;
        m_PreviousFrameTick = endFrameTick;
        return;
    }

    int64_t cpuTickElapsed = (endFrameTick - m_PreviousFrameTick).GetInt64Value();

    if (m_MinTimeSpan > cpuTickElapsed)
        m_MinTimeSpan = cpuTickElapsed;

    if (m_MaxTimeSpan < cpuTickElapsed)
        m_MaxTimeSpan = cpuTickElapsed;

    int nextHistoryIndex = (m_FrameCounter - 1) % m_HistoryLength;
    m_FrameTimeSpanHistory[nextHistoryIndex] = cpuTickElapsed;

    m_PreviousFrameTick = endFrameTick;
    m_FrameCounter++;

    m_AverageTimeSpan = ComputeAverageDurationTickFromHistory();
}

int64_t FrameRateTracker::ComputeAverageDurationTickFromHistory()
{
    int count = m_FrameCounter < m_HistoryLength ? m_FrameCounter : m_HistoryLength;


    int64_t sum = 0;

    for (int i = 0; i < count; ++i)
    {
        sum += m_FrameTimeSpanHistory[i];
    }

    sum /= count;

    return sum;
}

void FrameRateTracker::GetFps(float* pOutMinFps, float* pOutMaxFps, float* pOutAverageFps)
{
    float minTimeSpanInMs = static_cast<float>(nn::os::Tick(m_MinTimeSpan).ToTimeSpan().GetMicroSeconds());
    float maxTimeSpanInMs = static_cast<float>(nn::os::Tick(m_MaxTimeSpan).ToTimeSpan().GetMicroSeconds());
    float averageTimeSpanInMs = static_cast<float>(nn::os::Tick(m_AverageTimeSpan).ToTimeSpan().GetMicroSeconds());

    const float baseSpanInMs = 1000000.0f;

    if (pOutMinFps != nullptr)
    {
        *pOutMinFps = baseSpanInMs / maxTimeSpanInMs;
    }
    if (pOutMaxFps != nullptr)
    {
        *pOutMaxFps = baseSpanInMs / minTimeSpanInMs;
    }
    if (pOutAverageFps != nullptr)
    {
        *pOutAverageFps = baseSpanInMs / averageTimeSpanInMs;
    }
}


} } } } // namespace nnt { namespace gfx { namespace util { namespace agingtest {
