﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "WaterSimulation.h"

#include <nn/util/util_BitUtil.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#endif

namespace LegacyGPGPUShallowWater {

////////////////////////////////////////////////////
//
// Common Quad VB/IB, Sampler, ContextState
//
////////////////////////////////////////////////////

// -----
const u32 QUAD_VTX_STRIDE = (sizeof(f32) * 2 + sizeof(f32) * 2);
const u32 QUAD_POS_OFFSET = 0;
const u32 QUAD_TEXCOORD_OFFSET = 2;
const u32 BUFFER_IDX = 0;
#if NN_GFX_IS_TARGET_GX || NN_GFX_IS_TARGET_D3D
// ----- Model Vertex Data
f32 modelVtxs[] =
{
    -1.0f, -1.0f,  0.0f,  1.0f,
    1.0f, -1.0f,  1.0f,  1.0f,
    1.0f,  1.0f,  1.0f,  0.0f,
    -1.0f,  1.0f,  0.0f,  0.0f
};
#else
// GL4 needs to flip the texture coordinates.
// ----- Model Vertex Data
f32 modelVtxs[] =
{
    -1.0f, -1.0f,  0.0f,  0.0f,
    1.0f, -1.0f,  1.0f,  0.0f,
    1.0f,  1.0f,  1.0f,  1.0f,
    -1.0f,  1.0f,  0.0f,  1.0f
};
#endif
// ----- Quad Triangles
//u32 quadTrianglesIdxs[] = {0, 1, 2, 3};
u32 quadTrianglesIdxs[] = { 0, 1, 2,  0, 2, 3 };

struct Common
{
    // Pointers to GPU buffers
    DEMOGfxBuffer vtxModel;
    DEMOGfxBuffer idxModel;

    int idxCount;

    nn::gfx::Sampler mySampler[2];
    nn::gfx::DescriptorSlot mySamplerSlot[2];
} g_Common;

void Init()
{
    // Vertex buffer
    g_Common.vtxModel.Initialize( sizeof( modelVtxs ), modelVtxs ,
        nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read, 0 );

    // Index buffer
    g_Common.idxModel.Initialize( sizeof( quadTrianglesIdxs ), quadTrianglesIdxs,
        nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read, 0 );

    // Setup the index count
    g_Common.idxCount = sizeof(quadTrianglesIdxs) / sizeof(quadTrianglesIdxs[0]);

    // Sampler setting
    DEMOGfxInitSampler(&g_Common.mySampler[POINT], &g_Common.mySamplerSlot[POINT],
        nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint,
        nn::gfx::ComparisonFunction_Always );
    DEMOGfxInitSampler(&g_Common.mySampler[LINEAR], &g_Common.mySamplerSlot[LINEAR],
        nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint,
        nn::gfx::ComparisonFunction_Always );
}

void Free()
{
    g_Common.vtxModel.Finalize();
    g_Common.idxModel.Finalize();

    g_Common.mySampler[ POINT ].Finalize( &DEMODevice );
    g_Common.mySampler[ LINEAR ].Finalize( &DEMODevice );
}


////////////////////////////////////////////////////
//
// Texture2D
//
////////////////////////////////////////////////////

void InitTexture2D(Texture2D& texture, u32 w, u32 h,
                          nn::gfx::ImageFormat format, Sampler sampler, void* pinitvalues)
{
    texture.textureInfo.SetDefault();
    texture.textureInfo.SetWidth( w );
    texture.textureInfo.SetHeight( h );
    texture.textureInfo.SetDepth( 1 );
    texture.textureInfo.SetImageFormat( format );
    texture.textureInfo.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    texture.textureInfo.SetTileMode( nn::gfx::TileMode_Linear );
    texture.textureInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_Read );

    size_t imageSize = nn::gfx::Texture::CalculateMipDataSize( &DEMODevice, texture.textureInfo );
    size_t imageAlign = nn::gfx::Texture::CalculateMipDataAlignment( &DEMODevice, texture.textureInfo );
#if NN_GFX_IS_TARGET_NVN
    // FIXME: Need 128K alignment for HR6
    void *imagePtr = DEMOGfxAllocMEM2( nn::util::align_up( imageSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT ),
        nn::util::align_up( imageAlign, NVN_MEMORY_POOL_STORAGE_ALIGNMENT ) );
#else
    void *imagePtr = DEMOGfxAllocMEM2( imageSize, imageAlign );
#endif

    if(pinitvalues)
    {
        memcpy( imagePtr, pinitvalues, imageSize );
    }
    else
    {
        memset( imagePtr, 0, imageSize);
    }

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( imagePtr, imageSize );
#endif

    // Create the memory pool
    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached );
        info.SetPoolMemory( imagePtr, imageSize );
        texture.memoryPool.Initialize( &DEMODevice, info );
    }

    texture.imagePtr = imagePtr;
    texture.textureBuffer.Initialize( &DEMODevice, texture.textureInfo, &texture.memoryPool, 0, imageSize );

    texture.pSamplerSlot = &g_Common.mySamplerSlot[sampler];

    texture.width = w;
    texture.height = h;

    // set up a texture view
    DEMOGfxSetupTextureView( &texture.textureBuffer, &texture.textureView, &texture.textureSlot,
        nn::gfx::ImageDimension_2d, format, nn::gfx::DepthStencilFetchMode_DepthComponent );
}

void FreeTexture2D(Texture2D& texture)
{
    texture.textureView.Finalize( &DEMODevice );
    texture.textureBuffer.Finalize( &DEMODevice );
    texture.memoryPool.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( texture.imagePtr );
}


////////////////////////////////////////////////////
//
// Render Target Texture
//
////////////////////////////////////////////////////

void InvalidateRenderTarget(nn::gfx::CommandBuffer* cmd, RenderTarget& renderTarget)
{
    NN_UNUSED( renderTarget );
    cmd->InvalidateMemory( nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Texture );
}

void InitRenderTarget(RenderTarget& renderTarget, u32 w, u32 h,
                    nn::gfx::ImageFormat format, Sampler sampler)
{
    // initialize the texture
    DEMOGfxSetupTextureBuffer( &renderTarget.textureBuffer, NULL, NULL, &renderTarget.colorBuffer, NULL,
        &renderTarget.imagePtr, w, h, 1, 1, nn::gfx::ImageDimension_2d, format, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    renderTarget.pSamplerSlot = &g_Common.mySamplerSlot[sampler];

    renderTarget.width = w;
    renderTarget.height = h;

    // set up a texture view with special channel mapping
    {
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetImageDimension( nn::gfx::ImageDimension_2d );
        info.SetImageFormat( format );
        info.SetTexturePtr( &renderTarget.textureBuffer );

        if ( format == nn::gfx::ImageFormat_R32_Float )
        {
            info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Zero, nn::gfx::ChannelMapping_Zero, nn::gfx::ChannelMapping_One );
        }
        else if ( format == nn::gfx::ImageFormat_R32_G32_Float )
        {
            info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Green, nn::gfx::ChannelMapping_Zero, nn::gfx::ChannelMapping_One );
        }
        renderTarget.textureView.Initialize( &DEMODevice, info );

        DEMOTextureDescriptorPool.GetDescriptorSlot(&renderTarget.textureSlot, DEMOGfxRegisterTextureView( &renderTarget.textureView ));
    }
}

void FreeRenderTarget(RenderTarget& renderTarget)
{
    renderTarget.colorBuffer.Finalize( &DEMODevice );
    renderTarget.textureView.Finalize( &DEMODevice );
    renderTarget.textureBuffer.Finalize( &DEMODevice );
    renderTarget.imagePtr->Finalize();
}


////////////////////////////////////////////////////
//
// Legacy GPGPU pixel shader operations pass
//
////////////////////////////////////////////////////
//
// Wrapper to facilitate execution of Legacy GPGPU (pixel shader operations)
//
// Initialize pass in InitPass()
// Free the pass in FreePass()
//
// Processing for each frame
// Begin();
// SetInputUniform();               // Set input constant
// SetInputRenderTarget();          // Set input RenderTarget
// SetInputTexture2D();             // Set input Texture2D
// SetInputTextureSampler();            // Set input GX2Texture
// ComputePass(output RenderTargets);  //Run pixel shader operations
// End();
//
//

void SetInputUniform(Pass& pass, DEMOGfxBuffer& uniformView, const char* uniform_name)
{
    int loc = pass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, uniform_name );

    if( loc >= 0 )
    {
        DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ConstantBuffer );
        DEMOCommandBuffer.SetConstantBuffer( loc, nn::gfx::ShaderStage_Pixel, uniformView.gpuAddress, uniformView.size );
    }
    else
    {
        printf("Warning: SetInputUniform(%s) Failure. loc(%d) ps(%p)\n",
            uniform_name, loc, pass.pipeline.shaders.pixelShaderData );
    }
}

void SetInputRenderTarget(Pass& pass, RenderTarget& texture, const char* texture_name)
{
    nn::gfx::CommandBuffer *cmd = &DEMOCommandBuffer;

    // Texture Sampler lookup
    s32 loc = pass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, texture_name );

    if(loc >= 0)
    {
        // Set Texture
        cmd->SetTextureAndSampler( loc, nn::gfx::ShaderStage_Pixel, texture.textureSlot, *texture.pSamplerSlot );
    }
    else
    {
        printf("Warning: SetInputRenderTarget(%s) Failure. loc(%d)\n",
            texture_name, loc );
    }
}

void SetInputTexture2D(Pass& pass, Texture2D& texture, const char* texture_name)
{
    nn::gfx::CommandBuffer *cmd = &DEMOCommandBuffer;

    // Texture Sampler lookup
    int loc = pass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, texture_name);
    if(loc >= 0)
    {
        // Set Texture
        cmd->SetTextureAndSampler( loc, nn::gfx::ShaderStage_Pixel, texture.textureSlot, *texture.pSamplerSlot );
    }
    else
    {
        printf("Warning: SetInputTexture2D(%s) Failure. loc(%d)\n",
            texture_name, loc );
    }
}

void SetInputTextureSampler(Pass& pass, nn::gfx::DescriptorSlot& pTextureSlot, nn::gfx::DescriptorSlot& pSamplerSlot,
                        const char* texture_name)
{
    nn::gfx::CommandBuffer* cmd = &DEMOCommandBuffer;

    // Texture Sampler lookup
    s32 loc = pass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, texture_name);
    if(loc >= 0)
    {
        // Set Texture
        cmd->SetTextureAndSampler( loc, nn::gfx::ShaderStage_Pixel, pTextureSlot, pSamplerSlot );
    }
    else
    {
        printf("Warning: SetInputTextureSampler(%s) Failure. loc(%d)\n",
            texture_name, loc);
    }
}

void ComputePass(Pass& pass, RenderTarget* renderTarget[], u32 renderTargetNum)
{
    nn::gfx::CommandBuffer *cmd = &DEMOCommandBuffer;

    // Set Render Targets
    const int MAX_RENDER_TARGETS=4;
    nn::gfx::ColorTargetView *pTarget[MAX_RENDER_TARGETS];

    // Clear buffers
    for(u32 i=0; i<renderTargetNum; ++i)
    {
        cmd->ClearColor( &renderTarget[i]->colorBuffer,
                         0.0f, 0.0f, 0.0f, 0.0f, NULL );
    }
    cmd->ClearDepthStencil( &pass.myDepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    for(u32 i=0; i<renderTargetNum; ++i)
    {
        pTarget[i] = &renderTarget[i]->colorBuffer;
    }
    cmd->SetRenderTargets( renderTargetNum, pTarget, &pass.myDepthBufferView );

    // Set Viewport
    cmd->SetViewportScissorState( &pass.viewportScissor );

    // Set Pipeline
    cmd->SetPipeline( &pass.pipeline.pipeline );
    cmd->InvalidateMemory( nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_Texture );

    // Bind vertex buffer
    cmd->SetVertexBuffer( BUFFER_IDX, g_Common.vtxModel.gpuAddress, QUAD_VTX_STRIDE, g_Common.vtxModel.size );
    cmd->DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32,
        g_Common.idxModel.gpuAddress, g_Common.idxCount, 0 );

    cmd->InvalidateMemory( nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Texture );
}

#if NN_GFX_IS_TARGET_D3D
void ResetRenderTarget()
{
    nn::gfx::CommandBuffer *cmd = &DEMOCommandBuffer;

    cmd->SetRenderTargets( 0, NULL, NULL );
}

#endif

void InitPipeline(Pass& pass, int numRenderTargets, nn::gfx::ImageFormat* formats )
{
    pass.pipeline.SetDefaults();

    pass.pipeline.colorTargetStateCount = numRenderTargets;
    pass.pipeline.blendTargetStateCount = numRenderTargets;
    for(int i = 0; i < numRenderTargets; i++)
    {
        pass.pipeline.colorTargetStateInfoArray[i].SetDefault();
        pass.pipeline.colorTargetStateInfoArray[i].SetFormat( formats[i] );
        pass.pipeline.blendTargetStateInfoArray[i].SetDefault();
    }

    DEMOGfxInitShaderVertexBuffer( &pass.pipeline.shaders, BUFFER_IDX, QUAD_VTX_STRIDE, 0 );

    pass.pipeline.Initialize( &DEMODevice );
}

void InitPass(Pass& pass, const char* shaderName,
              const u32 domain_width, const u32 domain_height,
              const u32 numRenderTargets, nn::gfx::ImageFormat* imageFormats )
{
    DEMOGfxLoadShadersFromFile(&pass.pipeline.shaders, 0, shaderName);

    // Attribute Location Lookup
    DEMOGfxInitShaderAttribute( &pass.pipeline.shaders, "a_position", BUFFER_IDX, QUAD_POS_OFFSET * sizeof(f32), nn::gfx::AttributeFormat_32_32_Float );
    DEMOGfxInitShaderAttribute( &pass.pipeline.shaders, "a_texCoord", BUFFER_IDX, QUAD_TEXCOORD_OFFSET * sizeof(f32), nn::gfx::AttributeFormat_32_32_Float );

    // Initialize render depth buffer
    DEMOGfxSetupTextureBuffer( &pass.myDepthBufferTexture,
        NULL, NULL, NULL, &pass.myDepthBufferView, &pass.pDepthBufferData,
        domain_width, domain_height, 1, 1, nn::gfx::ImageDimension_2d,
        nn::gfx::ImageFormat_D32_Float,
        nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // initialize viewport
    pass.myColorBufferWidth = domain_width;
    pass.myColorBufferHeight = domain_height;
    DEMOGfxSetViewportScissorState( &pass.viewportScissor, &pass.viewportMem, 0.0f, 0.0f,
        static_cast< float >( domain_width ), static_cast< float >( domain_height ),
        0.0f, 1.0f, static_cast< float >( domain_height ), true );

    // initialize pipeline
    InitPipeline( pass, numRenderTargets, imageFormats );
}

void FreePass(Pass& pass)
{
    pass.viewportScissor.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(pass.viewportMem);

    pass.myDepthBufferView.Finalize( &DEMODevice );
    pass.myDepthBufferTexture.Finalize( &DEMODevice );
    pass.pDepthBufferData->Finalize();

    pass.pipeline.Finalize( &DEMODevice );
}

} //namespace LegacyGPGPUShallowWater
