﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>
#include "WaterSimulation.h"

#include <nnt.h>
#include <nnt/nnt_Argument.h>

static int g_ShaderFileIdx = 0;
static const char* SHALLOW_WATER_ADVECT_SHADERS[] =
{
    "shaders/sweAdvect",
    "shaders/sweAdvectHlslcc",
};
static const char* SHALLOW_WATER_HEIGHT_SHADERS[] =
{
    "shaders/sweHeight",
    "shaders/sweHeightHlslcc",
};
static const char* SHALLOW_WATER_VELOCITY_SHADERS[] =
{
    "shaders/sweVelocity",
    "shaders/sweVelocityHlslcc",
};
static const char* SHALLOW_WATER_COPY_SHADERS[] =
{
    "shaders/copyTexture",
    "shaders/copyTextureHlslcc",
};
#if NN_GFX_IS_TARGET_GX
static const char* const SHALLOW_WATER_LIGHTING_SHADERS[] =
{
    "shaders/sweLighting"
    "shaders/sweLightingHlslcc"
};
#else
static const char* const SHALLOW_WATER_LIGHTING_SHADERS[] =
{
    "shaders/sweLightingTessellated",
    "shaders/sweLightingTessellatedHlslcc",
};
#endif

static const char* const COLOR_TEXTURE     = "textures/shallowWater/pond";
static const char* const HEIGHTMAP_TEXTURE = "textures/shallowWater/pondTerrainMap";

static u32 s_FrameCount=0;

static void SetWiredRasterizer( nn::gfx::RasterizerState* pRasterizer, nn::gfx::CullMode cullMode,
    nn::gfx::FrontFace frontFace, int bias, float scale, float clamp )
{
    nn::gfx::RasterizerState::InfoType info;

    info.SetDefault();
    info.SetFrontFace( frontFace );
    info.SetSlopeScaledDepthBias( scale );
    info.SetDepthBias( bias );
    info.SetDepthBiasClamp( clamp );
    info.SetCullMode( cullMode );
    info.SetFillMode( nn::gfx::FillMode_Wireframe );
    info.SetScissorEnabled( true );
    pRasterizer->Initialize( &DEMODevice, info );
}

// ----- Surface Information

#define SURFACE_WIDTH  (DEMOColorBufferInfo.GetWidth())
#define SURFACE_HEIGHT  (DEMOColorBufferInfo.GetHeight())

// ----- Shader & Texture information

static const int DOMAIN_SIZE = 512;
static const int REFMAP_SIZE = 512;

//////////////////////////////////////////////

static LegacyGPGPUShallowWater::RenderTarget depthMap;
static LegacyGPGPUShallowWater::RenderTarget velocityMap[2];
static LegacyGPGPUShallowWater::RenderTarget waterMap[2];
static LegacyGPGPUShallowWater::RenderTarget heightMap;
static LegacyGPGPUShallowWater::Pass heightPass;
static LegacyGPGPUShallowWater::Pass velocityPass;
static LegacyGPGPUShallowWater::Pass advectPass;
static LegacyGPGPUShallowWater::Pass initWaterPass;
static LegacyGPGPUShallowWater::Pass initDepthPass;
static LegacyGPGPUShallowWater::Pass initHeightPass;
static LegacyGPGPUShallowWater::Pass initVelocityPass;

static LegacyGPGPUShallowWater::Texture2D initHeightMap;
static LegacyGPGPUShallowWater::Texture2D initDepthMap;
static LegacyGPGPUShallowWater::Texture2D initVelocityMap;
static LegacyGPGPUShallowWater::Texture2D initWaterMap;

static LegacyGPGPUShallowWater::RenderTarget reflectionMap;
static LegacyGPGPUShallowWater::RenderTarget refractionMap;

static nn::gfx::Texture          refDepthTexture;
static nn::gfx::DepthStencilView refDepthBufferView;
static DEMOGfxMemPool* refDepthTextureMem;

//////////////////////////////////////////////

struct UbHeight
{
    f32 u_shotCoord[2];
    f32 u_shotPower;
    f32 padding;
};

struct STATE
{
    f32 tessFactor;
    u32 tessMode;
    f32 u_shotCoord[2];
    f32 u_shotPower;
    f32 padding;
    bool drawWire;

    DEMOGfxBuffer ub_heightBuffer;
};
static STATE g_State;

static OSTime lastTime = 0;

//////////////////////////////////////////////


#if NN_GFX_IS_TARGET_GX
static const int NUM_ATTRIB = 4;
static GX2FetchShader fetchShader; // needed because we use tessellation
static GX2AttribStream attribs[NUM_ATTRIB];
#endif

static void* fsBuf;

// uniform location
static int ub_psLightingSlot = -1;  // slot for the ub_lighting uniform block in the pixel shader
#if NN_GFX_IS_TARGET_GX
static int ub_vsLightingSlot = -1;  // slot for the ub_lighting uniform block in the vertex shader
#else
static int ub_tcsLightingSlot = -1;  // slot for the ub_lighting uniform block in the hull shader
static int ub_tesLightingSlot = -1;  // slot for the ub_lighting uniform block in the domain shader
#endif

// structures for the uniform blocks
struct UbLighting
{
    Mtx44 u_ModelMtx;
    Mtx44 u_ViewMtx;
    Mtx44 u_ProjMtx;
    f32   isTerrain[4];
    f32   u_tessLevel;
};


static DEMOGfxBuffer ub_reflectBuffer;
static DEMOGfxBuffer ub_refractBuffer;
static DEMOGfxBuffer ub_waterBuffer;
static DEMOGfxBuffer ub_terrainBuffer;

static int heightmapLocVs = -1;
static int heightmapLocPs = -1;
static int colormapLoc = -1;
static int reflectionLoc = -1;
static int refractionLoc = -1;

static f32 isTerrain[4];

// Matrices
static Mtx44   viewMtx44;
static Mtx44   projMtx44;
static Mtx44  modelMtx44;

// ----- GX2 Triangle Data

static const int POSITION_STRIDE = (sizeof(f32) * 3);
static const int POSITION_OFFSET = 0;
static const int POSITION_BUFFER_IDX = 0;
static const int TEXCOORD_STRIDE = (sizeof(f32) * 2);
static const int TEXCOORD_OFFSET = 0;
static const int TEXCOORD_BUFFER_IDX = 1;

#define FIELD_SIZE 100.0f
#define FIELD_INTERVAL FIELD_SIZE * 2 / NUM_GRID
static const int NUM_GRID = 25;

// Number of vertices
static const u32 TRIANGLE_VERTEX_NUM = 6 * NUM_GRID * NUM_GRID;
static f32 posDataGX2[ 3 * TRIANGLE_VERTEX_NUM];
static f32 tcDataGX2[ 2 * TRIANGLE_VERTEX_NUM];

// ----- Attribute buffers
static DEMOGfxBuffer posBuf;
static DEMOGfxBuffer tcBuf;

// ----- Texture & sampler objects
static DEMOGfxTexture colorTex;
static DEMOGfxTexture heightMapTex;

static nn::gfx::Sampler mySampler;
static nn::gfx::DescriptorSlot mySamplerSlot;

static nn::gfx::ViewportScissorState refmapViewport;
static nn::gfx::ViewportScissorState surfaceViewport;
static void *refmapViewportMem;
static void *surfaceViewportMem;

static DEMOGfxPipeline reflectPipeline;
static DEMOGfxPipeline terrainPipeline;
static DEMOGfxPipeline refractPipeline;
static DEMOGfxPipeline terrainWiredPipeline;
static nn::gfx::RasterizerState terrainWiredRasterizer;

// initialization data for matrices
static Vec     up = {0.0f, 1.0f, 0.0f};
static Vec  objPt = {5.0f, -35.0f, -50.0f};
static Vec camLoc = {10.0f, 40.0f, 115.0f};

// Prototype
static void InitCamera(Mtx44 resultProjMtx44, Mtx44 resultViewMtx44);
static void InitTex();
static void CleanupTex();
static void InitScene();
static void CleanupScene();
static void DrawScene();
static void DrawReflectionMap();
static void DrawRefractionMap();
static void SetClipPlane(const f32 clipPlane[4] = NULL);
static void RunSimulationShader();
static void InitSimulationShader();
static void ProcessPad();

// Init function for setting projection matrix
static void InitCamera(Mtx44 resultProjMtx44, Mtx44 resultViewMtx44)
{
    // Row major matrices
    Mtx   lookAtMtx34;

    f32   pers = 50.0f;
    f32 aspect = (f32)SURFACE_WIDTH / (f32)SURFACE_HEIGHT;
    f32  znear = 10.0f;
    f32   zfar = 4000.0f;

    // Compute perspective matrix
    MTXPerspective(resultProjMtx44, pers, aspect, znear, zfar);

    // Compute lookAt matrix
    MTXLookAt(lookAtMtx34, &camLoc, &up, &objPt);
    MTX34To44(lookAtMtx34, resultViewMtx44);
}

static void InitTex()
{
    BOOL fOK = heightMapTex.Initialize( HEIGHTMAP_TEXTURE );
    ASSERT(fOK && "Invalid Height Texture File");
    fOK = colorTex.Initialize( COLOR_TEXTURE );
    ASSERT(fOK && "Invalid Height Color Texture File");

    // Initialize the sampler
    DEMOGfxInitSampler( &mySampler, &mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint,
        nn::gfx::ComparisonFunction_Always );
}

static void CleanupTex()
{
    // Free texture
    colorTex.Finalize();
    heightMapTex.Finalize();

    // Free Sampler
    mySampler.Finalize( &DEMODevice );

}

static void CreateWaterSimulationDataAndShaders()
{
    // Create textures for initialization
    f32* init_depth  = (f32*)DEMOAlloc(DOMAIN_SIZE*DOMAIN_SIZE*sizeof(f32));
    f32* init_height = (f32*)DEMOAlloc(DOMAIN_SIZE*DOMAIN_SIZE*sizeof(f32));
    f32* init_water  = (f32*)DEMOAlloc(DOMAIN_SIZE*DOMAIN_SIZE*sizeof(f32));

    for(u32 i=0; i<DOMAIN_SIZE*DOMAIN_SIZE; ++i)
    {
        init_depth[i] = 0.1f;
        init_water[i] = 0.0f;
        init_height[i] = init_depth[i] + init_water[i];
    }

    LegacyGPGPUShallowWater::InitTexture2D(initDepthMap, DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_Float, LegacyGPGPUShallowWater::POINT, init_depth);
    LegacyGPGPUShallowWater::InitTexture2D(initVelocityMap, DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_G32_Float, LegacyGPGPUShallowWater::POINT, NULL);
    LegacyGPGPUShallowWater::InitTexture2D(initHeightMap, DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_Float, LegacyGPGPUShallowWater::POINT, init_height);
    LegacyGPGPUShallowWater::InitTexture2D(initWaterMap, DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_Float, LegacyGPGPUShallowWater::POINT, init_height);

    DEMOFree(init_depth);
    DEMOFree(init_height);
    DEMOFree(init_water);

    // RenderTarget Setup
    LegacyGPGPUShallowWater::InitRenderTarget(depthMap, DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_Float, LegacyGPGPUShallowWater::LINEAR);
    LegacyGPGPUShallowWater::InitRenderTarget(velocityMap[0], DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_G32_Float, LegacyGPGPUShallowWater::LINEAR);
    LegacyGPGPUShallowWater::InitRenderTarget(velocityMap[1], DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_G32_Float, LegacyGPGPUShallowWater::LINEAR);
    LegacyGPGPUShallowWater::InitRenderTarget(waterMap[0], DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_Float, LegacyGPGPUShallowWater::LINEAR);
    LegacyGPGPUShallowWater::InitRenderTarget(waterMap[1], DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_Float, LegacyGPGPUShallowWater::LINEAR);
    LegacyGPGPUShallowWater::InitRenderTarget(heightMap, DOMAIN_SIZE, DOMAIN_SIZE, nn::gfx::ImageFormat_R32_Float, LegacyGPGPUShallowWater::LINEAR);

    // Legacy GPGPU Simulation Shader Setup
    nn::gfx::ImageFormat formats[2];

    formats[ 0 ] = nn::gfx::ImageFormat_R32_G32_Float;
    formats[ 1 ] = nn::gfx::ImageFormat_R32_Float;
    LegacyGPGPUShallowWater::InitPass( advectPass, SHALLOW_WATER_ADVECT_SHADERS[g_ShaderFileIdx], DOMAIN_SIZE, DOMAIN_SIZE, 2, formats );

    formats[ 0 ] = nn::gfx::ImageFormat_R32_Float;
    formats[ 1 ] = nn::gfx::ImageFormat_R32_G32_Float;
    LegacyGPGPUShallowWater::InitPass( heightPass, SHALLOW_WATER_HEIGHT_SHADERS[g_ShaderFileIdx], DOMAIN_SIZE, DOMAIN_SIZE, 2, formats );

    formats[ 0 ] = nn::gfx::ImageFormat_R32_G32_Float;
    LegacyGPGPUShallowWater::InitPass( velocityPass, SHALLOW_WATER_VELOCITY_SHADERS[g_ShaderFileIdx], DOMAIN_SIZE, DOMAIN_SIZE, 1, formats );

    formats[ 0 ] = nn::gfx::ImageFormat_R32_Float;
    LegacyGPGPUShallowWater::InitPass( initWaterPass, SHALLOW_WATER_COPY_SHADERS[g_ShaderFileIdx], DOMAIN_SIZE, DOMAIN_SIZE, 1, formats );

    formats[ 0 ] = nn::gfx::ImageFormat_R32_Float;
    LegacyGPGPUShallowWater::InitPass( initDepthPass, SHALLOW_WATER_COPY_SHADERS[g_ShaderFileIdx], DOMAIN_SIZE, DOMAIN_SIZE, 1, formats );

    formats[ 0 ] = nn::gfx::ImageFormat_R32_Float;
    LegacyGPGPUShallowWater::InitPass( initHeightPass, SHALLOW_WATER_COPY_SHADERS[g_ShaderFileIdx], DOMAIN_SIZE, DOMAIN_SIZE, 1, formats );

    formats[ 0 ] = nn::gfx::ImageFormat_R32_G32_Float;
    LegacyGPGPUShallowWater::InitPass( initVelocityPass, SHALLOW_WATER_COPY_SHADERS[g_ShaderFileIdx], DOMAIN_SIZE, DOMAIN_SIZE, 1, formats );
}

// set up a default pipeline
void SetDefaultPipeline( DEMOGfxPipeline* defaultPipeline, bool blendEnable,
    nn::gfx::CullMode cullMode, nn::gfx::FrontFace frontFace, nn::gfx::LogicOperation logicOp )
{

    defaultPipeline->SetDefaults();

    defaultPipeline->colorTargetStateCount = 1;
    defaultPipeline->blendTargetStateCount = 1;
    defaultPipeline->blendTargetStateInfoArray[ 0 ].SetDefault();
    defaultPipeline->colorTargetStateInfoArray[ 0 ].SetDefault();
    defaultPipeline->colorTargetStateInfoArray[0].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

#if NN_GFX_IS_TARGET_GX
    DEMOGfxInitShaderAttribute(&defaultPipeline->shaders, "tes_position", POSITION_BUFFER_IDX, POSITION_OFFSET * sizeof(float),
        nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute(&defaultPipeline->shaders, "tes_texCoord", TEXCOORD_BUFFER_IDX, TEXCOORD_OFFSET * sizeof(float),
        nn::gfx::AttributeFormat_32_32_Float );
#else
    DEMOGfxInitShaderAttribute(&defaultPipeline->shaders, "a_position", POSITION_BUFFER_IDX, POSITION_OFFSET * sizeof(float),
        nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute(&defaultPipeline->shaders, "a_texCoord", TEXCOORD_BUFFER_IDX, TEXCOORD_OFFSET * sizeof(float),
        nn::gfx::AttributeFormat_32_32_Float );
#endif
    DEMOGfxInitShaderVertexBuffer( &defaultPipeline->shaders, POSITION_BUFFER_IDX, POSITION_STRIDE, 0 );
    DEMOGfxInitShaderVertexBuffer( &defaultPipeline->shaders, TEXCOORD_BUFFER_IDX, TEXCOORD_STRIDE, 0 );

    // Set the blend state
    defaultPipeline->blendTargetStateInfoArray[0].SetBlendEnabled( blendEnable );
    defaultPipeline->blendStateInfo.SetLogicOperationEnabled( logicOp != nn::gfx::LogicOperation_Copy );
    defaultPipeline->blendStateInfo.SetLogicOperation( logicOp );

    // Setup the rasterizer state
    defaultPipeline->rasterizerStateInfo.SetDepthClipEnabled( true );
    defaultPipeline->rasterizerStateInfo.SetCullMode( cullMode );
    defaultPipeline->rasterizerStateInfo.SetFrontFace( frontFace );

    // Setup Tessellation
    defaultPipeline->tessellationStateInfo.SetPatchControlPointCount( 3 );

    // Initialize the pipeline
    defaultPipeline->Initialize( &DEMODevice );
}

#if NN_GFX_IS_TARGET_GX
// create a special tessellation aware fetch shader
static void CreateFetchShader()
{
    u32 posLoc = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "tes_position");
    u32 tcLoc = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,nn::gfx::ShaderInterfaceType_Input, "tes_texCoord");
    u32 barycLoc = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "gl_BarycentricCoord" );
    u32 indexLoc = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "gl_VertexTriangleIndex");

    // in order to use tessellation we need to set up GX2_SYSTEM_GENERATED_ATTRIBUTEs, which nn::gfx cannot do
    // Attribute Format setup
    GX2InitAttribStream(&attribs[0], posLoc, POSITION_BUFFER_IDX,
        POSITION_OFFSET * sizeof(float),
        GX2_ATTRIB_FORMAT_32_32_32_FLOAT);
    GX2InitAttribStream(&attribs[1], tcLoc, TEXCOORD_BUFFER_IDX,
        TEXCOORD_OFFSET * sizeof(float),
        GX2_ATTRIB_FORMAT_32_32_FLOAT);
    GX2InitAttribStream(&attribs[2], indexLoc, GX2_SYSTEM_GENERATED_ATTRIBUTE,
        GX2_SYSGEN_ATTRIB_TESSELLATION_INDEX,
        GX2_ATTRIB_FORMAT_32_32_32_UINT);
    GX2InitAttribStream(&attribs[3], barycLoc, GX2_SYSTEM_GENERATED_ATTRIBUTE,
        GX2_SYSGEN_ATTRIB_TESSELLATION_BARYC,
        GX2_ATTRIB_FORMAT_32_32_32_FLOAT);

    // GX2 Fetch Shader setup
    u32 fsSize = GX2CalcFetchShaderSizeEx(NUM_ATTRIB,
                                          GX2_FETCH_SHADER_TESSELATION_TRIANGLES,
                                          GX2_TESSELLATION_MODE_DISCRETE);
    fsBuf = DEMOGfxAllocMEM2(fsSize, GX2_SHADER_ALIGNMENT);
    GX2InitFetchShaderEx(&fetchShader, fsBuf, NUM_ATTRIB, attribs,
        GX2_FETCH_SHADER_TESSELATION_TRIANGLES, GX2_TESSELLATION_MODE_DISCRETE);
    GX2Invalidate(GX2_INVALIDATE_CPU_SHADER, fsBuf, fsSize);
}
#endif

// create grid for rendering
static void CreateGrid()
{
    u32 idx = 0;
    for(u32 i=0; i<NUM_GRID; i++)
    {
        for(u32 j=0; j<NUM_GRID; j++)
        {
            // position
            posDataGX2[3 * 6 * idx + 0 ] = -FIELD_SIZE + (j + 0) * FIELD_INTERVAL;
            posDataGX2[3 * 6 * idx + 1 ] = -5.0f;
            posDataGX2[3 * 6 * idx + 2 ] = -FIELD_SIZE + (i + 1) * FIELD_INTERVAL;

            posDataGX2[3 * 6 * idx + 3 ] = -FIELD_SIZE + (j + 0) * FIELD_INTERVAL;
            posDataGX2[3 * 6 * idx + 4 ] = -5.0f;
            posDataGX2[3 * 6 * idx + 5 ] = -FIELD_SIZE + (i + 0) * FIELD_INTERVAL;

            posDataGX2[3 * 6 * idx + 6 ] = -FIELD_SIZE + (j + 1) * FIELD_INTERVAL;
            posDataGX2[3 * 6 * idx + 7 ] = -5.0f;
            posDataGX2[3 * 6 * idx + 8 ] = -FIELD_SIZE + (i + 0) * FIELD_INTERVAL;

            posDataGX2[3 * 6 * idx + 9 ] = -FIELD_SIZE + (j + 1) * FIELD_INTERVAL;
            posDataGX2[3 * 6 * idx + 10] = -5.0f;
            posDataGX2[3 * 6 * idx + 11] = -FIELD_SIZE + (i + 0) * FIELD_INTERVAL;

            posDataGX2[3 * 6 * idx + 12] = -FIELD_SIZE + (j + 1) * FIELD_INTERVAL;
            posDataGX2[3 * 6 * idx + 13] = -5.0f;
            posDataGX2[3 * 6 * idx + 14] = -FIELD_SIZE + (i + 1) * FIELD_INTERVAL;

            posDataGX2[3 * 6 * idx + 15] = -FIELD_SIZE + (j + 0) * FIELD_INTERVAL;
            posDataGX2[3 * 6 * idx + 16] = -5.0f;
            posDataGX2[3 * 6 * idx + 17] = -FIELD_SIZE + (i + 1) * FIELD_INTERVAL;

            // texture coordinate
            tcDataGX2[2 * 6 * idx + 0 ] = 1.0f / (f32)(NUM_GRID) * (j + 0);
            tcDataGX2[2 * 6 * idx + 1 ] = 1.0f / (f32)(NUM_GRID) * (i + 1);

            tcDataGX2[2 * 6 * idx + 2 ] = 1.0f / (f32)(NUM_GRID) * (j + 0);
            tcDataGX2[2 * 6 * idx + 3 ] = 1.0f / (f32)(NUM_GRID) * (i + 0);

            tcDataGX2[2 * 6 * idx + 4 ] = 1.0f / (f32)(NUM_GRID) * (j + 1);
            tcDataGX2[2 * 6 * idx + 5 ] = 1.0f / (f32)(NUM_GRID) * (i + 0);

            tcDataGX2[2 * 6 * idx + 6 ] = 1.0f / (f32)(NUM_GRID) * (j + 1);
            tcDataGX2[2 * 6 * idx + 7 ] = 1.0f / (f32)(NUM_GRID) * (i + 0);

            tcDataGX2[2 * 6 * idx + 8 ] = 1.0f / (f32)(NUM_GRID) * (j + 1);
            tcDataGX2[2 * 6 * idx + 9 ] = 1.0f / (f32)(NUM_GRID) * (i + 1);

            tcDataGX2[2 * 6 * idx + 10] = 1.0f / (f32)(NUM_GRID) * (j + 0);
            tcDataGX2[2 * 6 * idx + 11] = 1.0f / (f32)(NUM_GRID) * (i + 1);

            idx++;
        }
    }
}

static void LoadTessellationShader( DEMOGfxShader* pShader )
{
    DEMOGfxLoadShadersFromFile(pShader, 0, SHALLOW_WATER_LIGHTING_SHADERS[g_ShaderFileIdx]);
}

// The init function for the rendering portions of this app
static void InitScene()
{
    LegacyGPGPUShallowWater::Init();

    CreateWaterSimulationDataAndShaders();

    //
    // Water Rendering / Tessellation Setup
    //

    // Load the shaders for each pass.
    LoadTessellationShader( &reflectPipeline.shaders );
    LoadTessellationShader( &terrainPipeline.shaders );
    LoadTessellationShader( &refractPipeline.shaders );
    LoadTessellationShader( &terrainWiredPipeline.shaders );

    // set up uniform blocks for shaders
    // FIXME: Do we really need more of these
    ub_reflectBuffer.Initialize( sizeof( UbLighting ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    ub_refractBuffer.Initialize( sizeof( UbLighting ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    ub_waterBuffer.Initialize( sizeof( UbLighting ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    ub_terrainBuffer.Initialize( sizeof( UbLighting ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

#if NN_GFX_IS_TARGET_GX
    CreateFetchShader();
#endif

    // position and texcoord data
    CreateGrid();

    // Vertex position buffer
    posBuf.Initialize( sizeof( posDataGX2 ), posDataGX2, nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Vertex texcoord buffer
    tcBuf.Initialize( sizeof( tcDataGX2 ), tcDataGX2 , nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer, 0 );

    // reflection / refraction buffer
    LegacyGPGPUShallowWater::InitRenderTarget(reflectionMap, REFMAP_SIZE, REFMAP_SIZE,
        nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, LegacyGPGPUShallowWater::LINEAR);
    LegacyGPGPUShallowWater::InitRenderTarget(refractionMap, REFMAP_SIZE, REFMAP_SIZE,
        nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, LegacyGPGPUShallowWater::LINEAR);

    // set up viewports
    DEMOGfxSetViewportScissorState( &surfaceViewport, &surfaceViewportMem, 0.0f, 0.0f,
        static_cast< float >( SURFACE_WIDTH ), static_cast< float >( SURFACE_HEIGHT ), 0.0f, 1.0f,
        static_cast< float >( SURFACE_HEIGHT ), true );
    DEMOGfxSetViewportScissorState( &refmapViewport, &refmapViewportMem, 0.0f, 0.0f,
        static_cast< float >( REFMAP_SIZE ), static_cast< float >( REFMAP_SIZE ),
        0.0f, 1.0f, static_cast< float >( REFMAP_SIZE ), true );

    // set up a depth buffer for the reflection/refraction
    DEMOGfxSetupTextureBuffer( &refDepthTexture, NULL, NULL, NULL, &refDepthBufferView, &refDepthTextureMem,
        REFMAP_SIZE, REFMAP_SIZE, 1, 1, nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // initialize the pipelines
    DEMOTestIsUseHlslccGlsl()
        ? SetDefaultPipeline( &reflectPipeline, false, nn::gfx::CullMode_Back, nn::gfx::FrontFace_Cw, nn::gfx::LogicOperation_Copy )
        : SetDefaultPipeline( &reflectPipeline, false, nn::gfx::CullMode_Back, nn::gfx::FrontFace_Ccw, nn::gfx::LogicOperation_Copy );
    DEMOTestIsUseHlslccGlsl()
        ? SetDefaultPipeline( &refractPipeline, false, nn::gfx::CullMode_Back, nn::gfx::FrontFace_Ccw, nn::gfx::LogicOperation_Copy )
        : SetDefaultPipeline( &refractPipeline, false, nn::gfx::CullMode_Back, nn::gfx::FrontFace_Cw, nn::gfx::LogicOperation_Copy );
    DEMOTestIsUseHlslccGlsl()
        ? SetDefaultPipeline( &terrainPipeline, true, nn::gfx::CullMode_Back, nn::gfx::FrontFace_Ccw, nn::gfx::LogicOperation_Copy )
        : SetDefaultPipeline( &terrainPipeline, true, nn::gfx::CullMode_Back, nn::gfx::FrontFace_Cw, nn::gfx::LogicOperation_Copy );

    SetDefaultPipeline( &terrainWiredPipeline, true, nn::gfx::CullMode_None, nn::gfx::FrontFace_Cw, nn::gfx::LogicOperation_Clear );
    SetWiredRasterizer( &terrainWiredRasterizer, nn::gfx::CullMode_None, nn::gfx::FrontFace_Cw, -5, 0.0f, 0.0f ); // This will need some GX2 overrides

    // Uniform Location Lookup
    ub_psLightingSlot = reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_lighting" );
#if NN_GFX_IS_TARGET_GX
    ub_vsLightingSlot = reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_lighting" );
#else
    ub_tcsLightingSlot = reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Hull, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_lighting" );
    ub_tesLightingSlot = reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Domain, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_lighting" );
#endif

    // Texture Sampler lookup
#if NN_GFX_IS_TARGET_GX
    heightmapLocVs = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Sampler, "s_heightmap" );
#else
    heightmapLocVs = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Domain, nn::gfx::ShaderInterfaceType_Sampler, "s_heightmap" );
#endif

    heightmapLocPs = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_heightmap");
    colormapLoc = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture");
    reflectionLoc = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_reflection");
    refractionLoc = (u32)reflectPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_refraction");

    InitCamera(projMtx44, viewMtx44);

    // Init State
    g_State.tessFactor = 7.0f;
#if NN_GFX_IS_TARGET_GX
    g_State.tessMode = GX2_TESSELLATION_MODE_DISCRETE;
#endif
    g_State.drawWire = false;

    g_State.ub_heightBuffer.Initialize( sizeof( UbHeight ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

    UbHeight* pUb = g_State.ub_heightBuffer.Map< UbHeight >();
    pUb->u_shotPower = 0.0f;
    pUb->u_shotCoord[0] = g_State.u_shotCoord[0] = 0.6f;
    pUb->u_shotCoord[1] = g_State.u_shotCoord[1] = 0.7f;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pUb, sizeof(UbHeight) );
#endif
    g_State.ub_heightBuffer.Unmap();

    isTerrain[1] = -1;
}

static void CleanupScene()
{
    // Free up g_State
    g_State.ub_heightBuffer.Finalize();

    // Free viewports
    refmapViewport.Finalize( &DEMODevice );
    surfaceViewport.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( refmapViewportMem );
    DEMOGfxFreeMEM2( surfaceViewportMem );

    // Free buffers
    ub_reflectBuffer.Finalize();
    ub_refractBuffer.Finalize();
    ub_waterBuffer.Finalize();
    ub_terrainBuffer.Finalize();

    refDepthBufferView.Finalize( &DEMODevice );
    refDepthTexture.Finalize( &DEMODevice );

    DEMOGfxFreeMEM2(fsBuf);
    posBuf.Finalize();
    tcBuf.Finalize();
    refDepthTextureMem->Finalize();

    refractPipeline.Finalize( &DEMODevice );
    reflectPipeline.Finalize( &DEMODevice );
    terrainPipeline.Finalize( &DEMODevice );
    terrainWiredPipeline.Finalize( &DEMODevice );

    LegacyGPGPUShallowWater::FreePass(heightPass);
    LegacyGPGPUShallowWater::FreePass(velocityPass);
    LegacyGPGPUShallowWater::FreePass(initHeightPass);
    LegacyGPGPUShallowWater::FreePass(initWaterPass);
    LegacyGPGPUShallowWater::FreePass(initDepthPass);
    LegacyGPGPUShallowWater::FreePass(initVelocityPass);
    LegacyGPGPUShallowWater::FreePass(advectPass);
    LegacyGPGPUShallowWater::FreeTexture2D(initHeightMap);
    LegacyGPGPUShallowWater::FreeTexture2D(initDepthMap);
    LegacyGPGPUShallowWater::FreeTexture2D(initVelocityMap);
    LegacyGPGPUShallowWater::FreeTexture2D(initWaterMap);
    for(u32 i=0; i<2; ++i)
    {
        LegacyGPGPUShallowWater::FreeRenderTarget(velocityMap[i]);
        LegacyGPGPUShallowWater::FreeRenderTarget(waterMap[i]);
    }
    LegacyGPGPUShallowWater::FreeRenderTarget(depthMap);
    LegacyGPGPUShallowWater::FreeRenderTarget(heightMap);
    LegacyGPGPUShallowWater::FreeRenderTarget(reflectionMap);
    LegacyGPGPUShallowWater::FreeRenderTarget(refractionMap);
    LegacyGPGPUShallowWater::Free();

    terrainWiredRasterizer.Finalize( &DEMODevice );

}

static void RunSimulationShader()
{
    // Spark Instrumentation
    DEMOGfxDebugTagIndent( "RunSimulationShader" );

    for(u32 i=0; i<2; ++i)
    {
        {
            LegacyGPGPUShallowWater::SetInputRenderTarget(advectPass, velocityMap[0], "s_velocitymap");
            LegacyGPGPUShallowWater::SetInputRenderTarget(advectPass, waterMap[0], "s_watermap");
            LegacyGPGPUShallowWater::SetInputRenderTarget(advectPass, depthMap, "s_depthmap");
            LegacyGPGPUShallowWater::RenderTarget* rt[] = { &velocityMap[1], &waterMap[1] };
            LegacyGPGPUShallowWater::ComputePass(advectPass, rt, 2);
#if NN_GFX_IS_TARGET_D3D
            LegacyGPGPUShallowWater::ResetRenderTarget();
#endif
        }
        {
            LegacyGPGPUShallowWater::SetInputRenderTarget(heightPass, velocityMap[1], "s_velocitymap");
            LegacyGPGPUShallowWater::SetInputRenderTarget(heightPass, waterMap[1], "s_watermap");
            LegacyGPGPUShallowWater::SetInputRenderTarget(heightPass, depthMap, "s_depthmap");
            LegacyGPGPUShallowWater::SetInputUniform( heightPass, g_State.ub_heightBuffer, "ub_height" );

            LegacyGPGPUShallowWater::RenderTarget* rt[] = { &heightMap, &waterMap[0] };
            LegacyGPGPUShallowWater::ComputePass(heightPass, rt, 2);
#if NN_GFX_IS_TARGET_D3D
            LegacyGPGPUShallowWater::ResetRenderTarget();
#endif
        }
        {
            LegacyGPGPUShallowWater::SetInputRenderTarget(velocityPass, velocityMap[1], "s_velocitymap");
            LegacyGPGPUShallowWater::SetInputRenderTarget(velocityPass, heightMap, "s_heightmap");
            LegacyGPGPUShallowWater::SetInputRenderTarget(velocityPass, depthMap, "s_depthmap");
            LegacyGPGPUShallowWater::RenderTarget* rt[] = { &velocityMap[0] };
            LegacyGPGPUShallowWater::ComputePass(velocityPass, rt, 1);
#if NN_GFX_IS_TARGET_D3D
            LegacyGPGPUShallowWater::ResetRenderTarget();
#endif
        }
    }

    // Spark Instrumentation
    DEMOGfxDebugTagUndent();
}

static void InitSimulationShader()
{
    for(u32 i=0; i<2; ++i)
    {
        LegacyGPGPUShallowWater::SetInputTexture2D(initWaterPass, initWaterMap, "s_sourceTexture");
        LegacyGPGPUShallowWater::RenderTarget* rt[] = { &waterMap[i] };
        LegacyGPGPUShallowWater::ComputePass(initWaterPass, rt, 1);
    }
    {
        LegacyGPGPUShallowWater::SetInputTexture2D(initHeightPass, initHeightMap, "s_sourceTexture");
        LegacyGPGPUShallowWater::RenderTarget* rt[] = { &heightMap };
        LegacyGPGPUShallowWater::ComputePass(initHeightPass, rt, 1);
    }
    for(u32 i=0; i<2; ++i)
    {
        LegacyGPGPUShallowWater::SetInputTexture2D(initVelocityPass, initVelocityMap, "s_sourceTexture");
        LegacyGPGPUShallowWater::RenderTarget* rt[] = { &velocityMap[i] };
        LegacyGPGPUShallowWater::ComputePass(initVelocityPass, rt, 1);
    }
    {
        LegacyGPGPUShallowWater::SetInputTextureSampler(initDepthPass, heightMapTex.GetDescriptorSlot(0), mySamplerSlot, "s_sourceTexture");
        LegacyGPGPUShallowWater::RenderTarget* rt[] = { &depthMap };
        LegacyGPGPUShallowWater::ComputePass(initDepthPass, rt, 1);
    }
}

static void SetClipPlane(const f32 clipPlane[4])
{
    InitCamera(projMtx44, viewMtx44);

    if(clipPlane == NULL)
    {
        return;
    }

    // Transform the clipping plane into camera space
    Vec vec, tempVec;
    tempVec.x = clipPlane[0];
    tempVec.y = clipPlane[1];
    tempVec.z = clipPlane[2];
    Vec onThePlane, tempOnThePlane;
    tempOnThePlane.x = 0.0f;
    tempOnThePlane.y = clipPlane[3];
    tempOnThePlane.z = 0.0f;
    MTX44MultVecSR(viewMtx44, &tempVec, &vec);
    MTX44MultVec(viewMtx44, &tempOnThePlane, &onThePlane);

    f32 cameraClipPlane[4];
    cameraClipPlane[0] = vec.x;
    cameraClipPlane[1] = vec.y;
    cameraClipPlane[2] = vec.z;
    cameraClipPlane[3] = -VECDotProduct(&vec, &onThePlane);

    if(cameraClipPlane[3] > 0.0f)
    {
        return;
    }

    // Oblique Near-Plane Clipping
    f32    cornerPoint[4];
    f32    sign[2];

    sign[0] = (f32)((cameraClipPlane[0] > 0) - (cameraClipPlane[0] < 0));
    sign[1] = (f32)((cameraClipPlane[1] > 0) - (cameraClipPlane[1] < 0));
    cornerPoint[0] = (sign[0] + projMtx44[0][2]) / projMtx44[0][0];
    cornerPoint[1] = (sign[1] + projMtx44[1][2]) / projMtx44[1][1];
    cornerPoint[2] = -1.0f;
    cornerPoint[3] = (1.0f + projMtx44[2][2]) / projMtx44[2][3];

    f32 scaledPlane[4];
    f32 dot = 0.0f;
    for(u32 i=0; i<4; ++i)
    {
        dot += cameraClipPlane[i] * cornerPoint[i];
    }
    for(u32 i=0; i<4; ++i)
    {
        scaledPlane[i] = cameraClipPlane[i] * (2.0f / dot);
    }

    projMtx44[2][0] = scaledPlane[0];
    projMtx44[2][1] = scaledPlane[1];
    projMtx44[2][2] = scaledPlane[2] + 1.0f;
    projMtx44[2][3] = scaledPlane[3];
}

static void MaybeSetConstantBuffer( nn::gfx::CommandBuffer* cmd, int loc, nn::gfx::ShaderStage stage, DEMOGfxBuffer* bufView )
{
    if (loc >= 0)
    {
        cmd->InvalidateMemory( nn::gfx::GpuAccess_ConstantBuffer );
        cmd->SetConstantBuffer( loc, stage, bufView->gpuAddress, bufView->size );
    }
}
static void MaybeSetTextureAndSampler( nn::gfx::CommandBuffer* cmd, int loc, nn::gfx::ShaderStage stage, nn::gfx::DescriptorSlot& view, nn::gfx::DescriptorSlot& sampler )
{
    if (loc >= 0)
    {
        cmd->SetTextureAndSampler( loc, stage, view, sampler );
    }
}

static void SetLightingUniforms( nn::gfx::CommandBuffer* cmd, DEMOGfxBuffer* buffer, Mtx44* proj, Mtx44* model, Mtx44* view )
{
    UbLighting* pUb;

    pUb = buffer->Map< UbLighting >();
    memcpy( pUb->u_ProjMtx, proj, sizeof(Mtx44) );
    memcpy( pUb->u_ModelMtx, model, sizeof(Mtx44) );
    memcpy( pUb->u_ViewMtx, view, sizeof(Mtx44) );
    memcpy( pUb->isTerrain, isTerrain, 4 * sizeof(f32) );
    pUb->u_tessLevel = g_State.tessFactor;

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pUb, sizeof(*pUb) );
#endif
    buffer->Unmap();
    cmd->InvalidateMemory( nn::gfx::GpuAccess_ConstantBuffer );

#if NN_GFX_IS_TARGET_GX
    MaybeSetConstantBuffer( cmd, ub_vsLightingSlot, nn::gfx::ShaderStage_Vertex, buffer );
#else
    MaybeSetConstantBuffer( cmd, ub_tcsLightingSlot, nn::gfx::ShaderStage_Hull, buffer );
    MaybeSetConstantBuffer( cmd, ub_tesLightingSlot, nn::gfx::ShaderStage_Domain, buffer );
#endif
    MaybeSetConstantBuffer( cmd, ub_psLightingSlot, nn::gfx::ShaderStage_Pixel, buffer );
}

static void DrawTessellated()
{
#if NN_GFX_IS_TARGET_GX
    // Draw with triangle
    GX2DrawEx(GX2_PRIMITIVE_TESSELLATE_TRIANGLES, TRIANGLE_VERTEX_NUM, 0, 1);
#else
    DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_PatchList, TRIANGLE_VERTEX_NUM, 0 );
#endif
}

static void DrawReflectionMap()
{
    nn::gfx::CommandBuffer* cmd = &DEMOCommandBuffer;

    // Spark Instrumentation
    DEMOGfxDebugTagIndent( "DrawReflectionMap" );

    // clear
    cmd->ClearColor( &reflectionMap.colorBuffer, 0.2f, 0.65f, 0.9f, 1.0f, NULL );
    cmd->ClearDepthStencil( &refDepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    cmd->SetPipeline( &reflectPipeline.pipeline );
    cmd->InvalidateMemory( nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_Texture );

#if NN_GFX_IS_TARGET_GX
    // Need a special tessellation fetch shader
    GX2SetFetchShader( &fetchShader );
#endif

    nn::gfx::ColorTargetView* rt[1] = { &reflectionMap.colorBuffer };
    cmd->SetRenderTargets( 1, rt, &refDepthBufferView );

    cmd->SetViewportScissorState( &refmapViewport );

#if NN_GFX_IS_TARGET_GX
    GX2SetMaxTessellationLevel(1.0f);
#endif

    f32 clipPlane[4] = {0.0f, -1.0f, 0.0f, 0.0f};
    SetClipPlane(clipPlane);

    Mtx44 terrainModelMtx;
    MTX44Copy(modelMtx44, terrainModelMtx);
    terrainModelMtx[1][1] *= -1.0f;

    isTerrain[0] = 1.0f;

#if NN_GFX_IS_TARGET_GX
    MaybeSetTextureAndSampler( cmd, heightmapLocVs, nn::gfx::ShaderStage_Vertex, heightMapTex.GetDescriptorSlot(0), mySamplerSlot);
#else
    MaybeSetTextureAndSampler( cmd, heightmapLocVs, nn::gfx::ShaderStage_Domain, heightMapTex.GetDescriptorSlot(0), mySamplerSlot);
#endif
    MaybeSetTextureAndSampler( cmd, heightmapLocPs, nn::gfx::ShaderStage_Pixel, heightMapTex.GetDescriptorSlot(0), mySamplerSlot);
    MaybeSetTextureAndSampler( cmd, colormapLoc, nn::gfx::ShaderStage_Pixel, colorTex.GetDescriptorSlot( 0 ), mySamplerSlot );

    // set up uniform buffer for this iteration
    SetLightingUniforms( cmd, &ub_reflectBuffer, &projMtx44, &terrainModelMtx, &viewMtx44 );

#if NN_GFX_IS_TARGET_GX
    // Set tessellation mode
    GX2SetTessellation((GX2TessellationMode)g_State.tessMode,
                       GX2_PRIMITIVE_TESSELLATE_TRIANGLES,
                       GX2_INDEX_FORMAT_U32);  // Must be set to 32 for non-indexed triangles
#endif
    DrawTessellated();

    LegacyGPGPUShallowWater::InvalidateRenderTarget(cmd, reflectionMap);

    // Spark Instrumentation
    DEMOGfxDebugTagUndent();
}

static void DrawRefractionMap()
{
    nn::gfx::CommandBuffer* cmd = &DEMOCommandBuffer;

    // Spark Instrumentation
    DEMOGfxDebugTagIndent( "DrawRefractionMap" );

    // clear
    cmd->ClearColor( &refractionMap.colorBuffer, 0.2f, 0.65f, 0.9f, 1.0f, NULL );
    cmd->ClearDepthStencil( &refDepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    cmd->SetPipeline( &refractPipeline.pipeline );
    cmd->InvalidateMemory( nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_Texture );

#if NN_GFX_IS_TARGET_GX
    // Need a special tessellation fetch shader
    GX2SetFetchShader( &fetchShader );
#endif

    nn::gfx::ColorTargetView* rt[1] = { &refractionMap.colorBuffer };
    cmd->SetRenderTargets( 1, rt, &refDepthBufferView );
    cmd->SetViewportScissorState( &refmapViewport );

#if NN_GFX_IS_TARGET_GX
    // NOTE: tessellation must be done with direct GX2 calls
    GX2SetMaxTessellationLevel(1.0f);
#endif

    f32 clipPlane[4] = {0.0f, -1.0f, 0.0f, 7.0f};
    SetClipPlane(clipPlane);

    Mtx44 terrainModelMtx;
    MTX44Copy(modelMtx44, terrainModelMtx);
    terrainModelMtx[1][1] *= 1.0f;

    isTerrain[0] = 1.0f;

#if NN_GFX_IS_TARGET_GX
    MaybeSetTextureAndSampler( cmd, heightmapLocVs, nn::gfx::ShaderStage_Vertex, heightMapTex.GetDescriptorSlot( 0 ), mySamplerSlot);
#else
    MaybeSetTextureAndSampler( cmd, heightmapLocVs, nn::gfx::ShaderStage_Domain, heightMapTex.GetDescriptorSlot( 0 ), mySamplerSlot);
#endif
    MaybeSetTextureAndSampler( cmd, heightmapLocPs, nn::gfx::ShaderStage_Pixel, heightMapTex.GetDescriptorSlot( 0 ), mySamplerSlot);
    MaybeSetTextureAndSampler( cmd, colormapLoc, nn::gfx::ShaderStage_Pixel, colorTex.GetDescriptorSlot( 0 ), mySamplerSlot );

    // set up uniform buffer for this iteration
    SetLightingUniforms( cmd, &ub_refractBuffer, &projMtx44, &terrainModelMtx, &viewMtx44 );

#if NN_GFX_IS_TARGET_GX
    // Set tessellation mode
    GX2SetTessellation((GX2TessellationMode)g_State.tessMode,
                       GX2_PRIMITIVE_TESSELLATE_TRIANGLES,
                       GX2_INDEX_FORMAT_U32);  // Must be set to 32 for non-indexed triangles
#endif
    DrawTessellated();

    LegacyGPGPUShallowWater::InvalidateRenderTarget(cmd, refractionMap);

    // Spark Instrumentation
    DEMOGfxDebugTagUndent();
}

static void DrawPass()
{
    nn::gfx::CommandBuffer* cmd = &DEMOCommandBuffer;

    // Spark Instrumentation
    DEMOGfxDebugTagIndent( "DrawPass" );

    // Clear and set buffers

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    cmd->ClearColor( pCurrentScanBuffer, 0.2f, 0.65f, 0.9f, 1.0f, NULL );
    cmd->ClearDepthStencil( &DEMODepthBufferView, 1.0, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    nn::gfx::ColorTargetView *rt[1] = { pCurrentScanBuffer };
    cmd->SetRenderTargets( 1, rt, &DEMODepthBufferView );

    cmd->SetViewportScissorState( &surfaceViewport );

    cmd->SetPipeline( &terrainPipeline.pipeline );

#if NN_GFX_IS_TARGET_GX
    // Need a special tessellation fetch shader
    GX2SetFetchShader( &fetchShader );
#endif

    // Bind position buffer
    cmd->SetVertexBuffer( POSITION_BUFFER_IDX, posBuf.gpuAddress, POSITION_STRIDE, posBuf.size );

    // Bind texcoord buffer
    cmd->SetVertexBuffer( TEXCOORD_BUFFER_IDX, tcBuf.gpuAddress, TEXCOORD_STRIDE, tcBuf.size );

    MTX44Identity(modelMtx44);

#if NN_GFX_IS_TARGET_GX
    GX2SetMinTessellationLevel(1.0f);
#endif
    nn::gfx::TessellationState::InfoType tessInfo;
    tessInfo.SetDefault();
    tessInfo.SetPatchControlPointCount( 3 );

    // Reflection & Refraction
    if(isTerrain[1] < 0.0f)
    {
        DrawReflectionMap();

        DrawRefractionMap();

        const nn::gfx::ColorTargetView* const colorTargets[] = { pCurrentScanBuffer };
        cmd->SetRenderTargets( 1, colorTargets, &DEMODepthBufferView );

        cmd->SetViewportScissorState( &surfaceViewport );
    }

    // Draw Water
    DEMOGfxDebugTagIndent( "Water" );

    // Restore the pipeline since Reflect/Refract have different pipelines
    cmd->SetPipeline( &terrainPipeline.pipeline );

#if NN_GFX_IS_TARGET_GX
    // Need a special tessellation fetch shader
    GX2SetFetchShader( &fetchShader );
#endif

    SetClipPlane(NULL);

    isTerrain[0] = -1.0f;

#if NN_GFX_IS_TARGET_GX
    MaybeSetTextureAndSampler( cmd, heightmapLocVs, nn::gfx::ShaderStage_Vertex, heightMap.textureSlot, mySamplerSlot);
#else
    MaybeSetTextureAndSampler( cmd, heightmapLocVs, nn::gfx::ShaderStage_Domain, heightMap.textureSlot, mySamplerSlot);
#endif
    MaybeSetTextureAndSampler( cmd, heightmapLocPs, nn::gfx::ShaderStage_Pixel, heightMap.textureSlot, mySamplerSlot);
    MaybeSetTextureAndSampler( cmd, reflectionLoc, nn::gfx::ShaderStage_Pixel, reflectionMap.textureSlot, mySamplerSlot );
    MaybeSetTextureAndSampler( cmd, refractionLoc, nn::gfx::ShaderStage_Pixel, refractionMap.textureSlot, mySamplerSlot );

    // set up uniform buffer for this iteration
    SetLightingUniforms( cmd, &ub_waterBuffer, &projMtx44, &modelMtx44, &viewMtx44 );

#if NN_GFX_IS_TARGET_GX
    // Set tessellation mode
    GX2SetTessellation((GX2TessellationMode)g_State.tessMode,
                       GX2_PRIMITIVE_TESSELLATE_TRIANGLES,
                       GX2_INDEX_FORMAT_U32);  // Must be set to 32 for non-indexed triangles

    // Set tessellation Factor
    GX2SetMaxTessellationLevel(g_State.tessFactor);
#endif

    // Wire Frame
    if(g_State.drawWire)
    {
        cmd->SetPipeline( &terrainWiredPipeline.pipeline );

#if NN_GFX_IS_TARGET_GX
        // Need a special tessellation fetch shader
        GX2SetFetchShader( &fetchShader );
#endif

        // Override the defaults
        cmd->SetRasterizerState( &terrainWiredRasterizer );

        // GX2 Allows for back/front polygon offsets as well as other fine grained
        // controls

#if NN_GFX_IS_TARGET_GX
        // Set Polygon Control Register
        GX2SetPolygonControl(GX2_FRONT_FACE_CW,       // frontFaceMode
            GX2_DISABLE,              // cullFront
            GX2_DISABLE,              // cullBack
            GX2_ENABLE,               // enablePolygonModes
            GX2_POLYGON_MODE_LINE,    // polygonModeFront
            GX2_POLYGON_MODE_LINE,    // polygonModeBack
            GX2_ENABLE,               // polyOffsetFrontEnable
            GX2_ENABLE,               // polyOffsetBackEnable
            GX2_DISABLE);             // pointLineOffsetEnable

        // Set Polygon Offset
        GX2SetPolygonOffset(-5.0f, 0.0f, 5.0f, 0.0f, 0.0f);
#endif
        DrawTessellated();

        // Restore the normal terrain pipeline
        cmd->SetPipeline( &terrainPipeline.pipeline );

#if NN_GFX_IS_TARGET_GX
        // Need a special tessellation fetch shader
        GX2SetFetchShader( &fetchShader );
#endif
    }

    DrawTessellated();

    DEMOGfxDebugTagUndent();

    // Draw Terrain
    DEMOGfxDebugTagIndent( "Terrain" );
    Mtx44 terrainModelMtx;
    MTX44Copy(modelMtx44, terrainModelMtx);

    isTerrain[0] = 1.0f;

    // set up uniform buffer for this iteration
    SetLightingUniforms( cmd, &ub_terrainBuffer, &projMtx44, &terrainModelMtx, &viewMtx44 );

#if NN_GFX_IS_TARGET_GX
    MaybeSetTextureAndSampler( cmd, heightmapLocVs, nn::gfx::ShaderStage_Vertex, heightMapTex.GetDescriptorSlot( 0 ), mySamplerSlot);
#else
    MaybeSetTextureAndSampler( cmd, heightmapLocVs, nn::gfx::ShaderStage_Domain, heightMapTex.GetDescriptorSlot( 0 ), mySamplerSlot);
#endif
    MaybeSetTextureAndSampler( cmd, heightmapLocPs, nn::gfx::ShaderStage_Pixel, heightMapTex.GetDescriptorSlot( 0 ), mySamplerSlot);
    MaybeSetTextureAndSampler( cmd, colormapLoc, nn::gfx::ShaderStage_Pixel, colorTex.GetDescriptorSlot( 0 ), mySamplerSlot );

#if NN_GFX_IS_TARGET_GX
    GX2SetMaxTessellationLevel(1.0f);
#endif
    DrawTessellated();

    DEMOGfxDebugTagUndent();

    // Spark Instrumentation
    DEMOGfxDebugTagUndent();
}

// The draw function for the rendering portions of this app
static DEMOGfxGpuTimestamp s_startRender;
static DEMOGfxGpuTimestamp s_markRender;
static DEMOGfxGpuTimestamp s_endRender;
static uint64_t s_startGpuTick;
static uint64_t s_markGpuTick;
static uint64_t s_endGpuTick;

static void UpdateTimestamps()
{
    // The command buffer must be recording for GetTimestampResult
    DEMOCommandBuffer.Begin();
    s_startGpuTick = s_startRender.GetTimestampResult();
    s_markGpuTick = s_markRender.GetTimestampResult();
    s_endGpuTick = s_endRender.GetTimestampResult();
    // GetTimestampResult leaves the command buffer recording
    DEMOCommandBuffer.End();
}

static void DrawScene()
{

    f32 dtime = OSTicksToMilliseconds(OSGetTime() - lastTime) / 1000.0f;
    lastTime = OSGetTime();
#if 0
    f32 dtime = DEMOGfxGpuTimestamp::TicksToMicroseconds(s_endRenderTicks - s_startRenderTicks) / 1000000.0;
#endif
    DEMOGfxBeforeRender();

    s_startRender.QueryTimestamp();
    if( s_FrameCount == 0 )
    {
        InitSimulationShader();
    }
    else
    {
        RunSimulationShader();
    }
    s_markRender.QueryTimestamp();
    DrawPass();
    s_endRender.QueryTimestamp();

    // Spark Instrumentation
    DEMOGfxDebugTagIndent( "PrintInfo" );

    if ( s_FrameCount != 0 )
    {
        double simTime, renderTime;
        simTime = DEMOGfxGpuTimestamp::TicksToMicroseconds(s_markGpuTick - s_startGpuTick);
        renderTime = DEMOGfxGpuTimestamp::TicksToMicroseconds(s_endGpuTick - s_markGpuTick);
        DEMOFontPrintf( 5, 2, "Shallow Water Simulation : %u x %u grid  %.1f fps\n",
            DOMAIN_SIZE, DOMAIN_SIZE, 1.0f / std::max( 1.0e-7f, dtime ));
        DEMOFontPrintf(5, 3, "Simulation GPU Time : %0.f usec\n", simTime);
        DEMOFontPrintf(5, 4, "Rendering GPU Time: %0.f usec\n", renderTime);
        DEMOFontPrintf( 5, 6, "Tessellation Factor (up/down) : %.1f\n", g_State.tessFactor );
#if NN_GFX_IS_TARGET_GX
        if ( g_State.tessMode == GX2_TESSELLATION_MODE_DISCRETE )
        {
            DEMOFontPrintf( 5, 7, "Tessellation Type (a) : DISCRETE\n" );
        }
        else
        {
            DEMOFontPrintf( 5, 7, "Tessellation Type (a) : CONTINUOUS\n" );
        }
#endif
        DEMOFontPrintf( 5, 8, "Effector Power (y) : %.2f\n", g_State.u_shotPower );
        DEMOFontPrintf( 5, 9, "Effector Position (L Stick) : u %.2f v %.2f\n",
            g_State.u_shotCoord[ 0 ], g_State.u_shotCoord[ 1 ] );
        DEMOFontPrintf( 5, 10, "Move Camera (L Trigger + L Stick/R Stick)" );
        if ( isTerrain[ 1 ] < 0.0f )
        {
            DEMOFontPrintf( 5, 11, "Reflection / Refraction (b) : ON\n" );
        }
        else
        {
            DEMOFontPrintf( 5, 11, "Reflection / Refraction (b) : OFF\n" );
        }
        if ( g_State.drawWire )
        {
            DEMOFontPrintf( 5, 12, "Wire Frame (x) : ON\n" );
        }
        else
        {
            DEMOFontPrintf( 5, 12, "Wire Frame (x) : OFF\n" );
        }
    }

    // Spark Instrumentation
    DEMOGfxDebugTagUndent();

    DEMOGfxDoneRender();

    // Update the elapsed time and print next frame
    UpdateTimestamps();
}

static void ProcessModifyReflectRefract()
{
    if ( isTerrain[ 1 ] < 0.0f )
    {
        isTerrain[1] = 1.0f;
    }
    else
    {
        isTerrain[1] = -1.0f;
    }
}

static void RotateCamera(f32 sx, f32 sy)
{
    Vec eyev, tempEyev;
    VECSubtract(&objPt, &camLoc, &eyev);
    Vec wupv =  {0.0f, 1.0f, 0.0f};
    VECCrossProduct(&eyev, &up, &tempEyev);
    VECNormalize(&tempEyev, &eyev);

    Mtx44 rot,rot0,rot1;
    MTX44RotAxisRad(rot0, &eyev, MTXDegToRad( sy * 5.0f));
    MTX44RotAxisRad(rot1, &wupv, MTXDegToRad(-sx * 5.0f));

    MTX44Concat(rot0, rot1, rot);

    Vec camv, tempCamv;
    VECSubtract(&objPt, &camLoc, &tempCamv);
    MTX44MultVecSR(rot, &tempCamv, &camv);

    VECAdd(&camv, &camLoc, &objPt);
    Vec tempUp = up;
    MTX44MultVecSR(rot, &tempUp, &up);

    InitCamera(projMtx44, viewMtx44);
}

static void TranslateCamera( f32 sx, f32 sy )
{
    Vec vec, tempVec;
    tempVec.x =  sx * 10.0f;
    tempVec.y = 0.0f;
    tempVec.z = -sy * 10.0f;

    Mtx44 inv;
    MTX44Inverse(viewMtx44, inv);
    MTX44MultVecSR(inv, &tempVec, &vec);
    Vec tempCamLoc = camLoc;
    Vec tempObjPt = objPt;
    VECAdd(&vec, &tempCamLoc, &camLoc);
    VECAdd(&vec, &tempObjPt, &objPt);

    InitCamera(projMtx44, viewMtx44);
}

static void ProcessPad()
{
    DEMOPadRead();
    u16 button = DEMOPadGetButton(0);
    u16 buttonDown = DEMOPadGetButtonDown(0);
    u16 buttonUp = DEMOPadGetButtonUp(0);

    if(DEMO_PAD_BUTTON_B & buttonUp)
    {
        // Reflection / Refraction
        ProcessModifyReflectRefract();
    }

    UbHeight* pUb = g_State.ub_heightBuffer.Map< UbHeight >();
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pUb, sizeof(UbHeight) );
#endif

    // Effector Position
    f32 effectorTrans = 0.0f;
    {
        f32 sx = (f32)DEMOPadGetStickX(0) / 255.0f;
        f32 sy = (f32)DEMOPadGetStickY(0) / 255.0f;
        effectorTrans = fabs(sx) + fabs(sy);
        Vec vec, tempVec;
        tempVec.x =  sx * 0.05f;
        tempVec.y = 0.0f;
        tempVec.z = -sy * 0.05f;
        Mtx44 inv;
        MTX44Inverse(viewMtx44, inv);
        MTX44MultVecSR(inv, &tempVec, &vec);

#if !( NN_GFX_IS_TARGET_GX || NN_GFX_IS_TARGET_D3D )
        // Flip the z for GL style texture coordinates
        vec.x = -vec.x;
#endif
        g_State.u_shotCoord[ 0 ] = pUb->u_shotCoord[ 0 ] = std::min( std::max( pUb->u_shotCoord[ 0 ] + vec.x, 0.0f ), 1.0f );
        g_State.u_shotCoord[ 1 ] = pUb->u_shotCoord[ 1 ] = std::min( std::max( pUb->u_shotCoord[ 1 ] + vec.z, 0.0f ), 1.0f );
    }

    g_State.u_shotPower = pUb->u_shotPower = 0.0f;
    if(DEMO_PAD_BUTTON_Y & buttonUp)
    {
        // Effector
        g_State.u_shotPower = pUb->u_shotPower = 0.1f * (1.0f / 3.0f);
    }
    if(DEMO_PAD_BUTTON_Y & button && effectorTrans > 0.05f)
    {
        // Effector
        g_State.u_shotPower = pUb->u_shotPower = 0.1f * (1.0f / 3.0f);
    }
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pUb, sizeof(UbHeight) );
#endif
    g_State.ub_heightBuffer.Unmap();

    if(DEMO_PAD_BUTTON_UP & button)
    {
        // Increase Tessellation Factor
        g_State.tessFactor = std::min( std::max( g_State.tessFactor + 0.1f, 1.0f ), 14.0f );
    }

    if(DEMO_PAD_BUTTON_DOWN & button)
    {
        // Decrease Tessellation Factor
        g_State.tessFactor = std::min( std::max( g_State.tessFactor - 0.1f, 1.0f ), 14.0f );
    }

    if(DEMO_PAD_BUTTON_X & buttonUp)
    {
        // Wire Frame
        g_State.drawWire = !g_State.drawWire;
    }

    if(DEMO_PAD_BUTTON_A & buttonUp)
    {
#if NN_GFX_IS_TARGET_GX
        // Change Tessellation Type
        ++g_State.tessMode;
        if(g_State.tessMode > GX2_TESSELLATION_MODE_CONTINUOUS)
        {
            g_State.tessMode = GX2_TESSELLATION_MODE_DISCRETE;
        }
#endif
    }

    if(DEMO_PAD_TRIGGER_L & button)
    {
        // L Stick translates the camera
        f32 sx = (f32)DEMOPadGetStickX(0) / 255.0f;
        f32 sy = (f32)DEMOPadGetStickY(0) / 255.0f;
        TranslateCamera( sx, sy );
    }

    {
        // R Stick rotates the camera
        f32 sx = (f32)DEMOPadGetSubStickX(0) / 255.0f;
        f32 sy = (f32)DEMOPadGetSubStickY(0) / 255.0f;

        RotateCamera(sx, sy);
    }

    if(DEMO_PAD_BUTTON_LEFT & buttonDown)
    {
        // Increase Tessellation Factor
    }

    if(DEMO_PAD_BUTTON_RIGHT & buttonDown)
    {
    }
}

//extern "C" void nnMain()
TEST(GfxShallowWater, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

#if NN_GFX_IS_TARGET_GX
    extern BOOL gDemoGfxForceMEM1;
    gDemoGfxForceMEM1 = TRUE;
#endif
    s_FrameCount=0;

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    s_startRender.Initialize();
    s_markRender.Initialize();
    s_endRender.Initialize();

    // Texture setup
    InitTex();
#if NN_GFX_IS_TARGET_GX
    gDemoGfxForceMEM1 = FALSE;
#endif
    InitScene();

    while (DEMOIsRunning())
    {
        ProcessPad();

        DrawScene();
        s_FrameCount++;
    }

    CleanupTex();

    CleanupScene();

    s_startRender.Finalize();
    s_markRender.Finalize();
    s_endRender.Finalize();

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
