﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/////////////////////////////////////////////////////////////////////////////
//
// Draws many textured quads which are loaded from multi texture gtx miped files.
//rm
//////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <cstring>

#include <gfx/demo.h>
#include <nnt.h>
#include <nnt/nnt_Argument.h>

#define countof(buf) ( sizeof( buf ) / sizeof( buf[0] ) )

// ----- Revert frame if true
#define REVERSE TRUE

// ----- Input textures
static const char *inputTextures[] =
{
    "textures/mipmap/frame1_multi_mip",
    "textures/mipmap/frame2_multi_mip",
};
static const int NUM_TEXTURE_FILES = countof( inputTextures );

static const int NUM_TEXTURE = 30;           // frames per sec
static const int FULL_FRAME = ( NUM_TEXTURE_FILES * NUM_TEXTURE );

// ----- Input gsh files
static int g_ShaderFileIdx = 0;
static const char *SHADER_FILE[] =
{
    "shaders/simple/texture2DLOD",
    "shaders/simple/texture2DLODHlslcc",
};

DEMOGfxPipeline pipeline;

// ----- Shader Info
static int levelLoc;

// ----- Data

#define WUNIT 1.0f / 8.0f
#define HUNIT 1.0f / 8.0f

static f32 posDataGX2[] = {
    -WUNIT, HUNIT, 0.0f,
    WUNIT, HUNIT, 0.0f,
    -WUNIT, -HUNIT, 0.0f,
    WUNIT, -HUNIT, 0.0f
};

static f32 tcDataGX2[] = {
    0.0f, 0.0f,
    1.0f, 0.0f,
    0.0f, 1.0f,
    1.0f, 1.0f
};

static const int NUM_VERTEX = 4;
static const int POSITION_STRIDE = ( sizeof( f32 ) * 3 );
static const int TEXCOORD_STRIDE = ( sizeof( f32 ) * 2 );
static const int OFFSET_STRIDE = ( sizeof( f32 ) * 4 );

static const int POSITION_BUFFER_IDX = 0;
static const int TEXCOORD_BUFFER_IDX = 1;
static const int OFFSET_BUFFER_IDX = 2;

// ----- Attribute buffers
static DEMOGfxBuffer positionBuffer;
static DEMOGfxBuffer textureCoordinateBuffer;
static DEMOGfxBuffer offsetBuffer[2];

// ----- Uniform buffers
static DEMOGfxBuffer levelBuffer[ 2 ];

// ----- Texture & sampler objects
static const int NUM_SAMPLER = 1;

static DEMOGfxTexture textureData[ FULL_FRAME ];
static int samplerLoc[ NUM_SAMPLER ];
static nn::gfx::Sampler mySampler[ NUM_SAMPLER ];
static nn::gfx::DescriptorSlot mySamplerSlot[ NUM_SAMPLER ];

static int frame = 0;
static size_t totalSize = 0;
static f32 val = 0;

// Prototype
static BOOL SceneInit();
static BOOL SceneDraw();

// Print Init Informations
static void PrintInitInfo()
{
    for ( int i = 0; i < 2; i++ )
    {
        DEMOGfxBeforeRender();

        nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
        DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, 0.0f, 0.0f, 0.0f, 1.0f, NULL );
        DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
        DEMOGfxSetDefaultRenderTarget();
        DEMOGfxSetDefaultViewportScissor();

        // Set Demo Font state
        DEMOFontPrintf( 22, 11, "Loading %d textures wait ...", FULL_FRAME );

        DEMOGfxDoneRender();
    }
}

// Print Informations
static void PrintInfo( DEMOGfxTexture* pTextureData )
{
    // Set Demo Font state
    DEMOGfxSetDefaultViewportScissor();

    nn::gfx::TextureInfo* pInfo;
    pInfo = pTextureData->GetTextureInfo( 0 );

    DEMOFontSetSpacing( 0 );
    DEMOFontSetGridSize( 110, 45 );
    DEMOFontPrintf( 69, 0.5f, "Frame : %d", frame );
    DEMOFontPrintf( 83, 0.5f, "MEM2 Usage : %d MB", totalSize / 1000000 );
    DEMOFontPrintf( 69, 1.5f, "size : [ TB: %d CB: %d ]",
        nn::gfx::Texture::CalculateMipDataSize( &DEMODevice, *pInfo ),
        nn::gfx::Texture::CalculateMipDataSize( &DEMODevice, DEMOColorBufferInfo ) );
    DEMOFontPrintf( 69, 2.5f, "TB : [ width: %d height: %d ]",
        pInfo->GetWidth(),
        pInfo->GetHeight() );
    DEMOFontPrintf( 69, 3.5f, "CB : [ width: %d height: %d ]",
        DEMOColorBufferInfo.GetWidth(),
        DEMOColorBufferInfo.GetHeight() );
    DEMOFontPrintf( 69, 4.5f, "[ first mip: %d num mip: %d level: %d]",
        0, pInfo->GetMipCount(),
        ( u32 ) val );
}

// The init function for the rendering portions of this app
static BOOL SceneInit()
{
    u32 texcount = 0;

    frame = 0;
    totalSize = 0;

    OSReport( "Loading Shader Binary\n" );
    DEMOGfxLoadShadersFromFile(&pipeline.shaders, 0, SHADER_FILE[g_ShaderFileIdx]);

    for ( int j = 0; j < NUM_TEXTURE_FILES; j++ )
    {
        OSReport( "Loading Texture Binary %d\n", j );

        for ( int i = 0; i < NUM_TEXTURE; i++ )
        {
            OSReport( "Loading Texture %d ...\n", texcount );
            char filename[ 256 ];

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
            _snprintf( filename, 256, "%s%02d", inputTextures[ j ], j * NUM_TEXTURE + i + 1 );
#else
            snprintf( filename, 256, "%s%02d", inputTextures[ j ], j * NUM_TEXTURE + i + 1 );
#endif
            textureData[ texcount ].Initialize( filename );

            // Keep track of memory used
            totalSize += nn::gfx::Texture::CalculateMipDataSize( &DEMODevice, *textureData[ texcount ].GetTextureInfo( 0 ) );

            texcount++;
        }
    }

    OSReport( "Done.\n" );

    // Uniform Location lookup
    levelLoc = pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_pixel" );

    // Texture Sampler lookup
    samplerLoc[ 0 ] = pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    // Attribute Format setup
    DEMOGfxInitShaderAttribute( &pipeline.shaders, "a_position", POSITION_BUFFER_IDX, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &pipeline.shaders, "a_texcoord", TEXCOORD_BUFFER_IDX, 0, nn::gfx::AttributeFormat_32_32_Float );
    DEMOGfxInitShaderAttribute( &pipeline.shaders, "a_offset", OFFSET_BUFFER_IDX, 0, nn::gfx::AttributeFormat_32_32_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &pipeline.shaders, POSITION_BUFFER_IDX, sizeof( float ) * 3, 0 );
    DEMOGfxInitShaderVertexBuffer( &pipeline.shaders, TEXCOORD_BUFFER_IDX, sizeof( float ) * 2, 0 );
    DEMOGfxInitShaderVertexBuffer( &pipeline.shaders, OFFSET_BUFFER_IDX, sizeof( float ) * 4, 1 ); // Instanced

    // Vertex position buffer
    nn::gfx::Buffer::InfoType bufferInfo;
    bufferInfo.SetDefault();
    bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer );
    bufferInfo.SetSize( sizeof( posDataGX2 ) );
    void* pData = DEMOGfxAllocMEM2( sizeof( posDataGX2 ), nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo ) );
    memcpy( pData, posDataGX2, sizeof( posDataGX2 ) );
    positionBuffer.Initialize( sizeof( posDataGX2 ), pData, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Vertex texcoord buffer
    bufferInfo.SetSize( sizeof( tcDataGX2 ) );
    pData = DEMOGfxAllocMEM2( sizeof( tcDataGX2 ), nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo ) );
    memcpy( pData, tcDataGX2, sizeof( tcDataGX2 ) );
    textureCoordinateBuffer.Initialize( sizeof( tcDataGX2 ), pData, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Offset buffer
    offsetBuffer[ 0 ].Initialize( sizeof( float ) * 4 * ( FULL_FRAME + 1 ), NULL, nn::gfx::GpuAccess_VertexBuffer, 0 );
    offsetBuffer[ 1 ].Initialize( sizeof( float ) * 4 * ( FULL_FRAME + 1 ), NULL, nn::gfx::GpuAccess_VertexBuffer, 0  );

    // Create the level buffer
    levelBuffer[ 0 ].Initialize( sizeof( float ) * 4, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    levelBuffer[ 1 ].Initialize( sizeof( float ) * 4, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

    // Setup the sampler
    DEMOGfxInitSampler( &mySampler[ 0 ], &mySamplerSlot[ 0 ],
        nn::gfx::TextureAddressMode_Repeat,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint,
        nn::gfx::ComparisonFunction_Always );

    // Setup the pipeline
    pipeline.blendTargetStateCount = 1;
    pipeline.colorTargetStateCount = 1;
    pipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    pipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    pipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

    pipeline.Initialize( &DEMODevice );

    return 1;
}

static void UpdateOffsets()
{
    f32 mOffset[ 4 ] = { 0.0f, 0.0f, 0.0f, 0.0f };

    float* pOffset = offsetBuffer[ DEMOTestFrame() % 2 ].Map< float >( );

    // The animated draw happens first
    mOffset[ 0 ] = 2.0f * WUNIT * ( FULL_FRAME % 8 ) - 2.0f * WUNIT * 3.5f;
    mOffset[ 1 ] = 2.0f * HUNIT * ( FULL_FRAME / 8 ) - 2.0f * HUNIT * 3.5f;
    memcpy( pOffset, mOffset, sizeof( mOffset ) );
    pOffset += 4;


    for ( int i = 0; i < FULL_FRAME; ++i )
    {
        mOffset[ 0 ] = 2.0f * WUNIT * ( i % 8 ) - 2.0f * WUNIT * 3.5f;
        mOffset[ 1 ] = 2.0f * HUNIT * ( i / 8 ) - 2.0f * HUNIT * 3.5f;
        memcpy( pOffset, mOffset, sizeof( mOffset ) );
        pOffset += 4;
    }

    offsetBuffer[ DEMOTestFrame() % 2 ].Unmap();
}

static void UpdateLevel()
{
    val += 0.01f;

    if ( val > textureData[ frame ].GetTextureInfo( 0 )->GetMipCount() )
    {
        val = 0;
    }

    f32 level[ 4 ] = { val, 0.0f, 0.0f, 0.0f };

    float* pLevel = levelBuffer[ DEMOTestFrame() % 2 ].Map< float >( );

    memcpy( pLevel, level, sizeof( level ) );

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pLevel, sizeof( level ) );
#endif

    levelBuffer[ DEMOTestFrame() % 2 ].Unmap();
}

// The draw function for the rendering portions of this app
static BOOL SceneDraw()
{
    u32 i;
    static f32 gray = 0.3f;
    static u32 rev = 0;

    // Need explicit reset in case of multiple runs
    if ( DEMOTestFrame() == 1 )
    {
        gray = 0.3f;
        rev = 0;
    }

    DEMOGfxBeforeRender();

    UpdateOffsets();
    UpdateLevel();


    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, gray, gray, gray, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOGfxSetDefaultRenderTarget();
    DEMOGfxSetDefaultViewportScissor();

    DEMOCommandBuffer.SetPipeline( &pipeline.pipeline );

    // Set the vertex buffers
    DEMOCommandBuffer.SetVertexBuffer( POSITION_BUFFER_IDX, positionBuffer.gpuAddress, POSITION_STRIDE, positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( TEXCOORD_BUFFER_IDX, textureCoordinateBuffer.gpuAddress, TEXCOORD_STRIDE, textureCoordinateBuffer.size );

    int testframe = DEMOTestFrame() % 2;

    // Instanced variable to simplify rendering each square
    DEMOCommandBuffer.SetVertexBuffer( OFFSET_BUFFER_IDX, offsetBuffer[testframe].gpuAddress, OFFSET_STRIDE, offsetBuffer[testframe].size );


    // Set the Level Uniform buffer
    DEMOCommandBuffer.SetConstantBuffer( levelLoc, nn::gfx::ShaderStage_Pixel, levelBuffer[ testframe ].gpuAddress, levelBuffer[testframe].size );

    // Draw the animated frame
    DEMOCommandBuffer.SetTextureAndSampler( samplerLoc[ 0 ], nn::gfx::ShaderStage_Pixel, textureData[ frame ].GetDescriptorSlot( 0 ), mySamplerSlot[ 0 ] );

    // Draw
    DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_TriangleStrip, NUM_VERTEX, 0, 1, 0 );

    // Draw all non-animated frames
    for ( i = 0; i < FULL_FRAME; ++i )
    {
        DEMOCommandBuffer.SetTextureAndSampler( samplerLoc[ 0 ], nn::gfx::ShaderStage_Pixel, textureData[ i ].GetDescriptorSlot( 0 ), mySamplerSlot[ 0 ] );

        // Draw
        DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_TriangleStrip, NUM_VERTEX, 0, 1, i + 1 );
    }

    PrintInfo( &textureData[ frame ] );

    DEMOGfxDoneRender();

#ifdef REVERSE
    if ( frame >= FULL_FRAME - 1 )
    {
        rev = 1;
    }

    if ( frame <= 0 )
    {
        rev = 0;
    }

    if ( rev )
    {
        frame--;
    }
    else
    {
        frame++;
    }
#else
    frame++;
    if ( frame >= FULL_FRAME )
    {
        frame = 0;
    }
#endif

    return 1; // 0 makes it exit
}

//extern "C" void nnMain()
TEST(GfxMultiTextureLod, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit( argc, argv );
    DEMOGfxInit( argc, argv );
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;
    PrintInitInfo();
    SceneInit();

    while ( DEMOIsRunning() )
    {
        SceneDraw();
    }

    // Free pipeline
    pipeline.Finalize( &DEMODevice );

    // Free Textures
    for ( int i = 0; i < FULL_FRAME; i++ )
    {
        textureData[ i ].Finalize();
    }

    // Free Sampler
    mySampler[ 0 ].Finalize( &DEMODevice );

    // Free Buffers
    positionBuffer.Finalize();
    textureCoordinateBuffer.Finalize();
    offsetBuffer[0].Finalize();
    offsetBuffer[1].Finalize();
    levelBuffer[0].Finalize();
    levelBuffer[1].Finalize();

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
