﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>
#include <cfloat>

#include <gfx/demo.h>
#include <nn/gfx.h>

#include <nnt.h>
#include <nnt/nnt_Argument.h>

#if NN_GFX_IS_TARGET_GX
#include <cafe/gx2ut.h>
#endif

#ifdef WIN32
#define snprintf _snprintf
#endif


////////////////////////////////////////////////////
//
// Assets data, types and interface for demos
//
////////////////////////////////////////////////////

// ----- Surface Information

static const int RSM_WIDTH = 256;
static const int RSM_HEIGHT = 256;

#define SURFACE_WIDTH DEMOColorBufferInfo.GetWidth()
#define SURFACE_HEIGHT DEMOColorBufferInfo.GetHeight()

#define ILLUMINATION_BUFFER_WIDTH (SURFACE_WIDTH / 4)
#define ILLUMINATION_BUFFER_HEIGHT (SURFACE_HEIGHT / 4)

// A .gsh file is a GX2 specific file that contains vertex and pixel
// shader data. In this case the vertex shader just passes through
// the position and color without modification. and the pixel shader
// outputs the interpolated vertex color.
static int g_ShaderFileIdx = 0;
static const char * const SHADER_FILE_GBUFFER[] =
{
    "shaders/indirectIllumination/gbuffer",
    "shaders/indirectIllumination/gbufferHlslcc",
};
static const char * const SHADER_FILE_RSM[] =
{
    "shaders/indirectIllumination/reflectiveShadowMaps",
    "shaders/indirectIllumination/reflectiveShadowMapsHlslcc",
};
static const char * const SHADER_FILE_IL[] =
{
    "shaders/indirectIllumination/indirectIllumination",
    "shaders/indirectIllumination/indirectIlluminationHlslcc",
};
static const char * const SHADER_FILE_LIGHTING[] =
{
    "shaders/indirectIllumination/lighting",
    "shaders/indirectIllumination/lightingHlslcc",
};
static const char * const SHADER_FILE_VIEWER[] =
{
    "shaders/indirectIllumination/viewer",
    "shaders/indirectIllumination/viewerHlslcc",
};
static const char * const MODEL_FILE_TEMPLE = "geometries/indirectIllumination/temple.nmod";
static const char * const MODEL_FILE_TERRAIN = "geometries/indirectIllumination/ground.nmod";
static const char * const MODEL_TEXTURE_DIRECTORY = "textures/indirectIllumination";

// Model file path

// Texture directory

static const int NUMBER_GBUF = 3;

static const int NUMBER_RSM = 4;
static const float RSM_RANGE = 4000.0f;
static const float RSM_CASTING_RANGE = 4000.0f;

static const int MAX_RENDER_TARGETS = NUMBER_RSM - 1;

static DemoModelData g_modelTemple;
static DemoModelData g_modelTerrain;

static Mtx44   g_viewMtx44;
static Mtx44   g_projMtx44;

struct DrawVariablesGBUF
{
    u32 vertexUniformLoc;

    u32 samplerLoc;
} g_drawVarGBUF;

struct DrawVariablesRSM
{
    u32 vertexUniformLoc;

    u32 samplerLoc;
} g_drawVarRSM;

struct DrawVariablesIL
{
    u32 pixelUniformLoc;

    u32 rsmLoc[NUMBER_RSM];
    u32 gbufLoc[NUMBER_GBUF];
} g_drawVarIL;

struct  DrawVariablesLighting {
    u32 vertexUniformLoc;

    u32 pixelUniformLoc;
    u32 samplerLoc;
    u32 gbufLoc[NUMBER_GBUF];
    u32 illuminationLoc;
} g_drawVarLighting;

static nn::gfx::Sampler g_samplerModel;
static nn::gfx::Sampler g_samplerRSM;
static nn::gfx::Sampler g_samplerViewer;
static nn::gfx::DescriptorSlot g_samplerModelSlot;
static nn::gfx::DescriptorSlot g_samplerRSMSlot;
static nn::gfx::DescriptorSlot g_samplerViewerSlot;

static OSTime g_lastTime = 0;
static f32 g_dtime = 0.0f;

static BOOL g_enableBaseColor = 0;
static BOOL g_enableIL = 0;
static BOOL g_showViewer = 0;

static Vec g_up = {0.0f, 1.0f, 0.0f};
static Vec g_objPt = {0.0f, 0.0f, 0.0f};
static Vec g_camPt = {-2000.0f, 1000.0f, 0.0f};
static const Vec g_lightUp = {0.0f, 1.0f, 0.0f};
static Vec g_lightPt = {-2000.0f, 2000.0f, 0.0f};
static const Vec g_origin = {0.0f, 0.0f, 0.0f};

static float g_offsetY = 40.0f;

static nn::gfx::Texture g_rsmT[NUMBER_RSM];
static nn::gfx::ColorTargetView g_rsmColor[NUMBER_RSM - 1];
static nn::gfx::DepthStencilView g_rsmDepth;
static nn::gfx::TextureView g_rsmTV[NUMBER_RSM];
static nn::gfx::DescriptorSlot g_rsmTVSlot[NUMBER_RSM];
static DEMOGfxMemPool* g_rsmData[NUMBER_RSM];

static nn::gfx::Texture g_gbufT[NUMBER_GBUF];
static nn::gfx::TextureView g_gbufTV[NUMBER_GBUF];
static nn::gfx::DescriptorSlot g_gbufTVSlot[NUMBER_GBUF];
static nn::gfx::ColorTargetView g_gbufColor[NUMBER_GBUF - 1];
static nn::gfx::DepthStencilView g_gbufDepth;
static DEMOGfxMemPool* g_gbufData[NUMBER_GBUF];

static nn::gfx::Texture g_illumT;
static nn::gfx::TextureView g_illumTV;
static nn::gfx::DescriptorSlot g_illumTVSlot;
static nn::gfx::ColorTargetView g_illumColor;
static DEMOGfxMemPool* g_illumData;

static bool s_firstFrame;

static const int DEMO_BUFFER_ALIGN = 4096;
static const int viewerPositionSize = sizeof( f32 ) * 12;
static const int viewerTexCoordSize = sizeof( f32 ) * 8;
static const int viewerIndexSize = sizeof( u32 ) * 6;

DEMOGfxPipeline gbufPipeline;
DEMOGfxPipeline ilPipeline;
DEMOGfxPipeline viewerPipeline;
DEMOGfxPipeline rsmPipeline;
DEMOGfxPipeline lightingPipeline;

nn::gfx::ViewportScissorState fullViewport;
static void* pFullViewportMem;
nn::gfx::ViewportScissorState rsmViewport;
static void* pRsmViewportMem;
nn::gfx::ViewportScissorState ilViewport;
static void* pIlViewportMem;
nn::gfx::ViewportScissorState viewerViewport[NUMBER_RSM];
static void* pViewerViewportMem[NUMBER_RSM];

struct vertexUniforms
{
    Mtx44   modelMtx44;
    Mtx44   viewMtx44;
    Mtx44   projMtx44;
};

static DEMOGfxBuffer rsmTempleVertexUniformBuffer;
static DEMOGfxBuffer rsmTerrainVertexUniformBuffer;
static DEMOGfxBuffer gbufTempleVertexUniformBuffer;
static DEMOGfxBuffer gbufTerrainVertexUniformBuffer;

struct lightingPixelUniforms
{
    Mtx44 projMtx44;
    float param[4];
    Vec   lightDir;
    float pad0;
    Vec   eyePos;
    float pad1;
};

static DEMOGfxBuffer lightingTempleVertexUniformBuffer;
static DEMOGfxBuffer lightingTerrainVertexUniformBuffer;
static DEMOGfxBuffer lightingPixelUniformBuffer;

struct viewerPixelUniforms
{
    float   index;
};

static int viewerPixelUniformBlock;
static int viewerTextureLoc;
DEMOGfxBuffer viewerPixelUniformBuffer[NUMBER_RSM];

DEMOGfxBuffer viewerPositionBuffer;
DEMOGfxBuffer viewerTexCoordBuffer;
DEMOGfxBuffer viewerIndexBuffer;

struct ilPixelUniforms
{
    Mat44 projMtx;
    Mat44 lightMtx;
    float coeff[4];
    float distance[4];
};

DEMOGfxBuffer ilPixelUniformBuffer;

void* g_pMaterialData;
nn::gfx::BlendState material;
void* g_pMaterialAlphaData;
nn::gfx::BlendState materialAlpha;

////////////////////////////////////////////////////
//
// Prototypes
//
////////////////////////////////////////////////////
static void InitShaderGBUF(DEMOGfxShader *pShader,const DemoAttributeData* attributes,s32 attributeCount);
static void InitShaderRSM(DEMOGfxShader *pShader,const DemoAttributeData* attributes,s32 attributeCount);
static void InitShaderIL(DEMOGfxShader *pShader);
static void InitShaderLighting(DEMOGfxShader *pShader, const DemoAttributeData* attributes, s32 attributeCount);
static void SetMaterial(const DemoMaterialData* pMaterial, DEMOGfxTexture* pTextures,u32 textureLoc);
static void DrawModel(const DemoModelData* pModelData,u32 textureLoc, bool bSkipMaterial);
static void InitCamera(Mtx44 resultProjMtx44, Mtx44 resultViewMtx44);
static void InitScene();
static void InitViewer(DEMOGfxShader *pShader);
static f32 ComputeLightMatrix(Mtx44 lightViewMtx44, Mtx44 lightProjMtx44);
static void DrawSceneFromLight();
static void DrawGBUFFER();
static void DrawIndirectIllumination();
static void DrawSceneFromCamera();
static void DrawViewer();
static void PrintInfo();
static void DrawScene();
static void ProcessPad();


// The initialization function for the rendering portions of this sample.
// It is responsible for allocating the three types of shaders and buffers
// as well as ensuring that data is flushed from the CPU to GPU memory
// for Simple Shader

static void InitShaderGBUF(DEMOGfxShader *pShader, const DemoAttributeData* attributes, s32 attributeCount)
{
    DEMOGfxLoadShadersFromFile(pShader, 0, SHADER_FILE_GBUFFER[g_ShaderFileIdx]);
    int attributeStride = 0;
    // Set attribute buffer to shader
    for (s32 i = 0; i < attributeCount; i++)
    {
        const DemoAttributeData* pAttribute = &attributes[i];
        nn::gfx::AttributeFormat format = nn::gfx::AttributeFormat_Undefined;

        const char* attributeName = NULL;

        switch (pAttribute->type){
        case DEMO_ATTRIBUTE_POSITION:
        {
            //Local coordinate attributes
            attributeName = "a_position";
            format = nn::gfx::AttributeFormat_32_32_32_Float;
            attributeStride += sizeof(float) * 3;
        }
        break;
        case DEMO_ATTRIBUTE_NORMAL:
        {
            //Normal attributes
            attributeName = "a_normal";
            format = nn::gfx::AttributeFormat_32_32_32_Float;
            attributeStride += sizeof(float) * 3;
        }
        break;
        case DEMO_ATTRIBUTE_TEXCOORD:
        {
            //Texture coordinate attributes
            attributeName = "a_texCoord";
            format = nn::gfx::AttributeFormat_32_32_Float;
            attributeStride += sizeof(float) * 2;
        }
        break;
        default:
        {
            //Attributes not used in shader
        }
        continue;
        }

        DEMOGfxInitShaderAttribute(pShader,
            attributeName,
            0,
            pAttribute->offset,
            format);
    }
    DEMOGfxInitShaderVertexBuffer(pShader, 0, attributeStride, 0);

    // Uniform Location Lookup
    g_drawVarGBUF.vertexUniformLoc = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_matrices");

    // uniform buffers
    gbufTempleVertexUniformBuffer.Initialize(sizeof(vertexUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
    gbufTerrainVertexUniformBuffer.Initialize(sizeof(vertexUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);

    // Texture Sampler lookup
    g_drawVarGBUF.samplerLoc = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture");

    gbufPipeline.SetDefaults();

    gbufPipeline.colorTargetStateCount = 2;
    gbufPipeline.colorTargetStateInfoArray[0].SetDefault();
    gbufPipeline.colorTargetStateInfoArray[0].SetFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    gbufPipeline.colorTargetStateInfoArray[1].SetDefault();
    gbufPipeline.colorTargetStateInfoArray[1].SetFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);

    gbufPipeline.blendTargetStateCount = 2;
    gbufPipeline.blendTargetStateInfoArray[0].SetDefault();
    gbufPipeline.blendTargetStateInfoArray[1].SetDefault();

#if NN_GFX_IS_TARGET_GX || NN_GFX_IS_TARGET_D3D
    gbufPipeline.rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Back );
#else
    // Flip the winding since we flip the y-coordinates in the shader
    gbufPipeline.rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Front );
#endif

    gbufPipeline.depthStencilStateInfo.SetDepthTestEnabled(true);
    gbufPipeline.depthStencilStateInfo.SetDepthWriteEnabled(true);
    gbufPipeline.depthStencilStateInfo.SetDepthComparisonFunction(nn::gfx::ComparisonFunction_Less);
    gbufPipeline.Initialize(&DEMODevice);
}

static void InitShaderRSM(DEMOGfxShader *pShader,const DemoAttributeData* attributes,s32 attributeCount)
{
    DEMOGfxLoadShadersFromFile(pShader, 0, SHADER_FILE_RSM[g_ShaderFileIdx]);
    int attributeStride = 0;
    // Set attribute buffer to shader
    for (s32 i = 0; i < attributeCount; i++)
    {
        const DemoAttributeData* pAttribute = &attributes[i];
        nn::gfx::AttributeFormat format = nn::gfx::AttributeFormat_Undefined;

        const char* attributeName = NULL;

        switch (pAttribute->type){
           case DEMO_ATTRIBUTE_POSITION:
               {
                   //Local coordinate attributes
                   attributeName = "a_position";
                   format = nn::gfx::AttributeFormat_32_32_32_Float;
                   attributeStride += sizeof( float ) * 3;
           }
               break;
           case DEMO_ATTRIBUTE_NORMAL:
               {
                   //Normal attributes
                   attributeName = "a_normal";
                   format = nn::gfx::AttributeFormat_32_32_32_Float;
                   attributeStride += sizeof( float ) * 3;
           }
               break;
           case DEMO_ATTRIBUTE_TEXCOORD:
               {
                   //Texture coordinate attributes
                   attributeName = "a_texCoord";
                   format = nn::gfx::AttributeFormat_32_32_Float;
                   attributeStride += sizeof( float ) * 2;
           }
               break;
           default:
               {
                   //Attributes not used in shader
               }
               continue;
        }

        DEMOGfxInitShaderAttribute(pShader,
                                   attributeName,
                                   0,
                                   pAttribute->offset,
                                   format);
    }
    DEMOGfxInitShaderVertexBuffer( pShader, 0, attributeStride, 0 );

    // Uniform Location Lookup
    g_drawVarRSM.vertexUniformLoc = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_matrices" );

    // uniform buffers
    rsmTempleVertexUniformBuffer.Initialize(sizeof(vertexUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
    rsmTerrainVertexUniformBuffer.Initialize(sizeof(vertexUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);

    // Texture Sampler lookup
    g_drawVarRSM.samplerLoc = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    // pipeline
    rsmPipeline.SetDefaults();
    rsmPipeline.colorTargetStateCount = NUMBER_RSM - 1;
    rsmPipeline.blendTargetStateCount = NUMBER_RSM - 1;
    for (int i = 0; i < NUMBER_RSM - 1; i++)
    {
        rsmPipeline.colorTargetStateInfoArray[i].SetDefault();
        rsmPipeline.colorTargetStateInfoArray[i].SetFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    }
    rsmPipeline.blendTargetStateCount = NUMBER_RSM - 1;
    for (int i = 0; i < NUMBER_RSM - 1; i++)
    {
        rsmPipeline.blendTargetStateInfoArray[i].SetDefault();
    }

#if NN_GFX_IS_TARGET_GX || NN_GFX_IS_TARGET_D3D
    rsmPipeline.rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Back );
#else
    // Flip the winding since we flip the y-coordinates in the shader
    rsmPipeline.rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Front );
#endif

    rsmPipeline.Initialize( &DEMODevice );
}

static void InitShaderIL(DEMOGfxShader *pShader)
{
    DEMOGfxLoadShadersFromFile(pShader, 0, SHADER_FILE_IL[g_ShaderFileIdx]);

    // Set attribute buffer to shader

    DEMOGfxInitShaderAttribute(pShader,
        "a_position",
        0,
        0,
        nn::gfx::AttributeFormat_32_32_32_Float);

    DEMOGfxInitShaderVertexBuffer(pShader, 0, 3 * sizeof(float), 0);

    // Uniform Location Lookup
    g_drawVarIL.pixelUniformLoc = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_pixelData");
    // Texture Sampler lookup
    g_drawVarIL.gbufLoc[0] = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_gbuffer0");
    g_drawVarIL.gbufLoc[1] = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_gbuffer1");
    g_drawVarIL.gbufLoc[2] = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_gbufferD");
    g_drawVarIL.rsmLoc[0] = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_rsm0");
    g_drawVarIL.rsmLoc[1] = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_rsm1");
    g_drawVarIL.rsmLoc[2] = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_rsm2");

    ilPipeline.SetDefaults();
    ilPipeline.colorTargetStateCount = 1;
    ilPipeline.colorTargetStateInfoArray[0].SetDefault();
    ilPipeline.colorTargetStateInfoArray[0].SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
    ilPipeline.blendTargetStateCount = 1;
    ilPipeline.blendTargetStateInfoArray[0].SetDefault();

    ilPipeline.depthStencilStateInfo.SetDepthTestEnabled(false);
    ilPipeline.depthStencilStateInfo.SetDepthWriteEnabled(false);
    ilPipeline.Initialize(&DEMODevice);

    // set up uniform buffers
    ilPixelUniformBuffer.Initialize(sizeof(ilPixelUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);

}

static void InitShaderLighting(DEMOGfxShader *pShader, const DemoAttributeData* attributes, s32 attributeCount)
{
    DEMOGfxLoadShadersFromFile(pShader, 0, SHADER_FILE_LIGHTING[g_ShaderFileIdx]);

    int attributeStride = 0;
    // Set attribute buffer to shader
    for (s32 i = 0; i < attributeCount; i++)
    {
        const DemoAttributeData* pAttribute = &attributes[i];
        nn::gfx::AttributeFormat format = nn::gfx::AttributeFormat_Undefined;

        const char* attributeName = NULL;

        switch (pAttribute->type){
        case DEMO_ATTRIBUTE_POSITION:
        {
            //Local coordinate attributes
            attributeName = "a_position";
            format = nn::gfx::AttributeFormat_32_32_32_Float;
            attributeStride += sizeof(float) * 3;
        }
        break;
        case DEMO_ATTRIBUTE_NORMAL:
        {
            //Normal attributes
            attributeName = "a_normal";
            format = nn::gfx::AttributeFormat_32_32_32_Float;
            attributeStride += sizeof(float) * 3;
        }
        break;
        case DEMO_ATTRIBUTE_TEXCOORD:
        {
            //Texture coordinate attributes
            attributeName = "a_texCoord";
            format = nn::gfx::AttributeFormat_32_32_Float;
            attributeStride += sizeof(float) * 2;
        }
        break;
        default:
        {
            //Attributes not used in shader
        }
        continue;
        }

        DEMOGfxInitShaderAttribute(pShader,
            attributeName,
            0,
            pAttribute->offset,
            format);
    }
    DEMOGfxInitShaderVertexBuffer(pShader, 0, attributeStride, 0);

    // Uniform Location Lookup
    g_drawVarLighting.vertexUniformLoc = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_matrices");
    g_drawVarLighting.pixelUniformLoc = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_lighting");

    // Texture Sampler lookup
    g_drawVarLighting.samplerLoc = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture");
    g_drawVarLighting.gbufLoc[0] = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_gbuffer0");
    g_drawVarLighting.gbufLoc[1] = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_gbuffer1");
    g_drawVarLighting.gbufLoc[2] = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_gbufferD");
    g_drawVarLighting.illuminationLoc = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_illumination");

    lightingPipeline.SetDefaults();
    lightingPipeline.colorTargetStateCount = 1;
    lightingPipeline.colorTargetStateInfoArray[0].SetDefault();
    lightingPipeline.colorTargetStateInfoArray[0].SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
    lightingPipeline.blendTargetStateCount = 1;
    lightingPipeline.blendTargetStateInfoArray[0].SetDefault();

    lightingPipeline.rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Back );

    lightingPipeline.Initialize(&DEMODevice);

    // initialize uniform buffers
    lightingPixelUniformBuffer.Initialize(sizeof(lightingPixelUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
    lightingTempleVertexUniformBuffer.Initialize(sizeof(vertexUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
    lightingTerrainVertexUniformBuffer.Initialize(sizeof(vertexUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
}

// Init function for setting projection matrix
static void InitCamera(Mtx44 resultProjMtx44, Mtx44 resultViewMtx44)
{
    // Row major matrices
    Mtx lookAtMtx34;

    f32 pers = 60.0f;
    f32 aspect = (f32)SURFACE_WIDTH / (f32)SURFACE_HEIGHT;
    f32 znear = 10.0f;
    f32 zfar = 100000.0f;

    // Compute perspective matrix
    MTXPerspective(resultProjMtx44, pers, aspect, znear, zfar);

    // Compute lookAt matrix
    MTXLookAt(lookAtMtx34, &g_camPt, &g_up, &g_objPt);
    MTX34To44(lookAtMtx34, resultViewMtx44);
}

// The init function for the rendering portions of this app
static void InitScene()
{
    // Load the model data
    DEMOLoadModelData(&g_modelTemple, MODEL_FILE_TEMPLE, MODEL_TEXTURE_DIRECTORY );
    DEMOLoadModelData( &g_modelTerrain, MODEL_FILE_TERRAIN, MODEL_TEXTURE_DIRECTORY );

    // Initialize shader
    InitShaderGBUF( &gbufPipeline.shaders, g_modelTemple.attributes,
                    g_modelTemple.header.attributeCount);
    InitShaderRSM( &rsmPipeline.shaders, g_modelTemple.attributes,
                     g_modelTemple.header.attributeCount);
    InitShaderIL(&ilPipeline.shaders);
    InitShaderLighting(&lightingPipeline.shaders, g_modelTemple.attributes, g_modelTemple.header.attributeCount);

    // Camera initialization
    InitCamera(g_projMtx44, g_viewMtx44);

    // Sampler setting
    DEMOGfxInitSampler( &g_samplerModel, &g_samplerModelSlot, nn::gfx::TextureAddressMode_Repeat, nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear, nn::gfx::ComparisonFunction_Always );
    DEMOGfxInitSampler( &g_samplerRSM, &g_samplerRSMSlot, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_Comparison_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Less );

    // color buffer for GBUFFER
    for (int i = 0; i < NUMBER_GBUF - 1; i++)
    {
        DEMOGfxSetupTextureBuffer(&g_gbufT[i], &g_gbufTV[i], &g_gbufTVSlot[i], &g_gbufColor[i], NULL, &g_gbufData[i],
            ILLUMINATION_BUFFER_WIDTH, ILLUMINATION_BUFFER_HEIGHT, 1, 1, nn::gfx::ImageDimension_2d,
            nn::gfx::ImageFormat_R16_G16_B16_A16_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0);
    }
    // depth buffer for GBUFFER
    DEMOGfxSetupTextureBuffer(&g_gbufT[NUMBER_GBUF - 1], &g_gbufTV[NUMBER_GBUF - 1], &g_gbufTVSlot[NUMBER_GBUF - 1], NULL, &g_gbufDepth, &g_gbufData[NUMBER_GBUF - 1],
        ILLUMINATION_BUFFER_WIDTH, ILLUMINATION_BUFFER_HEIGHT, 1, 1, nn::gfx::ImageDimension_2d,
        nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0);

    // texture for Reflective Shadow Maps
    for (int i = 0; i < NUMBER_RSM - 1; i++)
    {
        DEMOGfxSetupTextureBuffer(&g_rsmT[i], &g_rsmTV[i], &g_rsmTVSlot[i], &g_rsmColor[i], NULL, &g_rsmData[i],
            RSM_WIDTH, RSM_HEIGHT, 1, 1, nn::gfx::ImageDimension_2d,
            nn::gfx::ImageFormat_R16_G16_B16_A16_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0);
    }

    DEMOGfxSetupTextureBuffer(&g_rsmT[NUMBER_RSM - 1], &g_rsmTV[NUMBER_RSM - 1], &g_rsmTVSlot[NUMBER_RSM - 1], NULL, &g_rsmDepth, &g_rsmData[NUMBER_RSM - 1],
        RSM_WIDTH, RSM_HEIGHT, 1, 1, nn::gfx::ImageDimension_2d,
        nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0);

    // other buffers
    DEMOGfxSetupTextureBuffer(&g_illumT, &g_illumTV, &g_illumTVSlot, &g_illumColor, NULL, &g_illumData, ILLUMINATION_BUFFER_WIDTH, ILLUMINATION_BUFFER_HEIGHT,
        1, 1, nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R16_G16_B16_A16_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0);

    // Create viewports
    DEMOGfxSetViewportScissorState( &fullViewport, &pFullViewportMem, 0.0f,
        0.0f,
        static_cast< float >( SURFACE_WIDTH ),
        static_cast< float >( SURFACE_HEIGHT ),
        0.0f, 1.0f,
        static_cast< float >( SURFACE_HEIGHT ), false );
    DEMOGfxSetViewportScissorState( &rsmViewport, &pRsmViewportMem, 0.0f,
        0.0f,
        static_cast< float >( RSM_WIDTH ),
        static_cast< float >( RSM_HEIGHT ),
        0.0f, 1.0f,
        static_cast< float >( RSM_HEIGHT ), false );
    DEMOGfxSetViewportScissorState(&ilViewport, &pIlViewportMem, 0.0f,
        0.0f,
        static_cast< float >(ILLUMINATION_BUFFER_WIDTH),
        static_cast< float >(ILLUMINATION_BUFFER_HEIGHT),
        0.0f, 1.0f,
        static_cast< float >(ILLUMINATION_BUFFER_HEIGHT), false);

    {
        nn::gfx::BlendStateInfo info;
        nn::gfx::BlendTargetStateInfo blendTargets[ MAX_RENDER_TARGETS ];

        for (int i = 0; i < MAX_RENDER_TARGETS; i++)
        {
            blendTargets[i].SetDefault();
            blendTargets[i].SetBlendEnabled(false);
            blendTargets[i].SetChannelMask(nn::gfx::ChannelMask_All);
        }

        info.SetDefault();
        info.SetAlphaToCoverageEnabled( false );
        info.SetBlendTargetStateInfoArray( &blendTargets[0], MAX_RENDER_TARGETS );
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
        g_pMaterialData = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        material.SetMemory( g_pMaterialData, size );
        material.Initialize( &DEMODevice, info );
        info.SetAlphaToCoverageEnabled( true );
        size = nn::gfx::BlendState::GetRequiredMemorySize( info );
        g_pMaterialAlphaData = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        materialAlpha.SetMemory( g_pMaterialAlphaData, size );
        materialAlpha.Initialize( &DEMODevice, info );
    }
}

static void InitViewer(DEMOGfxShader *pShader)
{
    DEMOGfxLoadShadersFromFile(pShader, 0, SHADER_FILE_VIEWER[g_ShaderFileIdx]);

    // Set attribute buffer to shader
    DEMOGfxInitShaderAttribute(pShader,
                               "a_position",
                               0,
                               0,
                               nn::gfx::AttributeFormat_32_32_32_Float);
    DEMOGfxInitShaderAttribute(pShader,
                               "a_texCoord",
                               1,
                               0,
                               nn::gfx::AttributeFormat_32_32_Float );
    DEMOGfxInitShaderVertexBuffer(pShader, 0, 3 * sizeof(float), 0);
    DEMOGfxInitShaderVertexBuffer(pShader, 1, 2 * sizeof(float), 0);

    // Uniform Location Lookup
    viewerPixelUniformBlock = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_index");
    viewerTextureLoc = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture");

    f32 *posInitData = ( f32 *)DEMOAllocEx( viewerPositionSize, DEMO_BUFFER_ALIGN );
    f32 *texInitData = ( f32 * )DEMOAllocEx( viewerTexCoordSize, DEMO_BUFFER_ALIGN );
    u32 *idxInitData = ( u32 * )DEMOAllocEx( viewerIndexSize, DEMO_BUFFER_ALIGN );

    posInitData[0] = -1.0f; posInitData[1] = -1.0f;
    posInitData[2] =  0.0f; posInitData[3] =  1.0f;
    posInitData[4] = -1.0f; posInitData[5] =  0.0f;
    posInitData[6] =  1.0f; posInitData[7] =  1.0f;
    posInitData[8] =  0.0f; posInitData[9] = -1.0f;
    posInitData[10] = 1.0f; posInitData[11] = 0.0f;

    texInitData[0] =  0.0f; texInitData[1] =  1.0f;
    texInitData[2] =  1.0f; texInitData[3] =  1.0f;
    texInitData[4] =  1.0f; texInitData[5] =  0.0f;
    texInitData[6] =  0.0f; texInitData[7] =  0.0f;

    idxInitData[0] = 0; idxInitData[1] = 1;
    idxInitData[2] = 2; idxInitData[3] = 3;
    idxInitData[4] = 0; idxInitData[5] = 2;

    viewerPositionBuffer.Initialize( viewerPositionSize, posInitData, nn::gfx::GpuAccess_VertexBuffer, 0 );
    viewerTexCoordBuffer.Initialize( viewerTexCoordSize, texInitData, nn::gfx::GpuAccess_VertexBuffer, 0);
    viewerIndexBuffer.Initialize( viewerIndexSize, idxInitData, nn::gfx::GpuAccess_IndexBuffer, 0);

    DEMOFree( posInitData );
    DEMOFree( texInitData );
    DEMOFree( idxInitData );

    DEMOGfxInitSampler( &g_samplerViewer, &g_samplerViewerSlot, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    for (int i = 0; i < NUMBER_RSM; i++)
    {
        viewerPixelUniformBuffer[ i ].Initialize( sizeof( viewerPixelUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    }
    float xo, yo;
    for ( u32 i = 0; i < NUMBER_RSM; ++i )
    {
        switch ( i )
        {
        default: /* fall through */
        case 0: xo = ( float )SURFACE_WIDTH / 7.0f * 4.0f; yo = ( float )SURFACE_HEIGHT / 4.0f * 0.0f; break;
        case 1: xo = ( float )SURFACE_WIDTH / 7.0f * 5.0f; yo = ( float )SURFACE_HEIGHT / 4.0f * 0.0f; break;
        case 2: xo = ( float )SURFACE_WIDTH / 7.0f * 6.0f; yo = ( float )SURFACE_HEIGHT / 4.0f * 0.0f; break;
        case 3: xo = ( float )SURFACE_WIDTH / 7.0f * 6.0f; yo = ( float )SURFACE_HEIGHT / 4.0f * 1.0f; break;
        }

        DEMOGfxSetViewportScissorState( &viewerViewport[i], &pViewerViewportMem[i], xo, yo,
            SURFACE_WIDTH / 7.0f, SURFACE_HEIGHT / 4.0f, 0.0f, 1.0f, (float)SURFACE_HEIGHT, false );
    }

    viewerPipeline.colorTargetStateCount = 1;
    viewerPipeline.colorTargetStateInfoArray[0].SetDefault();
    viewerPipeline.colorTargetStateInfoArray[0].SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
    viewerPipeline.blendTargetStateCount = 1;
    viewerPipeline.blendTargetStateInfoArray[0].SetDefault();
    viewerPipeline.rasterizerStateInfo.SetCullMode(nn::gfx::CullMode_None);
    viewerPipeline.depthStencilStateInfo.SetDepthWriteEnabled(false);
    viewerPipeline.depthStencilStateInfo.SetDepthTestEnabled(false);
    viewerPipeline.Initialize(&DEMODevice);
}

static void VecBounds(const Vec& pt, Vec& minPt, Vec& maxPt)
{
    if (minPt.x > pt.x)
    {
        minPt.x = pt.x;
    }
    if (minPt.y > pt.y)
    {
        minPt.y = pt.y;
    }
    if (minPt.z > pt.z)
    {
        minPt.z = pt.z;
    }
    if (maxPt.x < pt.x)
    {
        maxPt.x = pt.x;
    }
    if (maxPt.y < pt.y)
    {
        maxPt.y = pt.y;
    }
    if (maxPt.z < pt.z)
    {
        maxPt.z = pt.z;
    }
}
static f32 ComputeLightMatrix(Mtx44 lightViewMtx44, Mtx44 lightProjMtx44)
{
    Mtx44 invProjMtx44;
    MTX44Inverse(g_projMtx44, invProjMtx44);
    Vec cube[8] =
    {
        {-1.0f, -1.0f, -1.0f,},
        {+1.0f, -1.0f, -1.0f,},
        {-1.0f, +1.0f, -1.0f,},
        {+1.0f, +1.0f, -1.0f,},
        {-1.0f, -1.0f, +1.0f,},
        {+1.0f, -1.0f, +1.0f,},
        {-1.0f, +1.0f, +1.0f,},
        {+1.0f, +1.0f, +1.0f,},
    };
    Vec frustum[8];
    for (u32 i = 0; i < 8; ++i)
    {
        MTX44MultVec(invProjMtx44, &cube[i], &frustum[i]);
    }

    // Calculate the splitting locations for the cascade
    f32 nearDist = -frustum[0].z;
    f32 farDist = -frustum[4].z;

    // Calculate center coordinates of rendering range
    f32 f = (RSM_RANGE - nearDist) / (farDist - nearDist);
    f = (f < 0.0f) ? 0.0f : (f > 1.0f ? 1.0f : f);

    Vec minPt = {FLT_MAX, FLT_MAX, FLT_MAX};
    Vec maxPt = {-FLT_MAX, -FLT_MAX, -FLT_MAX};
    for (u32 i = 0; i < 4; ++i)
    {
        Vec t0, t1, pt;
        pt = frustum[i];
        VecBounds(pt, minPt, maxPt);

        VECScale(&frustum[i], &t0, (1.0f - f));
        VECScale(&frustum[i + 4], &t1, f);
        VECAdd(&t0, &t1, &pt);
        VecBounds(pt, minPt, maxPt);
    }
    Vec dir;
    VECSubtract(&maxPt, &minPt, &dir);
    VECScale(&dir, &dir, 0.5f);
    Vec centerPt;
    VECAdd(&minPt, &dir, &centerPt);
    f32 radius = 0.5f * VECDistance(&maxPt, &minPt);
    Mtx44 invViewMtx44;
    MTX44Inverse(g_viewMtx44, invViewMtx44);
    Vec center;
    MTX44MultVec(invViewMtx44, &centerPt, &center);

    Vec lightDir;
    VECNormalize(&g_lightPt, &lightDir);

    // Round center coordinates to nearest texel
    Vec ax, ay, temp;
    VECCrossProduct(&g_lightPt, &g_lightUp, &temp);
    VECNormalize(&temp, &ax);
    VECCrossProduct(&ax, &g_lightPt, &temp);
    VECNormalize(&temp, &ay);
    f32 sx, sy, sz;
    f32 ux = RSM_WIDTH / (2.0f * radius);
    f32 uy = RSM_HEIGHT / (2.0f * radius);
    sx = ceilf(VECDotProduct(&center, &ax) * ux) / ux;
    sy = ceilf(VECDotProduct(&center, &ay) * uy) / uy;
    sz = VECDotProduct(&center, &lightDir);
    Vec target;
    VECScale(&ax, &target, sx);
    VECScale(&ay, &temp, sy);
    VECAdd(&target, &temp, &target);
    VECScale(&lightDir, &temp, sz);
    VECAdd(&target, &temp, &target);

    // Compute lookAt matrix
    Vec lightDist;
    VECScale(&lightDir, &lightDist, radius + RSM_CASTING_RANGE);
    Vec lightPt;
    VECAdd(&target, &lightDist, &lightPt);
    Mtx lightViewMtx34;
    MTXLookAt(lightViewMtx34, &lightPt, &g_lightUp, &target);
    MTX34To44(lightViewMtx34, lightViewMtx44);

    // Compute projection matrix
    MTXOrtho(lightProjMtx44, radius, -radius, -radius, radius,
             0.0f, 2.0f * radius + RSM_CASTING_RANGE);

    return RSM_RANGE - nearDist;
}

// Set material
static void SetMaterial( const DemoMaterialData* pMaterial, DEMOGfxTexture* pTextures, u32 textureLoc )
{
    // Set Texture
    DEMOCommandBuffer.SetTextureAndSampler( textureLoc, nn::gfx::ShaderStage_Pixel, pTextures[ pMaterial->textureIndex ].GetDescriptorSlot( 0 ), g_samplerModelSlot );

    // Rendering state settings
    switch ( pMaterial->type )
    {
    case DEMO_MATERIAL_OPAQUE:
        //Translucent material
        DEMOCommandBuffer.SetBlendState( &material );
        break;
    case DEMO_MATERIAL_OVERLAP:
        //Translucent material (no Z-write)
        DEMOCommandBuffer.SetBlendState( &materialAlpha );
        break;
    case DEMO_MATERIAL_TRANSPARENT:
        //Translucent material
        DEMOCommandBuffer.SetBlendState( &materialAlpha );
        break;
    case DEMO_MATERIAL_ALPHA_TO_COVERAGE:
        //Coverage mask material
        DEMOCommandBuffer.SetBlendState( &materialAlpha );
        break;
    default:
        ASSERT(false && "Illegal material type");
        break;
    }
}

// Render the model data
static void DrawModel( const DemoModelData* pModelData, u32 textureLoc, bool bSkipMaterial )
{
    s32 currentMaterialIndex = -1;

    for ( s32 i = 0; i < pModelData->header.meshCount; i++ )
    {
        const DemoMeshData* pMesh = &pModelData->meshes[ i ];

        // Set material
        if ( bSkipMaterial && currentMaterialIndex != pMesh->materialIndex )
        {
            currentMaterialIndex = pMesh->materialIndex;

            SetMaterial( &pModelData->materials[ pMesh->materialIndex ],
                pModelData->pTextures, textureLoc );
        }

        // Set attribute buffers
        nn::gfx::GpuAddress gpuAddress = pModelData->attributeBuffer.buffer.gpuAddress;
        gpuAddress.Offset( pMesh->attributeOffset );
        DEMOCommandBuffer.SetVertexBuffer( 0, gpuAddress, pMesh->attributeStride, pModelData->attributeBuffer.size - pMesh->attributeOffset );

        // Draw Triangle. pMesh->indexOffset is the index offset, but we already have that in the index buffer indexView[]
        gpuAddress = pModelData->indexBuffer.buffer.gpuAddress;
        gpuAddress.Offset( pMesh->indexOffset );
        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList,
            pModelData->indexBuffer.stride == 2 ? nn::gfx::IndexFormat_Uint16 : nn::gfx::IndexFormat_Uint32,
            gpuAddress, pMesh->indexCount, 0 );
    }
}

static void ExpandZ()
{
#ifdef CAFE
    // Need to expand HiZ
    GX2UTSetExpandDepthState(GX2_ENABLE);
    GX2UTExpandDepthBufferOp(reinterpret_cast< GX2DepthBuffer* >(&g_gbufDepth.ToData()->gx2DepthBuffer));
    GX2UTSetExpandDepthState(GX2_DISABLE);

    GX2UTSetExpandDepthState(GX2_ENABLE);
    GX2UTExpandDepthBufferOp(reinterpret_cast< GX2DepthBuffer* >(&g_rsmDepth.ToData()->gx2DepthBuffer));
    GX2UTSetExpandDepthState(GX2_DISABLE);

    DEMOCommandBuffer.FlushMemory(nn::gfx::GpuAccess_DepthStencil);
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture);

    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
}

static void DrawGBUFFER()
{
    for (int i = 0; i < 2; i++)
    {
        DEMOCommandBuffer.ClearColor(&g_gbufColor[i], 0.0f, 0.0f, 0.0f, 0.0f, NULL);
    }
    DEMOCommandBuffer.ClearDepthStencil(&g_gbufDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_Depth, NULL);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetViewportScissorState(&ilViewport);
    nn::gfx::ColorTargetView *rtx[2] = { &g_gbufColor[0], &g_gbufColor[1] };
    DEMOCommandBuffer.SetRenderTargets(2, rtx, &g_gbufDepth);
    DEMOCommandBuffer.SetPipeline(&gbufPipeline.pipeline);

    vertexUniforms* pVU;

    pVU = gbufTempleVertexUniformBuffer.Map< vertexUniforms >();
    memcpy(&pVU->viewMtx44, &g_viewMtx44, sizeof(g_viewMtx44));
    memcpy(&pVU->projMtx44, &g_projMtx44, sizeof(g_projMtx44));
    MTX44Identity(pVU->modelMtx44);
    pVU->modelMtx44[1][3] += g_offsetY;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(pVU, sizeof(*pVU));
#endif
    gbufTempleVertexUniformBuffer.Unmap();
    DEMOCommandBuffer.SetConstantBuffer(g_drawVarGBUF.vertexUniformLoc, nn::gfx::ShaderStage_Vertex, gbufTempleVertexUniformBuffer.gpuAddress, gbufTempleVertexUniformBuffer.size);
    DrawModel(&g_modelTemple, g_drawVarGBUF.samplerLoc, true);

    pVU = gbufTerrainVertexUniformBuffer.Map< vertexUniforms >();
    memcpy(&pVU->viewMtx44, &g_viewMtx44, sizeof(g_viewMtx44));
    memcpy(&pVU->projMtx44, &g_projMtx44, sizeof(g_projMtx44));
    MTX44Identity(pVU->modelMtx44);
    pVU->modelMtx44[0][0] *= 2.0f;
    pVU->modelMtx44[2][2] *= 2.0f;
    pVU->modelMtx44[1][3] += -1.0f;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(pVU, sizeof(*pVU));
#endif
    gbufTerrainVertexUniformBuffer.Unmap();
    DEMOCommandBuffer.SetConstantBuffer(g_drawVarGBUF.vertexUniformLoc, nn::gfx::ShaderStage_Vertex, gbufTerrainVertexUniformBuffer.gpuAddress, gbufTerrainVertexUniformBuffer.size);
    DrawModel(&g_modelTerrain, g_drawVarGBUF.samplerLoc, true);

    DEMOCommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_DepthStencil);
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ConstantBuffer);
}

static void DrawIndirectIllumination()
{
    DEMOCommandBuffer.ClearColor(&g_illumColor, 0.0f, 0.0f, 0.0f, 0.0f, NULL);
    DEMOCommandBuffer.SetViewportScissorState(&ilViewport);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    nn::gfx::ColorTargetView *rtx[1] = { &g_illumColor };
    DEMOCommandBuffer.SetRenderTargets(1, rtx, NULL);
    DEMOCommandBuffer.SetPipeline(&ilPipeline.pipeline);

    struct ilPixelUniforms *pUniforms = ilPixelUniformBuffer.Map< ilPixelUniforms >();

    memcpy(&pUniforms->projMtx, &g_projMtx44, sizeof(g_projMtx44));

    Mtx44 tex;
    MTX44Identity(tex);
    tex[0][0] = 0.5f;
    tex[1][1] = -0.5f;
    tex[2][2] = 0.5f;
    tex[0][3] = 0.5f;
    tex[1][3] = 0.5f;
    tex[2][3] = 0.5f;
    Mtx44 lightViewMtx44, lightProjMtx44, temp, vpt;
    f32 splitDistance = ComputeLightMatrix(lightViewMtx44, lightProjMtx44);
    memcpy(&pUniforms->distance, &splitDistance, sizeof(float));
    MTX44Concat(tex, lightProjMtx44, temp);
    MTX44Concat(temp, lightViewMtx44, vpt);
    memcpy(&pUniforms->lightMtx, vpt, sizeof(Mtx44));

    f32 coeff[4] = { 4.0f, 1.0f, 80.0f, 10000.0f };
    memcpy(&pUniforms->coeff, coeff, sizeof(coeff));
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(pUniforms, sizeof(*pUniforms));
#endif
    ilPixelUniformBuffer.Unmap();

    DEMOCommandBuffer.SetConstantBuffer(g_drawVarIL.pixelUniformLoc, nn::gfx::ShaderStage_Pixel, ilPixelUniformBuffer.gpuAddress, ilPixelUniformBuffer.size);

    // Set Texture
    for (int i = 0; i < NUMBER_RSM - 1; i++)
    {
        DEMOCommandBuffer.SetTextureAndSampler(g_drawVarIL.rsmLoc[i], nn::gfx::ShaderStage_Pixel, g_rsmTVSlot[i], g_samplerRSMSlot);
    }
    for (int i = 0; i < NUMBER_GBUF; i++)
    {
        DEMOCommandBuffer.SetTextureAndSampler(g_drawVarIL.gbufLoc[i], nn::gfx::ShaderStage_Pixel, g_gbufTVSlot[i], g_samplerRSMSlot);
    }
    DEMOCommandBuffer.SetVertexBuffer(0, viewerPositionBuffer.gpuAddress, sizeof(float) * 3, viewerPositionBuffer.size);
    DEMOCommandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
        viewerIndexBuffer.gpuAddress, 6, 0);

    DEMOCommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_DepthStencil);
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ConstantBuffer);
}
static void DrawSceneFromLight()
{
    for (int i = 0; i < 3; i++)
    {
        DEMOCommandBuffer.ClearColor(&g_rsmColor[i], 0.0f, 0.0f, 0.0f, 0.0f, NULL);
    }
    DEMOCommandBuffer.ClearDepthStencil(&g_rsmDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_Depth, NULL);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    nn::gfx::ColorTargetView *rtx[3] = { &g_rsmColor[0], &g_rsmColor[1], &g_rsmColor[2] };
    DEMOCommandBuffer.SetRenderTargets(3, rtx, &g_rsmDepth);
    DEMOCommandBuffer.SetViewportScissorState(&rsmViewport);
    DEMOCommandBuffer.SetPipeline(&rsmPipeline.pipeline);

    Mtx44 lightViewMtx44;
    Mtx44 lightProjMtx44;
    ComputeLightMatrix(lightViewMtx44, lightProjMtx44);

    vertexUniforms* pVU;

    pVU = rsmTempleVertexUniformBuffer.Map< vertexUniforms >();
    memcpy(&pVU->viewMtx44, &lightViewMtx44, sizeof(lightViewMtx44));
    memcpy(&pVU->projMtx44, &lightProjMtx44, sizeof(lightProjMtx44));
    MTX44Identity(pVU->modelMtx44);
    pVU->modelMtx44[1][3] += g_offsetY;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(pVU, sizeof(*pVU));
#endif
    rsmTempleVertexUniformBuffer.Unmap();
    DEMOCommandBuffer.SetConstantBuffer(g_drawVarRSM.vertexUniformLoc, nn::gfx::ShaderStage_Vertex, rsmTempleVertexUniformBuffer.gpuAddress, rsmTempleVertexUniformBuffer.size);
    DrawModel(&g_modelTemple, g_drawVarRSM.samplerLoc, true);

    pVU = rsmTerrainVertexUniformBuffer.Map< vertexUniforms >();
    memcpy(&pVU->viewMtx44, &lightViewMtx44, sizeof(lightViewMtx44));
    memcpy(&pVU->projMtx44, &lightProjMtx44, sizeof(lightProjMtx44));
    MTX44Identity(pVU->modelMtx44);
    pVU->modelMtx44[0][0] *= 2.0f;
    pVU->modelMtx44[2][2] *= 2.0f;
    pVU->modelMtx44[1][3] += -1.0f;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(pVU, sizeof(*pVU));
#endif
    rsmTerrainVertexUniformBuffer.Unmap();
    DEMOCommandBuffer.SetConstantBuffer(g_drawVarRSM.vertexUniformLoc, nn::gfx::ShaderStage_Vertex, rsmTerrainVertexUniformBuffer.gpuAddress, rsmTerrainVertexUniformBuffer.size);
    DrawModel(&g_modelTerrain, g_drawVarRSM.samplerLoc, true);
    DEMOCommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_DepthStencil);
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ConstantBuffer);
}

static void DrawSceneFromCamera()
{
    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, 0.2f, 0.65f, 0.9f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOGfxSetDefaultRenderTarget();
    DEMOCommandBuffer.SetViewportScissorState( &fullViewport );
    DEMOCommandBuffer.SetPipeline( &lightingPipeline.pipeline );

    struct vertexUniforms vertUniformValues;

    memcpy(&vertUniformValues.viewMtx44, &g_viewMtx44, sizeof(g_viewMtx44));
    memcpy(&vertUniformValues.projMtx44, &g_projMtx44, sizeof(g_projMtx44));
    MTX44Identity(vertUniformValues.modelMtx44);
    vertUniformValues.modelMtx44[1][3] += g_offsetY;

    struct lightingPixelUniforms * pPU = lightingPixelUniformBuffer.Map< struct lightingPixelUniforms >();
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pPU, sizeof( *pPU ) );
#endif
    memcpy(&pPU->projMtx44, &g_projMtx44, sizeof(g_projMtx44));

    VECSubtract(&g_lightPt, &g_origin, &pPU->lightDir);
    memcpy(&pPU->eyePos, &g_camPt, sizeof(g_camPt));

    pPU->param[0] = g_enableBaseColor ? 1.0f : 0.0f;
    pPU->param[1] = g_enableIL ? 1.0f : 0.0f;

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pPU, sizeof( *pPU ) );
#endif
    lightingPixelUniformBuffer.Unmap( );

    for (u32 i = 1; i < NUMBER_GBUF; i++)
    {
        DEMOCommandBuffer.SetTextureAndSampler(g_drawVarLighting.gbufLoc[i], nn::gfx::ShaderStage_Pixel, g_gbufTVSlot[i], g_samplerRSMSlot);
    }
    DEMOCommandBuffer.SetTextureAndSampler(g_drawVarLighting.illuminationLoc, nn::gfx::ShaderStage_Pixel, g_illumTVSlot, g_samplerModelSlot);

    struct vertexUniforms *pVU = lightingTempleVertexUniformBuffer.Map< struct vertexUniforms >();
    memcpy(pVU, &vertUniformValues, sizeof(vertUniformValues));
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(pVU, sizeof(*pVU));
#endif
    lightingTempleVertexUniformBuffer.Unmap();
    DEMOCommandBuffer.SetConstantBuffer( g_drawVarLighting.pixelUniformLoc, nn::gfx::ShaderStage_Pixel, lightingPixelUniformBuffer.gpuAddress, lightingPixelUniformBuffer.size );
    DEMOCommandBuffer.SetConstantBuffer(g_drawVarLighting.vertexUniformLoc, nn::gfx::ShaderStage_Vertex, lightingTempleVertexUniformBuffer.gpuAddress, lightingTempleVertexUniformBuffer.size);

    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer );

    DrawModel(&g_modelTemple, g_drawVarLighting.samplerLoc, true);


    MTX44Identity(vertUniformValues.modelMtx44);
    vertUniformValues.modelMtx44[0][0] *= 2.0f;
    vertUniformValues.modelMtx44[2][2] *= 2.0f;
    vertUniformValues.modelMtx44[1][3] += -1.0f;

    pVU = lightingTerrainVertexUniformBuffer.Map< struct vertexUniforms >();
    memcpy(pVU, &vertUniformValues, sizeof(vertUniformValues));
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(pVU, sizeof(*pVU));
#endif
    lightingTerrainVertexUniformBuffer.Unmap();

    DEMOCommandBuffer.SetConstantBuffer( g_drawVarLighting.vertexUniformLoc, nn::gfx::ShaderStage_Vertex, lightingTerrainVertexUniformBuffer.gpuAddress, lightingTerrainVertexUniformBuffer.size );

    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer );

    DrawModel( &g_modelTerrain, g_drawVarLighting.samplerLoc, true );
}

static void DrawViewer()
{
    DEMOCommandBuffer.SetPipeline(&viewerPipeline.pipeline);
    //DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_VertexBuffer);
    DEMOCommandBuffer.SetVertexBuffer(0, viewerPositionBuffer.gpuAddress, sizeof(float) * 3, viewerPositionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer(1, viewerTexCoordBuffer.gpuAddress, sizeof(float) * 2, viewerTexCoordBuffer.size );

    for (u32 i = 0; i < NUMBER_RSM - 1; ++i)
    {
        DEMOCommandBuffer.SetViewportScissorState( &viewerViewport[ i ] );

        DEMOCommandBuffer.SetTextureAndSampler(viewerTextureLoc, nn::gfx::ShaderStage_Pixel, g_rsmTVSlot[i], g_samplerViewerSlot);

        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
            viewerIndexBuffer.gpuAddress, 6, 0);
    }
    DEMOCommandBuffer.SetViewportScissorState(&viewerViewport[NUMBER_RSM - 1]);
    DEMOCommandBuffer.SetTextureAndSampler(viewerTextureLoc, nn::gfx::ShaderStage_Pixel, g_illumTVSlot, g_samplerViewerSlot);
    DEMOCommandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
        viewerIndexBuffer.gpuAddress, 6, 0);
}

static void PrintInfo()
{
    float denom = (g_dtime > 1.0e-7f ? g_dtime : 1.0e-7f);
    DEMOCommandBuffer.SetViewportScissorState( &fullViewport );

    DEMOFontPrintf(5, 2, "Indirect Illumination : %.1f fps", 1.0f / denom);
    DEMOFontPrintf(5, 3, "Camera : %g,%g,%g", g_camPt.x, g_camPt.y, g_camPt.z);
    DEMOFontPrintf(5, 4, "Target : %g,%g,%g", g_objPt.x, g_objPt.y, g_objPt.z);
    DEMOFontPrintf(5, 5, "Move Camera (L Stick/R Stick)");
    DEMOFontPrintf(5, 6, "Rotate Light Direction (up/down/left/right)");
    DEMOFontPrintf(5, 7, "Enable Indirect Illumination (x)");
    DEMOFontPrintf(5, 8, "Enable Diffuse Texture (a)");
    DEMOFontPrintf(5, 9, "Show Reflective Shadow Maps (b)");
}

// The draw function for the rendering portions of this sample.
static void DrawScene()
{
    DEMOGfxBeforeRender();

    DrawSceneFromLight();

    DrawGBUFFER();

    ExpandZ();

    DrawIndirectIllumination();

    DrawSceneFromCamera();

    if (g_showViewer)
    {
        DrawViewer();
    }

    if (s_firstFrame == false)
    {
        PrintInfo();
    }
    else
    {
        s_firstFrame = false;
    }

    DEMOGfxDoneRender();
}

static void ProcessPad()
{
    DEMOPadRead();
    u16 button = DEMOPadGetButton(0);
    u16 buttonDown = DEMOPadGetButtonDown(0);

    if (DEMO_PAD_BUTTON_A & buttonDown)
    {
        g_enableBaseColor = ~g_enableBaseColor;
    }
    if (DEMO_PAD_BUTTON_B & buttonDown)
    {
        g_showViewer = ~g_showViewer;
    }
    if (DEMO_PAD_BUTTON_X & buttonDown)
    {
        g_enableIL = ~g_enableIL;
    }
    {
        // L Pad rotates light direction
        static f32 v = 1.0f;
        f32 sx = 0.0f;
        f32 sy = 0.0f;
        sx += (DEMO_PAD_BUTTON_LEFT & button) ? v : 0.0f;
        sx -= (DEMO_PAD_BUTTON_RIGHT & button) ? v : 0.0f;
        sy += (DEMO_PAD_BUTTON_UP & button) ? v : 0.0f;
        sy -= (DEMO_PAD_BUTTON_DOWN & button) ? v : 0.0f;

        Mtx44 rot;
        MTX44RotAxisRad(rot, &g_lightUp, MTXDegToRad(sx));
        MTX44MultVecSR(rot, &g_lightPt, &g_lightPt);

        Vec ld;
        VECNormalize(&g_lightPt, &ld);
        f32 c = VECDotProduct(&g_lightUp, &ld);
        if ((0.25f < c && sy > 0.0f) || (c < 0.9f && sy < 0.0f))
        {
            Vec ax;
            VECCrossProduct(&g_lightUp, &g_lightPt, &ax);
            MTX44RotAxisRad(rot, &ax, MTXDegToRad(sy));
            MTX44MultVecSR(rot, &g_lightPt, &g_lightPt);
        }
    }

    {
        // L Stick translates camera
        f32 sx = (f32)DEMOPadGetStickX(0) / 255.0f;
        f32 sy = (f32)DEMOPadGetStickY(0) / 255.0f;

        Vec vec, tempVec;
        tempVec.x =  sx * 100.0f;
        tempVec.y = 0.0f;
        tempVec.z = -sy * 100.0f;

        Mtx44 inv;
        MTX44Inverse(g_viewMtx44, inv);
        MTX44MultVecSR(inv, &tempVec, &vec);
        Vec tempCamLoc = g_camPt;
        Vec tempObjPt = g_objPt;
        VECAdd(&vec, &tempCamLoc, &g_camPt);
        VECAdd(&vec, &tempObjPt, &g_objPt);

        InitCamera(g_projMtx44, g_viewMtx44);
    }

    {
        // R Stick rotates camera
        f32 sx = (f32)DEMOPadGetSubStickX(0) / 255.0f;
        f32 sy = (f32)DEMOPadGetSubStickY(0) / 255.0f;

        Vec eyev, tempEyev;
        VECSubtract(&g_objPt, &g_camPt, &eyev);
        Vec wupv =  {0.0f, 1.0f, 0.0f};
        VECCrossProduct(&eyev, &g_up, &tempEyev);
        VECNormalize(&tempEyev, &eyev);

        Mtx44 rot,rot0,rot1;
        MTX44RotAxisRad(rot0, &eyev, MTXDegToRad( sy * 5.0f));
        MTX44RotAxisRad(rot1, &wupv, MTXDegToRad(-sx * 5.0f));

        MTX44Concat(rot0, rot1, rot);

        Vec camv, tempCamv;
        VECSubtract(&g_objPt, &g_camPt, &tempCamv);
        MTX44MultVecSR(rot, &tempCamv, &camv);

        VECAdd(&camv, &g_camPt, &g_objPt);
        Vec tempUp = g_up;
        MTX44MultVecSR(rot, &tempUp, &g_up);

        InitCamera(g_projMtx44, g_viewMtx44);
    }
}

//extern "C" void nnMain()
TEST(GfxIndirectIllumination, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    InitScene();
    s_firstFrame = true;
    InitViewer(&viewerPipeline.shaders);

    while (DEMOIsRunning())
    {
        g_dtime = OSTicksToMilliseconds(OSGetTime() - g_lastTime) / 1000.0f;
        g_lastTime = OSGetTime();

        ProcessPad();
        DrawScene();
    }

    DEMOFreeModelData(&g_modelTemple);
    DEMOFreeModelData(&g_modelTerrain);

    lightingTempleVertexUniformBuffer.Finalize();
    lightingTerrainVertexUniformBuffer.Finalize();
    lightingPixelUniformBuffer.Finalize();
    lightingPipeline.Finalize( &DEMODevice );

    for (int i = 0; i < NUMBER_GBUF - 1; i++)
    {
        g_gbufT[i].Finalize(&DEMODevice);
        g_gbufColor[i].Finalize(&DEMODevice);
        g_gbufTV[i].Finalize(&DEMODevice);
        g_gbufData[i]->Finalize();
        delete g_gbufData[ i ];
    }
    g_gbufDepth.Finalize(&DEMODevice);
    g_gbufT[NUMBER_GBUF - 1].Finalize(&DEMODevice);
    g_gbufTV[NUMBER_GBUF - 1].Finalize(&DEMODevice);
    g_gbufData[NUMBER_GBUF - 1]->Finalize();
    delete g_gbufData[NUMBER_GBUF - 1];

    gbufTempleVertexUniformBuffer.Finalize();
    gbufTerrainVertexUniformBuffer.Finalize();
    gbufPipeline.Finalize( &DEMODevice );

    for (int i = 0; i < NUMBER_RSM - 1; i++)
    {
        g_rsmT[i].Finalize( &DEMODevice );
        g_rsmColor[i].Finalize(&DEMODevice);
        g_rsmTV[i].Finalize(&DEMODevice);
        g_rsmData[ i ]->Finalize();
        delete g_rsmData[ i ];
    }
    g_rsmDepth.Finalize(&DEMODevice);
    g_rsmT[NUMBER_RSM - 1].Finalize(&DEMODevice);
    g_rsmTV[NUMBER_RSM - 1].Finalize(&DEMODevice);
    g_rsmData[NUMBER_RSM - 1]->Finalize();
    delete g_rsmData[NUMBER_RSM - 1];

    rsmTempleVertexUniformBuffer.Finalize();
    rsmTerrainVertexUniformBuffer.Finalize();
    rsmPipeline.Finalize(&DEMODevice);

    g_illumColor.Finalize(&DEMODevice);
    g_illumTV.Finalize(&DEMODevice);
    g_illumT.Finalize(&DEMODevice);
    g_illumData->Finalize();
    delete g_illumData;
    ilPipeline.Finalize(&DEMODevice);

    material.Finalize( &DEMODevice );
    materialAlpha.Finalize( &DEMODevice );

    viewerTexCoordBuffer.Finalize();
    viewerPositionBuffer.Finalize();
    viewerIndexBuffer.Finalize();

    for (int i = 0; i < NUMBER_RSM; i++)
    {
        viewerViewport[i].Finalize(&DEMODevice);
        viewerPixelUniformBuffer[i].Finalize(&DEMODevice);
    }
    viewerPipeline.Finalize(&DEMODevice);

    ilPixelUniformBuffer.Finalize(&DEMODevice);

    ilViewport.Finalize(&DEMODevice);
    rsmViewport.Finalize(&DEMODevice);
    fullViewport.Finalize(&DEMODevice);

    g_samplerViewer.Finalize(&DEMODevice);
    g_samplerModel.Finalize(&DEMODevice);
    g_samplerRSM.Finalize(&DEMODevice);

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
