﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>

#include "textureHandle.h"
#include "simpleParticle.h"

// Particle vertex number.
static const u32 VERTEX_NUM = MAX_PARTICLE * 6;

// Simple particle shader.
extern int g_ShaderFileIdx;
static const char* const SIMPLE_PARTICLE[] =
{
    "shaders/gpuParticle/simpleParticle",
    "shaders/gpuParticle/simpleParticleHlslcc",
};

typedef struct SIMPLE_PARTICLE_UNIFORM
{
    Mtx44 u_viewMtx;
    Mtx44 u_viewInvMtx;
    Mtx44 u_projMtx;

    // Model matrix.
    Mtx44 u_modelMtx;

    // Bounding boxes.
    Mtx44 u_bndBoxPos;
    Mtx44 u_bndBoxSize;

    // Node color.
    Mtx44 u_nodeColor;

    // Loop attribute (x: loop length, y: current time, z: unused, w: particle scale).
    float u_loopAttr[4];

    // Texture attribute.
    float u_textureSize[4];

    // Pattern constant (x: row, y: column, z: num. of pattern, w: unused).
    uint32_t u_patternConst[4];
} TSimpleParticleUniform;
static DEMOGfxBuffer s_SimpleParticleUniforms[ 2 ];
static int s_SimpleParticleBuffer = 0;
static int s_SimpleParticleUniformsLoc = -1;

typedef struct SIMPLE_PIXEL_UNIFORM
{
    float u_alphaTestValue[ 4 ];
} TSimplePixelUniform;
static DEMOGfxBuffer s_SimplePixelUniform[ 2 ];
static int s_SimplePixelUniformLoc = -1;

static int s_SimpleParticleBaseMapLoc = -1;

// Simple particle attribute.
enum SIMPLE_PARTICLE_ATTRIB
{
    SIMPLE_PARTICLE_VPOS = 0,
    SIMPLE_PARTICLE_VCOLOR,
    SIMPLE_PARTICLE_VSCALE,
    SIMPLE_PARTICLE_VLIFE,
    SIMPLE_PARTICLE_VATTR0,
    SIMPLE_PARTICLE_VATTR1,
    SIMPLE_PARTICLE_MAX_ATTR,
};

// Sampler.
static DEMOGfxTexture* s_pBaseMap = NULL;
static nn::gfx::Sampler s_BaseMapSampler;
static nn::gfx::DescriptorSlot s_BaseMapSamplerSlot;

// Vertices.
typedef struct SIMPLE_PARTICLE_VTX
{
    f32 pos[3];
    u32 color;
    u8 scale[4];
    u16 life[2];
    u8 attr0[4];
    u16 attr1[2];
} TSimpleParticle;
static DEMOGfxBuffer s_SimpleParticleVtx;

// Model matrix.
static Mtx44 s_ModelMatrix44;

// Bounding box.
static f32 s_BoundingBox[4][4];

// Bounding box size.
static f32 s_BoundingBoxSize[4][4];

// Node color.
static f32 s_NodeColor[4][4];

// Blend state.
enum BLEND_STATE
{
    BLEND_NONE = 0,
    BLEND_ALPHA,
    BLEND_ADD,
};
static BLEND_STATE s_BlendState = BLEND_NONE;
static DEMOGfxPipeline s_SimpleParticlePipeline;
static nn::gfx::BlendState s_SimpleParticleBlendStates[BLEND_ADD + 1];
static nn::gfx::DepthStencilState s_SimpleParticleDepthStencilStates[BLEND_ADD + 1];
static void* s_pSimpleParticleBlendStateData[ BLEND_ADD + 1 ];

// Alpha test.
static bool s_AlphaTest = false;

// Texture attributes.
static f32 s_TextureSize[4];
static s32 s_PatternAttr[4];

// Particle scale.
static f32 s_ParticleScale = 1.0f;

void InitSimpleParticlePipelines()
{
    s_SimpleParticlePipeline.SetDefaults();

    // Set Common State
    s_SimpleParticlePipeline.colorTargetStateCount = 1;
    s_SimpleParticlePipeline.blendTargetStateCount = 1;
    s_SimpleParticlePipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDefault();

    s_SimpleParticlePipeline.depthStencilStateInfo.SetDepthTestEnabled( true );
    s_SimpleParticlePipeline.depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_LessEqual );

    s_SimpleParticlePipeline.Initialize( &DEMODevice );

    // BLEND_NONE state
    {
        s_SimpleParticlePipeline.depthStencilStateInfo.SetDepthWriteEnabled( true );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetBlendEnabled( false );
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( s_SimpleParticlePipeline.blendStateInfo );
        s_pSimpleParticleBlendStateData[ BLEND_NONE ] = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        s_SimpleParticleBlendStates[ BLEND_NONE ].SetMemory( s_pSimpleParticleBlendStateData[ BLEND_NONE ], size );
        s_SimpleParticleBlendStates[ BLEND_NONE ].Initialize( &DEMODevice, s_SimpleParticlePipeline.blendStateInfo );
        s_SimpleParticleDepthStencilStates[ BLEND_NONE ].Initialize( &DEMODevice, s_SimpleParticlePipeline.depthStencilStateInfo );
    }

    // BLEND_ADD state
    {
        s_SimpleParticlePipeline.depthStencilStateInfo.SetDepthWriteEnabled( false );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetBlendEnabled( true );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_One );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_One );
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( s_SimpleParticlePipeline.blendStateInfo );
        s_pSimpleParticleBlendStateData[ BLEND_ADD ] = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        s_SimpleParticleBlendStates[ BLEND_ADD ].SetMemory( s_pSimpleParticleBlendStateData[ BLEND_ADD ], size );
        s_SimpleParticleBlendStates[ BLEND_ADD ].Initialize( &DEMODevice, s_SimpleParticlePipeline.blendStateInfo );
        s_SimpleParticleDepthStencilStates[ BLEND_ADD ].Initialize( &DEMODevice, s_SimpleParticlePipeline.depthStencilStateInfo );
    }

    // BLEND_ALPHA state
    {
        s_SimpleParticlePipeline.depthStencilStateInfo.SetDepthWriteEnabled( false );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetBlendEnabled( true );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        s_SimpleParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( s_SimpleParticlePipeline.blendStateInfo );
        s_pSimpleParticleBlendStateData[ BLEND_ALPHA ] = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        s_SimpleParticleBlendStates[ BLEND_ALPHA ].SetMemory( s_pSimpleParticleBlendStateData[ BLEND_ALPHA ], size );
        s_SimpleParticleBlendStates[ BLEND_ALPHA ].Initialize( &DEMODevice, s_SimpleParticlePipeline.blendStateInfo );
        s_SimpleParticleDepthStencilStates[ BLEND_ALPHA ].Initialize( &DEMODevice, s_SimpleParticlePipeline.depthStencilStateInfo );
    }
}

// Initialize simple particle.
void InitSimpleParticle()
{
    DEMOGfxLoadShadersFromFile(&s_SimpleParticlePipeline.shaders, 0, SIMPLE_PARTICLE[g_ShaderFileIdx]);

    // Get uniform locations.
    s_SimpleParticleBaseMapLoc =
        s_SimpleParticlePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_basemap" );

    // Initialize attibute stream.
    {
        u32 offset = 0;
        DEMOGfxInitShaderAttribute( &s_SimpleParticlePipeline.shaders, "a_position", 0, offset, nn::gfx::AttributeFormat_32_32_32_Float );
        offset += 3 * sizeof(f32);

        DEMOGfxInitShaderAttribute( &s_SimpleParticlePipeline.shaders, "a_color", 0, offset, nn::gfx::AttributeFormat_8_8_8_8_Unorm );
        offset += 4 * sizeof(u8);

        DEMOGfxInitShaderAttribute( &s_SimpleParticlePipeline.shaders, "a_scale", 0, offset, nn::gfx::AttributeFormat_8_8_8_8_Unorm );
        offset += 4 * sizeof(u8);

        DEMOGfxInitShaderAttribute( &s_SimpleParticlePipeline.shaders, "a_life", 0, offset, nn::gfx::AttributeFormat_16_16_Uint );
        offset += 2 * sizeof(u16);

        DEMOGfxInitShaderAttribute( &s_SimpleParticlePipeline.shaders, "a_attr0", 0, offset, nn::gfx::AttributeFormat_8_8_8_8_Uint );
        offset += 4 * sizeof(u8);

        DEMOGfxInitShaderAttribute( &s_SimpleParticlePipeline.shaders, "a_attr1", 0, offset, nn::gfx::AttributeFormat_16_16_Uint );

        DEMOGfxInitShaderVertexBuffer( &s_SimpleParticlePipeline.shaders, 0, sizeof( TSimpleParticle ), 0 );
    }

    // Initialize sampler.
    DEMOGfxInitSampler( &s_BaseMapSampler, &s_BaseMapSamplerSlot,
        nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear,
        nn::gfx::ComparisonFunction_Always );

    // Create & set-up vertex buffer.
    // Simple particle requires 4 vertices in quad-order.
    s_SimpleParticleVtx.Initialize( sizeof( TSimpleParticle ) * VERTEX_NUM, NULL,  nn::gfx::GpuAccess_VertexBuffer, 0 );

    TSimpleParticle* pvtx = s_SimpleParticleVtx.Map< TSimpleParticle >();

    // Reset random number.
    DEMOSRand(1);
    for (u32 ui = 0; ui < MAX_PARTICLE; ui++)
    {
        // Vertex position [-1.0..1.0].
        // This position is relative to bounding box position and size. Assigning 1.0 or -1.0
        // will make vertex positioned at the farthest point of particular axis in the bounding
        // box.
        f32 posx = (static_cast< f32 >(DEMORand() * 2) / static_cast< f32 >(DEMO_RAND_MAX)) - 1.0f;
        f32 posy = (static_cast< f32 >(DEMORand() * 2) / static_cast< f32 >(DEMO_RAND_MAX)) - 1.0f;
        f32 posz = (static_cast< f32 >(DEMORand() * 2) / static_cast< f32 >(DEMO_RAND_MAX)) - 1.0f;

        // Particle birth time and life.
        // Birth time assumes the first frame a particle being displayed.
        u16 birth = DEMORand() % 100;
        // Life time assumes the number of frames required to travel from the first node to
        // the last node. Smaller value will make a particle moves faster.
        u16 life = (DEMORand() % 500) + 60;

        // Particle size.
        u8 particle_size = DEMORand() % 3 + 2;
        // Time shift, used to decrease regularity for particles with similar property.
        u8 time_shift = DEMORand() % 256;
        // Flip animation start index and end index.
        u8 animation_start = 0;
        u8 animation_end = 255;

        // Rotation (in degrees).
        u16 rotation_start = DEMORand() % 360;
        u16 rotation_end = DEMORand() % 360;

        pvtx[0].pos[0] = posx;
        pvtx[0].pos[1] = posy;
        pvtx[0].pos[2] = posz;
        // Vertex color (vertex color will be multiplied with fragment color).
        pvtx[0].color = 0xffffffff;
        // Particle scale, used to calculate position in vertex shader.
        pvtx[0].scale[0] = 0x0;
        pvtx[0].scale[1] = 0xff;
        pvtx[0].life[0] = birth;
        pvtx[0].life[1] = life;
        pvtx[0].attr0[0] = particle_size;
        pvtx[0].attr0[1] = time_shift;
        pvtx[0].attr0[2] = animation_start;
        pvtx[0].attr0[3] = animation_end;
        pvtx[0].attr1[0] = rotation_start;
        pvtx[0].attr1[1] = rotation_end;

        pvtx[1].pos[0] = posx;
        pvtx[1].pos[1] = posy;
        pvtx[1].pos[2] = posz;
        pvtx[1].color = 0xffffffff;
        pvtx[1].scale[0] = 0x0;
        pvtx[1].scale[1] = 0x0;
        pvtx[1].life[0] = birth;
        pvtx[1].life[1] = life;
        pvtx[1].attr0[0] = particle_size;
        pvtx[1].attr0[1] = time_shift;
        pvtx[1].attr0[2] = animation_start;
        pvtx[1].attr0[3] = animation_end;
        pvtx[1].attr1[0] = rotation_start;
        pvtx[1].attr1[1] = rotation_end;

        pvtx[2].pos[0] = posx;
        pvtx[2].pos[1] = posy;
        pvtx[2].pos[2] = posz;
        pvtx[2].color = 0xffffffff;
        pvtx[2].scale[0] = 0xff;
        pvtx[2].scale[1] = 0x0;
        pvtx[2].life[0] = birth;
        pvtx[2].life[1] = life;
        pvtx[2].attr0[0] = particle_size;
        pvtx[2].attr0[1] = time_shift;
        pvtx[2].attr0[2] = animation_start;
        pvtx[2].attr0[3] = animation_end;
        pvtx[2].attr1[0] = rotation_start;
        pvtx[2].attr1[1] = rotation_end;

        pvtx[3].pos[0] = posx;
        pvtx[3].pos[1] = posy;
        pvtx[3].pos[2] = posz;
        pvtx[3].color = 0xffffffff;
        pvtx[3].scale[0] = 0xff;
        pvtx[3].scale[1] = 0x0;
        pvtx[3].life[0] = birth;
        pvtx[3].life[1] = life;
        pvtx[3].attr0[0] = particle_size;
        pvtx[3].attr0[1] = time_shift;
        pvtx[3].attr0[2] = animation_start;
        pvtx[3].attr0[3] = animation_end;
        pvtx[3].attr1[0] = rotation_start;
        pvtx[3].attr1[1] = rotation_end;

        pvtx[4].pos[0] = posx;
        pvtx[4].pos[1] = posy;
        pvtx[4].pos[2] = posz;
        pvtx[4].color = 0xffffffff;
        pvtx[4].scale[0] = 0xff;
        pvtx[4].scale[1] = 0xff;
        pvtx[4].life[0] = birth;
        pvtx[4].life[1] = life;
        pvtx[4].attr0[0] = particle_size;
        pvtx[4].attr0[1] = time_shift;
        pvtx[4].attr0[2] = animation_start;
        pvtx[4].attr0[3] = animation_end;
        pvtx[4].attr1[0] = rotation_start;
        pvtx[4].attr1[1] = rotation_end;

        pvtx[5].pos[0] = posx;
        pvtx[5].pos[1] = posy;
        pvtx[5].pos[2] = posz;
        pvtx[5].color = 0xffffffff;
        pvtx[5].scale[0] = 0x0;
        pvtx[5].scale[1] = 0xff;
        pvtx[5].life[0] = birth;
        pvtx[5].life[1] = life;
        pvtx[5].attr0[0] = particle_size;
        pvtx[5].attr0[1] = time_shift;
        pvtx[5].attr0[2] = animation_start;
        pvtx[5].attr0[3] = animation_end;
        pvtx[5].attr1[0] = rotation_start;
        pvtx[5].attr1[1] = rotation_end;

        // Write 6 vertices.
        pvtx += 6;
    }

    // Send vertex data to GPU.
    s_SimpleParticleVtx.Unmap();

    SimpleParticleMode(SNOW);

    // Initialize model matrix.
    MTX44Identity(s_ModelMatrix44);

    // Initialize the vertex uniform blocks
    s_SimpleParticleUniforms[ 0 ].Initialize( sizeof( TSimpleParticleUniform ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    s_SimpleParticleUniforms[ 1 ].Initialize( sizeof( TSimpleParticleUniform ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    s_SimpleParticleUniformsLoc = s_SimpleParticlePipeline.shaders.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex,
        nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_simpleParticle" );

    // Initialize Alpha Test
    s_SimplePixelUniform[ 0 ].Initialize( sizeof( TSimplePixelUniform ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    s_SimplePixelUniform[ 1 ].Initialize( sizeof( TSimplePixelUniform ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    s_SimplePixelUniformLoc = s_SimpleParticlePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_alpha" );

    // Initialize the pipelines
    InitSimpleParticlePipelines();
} // NOLINT(impl/function_size)

// Update simple particle.
void DrawSimpleParticle(Mtx44 viewMatrix, Mtx44 projMatrix)
{
    f32 loopAttr[4] = { loopLength, currentLoop, 0.0f, s_ParticleScale };
    u32 vnum = currentParticleNum * 4;
    Mtx44 invView;
    MTX44Inverse(viewMatrix, invView);

    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ConstantBuffer );

    // Set-up uniforms.
    s_SimpleParticleBuffer ^= 1;
    TSimpleParticleUniform* pUniforms = s_SimpleParticleUniforms[ s_SimpleParticleBuffer ].Map< TSimpleParticleUniform >();
    memcpy( pUniforms->u_viewMtx, viewMatrix, sizeof( Mtx44 ) );
    memcpy( pUniforms->u_viewInvMtx, invView, sizeof( Mtx44 ) );
    memcpy( pUniforms->u_projMtx, projMatrix, sizeof( Mtx44 ) );
    memcpy( pUniforms->u_modelMtx, s_ModelMatrix44, sizeof( Mtx44 ) );
    memcpy( pUniforms->u_bndBoxPos, s_BoundingBox, sizeof( Mtx44 ) );
    memcpy( pUniforms->u_bndBoxSize, s_BoundingBoxSize, sizeof( Mtx44 ) );
    memcpy( pUniforms->u_nodeColor, s_NodeColor, sizeof( Mtx44 ) );
    memcpy( pUniforms->u_loopAttr, loopAttr, 4 * sizeof( float ) );
    memcpy( pUniforms->u_textureSize, s_TextureSize, sizeof( s_TextureSize ) );
    memcpy( pUniforms->u_patternConst, s_PatternAttr, sizeof( s_PatternAttr ) );

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pUniforms, sizeof( *pUniforms ) );
#endif
    s_SimpleParticleUniforms[ s_SimpleParticleBuffer ].Unmap();

    DEMOCommandBuffer.SetConstantBuffer(s_SimpleParticleUniformsLoc,
        nn::gfx::ShaderStage_Vertex,
        s_SimpleParticleUniforms[ s_SimpleParticleBuffer ].gpuAddress, s_SimpleParticleUniforms[ s_SimpleParticleBuffer ].size );

    // Set-up alpha test state.
    TSimplePixelUniform* pPixelUniform = s_SimplePixelUniform[ s_SimpleParticleBuffer ].Map< TSimplePixelUniform >();
    pPixelUniform->u_alphaTestValue[ 0 ] = ( s_AlphaTest ? 0.5f : -1.0f );
    pPixelUniform->u_alphaTestValue[ 1 ] = 0.0f;
    pPixelUniform->u_alphaTestValue[ 2 ] = 0.0f;
    pPixelUniform->u_alphaTestValue[ 3 ] = 0.0f;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pPixelUniform, sizeof( *pPixelUniform ) );
#endif
    s_SimplePixelUniform[ s_SimpleParticleBuffer ].Unmap();

    DEMOCommandBuffer.SetConstantBuffer(s_SimplePixelUniformLoc,
        nn::gfx::ShaderStage_Pixel,
        s_SimplePixelUniform[ s_SimpleParticleBuffer ].gpuAddress, s_SimplePixelUniform[ s_SimpleParticleBuffer ].size );

    // Set-up vertex attributes & vertex buffer.
    DEMOCommandBuffer.SetVertexBuffer( 0, s_SimpleParticleVtx.gpuAddress, sizeof( TSimpleParticle ), s_SimpleParticleVtx.size );

    // Set-up texture.
    DEMOCommandBuffer.SetTextureAndSampler( s_SimpleParticleBaseMapLoc, nn::gfx::ShaderStage_Pixel, s_pBaseMap->GetDescriptorSlot( 0 ), s_BaseMapSamplerSlot );

    // Setup render state
    DEMOCommandBuffer.SetPipeline( &s_SimpleParticlePipeline.pipeline );
    DEMOCommandBuffer.SetDepthStencilState( &s_SimpleParticleDepthStencilStates[ s_BlendState ] );
    DEMOCommandBuffer.SetBlendState( &s_SimpleParticleBlendStates[ s_BlendState ] );

    // Draw quads (each particle needs 4 vertices).
    DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_TriangleList, vnum, 0 );
}

// Terminate simple particle.
void TermSimpleParticle()
{
    // Free pipelines and blend/depth states
    s_SimpleParticleBlendStates[ BLEND_NONE ].Finalize( &DEMODevice );
    s_SimpleParticleBlendStates[ BLEND_ADD ].Finalize( &DEMODevice );
    s_SimpleParticleBlendStates[ BLEND_ALPHA ].Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( s_pSimpleParticleBlendStateData[ BLEND_NONE ] );
    DEMOGfxFreeMEM2( s_pSimpleParticleBlendStateData[ BLEND_ADD ] );
    DEMOGfxFreeMEM2( s_pSimpleParticleBlendStateData[ BLEND_ALPHA ] );

    s_SimpleParticleDepthStencilStates[ BLEND_NONE ].Finalize( &DEMODevice );
    s_SimpleParticleDepthStencilStates[ BLEND_ADD ].Finalize( &DEMODevice );
    s_SimpleParticleDepthStencilStates[ BLEND_ALPHA ].Finalize( &DEMODevice );

    s_SimpleParticlePipeline.Finalize( &DEMODevice );

    // Free buffers.
    s_SimpleParticleVtx.Finalize();

    s_SimpleParticleUniforms[ 0 ].Finalize();
    s_SimpleParticleUniforms[ 1 ].Finalize();

    s_SimplePixelUniform[ 0 ].Finalize();
    s_SimplePixelUniform[ 1 ].Finalize();

    s_BaseMapSampler.Finalize( &DEMODevice );
}

// Set simple particle mode.
void SimpleParticleMode(PARTICLE_MODE mode)
{
    switch (mode)
    {
    case SNOW:
        // Snow/blizzard showcase.
        s_pBaseMap = &g_SnowTex;
        s_BlendState = BLEND_ALPHA;
        s_AlphaTest = false;

        // Texture size: 32*32 pixels.
        s_TextureSize[0] = 32.0f;
        s_TextureSize[1] = 32.0f;
        s_TextureSize[2] = 1.0f / 32.0f;
        s_TextureSize[3] = 1.0f / 32.0f;
        // 1 row, 1 column, 1 pattern.
        s_PatternAttr[0] = 1;
        s_PatternAttr[1] = 1;
        s_PatternAttr[2] = 1;
        // Unused.
        s_PatternAttr[3] = 0;

        // Particle scale, scales each particle equally.
        s_ParticleScale = 0.2f;

        // Node information.
        {
            // Position for bounding box 1-4.
            const f32 bb_pos[4][4] = {
                { -40.0f, 100.0f, 0.0f, 1.0f },
                { 20.0f, 60.0f, 0.0f, 1.0f },
                { -30.0f, 50.0f, 0.0f, 1.0f },
                { 0.0f, 0.0f, 0.0f, 1.0f },
            };
            // Bounding box size (x: width, y: height, z: depth, w: unused).
            const f32 bb_size[4][4] = {
                { 250.0f, 20.0f, 250.0f, 0.0f },
                { 250.0f, 1.0f, 250.0f, 0.0f },
                { 250.0f, 1.0f, 250.0f, 0.0f },
                { 400.0f, 1.0f, 400.0f, 0.0f },
            };
            // Bounding box color (RGBA)
            const f32 n_color[4][4] = {
                { 0.25f, 0.25f, 0.25f, 1.0f },
                { 0.8f, 0.8f, 0.8f, 1.0f },
                { 1.0f, 1.0f, 1.0f, 1.0f },
                { 0.3f, 0.3f, 0.3f, 1.0f },
            };
            // Assign to node informations.
            memcpy(s_BoundingBox, bb_pos, sizeof(f32) * 4 * 4);
            memcpy(s_BoundingBoxSize, bb_size, sizeof(f32) * 4 * 4);
            memcpy(s_NodeColor, n_color, sizeof(f32) * 4 * 4);
        }

        break;
    case SAKURA:
        // Sakura showcase.
        s_pBaseMap = &g_SakuraTex;
        s_BlendState = BLEND_NONE;
        s_AlphaTest = true;

        // Texture size: 1024*1024 pixels.
        s_TextureSize[0] = 1024.0f;
        s_TextureSize[1] = 1024.0f;
        s_TextureSize[2] = 1.0f / 1024.0f;
        s_TextureSize[3] = 1.0f / 1024.0f;
        // 16 rows, 16 columns, 256 patterns.
        s_PatternAttr[0] = 16;
        s_PatternAttr[1] = 16;
        s_PatternAttr[2] = 256;
        // Unused.
        s_PatternAttr[3] = 0;

        // Particle scale.
        s_ParticleScale = 0.5f;

        // Node information.
        {
            const f32 bb_pos[4][4] = {
                { -40.0f, 100.0f, 0.0f, 1.0f },
                { 20.0f, 80.0f, 0.0f, 1.0f },
                { 0.0f, 40.0f, 0.0f, 1.0f },
                { 40.0f, 0.0f, 0.0f, 1.0f },
            };
            const f32 bb_size[4][4] = {
                { 250.0f, 40.0f, 250.0f, 0.0f },
                { 250.0f, 20.0f, 250.0f, 0.0f },
                { 250.0f, 50.0f, 250.0f, 0.0f },
                { 250.0f, 10.0f, 250.0f, 0.0f },
            };
            const f32 n_color[4][4] = {
                { 0.50f, 0.25f, 0.25f, 1.0f },
                { 0.9f, 0.75f, 0.75f, 1.0f },
                { 1.0f, 0.75f, 0.75f, 1.0f },
                { 0.5f, 0.3f, 0.3f, 1.0f },
            };
            memcpy(s_BoundingBox, bb_pos, sizeof(f32) * 4 * 4);
            memcpy(s_BoundingBoxSize, bb_size, sizeof(f32) * 4 * 4);
            memcpy(s_NodeColor, n_color, sizeof(f32) * 4 * 4);
        }

        break;
    case MIST:
        // Mist showcase.
        s_pBaseMap = &g_MistTex;
        s_BlendState = BLEND_ADD;
        s_AlphaTest = false;

        // Texture size.
        s_TextureSize[0] = 1024.0f;
        s_TextureSize[1] = 1024.0f;
        s_TextureSize[2] = 1.0f / 1024.0f;
        s_TextureSize[3] = 1.0f / 1024.0f;
        // 8 rows, 8 columns, 32 patterns.
        s_PatternAttr[0] = 8;
        s_PatternAttr[1] = 8;
        s_PatternAttr[2] = 32;
        // Unused.
        s_PatternAttr[3] = 0;

        // Particle scale.
        s_ParticleScale = 5.0f;

        // Node information.
        {
            const f32 bb_pos[4][4] = {
                { 0.0f, 50.0f, -200.0f, 1.0f },
                { 0.0f, 50.0f, -100.0f, 1.0f },
                { 0.0f, 50.0f, 100.0f, 1.0f },
                { 0.0f, 50.0f, 200.0f, 1.0f },
            };
            const f32 bb_size[4][4] = {
                { 250.0f, 250.0f, 100.0f, 0.0f },
                { 250.0f, 250.0f, 100.0f, 0.0f },
                { 250.0f, 250.0f, 100.0f, 0.0f },
                { 250.0f, 250.0f, 200.0f, 0.0f },
            };
            const f32 n_color[4][4] = {
                { 0.8f, 0.8f, 0.8f, 0.0f },
                { 0.4f, 0.4f, 0.4f, 1.0f },
                { 0.4f, 0.4f, 0.4f, 1.0f },
                { 0.8f, 0.8f, 0.8f, 0.0f },
            };
            memcpy(s_BoundingBox, bb_pos, sizeof(f32) * 4 * 4);
            memcpy(s_BoundingBoxSize, bb_size, sizeof(f32) * 4 * 4);
            memcpy(s_NodeColor, n_color, sizeof(f32) * 4 * 4);
        }

        break;
    default:
        break;
    }
} // NOLINT(impl/function_size)

// Vertex buffer size.
u32 SimpleParticleVtxBufSize()
{
    // 1 particle is a quad (4 vertices).
    return currentParticleNum * sizeof(TSimpleParticle) * 4;
}
