﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/result/result_ErrorResult.h>
#include <nn/util/util_Color.h>

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#include "Graphics.h"
#include <nvn/nvn_FuncPtrInline.h>

namespace Graphics {

bool CheckGpuResource::MeasureMaxGpuLoad(int* objectCount, int* liteObjectCount) NN_NOEXCEPT
{
    // ギリギリ処理落ちしない大オブジェクトの個数を決定する
    if (!isDefineLargeObjectCount)
    {
        if (MeasureMaxGpuLoadImpl(objectCount, false))
        {
            isDefineLargeObjectCount = true;
            m_StableFrameCount = 0;
            m_GpuLoadSum = 0;
            m_GpuDelaySum  = 0;
            m_StableFrameCount = 0;
            m_ObjectCountMax = 999;
        }
    }
    // ギリギリ処理落ちしない小オブジェクトの個数を決定する
    else
    {
        if (MeasureMaxGpuLoadImpl(liteObjectCount, true))
        {
            return true;
        }
    }
    return false;
}

bool CheckGpuResource::MeasureMaxGpuLoad(int* liteObjectCount) NN_NOEXCEPT
{
    // ギリギリ処理落ちしない小オブジェクトの個数を決定する
    if (MeasureMaxGpuLoadImpl(liteObjectCount, true))
    {
        return true;
    }
    return false;
}

bool CheckGpuResource::MeasureMaxGpuLoadImpl(int* objectCount, bool enableShowResult) NN_NOEXCEPT
{
    nn::perf::CpuMeter* pCpuMeter = NN_PERF_GET_FRAME_METER();
    float fps = 1000.f / (static_cast<float>(pCpuMeter->GetLastTotalSpan().GetNanoSeconds()) / 1000000.0f);
    nn::perf::GpuMeter* pGpuMeter = NN_PERF_GET_GPU_METER();
    nn::perf::LoadMeterBase::Section gpuResult = pGpuMeter->GetLastResult(0);
    int64_t gpuLoad = gpuResult.end.ToTimeSpan().GetMicroSeconds() - gpuResult.begin.ToTimeSpan().GetMicroSeconds();
    static int intervalFrame = 0;
    static int largeDifferenceCount = 0;
    static int objectCountMax = 0;

    // 一度処理落ちした場合、連続して処理落ちすることがあるため 40フレームほど何もしない時間を設ける
    if (intervalFrame > 0)
    {
        intervalFrame--;
    }
    else
    {
        // 処理落ちした場合、もしくは 20 秒間 60fps をキープできた場合
        if (fps < m_FpsThreshold || gpuLoad > 16000 + 16666 * (m_PresentInterval - 1) || m_StableFrameCount >= 1200 / m_PresentInterval)
        {
            if (*objectCount > 0)
            {
                if (m_StableFrameCount == 0)
                {
                    NN_LOG("objectCountMax=%d(act %d), stableFrame=%d\n", m_ObjectCountMax, *objectCount, m_StableFrameCount);
                }
                else
                {
                    NN_LOG("objectCountMax=%d(act %d), stableFrame=%d, gpuAverage=%dus\n", m_ObjectCountMax, *objectCount, m_StableFrameCount, m_GpuLoadSum / m_StableFrameCount);
                    // 同じ GPU コマンドで 20秒間 60fps をキープできたら終了する
                    if (m_StableFrameCount >= 1200 / m_PresentInterval)
                    {
                        if (enableShowResult)
                        {
                            m_GpuLoadAverage = m_GpuLoadSum / m_StableFrameCount;
                            m_GpuDelayAverage = m_GpuDelaySum / m_StableFrameCount;
                            NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
                            NN_LOG("         MAXGPU = %lld         \n", m_GpuLoadAverage);
                            NN_LOG("         GPUDELAY = %lld       \n", m_GpuDelayAverage);
                            NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
                        }
                        return true;
                    }
                }
                // 目標オブジェクト数との差が大きい場合は、目標オブジェクト数を大幅に小さくする。
                // 稀に不当に処理落ちするため、連続して4回処理落ちした場合のみ適用する。
                // 処理落ちした際の最大オブジェクト数+15個に減らす。稀に不当に少ないオブジェクト数で処理落ちするため最大を取る。
                if (m_ObjectCountMax - *objectCount > 15)
                {
                    largeDifferenceCount++;
                    objectCountMax = std::max(*objectCount, objectCountMax);
                    if (largeDifferenceCount > 4)
                    {
                        m_ObjectCountMax = objectCountMax + 15;
                        largeDifferenceCount = 0;
                        objectCountMax = 0;
                    }
                }
                else
                {
                    largeDifferenceCount = 0;
                    objectCountMax = 0;
                }
                m_ObjectCountMax--;
                m_GpuLoadSum = 0;
                m_GpuDelaySum = 0;
                *objectCount = 0;
                m_StableFrameCount = 0;
                intervalFrame = 40;
            }
        }
        // 目標の負荷まで増加させる（GPU のスリープのため 徐々に増加させる必要がある）
        else if (*objectCount < m_ObjectCountMax)
        {
            *objectCount = *objectCount + 1;
        }
        // 目標の負荷をキープしている場合。処理落ちしないフレーム数と GPU 負荷の合計値を計測する
        else
        {
            m_GpuLoadSum += gpuResult.end.ToTimeSpan().GetMicroSeconds() - gpuResult.begin.ToTimeSpan().GetMicroSeconds();
            m_GpuDelaySum += pGpuMeter->GetLastResult(0).begin.ToTimeSpan().GetMicroSeconds() - pCpuMeter->GetLastResult(0).begin.ToTimeSpan().GetMicroSeconds();
            m_StableFrameCount++;
        }

        if (m_ObjectCountMax <= 0)
        {
            *objectCount = 0;
            m_ObjectCountMax = 0;
            return true;
        }
    }
    return false;
}

void CheckGpuResource::MaximizeGpuLoad(int* objectCount)
{
    nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
    float fps = 1000.f / (static_cast<float>(pFrameMeter->GetLastTotalSpan().GetNanoSeconds()) / 1000000.0f);
    nn::perf::GpuMeter* pGpuMeter = NN_PERF_GET_GPU_METER();
    nn::perf::LoadMeterBase::Section mainSection = pGpuMeter->GetLastResult(0);
    uint64_t gpuLoad = mainSection.end.ToTimeSpan().GetMicroSeconds() - mainSection.begin.ToTimeSpan().GetMicroSeconds();

    static int largeDifferenceCount = 0;

    if (fps < m_FpsThreshold || gpuLoad > 16666 * m_PresentInterval)
    {
        // 目標オブジェクト数との差が大きい場合は、目標オブジェクト数を大幅に小さくする。
        // 稀に不当に処理落ちするため、連続して処理落ちした場合のみ適用する。
        if (m_ObjectCountMax - *objectCount > 5)
        {
            largeDifferenceCount++;
            if (largeDifferenceCount > 4)
            {
                m_ObjectCountMax = *objectCount + 5;
                largeDifferenceCount = 0;
            }
        }
        m_ObjectCountMax--;
        *objectCount = *objectCount - 1;
    }

    // GPU 負荷を高くする
    if (*objectCount < m_ObjectCountMax)
    {
        *objectCount = *objectCount + 1;
    }
}

namespace {

// クリアカラー
Rgba g_ClearColor;

// タイトル
std::string g_Title;

// フレームワークモード
FrameworkMode g_CurrentFrameworkMode;
TestMode g_CurrentTestMode;

// 描画するオブジェクト数
int g_ObjectCount = 0;
int g_LiteObjectCount = 0;

// バッファ数 (2 以上を指定してください)
const int g_BufferCount = 2;

// OA の表示のオンオフ
bool g_IsSuspendGraphicsThread = false;
bool g_IsPrepareSuspend = false;

int g_PresentInterval = 1;
CheckGpuResource checkGpuResouce;
bool g_IsProfiling = true;

//-----------------------------------------------------------------------------
// グラフィックスフレームワークを初期化
nns::gfx::GraphicsFramework g_GraphicsFramework;
nns::gfx::GraphicsFramework::DebugFontTextWriter g_Writer;
static int g_BufferIndex = 0;
void InitializeGraphicsFramework()
{
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetDepthStencilBufferEnabled(false);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, 8 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,  12 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, 3 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Shader, 0 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Data, 1 * 1024 * 1024);
    fwInfo.SetRootCommandBufferCommandMemorySize(2 * 1024 * 1024);
    fwInfo.SetBufferCount(g_BufferCount);
    fwInfo.SetSwapChainBufferCount(g_BufferCount);
    g_GraphicsFramework.Initialize(fwInfo);
}

// グラフィックスフレームワークを破棄
void FinalizeGraphicsFramework()
{
    g_GraphicsFramework.Finalize();
}

// プリミティブレンダラの初期化
nn::mem::StandardAllocator  g_PrimitiveRendererAllocator;
nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
void InitializePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(g_GraphicsFramework.DefaultAllocateFunction, NULL);
    info.SetAdditionalBufferSize(1024 * 4);
    info.SetDrawCallCountMax(1024 * 4);
    info.SetViewFunctionCallCountMax(1024 * 4);
    info.SetMultiBufferQuantity(g_BufferCount);

    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(g_GraphicsFramework.GetDevice(), info);
    g_pPrimitiveRenderer->SetScreenWidth(g_GraphicsFramework.GetDisplayWidth());
    g_pPrimitiveRenderer->SetScreenHeight(g_GraphicsFramework.GetDisplayHeight());
}

// プリミティブレンダラの破棄
void FinalizePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::DestroyRenderer(g_pPrimitiveRenderer, g_GraphicsFramework.GetDevice(), g_GraphicsFramework.DefaultFreeFunction, NULL);
}

// パフォーマンス計測クラスの初期化
nns::gfx::PrimitiveRenderer::MeterDrawer g_MeterDrawer;
void*  g_MeterMemory;
nn::gfx::MemoryPool g_MeterMemoryPool;
void* g_pMeterPoolMemory;
ptrdiff_t g_MeterMemoryPoolOffset;
void InitializeLoadMeter()
{
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(4);
    if ((g_CurrentFrameworkMode == FrameworkMode_QueueFinish) || (g_CurrentFrameworkMode == FrameworkMode_Immediate))
    {
        info.SetGpuBufferCount(2);
    }
    else
    {
        info.SetGpuBufferCount(3);
    }
    info.SetCpuSectionCountMax(16);
    info.SetGpuSectionCountMax(16);

    // 計測で使用するメモリの確保
    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    g_MeterMemory = g_GraphicsFramework.AllocateMemory(memorySize, memoryAlignment);

    // メモリプールの確保
    nn::gfx::MemoryPool::InfoType memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuUncached);

    size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(g_GraphicsFramework.GetDevice(), info);
    size_t alignment = NN_PERF_GET_MEMORY_POOL_ALIGNMENT(g_GraphicsFramework.GetDevice(), info);
    alignment = nn::util::align_up(alignment, nn::gfx::MemoryPool::GetPoolMemoryAlignment(g_GraphicsFramework.GetDevice(), memoryPoolInfo));
    memoryPoolSize = nn::util::align_up(memoryPoolSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(g_GraphicsFramework.GetDevice(), memoryPoolInfo));
    g_pMeterPoolMemory = g_GraphicsFramework.AllocateMemory(memoryPoolSize, alignment);
    NN_ASSERT_NOT_NULL(g_pMeterPoolMemory);
    memoryPoolInfo.SetPoolMemory(g_pMeterPoolMemory, memoryPoolSize);

    g_MeterMemoryPool.Initialize(g_GraphicsFramework.GetDevice(), memoryPoolInfo);
    g_MeterMemoryPoolOffset = 0;

    NN_PERF_INITIALIZE_METER(g_GraphicsFramework.GetDevice(), info,
        g_MeterMemory, memorySize,
        &g_MeterMemoryPool, g_MeterMemoryPoolOffset, memoryPoolSize);
}

// パフォーマンス計測クラスの破棄
void FinalizeLoadMeter()
{
    NN_PERF_FINALIZE_METER(g_GraphicsFramework.GetDevice());
    g_MeterMemoryPool.Finalize(g_GraphicsFramework.GetDevice());
    g_GraphicsFramework.FreeMemory(g_pMeterPoolMemory);
    g_GraphicsFramework.FreeMemory(g_MeterMemory);
}

nns::gfx::GraphicsFramework::CommandBuffer g_CommandBufferBegin[g_BufferCount];
nns::gfx::GraphicsFramework::CommandBuffer g_CommandBufferEnd[g_BufferCount];
void InitializeQueryCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
    size_t commandMemorySize = 1024 * 1024 * 1;
    size_t controlMemorySize = 256;

    for (int bufferIndex = 0; bufferIndex < g_BufferCount; bufferIndex++)
    {
        g_GraphicsFramework.InitializeCommandBuffer(&g_CommandBufferBegin[bufferIndex], info, commandMemorySize, controlMemorySize);
        g_GraphicsFramework.InitializeCommandBuffer(&g_CommandBufferEnd[bufferIndex], info, commandMemorySize, controlMemorySize);
    }
}

void FinalizeQueryCommandBuffer()
{
    for (int bufferIndex = 0; bufferIndex < g_BufferCount; bufferIndex++)
    {
        g_GraphicsFramework.FinalizeCommandBuffer(&g_CommandBufferBegin[bufferIndex]);
        g_GraphicsFramework.FinalizeCommandBuffer(&g_CommandBufferEnd[bufferIndex]);
    }
}

void ClearColor(nn::gfx::CommandBuffer* pCommandBuffer, int x, int y, int width, int height)
{
    int displayHeight = g_GraphicsFramework.GetDisplayHeight();
    int h = displayHeight - height - y;
    nn::gfx::ViewportStateInfo vinfo;
    vinfo.SetDefault();
    vinfo.SetOriginX(static_cast<float>(x));
    vinfo.SetOriginY(static_cast<float>(h));
    vinfo.SetWidth(static_cast<float>(width));
    vinfo.SetHeight(static_cast<float>(height));
    pCommandBuffer->SetViewports(0, 1, &vinfo);

    nn::gfx::ScissorStateInfo sinfo;
    sinfo.SetDefault();
    sinfo.SetOriginX(x);
    sinfo.SetOriginY(h);
    sinfo.SetWidth(width);
    sinfo.SetHeight(height);
    pCommandBuffer->SetScissors(0, 1, &sinfo);

    float c[4];
    std::memcpy(c, &g_ClearColor, 4 * sizeof(float));
    nvnCommandBufferClearColor(
        pCommandBuffer->ToData()->pNvnCommandBuffer,
        0,
        c,
        NVN_CLEAR_COLOR_MASK_RGBA
    );
}

// コマンド作成
void MakeCommand( unsigned int frame )
{
    g_pPrimitiveRenderer->Update(g_BufferIndex);

    nn::gfx::CommandBuffer* pCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(g_BufferIndex);
    g_GraphicsFramework.BeginFrame(g_BufferIndex);
    {
        if (g_IsPrepareSuspend)
        {
            if (NN_STATIC_CONDITION(g_CurrentFrameworkMode == FrameworkMode_Ocean))
            {
                NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Green());
                NN_PERF_BEGIN_MEASURE_GPU(pCommandBuffer);
            }
            int scanBufferIndex = g_GraphicsFramework.GetNextScanBufferIndex();
            nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetScanBufferView(scanBufferIndex);
            pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);
            ClearColor(pCommandBuffer, 0, 0, g_GraphicsFramework.GetDisplayWidth(), 140);
            pCommandBuffer->SetRenderTargets(1, &pTarget, g_GraphicsFramework.GetDepthStencilView());
            pCommandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());
            pCommandBuffer->SetRasterizerState(g_GraphicsFramework.GetRasterizerState(nns::gfx::GraphicsFramework::RasterizerStateType::RasterizerStateType_FillSolid_CullNone));
            pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);
            g_IsSuspendGraphicsThread = true;
            if (NN_STATIC_CONDITION(g_CurrentFrameworkMode == FrameworkMode_Ocean))
            {
                NN_PERF_END_MEASURE_GPU(pCommandBuffer);
            }
        }
        else
        {
            int scanBufferIndex = g_GraphicsFramework.GetNextScanBufferIndex();
            nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetScanBufferView(scanBufferIndex);

            if (g_CurrentTestMode == TestMode_ApplicationTest || g_CurrentTestMode == TestMode_ApplicationHighLoad || g_CurrentTestMode == TestMode_ApplicationLowLoad)
            {
                pCommandBuffer->ClearColor(pTarget, g_ClearColor.r, g_ClearColor.g, g_ClearColor.b, g_ClearColor.a, NULL);
            }
            else
            {
                if (frame < g_BufferCount)
                {
                    // 全画面クリアがオンもしくは初回描画時のときは全画面クリア
                    pCommandBuffer->ClearColor(pTarget, g_ClearColor.r, g_ClearColor.g, g_ClearColor.b, g_ClearColor.a, NULL);
                    pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);
                }
                else
                {
                    pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);
                    ClearColor(pCommandBuffer, 0, 0, g_GraphicsFramework.GetDisplayWidth(), 140);
                }
            }

            pCommandBuffer->SetRenderTargets(1, &pTarget, g_GraphicsFramework.GetDepthStencilView());
            pCommandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());
            pCommandBuffer->SetRasterizerState(g_GraphicsFramework.GetRasterizerState(nns::gfx::GraphicsFramework::RasterizerStateType::RasterizerStateType_FillSolid_CullNone));

            pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
                | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);

            g_pPrimitiveRenderer->SetDefaultParameters();
            nn::util::Matrix4x3fType viewMatrix;
            nn::util::Matrix4x4fType projectionMatrix;
            nn::util::Matrix4x3f modelMatrix;
            nn::util::MatrixIdentity(&modelMatrix);
            nn::util::Vector3f translate;
            nn::util::VectorSet(&translate, 0.f, 0.f, 1.f);
            nn::util::MatrixSetAxisW(&modelMatrix, translate);

            // Blend
            pCommandBuffer->SetBlendState(g_pPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal));

            // ビューとプロジェクションを設定
            float radius = 20.f;
            float x = radius * 1.0f;
            float z = radius * 0.0f;
            nn::util::Vector3fType camPos = { x, 10.f, z };
            nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
            nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
            nn::util::MatrixLookAtRightHanded(&viewMatrix, camPos, camTarget, camUp);

            const float fovy = nn::util::FloatPi / 3.0f;
            const float aspect = static_cast<float>(g_GraphicsFramework.GetDisplayWidth()) / static_cast<float>(g_GraphicsFramework.GetDisplayHeight());
            nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, fovy, aspect, 0.1f, 1000.f);

            g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
            g_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

            // Depth Disable
            g_pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

            // オブジェクトを描画
            nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
            nn::util::Vector3fType size = { 50.f, 50.f, 50.f };
            for (int i = 0; i < g_ObjectCount; ++i)
            {
                pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_IndirectBuffer);
                pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_IndirectBuffer);
                nn::util::VectorSet(&translate, 0, 0, 0);
                nn::util::MatrixSetAxisW(&modelMatrix, translate);
                g_pPrimitiveRenderer->SetColor(nn::util::Color4u8(0, 0, 0, 0));
                g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
                g_pPrimitiveRenderer->SetLineWidth(14.f);
                g_pPrimitiveRenderer->DrawCone(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
            }

            for (int i = 0; i < g_LiteObjectCount * 2; ++i)
            {
                pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_IndirectBuffer);
                pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_IndirectBuffer);
                g_pPrimitiveRenderer->SetColor(nn::util::Color4u8(0, 0, 0, 0));
                g_Writer.object.Print(" ");
            }

            float fontPosOffset = 0;
            if (g_CurrentTestMode == TestMode_ApplicationTest || g_CurrentTestMode == TestMode_ApplicationHighLoad || g_CurrentTestMode == TestMode_ApplicationLowLoad)
            {
                fontPosOffset = 400;
            }

            // タイトルを描画
            g_Writer.object.SetTextColor(nn::util::Color4u8::White());
            g_Writer.object.SetScale(1.5f, 1.5f);
            g_Writer.object.SetCursor(15, 15 + fontPosOffset);
            g_Writer.object.Print(g_Title.c_str());

            // フレームカウントを描画
            g_Writer.object.SetScale(1, 1);
            g_Writer.object.SetCursor(200, 15 + fontPosOffset);
            g_Writer.object.Print("Frame:%d", frame);

            // ビューをリセット
            g_pPrimitiveRenderer->SetDefaultParameters();

            // 処理負荷メータを描画
            g_Writer.object.SetScale(1, 1);
            nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
            if (g_CurrentTestMode == TestMode_ApplicationTest || g_CurrentTestMode == TestMode_ApplicationHighLoad || g_CurrentTestMode == TestMode_ApplicationLowLoad)
            {
                nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(32.f, g_GraphicsFramework.GetDisplayHeight() - g_MeterDrawer.GetHeight(pFrameMeter) - 10);
                g_MeterDrawer.SetWidth(g_GraphicsFramework.GetDisplayWidth() - 64.f);
                g_MeterDrawer.SetPosition(pos);
            }
            else
            {
                nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(35, 65);
                g_MeterDrawer.SetWidth(500);
                g_MeterDrawer.SetPosition(pos);
            }
            g_MeterDrawer.SetDebugFontTextWriter(&g_Writer.object);
            g_MeterDrawer.SetVisibleNoResult(false);
            g_MeterDrawer.Draw(pCommandBuffer, g_pPrimitiveRenderer, pFrameMeter);

            g_Writer.object.SetScale(1.5f, 1.5f);

            // GPU 負荷を描画
            nn::perf::GpuMeter* pGpuMeter = NN_PERF_GET_GPU_METER();
            g_Writer.object.SetScale(1, 1);
            g_Writer.object.SetCursor(200, 40 + fontPosOffset);
            if (NN_STATIC_CONDITION(g_CurrentFrameworkMode == FrameworkMode_Ocean))
            {
                nn::perf::LoadMeterBase::Section drawSection = pGpuMeter->GetLastResult(0);
                nn::perf::LoadMeterBase::Section presentSection = pGpuMeter->GetLastResult(1);
                int64_t drawUs = drawSection.end.ToTimeSpan().GetMicroSeconds() - drawSection.begin.ToTimeSpan().GetMicroSeconds();
                int64_t presentUs = presentSection.end.ToTimeSpan().GetMicroSeconds() - presentSection.begin.ToTimeSpan().GetMicroSeconds();
                g_Writer.object.Print("GPU: %lldus", drawUs + presentUs);
            }
            else
            {
                nn::perf::LoadMeterBase::Section mainSection = pGpuMeter->GetLastResult(0);
                g_Writer.object.Print("GPU: %lldus", mainSection.end.ToTimeSpan().GetMicroSeconds() - mainSection.begin.ToTimeSpan().GetMicroSeconds());
            }

            // まとめて文字を書き出す
            g_Writer.object.Draw(pCommandBuffer);

            if (NN_STATIC_CONDITION(g_CurrentFrameworkMode == FrameworkMode_Ocean))
            {
                NN_PERF_END_MEASURE_GPU(pCommandBuffer);
            }
        }
    }
    pCommandBuffer->End();
} // NOLINT(impl/function_size)

} // namespace

// グラフィックスの初期処理
void InitializeGraphics( Rgba clearColor, std::string title, FrameworkMode frameworkMode, TestMode testMode, int presentInterval)
{
    g_ClearColor = clearColor;
    g_Title = title;
    g_CurrentFrameworkMode = frameworkMode;
    g_CurrentTestMode = testMode;
    g_PresentInterval = presentInterval;

    const size_t graphicsSystemMemorySize = 6 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(graphicsSystemMemorySize);

    // グラフィックスフレームワークの初期化
    InitializeGraphicsFramework();

    // プリミティブレンダラの初期化
    InitializePrimitiveRenderer();

    // デバッグフォントの初期化
    g_GraphicsFramework.InitializeDebugFontTextWriter(&g_Writer, 2560, g_BufferCount);

    // 負荷計測の初期化
    InitializeLoadMeter();

    InitializeQueryCommandBuffer();
}

// グラフィックスの終了処理
void FinalizeGraphics()
{
    FinalizeQueryCommandBuffer();

    // 負荷計測の破棄
    FinalizeLoadMeter();

    // デバッグフォントの破棄
    g_GraphicsFramework.FinalizeDebugFontTextWriter(&g_Writer);

    // プリミティブレンダラの破棄
    FinalizePrimitiveRenderer();

    // グラフィックスフレームワークの破棄
    FinalizeGraphicsFramework();
}

void SetMeasureParameters(int presentInterval)
{
    checkGpuResouce.Initialize(presentInterval);
}

int GetGpuLoadAverage()
{
    return checkGpuResouce.GetGpuLoadAverage();
}

int GetGpuDelayAverage()
{
    return checkGpuResouce.GetGpuDelayAverage();
}

nn::os::Event g_SuspendGraphicsThread(nn::os::EventClearMode_AutoClear);
void SuspendGraphicsThread()
{
    g_IsPrepareSuspend = true;
}

void ResumeGraphicsThread()
{
    g_SuspendGraphicsThread.Signal();
    g_IsSuspendGraphicsThread = false;
    g_IsPrepareSuspend = false;
}

void WaitForResumeGraphicsThread()
{
    if (g_IsSuspendGraphicsThread && g_IsPrepareSuspend)
    {
        g_SuspendGraphicsThread.Wait();
    }
}

void WindowAcquireTexture()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());
    NN_PERF_AUTO_MEASURE_NAME("AcquireTexture");
    g_GraphicsFramework.AcquireTexture(g_BufferIndex);
}

void QueueWaitSync()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE();
    queue->SyncSemaphore(g_GraphicsFramework.GetDisplaySemaphore(g_BufferIndex));
    queue->Flush();
}

void QueueAcquireTexture()
{
    WindowAcquireTexture();
    QueueWaitSync();
}

void QueueFinish()
{
    g_GraphicsFramework.QueueFinish();
}

void WaitSync()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::White());
    NN_PERF_AUTO_MEASURE();
    g_GraphicsFramework.WaitDisplaySync(g_BufferIndex, nn::TimeSpan::FromSeconds(2));
}

void UpdateBufferIndex()
{
    g_BufferIndex = (g_BufferIndex + 1) % g_BufferCount;
}

void MakeAppCommands(unsigned int frame)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE_NAME("RecordCommand");
    MakeCommand(frame);

    // MakeBeginTimeStampQueryCommand
    g_GraphicsFramework.ResetCommandBuffer(&g_CommandBufferBegin[g_BufferIndex]);
    g_CommandBufferBegin[g_BufferIndex].object.Begin();
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
    NN_PERF_BEGIN_MEASURE_GPU(&g_CommandBufferBegin[g_BufferIndex].object);
    g_CommandBufferBegin[g_BufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferBegin[g_BufferIndex].object.End();

    // MakeEndTimeStampQueryCommand
    g_GraphicsFramework.ResetCommandBuffer(&g_CommandBufferEnd[g_BufferIndex]);
    g_CommandBufferEnd[g_BufferIndex].object.Begin();
    NN_PERF_END_MEASURE_GPU(&g_CommandBufferEnd[g_BufferIndex].object);
    g_CommandBufferEnd[g_BufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferEnd[g_BufferIndex].object.End();
}

void QueueAppCommands()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE_NAME("RecordCommand");
    queue->ExecuteCommand(g_GraphicsFramework.GetRootCommandBuffer(g_BufferIndex), NULL);
}

void QueuePresentTexture()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Yellow());
    NN_PERF_AUTO_MEASURE_NAME("PresentTexture");
    queue->Present(g_GraphicsFramework.GetSwapChain(), g_PresentInterval);
}

void QueueBeginTimeStampQueryCommand()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE();
    queue->ExecuteCommand(&g_CommandBufferBegin[g_BufferIndex].object, NULL);
    NN_PERF_END_MEASURE();
}

void QueueEndTimeStampQueryCommand()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE();
    queue->ExecuteCommand(&g_CommandBufferEnd[g_BufferIndex].object, NULL);
    queue->Flush();
    NN_PERF_END_MEASURE();
}

void GraphicsThreadFunction(void *arg)
{
    int frame = 0;
    g_IsProfiling = true;

    NN_UNUSED(arg);
    NN_LOG("@@@ Start GraphicsThreadFunction @@@\n");

    // 毎フレームのレンダリング
    while ( g_IsProfiling )
    {
        if (g_CurrentTestMode == TestMode_OverlayInvisible &&  frame == 10)
        {
            SuspendGraphicsThread();
        }

        WaitForResumeGraphicsThread();
        // フレーム全体の計測
        NN_PERF_BEGIN_FRAME();
        {
            // 自動負荷調整
            if (g_CurrentTestMode == TestMode_ApplicationTest || g_CurrentTestMode == TestMode_OverlayTest)
            {
                if (g_CurrentTestMode == TestMode_OverlayTest && g_PresentInterval == 1)
                {
                    if (checkGpuResouce.MeasureMaxGpuLoad(&g_LiteObjectCount))
                    {
                        g_IsProfiling = false;
                    }
                }
                else
                {
                    if (checkGpuResouce.MeasureMaxGpuLoad(&g_ObjectCount, &g_LiteObjectCount))
                    {
                        g_IsProfiling = false;
                    }
                }
            }
            else if (g_CurrentTestMode == TestMode_ApplicationHighLoad)
            {
                checkGpuResouce.MaximizeGpuLoad(&g_ObjectCount);
            }

            if ((g_CurrentFrameworkMode == FrameworkMode_AppletOption1) || (g_CurrentFrameworkMode == FrameworkMode_QueueFinish))
            {
                // Update bufferIndex
                UpdateBufferIndex();

                if (frame == 0)
                {
                    // AcquireTexture
                    QueueAcquireTexture();
                }

                // Make commands
                MakeAppCommands(frame++);

                // AddBeginTimeStampQueryCommand
                QueueBeginTimeStampQueryCommand();

                // AddCommand
                QueueAppCommands();

                // AddPresentCommand
                QueuePresentTexture();

                // AddEndTimeStampQueryCommand
                QueueEndTimeStampQueryCommand();

                // AcquireTexture
                QueueAcquireTexture();

                if (g_CurrentFrameworkMode == FrameworkMode_QueueFinish)
                {
                    // QueueFinish
                    QueueFinish();
                }
            }
            else if (g_CurrentFrameworkMode == FrameworkMode_DeferredExecution)
            {
                // Update bufferIndex
                UpdateBufferIndex();

                // WindowAcquireTexture
                WindowAcquireTexture();

                // Add sync command to queue
                QueueWaitSync();

                // Make commands
                MakeAppCommands(frame++);

                // AddBeginTimeStampQueryCommand
                QueueBeginTimeStampQueryCommand();

                // AddCommand
                QueueAppCommands();

                // PresentTexture
                QueuePresentTexture();

                // AddEndTimeStampQueryCommand
                QueueEndTimeStampQueryCommand();

                // Wait for vsync by waiting for the release fence from previous acquire.
                // That is, when above commands start executing on the GPU.
                WaitSync();
            }
            else if (g_CurrentFrameworkMode == FrameworkMode_DeferredSubmission)
            {
                // PresentTexture & flush
                if (frame > 0)
                {
                    // AddBeginTimeStampQueryCommand
                    QueueBeginTimeStampQueryCommand();

                    // AddCommand
                    QueueAppCommands();

                    // PresentTexture
                    QueuePresentTexture();

                    // AddEndTimeStampQueryCommand
                    QueueEndTimeStampQueryCommand();
                }

                // Update bufferIndex
                UpdateBufferIndex();

                // WindowAcquireTexture
                WindowAcquireTexture();

                // Add sync command to queue
                QueueWaitSync();

                // Make and add commands
                MakeAppCommands(frame++);

                // Wait for vsync by waiting for the release fence from previous acquire.
                // That is, when above commands start executing on the GPU.
                WaitSync();
            }
            else if (g_CurrentFrameworkMode == FrameworkMode_Immediate)
            {
                if (frame == 0)
                {
                    // Update bufferIndex
                    UpdateBufferIndex();

                    // WindowAcquireTexture
                    WindowAcquireTexture();

                    WaitSync();
                }

                // Make and add commands
                MakeAppCommands(frame++);

                // AddBeginTimeStampQueryCommand
                QueueBeginTimeStampQueryCommand();

                // AddCommand
                QueueAppCommands();

                // PresentTexture
                QueuePresentTexture();

                // AddEndTimeStampQueryCommand
                QueueEndTimeStampQueryCommand();

                // Update bufferIndex
                UpdateBufferIndex();

                // WindowAcquireTexture
                WindowAcquireTexture();

                // Add sync command to queue
                //QueueWaitSync();

                // Wait for vsync by waiting for the release fence from previous acquire.
                // That is, when above commands start executing on the GPU.
                WaitSync();
            }
            else if (NN_STATIC_CONDITION(g_CurrentFrameworkMode == FrameworkMode_Ocean))
            {
                nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

                // Draw Command
                {
                    // Update bufferIndex
                    UpdateBufferIndex();

                    // WindowAcquireTexture
                    WindowAcquireTexture();

                    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(2166));

                    // Add sync command to queue
                    QueueWaitSync();

                    // Make commands
                    MakeAppCommands(frame++);

                    // AddCommand
                    QueueAppCommands();

                    queue->Flush();

                    // Wait for vsync by waiting for the release fence from previous acquire.
                    // That is, when above commands start executing on the GPU.
                    WaitSync();
                }

                // Present Command
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1000));

                    // AddBeginTimeStampQueryCommand
                    QueueBeginTimeStampQueryCommand();

                    // PresentTexture
                    QueuePresentTexture();

                    // AddEndTimeStampQueryCommand
                    QueueEndTimeStampQueryCommand();

                    queue->Flush();

                    QueueFinish();
                }
            }
        }
        NN_PERF_END_FRAME();
    }
    QueueFinish();
} // NOLINT(impl/function_size)

} // Graphics
