﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <gfx/demo.h>
#include <nvperfapi.h>
#include <nvperfapi_nvn.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_FROM_WINDOWS_SDK_HEADERS
#endif

#include <nvperfapi_user_impl.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
NN_PRAGMA_POP_WARNINGS
#endif

#include "nvperfapi_debug.h"
#include "nvperfapi_profiler.h"

#include <iostream>
#include <fstream>
#include <vector>
#include <cstring>

struct _DEMOPerfData
{
    bool allMetrics;
    std::vector< NvPerfApiSamples::Profiler::MetricSpec > metrics;
    NVPA_Config const* pConfig;
    nn::gfx::Queue* pQueue;
    nn::gfx::CommandBuffer* pCommandBuffer;
};

static const size_t PERF_MAX_PATH = 1024;
static char s_PerfOutputPath[PERF_MAX_PATH] = { 0 };
static bool s_bRedirectPerfOutput = false;

// This should be sufficient for all the demos
uint32_t DEMOPerfMaxTags = 100;

bool _DEMOPerfCheckEnabled( int argc, char** argv )
{
    int i;
    char *p;
    bool result = false;

    // Analyze arguments
    // Note that all arguments might be in a single string!
    for (i = 0; i < argc; ++i)
    {
        p = strstr(argv[i], "ENABLE_PERF");
        if (p != 0)
        {
            result = true;
        }
        p = strstr(argv[i], "PERF_OUTPUT_PATH=");
        if (p != 0)
        {
            std::strncpy(s_PerfOutputPath, p + std::strlen("PERF_OUTPUT_PATH="),
                         PERF_MAX_PATH);
            if ( (p = strchr(s_PerfOutputPath, ',') ) != NULL )
            {
                *p = '\0';
            }

            // Create the path to the file
            char oldValue;
            char* pPath = strrchr( s_PerfOutputPath, '\\' );
            if ( !pPath )
            {
                pPath = strrchr( s_PerfOutputPath, '/' );
            }
            DEMOAssert( pPath != nullptr );

            // Create the path to the file if it doesn't exist.
            oldValue = *pPath;
            *pPath = '\0';
            s32 ok = DEMOFSCreateDirectory( s_PerfOutputPath );
            DEMOAssert( ok == DEMO_FS_RESULT_OK );
            NN_UNUSED( ok );
            *pPath = oldValue;

            s_bRedirectPerfOutput = true;
        }
    }

    return result;
}

void* DEMOPerfInit( uint32_t maxTags, uint32_t numMetrics, void* pMetrics)
{
    NN_UNUSED( maxTags );
    struct _DEMOPerfData* pPerfData;

    if ( !NvPerfApiSamples::Profiler::Init( NVPA_NVN_LoadDriver, 1 ) )
    {
        DEMOPrintf( "%s: Failed to initialize NVPA!\n", NN_CURRENT_FUNCTION_NAME );
        return nullptr;
    }

    pPerfData = new struct _DEMOPerfData;
    DEMOAssert( nullptr != pPerfData );

    if ( numMetrics == 0 || pMetrics == nullptr )
    {
        pPerfData->allMetrics = true;
        pPerfData->pConfig = nullptr;
    }
    else
    {
        const char** metricNames = reinterpret_cast< const char** >( pMetrics );
        pPerfData->allMetrics = false;
        pPerfData->metrics.clear();
        pPerfData->pConfig = nullptr;

        for ( uint32_t i = 0; i < numMetrics; i++ )
        {
            NvPerfApiSamples::Profiler::MetricSpec spec;
            spec.name = metricNames[ i ];
            spec.serialized = true;
            pPerfData->metrics.push_back( spec );
        }
    }
    return pPerfData;
}

void DEMOPerfRegisterQueue( void* pContext, nn::gfx::Queue* pQueue )
{
    NVPA_Status result;
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );

    result = NVPA_NVN_Queue_Register( pQueue->ToData()->pNvnQueue );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );
    NN_UNUSED( result );

    size_t deviceIndex = 0;
    result = NVPA_NVN_Queue_GetDeviceIndex( pQueue->ToData()->pNvnQueue, 0, &deviceIndex );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );

    if ( pPerfData->allMetrics )
    {
        NVPA_ActivityOptions* pActivityOptions = nullptr;
        result = NVPA_ActivityOptions_Create( &pActivityOptions );
        DEMOAssert( NVPA_STATUS_SUCCESS == result );
        result = NVPA_ActivityOptions_SetActivityKind( pActivityOptions, NVPA_ACTIVITY_KIND_PROFILER );
        DEMOAssert( NVPA_STATUS_SUCCESS == result );

        NVPA_Activity* pActivity = nullptr;
        result = NVPA_Activity_CreateForDevice( deviceIndex, pActivityOptions, &pActivity );
        DEMOAssert( NVPA_STATUS_SUCCESS == result );

        result = NVPA_ActivityOptions_Destroy( pActivityOptions );
        DEMOAssert( NVPA_STATUS_SUCCESS == result );

        size_t numMetrics;
        result = NVPA_Activity_GetNumMetrics( pActivity, &numMetrics );
        DEMOAssert( NVPA_STATUS_SUCCESS == result );
        DEMOAssert( numMetrics > 0 );

        std::vector<NVPA_MetricId> metricIds( numMetrics );
        result = NVPA_Activity_GetMetricIds( pActivity, numMetrics, &metricIds[ 0 ], nullptr );
        DEMOAssert( NVPA_STATUS_SUCCESS == result );

        std::vector<char const*> metricNames( numMetrics );
        result = NVPA_GetMetricNames( numMetrics, &metricIds[ 0 ], &metricNames[ 0 ] );
        DEMOAssert( NVPA_STATUS_SUCCESS == result );

        for ( size_t index = 0; index < numMetrics; ++index )
        {
            NvPerfApiSamples::Profiler::MetricSpec spec;
            spec.name = metricNames[ index ];
            spec.serialized = true;
            pPerfData->metrics.push_back( spec );
        }
        result = NVPA_Activity_Destroy( pActivity );
        DEMOAssert( NVPA_STATUS_SUCCESS == result );
    }
    pPerfData->pConfig = NvPerfApiSamples::Profiler::CreateConfig( deviceIndex, NVPA_ACTIVITY_KIND_PROFILER, &pPerfData->metrics.front(), &pPerfData->metrics.back() + 1, 1 );
    DEMOAssert( nullptr != pPerfData->pConfig );

    pPerfData->pQueue = pQueue;
}

void DEMOPerfRegisterCommandBuffer( void* pContext, nn::gfx::CommandBuffer* pCommandBuffer )
{
    NVPA_Status result;
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );
    result = NVPA_NVN_CommandBuffer_Register( pCommandBuffer->ToData()->pNvnCommandBuffer );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );
    NN_UNUSED( result );

    pPerfData->pCommandBuffer = pCommandBuffer;
}

void DEMOPerfUnregisterQueue( void* pContext )
{
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );

    NVPA_NVN_Queue_Unregister( pPerfData->pQueue->ToData()->pNvnQueue );
    pPerfData->pQueue = nullptr;
}

void DEMOPerfUnregisterCommandBuffer( void* pContext )
{
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );

    NVPA_NVN_CommandBuffer_Unregister( pPerfData->pCommandBuffer->ToData()->pNvnCommandBuffer );
    pPerfData->pCommandBuffer = nullptr;
}

void DEMOPerfStartFrame( void* pContext )
{
    NVPA_Status result;
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );
    result = NVPA_NVN_Queue_BeginSession( pPerfData->pQueue->ToData()->pNvnQueue, pPerfData->pConfig );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );
    NN_UNUSED( result );
}

void DEMOPerfEndFrame( void* pContext )
{
    NVPA_Status result;
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );
    result = NVPA_NVN_Queue_EndSession( pPerfData->pQueue->ToData()->pNvnQueue );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );
    NN_UNUSED( result );
}

void DEMOPerfStartPass( void* pContext )
{
    NVPA_Status result;
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );
    result = NVPA_NVN_Queue_BeginPass( pPerfData->pQueue->ToData()->pNvnQueue );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );
    NN_UNUSED( result );
}

void DEMOPerfEndPass( void* pContext )
{
    NVPA_Status result;
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );
    result = NVPA_NVN_Queue_EndPass( pPerfData->pQueue->ToData()->pNvnQueue );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );
    NN_UNUSED( result );
}

void DEMOPerfPushTag(void* pContext, uint32_t tag)
{
    NVPA_Status result;
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );
    result = NVPA_NVN_CommandBuffer_PushRange( pPerfData->pCommandBuffer->ToData()->pNvnCommandBuffer,
        static_cast< NVPA_RangeId >( tag ) );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );
    NN_UNUSED( result );
}

void DEMOPerfPopTag(void* pContext, uint32_t tag)
{
    NN_UNUSED( tag );
    NVPA_Status result;
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );
    result = NVPA_NVN_CommandBuffer_PopRange( pPerfData->pCommandBuffer->ToData()->pNvnCommandBuffer );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );
    NN_UNUSED( result );
}

bool DEMOPerfIsDataReady(void* pContext)
{
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );

    const NVPA_StackData* pStackData = nullptr;
    INSTRUMENT_NVPA(NVPA_NVN_Queue_GetStackData(pPerfData->pQueue->ToData()->pNvnQueue, &pStackData));
    if (pStackData == nullptr)
    {
        return false;
    }

    NVPA_Bool isReady = 0;
    INSTRUMENT_NVPA(NVPA_StackData_IsReady(pStackData, &isReady));
    INSTRUMENT_NVPA(NVPA_StackData_Release(pStackData));
    return isReady != 0;
}

void DEMOPerfPrintMetricData( void* pContext )
{
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );

    NVPA_NVN_Queue_Finish( pPerfData->pQueue->ToData()->pNvnQueue );
    const NVPA_StackData* pStackData = nullptr;
    INSTRUMENT_NVPA( NVPA_NVN_Queue_GetStackData( pPerfData->pQueue->ToData()->pNvnQueue, &pStackData ) );
    std::string str = NvPerfApiSamples::Profiler::CollectAndPrint( pStackData );

    if ( s_bRedirectPerfOutput )
    {
        DEMOFSFileInfo info;
        s32 result = DEMOFSOpenFileMode( s_PerfOutputPath, &info, "w" );
        if ( DEMO_FS_RESULT_OK == result )
        {
            DEMOPrintf( "DEMO: Redirecting PerfWorks data to '%s'\n", s_PerfOutputPath );
            DEMOFSWrite( &info, str.c_str(), str.length() );
            DEMOFSCloseFile( &info );
        }
        else
        {
            DEMOPrintf( "DEMO: Failed to open file %s\n", s_PerfOutputPath );
        }

        DEMOAssert( DEMO_FS_RESULT_OK == result );
    }
    else
    {
        DEMOPrintf( "%s\n", str.c_str() );
    }
}

void DEMOPerfShutdown( void* pContext )
{
    struct _DEMOPerfData* pPerfData = reinterpret_cast< struct _DEMOPerfData* >( pContext );
    DEMOAssert( nullptr != pPerfData );
    pPerfData->metrics.clear();
    NVPA_Status result = NVPA_Config_Release( pPerfData->pConfig );
    NN_UNUSED( result );
    DEMOAssert( NVPA_STATUS_SUCCESS == result );
    NN_UNUSED( result );
    delete pPerfData;
}
