﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////===========================================================================
///  DEMOGfx.c
///
///     This is graphics system code for the DEMO library.
///
////===========================================================================

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Macro.h>
#include <nn/vi.h>
#include <gfx/demo.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#if NN_GFX_IS_TARGET_GL
#include <GL/glew.h>
#endif

#define xstr(a) str(a)
#define str(a) #a

#if NN_GFX_IS_TARGET_GX
#include <stdint.h>
#include <cafe/procui.h>
#include <cafe/env.h>

// Cafe doesn't define SIZE_MAX
static const int SIZE_MAX = INT_MAX;

// A bit like asserts, but even for NDEBUG=TRUE:
#define DEMOCheck(x,y) if (!(x)) { OSReport("%s\n", y); while(1) OSSleepSeconds(1); }
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_FROM_WINDOWS_SDK_HEADERS
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#include <nn/hws/hws_Message.h>
#include <malloc.h>
#define DEMOCheck(x,y) if (!(x)) { fprintf(stderr, "%s\n", y); abort(); }
NN_PRAGMA_POP_WARNINGS
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
#include <nn/fgm.h>
#include <nn/oe.h>
#include <nn/nn_Log.h>
#define DEMOCheck(x,y) if (!(x)) { NN_LOG("%s\n", y); abort(); }

#if defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

enum DEMOPerfConfigTypes
{
    DEMOPerfConfigType_First,

    DEMOPerfConfigType_HandHeld = DEMOPerfConfigType_First,
    DEMOPerfConfigType_Console,
    DEMOPerfConfigType_Boost,

    DEMOPerfConfigType_Last,
};

typedef struct _DEMOPerfConfig
{
    const char* name;
    nn::oe::PerformanceConfiguration configuration;
} DEMOPerfConfig;

static DEMOPerfConfigTypes s_PerfConfigIndex = DEMOPerfConfigType_Last;
static DEMOPerfConfig s_PerfConfigs[ DEMOPerfConfigType_Last ] = {
    { "HandHeld", nn::oe::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz },
    { "Console", nn::oe::PerformanceConfiguration_Cpu1020MhzGpu384MhzEmc1331Mhz },
    { "Boost", nn::oe::PerformanceConfiguration_Cpu1020MhzGpu768MhzEmc1600Mhz },
};
nn::oe::PerformanceConfiguration s_OriginalPerformanceConfiguration;

#endif

#define SKIP_NON_DIGIT(c) ((c)!=0&&((c)<'0'||(c)>'9'))

static const size_t DEMO_GFX_SWAP_CHAIN_COUNT = 3;

static const size_t  DEMO_GFX_POOL_BAD_ALLOC = SIZE_MAX;
nn::gfx::Device DEMODevice;
nn::gfx::SwapChain DEMOSwapChain;
DEMOGfxMemPool* DEMOSwapChainPool;

nn::gfx::Queue DEMOQueue;
nn::gfx::CommandBuffer DEMOCommandBuffer;

nn::gfx::Texture::InfoType DEMOColorBufferInfo;
nn::gfx::ImageDimension DEMOColorBufferImageDimension;

nn::gfx::Texture::InfoType DEMODepthBufferInfo;
nn::gfx::Texture DEMODepthBuffer;
nn::gfx::ImageDimension DEMODepthBufferImageDimension;
nn::gfx::DepthStencilView DEMODepthBufferView;

nn::gfx::DescriptorPool DEMOTextureDescriptorPool;
nn::gfx::DescriptorPool DEMOSamplerDescriptorPool;
nn::gfx::DescriptorPool DEMOBufferViewDescriptorPool;
static DEMOGfxMemPool* s_TextureMemoryPool;
static DEMOGfxMemPool* s_SamplerMemoryPool;
static DEMOGfxMemPool* s_BufferDescriptorMemoryPool;
static const int DEMO_GFX_MAX_TEXTURES = 8192;
static const int DEMO_GFX_MAX_SAMPLES = 2048;
static const int DEMO_GFX_MAX_BUFFERS = 2048;
static int s_NextTextureID = 256; // FIXME: Start at 256 to deal with NVN issue
static int s_NextSamplerID = 256; // FIXME: Start at 256 to deal with NVN issue
static int s_NextBufferID = 0;

static nn::gfx::Texture* s_pSwapColorBuffers[ DEMO_GFX_SWAP_CHAIN_COUNT ] = { NULL };
static nn::gfx::Texture* s_pSwapColorBuffer = NULL;
static nn::gfx::ColorTargetView* s_pSwapColorTargetView = NULL;

static const int DEMO_NUM_CHUNKS = 8;
static int s_cmdChunk = 0;
DEMOGfxMemPool* DEMOCommandBufferCommandPool;
static int s_ctrlChunk = 0;
void* DEMOCommandBufferControlMemory;
DEMOGfxMemPool* DEMOColorBufferMemoryPool;
DEMOGfxMemPool* DEMODepthBufferMemoryPool;
static const int DEMO_GFX_COMMAND_BUFFER_CHUNK_SIZE = 512 * 1024;
static const int DEMO_GFX_CONTROL_BUFFER_CHUNK_SIZE = 64 * 1024;
static const int DEMO_GFX_COMMAND_BUFFER_POOL_SIZE = DEMO_NUM_CHUNKS * DEMO_GFX_COMMAND_BUFFER_CHUNK_SIZE;
static const int DEMO_GFX_CONTROL_BUFFER_POOL_SIZE = DEMO_NUM_CHUNKS * DEMO_GFX_CONTROL_BUFFER_CHUNK_SIZE;

static const int DEMO_GFX_CPU_POOL_SIZE = 64 * 1024 * 1024;
static const int DEMO_GFX_GPU_POOL_SIZE = 64 * 1024 * 1024;
static const int DEMO_GFX_MAX_POOL_ALIGNMENT = 128 * 1024;
#if NN_GFX_IS_TARGET_GX
static const int DEMO_GFX_GPU_POOL_SIZE_MEM1 = 28 * 1024 * 1024;
#endif
DEMOGfxMemPool* DEMOGfxSharedPool;
DEMOGfxMemPool* DEMOGfxGpuPool;

#if NN_GFX_IS_TARGET_NVN
DEMOGfxMemPool* DEMOGfxShaderScratchMemory;
#endif


static nn::gfx::Fence DEMOGfxFrameSync;
static BOOL gDemoGfxRunningFlag=FALSE;
static BOOL gDemoGfxForceMEM2=FALSE;
BOOL gDemoGfxForceMEM1 = FALSE;

BOOL gDemoGfxInForeground=TRUE;
static bool s_bEnablePerf = false;
void* DEMOPerfContext = NULL;
extern uint32_t DEMOPerfMaxTags;
extern bool _DEMOPerfCheckEnabled( int argc, char** argv );

static nn::gfx::ViewportScissorState s_DefaultViewportScissorState;

static char gDemoAssetsDir[ MAX_ASSET_DIR_LEN ] = { 0 };

// State from init that needs to be preserved when reacquiring foreground.
static nn::gfx::ImageFormat gDemoGfxScanOutCBFormat;
// Other preserved items
static u32 gDemoGfxSwapInterval = 1;

static nn::vi::Display* g_pDisplay = NULL;
static nn::vi::Layer* g_pLayer = NULL;

// nn::perf resources
static void* s_pMeterMemory = NULL;
static DEMOGfxMemPool* s_pGpuMeterMemPool = NULL;
#ifdef NN_PERF_PROFILE_ENABLED
static int GetCoreNumber()
{
#ifdef _WIN32
    static const nn::os::ThreadId s_ThreadId = nn::os::GetThreadId( nn::os::GetCurrentThread() );
    if( s_ThreadId != nn::os::GetThreadId( nn::os::GetCurrentThread() ) )
    {
        OSReport( "Warning : Two or more thread exists, nn::perf result may be incorrect.\n" );
    }
    return 0;    // workaround for cpu affinity mask isn't assured for windows.
#else
    return nn::os::GetCurrentCoreNumber();
#endif
}
#endif

#if NN_GFX_IS_TARGET_GX
static BOOL gDemoNeedMEM1HeapInit = TRUE;
static MEMHeapHandle  gDEMOMem1Heap = MEM_HEAP_INVALID_HANDLE;
static MEMHeapHandle  gDEMOBucketHeap = MEM_HEAP_INVALID_HANDLE;


static void* DEMOGX2RAlloc(GX2RResourceFlags resourceFlags, u32 byteCount, u32 alignment)
{
    if (GX2TestBitFlagsAny(resourceFlags, GX2R_BIND_COLOR_BUFFER | GX2R_BIND_DEPTH_BUFFER | GX2R_BIND_SCAN_BUFFER | GX2R_BIND_GS_RING | GX2R_USAGE_FORCE_MEM1) &&
        !GX2TestBitFlagsAny(resourceFlags, GX2R_USAGE_FORCE_MEM2))
    {
        return DEMOGfxAllocMEM1(byteCount, alignment);
    }
    else
    {
        return DEMOGfxAllocMEM2(byteCount, alignment);
    }
}


static void DEMOGX2RFree(GX2RResourceFlags resourceFlags, void* pMem)
{
    if (GX2TestBitFlagsAny(resourceFlags, GX2R_BIND_COLOR_BUFFER | GX2R_BIND_DEPTH_BUFFER | GX2R_BIND_SCAN_BUFFER | GX2R_BIND_GS_RING | GX2R_USAGE_FORCE_MEM1) &&
        !GX2TestBitFlagsAny(resourceFlags, GX2R_USAGE_FORCE_MEM2))
    {
        DEMOGfxFreeMEM1(pMem);
    }
    else
    {
        DEMOGfxFreeMEM2(pMem);
    }
}
#endif


void *DEMOGfxAllocMEM1(size_t size, size_t align)
{
    void *ptr;

    if (gDemoGfxForceMEM2)
    {
        return DEMOGfxAllocMEM2(size, align);
    }

    if (align < 4)
    {
        align = 4;
    }
#if NN_GFX_IS_TARGET_GX
    // DEMOGfxMem1HeapInit() must be called first to init the heap.
    DEMOAssert(gDEMOMem1Heap != MEM_HEAP_INVALID_HANDLE);

    // Use the expanded heap for demos.
    ptr = MEMAllocFromExpHeapEx(gDEMOMem1Heap, size, align);
    GX2NotifyMemAlloc(ptr, size, align);
#else
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    ptr = DEMOAllocEx( size, align );
#else
    ptr = _aligned_malloc( size, align );
#endif
#endif
    DEMOAssert(ptr&&"Failed MEM1 alloc");

    return ptr;
}

void DEMOGfxFreeMEM1(void * ptr)
{
#if NN_GFX_IS_TARGET_GX
    if (gDemoGfxForceMEM2)
    {
        DEMOGfxFreeMEM2(ptr);
        return;
    }

    if (gDemoGfxInForeground)
    {
        MEMFreeToExpHeap(gDEMOMem1Heap, ptr);
    }
    GX2NotifyMemFree(ptr);
#else
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    DEMOFree( ptr );
#else
    _aligned_free( ptr );
#endif
#endif
}

void *DEMOGfxAllocBucket(u32 size, u32 align)
{
    void* ptr = NULL;
#if NN_GFX_IS_TARGET_GX
    DEMOAssert(gDEMOBucketHeap != MEM_HEAP_INVALID_HANDLE);

    ptr = MEMAllocFromExpHeapEx(gDEMOBucketHeap, size, align);
    GX2NotifyMemAlloc(ptr, size, align);
#else
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    ptr = DEMOAllocEx( size, align );
#else
    ptr = _aligned_malloc( size, align );
#endif
#endif
    DEMOAssert(ptr);

    return ptr;
}

void DEMOGfxFreeBucket(void * ptr)
{
#if NN_GFX_IS_TARGET_GX
    DEMOAssert(gDEMOBucketHeap != MEM_HEAP_INVALID_HANDLE);

    if (gDemoGfxInForeground)
    {
        MEMFreeToExpHeap(gDEMOBucketHeap, ptr);
    }
    GX2NotifyMemFree(ptr);
#else
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    DEMOFree( ptr );
#else
    _aligned_free( ptr );
#endif
#endif
}

void *DEMOGfxAllocMEM2( size_t size, size_t align)
{
    void* ptr = DEMOAllocEx(size, align);

#if NN_GFX_IS_TARGET_GX
    GX2NotifyMemAlloc(ptr, size, align);
#endif

    DEMOAssert(ptr&&"Failed MEM2 alloc");

    return ptr;
}


void DEMOGfxFreeMEM2(void * ptr)
{
    DEMOFree(ptr);

#if NN_GFX_IS_TARGET_GX
    GX2NotifyMemFree(ptr);
#endif
}

#if NN_GFX_IS_TARGET_GX
static u32 DEMOGfxCallbackAcquiredForeground(void* unused)
{
    (void)unused;
    DEMOGfxAcquiredForeground();

    // No issues
    return 0;
}

static u32 DEMOGfxCallbackReleaseForeground(void* unused)
{
    (void) unused;
    DEMOGfxReleaseForeground();

    // No issues
    return 0;
}
#endif
static void ProcessColorFormat( char* p )
{
    if(strncmp(p, "10_10_10_2", 10) == 0)
    {
        DEMOColorBufferInfo.SetImageFormat( nn::gfx::ImageFormat_R10_G10_B10_A2_Unorm );
    } else if(strncmp(p, "8_8_8_8_SRGB", 12) == 0)
    {
        DEMOColorBufferInfo.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    } else if(strncmp(p, "8_8_8_8", 7) == 0)
    {
        DEMOColorBufferInfo.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    } else if(strncmp(p, "16_16_16_16F", 12) == 0)
    {
        DEMOColorBufferInfo.SetImageFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
    } else if(strncmp(p, "32_32_32_32F", 12) == 0)
    {
        DEMOColorBufferInfo.SetImageFormat( nn::gfx::ImageFormat_R32_G32_B32_A32_Float );
    } else if(strncmp(p, "16", 2) == 0)
    {
        DEMOColorBufferInfo.SetImageFormat( nn::gfx::ImageFormat_R16_Unorm );
    } else if(strncmp(p, "32F", 3) == 0)
    {
        DEMOColorBufferInfo.SetImageFormat( nn::gfx::ImageFormat_R32_Float );
    } else
    {
        DEMOPrintf("Unrecognized CB format: %s\n",p);
    }
}

static void ProcessDepthFormat( char* p )
{
    if(strncmp(p, "16", 2) == 0)
    {
        DEMODepthBufferInfo.SetImageFormat( nn::gfx::ImageFormat_D16_Unorm );
    }else if(strncmp(p, "8_24", 4) == 0)
    {
        DEMODepthBufferInfo.SetImageFormat( nn::gfx::ImageFormat_D24_Unorm_S8_Uint );
    }else if(strncmp(p, "X24_8_32F", 9) == 0)
    {
        DEMODepthBufferInfo.SetImageFormat( nn::gfx::ImageFormat_D32_Float_S8_Uint_X24 );
    }else if(strncmp(p, "32F", 3) == 0)
    {
        DEMODepthBufferInfo.SetImageFormat( nn::gfx::ImageFormat_D32_Float );
    } else
    {
        DEMOPrintf("Unrecognized DB format: %s\n",p);
    }
}

static void ProcessScanBufferFormat( char* p )
{
    if(strncmp(p, "10_10_10_2", 10) == 0)
    {
        gDemoGfxScanOutCBFormat = nn::gfx::ImageFormat_R10_G10_B10_A2_Unorm;
    }else if(strncmp(p, "8_8_8_8_SRGB", 12) == 0)
    {
        gDemoGfxScanOutCBFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb;
    }else if(strncmp(p, "8_8_8_8", 7) == 0)
    {
        gDemoGfxScanOutCBFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
    } else
    {
        DEMOPrintf("Unrecognized SCAN format: %s\n",p);
    }
}

nn::gfx::ImageStorageDimension DEMOConvertToStorageDimension(nn::gfx::ImageDimension imDim)
{
    switch (imDim)
    {
    case nn::gfx::ImageDimension_1d:
    case nn::gfx::ImageDimension_1dArray:
        return nn::gfx::ImageStorageDimension_1d;
    case nn::gfx::ImageDimension_2d:
    case nn::gfx::ImageDimension_2dArray:
    case nn::gfx::ImageDimension_2dMultisample:
    case nn::gfx::ImageDimension_2dMultisampleArray:
    case nn::gfx::ImageDimension_CubeMap:
    case nn::gfx::ImageDimension_CubeMapArray:
        return nn::gfx::ImageStorageDimension_2d;
    case nn::gfx::ImageDimension_3d:
        return nn::gfx::ImageStorageDimension_3d;
    default:
        return nn::gfx::ImageStorageDimension_Undefined;
    }
}

static void ProcessArguments(int argc, char *argv[])
{
    int i;
    char *p;
    s32 gx2DebugMode = -1;
    s32 gx2rDebugOptions = -1;

    // Analyze arguments
    // Note that all arguments might be in a single string!
    for (i = 0; i < argc; ++i)
    {
        DEMOPrintf( "argv[%d]: %s\n", i, argv[i] );

        p = strstr(argv[i], "DEMO_WIDTH");
        if (p != 0)
        {
            DEMOColorBufferInfo.SetWidth( atoi(p + 10 + SKIP_NON_DIGIT(p[10])) );
        }
        p = strstr(argv[i], "DEMO_HEIGHT");
        if (p != 0)
        {
            DEMOColorBufferInfo.SetHeight( atoi(p + 11 + SKIP_NON_DIGIT(p[11])) );
        }
        p = strstr(argv[i], "DEMO_CB_FORMAT");
        if (p != 0)
        {
            p = p + 14 + SKIP_NON_DIGIT(p[14]);

            ProcessColorFormat( p );
        }
        p = strstr(argv[i], "DEMO_SCAN_FORMAT");
        if (p != 0)
        {
            p = p + 16 + SKIP_NON_DIGIT(p[16]);

            ProcessScanBufferFormat( p );
        }
        p = strstr(argv[i], "DEMO_DB_FORMAT");
        if (p != 0)
        {
            p = p + 14 + SKIP_NON_DIGIT(p[14]);

            ProcessDepthFormat( p );
        }
        p = strstr(argv[i], "DEMO_AA_MODE");
        if (p != 0)
        {
            int numSamples = atoi( p + 12 + SKIP_NON_DIGIT(p[12]) );
            if ( numSamples != 0 )
            {
                // Use GX2 behavior where the number indicated what power of two to use
                DEMOColorBufferInfo.SetMultiSampleCount( 1 << numSamples );
                DEMOColorBufferImageDimension = nn::gfx::ImageDimension_2dMultisample;
                DEMOColorBufferInfo.SetImageStorageDimension( DEMOConvertToStorageDimension(DEMOColorBufferImageDimension) );
            }
        }
        p = strstr(argv[i], "DEMO_FONT_DISABLE");
        if (p != 0)
        {
            DEMOFontDrawEnable(FALSE);
            DEMOPrintf("DEMO: DISABLE_FONT\n");
        }

        p = strstr(argv[i], "DEMO_GX2_DEBUG_MODE");
        if (p != 0)
        {
            sscanf(p + strlen("DEMO_GX2_DEBUG_MODE="), "%i", &gx2DebugMode);       // allows hex notation
        }

        p = strstr(argv[i], "DEMO_GX2R_DEBUG_OPTIONS");
        if (p != 0)
        {
            sscanf(p + strlen("DEMO_GX2R_DEBUG_OPTIONS="), "%i", &gx2rDebugOptions); // allows hex notation
        }

        p = strstr(argv[i], "DEMO_SWAP_INTERVAL");
        if (p != 0)
        {
#if 1 || NN_GFX_IS_TARGET_GX
            size_t offset = strlen("DEMO_SWAP_INTERVAL");
            gDemoGfxSwapInterval = atoi(p + offset + SKIP_NON_DIGIT(p[offset]));
            OSReport("DEMO: swap interval=%d\n", gDemoGfxSwapInterval);
#endif
        }

        p = strstr(argv[i], "DEMO_FORCE_MEM2");

        if (p != 0)
        {

            gDemoGfxForceMEM2 = TRUE;
            gDemoGfxForceMEM1 = FALSE;
        }
        else
        {
            p = strstr(argv[i], "DEMO_FORCE_MEM1");
            if (p != 0)
            {
                gDemoGfxForceMEM2 = FALSE;
                gDemoGfxForceMEM1 = TRUE;
            }
        }

        // Modify the assets directory if DEMO_ASSETS_DIR is passed in
        p = strstr(argv[i], "DEMO_ASSETS_DIR=");
        if (p != 0)
        {
            char tempBuffer[MAX_ASSET_DIR_LEN];
            tempBuffer[MAX_ASSET_DIR_LEN - 2] = 0;

            strncpy(tempBuffer, p + strlen("DEMO_ASSETS_DIR="), MAX_ASSET_DIR_LEN - 2);
            // replace EOL or ',' with terminating 0
            char* p2 = tempBuffer;
            for (; *p2 != '\n' && *p2 != '\r' && *p2 != ',' && *p2 != 0; p2++) {}
            *p2 = 0;

            // Make sure the asset directory ends with a forward slash
            if (*(p2 - 1) != '/')
            {
                *p2 = '/';
                *(p2 + 1) = 0;
            }

            strcpy(gDemoAssetsDir, tempBuffer);
        }

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
        p = strstr(argv[i], "DEMO_PERF_CONFIG=");
        if (p != 0)
        {
            s_PerfConfigIndex = static_cast< DEMOPerfConfigTypes >( atoi(p + 17 ) );
            if ( s_PerfConfigIndex < DEMOPerfConfigType_First || s_PerfConfigIndex >= DEMOPerfConfigType_Last )
            {
                s_PerfConfigIndex = DEMOPerfConfigType_Last;
                DEMOPrintf( "Warning: Invalid DEMO_PERF_CONFIG value (%d)\n", s_PerfConfigIndex );
            }
            else
            {
                DEMOPerfConfig* pConfig = &s_PerfConfigs[ s_PerfConfigIndex ];
                DEMOPrintf( "DEMO Performance Configuration: %s\n",
                    pConfig->name );
            }
        }
#endif
    }

    if (!gDemoGfxForceMEM1 && !gDemoGfxForceMEM2)
    {
        // make the default be to use MEM2
        gDemoGfxForceMEM2 = TRUE;
    }
    if (gDemoGfxForceMEM2)
    {
        DEMOPrintf("DEMO: FORCE_MEM2\nDEMO: Performance will be signifcantly worse!\n");
    }

#if NN_GFX_IS_TARGET_GX
    if (gx2DebugMode != -1)
    {
        DEMOPrintf("DEMO: GX2 debug mode=%#04x\n", gx2DebugMode);
        GX2SetDebugMode((GX2DebugMode)gx2DebugMode);
    }

#ifdef _DEBUG
    // If no GX2R debug options specified, default some of them in debug builds
    if (gx2rDebugOptions == -1)
    {
        gx2rDebugOptions = GX2R_DEBUG_GUARD_BANDS_ENABLED | GX2R_DEBUG_LEAK_CHECK | GX2R_DEBUG_CHECK_GPU_CONTENTION;
    }
#endif

    if (gx2rDebugOptions != -1)
    {
        DEMOPrintf("DEMO: GX2R debug options=%#04x\n", gx2rDebugOptions);
        GX2RSetDebugOptions((GX2RDebugOptions)gx2rDebugOptions);
    }
#endif
} //NOLINT(readability/fn_size)

DEMOGfxMemPool* DEMOGfxMemPool::AllocNewPool(size_t size, size_t align, int pageProperty)
{
    DEMOGfxMemPool* result;
    nn::gfx::MemoryPoolInfo info;
    void* memPtr;
    size_t allocGranularity;
    size_t allocSize;
    size_t allocAlign;

#if NN_GFX_IS_TARGET_NVN
    allocGranularity = 128 * 1024;  //FIXME NVN_MEMORY_POOL_STORAGE_ALIGNMENT;
    allocAlign = 128 * 1024;
    NN_UNUSED( align );
#elif NN_GFX_IS_TARGET_GX
    allocGranularity = 32;
    allocAlign = ( align + 0xfff ) & ~( 0xfff );
#else
    allocGranularity = 4;
    allocAlign = ( align + 0xfff ) & ~( 0xfff );
#endif
    allocSize = (size + allocGranularity - 1) & ~(allocGranularity - 1);
    result = new DEMOGfxMemPool;
#if NN_GFX_IS_TARGET_GX
    if (gDemoGfxForceMEM1 && (pageProperty & nn::gfx::MemoryPoolProperty_CpuInvisible) != 0)
    {
        memPtr = DEMOGfxAllocMEM1(allocSize, allocAlign);
    }
    else
#endif
    {
        memPtr = DEMOGfxAllocMEM2(allocSize, allocAlign);
    }
    ASSERT(result && memPtr);
    result->pMemPool = new nn::gfx::MemoryPool;

    info.SetDefault();
    info.SetMemoryPoolProperty(pageProperty);
    info.SetPoolMemory(memPtr, allocSize);
    result->GetPool()->Initialize(&DEMODevice, info);

    result->dataPtr = static_cast<u8*>(memPtr);
    result->size = allocSize;
    result->curPos = result->origPos = 0;
    result->end = size;
    result->poolNeedsFree = true;
    result->dataNeedsFree = true;
    return result;
}

DEMOGfxMemPool* DEMOGfxMemPool::CreateFromMemory(void* memPtr, size_t size, int memoryPoolProperty)
{
    DEMOGfxMemPool* result;
    nn::gfx::MemoryPoolInfo info;

    result = new DEMOGfxMemPool;
    ASSERT(result);
    result->pMemPool = new nn::gfx::MemoryPool;

    info.SetDefault();
    info.SetMemoryPoolProperty( memoryPoolProperty );
    info.SetPoolMemory(memPtr, size);
    result->GetPool()->Initialize(&DEMODevice, info);

    result->dataPtr = static_cast<u8*>(memPtr);
    result->size = size;
    result->curPos = result->origPos = 0;
    result->end = size;
    result->dataNeedsFree = false;
    result->poolNeedsFree = true;

    return result;
}

size_t DEMOGfxMemPool::AllocSpace(size_t allocSize, size_t allocAlign)
{
    size_t pos = curPos;

    ASSERT(allocAlign <= DEMO_GFX_MAX_POOL_ALIGNMENT);

    pos = (pos + allocAlign - 1) & ~(allocAlign - 1);
    if (pos + allocSize > end)
    {
        return DEMO_GFX_POOL_BAD_ALLOC; // unable to allocate
    }
    curPos = pos + allocSize;
    return pos;
}

DEMOGfxMemPool* DEMOGfxMemPool::AllocSubPool(size_t allocSize, size_t allocAlign)
{
    size_t subPos = curPos;
    size_t offset = this->AllocSpace(allocSize, allocAlign);
    ASSERT(offset != DEMO_GFX_POOL_BAD_ALLOC);

    DEMOGfxMemPool* pPool = new DEMOGfxMemPool;
    pPool->curPos = pPool->origPos = offset;
    pPool->size = curPos - subPos;
    pPool->end = curPos;
    pPool->poolNeedsFree = false;
    pPool->dataNeedsFree = false;
    pPool->pMemPool = this->pMemPool;
    pPool->dataPtr = this->dataPtr;
    return pPool;
}

void DEMOGfxMemPool::Reset()
{
    curPos = origPos;
}

void DEMOGfxMemPool::Finalize()
{
    DEMOGfxMemPool* pPool = this;
    if (pPool)
    {
        if (pPool->dataNeedsFree)
        {
            DEMOGfxFreeMEM2(pPool->dataPtr);
        }
        if (pPool->poolNeedsFree)
        {
            pPool->GetPool()->Finalize(&DEMODevice);
            delete pPool->pMemPool;
        }
    }
}

static void OutOfCommandMemoryEventCallback(nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg&)
{
    s_cmdChunk++;
    if (s_cmdChunk >= DEMO_NUM_CHUNKS)
    {
        s_cmdChunk = 0;
    }
    {
        size_t start = s_cmdChunk * DEMO_GFX_COMMAND_BUFFER_CHUNK_SIZE;
        pCommandBuffer->AddCommandMemory(DEMOCommandBufferCommandPool->GetPool(), start, DEMO_GFX_COMMAND_BUFFER_CHUNK_SIZE);
    }
}

static void OutOfControlMemoryEventCallback(nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg&)
{
    s_ctrlChunk++;
    if (s_ctrlChunk >= DEMO_NUM_CHUNKS)
    {
        s_ctrlChunk = 0;
    }
    {
        size_t start = s_ctrlChunk * DEMO_GFX_CONTROL_BUFFER_CHUNK_SIZE;
        pCommandBuffer->AddControlMemory(nn::util::BytePtr(DEMOCommandBufferControlMemory, start).Get(), DEMO_GFX_CONTROL_BUFFER_CHUNK_SIZE);
    }
}

static void DEMOGfxInitCommandBufferQueue()
{
    {
        nn::gfx::Queue::InfoType info;
        info.SetDefault();
        info.SetCapability( nn::gfx::QueueCapability_Graphics | nn::gfx::QueueCapability_Compute );
        DEMOQueue.Initialize( &DEMODevice, info );
    }
    {
        // Alloc Gfx command buffer pool
        DEMOCommandBufferCommandPool = DEMOGfxSharedPool->AllocSubPool( DEMO_GFX_COMMAND_BUFFER_POOL_SIZE, nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &DEMODevice ) );
        DEMOCommandBufferControlMemory = DEMOGfxAllocMEM2( DEMO_GFX_CONTROL_BUFFER_POOL_SIZE, nn::gfx::CommandBuffer::GetControlMemoryAlignment( &DEMODevice ) );
        DEMOCheck(DEMOCommandBufferCommandPool && DEMOCommandBufferControlMemory, "Failed to allocate command buffer pool");
    }
    {
        nn::gfx::CommandBuffer::InfoType info;

        info.SetDefault();
        info.SetQueueCapability( nn::gfx::QueueCapability_Graphics | nn::gfx::QueueCapability_Compute );
        info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);

        DEMOCommandBuffer.Initialize( &DEMODevice, info );

        DEMOCommandBuffer.AddCommandMemory(DEMOCommandBufferCommandPool->GetPool(), DEMOCommandBufferCommandPool->GetBaseOffset(), DEMO_GFX_COMMAND_BUFFER_CHUNK_SIZE);
        DEMOCommandBuffer.AddControlMemory(DEMOCommandBufferControlMemory, DEMO_GFX_CONTROL_BUFFER_CHUNK_SIZE);

        DEMOCommandBuffer.SetOutOfCommandMemoryEventCallback(OutOfCommandMemoryEventCallback);
        DEMOCommandBuffer.SetOutOfControlMemoryEventCallback(OutOfControlMemoryEventCallback);
    }
}

static void DEMOGfxInitDefaultViewportScissor()
{
    void* data;
    DEMOGfxSetViewportScissorState( &s_DefaultViewportScissorState, &data,
        0.0f, 0.0f,
        static_cast< float >( DEMOColorBufferInfo.GetWidth() ),
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ),
        0.0f, 1.0f,
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ), false );
}

int DEMOGfxRegisterTextureView( nn::gfx::TextureView* pTextureView )
{
    int id = s_NextTextureID++;

    DEMOTextureDescriptorPool.BeginUpdate();
    DEMOTextureDescriptorPool.SetTextureView( id, pTextureView );
    DEMOTextureDescriptorPool.EndUpdate();

    return id;
}

int DEMOGfxRegisterSampler( nn::gfx::Sampler* pSampler )
{
    int id = s_NextSamplerID++;

    DEMOSamplerDescriptorPool.BeginUpdate();
    DEMOSamplerDescriptorPool.SetSampler( id, pSampler );
    DEMOSamplerDescriptorPool.EndUpdate();

    return id;
}

int DEMOGfxRegisterBufferView( const nn::gfx::GpuAddress& gpuAddress, size_t size )
{
    int id = s_NextBufferID++;

    DEMOBufferViewDescriptorPool.BeginUpdate();
    DEMOBufferViewDescriptorPool.SetBufferView( id, gpuAddress, size );
    DEMOBufferViewDescriptorPool.EndUpdate();

    return id;
}

void DEMOGfxInitDescriptorPools()
{
    // Setup the Texture Descriptor pool
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
        info.SetSlotCount( DEMO_GFX_MAX_TEXTURES );

        s_TextureMemoryPool = DEMOGfxSharedPool->AllocSubPool(
            nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &DEMODevice, info ),
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &DEMODevice, info ) );

        DEMOTextureDescriptorPool.Initialize( &DEMODevice, info, s_TextureMemoryPool->GetPool(), s_TextureMemoryPool->GetBaseOffset(),
            s_TextureMemoryPool->GetSize() );
    }

    // Setup the Sampler Descriptor pool
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
        info.SetSlotCount( DEMO_GFX_MAX_SAMPLES );

        s_SamplerMemoryPool = DEMOGfxSharedPool->AllocSubPool(
            nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &DEMODevice, info ),
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &DEMODevice, info ) );

        DEMOSamplerDescriptorPool.Initialize( &DEMODevice, info, s_SamplerMemoryPool->GetPool(), s_SamplerMemoryPool->GetBaseOffset(),
            s_SamplerMemoryPool->GetSize() );
    }

    // Setup the Buffer Descriptor pool
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_BufferView );
        info.SetSlotCount( DEMO_GFX_MAX_BUFFERS );

        s_BufferDescriptorMemoryPool = DEMOGfxSharedPool->AllocSubPool(
            nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &DEMODevice, info ),
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &DEMODevice, info ) );

        DEMOBufferViewDescriptorPool.Initialize(&DEMODevice, info,
            s_BufferDescriptorMemoryPool->GetPool(),
            s_BufferDescriptorMemoryPool->GetBaseOffset(),
            s_BufferDescriptorMemoryPool->GetSize() );
    }
}

void DEMOGfxFreeDescriptorPools()
{
    DEMOTextureDescriptorPool.Finalize( &DEMODevice );
    DEMOSamplerDescriptorPool.Finalize( &DEMODevice );
    DEMOBufferViewDescriptorPool.Finalize( &DEMODevice );

    s_TextureMemoryPool->Finalize();
    s_SamplerMemoryPool->Finalize();
}

#if NN_GFX_IS_TARGET_NVN
void NVNAPIENTRY DEMOGfxNvnDebugCallback( NVNdebugCallbackSource source, NVNdebugCallbackType type, int id, NVNdebugCallbackSeverity severity, const char* message, void* userParam )
{
    // We won't print the message since nn::gfx already does.
    NN_UNUSED( source );
    NN_UNUSED( type );
    NN_UNUSED( id );
    NN_UNUSED( severity );
    NN_UNUSED( message );
    NN_UNUSED( userParam );

    // Output failure message that RunOnTarget will catch.
    DEMOTestSetResult( false );
}

void DEMOGfxInitShaderScratchMemory()
{
    int alignment = 4;
    nvnDeviceGetInteger( DEMODevice.ToData()->pNvnDevice, NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_ALIGNMENT, &alignment );
    DEMOGfxShaderScratchMemory = DEMOGfxGpuPool->AllocSubPool( 1024 * 1024, alignment );
}

void DEMOGfxFreeShaderScratchMemory()
{
    DEMOGfxShaderScratchMemory->Finalize();
}
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}

void InitializeClocks()
{
    if ( s_PerfConfigIndex != DEMOPerfConfigType_Last )
    {
        DEMOPerfConfig* pConfig = &s_PerfConfigs[ s_PerfConfigIndex ];

        nn::oe::Initialize();
        if ( s_PerfConfigIndex == DEMOPerfConfigType_Boost )
        {
            DEMOAssert( nn::oe::GetPerformanceMode() == nn::oe::PerformanceMode_Boost );
            nn::oe::SetPerformanceConfiguration( nn::oe::PerformanceMode_Boost, pConfig->configuration );
        }
        else
        {
            // Sets both boost mode and normal mode
            nn::oe::SetPerformanceConfiguration( pConfig->configuration );
        }
    }
}
#endif

bool DEMOGfxInit( int argc, char *argv[] )
{
    u32 renderWidth = 1280;
    u32 renderHeight = 720;
    nn::gfx::ImageFormat renderCBFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb;
    nn::gfx::ImageFormat renderDBFormat = nn::gfx::ImageFormat_D32_Float;

    // Select the starting scanout format.
    gDemoGfxScanOutCBFormat = renderCBFormat;

    // Init flags
    gDemoGfxRunningFlag = FALSE;
    gDemoGfxForceMEM2 = FALSE;

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::InitializeGraphics( std::malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
    }
#endif

    // Initialize nn::gfx and the display
    nn::vi::Initialize();

    {
        nn::Result result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
        NN_ASSERT( result.IsSuccess() );
        NN_UNUSED( result );

        result = nn::vi::CreateLayer( &g_pLayer, g_pDisplay );
        NN_ASSERT( result.IsSuccess() );

        result = nn::vi::SetLayerScalingMode( g_pLayer, nn::vi::ScalingMode_FitToLayer );
        NN_ASSERT( result.IsSuccess() );
    }

    nn::gfx::Initialize();

    s_bEnablePerf = _DEMOPerfCheckEnabled( argc, argv );
    if ( s_bEnablePerf )
    {
        DEMOPerfContext = DEMOPerfInit( DEMOPerfMaxTags, 0, NULL );
    }

    {
        nn::gfx::Device::InfoType info;
        info.SetDefault();
        info.SetDebugMode( nn::gfx::DebugMode_Enable );
        DEMODevice.Initialize( info );
    }

#if NN_GFX_IS_TARGET_NVN
    nvnDeviceInstallDebugCallback( DEMODevice.ToData()->pNvnDevice, DEMOGfxNvnDebugCallback, NULL, NVN_TRUE );
#endif

    // Initialize memory pools
    DEMOGfxMem1HeapInit();
    DEMOGfxSharedPool = DEMOGfxMemPool::AllocNewPool(
        DEMO_GFX_CPU_POOL_SIZE, DEMO_GFX_MAX_POOL_ALIGNMENT,
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );
    // Initialize the command buffer and queue
    DEMOGfxInitCommandBufferQueue();

    // Reset the color/depth buffer info structures
    DEMOColorBufferInfo.SetDefault();
    DEMODepthBufferInfo.SetDefault();

#if NN_GFX_IS_TARGET_GX
    // Hook in the demo allocators for GX2R
    GX2RSetAllocator(DEMOGX2RAlloc, DEMOGX2RFree);
#endif

    // Setup DEMOColorBufferInfo/DEMODepthBufferInfo
    DEMOColorBufferImageDimension = nn::gfx::ImageDimension_2d;
    DEMOColorBufferInfo.SetWidth( renderWidth );
    DEMOColorBufferInfo.SetHeight( renderHeight );
    DEMOColorBufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_ColorBuffer );
    DEMOColorBufferInfo.SetImageStorageDimension( DEMOConvertToStorageDimension( DEMOColorBufferImageDimension ) );
    DEMOColorBufferInfo.SetImageFormat( renderCBFormat );

    // ProcessArguments should override this
    DEMODepthBufferInfo.SetImageFormat( renderDBFormat );

    // This may override values in DEMOColorBufferInfo which are used to setup
    // DEMODepthBufferInfo in the following steps
    ProcessArguments( argc, argv );

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    InitializeClocks();
#endif

    // For ease of testing create the pat
    if ( strlen( gDemoAssetsDir ) == 0 )
    {
#if NN_GFX_IS_TARGET_GX
        char nxSdkRoot[ MAX_ASSET_DIR_LEN ];
        strcpy( gDemoAssetsDir, "/vol/hfio01/" );
        int retVal = ENVGetEnvironmentVariable( "NX_SDK_ROOT", nxSdkRoot, sizeof( nxSdkRoot ) );
        DEMOAssert( !retVal );

        char* p = strstr( nxSdkRoot, "\\" );
        while (p)
        {
            *p = '/';
            p = strstr( p, "\\" );
        }

        p = strstr( nxSdkRoot, ":/" );

        // Append the drive letter
        char driveLetter = tolower( p[ -1 ] );
        strncat( gDemoAssetsDir, &driveLetter, 1 );

        // Append the path
        strcat( gDemoAssetsDir, p + 1 );

        if ( gDemoAssetsDir[ strlen(gDemoAssetsDir) - 1 ] != '/' )
        {
            strcat( gDemoAssetsDir, "/" );
        }
#else
        strcpy( gDemoAssetsDir, "assets:/" );

        // Setup the remainder of the asset path
        strcat( gDemoAssetsDir, "Externals/TestBinaries/Gfx/Prebuilt/" );

        // Append the test name
        strcat( gDemoAssetsDir, DEMOTestGetTestName() );

#if NN_GFX_IS_TARGET_NVN
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
        std::strcat( gDemoAssetsDir, "/NX/" );
#else
        std::strcat( gDemoAssetsDir, "/NXWin32/" );
#endif
#elif NN_GFX_IS_TARGET_GL
        std::strcat( gDemoAssetsDir, "/Generic/" );
#elif NN_GFX_IS_TARGET_GX
        std::strcat( gDemoAssetsDir, "/Cafe/" );
#elif NN_GFX_IS_TARGET_D3D
        std::strcat( gDemoAssetsDir, "/GenericD3d/" );
#endif
#endif
    }

    // Allocate the GPU pool after argument processing so that command line can
    // select MEM1 or MEM2 on Cafe
#if NN_GFX_IS_TARGET_GX
    if (gDemoGfxForceMEM1)
    {
        DEMOGfxGpuPool = DEMOGfxMemPool::AllocNewPool(DEMO_GFX_GPU_POOL_SIZE_MEM1, DEMO_GFX_MAX_POOL_ALIGNMENT, nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached);
    }
    else
#endif
    {
        DEMOGfxGpuPool = DEMOGfxMemPool::AllocNewPool(DEMO_GFX_GPU_POOL_SIZE, DEMO_GFX_MAX_POOL_ALIGNMENT, nn::gfx::MemoryPoolProperty_Compressible | nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached);
    }
    DEMODepthBufferInfo.SetWidth( DEMOColorBufferInfo.GetWidth() );
    DEMODepthBufferInfo.SetHeight( DEMOColorBufferInfo.GetHeight() );
#if NN_GFX_IS_TARGET_D3D
    DEMODepthBufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_Texture );
#else
    DEMODepthBufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_DepthStencil );
#endif
    DEMODepthBufferImageDimension = nn::gfx::ImageDimension_2d;
    DEMODepthBufferInfo.SetImageStorageDimension( DEMOConvertToStorageDimension(DEMOColorBufferImageDimension) );
    DEMODepthBufferInfo.SetMultiSampleCount( DEMOColorBufferInfo.GetMultisampleCount() );

    DEMOPrintf("DEMO: Rendering TV:%dx%d CB:%s DB:%s ScanCB:%s AA:%dX\n",
               DEMOColorBufferInfo.GetWidth(), DEMOColorBufferInfo.GetHeight(),
               &(DEMOGfxGetSurfaceFormatName(DEMOColorBufferInfo.GetImageFormat())[12]),
               &(DEMOGfxGetSurfaceFormatName(DEMODepthBufferInfo.GetImageFormat())[12]),
               &(DEMOGfxGetSurfaceFormatName(gDemoGfxScanOutCBFormat)[12]),
               DEMOColorBufferInfo.GetMultisampleCount());

    // Initialize all of the display buffers
    DEMOGfxAcquiredForeground();
#if NN_GFX_IS_TARGET_GX
    // Register a callback upon foreground acquire and release
    ProcUIRegisterCallback(PROCUI_MESSAGE_ACQUIRE, DEMOGfxCallbackAcquiredForeground, NULL, 100);
    ProcUIRegisterCallback(PROCUI_MESSAGE_RELEASE, DEMOGfxCallbackReleaseForeground, NULL, 100);

    // The main core is the one with GFX
    DEMOSetMainCore(OSGetCoreId());
#endif

    // Setup default viewport scissor
    DEMOGfxInitDefaultViewportScissor();

    // Setup the descriptor pools
    DEMOGfxInitDescriptorPools();

    // Setup the frame fence
    {
        nn::gfx::Fence::InfoType info;
        info.SetDefault();
        DEMOGfxFrameSync.Initialize( &DEMODevice, info );
    }

#if NN_GFX_IS_TARGET_NVN
    DEMOGfxInitShaderScratchMemory();
#endif

    // Indicate that graphics has been started
    gDemoGfxRunningFlag = TRUE;

    if ( s_bEnablePerf )
    {
        DEMOPerfRegisterQueue( DEMOPerfContext, &DEMOQueue );
        DEMOPerfRegisterCommandBuffer( DEMOPerfContext, &DEMOCommandBuffer );
    }

#ifdef NN_PERF_PROFILE_ENABLED
    // Initialize nn::perf
    {
        nn::perf::LoadMeterCenterInfo info;
#if NN_BUILD_CONFIG_OS_SUPPORTS_WIN32
        SYSTEM_INFO sysInfo = {};
        GetSystemInfo( &sysInfo );
        info.SetCoreCount( sysInfo.dwNumberOfProcessors );
#else
        info.SetCoreCount( 3 );
#endif
        info.SetUserMeterCount( 2 );
        info.SetCpuBufferCount( 2 );
        info.SetGpuBufferCount( 2 );
        info.SetCpuSectionCountMax( 32 * 1024 );
        info.SetGpuSectionCountMax( 64 );

        size_t memorySize = NN_PERF_GET_BUFFER_SIZE( info );
        size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
        s_pMeterMemory = DEMOGfxAllocMEM2( memorySize, memoryAlignment );

        size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE( &DEMODevice, info );
        size_t memoryPoolAlignment = NN_PERF_GET_MEMORY_POOL_ALIGNMENT( &DEMODevice, info );
        s_pGpuMeterMemPool = DEMOGfxMemPool::AllocNewPool(memoryPoolSize, memoryPoolAlignment, nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached);
        NN_PERF_INITIALIZE_METER( &DEMODevice, info, s_pMeterMemory, memorySize, s_pGpuMeterMemPool->GetPool(), s_pGpuMeterMemPool->GetBaseOffset(), memoryPoolSize );
        NN_PERF_SET_GET_CORE_NUMBER_FUNCTION( GetCoreNumber );
    }
#endif

    return true;
} // NOLINT

void DEMOGfxShutdown()
{
    DEMOGfxFrameSync.Finalize( &DEMODevice );

#ifdef NN_PERF_PROFILE_ENABLED
    NN_PERF_FINALIZE_METER( &DEMODevice );
    s_pGpuMeterMemPool->Finalize();
    DEMOFree( s_pMeterMemory );
#endif

    if ( s_bEnablePerf )
    {
        DEMOPerfUnregisterCommandBuffer( DEMOPerfContext );
        DEMOPerfUnregisterQueue( DEMOPerfContext );
        DEMOPerfShutdown( DEMOPerfContext );
    }

#if NN_GFX_IS_TARGET_NVN
    DEMOGfxFreeShaderScratchMemory();
#endif

    DEMOGfxFreeDescriptorPools();

    // It is possible to exit from the background, but some API calls
    // can only happen from the foreground. Check a flag to see if this
    // processes is exiting from the foreground before calling APIs
    // that deal with foreground-only resources.
    if( gDemoGfxInForeground )
    {
        // Free allocated buffers
        DEMOSwapChain.Finalize( &DEMODevice );
        DEMODepthBufferView.Finalize( &DEMODevice );
        DEMODepthBuffer.Finalize( &DEMODevice );

        DEMOColorBufferMemoryPool->Finalize();
        DEMODepthBufferMemoryPool->Finalize();

        DEMOSwapChainPool->Finalize();
    }

    // Free up the CommandBuffer/Queue
    DEMOCommandBuffer.Finalize( &DEMODevice );
    DEMOQueue.Finalize( &DEMODevice );

    {
        if (DEMOCommandBufferControlMemory)
        {
            DEMOFree(DEMOCommandBufferControlMemory);
        }
        if (DEMOCommandBufferCommandPool)
        {
            DEMOCommandBufferCommandPool->Finalize();
        }
    }
    {
        void* data = s_DefaultViewportScissorState.GetMemory();
        s_DefaultViewportScissorState.Finalize( &DEMODevice );
        if ( data )
        {
            DEMOGfxFreeMEM2( data );
        }
    }

    // Cleanup the memory pools
    DEMOGfxGpuPool->Finalize();
    delete DEMOGfxGpuPool;
    DEMOGfxGpuPool = NULL;
    DEMOGfxSharedPool->Finalize();
    delete DEMOGfxSharedPool;
    DEMOGfxSharedPool = NULL;

    // Shutdown GX2 (which is safe to call from the background);
    DEMODevice.Finalize();
    nn::gfx::Finalize();

    nn::vi::DestroyLayer( g_pLayer );
    nn::vi::CloseDisplay( g_pDisplay );
    nn::vi::Finalize();

#if NN_GFX_IS_TARGET_GX
    // Dump any leaked resources if enabled
    if (GX2RGetDebugOptions() & GX2R_DEBUG_LEAK_CHECK)
    {
        if (GX2TempGetResourceCount()>0)
        {
            OSReport("***DEMO: resources were not destroyed on shutdown\n");
            GX2TempDumpResources();
            ASSERT(FALSE && "Resources were not destroyed on shutdown");
        }
    }
    // Unhook the GX2R allocator functions.
    GX2RSetAllocator(NULL, NULL);

    // Only destroy these heaps if this process is in the foreground.
    if (gDemoGfxInForeground)
    {
        DEMOGfxMem1HeapDestroy();
        DEMOGfxBucketHeapDestroy();
    }
#endif
    gDemoGfxRunningFlag = FALSE;
}

void DEMOGfxReleaseForeground()
{
    // First wait for all pending draws to complete.
    DEMOQueue.Sync();

    // Destroy the CB/DB
    DEMOColorBufferMemoryPool->Finalize();
    DEMODepthBufferMemoryPool->Finalize();

    DEMODepthBufferView.Finalize( &DEMODevice );
    DEMODepthBuffer.Finalize( &DEMODevice );

    DEMOSwapChain.Finalize( &DEMODevice );
    DEMOSwapChainPool->Finalize();
    DEMOSwapChainPool = NULL;

    // Destroy heaps that will not be available.
    DEMOGfxMem1HeapDestroy();
    DEMOGfxBucketHeapDestroy();


    // Set a flag to indicate that this process is in the background.
    gDemoGfxInForeground=FALSE;
    OSMemoryBarrier();
}

void DEMOGfxAcquiredForeground()
{
    // Set a flag to indicate that this process is in the foreground.
    gDemoGfxInForeground=TRUE;
    OSMemoryBarrier();

    // Use a different MEM1 allocator for DEMO lib.
    DEMOGfxMem1HeapInit();
    DEMOGfxBucketHeapInit();

    // Setup the swap chain
    {
        nn::gfx::SwapChain::InfoType info;

        info.SetDefault();
        info.SetLayer( g_pLayer );
        info.SetFormat( gDemoGfxScanOutCBFormat );
        info.SetHeight( DEMOColorBufferInfo.GetHeight() );
        info.SetWidth( DEMOColorBufferInfo.GetWidth() );
        info.SetBufferCount( DEMO_GFX_SWAP_CHAIN_COUNT );

        if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
        {
            size_t align = nn::gfx::SwapChain::GetScanBufferAlignment( &DEMODevice, info );
            size_t memSize;
            memSize = nn::gfx::SwapChain::CalculateScanBufferSize( &DEMODevice, info );
            DEMOSwapChainPool = DEMOGfxGpuPool->AllocSubPool(memSize, align);
            DEMOSwapChain.Initialize(&DEMODevice, info, DEMOSwapChainPool->GetPool(), DEMOSwapChainPool->GetBaseOffset(), DEMOSwapChainPool->GetSize());
        }
        else
        {
            DEMOSwapChainPool = NULL;
            DEMOSwapChain.Initialize(&DEMODevice, info, NULL, 0, 0);
        }

        DEMOAssert( DEMO_GFX_SWAP_CHAIN_COUNT == DEMOSwapChain.GetScanBuffers( s_pSwapColorBuffers, DEMO_GFX_SWAP_CHAIN_COUNT ) );
    }

    // Setup the Depth buffer
    {
        if ( NN_STATIC_CONDITION( nn::gfx::Texture::IsMemoryPoolRequired ) )
        {
            size_t size = nn::gfx::Texture::CalculateMipDataSize( &DEMODevice, DEMODepthBufferInfo );
            size_t align = nn::gfx::Texture::CalculateMipDataAlignment(&DEMODevice, DEMODepthBufferInfo);
            DEMODepthBufferMemoryPool = DEMOGfxGpuPool->AllocSubPool(size, align);
            DEMODepthBuffer.Initialize(&DEMODevice, DEMODepthBufferInfo, DEMODepthBufferMemoryPool->GetPool(), DEMODepthBufferMemoryPool->GetBaseOffset(),
                DEMODepthBufferMemoryPool->GetSize());
        }
        else
        {
            DEMODepthBuffer.Initialize(&DEMODevice, DEMODepthBufferInfo, NULL, 0, 0);
        }
    }

    {
        nn::gfx::DepthStencilView::InfoType info;
        info.SetDefault();
        info.SetImageDimension( DEMODepthBufferImageDimension );
        info.SetTexturePtr( &DEMODepthBuffer );
        DEMODepthBufferView.Initialize( &DEMODevice, info );
    }


    // Indicate that graphics has been started
    gDemoGfxRunningFlag = TRUE;
}

BOOL DEMOGfxIsRunning()
{
    return gDemoGfxRunningFlag;
}

void DEMOGfxBeforeRender()
{
    // Acquire the next scanbuffer and setup pointers for query
    int swapIndex;
    s_pSwapColorTargetView = DEMOSwapChain.AcquireNextScanBufferView();
    swapIndex = DEMOSwapChain.AcquireNextScanBufferIndex();
    s_pSwapColorBuffer = s_pSwapColorBuffers[ swapIndex ];

    // FIXME: Cafe eventually fails in some demos. See NNGFX-334 for details.
    nn::gfx::OutOfMemoryEventArg arg;
    arg.minRequiredSize = DEMO_GFX_COMMAND_BUFFER_CHUNK_SIZE;
    OutOfCommandMemoryEventCallback( &DEMOCommandBuffer, arg );
    // END FIXME

    if ( s_bEnablePerf )
    {
        DEMOPerfStartFrame( DEMOPerfContext );
    }

    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_INDEX( DEMOTestPerfIndices_FrameCpu );

    DEMOCommandBuffer.Begin();

    DEMOGfxDebugTagIndent("DEMOGfxBeforeRender()");

    if ( s_bEnablePerf )
    {
        DEMOPerfPushTag( DEMOPerfContext, 1000 );
    }

    NN_PERF_BEGIN_MEASURE_GPU( &DEMOCommandBuffer );

    // Enabled when Demo Perf condition is met
    DEMOTestCheckPerfBegin();

    // Inform DEMOFont that it should free buffers left over from last frame
    DEMOFontNewFrame();

    // Setup descriptor pools for the frame.
    DEMOCommandBuffer.SetDescriptorPool( &DEMOTextureDescriptorPool );
    DEMOCommandBuffer.SetDescriptorPool( &DEMOSamplerDescriptorPool );
    DEMOCommandBuffer.SetDescriptorPool( &DEMOBufferViewDescriptorPool );

#if NN_GFX_IS_TARGET_NVN
    nvnCommandBufferSetShaderScratchMemory( DEMOCommandBuffer.ToData()->pNvnCommandBuffer,
        DEMOGfxShaderScratchMemory->GetPool()->ToData()->pNvnMemoryPool,
        DEMOGfxShaderScratchMemory->GetBaseOffset(), DEMOGfxShaderScratchMemory->GetSize());
#endif

    DEMOGfxDebugTagUndent();
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
//
// Determine whether one of our windows has focus
// this is somewhat crude; a better solution would be to
// look for appropriate messages. However, our PeekMessageA
// function in DEMOGfxDoneRender() never sees WM_ACTIVATE or WM_SETFOCUS
// messages, so this would have to be done at another level
//
bool DEMO_HasFocus()
{
    HANDLE hOurWindow;
    HANDLE hDeskWindow;

    hOurWindow = GetActiveWindow();
    hDeskWindow = GetForegroundWindow();

    return (hOurWindow == hDeskWindow);
}
#endif

nn::gfx::ColorTargetView* DEMOGetColorBufferView()
{
    return s_pSwapColorTargetView;
}

nn::gfx::Texture* DEMOGetColorBuffer()
{
    return s_pSwapColorBuffer;
}

void DEMOGfxSetDefaultRenderTarget()
{
    DEMOGfxDebugTagIndent("DEMOGfxSetDefaultRenderTarget()");
    const nn::gfx::ColorTargetView* renderTargets[ 1 ] = {
        s_pSwapColorTargetView,
    };
    DEMOCommandBuffer.SetRenderTargets( 1, renderTargets, &DEMODepthBufferView );
    DEMOGfxDebugTagUndent();
}

void DEMOGfxDoneRender()
{
    DEMOGfxDebugTagIndent("DEMOGfxDoneRender()");

    if ( s_bEnablePerf )
    {
        DEMOPerfPopTag( DEMOPerfContext, 1000 );
    }

    NN_PERF_END_MEASURE_GPU( &DEMOCommandBuffer );

    // Include the debug tag in the current CB.
    DEMOGfxDebugTagUndent();

    // Finish up the current command buffer
    DEMOCommandBuffer.End();

    do
    {
        if ( s_bEnablePerf )
        {
            DEMOPerfStartPass( DEMOPerfContext );
        }

        // Make sure that the command buffer has been sent.
        DEMOQueue.ExecuteCommand( &DEMOCommandBuffer, &DEMOGfxFrameSync );

        if ( s_bEnablePerf )
        {
            DEMOPerfEndPass( DEMOPerfContext );
            DEMOQueue.Sync();
        }
        else
        {
            // Flush out all previous commands
            DEMOQueue.Flush();
        }
    } while ( s_bEnablePerf && !DEMOPerfIsDataReady( DEMOPerfContext ) );

    if ( s_bEnablePerf )
    {
        DEMOPerfPrintMetricData( DEMOPerfContext );
        DEMOPerfEndFrame( DEMOPerfContext );
    }

    NN_PERF_END_MEASURE_INDEX( DEMOTestPerfIndices_FrameCpu );

    DEMOQueue.Present( &DEMOSwapChain, gDemoGfxSwapInterval );

    DEMOQueue.Flush();

    // Wait for the frame to complete before starting the next frame.
    // This is also important for NN_PERF_END_FRAME to work correctly.
    DEMOGfxWaitForSwap(0, 0);

    NN_PERF_END_FRAME();
#if NN_GFX_IS_TARGET_NVN
    {
        NVNqueueErrorInfo errorInfo;
        NVNqueueGetErrorResult result = nvnQueueGetError( DEMOQueue.ToData()->pNvnQueue, &errorInfo );
        static const char* s_QueueErrorStrings[] =
        {
            "No Error",
            "Unknown Error",
            "MMU Fault",
            "PDMA Exception",
            "Engine Exception",
            "Timeout",
        };

        if ( result != NVN_QUEUE_GET_ERROR_RESULT_GPU_NO_ERROR )
        {
            DEMOTestSetResult( false );
            DEMOPrintf( "DEMO: nvnQueueGetError returned %s\n", s_QueueErrorStrings[ result ] );
            if ( result == NVN_QUEUE_GET_ERROR_RESULT_GPU_ERROR_MMU_FAULT )
            {
                const char* s_MmuAccessType[] =
                {
                    "read",
                    "write",
                };
                DEMOPrintf( "DEMO: MMU Fault Info (faultAddress = %llx, accessType = %s)\n",
                    errorInfo.mmuFault.accessType, s_MmuAccessType[ errorInfo.mmuFault.accessType ] );
            }
        }
    }
#endif

    // Enabled when Demo Perf condition is met
    DEMOTestCheckPerfEnd();

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
    MSG msg;
    while ( PeekMessageA( &msg, nullptr, 0, 0, PM_REMOVE ) )
    {
        if ( msg.message == WM_QUIT )
        {
            DEMOStopRunning();
        }
        TranslateMessage( &msg );
        DispatchMessageA( &msg );
    }
#endif
}

void DEMOGfxDrawDone()
{
#if NN_GFX_IS_TARGET_GX
    // Only if running and in the foreground
    if (APP_IN_FOREGROUND && DEMOGfxIsRunning())
#endif
    {
        DEMOQueue.Sync();
    }
}

void DEMOGfxWaitForSwap(u32 depth, u32 percent)
{
#if NN_GFX_IS_TARGET_GX
    DEMOGfxDebugTagIndent("DEMOGfxWaitForSwap()");
    u32 swapCount, flipCount, waitCount=0;
    OSTime tLastFlip, tLastVsync;
    OSTime tNow;
    const OSTime t60  = (OSTime)OSSecondsToTicks(1.0f / 59.94f);
    OSTime period;
    u32    swapInt = GX2GetSwapInterval();

    if (swapInt != 0)
    {
        // Note: must be careful about unsigned wrap-around!

        // Wait for "depth" frames ago to post
        while (1)
        {
            GX2GetSwapStatus(&swapCount, &flipCount, &tLastFlip, &tLastVsync);
            if (flipCount + depth >= swapCount)
            {
                break;
            }

            // If we've waited over 10 seconds for a flip, consider the GPU hung
            // and stop running.
            if (waitCount++ > 60 * GX2GetGPUTimeout() / 1000)
            {
                OSReport("DEMOGfxWaitForSwap timed out. Potential GPU hang detected?\n");
                GX2SetMiscParam(GX2_MISC_HANG_STATE, GX2_HANG_STATE_ETC);
                if (GX2GetMiscParam(GX2_MISC_HANG_RESPONSE) == GX2_HANG_RESPONSE_DEBUG)
                {
                    GX2PrintGPUStatus();
                    OSDebugStrIfConnected(0);
                    DEMOStopRunning();
                }
                break;
            }

            // Call WaitForVsync instead of WaitForFlip due to possible
            // race condition of flip happening right after above test.
            // (There will always be more vsyncs, but not always more flips.)
            GX2WaitForVsync();
        }

        // Wait for (percent * Swap Period) milliseconds since last flip
        period = (percent * swapInt * t60 / 100);
        tNow = OSGetSystemTime();
        if (period > (tNow - tLastFlip))
        {
            OSSleepTicks(period - (tNow - tLastFlip));
        }
    }

    DEMOGfxDebugTagUndent();
#else
    NN_UNUSED( depth );
    NN_UNUSED( percent );

    // Wait for the worst-case amount of time
    {
        nn::gfx::SyncResult result = DEMOGfxFrameSync.Sync( nn::TimeSpan::FromSeconds( 1 ) );
        NN_SDK_ASSERT( nn::gfx::SyncResult_Success == result );
        NN_UNUSED( result );
    }

#endif
}

void DEMOGfxMem1HeapInit()
{
#if NN_GFX_IS_TARGET_GX
    if (!gDemoNeedMEM1HeapInit)
    {
        // already initialized, skip
        return;
    }
    //A real game should use the frame heap directly to save memory,
    //but using an expanded heap makes the demos simpler.

    //For more information on MEM1 usage, go to
    // Operating System->Cafe Core OS (COS) Overview->Basic Memory Allocation->MEM1
    //in the MAN pages.
    MEMHeapHandle hMEM1 = MEMGetBaseHeapHandle(MEM_ARENA_1);
    u32 uMEM1Size = MEMGetAllocatableSizeForFrmHeapEx(hMEM1,4);
    void* pStartOfMem1 = MEMAllocFromFrmHeapEx(hMEM1, uMEM1Size, 4);
    gDEMOMem1Heap = MEMCreateExpHeap(pStartOfMem1, uMEM1Size);
    DEMOAssert(gDEMOMem1Heap != MEM_HEAP_INVALID_HANDLE);
    gDemoNeedMEM1HeapInit = FALSE;
#endif
}

void DEMOGfxMem1HeapDestroy()
{
#if NN_GFX_IS_TARGET_GX
    DEMOAssert(gDEMOMem1Heap != MEM_HEAP_INVALID_HANDLE);

    //Destroy the expanded heap and reset the frame heap to restore the allocation.
    MEMDestroyExpHeap(gDEMOMem1Heap);
    gDEMOMem1Heap = MEM_HEAP_INVALID_HANDLE;
    MEMHeapHandle hMEM1 = MEMGetBaseHeapHandle(MEM_ARENA_1);
    MEMFreeToFrmHeap(hMEM1, MEM_FRMHEAP_FREE_ALL);
    gDemoNeedMEM1HeapInit = TRUE;
#endif
}

void DEMOGfxBucketHeapInit()
{
#if NN_GFX_IS_TARGET_GX
    //A real game should use the frame heap directly to save memory,
    //but using an expanded heap makes the demos simpler.

    //For more information on Foreground Bucket usage, go to
    // Operating System->Cafe Core OS (COS) Overview->Basic Memory Allocation->Foreground Bucket
    //in the MAN pages.
    MEMHeapHandle hMEMFg = MEMGetBaseHeapHandle(MEM_ARENA_FG);
    u32 uMEMFgSize = MEMGetAllocatableSizeForFrmHeapEx(hMEMFg,4);
    void* pStartOfMemFg = MEMAllocFromFrmHeapEx(hMEMFg, uMEMFgSize, 4);
    gDEMOBucketHeap = MEMCreateExpHeap(pStartOfMemFg, uMEMFgSize);
    DEMOAssert(gDEMOBucketHeap != MEM_HEAP_INVALID_HANDLE);
#endif
}

void DEMOGfxBucketHeapDestroy()
{
#if NN_GFX_IS_TARGET_GX
    DEMOAssert(gDEMOBucketHeap != MEM_HEAP_INVALID_HANDLE);

    //Destroy the expanded heap and reset the frame heap to restore the allocation.
    MEMDestroyExpHeap(gDEMOBucketHeap);
    gDEMOBucketHeap = MEM_HEAP_INVALID_HANDLE;
    MEMHeapHandle hMEMFg = MEMGetBaseHeapHandle(MEM_ARENA_FG);
    MEMFreeToFrmHeap(hMEMFg, MEM_FRMHEAP_FREE_ALL);
#endif
}

char* DEMOGfxAssetFileName(const char *name)
{
    char* filename = static_cast< char* >( malloc(strlen(name) + strlen(gDemoAssetsDir) + 4) );
    strcpy(filename, gDemoAssetsDir);
    strcat(filename, name);
    return filename;
}

s32 DEMOGfxOpenAssetFile(const char* path, DEMOFSFileInfo* info)
{
    char filename[MAX_ASSET_DIR_FULL_LEN];

    strcpy(filename, gDemoAssetsDir);
    strncat(filename, path, MAX_ASSET_DIR_FULL_LEN - MAX_ASSET_DIR_LEN);

    return DEMOFSOpenFile(filename, info);
}

s32 DEMOGfxGetAssetFileLength(const DEMOFSFileInfo* fileInfo, u32* length)
{
    return DEMOFSGetLength(fileInfo, length);
}

s32 DEMOGfxReadAssetFile(DEMOFSFileInfo* fileInfo, void* addr, s32 length, s32 offset)
{
    return DEMOFSRead(fileInfo, addr, length, offset);
}

s32 DEMOGfxCloseAssetFile(DEMOFSFileInfo* fileInfo)
{
    return DEMOFSCloseFile(fileInfo);
}

void* DEMOGfxLoadAssetFile( const char* path, u32* len )
{
    char filename[MAX_ASSET_DIR_FULL_LEN];

    strcpy(filename, gDemoAssetsDir);
    strncat(filename, path, MAX_ASSET_DIR_FULL_LEN - MAX_ASSET_DIR_LEN);

    return DEMOFSSimpleRead(filename, len);
}

void* DEMOGfxLoadAssetFileAlign( const char* path, u32* len, u32 alignSize )
{
    char filename[MAX_ASSET_DIR_FULL_LEN];

    strcpy(filename, gDemoAssetsDir);
    strncat(filename, path, MAX_ASSET_DIR_FULL_LEN - MAX_ASSET_DIR_LEN);

    return DEMOFSSimpleReadAlign(filename, len, alignSize);
}

void* DEMOGfxLoadModelFile( const char* path, u32* len )
{
    void *pRet = DEMOGfxLoadAssetFileAlign( path, len, 1024 );  // FIXME - temporary code
#if !NN_GFX_IS_TARGET_GX
    DEMOSwapBuffer32( pRet, *len );
#endif

    return pRet;
}

s32 DEMOGfxScanAssetDir( const char *path, u32 *pFileCount, u32 maxFiles, char** ppFileNames )
{
    char filename[MAX_ASSET_DIR_FULL_LEN];

    strcpy(filename, gDemoAssetsDir);
    strncat(filename, path, MAX_ASSET_DIR_FULL_LEN - MAX_ASSET_DIR_LEN);

    return DEMOFSScanDir(filename, path, pFileCount, maxFiles, ppFileNames);
}

const char *DEMOGfxGetAttribFormatName(nn::gfx::AttributeFormat format)
{
    switch(format)
    {
    case nn::gfx::AttributeFormat_Undefined:                     return "AttributeFormat_Undefined";
    case nn::gfx::AttributeFormat_8_Unorm:                       return "AttributeFormat_8_Unorm";
    case nn::gfx::AttributeFormat_8_Uint:                        return "AttributeFormat_8_Uint";
    case nn::gfx::AttributeFormat_8_Snorm:                       return "AttributeFormat_8_Snorm";
    case nn::gfx::AttributeFormat_8_Sint:                        return "AttributeFormat_8_Sint";
    case nn::gfx::AttributeFormat_8_UintToFloat:                 return "AttributeFormat_8_UintToFloat";
    case nn::gfx::AttributeFormat_8_SintToFloat:                 return "AttributeFormat_8_SintToFloat";
    case nn::gfx::AttributeFormat_4_4_Unorm:                     return "AttributeFormat_4_4_Unorm";
    case nn::gfx::AttributeFormat_16_Unorm:                      return "AttributeFormat_16_Unorm";
    case nn::gfx::AttributeFormat_16_Uint:                       return "AttributeFormat_16_Uint";
    case nn::gfx::AttributeFormat_16_Snorm:                      return "AttributeFormat_16_Snorm";
    case nn::gfx::AttributeFormat_16_Sint:                       return "AttributeFormat_16_Sint";
    case nn::gfx::AttributeFormat_16_Float:                      return "AttributeFormat_16_Float";
    case nn::gfx::AttributeFormat_16_UintToFloat:                return "AttributeFormat_16_UintToFloat";
    case nn::gfx::AttributeFormat_16_SintToFloat:                return "AttributeFormat_16_SintToFloat";
    case nn::gfx::AttributeFormat_8_8_Unorm:                     return "AttributeFormat_8_8_Unorm";
    case nn::gfx::AttributeFormat_8_8_Uint:                      return "AttributeFormat_8_8_Uint";
    case nn::gfx::AttributeFormat_8_8_Snorm:                     return "AttributeFormat_8_8_Snorm";
    case nn::gfx::AttributeFormat_8_8_Sint:                      return "AttributeFormat_8_8_Sint";
    case nn::gfx::AttributeFormat_8_8_UintToFloat:               return "AttributeFormat_8_8_UintToFloat";
    case nn::gfx::AttributeFormat_8_8_SintToFloat:               return "AttributeFormat_8_8_SintToFloat";
    case nn::gfx::AttributeFormat_32_Uint:                       return "AttributeFormat_32_Uint";
    case nn::gfx::AttributeFormat_32_Sint:                       return "AttributeFormat_32_Sint";
    case nn::gfx::AttributeFormat_32_Float:                      return "AttributeFormat_32_Float";
    case nn::gfx::AttributeFormat_16_16_Unorm:                   return "AttributeFormat_16_16_Unorm";
    case nn::gfx::AttributeFormat_16_16_Uint:                    return "AttributeFormat_16_16_Uint";
    case nn::gfx::AttributeFormat_16_16_Snorm:                   return "AttributeFormat_16_16_Snorm";
    case nn::gfx::AttributeFormat_16_16_Sint:                    return "AttributeFormat_16_16_Sint";
    case nn::gfx::AttributeFormat_16_16_Float:                   return "AttributeFormat_16_16_Float";
    case nn::gfx::AttributeFormat_16_16_UintToFloat:             return "AttributeFormat_16_16_UintToFloat";
    case nn::gfx::AttributeFormat_16_16_SintToFloat:             return "AttributeFormat_16_16_SintToFloat";
    case nn::gfx::AttributeFormat_8_8_8_8_Unorm:                 return "AttributeFormat_8_8_8_8_Unorm";
    case nn::gfx::AttributeFormat_8_8_8_8_Uint:                  return "AttributeFormat_8_8_8_8_Uint";
    case nn::gfx::AttributeFormat_8_8_8_8_Snorm:                 return "AttributeFormat_8_8_8_8_Snorm";
    case nn::gfx::AttributeFormat_8_8_8_8_Sint:                  return "AttributeFormat_8_8_8_8_Sint";
    case nn::gfx::AttributeFormat_8_8_8_8_UintToFloat:           return "AttributeFormat_8_8_8_8_UintToFloat";
    case nn::gfx::AttributeFormat_8_8_8_8_SintToFloat:           return "AttributeFormat_8_8_8_8_SintToFloat";
    case nn::gfx::AttributeFormat_10_10_10_2_Unorm:              return "AttributeFormat_10_10_10_2_Unorm";
    case nn::gfx::AttributeFormat_10_10_10_2_Uint:               return "AttributeFormat_10_10_10_2_Uint";
    case nn::gfx::AttributeFormat_10_10_10_2_Snorm:              return "AttributeFormat_10_10_10_2_Snorm";
    case nn::gfx::AttributeFormat_10_10_10_2_Sint:               return "AttributeFormat_10_10_10_2_Sint";
    case nn::gfx::AttributeFormat_32_32_Uint:                    return "AttributeFormat_32_32_Uint";
    case nn::gfx::AttributeFormat_32_32_Sint:                    return "AttributeFormat_32_32_Sint";
    case nn::gfx::AttributeFormat_32_32_Float:                   return "AttributeFormat_32_32_Float";
    case nn::gfx::AttributeFormat_16_16_16_16_Unorm:             return "AttributeFormat_16_16_16_16_Unorm";
    case nn::gfx::AttributeFormat_16_16_16_16_Uint:              return "AttributeFormat_16_16_16_16_Uint";
    case nn::gfx::AttributeFormat_16_16_16_16_Snorm:             return "AttributeFormat_16_16_16_16_Snorm";
    case nn::gfx::AttributeFormat_16_16_16_16_Sint:              return "AttributeFormat_16_16_16_16_Sint";
    case nn::gfx::AttributeFormat_16_16_16_16_Float:             return "AttributeFormat_16_16_16_16_Float";
    case nn::gfx::AttributeFormat_16_16_16_16_UintToFloat:       return "AttributeFormat_16_16_16_16_UintToFloat";
    case nn::gfx::AttributeFormat_16_16_16_16_SintToFloat:       return "AttributeFormat_16_16_16_16_SintToFloat";
    case nn::gfx::AttributeFormat_32_32_32_Uint:                 return "AttributeFormat_32_32_32_Uint";
    case nn::gfx::AttributeFormat_32_32_32_Sint:                 return "AttributeFormat_32_32_32_Sint";
    case nn::gfx::AttributeFormat_32_32_32_Float:                return "AttributeFormat_32_32_32_Float";
    case nn::gfx::AttributeFormat_32_32_32_32_Uint:              return "AttributeFormat_32_32_32_32_Uint";
    case nn::gfx::AttributeFormat_32_32_32_32_Sint:              return "AttributeFormat_32_32_32_32_Sint";
    case nn::gfx::AttributeFormat_32_32_32_32_Float:             return "AttributeFormat_32_32_32_32_Float";
    default:                                            return "invalid format";
    }
}

const char *DEMOGfxGetSurfaceFormatName(nn::gfx::ImageFormat format)
{
    switch(format)
    {
        case nn::gfx::ImageFormat_Undefined:    return "ImageFormat_Undefined";
        case nn::gfx::ImageFormat_R8_Unorm:    return "ImageFormat_R8_Unorm";
        case nn::gfx::ImageFormat_R8_Snorm:    return "ImageFormat_R8_Snorm";
        case nn::gfx::ImageFormat_R8_Uint:    return "ImageFormat_R8_Uint";
        case nn::gfx::ImageFormat_R8_Sint:    return "ImageFormat_R8_Sint";
        case nn::gfx::ImageFormat_R4_G4_B4_A4_Unorm:    return "ImageFormat_R4_G4_B4_A4_Unorm";
        case nn::gfx::ImageFormat_A4_B4_G4_R4_Unorm:    return "ImageFormat_A4_B4_G4_R4_Unorm";
        case nn::gfx::ImageFormat_R5_G5_B5_A1_Unorm:    return "ImageFormat_R5_G5_B5_A1_Unorm";
        case nn::gfx::ImageFormat_A1_B5_G5_R5_Unorm:    return "ImageFormat_A1_B5_G5_R5_Unorm";
        case nn::gfx::ImageFormat_R5_G6_B5_Unorm:    return "ImageFormat_R5_G6_B5_Unorm";
        case nn::gfx::ImageFormat_B5_G6_R5_Unorm:    return "ImageFormat_B5_G6_R5_Unorm";
        case nn::gfx::ImageFormat_R8_G8_Unorm:    return "ImageFormat_R8_G8_Unorm";
        case nn::gfx::ImageFormat_R8_G8_Snorm:    return "ImageFormat_R8_G8_Snorm";
        case nn::gfx::ImageFormat_R8_G8_Uint:    return "ImageFormat_R8_G8_Uint";
        case nn::gfx::ImageFormat_R8_G8_Sint:    return "ImageFormat_R8_G8_Sint";
        case nn::gfx::ImageFormat_R16_Unorm:    return "ImageFormat_R16_Unorm";
        case nn::gfx::ImageFormat_R16_Snorm:    return "ImageFormat_R16_Snorm";
        case nn::gfx::ImageFormat_R16_Uint:    return "ImageFormat_R16_Uint";
        case nn::gfx::ImageFormat_R16_Sint:    return "ImageFormat_R16_Sint";
        case nn::gfx::ImageFormat_R16_Float:    return "ImageFormat_R16_Float";
        case nn::gfx::ImageFormat_D16_Unorm:    return "ImageFormat_D16_Unorm";
        case nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm:    return "ImageFormat_R8_G8_B8_A8_Unorm";
        case nn::gfx::ImageFormat_R8_G8_B8_A8_Snorm:    return "ImageFormat_R8_G8_B8_A8_Snorm";
        case nn::gfx::ImageFormat_R8_G8_B8_A8_Uint:    return "ImageFormat_R8_G8_B8_A8_Uint";
        case nn::gfx::ImageFormat_R8_G8_B8_A8_Sint:    return "ImageFormat_R8_G8_B8_A8_Sint";
        case nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb:    return "ImageFormat_R8_G8_B8_A8_UnormSrgb";
        case nn::gfx::ImageFormat_B8_G8_R8_A8_Unorm:    return "ImageFormat_B8_G8_R8_A8_Unorm";
        case nn::gfx::ImageFormat_B8_G8_R8_A8_Snorm:    return "ImageFormat_B8_G8_R8_A8_Snorm";
        case nn::gfx::ImageFormat_B8_G8_R8_A8_Uint:    return "ImageFormat_B8_G8_R8_A8_Uint";
        case nn::gfx::ImageFormat_B8_G8_R8_A8_Sint:    return "ImageFormat_B8_G8_R8_A8_Sint";
        case nn::gfx::ImageFormat_B8_G8_R8_A8_UnormSrgb:    return "ImageFormat_B8_G8_R8_A8_UnormSrgb";
        case nn::gfx::ImageFormat_R9_G9_B9_E5_SharedExp:    return "ImageFormat_R9_G9_B9_E5_SharedExp";
        case nn::gfx::ImageFormat_R10_G10_B10_A2_Unorm:    return "ImageFormat_R10_G10_B10_A2_Unorm";
        case nn::gfx::ImageFormat_R10_G10_B10_A2_Uint:    return "ImageFormat_R10_G10_B10_A2_Uint";
        case nn::gfx::ImageFormat_R11_G11_B10_Float:    return "ImageFormat_R11_G11_B10_Float";
        case nn::gfx::ImageFormat_B10_G11_R11_Float:    return "ImageFormat_B10_G11_R11_Float";
        case nn::gfx::ImageFormat_R16_G16_Unorm:    return "ImageFormat_R16_G16_Unorm";
        case nn::gfx::ImageFormat_R16_G16_Snorm:    return "ImageFormat_R16_G16_Snorm";
        case nn::gfx::ImageFormat_R16_G16_Uint:    return "ImageFormat_R16_G16_Uint";
        case nn::gfx::ImageFormat_R16_G16_Sint:    return "ImageFormat_R16_G16_Sint";
        case nn::gfx::ImageFormat_R16_G16_Float:    return "ImageFormat_R16_G16_Float";
        case nn::gfx::ImageFormat_D24_Unorm_S8_Uint:    return "ImageFormat_D24_Unorm_S8_Uint";
        case nn::gfx::ImageFormat_R32_Uint:    return "ImageFormat_R32_Uint";
        case nn::gfx::ImageFormat_R32_Sint:    return "ImageFormat_R32_Sint";
        case nn::gfx::ImageFormat_R32_Float:    return "ImageFormat_R32_Float";
        case nn::gfx::ImageFormat_D32_Float:    return "ImageFormat_D32_Float";
        case nn::gfx::ImageFormat_R16_G16_B16_A16_Unorm:    return "ImageFormat_R16_G16_B16_A16_Unorm";
        case nn::gfx::ImageFormat_R16_G16_B16_A16_Snorm:    return "ImageFormat_R16_G16_B16_A16_Snorm";
        case nn::gfx::ImageFormat_R16_G16_B16_A16_Uint:    return "ImageFormat_R16_G16_B16_A16_Uint";
        case nn::gfx::ImageFormat_R16_G16_B16_A16_Sint:    return "ImageFormat_R16_G16_B16_A16_Sint";
        case nn::gfx::ImageFormat_R16_G16_B16_A16_Float:    return "ImageFormat_R16_G16_B16_A16_Float";
        case nn::gfx::ImageFormat_D32_Float_S8_Uint_X24:    return "ImageFormat_D32_Float_S8_Uint_X24";
        case nn::gfx::ImageFormat_R32_G32_Uint:    return "ImageFormat_R32_G32_Uint";
        case nn::gfx::ImageFormat_R32_G32_Sint:    return "ImageFormat_R32_G32_Sint";
        case nn::gfx::ImageFormat_R32_G32_Float:    return "ImageFormat_R32_G32_Float";
        case nn::gfx::ImageFormat_R32_G32_B32_Uint:    return "ImageFormat_R32_G32_B32_Uint";
        case nn::gfx::ImageFormat_R32_G32_B32_Sint:    return "ImageFormat_R32_G32_B32_Sint";
        case nn::gfx::ImageFormat_R32_G32_B32_Float:    return "ImageFormat_R32_G32_B32_Float";
        case nn::gfx::ImageFormat_R32_G32_B32_A32_Uint:    return "ImageFormat_R32_G32_B32_A32_Uint";
        case nn::gfx::ImageFormat_R32_G32_B32_A32_Sint:    return "ImageFormat_R32_G32_B32_A32_Sint";
        case nn::gfx::ImageFormat_R32_G32_B32_A32_Float:    return "ImageFormat_R32_G32_B32_A32_Float";
        case nn::gfx::ImageFormat_Bc1_Unorm:    return "ImageFormat_Bc1_Unorm";
        case nn::gfx::ImageFormat_Bc1_UnormSrgb:    return "ImageFormat_Bc1_UnormSrgb";
        case nn::gfx::ImageFormat_Bc2_Unorm:    return "ImageFormat_Bc2_Unorm";
        case nn::gfx::ImageFormat_Bc2_UnormSrgb:    return "ImageFormat_Bc2_UnormSrgb";
        case nn::gfx::ImageFormat_Bc3_Unorm:    return "ImageFormat_Bc3_Unorm";
        case nn::gfx::ImageFormat_Bc3_UnormSrgb:    return "ImageFormat_Bc3_UnormSrgb";
        case nn::gfx::ImageFormat_Bc4_Unorm:    return "ImageFormat_Bc4_Unorm";
        case nn::gfx::ImageFormat_Bc4_Snorm:    return "ImageFormat_Bc4_Snorm";
        case nn::gfx::ImageFormat_Bc5_Unorm:    return "ImageFormat_Bc5_Unorm";
        case nn::gfx::ImageFormat_Bc5_Snorm:    return "ImageFormat_Bc5_Snorm";
        case nn::gfx::ImageFormat_Bc6_Float:    return "ImageFormat_Bc6_Float";
        case nn::gfx::ImageFormat_Bc6_Ufloat:    return "ImageFormat_Bc6_Ufloat";
        case nn::gfx::ImageFormat_Bc7_Unorm:    return "ImageFormat_Bc7_Unorm";
        case nn::gfx::ImageFormat_Bc7_UnormSrgb:    return "ImageFormat_Bc7_UnormSrgb";
        case nn::gfx::ImageFormat_Eac_R11_Unorm:    return "ImageFormat_Eac_R11_Unorm";
        case nn::gfx::ImageFormat_Eac_R11_Snorm:    return "ImageFormat_Eac_R11_Snorm";
        case nn::gfx::ImageFormat_Eac_R11_G11_Unorm:    return "ImageFormat_Eac_R11_G11_Unorm";
        case nn::gfx::ImageFormat_Eac_R11_G11_Snorm:    return "ImageFormat_Eac_R11_G11_Snorm";
        case nn::gfx::ImageFormat_Etc1_Unorm:    return "ImageFormat_Etc1_Unorm";
        case nn::gfx::ImageFormat_Etc2_Unorm:    return "ImageFormat_Etc2_Unorm";
        case nn::gfx::ImageFormat_Etc2_UnormSrgb:    return "ImageFormat_Etc2_UnormSrgb";
        case nn::gfx::ImageFormat_Etc2_Mask_Unorm:    return "ImageFormat_Etc2_Mask_Unorm";
        case nn::gfx::ImageFormat_Etc2_Mask_UnormSrgb:    return "ImageFormat_Etc2_Mask_UnormSrgb";
        case nn::gfx::ImageFormat_Etc2_Alpha_Unorm:    return "ImageFormat_Etc2_Alpha_Unorm";
        case nn::gfx::ImageFormat_Etc2_Alpha_UnormSrgb:    return "ImageFormat_Etc2_Alpha_UnormSrgb";
        case nn::gfx::ImageFormat_Pvrtc1_2Bpp_Unorm:    return "ImageFormat_Pvrtc1_2Bpp_Unorm";
        case nn::gfx::ImageFormat_Pvrtc1_2Bpp_UnormSrgb:    return "ImageFormat_Pvrtc1_2Bpp_UnormSrgb";
        case nn::gfx::ImageFormat_Pvrtc1_4Bpp_Unorm:    return "ImageFormat_Pvrtc1_4Bpp_Unorm";
        case nn::gfx::ImageFormat_Pvrtc1_4Bpp_UnormSrgb:    return "ImageFormat_Pvrtc1_4Bpp_UnormSrgb";
        case nn::gfx::ImageFormat_Pvrtc1_Alpha_2Bpp_Unorm:    return "ImageFormat_Pvrtc1_Alpha_2Bpp_Unorm";
        case nn::gfx::ImageFormat_Pvrtc1_Alpha_2Bpp_UnormSrgb:    return "ImageFormat_Pvrtc1_Alpha_2Bpp_UnormSrgb";
        case nn::gfx::ImageFormat_Pvrtc1_Alpha_4Bpp_Unorm:    return "ImageFormat_Pvrtc1_Alpha_4Bpp_Unorm";
        case nn::gfx::ImageFormat_Pvrtc1_Alpha_4Bpp_UnormSrgb:    return "ImageFormat_Pvrtc1_Alpha_4Bpp_UnormSrgb";
        case nn::gfx::ImageFormat_Pvrtc2_Alpha_2Bpp_Unorm:    return "ImageFormat_Pvrtc2_Alpha_2Bpp_Unorm";
        case nn::gfx::ImageFormat_Pvrtc2_Alpha_2Bpp_UnormSrgb:    return "ImageFormat_Pvrtc2_Alpha_2Bpp_UnormSrgb";
        case nn::gfx::ImageFormat_Pvrtc2_Alpha_4Bpp_Unorm:    return "ImageFormat_Pvrtc2_Alpha_4Bpp_Unorm";
        case nn::gfx::ImageFormat_Pvrtc2_Alpha_4Bpp_UnormSrgb:    return "ImageFormat_Pvrtc2_Alpha_4Bpp_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_4x4_Unorm:    return "ImageFormat_Astc_4x4_Unorm";
        case nn::gfx::ImageFormat_Astc_4x4_UnormSrgb:    return "ImageFormat_Astc_4x4_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_5x4_Unorm:    return "ImageFormat_Astc_5x4_Unorm";
        case nn::gfx::ImageFormat_Astc_5x4_UnormSrgb:    return "ImageFormat_Astc_5x4_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_5x5_Unorm:    return "ImageFormat_Astc_5x5_Unorm";
        case nn::gfx::ImageFormat_Astc_5x5_UnormSrgb:    return "ImageFormat_Astc_5x5_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_6x5_Unorm:    return "ImageFormat_Astc_6x5_Unorm";
        case nn::gfx::ImageFormat_Astc_6x5_UnormSrgb:    return "ImageFormat_Astc_6x5_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_6x6_Unorm:    return "ImageFormat_Astc_6x6_Unorm";
        case nn::gfx::ImageFormat_Astc_6x6_UnormSrgb:    return "ImageFormat_Astc_6x6_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_8x5_Unorm:    return "ImageFormat_Astc_8x5_Unorm";
        case nn::gfx::ImageFormat_Astc_8x5_UnormSrgb:    return "ImageFormat_Astc_8x5_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_8x6_Unorm:    return "ImageFormat_Astc_8x6_Unorm";
        case nn::gfx::ImageFormat_Astc_8x6_UnormSrgb:    return "ImageFormat_Astc_8x6_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_8x8_Unorm:    return "ImageFormat_Astc_8x8_Unorm";
        case nn::gfx::ImageFormat_Astc_8x8_UnormSrgb:    return "ImageFormat_Astc_8x8_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_10x5_Unorm:    return "ImageFormat_Astc_10x5_Unorm";
        case nn::gfx::ImageFormat_Astc_10x5_UnormSrgb:    return "ImageFormat_Astc_10x5_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_10x6_Unorm:    return "ImageFormat_Astc_10x6_Unorm";
        case nn::gfx::ImageFormat_Astc_10x6_UnormSrgb:    return "ImageFormat_Astc_10x6_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_10x8_Unorm:    return "ImageFormat_Astc_10x8_Unorm";
        case nn::gfx::ImageFormat_Astc_10x8_UnormSrgb:    return "ImageFormat_Astc_10x8_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_10x10_Unorm:    return "ImageFormat_Astc_10x10_Unorm";
        case nn::gfx::ImageFormat_Astc_10x10_UnormSrgb:    return "ImageFormat_Astc_10x10_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_12x10_Unorm:    return "ImageFormat_Astc_12x10_Unorm";
        case nn::gfx::ImageFormat_Astc_12x10_UnormSrgb:    return "ImageFormat_Astc_12x10_UnormSrgb";
        case nn::gfx::ImageFormat_Astc_12x12_Unorm:    return "ImageFormat_Astc_12x12_Unorm";
        case nn::gfx::ImageFormat_Astc_12x12_UnormSrgb:    return "ImageFormat_Astc_12x12_UnormSrgb";
        default:  NN_UNEXPECTED_DEFAULT;
    }
} // NOLINT(impl/function_size)

bool DEMOGfxInitBuffer(DEMOGfxMemPool *gfxPool, nn::gfx::Buffer* pBuffer, int gpuAccessFlags, void* data, size_t size)
{
    nn::gfx::Buffer::InfoType bufferInfo;
    nn::gfx::MemoryPool* memPool;
    size_t offset;

    // Setup Buffer
    bufferInfo.SetDefault();
    bufferInfo.SetGpuAccessFlags( gpuAccessFlags );
    bufferInfo.SetSize(size);
    if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
    {
        size_t align = nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo );
        NN_SDK_REQUIRES_NOT_NULL(gfxPool);
        memPool = gfxPool->GetPool();
        offset = gfxPool->AllocSpace(size, align);
        ASSERT(offset != DEMO_GFX_POOL_BAD_ALLOC);
        if (offset == DEMO_GFX_POOL_BAD_ALLOC )
        {
            return false; // unable to allocate from the pool
        }
    }
    else
    {
        memPool = NULL;
        offset = 0;
    }
    pBuffer->Initialize(&DEMODevice, bufferInfo, memPool, offset, size);

    if (data)
    {
        void* dest = pBuffer->Map< void >();
        memcpy(dest, data, size);
        pBuffer->FlushMappedRange( 0, size );
        pBuffer->Unmap();
    }
    return true;
}

DEMOGfxMemPool* DEMOGfxCreateBuffer( nn::gfx::Buffer* pBuffer, int gpuAccessFlags, void* data, size_t size )
{
    nn::gfx::Buffer::InfoType bufferInfo;
    DEMOGfxMemPool* pool = NULL;

    // Setup Buffer
    bufferInfo.SetDefault();
    bufferInfo.SetGpuAccessFlags( gpuAccessFlags );
    bufferInfo.SetSize(size);
    if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
    {
        size_t align = nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo );
        pool = DEMOGfxSharedPool->AllocSubPool(size, align);
    }

    bool ok = DEMOGfxInitBuffer(pool, pBuffer, gpuAccessFlags, data, size);
    NN_SDK_ASSERT( ok );
    NN_UNUSED( ok );

    return pool;
}

void DEMOGfxSetViewportScissorState( nn::gfx::ViewportScissorState* pState, void** ppData,
    float xOrig, float yOrig, float width, float height, float minZ, float maxZ,
    float renderTargetHeight, bool isTargetTexture )
{
    NN_UNUSED( renderTargetHeight );
    nn::gfx::ViewportScissorState::InfoType info;
    nn::gfx::ViewportStateInfo viewportStateInfo;
    nn::gfx::ScissorStateInfo scissorStateInfo;

    viewportStateInfo.SetDefault();
    viewportStateInfo.SetWidth( width );
    viewportStateInfo.SetHeight( height );
    viewportStateInfo.SetOriginX( xOrig );
#if NN_GFX_IS_TARGET_GX
    viewportStateInfo.SetOriginY( yOrig );
#else
    if ( !isTargetTexture )
    {
        // Because GL defines origin as bottom left, the render target height is required to map
        // the viewport Y origin to a top left coordinate system.
        NN_SDK_REQUIRES( renderTargetHeight >= height + yOrig );
        viewportStateInfo.SetOriginY( renderTargetHeight - height - yOrig );
    }
    else
    {
        // However on render to texture situations, we are consciously rendering "upside-down"
        // from GL's perspective since we want to use GX2/D3D style texcoords to sample.
        viewportStateInfo.SetOriginY( yOrig );
    }
#endif
    viewportStateInfo.SetMinDepth( minZ );
    viewportStateInfo.SetMaxDepth( maxZ );

    scissorStateInfo.SetDefault();
    scissorStateInfo.SetOriginX( static_cast< int >( xOrig ) );
#if NN_GFX_IS_TARGET_GX
    scissorStateInfo.SetOriginY( static_cast< int >( yOrig ) );
#else
    if ( !isTargetTexture )
    {
        scissorStateInfo.SetOriginY( static_cast< int >( renderTargetHeight - height - yOrig ) );
    }
    else
    {
        scissorStateInfo.SetOriginY( static_cast< int >( yOrig ) );
    }
#endif

    scissorStateInfo.SetWidth( static_cast< int >( width ) );
    scissorStateInfo.SetHeight( static_cast< int >( height ) );

    info.SetDefault();
    info.SetScissorEnabled( true );
    info.SetScissorStateInfoArray( &scissorStateInfo, 1 );
    info.SetViewportStateInfoArray( &viewportStateInfo, 1 );

    size_t stateSize = nn::gfx::ViewportScissorState::GetRequiredMemorySize( info );
    if ( stateSize )
    {
        *ppData = DEMOGfxAllocMEM2( stateSize, nn::gfx::ViewportScissorState::RequiredMemoryInfo_Alignment );
        pState->SetMemory( *ppData, stateSize );
    }
    pState->Initialize( &DEMODevice, info );
}

void DEMOGfxSetupTextureView( nn::gfx::Texture* pTexture,
    nn::gfx::TextureView* pTextureView,
    nn::gfx::DescriptorSlot* pDescriptorSlot,
    nn::gfx::ImageDimension dimensions,
    nn::gfx::ImageFormat format,
    nn::gfx::DepthStencilFetchMode fetchMode )
{
    nn::gfx::TextureView::InfoType info;
    info.SetDefault();
    info.SetImageDimension( dimensions );
    info.SetImageFormat( format );
    info.SetTexturePtr( pTexture );
    info.SetDepthStencilTextureMode( fetchMode );

    switch ( format )
    {
    case nn::gfx::ImageFormat_D16_Unorm:
    case nn::gfx::ImageFormat_D32_Float:
        info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red );
        break;
    case nn::gfx::ImageFormat_D32_Float_S8_Uint_X24:
    case nn::gfx::ImageFormat_D24_Unorm_S8_Uint:
        if ( fetchMode == nn::gfx::DepthStencilFetchMode_DepthComponent )
        {
            info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red );
        }
        else
        {
#if NN_GFX_IS_TARGET_GX
            info.SetChannelMapping( nn::gfx::ChannelMapping_Green, nn::gfx::ChannelMapping_Green, nn::gfx::ChannelMapping_Green, nn::gfx::ChannelMapping_Green );
#else
            info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red );
#endif
        }
        break;
    default: break;
    }

    pTextureView->Initialize( &DEMODevice, info );

    int slot = DEMOGfxRegisterTextureView( pTextureView );
    DEMOTextureDescriptorPool.GetDescriptorSlot( pDescriptorSlot, slot );
}

void DEMOGfxSetupColorView( nn::gfx::Texture* pTexture,
    nn::gfx::ColorTargetView* pColorView,
    nn::gfx::ImageDimension dimensions,
    nn::gfx::ImageFormat format )
{
    nn::gfx::ColorTargetView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension( dimensions );
    viewInfo.SetImageFormat( format );
    viewInfo.SetTexturePtr( pTexture );

    pColorView->Initialize( &DEMODevice, viewInfo );
}

void DEMOGfxSetupDepthView( nn::gfx::Texture* pTexture,
    nn::gfx::DepthStencilView* pDepthView,
    nn::gfx::ImageDimension dimensions )
{
    nn::gfx::DepthStencilView::InfoType info;

    info.SetDefault();
    info.SetImageDimension( dimensions );
    info.SetTexturePtr( pTexture );
    pDepthView->Initialize( &DEMODevice, info );
}

void DEMOGfxSetupTextureBuffer( nn::gfx::Texture* pTexture,
    nn::gfx::TextureView* pTextureView,
    nn::gfx::DescriptorSlot* pDescriptorSlot,
    nn::gfx::ColorTargetView* pColorView,
    nn::gfx::DepthStencilView* pDepthView,
    DEMOGfxMemPool** pMemPool,
    int width,
    int height,
    int depth,
    int mipCount,
    nn::gfx::ImageDimension dimensions,
    nn::gfx::ImageFormat format,
    nn::gfx::DepthStencilFetchMode fetchMode,
    int swizzle )
{
    DEMOGfxMemPool* pool = *pMemPool;

    if ( pTexture )
    {
        nn::gfx::Texture::InfoType info;

        info.SetDefault();
        info.SetWidth( width );
        info.SetHeight( height );
        if ( dimensions == nn::gfx::ImageDimension_1dArray ||
            dimensions == nn::gfx::ImageDimension_2dArray ||
            dimensions == nn::gfx::ImageDimension_2dMultisampleArray ||
            dimensions == nn::gfx::ImageDimension_CubeMapArray )
        {
            info.SetArrayLength( depth );
        }
        else
        {
            info.SetDepth( depth );
        }
        info.SetImageStorageDimension( DEMOConvertToStorageDimension(dimensions) );
        info.SetImageFormat( format );
        info.SetSwizzle( swizzle );
        info.SetMipCount( mipCount );

        info.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture |
            ( ( ( format & ( nn::gfx::TypeFormat_Bits - 1 ) ) == nn::gfx::TypeFormat_DepthStencil ) ?
            nn::gfx::GpuAccess_DepthStencil : nn::gfx::GpuAccess_ColorBuffer ) );

        if (!pool)
        {
            size_t memPoolSize = nn::gfx::Texture::CalculateMipDataSize(&DEMODevice, info);
            pool = DEMOGfxMemPool::AllocNewPool(memPoolSize, nn::gfx::Texture::CalculateMipDataAlignment(
                &DEMODevice, info), nn::gfx::MemoryPoolProperty_Compressible | nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached);
            *pMemPool = pool;
        }
        pTexture->Initialize( &DEMODevice, info, pool->GetPool(), pool->GetBaseOffset(), pool->GetSize() );
    }

    if ( pColorView )
    {
        DEMOGfxSetupColorView( pTexture, pColorView, dimensions, format );
    }

    if ( pTextureView )
    {
        DEMOGfxSetupTextureView( pTexture, pTextureView, pDescriptorSlot, dimensions, format, fetchMode );
    }

    if ( pDepthView )
    {
        DEMOGfxSetupDepthView( pTexture, pDepthView, dimensions );
    }
}

void DEMOGfxInitSampler( nn::gfx::Sampler* pSampler,
    nn::gfx::DescriptorSlot* pDescriptorSlot,
    nn::gfx::TextureAddressMode addressMode,
    nn::gfx::FilterMode filterMode,
    nn::gfx::ComparisonFunction compareFunction )
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetAddressU( addressMode );
    info.SetAddressV( addressMode );
    info.SetAddressW( addressMode );
    info.SetComparisonFunction( compareFunction );
    info.SetFilterMode( filterMode );

    pSampler->Initialize( &DEMODevice, info );

    int slot = DEMOGfxRegisterSampler( pSampler );
    DEMOSamplerDescriptorPool.GetDescriptorSlot( pDescriptorSlot, slot );
}

void DEMOGfxSetDefaultViewportScissor()
{
    DEMOCommandBuffer.SetViewportScissorState( &s_DefaultViewportScissorState );
}

void DEMOGfxPipeline::SetDefaults()
{
    depthStencilStateInfo.SetDefault();
    rasterizerStateInfo.SetDefault();
    blendStateInfo.SetDefault();
    renderTargetStateInfo.SetDefault();
    vertexStateInfo.SetDefault();
    tessellationStateInfo.SetDefault();

    blendTargetStateCount = 0;
    colorTargetStateCount = 0;

    pPipelineData = NULL;

    // Setup the default state like in Cafe
    pipelineInfo.SetDefault();

    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    depthStencilStateInfo.SetDepthTestEnabled( true );
    depthStencilStateInfo.SetDepthWriteEnabled( true );
    depthStencilStateInfo.SetStencilTestEnabled( false );
    depthStencilStateInfo.SetStencilReadMask( 0xFF );
    depthStencilStateInfo.SetStencilWriteMask( 0xFF );
    depthStencilStateInfo.EditBackStencilStateInfo().SetComparisonFunction( nn::gfx::ComparisonFunction_Always );
    depthStencilStateInfo.EditBackStencilStateInfo().SetDepthPassOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditBackStencilStateInfo().SetDepthFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditBackStencilStateInfo().SetStencilFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditBackStencilStateInfo().SetStencilRef( 0x1 );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetComparisonFunction( nn::gfx::ComparisonFunction_Always );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetDepthPassOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetDepthFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetStencilFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetStencilRef( 0x1 );

    rasterizerStateInfo.SetFrontFace( nn::gfx::FrontFace_Ccw );
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_None );
    rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Solid );
    rasterizerStateInfo.SetDepthBias( 0 );
    rasterizerStateInfo.SetMultisampleEnabled( false );
    rasterizerStateInfo.SetRasterEnabled( true );
    rasterizerStateInfo.SetScissorEnabled( true );
    rasterizerStateInfo.SetSlopeScaledDepthBias( 0.0f );
    rasterizerStateInfo.SetDepthClipEnabled( false );

    blendStateInfo.SetBlendConstant( 0.0f, 0.0f, 0.0f, 0.0f );

    renderTargetStateInfo.SetDepthStencilFormat( DEMODepthBufferInfo.GetImageFormat() );
}

DEMOGfxPipeline::DEMOGfxPipeline()
{
    SetDefaults();
}

void DEMOGfxPipeline::Initialize( nn::gfx::Device* pDevice )
{
    DEMOAssert( shaders.attribCount != 0 && "shaders.attribCount is zero!" );
    DEMOAssert( shaders.bufferCount != 0 && "shaders.bufferCount is zero!" );

    blendStateInfo.SetBlendTargetStateInfoArray( blendTargetStateInfoArray, blendTargetStateCount );
    renderTargetStateInfo.SetColorTargetStateInfoArray( colorTargetStateInfoArray, colorTargetStateCount );
    vertexStateInfo.SetVertexAttributeStateInfoArray( shaders.vertexAttributes, shaders.attribCount );
    vertexStateInfo.SetVertexBufferStateInfoArray( shaders.vertexBuffers, shaders.bufferCount );

    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetTessellationStateInfo( &tessellationStateInfo );

    pipelineInfo.SetShaderPtr(shaders.GetShader());

    size_t pipelineDataSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    pPipelineData = DEMOGfxAllocMEM2( pipelineDataSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    pipeline.SetMemory( pPipelineData, pipelineDataSize );
    pipeline.Initialize( pDevice, pipelineInfo );
}

void DEMOGfxPipeline::Finalize( nn::gfx::Device* pDevice )
{
    pipeline.Finalize( pDevice );
    DEMOGfxFreeMEM2( pPipelineData );

    DEMOGfxFreeShaders( &shaders );
}

void DEMOGfxBuffer::Initialize( size_t initsize, void* initdata, int gpuAccessFlags, int initoffset )
{
    NN_SDK_ASSERT(initoffset == 0);
    this->size = initsize;
    this->data = initdata;
    this->offset = initoffset;

    this->memPool = DEMOGfxCreateBuffer( &buffer, gpuAccessFlags, initdata, initsize );
    buffer.GetGpuAddress(&this->gpuAddress);
}

void DEMOGfxBuffer::AllocFromPool(DEMOGfxMemPool* pool, size_t allocsize, void* allocdata, int gpuAccessFlags, int allocoffset)
{
    NN_SDK_ASSERT(allocoffset == 0);
    this->size = allocsize;
    this->data = allocdata;
    this->offset = allocoffset;
    this->memPool = NULL;

    bool ok = DEMOGfxInitBuffer(pool, &buffer, gpuAccessFlags, allocdata, allocsize);
    NN_SDK_ASSERT( ok );
    NN_UNUSED( ok );

    buffer.GetGpuAddress(&this->gpuAddress);
}

void DEMOGfxBuffer::CreateFromPool(DEMOGfxMemPool *pPool, size_t createsize, int gpuAccessFlags)
{
    nn::gfx::Buffer::InfoType bufferInfo;
    nn::gfx::MemoryPool* nnPool;

    // Setup Buffer
    bufferInfo.SetDefault();
    bufferInfo.SetGpuAccessFlags(gpuAccessFlags);
    bufferInfo.SetSize(createsize);
    {
        size_t align = nn::gfx::Buffer::GetBufferAlignment(&DEMODevice, bufferInfo);
        NN_SDK_REQUIRES_NOT_NULL(pPool);
        nnPool = pPool->GetPool();
        offset = pPool->AllocSpace(createsize, align);
        ASSERT(offset != DEMO_GFX_POOL_BAD_ALLOC);
    }

    this->data = pPool->GetDataPtr();
    this->memPool = pPool;
    this->offset = 0;
    this->size = createsize;
    this->buffer.Initialize(&DEMODevice, bufferInfo, nnPool, 0, createsize);
    this->buffer.GetGpuAddress(&this->gpuAddress);
}

void DEMOGfxBuffer::Finalize( nn::gfx::Device* dev)
{
    buffer.Finalize( dev );
    this->memPool->Finalize();
}

void DEMOGfxBuffer::Unmap()
{
    buffer.FlushMappedRange( 0, this->size );
    buffer.Unmap();
}

//
// code for loading models
//
//
// Load model data from file
//
// Pass pModelData a pointer to the model-data structure at the output destination.
// Set pModelFilePath to path of model file to load, and pTextureDirectoryPath to directory path where texture is located.
//

#if !NN_GFX_IS_TARGET_GX
void DEMOSwapBuffer16(void* bufp, size_t len)
{
    // No need to swap in Cafe
    uint8_t* buf = static_cast< uint8_t* >(bufp);
    uint8_t a;

    while (len > 0)
    {
        a = buf[0];
        buf[0] = buf[1];
        buf[1] = a;
        len -= 2;
        buf += 2;
    }
}

void DEMOSwapBuffer32(void* bufp, size_t len)
{
    uint8_t* buf = static_cast< uint8_t* >(bufp);
    uint8_t a;

    while (len > 0)
    {
        a = buf[0];
        buf[0] = buf[3];
        buf[3] = a;
        a = buf[1];
        buf[1] = buf[2];
        buf[2] = a;
        len -= 4;
        buf += 4;
    }
}
#endif

//
// Load the model data
//
void DEMOLoadModelData(DemoModelData* pModelData,
    const char* pModelFilePath,
    const char* pTextureDirectoryPath)
{
    DEMOAssert(pModelData != NULL && "Destination pointer is null.");
    DEMOAssert(pModelFilePath != NULL && "Model path is null.");
    DEMOAssert(pTextureDirectoryPath != NULL && "Texture path is null.");

    const u32 MAX_FILE_PATH = 512;  // Maximum length of file path

    // Load file data
    u32 fileLen = 0;
    void* pFileData = DEMOGfxLoadAssetFile(pModelFilePath, &fileLen);

    DEMOAssert(pFileData != NULL && "Unable to load model file.");

    // now load
    u8* pSrc = (u8*)pFileData;

    // Initialize model header
    memcpy(&pModelData->header, pSrc, sizeof(DemoModelHeader));

#if !NN_GFX_IS_TARGET_GX
    DEMOSwapBuffer32(&pModelData->header, sizeof(DemoModelHeader));
#endif

    DEMOAssert(pModelData->header.magic == DEMO_MODEL_DATA_MAGIC   && "Invalid model magic.");
    DEMOAssert(pModelData->header.version == DEMO_MODEL_DATA_VERSION && "Invalid model version.");

    pSrc += sizeof(DemoModelHeader);

    // Initialize joints
    u32 jointDataSize = sizeof(DemoJointData) * pModelData->header.jointCount;

    pModelData->joints = (DemoJointData*)DEMOAlloc(jointDataSize);

    memcpy(pModelData->joints, pSrc, jointDataSize);

#if !NN_GFX_IS_TARGET_GX
    DEMOSwapBuffer32(pModelData->joints, jointDataSize);
#endif

    pSrc += jointDataSize;

    // Initialize attribute data
    u32 attributeDataSize = sizeof(DemoAttributeData) * pModelData->header.attributeCount;

    pModelData->attributes = (DemoAttributeData*)DEMOAlloc(attributeDataSize);

    memcpy(pModelData->attributes, pSrc, attributeDataSize);

#if !NN_GFX_IS_TARGET_GX
    DEMOSwapBuffer32(pModelData->attributes, attributeDataSize);
#endif

    pSrc += attributeDataSize;

    // Initialize texture data
    u32 textureDataSize = sizeof(DemoTextureData) * pModelData->header.textureCount;

    DemoTextureData* textures = (DemoTextureData*)pSrc;

    pModelData->pTextures = new DEMOGfxTexture[pModelData->header.textureCount];

    for (s32 i = 0; i<pModelData->header.textureCount; i++)
    {
        // Load texture from file
        char textureFilePath[MAX_FILE_PATH];

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
        _snprintf(textureFilePath,
            sizeof(textureFilePath),
            "%s/%s",
            pTextureDirectoryPath,
            textures[i].name);
#else
        snprintf(textureFilePath,
            sizeof(textureFilePath),
            "%s/%s",
            pTextureDirectoryPath,
            textures[i].name);
#endif

        BOOL fOK = pModelData->pTextures[ i ].Initialize( textureFilePath );

        DEMOAssert(fOK && "Unable to load texture file");
        NN_UNUSED( fOK );
    }

    pSrc += textureDataSize;

    // Initialize material data
    u32 materialDataSize = sizeof(DemoMaterialData) * pModelData->header.materialCount;

    pModelData->materials = (DemoMaterialData*)DEMOAlloc(materialDataSize);

    memcpy(pModelData->materials, pSrc, materialDataSize);

#if !NN_GFX_IS_TARGET_GX
    DEMOSwapBuffer32(pModelData->materials, materialDataSize);
#endif

    pSrc += materialDataSize;

    // Initialize mesh data
    u32 meshDataSize = sizeof(DemoMeshData) * pModelData->header.meshCount;

    pModelData->meshes = (DemoMeshData*)DEMOAlloc(meshDataSize);

    memcpy(pModelData->meshes, pSrc, meshDataSize);

#if !NN_GFX_IS_TARGET_GX
    DEMOSwapBuffer32(pModelData->meshes, meshDataSize);
#endif

    pSrc += meshDataSize;

    // Initialize attribute buffer
    pModelData->attributeBuffer.size = pModelData->header.attributeBufferSize;
    pModelData->attributeBuffer.stride = pModelData->meshes->attributeStride;
    pModelData->attributeBuffer.elementCount =
        pModelData->attributeBuffer.size / pModelData->attributeBuffer.stride;

    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer );
    info.SetSize( pModelData->attributeBuffer.size );

    pModelData->attributeBuffer.data = DEMOGfxAllocMEM2(
        pModelData->attributeBuffer.size, nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, info ) );

    memcpy(pModelData->attributeBuffer.data, pSrc, pModelData->attributeBuffer.size);

#if !NN_GFX_IS_TARGET_GX
    DEMOSwapBuffer32(pModelData->attributeBuffer.data, pModelData->attributeBuffer.size);
#endif

    // Setup the nn::gfx::Buffer
    pModelData->attributeBuffer.buffer.Initialize(
        pModelData->attributeBuffer.size, pModelData->attributeBuffer.data,
        nn::gfx::GpuAccess_VertexBuffer, 0 );

    pSrc += pModelData->header.attributeBufferSize;

    // Initialize index buffer
    pModelData->indexBuffer.size = pModelData->header.indexBufferSize;
    pModelData->indexBuffer.stride = sizeof(u16);
    pModelData->indexBuffer.elementCount =
        pModelData->indexBuffer.size / pModelData->indexBuffer.stride;

    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_IndexBuffer );
    info.SetSize( pModelData->indexBuffer.size );
    pModelData->indexBuffer.data = DEMOGfxAllocMEM2(
        pModelData->indexBuffer.size, nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, info ) );

    memcpy(pModelData->indexBuffer.data, pSrc, pModelData->indexBuffer.size);

#if !NN_GFX_IS_TARGET_GX
    DEMOSwapBuffer16(pModelData->indexBuffer.data, pModelData->indexBuffer.size);
#endif

    // Setup the nn::gfx::Buffer
    pModelData->indexBuffer.buffer.Initialize(
        pModelData->indexBuffer.size, pModelData->indexBuffer.data,
        nn::gfx::GpuAccess_IndexBuffer, 0 );

    pSrc += pModelData->header.indexBufferSize;

    // Free the file data
    DEMOFree(pFileData);

    DEMOPrintf("Model load %s.\n", pModelFilePath);
}

//
// Free the model data memory
//
// Free the memory that was allocated in LoadModelData function.
//
void DEMOFreeModelData(DemoModelData* pModelData)
{
    for (s32 i = 0; i < pModelData->header.textureCount; i++)
    {
        pModelData->pTextures[i].Finalize();
    }

    delete[] pModelData->pTextures;
    DEMOFree(pModelData->attributes);
    DEMOFree(pModelData->materials);
    DEMOFree(pModelData->meshes);
    DEMOFree(pModelData->joints);

    pModelData->attributeBuffer.buffer.Finalize();
    pModelData->indexBuffer.buffer.Finalize();
    DEMOGfxFreeMEM2(pModelData->attributeBuffer.data);
    DEMOGfxFreeMEM2(pModelData->indexBuffer.data);

    memset(pModelData, 0, sizeof(DemoModelData));
}

bool DEMOGfxTexture::Initialize( const char* pFileName )
{
    u32 length;
    char* tmpString = new char[ strlen( pFileName ) + 5 + 1 ]; // fileLength + .bntx + '\0'
    strcpy( tmpString, pFileName );
    strcat( tmpString, ".bntx" );
    pFileData = DEMOGfxLoadAssetFileAlign( tmpString, &length,
        static_cast< u32 >( nn::gfx::ResTextureFile::GetMaxFileAlignment() ) );
    DEMOAssert( pFileData );
    delete[] tmpString;

    pResTextureFile = nn::gfx::ResTextureFile::ResCast( pFileData );
    pResTextureFile->Initialize( &DEMODevice );

    DEMOTextureDescriptorPool.BeginUpdate();

    totalTextures = pResTextureFile->GetTextureDic()->GetCount();
    pSlots = new nn::gfx::DescriptorSlot[totalTextures];
    for ( int i = 0; i < totalTextures; i++ )
    {
        nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture( i );
        pResTexture->Initialize( &DEMODevice );

        // Register the texture view
        DEMOTextureDescriptorPool.SetTextureView( s_NextTextureID++, pResTexture->GetTextureView() );
        DEMOTextureDescriptorPool.GetDescriptorSlot( &pSlots[ i ], s_NextTextureID - 1 );
    }

    DEMOTextureDescriptorPool.EndUpdate();

    return true;
}

void DEMOGfxTexture::Finalize()
{
    for ( int i = 0; i < totalTextures; i++ )
    {
        nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture( i );
        pResTexture->Finalize( &DEMODevice );
    }

    delete[] pSlots;
    pResTextureFile->Finalize( &DEMODevice );
    DEMOFree( pFileData );
}

nn::gfx::Texture* DEMOGfxTexture::GetTexture( int index )
{
    nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture( index );
    return pResTexture->GetTexture();
}

nn::gfx::TextureInfo* DEMOGfxTexture::GetTextureInfo( int index )
{
    nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture( index );
    return pResTexture->GetTextureInfo();
}

nn::gfx::TextureView* DEMOGfxTexture::GetTextureView( int index )
{
    nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture( index );
    return pResTexture->GetTextureView();
}

nn::gfx::DescriptorSlot& DEMOGfxTexture::GetDescriptorSlot( int index )
{
    return pSlots[ index ];
}

#if NN_GFX_IS_TARGET_GL
typedef struct _TimeStampResult
{
    GLuint id;
    uint64_t value;
} TimeStampResult;

void DEMOGfxTimestampQueryCallback( const void* pParam )
{
    DEMOGfxGpuTimestamp* pTimestamp = const_cast< DEMOGfxGpuTimestamp* >(
        reinterpret_cast< const DEMOGfxGpuTimestamp* >( pParam ) );
    TimeStampResult* result = pTimestamp->buffer.Map< TimeStampResult >();

    glGenQueries( 1, &result->id );
    glQueryCounter( result->id, GL_TIMESTAMP );

    DEMOAssert( glGetError() == GL_NO_ERROR );

    pTimestamp->buffer.FlushMappedRange( 0, sizeof( TimeStampResult ) );
    pTimestamp->buffer.Unmap();
}

void DEMOGfxTimestampGetResultCallback( const void* pParam )
{
    DEMOGfxGpuTimestamp* pTimestamp = const_cast< DEMOGfxGpuTimestamp* >(
        reinterpret_cast< const DEMOGfxGpuTimestamp* >( pParam ) );
    TimeStampResult* result = pTimestamp->buffer.Map< TimeStampResult >();

    glBindBuffer( GL_QUERY_BUFFER, 0 );
    glGetQueryObjectui64v( result->id, GL_QUERY_RESULT, &result->value );
    glDeleteQueries( 1, &result->id );

    DEMOAssert( glGetError() == GL_NO_ERROR );

    pTimestamp->buffer.FlushMappedRange( 0, sizeof( TimeStampResult ) );
    pTimestamp->buffer.Unmap();
}
#endif

void DEMOGfxGpuTimestamp::Initialize()
{
    size_t size = 0;
    size_t alignment = 0;

#if NN_GFX_IS_TARGET_GX
    size = sizeof(u64);
    alignment = 8;
#endif

#if NN_GFX_IS_TARGET_NVN
    int tmp = 0;
    nvnDeviceGetInteger( DEMODevice.ToData()->pNvnDevice, NVN_DEVICE_INFO_COUNTER_ALIGNMENT, &tmp );
    alignment = tmp;

    // Counters are 128-bit with the top 64-bit unused for timestamps
    size = sizeof( uint64_t ) * 2;
#endif

#if NN_GFX_IS_TARGET_GL
    // Store the GL ID and the timestamp value for callbacks.
    size = sizeof(TimeStampResult);
    alignment = sizeof( uint64_t );
#endif

#if NN_GFX_IS_TARGET_D3D
    // This is temporal settings.
    size = sizeof( uint64_t );
    alignment = sizeof( uint64_t );
#endif

    pMemPool = DEMOGfxSharedPool->AllocSubPool( size, alignment );

    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_Write );
    info.SetSize( size );
    buffer.Initialize( &DEMODevice, info,
        pMemPool->GetPool(), pMemPool->GetBaseOffset(), pMemPool->GetSize() );

    buffer.GetGpuAddress( &address );
#if NN_GFX_IS_TARGET_GX
    u64* pData = buffer.Map< u64 >();
    pData[ 0 ] = GX2_INVALID_COUNTER_VALUE_U64;
    buffer.FlushMappedRange(0, sizeof( u64 ) );
    buffer.Unmap();
#endif
}

void DEMOGfxGpuTimestamp::Finalize()
{
    buffer.Finalize( &DEMODevice );
    pMemPool->Finalize();
}

void DEMOGfxGpuTimestamp::QueryTimestamp( nn::gfx::CommandBuffer* pCmd )
{
#if NN_GFX_IS_TARGET_GX
    GX2SampleBottomGPUCycle( reinterpret_cast< u64* >(
        static_cast< u32 >( address.ToData()->value ) ) );
#endif

#if NN_GFX_IS_TARGET_NVN
    nvnCommandBufferReportCounter( pCmd->ToData()->pNvnCommandBuffer, NVN_COUNTER_TYPE_TIMESTAMP, address.ToData()->value );
#endif

#if NN_GFX_IS_TARGET_GL
    pCmd->Gl4SetUserCommand( DEMOGfxTimestampQueryCallback, this );
#endif

#if NN_GFX_IS_TARGET_D3D
    NN_UNUSED( pCmd );
#endif
}

uint64_t DEMOGfxGpuTimestamp::GetTimestampResult( nn::gfx::Queue* pQueue, nn::gfx::CommandBuffer* pCmd )
{
    uint64_t value = 0;

#if NN_GFX_IS_TARGET_GL
    pCmd->Gl4SetUserCommand( DEMOGfxTimestampGetResultCallback, this );
#endif

    pCmd->End();
    pQueue->ExecuteCommand( &DEMOCommandBuffer, NULL );
    pQueue->Sync();
    pCmd->Begin();

#if NN_GFX_IS_TARGET_GX
    u64* pData = buffer.Map< u64 >();
    buffer.InvalidateMappedRange(0, sizeof( u64 ) );
    value = *pData;
    buffer.FlushMappedRange(0, sizeof( u64 ) );
    buffer.Unmap();
#endif

#if NN_GFX_IS_TARGET_NVN
    uint64_t* pData = buffer.Map< uint64_t >();
    buffer.InvalidateMappedRange(0, sizeof( uint64_t ) );
    value = pData[ 1 ]; // Second uint64_t is the timestamp
    buffer.FlushMappedRange( 0, sizeof( uint64_t ) );
    buffer.Unmap();
#endif

#if NN_GFX_IS_TARGET_GL
    TimeStampResult* pData = buffer.Map< TimeStampResult >();
    value = pData->value;
    buffer.FlushMappedRange( 0, sizeof( uint64_t ) );
    buffer.Unmap();
#endif

    return value;
}

float DEMOGfxGpuTimestamp::TicksToMicroseconds( uint64_t tick )
{
#if NN_GFX_IS_TARGET_GX
    return static_cast< float >(tick / static_cast< float >( GX2_TOP_BOTTOM_CLOCK_CYCLES / 1000000 ) );
#endif

#if NN_GFX_IS_TARGET_GL || NN_GFX_IS_TARGET_D3D
    return static_cast< float >( tick ) / 1000.0f;
#endif
#if NN_GFX_IS_TARGET_NVN
# ifdef __horizon__
    return static_cast< float >( tick ) * 1.625f / 1000.0f;
# else
    return static_cast< float >( tick ) / 1000.0f;
# endif
#endif
}

void DEMOGfxDebugTagIndent(const char* formatString, ...)
{
    va_list args;
    va_start(args, formatString);

#if NN_GFX_IS_TARGET_NVN
    const size_t BUFFER_SIZE = 1024;
    char buffer[ BUFFER_SIZE ];
    vsnprintf( buffer, BUFFER_SIZE, formatString, args );

    //nvnCommandBufferPushDebugGroup( DEMOCommandBuffer.ToData()->pNvnCommandBuffer, buffer );
    nvnCommandBufferInsertDebugMarker( DEMOCommandBuffer.ToData()->pNvnCommandBuffer, buffer );
#endif

#if NN_GFX_IS_TARGET_GX
    GX2DebugTagUserStringVA(GX2_DEBUG_TAG_INDENT, formatString, args);
#endif

    va_end(args);
}

void DEMOGfxDebugTagUndent()
{
#if NN_GFX_IS_TARGET_NVN
    nvnCommandBufferPopDebugGroup( DEMOCommandBuffer.ToData()->pNvnCommandBuffer );
#endif

#if NN_GFX_IS_TARGET_GX
    GX2DebugTagUserString(GX2_DEBUG_TAG_UNDENT, NULL);
#endif
}

void DEMOGfxDebugTagComment( const char* formatString, ... )
{
    va_list args;
    va_start(args, formatString);
#if NN_GFX_IS_TARGET_NVN
    const size_t BUFFER_SIZE = 1024;
    char buffer[ BUFFER_SIZE ];
    vsnprintf( buffer, BUFFER_SIZE, formatString, args );

    //nvnCommandBufferInsertDebugMarker( DEMOCommandBuffer.ToData()->pNvnCommandBuffer, buffer );
#endif

#if NN_GFX_IS_TARGET_GX
    GX2DebugTagUserStringVA(GX2_DEBUG_TAG_COMMENT, formatString, args);
#endif
    va_end(args);
}
