﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <gfx/demo.h>

/// @addtogroup demoTest
/// @{

enum _DEMOTestPerfIndices
{
    DEMOTestPerfIndices_FrameCpu = 0,
};

/// \brief Initializes Demo Test system
///
/// If the following arguments pass, Demo Test System runs.
/// If no arguments, Demo Test System is disabled.
///
/// - TEST_SELECT=#
/// - CAPTURE_SELECT=#
/// - CAPTURE_DIR=name
/// - COMPARE_FILE=name
/// - STOP_SELECT=#
/// - GX2R_DUMP_SELECT=#
/// - PM4_SELECT=#-#
/// - PM4_FILE=name
///
/// (e.g)\n
///   make run RUN_ARGS="TEST_SELECT=1,CAPTURE_SELECT=1,COMPARE_FILE=image/xxx.tga"\n
///   make run RUN_ARGS="STOP_SELECT=2,PM4_SELECT=2,PM4_FILE=xxx.4mp"\n
///
/// Where # should be replaced by a number, and "name" should be replaced by a file path name.\n
/// TEST_SELECT indicates the frame number when the DEMOTestCompare function is called.\n
/// CAPTURE_SELECT indicates the frame number when the DEMOTestCapture function is called.\n
/// CAPTURE_DIR indicates the output directory of captured image, it should be "assets:/somewhere" or DOS style full path. In NintendoSdk development tree assets indicates sdk root path. Only if specified file name with .bmp extension it outputs bmp format image.\n
/// COMPARE_FILE is the name(extension isn't needed) of the file containing the data to compare against, it should be relative path from "Externals\TestBinaries\Gfx\ReferenceImages\$test_name". Only works for tga file.\n
/// STOP_SELECT indicates the frame number when the demo should stop running.\n
/// GX2R_DUMP_SELECT indicates the frame number when the \ref GX2TempDumpResources function is called.\n
/// PM4_SELECT indicates the frame numbers when the \ref GX2DebugCaptureStart and \ref GX2DebugCaptureEnd functions are called.\n
/// PM4_FILE is the name of the .4mp file to store the PM4 capture in.\n
/// USE_HLSLCC_GLSL indicates that test uses HLSLcc converted GLSL.\n
/// \note The first frame is number 1. Using 0 for a SELECT disables that function.
/// \note This function should be called by the same core that calls GX2Init.
///
void DEMOTestInit(int argc, char *argv[]);

/// \brief Shuts down Demo Test system
///
void DEMOTestShutdown(void);

/// \brief Does per-frame check to see if other test APIs need to be called
/// (called internally by DEMOIsRunning)
void DEMOTestCheck(void);

/// \brief Called when Demo Test condition is met
/// (called internally by DEMOTestCheck)
void DEMOTestCompare(void);

/// \brief Called when Demo Capture condition is met
/// (called internally by DEMOTestCheck)
void DEMOTestCapture(void);

/// \brief Called in DEMOGfxBeforeRender when Demo Perf condition is met
void DEMOTestCheckPerfBegin(void);

/// \brief Called in DEMOGfxDoneRender when Demo Perf condition is met
void DEMOTestCheckPerfEnd(void);

/// \brief Set value of test result.
/// TRUE means test passed; FALSE means test failed.
void DEMOTestSetResult(bool result);

/// \brief Returns value of test result.
/// TRUE means test passed; FALSE means test failed.
/// The value -1 means the test has not yet occurred.
bool DEMOTestResult(void);

/// \brief Returns the current test frame number.
/// \note The first frame is number 1.
u32 DEMOTestFrame(void);

/// \brief Calc CRC-16 Hash (for Demo Test system)
///
/// \param pData Pointer to data to hash.
/// \param size  Size (in bytes) of data to hash.
///
/// \return result CRC-16 hash value
///
u16 DEMOTestHashCRC16(const void *pData, u32 size);

/// \brief Calc CRC-32 Hash (for Demo Test system)
///
/// \param pData Pointer to data to hash.
/// \param size  Size (in bytes) of data to hash.
///
/// \return result CRC-32 hash value
///
u32 DEMOTestHashCRC32(const void *pData, u32 size);

/// \brief Retrieve test name from file path
const char* DEMOTestGetTestName();

/// \brief Get HLSLcc is used or not
bool DEMOTestIsUseHlslccGlsl();

/// @}
