﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demo_Perf.h
//
// -----------------------------------------------------------------------------
#pragma once

#include <gfx/demo.h>

/// @addtogroup demoPerf
/// @{

/// \brief Initialize performance metric recording
/// \param maxTags Total number of debug tag sections
/// \param numMetrics Total metric entries in \p pMetrics. Setting to 0 indicates all metrics.
/// \param pMetrics Implementation specific metrics. For NX these correspond to
///        PerfWork metric names.
/// \return Performance metric context
void* DEMOPerfInit( uint32_t maxTags, uint32_t numMetrics, void* pMetrics );

/// \brief Registers an nn::gfx::Queue with the performance metrics utility
/// \param pContext Performance metric context returned by DEMOPerfInit.
/// \param pQueue nn::gfx::Queue that will be used for this metric data
void DEMOPerfRegisterQueue( void* pContext, nn::gfx::Queue* pQueue );

/// \brief Registers an nn::gfx::CommandBuffer with the performance metrics utility
/// \param pContext Performance metric context returned by DEMOPerfInit.
/// \param pCommandBuffer nn::gfx::CommandBuffer that will be used for this metric data
void DEMOPerfRegisterQueue( void* pContext, nn::gfx::Queue* pQueue );
void DEMOPerfRegisterCommandBuffer( void* pContext, nn::gfx::CommandBuffer* pCommandBuffer );

/// \brief Unregisters an nn::gfx::Queue with the performance metrics utility
/// \param pContext Performance metric context returned by DEMOPerfInit.
void DEMOPerfUnregisterQueue( void* pContext );

/// \brief Unregisters an nn::gfx::CommandBuffer with the performance metrics utility
/// \param pContext Performance metric context returned by DEMOPerfInit.
/// \param pCommandBuffer nn::gfx::Queue that was registered.
void DEMOPerfUnregisterCommandBuffer( void* pContext );

/// \brief Starts a performance metric frame
/// \param pContext Performance metric context returned by DEMOPerfInit.
void DEMOPerfStartFrame( void* pContext );

/// \brief Ends a performance metric frame
/// \param pContext Performance metric context returned by DEMOPerfInit.
void DEMOPerfEndFrame( void* pContext );

/// \brief Starts a performance metric pass
/// \param pContext Performance metric context returned by DEMOPerfInit.
void DEMOPerfStartPass( void* pContext );

/// \brief Ends a performance metric pass
/// \param pContext Performance metric context returned by DEMOPerfInit.
void DEMOPerfEndPass( void* pContext );

/// \brief Starts a performance metric tag section
/// \param pCommandBuffer nn::gfx::Queue that was registered.
/// \param tag Tag ID for the section
void DEMOPerfPushTag( void* pContext, uint32_t tag );

/// \brief End a performance metric tag section
/// \param pContext Performance metric context returned by DEMOPerfInit.
/// \param tag Tag ID for the section
void DEMOPerfPopTag( void* pContext, uint32_t tag );

/// \brief Shutdown performance metrics
/// \param pContext Performance metric context returned by DEMOPerfInit.
void DEMOPerfShutdown( void* pContext );

/// \brief Determine if enough passes have occurred to collect data.
/// \param pContext Performance metric context returned by DEMOPerfInit.
/// \return Returns true when the data is read or an internal error has occured.
bool DEMOPerfIsDataReady( void* pContext );

/// \brief Output metric data to the console
/// \param pContext Performance metric context returned by DEMOPerfInit.
void DEMOPerfPrintMetricData( void* pContext );

/// @} // demoCapture
