﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/// @defgroup VEC VEC
/// @ingroup MTXL

/// @defgroup QUAT QUAT
/// @ingroup MTXL

/// @defgroup MTX MTX
/// @ingroup MTXL

/*---------------------------------------------------------------------------*
    Matrix-Vector Library
 *---------------------------------------------------------------------------*/

#pragma once
#define __MTXVEC_H__ // Disable Cafe Headers

#ifdef CAFE
#include <cafe/os.h>
#else
#include <sys/types.h>
#include <stdint.h>
typedef uint32_t u32;
typedef int32_t s32;
typedef uint16_t u16;
typedef int16_t s16;
typedef float f32;
#ifdef _DEBUG
#ifndef ASSERTMSG
#define ASSERTMSG(exp, msg)                                     \
    (void) ((exp) ||                                            \
            (printf("%s %d: " msg, __FILE__, __LINE__)))
#endif
#else   // _DEBUG
#ifndef ASSERTMSG
#define ASSERTMSG(exp, msg)                                     ((void) 0)
#endif

#endif   // _DEBUG
#define MTX_USE_C
#endif

#include <gfx/demo_MtxGeoTypes.h>


/*---------------------------------------------------------------------------*
    Default function binding configuration
 *---------------------------------------------------------------------------*/
// [Binding Rule]
//
// "MTX_USE_ASM" -> When this flag is specified, it uses PS* (Paired-Single
//                 assembler code) functions for non-prefixed function calls.
// "MTX_USE_PS" -> When this flag is specified, it uses PS* (Paired-Single
//                 intrinsics code) functions for non-prefixed function calls.
// "MTX_USE_C " -> When this flag is specified, it uses C_* (C code) functions
//                 for non-prefixed function calls.
//
// The first binding specified in priority order listed will be used
// If nothing is specified, refers ASM* functions

#if ( !defined(MTX_USE_ASM) && !defined(MTX_USE_PS) && !defined(MTX_USE_C))
#define MTX_USE_ASM
#endif

/*---------------------------------------------------------------------------*
    Macro definitions
 *---------------------------------------------------------------------------*/

// MtxPtr offset to access next Mtx of an array
static const int MTX_PTR_OFFSET = 3;

// Mtx44Ptr offset to access next Mtx44 of an array
static const int MTX44_PTR_OFFSET = 4;

// "const" doesn't really work correctly for 2-dimensional mtx types
// If that changes, then redefine this as "const" (without quotes):
#define MTX_CONST

/// @addtogroup MTX
/// @{

// Degree <--> radian conversion macros
/// \brief Convert Degrees to Radians.
/// \param a Degrees
/// \return Radians
///
#define MTXDegToRad(a)   ( (a) *  0.01745329252f )

/// \brief Convert Radians to Degrees.
/// \param a Radians
/// \return Degrees
///
#define MTXRadToDeg(a)   ( (a) * 57.29577951f )

/// \brief Matrix-element-referencing macro.
/// Insulates user from changes from row-major to column-major and vice-versa.
/// Fully documents which index is row, which index is column.
///
/// \param m Matrix (\ref Mtx or \ref Mtx44)
/// \param r Row
/// \param c Column
/// \return Value of matrix at given row/column
///
#define MTXRowCol(m,r,c) ((m)[(r)][(c)])

/*---------------------------------------------------------------------------*
    Typedefs
 *---------------------------------------------------------------------------*/

/// Matrix stack for the \ref Mtx type.
typedef struct _MtxStack
{

    u32    numMtx; ///< Size of the matrix stack.
    MtxPtr stackBase; ///< Base pointer of the matrix stack.
    MtxPtr stackPtr; ///< Current stack pointer. NULL means an empty stack.

} MtxStack, *MtxStackPtr;

/// @}

/*---------------------------------------------------------------------------*
    GENERAL MATRIX SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup MTX
/// @{

// C version

/// \brief Set a 3x4 matrix to the identity.
///
/// \param m Matrix to be set.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXIdentity           ( Mtx m );

/// \brief Copies the contents of one 3x4 matrix into another
///
/// \param src Source matrix for the copy.
/// \param dst Destination matrix for the copy.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXCopy               ( MTX_CONST Mtx src, Mtx dst );

/// \brief Concatenates two 3x4 matrices
///
/// Order of operations is A x B = AB.
/// This function can handle the case when ab == a == b.
///
/// \param a First matrix to concatenate
/// \param b Second matrix to concatenate
/// \param ab Resulting matrix from concatenate
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXConcat             ( MTX_CONST Mtx a, MTX_CONST Mtx b, Mtx ab );

/// \brief Concatenates a 3x4 matrix to an array of 3x4 matrices.
/// The order of operations is A x B(array) = AB(array)
/// This routine is equivalent to:
///
/// dstBase[i] = A x srcBase[i] for all i = 0 to count - 1
///
/// \param a first matrix for concatenation
/// \param srcBase array base of second matrix for concatenation
/// \param dstBase array base of resulting matrix from concatenation
/// \param count number of matrices in srcBase and dstBase arrays
///
/// \warning This routine cannot check for array overflow.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXConcatArray        ( MTX_CONST Mtx a, MTX_CONST Mtx* srcBase, Mtx* dstBase, u32 count );

/// \brief Computes the transpose of a 3x4 matrix.
///
/// \note It is safe for src == xPose
///
/// \warning If the matrix is a 3x4 matrix, the fourth column (translation
/// component) is lost and becomes (0, 0, 0). This function is intended for
/// use in computing an inverse-transpose matrix to transform normals for
/// lighting. In this case, the loss of the translation component doesn't
/// matter.
///
/// \param src Source matrix
/// \param xPose Destination (transposed) matrix
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXTranspose          ( MTX_CONST Mtx src, Mtx xPose );

/// \brief Computes a fast inverse of a 3x4 matrix.
/// This algorithm works for matrices with a fourth row of (0, 0, 0, 1).
///
/// For a matrix:
///
/// M = | A  C |
///     | 0  1 |
///
/// Where A is the upper 3x3 submatrix and C is a 1x3 column vector:
///
/// INV(M) = | inv(A)   inv(A)*(-C) |
///          |   0          1       |
///
/// \note It is safe for src == inv.
///
/// \param src Source matrix
/// \param inv Destination (inverse) matrix
/// \return 0 if src is not invertible, 1 on success.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
u32     C_MTXInverse            ( MTX_CONST Mtx src, Mtx inv );

/// \brief Computes a fast inverse-transpose of a 3x4 matrix.
///
/// This algorithm works for matrices with a fourth row of (0, 0, 0, 1).
/// Commonly used for calculating normal transform matrices.
///
/// This function is equivalent to the combination of two functions
/// \ref MTXInverse + \ref MTXTranspose
///
/// \note It is safe to call this function if src == invX.
///
/// \param src Source matrix
/// \param invX Destination (inverse-transpose) matrix.
/// \return 0 if src is not invertible, 1 on success.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
u32     C_MTXInvXpose           ( MTX_CONST Mtx src, Mtx invX );

/// @}

// Bindings
#define MTXIdentity             C_MTXIdentity
#define MTXCopy                 C_MTXCopy
#define MTXConcat               C_MTXConcat
#define MTXConcatArray          C_MTXConcatArray
#define MTXTranspose            C_MTXTranspose
#define MTXInverse              C_MTXInverse
#define MTXInvXpose             C_MTXInvXpose

/*---------------------------------------------------------------------------*
    MATRIX-VECTOR SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup MTX
/// @{

// C version

/// \brief Multiplies a vector by a 3x4 matrix
///
/// dst = m x src
///
/// \note It is safe for src == dst.
///
/// \param m Matrix to multiply by
/// \param src Source vector of multiply
/// \param dst Resulting vector from multiply
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXMultVec            ( MTX_CONST Mtx m, const Vec *src, Vec *dst );

/// \brief Multiplies an array of vectors by a 3x4 matrix.
///
/// \note It is safe for srcBase == dstBase.
///
/// \warning This function cannot check for array overflow.
///
/// \param m Matrix to multiply by
/// \param srcBase Source vector array
/// \param dstBase Resulting vector array
/// \param count Number of vectors in srcBase and dstBase arrays.
///
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXMultVecArray       ( MTX_CONST Mtx m, const Vec *srcBase, Vec *dstBase, u32 count );

/// \brief Multiply a vector by a 3x4 Scaling and Rotating matrix
/// \note It is assumed that the 4th column (translation) is 0.
///
/// This is equivalent to:
///
/// dst = m x src
///
/// \note It is safe for src == dst.
///
/// \param m Matrix to multiply by
/// \param src Source vector for multiply
/// \param dst Resulting vector from multiply
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXMultVecSR          ( MTX_CONST Mtx m, const Vec *src, Vec *dst );

/// \brief Multiply an array of vector by a 3x4 Scaling and Rotating matrix
/// \note It is assumed that the 4th column (translation) is 0.
///
/// This is equivalent to:
///
/// dstBase[i] = m x srcBase[i]
///
/// \note It is safe for srcBase == dstBase.
///
/// \warning This function cannot check for array overflow
///
/// \param m Matrix to multiply by
/// \param srcBase Source vector array
/// \param dstBase Resulting vector array
/// \param count Number of vectors in srcBase and dstBase
///
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXMultVecArraySR     ( MTX_CONST Mtx m, const Vec *srcBase, Vec *dstBase, u32 count );

/// @}

// Bindings
#define MTXMultVec              C_MTXMultVec
#define MTXMultVecArray         C_MTXMultVecArray
#define MTXMultVecSR            C_MTXMultVecSR
#define MTXMultVecArraySR       C_MTXMultVecArraySR

/*---------------------------------------------------------------------------*
    MODEL MATRIX SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup MTX
/// @{

// C version

/// \brief Sets a rotation 3x4 matrix from a quaternion.
///
/// \param m Matrix to be set.
/// \param q Pointer to Quaternion
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXQuat             ( Mtx m, const Quaternion *q );

/// \brief Reflect a rotation 3x4 matrix with respect to a plane.
///
/// \param m Matrix to be set.
/// \param p point on the plane
/// \param n normal of the plane
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXReflect          ( Mtx m, const Vec *p, const Vec *n );

/// \brief Sets a translation 3x4 matrix
///
/// \param m Matrix to be set
/// \param xT x component of translation
/// \param yT y component of translation
/// \param zT z component of translation
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXTrans            ( Mtx m, f32 xT, f32 yT, f32 zT );

/// \brief Apply a translation to a 3x4 matrix.
/// This function is equivalent to \ref MTXTrans + \ref MTXConcat.
///
/// \note This is safe for the case where src == dst.
///
/// \param src Matrix to multiply the translation by
/// \param dst Resulting matrix from concatenation
/// \param xT x component of translation
/// \param yT y component of translation
/// \param zT z component of translation
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXTransApply       ( MTX_CONST Mtx src, Mtx dst, f32 xT, f32 yT, f32 zT );

/// \brief Sets a scale 3x4 matrix
///
/// \param m Matrix to be set
/// \param xS x component of scale
/// \param yS y component of scale
/// \param zS z component of scale
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXScale            ( Mtx m, f32 xS, f32 yS, f32 zS );

/// \brief Apply a scale to a 3x4 matrix.
/// This function is equivalent to \ref MTXScale + \ref MTXConcat.
///
/// \note This is safe for the case where src == dst.
///
/// \param src Matrix to multiply the scale by
/// \param dst Resulting matrix from concatenation
/// \param xS x component of scale
/// \param yS y component of scale
/// \param zS z component of scale
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXScaleApply       ( MTX_CONST Mtx src, Mtx dst, f32 xS, f32 yS, f32 zS );


/// \brief Sets a rotation 3x4 matrix about one of the X, Y or Z axes.
///
/// \note Counter clockwise rotation is positive.
///
/// \param m Matrix to be set
/// \param axis Principal axis of rotation. Must be 'X', 'x', 'Y', 'y', 'Z', or 'z'.
/// \param rad Rotation angle in radians
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXRotRad           ( Mtx m, char axis, f32 rad );

/// \brief Sets a rotation 3x4 matrix about one of the X, Y, or Z axes from specified trig ratios.
///
/// \note Counter clockwise rotation is positive.
///
/// \param m Matrix to be set
/// \param axis Principal axis of rotation. Must be 'X', 'x', 'Y', 'y', 'Z', or 'z'.
/// \param sinA Sine of rotation angle
/// \param cosA Cosine of rotation angle
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXRotTrig          ( Mtx m, char axis, f32 sinA, f32 cosA );

/// \brief Sets a rotation 3x4 matrix about an arbitrary axis.
///
/// \note Counter clockwise rotation is positive.
///
/// \param m Matrix to be set
/// \param axis Pointer to a vector containing the (x, y, z) axis components
/// \param rad Rotation angle in radians
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXRotAxisRad       ( Mtx m, const Vec *axis, f32 rad );

/// @}

// Bindings

#define MTXTrans                C_MTXTrans
#define MTXTransApply           C_MTXTransApply
#define MTXQuat                 C_MTXQuat
#define MTXReflect              C_MTXReflect
#define MTXScale                C_MTXScale
#define MTXScaleApply           C_MTXScaleApply
#define MTXRotRad               C_MTXRotRad
#define MTXRotTrig              C_MTXRotTrig
#define MTXRotDeg( m, axis, deg ) \
    C_MTXRotRad( m, axis, MTXDegToRad(deg) )
#define MTXRotAxisRad           C_MTXRotAxisRad
#define MTXRotAxisDeg( m, axis, deg ) \
    C_MTXRotAxisRad( m, axis, MTXDegToRad(deg) )

// Obsolete. Don't use this if possible.
#define MTXRotAxis              MTXRotAxisDeg


/*---------------------------------------------------------------------------*
    VIEW MATRIX SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup MTX
/// @{

// C version only so far

/// \brief Compute a 3x4 matrix to transform points to camera coordinates.
///
/// \param m Matrix to be set
/// \param camPos Camera position
/// \param camUp Camera 'up' direction
/// \param target Camera aim point
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXLookAt         ( Mtx            m,
                              const Point3d *camPos,
                              const Vec     *camUp,
                              const Point3d *target );

/// @}

// Bindings
#define MTXLookAt               C_MTXLookAt

/*---------------------------------------------------------------------------*
    PROJECTION MATRIX SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup MTX
/// @{

// C version only so far

/// \brief Compute a 4x4 perspective projection matrix from a specified view volume.
///
/// \param m 4x4 Matrix to be set
/// \param t Top coordinate of the viewing volume at the near clipping plane
/// \param b Bottom coordinate of the viewing volume at the near clipping plane
/// \param lf Left coordinate of the viewing volume at the near clipping plane
/// \param r Right coordinate of the viewing volume at the near clipping plane
/// \param n Positive distance from camera to the near clipping plane
/// \param f Positive distance from camera to the far clipping plane
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXFrustum        ( Mtx44 m, f32 t, f32 b, f32 lf, f32 r, f32 n, f32 f );

/// \brief Compute a 4x4 perspective projection matrix from the field of view and aspect ratio.
///
/// \param m 4x4 Matrix to be set
/// \param fovY Total field of view in degrees in the YZ plane
/// \param aspect Ratio of view window width:height (X / Y)
/// \param n Positive distance from camera to the near clipping plane
/// \param f Positive distance from camera to the far clipping plane
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXPerspective    ( Mtx44 m, f32 fovY, f32 aspect, f32 n, f32 f );

/// \brief Compute a 4x4 orthographic projection matrix.
///
/// \param m 4x4 Matrix to be set
/// \param t Top coordinate of the parallel view volume.
/// \param b Bottom coordinate of the parallel view volume.
/// \param lf Left coordinate of the parallel view volume.
/// \param r Right coordinate of the parallel view volume.
/// \param n Positive distance from camera to the near clipping plane
/// \param f Positive distance from camera to the far clipping plane
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXOrtho          ( Mtx44 m, f32 t, f32 b, f32 lf, f32 r, f32 n, f32 f );

/// @}

// Bindings
#define MTXFrustum              C_MTXFrustum
#define MTXPerspective          C_MTXPerspective
#define MTXOrtho                C_MTXOrtho

/*---------------------------------------------------------------------------*
    TEXTURE PROJECTION MATRIX SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup MTX
/// @{

// C version only so far

/// \brief Compute a 3x4 perspective projection matrix for texture projection.
///
/// \param m Matrix to be set
/// \param t Top coordinate of the viewing volume at the near clipping plane
/// \param b Bottom coordinate of the viewing volume at the near clipping plane
/// \param lf Left coordinate of the viewing volume at the near clipping plane
/// \param r Right coordinate of the viewing volume at the near clipping plane
/// \param n Positive distance from camera to the near clipping plane
/// \param scaleS Scale in the S direction for projected coordinates (usually 0.5)
/// \param scaleT Scale in the T direction for projected coordinates (usually 0.5)
/// \param transS Translate in the S direction for projected coordinates (usually 0.5)
/// \param transT Translate in the T direction for projected coordinates (usually 0.5)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXLightFrustum       ( Mtx m, f32 t, f32 b, f32 lf, f32 r, f32 n,
                                  f32 scaleS, f32 scaleT, f32 transS,
                                  f32 transT );

/// \brief Compute a 3x4 perspective projection matrix from field of view and aspect ratio for texture projection.
///
/// \param m Matrix to be set
/// \param fovY Total field of view in degrees in the YZ plane
/// \param aspect Ratio of view window width:height (X / Y)
/// \param scaleS Scale in the S direction for projected coordinates (usually 0.5)
/// \param scaleT Scale in the T direction for projected coordinates (usually 0.5)
/// \param transS Translate in the S direction for projected coordinates (usually 0.5)
/// \param transT Translate in the T direction for projected coordinates (usually 0.5)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXLightPerspective   ( Mtx m, f32 fovY, f32 aspect, f32 scaleS,
                                  f32 scaleT, f32 transS, f32 transT );

/// \brief Compute a 3x4 orthographic projection matrix for texture projection.
///
/// \param m Matrix to be set
/// \param t Top coordinate of the viewing volume at the near clipping plane
/// \param b Bottom coordinate of the viewing volume at the near clipping plane
/// \param lf Left coordinate of the viewing volume at the near clipping plane
/// \param r Right coordinate of the viewing volume at the near clipping plane
/// \param scaleS Scale in the S direction for projected coordinates (usually 0.5)
/// \param scaleT Scale in the T direction for projected coordinates (usually 0.5)
/// \param transS Translate in the S direction for projected coordinates (usually 0.5)
/// \param transT Translate in the T direction for projected coordinates (usually 0.5)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    C_MTXLightOrtho         ( Mtx m, f32 t, f32 b, f32 lf, f32 r, f32 scaleS,
                                  f32 scaleT, f32 transS, f32 transT );

/// @}

// Bindings
#define MTXLightFrustum         C_MTXLightFrustum
#define MTXLightPerspective     C_MTXLightPerspective
#define MTXLightOrtho           C_MTXLightOrtho

/*---------------------------------------------------------------------------*
    VECTOR SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup VEC
/// @{

// C version

/// \brief Add two vectors.
///
/// \note It is safe for ab == a == b
///
/// \param a First vector
/// \param b Second vector
/// \param ab Resulting vector (a + b)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void  C_VECAdd              ( const Vec *a, const Vec *b, Vec *ab );

/// \brief Subtract two vectors.
///
/// \note It is safe for a_b == a == b
///
/// \param a First vector
/// \param b Second vector
/// \param a_b Resulting vector (a - b)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void  C_VECSubtract         ( const Vec *a, const Vec *b, Vec *a_b );

/// \brief Scale a vector using a scalar.
///
/// \note It is safe for src == dst.
///
/// \param src Unscaled source vector
/// \param dst Scaled resultant vector (src * scale)
/// \param scale Scaling factor
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void  C_VECScale            ( const Vec *src, Vec *dst, f32 scale );

/// \brief Normalize a vector.
///
/// \note It is safe for src == unit.
///
/// \param src Non-unit source vector
/// \param unit Resulting unit vector (src / src magnitude)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void  C_VECNormalize        ( const Vec *src, Vec *unit );

/// \brief Compute the square of the magnitude of a vector.
///
/// \param v Source vector
/// \return Square magnitude of the vector
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
f32   C_VECSquareMag        ( const Vec *v );

/// \brief Compute the magnitude of a vector.
///
/// \param v Source vector
/// \return Magnitude of the vector
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
f32   C_VECMag              ( const Vec *v );

/// \brief Compute the dot product of two vectors.
///
/// \note Input vectors do not have to be normalized.
/// \note Input vectors are not normalized in the function.
///
/// \warning If direct cosine computation of the angle between a and b is desired, a and b should be normalized prior to calling \ref VECDotProduct.
///
/// \param a First vector
/// \param b Second vector
/// \result Dot product of the two vectors
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
f32   C_VECDotProduct       ( const Vec *a, const Vec *b );

/// \brief Compute the cross product of two vectors.
///
/// \note It is safe for axb == a == b
/// \note Input vectors do not have to be normalized.
/// \note Input vectors are not normalized in the function.
///
/// \param a First vector
/// \param b Second vector
/// \param axb Resulting cross product vector (a x b)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void  C_VECCrossProduct     ( const Vec *a, const Vec *b, Vec *axb );

/// \brief Returns the square of the distance between vectors a and b.
/// Distance can be calculated using the square root of the returned value.
///
/// \note It is safe for ab == a == b
///
/// \param a First vector
/// \param b Second vector
/// \return Square distance between the two vectors
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
f32   C_VECSquareDistance   ( const Vec *a, const Vec *b );

/// \brief Returns the distance between vectors a and b.
///
/// \note It is safe for ab == a == b
///
/// \param a First vector
/// \param b Second vector
/// \return Distance between the two vectors
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
f32   C_VECDistance         ( const Vec *a, const Vec *b );

/// \brief Reflect a vector about a normal to a surface.
///
/// \note It is safe for dst == src.
/// \note This function normalizes src and normal vectors.
///
/// \param src Incident vector
/// \param normal Normal to surface
/// \param dst Normalized reflected vector
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void  C_VECReflect          ( const Vec *src, const Vec *normal, Vec *dst );

/// \brief Compute the vector halfway between two vectors.
/// This is intended for use in computing specular highlights.
///
/// \note It is safe for half == a == b
/// \note Input vectors do not have to be normalized.
///
/// \param a First vector. This must point FROM the light source (tail) TO the surface (head).
/// \param b Second vector. This must point FROM the viewer (tail) TO the surface (head).
/// \param half Resulting normalized 'half-angle' vector.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void  C_VECHalfAngle        ( const Vec *a, const Vec *b, Vec *half );

/// @}

#define VECNormalize            C_VECNormalize
#define VECDotProduct           C_VECDotProduct
#define VECAdd                  C_VECAdd
#define VECSubtract             C_VECSubtract
#define VECScale                C_VECScale
#define VECSquareMag            C_VECSquareMag
#define VECMag                  C_VECMag
#define VECCrossProduct         C_VECCrossProduct
#define VECSquareDistance       C_VECSquareDistance
#define VECDistance             C_VECDistance

#define VECReflect              C_VECReflect
#define VECHalfAngle            C_VECHalfAngle

/*---------------------------------------------------------------------------*
    QUATERNION SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup QUAT
/// @{

/// \brief Quaternion epsilon comparison used for compare against 0.0f.
///
#define QUAT_EPSILON        0.00001F

// C version

/// \brief Returns the sum of two quaternions.
///
/// \note It is safe for p == q == r
///
/// \param p First quaternion
/// \param q Second quaternion
/// \param r Resulting quaternion (p + q)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATAdd              ( const Quaternion *p, const Quaternion *q, Quaternion *r );

/// \brief Returns the difference of two quaternions p-q.
///
/// \note It is safe for p == q == r
///
/// \param p First quaternion
/// \param q Second quaternion
/// \param r Resulting quaternion (p - q)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATSubtract         ( const Quaternion *p, const Quaternion *q, Quaternion *r );

/// \brief Returns the product of two quaternions.
/// The order of multiplication is important. (p*q != q*p)
///
/// \note It is safe for p == q == pq
///
/// \param p Left quaternion
/// \param q Right quaternion
/// \param pq Resulting quaternion product (p * q)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATMultiply         ( const Quaternion *p, const Quaternion *q, Quaternion *pq );

/// \brief Returns the ratio of two quaternions.
/// Creates a result r = p/q such that q*r = p (order of multiplication is important).
///
/// \note It is safe for p == q == r
///
/// \param p Left quaternion
/// \param q Right quaternion
/// \param r Resulting quaternion ratio (p / q)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATDivide           ( const Quaternion *p, const Quaternion *q, Quaternion *r );

/// \brief Scales a quaternion.
///
/// \note It is safe for q == r
///
/// \param q Quaternion
/// \param r Resulting scaled quaternion
/// \param scale Scaling factor
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATScale            ( const Quaternion *q, Quaternion *r, f32 scale );

/// \brief Returns the dot product of two quaternions.
///
/// \param p First quaternion
/// \param q Second quaternion
/// \return Dot product of the two quaternions
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
f32  C_QUATDotProduct       ( const Quaternion *p, const Quaternion *q );

/// \brief Normalizes a quaternion.
///
/// \note It is safe for src == unit
/// \warning If using MTX_USE_C and the magnitude of the quaternion is < \ref QUAT_EPSILON then the resulting quaternion is 0.
///
/// \param src Source quaternion
/// \param unit Resulting unit quaternion
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATNormalize        ( const Quaternion *src, Quaternion *unit );

/// \brief Returns the inverse of a quaternion.
///
/// \param src Source quaternion
/// \param inv Resulting inverse quaternion
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATInverse          ( const Quaternion *src, Quaternion *inv );

/// \brief Exponentiate quaternion (where q.w == 0).
///
/// \note It is safe for q == r
///
/// \param q Pure quaternion
/// \param r Resulting exponentiated quaternion (an unit quaternion)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATExp              ( const Quaternion *q, Quaternion *r );

/// \brief Returns the natural logarithm of a UNIT quaternion.
///
/// \note It is safe for q == r
///
/// \param q Unit quaternion
/// \param r Resulting logarithm quaternion (a pure quaternion)
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATLogN             ( const Quaternion *q, Quaternion *r );

/// \brief Modify q so it is on the same side of the hypersphere as qto
///
/// \note It is safe for p == q == r
///
/// \param q Quaternion
/// \param qto Quaternion to be close to
/// \param r Resulting modified quaternion
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATMakeClosest      ( const Quaternion *q, const Quaternion *qto, Quaternion *r );

/// \brief Returns the sum of two quaternions.
///
/// \note It is safe for p == q == r
///
/// \param r Resulting rotation quaternion
/// \param axis Rotation axis
/// \param rad Rotation angle in radians
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATRotAxisRad       ( Quaternion *r, const Vec *axis, f32 rad );

/// \brief Converts a matrix to a unit quaternion.
///
/// \note It is safe for p == q == r
///
/// \param r Resulting quaternion
/// \param m Input matrix
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATMtx              ( Quaternion *r, MTX_CONST Mtx m );

/// \brief Linear interpolation between two quaternions.
///
/// \note It is safe for p == q == r
///
/// \param p First quaternion
/// \param q Second quaternion
/// \param r Resulting quaternion (q*t + (1 - t) * p)
/// \param t Interpolation parameter
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATLerp             ( const Quaternion *p, const Quaternion *q, Quaternion *r, f32 t );

/// \brief Spherical linear interpolation of two quaternions
///
/// \note It is safe for p == q == r
///
/// \param p First quaternion
/// \param q Second quaternion
/// \param r Resulting interpolated quaternion (p + q)
/// \param t Interpolation parameter
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATSlerp            ( const Quaternion *p, const Quaternion *q, Quaternion *r, f32 t );

/// \brief Spherical cubic quadrangle interpolation of two quaternions with derived inner-quadrangle quaternions.
/// This will be used with the function \ref QUATCompA.
///
/// \note It is safe for p == q == r
///
/// \param p First quaternion
/// \param a Derived inner-quadrangle quaternion
/// \param b Derived inner-quadrangle quaternion
/// \param q Second quaternion
/// \param r Resulting quaternion (p + q)
/// \param t Interpolation value
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATSquad            ( const Quaternion *p, const Quaternion *a, const Quaternion *b,
                              const Quaternion *q, Quaternion *r, f32 t );

/// \brief Compute a, the term used in Boehm style interpolation.
///
/// a[n] = q[n] * qexp(-(1/4) * (logN(qinv(q[n])*q[n+1]) + logN(qinv(q[n])*q[n-1])))
///
/// \note This is safe for the case where qprev == q == qnext == a.
///
/// \param qprev Previous quaternion
/// \param q Current quaternion
/// \param qnext Next quaternion
/// \param a Resulting quaternion A
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_QUATCompA            ( const Quaternion *qprev, const Quaternion *q,
                              const Quaternion *qnext, Quaternion *a );

/// @}

#define QUATAdd                 C_QUATAdd
#define QUATSubtract            C_QUATSubtract
#define QUATMultiply            C_QUATMultiply
#define QUATDivide              C_QUATDivide
#define QUATScale               C_QUATScale
#define QUATDotProduct          C_QUATDotProduct
#define QUATNormalize           C_QUATNormalize
#define QUATInverse             C_QUATInverse

#define QUATExp                 C_QUATExp
#define QUATLogN                C_QUATLogN
#define QUATMakeClosest         C_QUATMakeClosest
#define QUATRotAxisRad          C_QUATRotAxisRad
#define QUATMtx                 C_QUATMtx
#define QUATLerp                C_QUATLerp
#define QUATSlerp               C_QUATSlerp
#define QUATSquad               C_QUATSquad
#define QUATCompA               C_QUATCompA

/*---------------------------------------------------------------------------*
    MATRIX STACK SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup MTX
/// @{


/// \brief Initializes a matrix stack size and stack ptr from a previously allocated stack
/// This resets the stack pointer to NULL(empty) and updates the stack size.
///
/// \note The stack (array) memory must have been previously allocated. Use \ref MTXAllocStack and \ref MTXFreeStack to create/destroy the stack.
///
/// \param sPtr Pointer to \ref MtxStack structure to be initialized
/// \param numMtx Number of matrices in the stack
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void    MTXInitStack          ( MtxStack *sPtr, u32 numMtx );

/// \brief Copy a matrix to stack pointer + 1.
/// Increment the stack pointer.
///
/// \param sPtr Pointer to MtxStack structure
/// \param m Matrix to copy into (stack pointer + 1) location
/// \return Returns the resulting stack pointer
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
MtxPtr  MTXPush               ( MtxStack *sPtr, MTX_CONST Mtx m );

/// \brief Concatenate a matrix with the current top of the stack push
/// the resulting matrix onto the stack.
/// This is intended for use in building forward transformations, so
/// concatentation is post-order:
///
/// (top of stack + 1) = (top of stack x m);
///
/// \param sPtr Pointer to MtxStack structure
/// \param m Matrix to concatenate with stack pointer and push to (stack pointer + 1)
/// \return Returns the resulting stack pointer
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
MtxPtr  MTXPushFwd            ( MtxStack *sPtr, MTX_CONST Mtx m );

/// \brief Concatenate the inverse of a matrix with the top of the stack
/// and push the resulting matrix onto the stack.
/// This is intended for building inverse transformations so concatenation
/// is pre-order:
///
/// (top of stack + 1) = (m x top of stack);
///
/// \note m is not modified by this function.
///
/// \param sPtr Pointer to MtxStack structure
/// \param m  Matrix to inverse-concatenate with stack pointer and push to (stack pointer + 1)
/// \return Returns the resulting stack pointer
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
MtxPtr  MTXPushInv            ( MtxStack *sPtr, MTX_CONST Mtx m );

/// \brief Concatenate the inverse-transpose of a matrix with the top of the
/// stack and push the resulting matrix onto the stack.
/// This is intended for building inverse-transpose matrix for forward
/// transformations of normals, so concatenation is post-order:
///
/// (top of stack + 1) = (top of stack x m);
///
/// \param sPtr Pointer to MtxStack structure
/// \param m  Matrix to inverse-concatenate with stack pointer and push to (stack pointer + 1)
/// \return Returns the resulting stack pointer
///
/// \par Usage
/// \note m is not modified by this function.
///
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
MtxPtr  MTXPushInvXpose       ( MtxStack *sPtr, MTX_CONST Mtx m );

/// \brief Decrement the stack pointer.
///
/// \param sPtr Pointer to MtxStack structure
/// \return Returns the stack pointer.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
MtxPtr  MTXPop                ( MtxStack *sPtr );

/// \brief Return the stack pointer.
///
/// \param sPtr Pointer to MtxStack structure
/// \return Returns the current stack pointer
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
MtxPtr  MTXGetStackPtr        ( const MtxStack *sPtr );

/// \brief Macro to create a matrix stack.
/// \note This allocates using MEMAllocFromDefaultHeap. This can be modified
/// by the user.
///
/// \param sPtr Pointer to MtxStack structure
/// \param numMtx Number of \ref Mtx structures to allocate for the stack.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \notthreadsafe \userheap \devonly \enddonotcall
///
#define MTXAllocStack( sPtr, numMtx ) (  ((MtxStackPtr)(sPtr))->stackBase = (MtxPtr)MEMAllocFromDefaultHeap( ( (numMtx) * sizeof(Mtx) ) )  )

/// \brief Macro to free a matrix stack.
/// \note This allocates using MEMFreeToDefaultHeap. This can be modified
/// by the user.
///
/// \param sPtr Pointer to MtxStack structure
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \notthreadsafe \userheap \devonly \enddonotcall
///
#define MTXFreeStack( sPtr )    (  MEMFreeToDefaultHeap( (void*)( ((MtxStackPtr)(sPtr))->stackBase ) )  )

/// @}

/*---------------------------------------------------------------------------*
    SPECIAL PURPOSE MATRIX SECTION
 *---------------------------------------------------------------------------*/

/// @addtogroup MTX
/// @{

/// \brief Creates a reordered (column-major) matrix from a row-major matrix.
/// This is useful for getting better performance for the MTXRO* functions.
///
/// \warning It is not safe to have src == dst.
///
/// \param src Source matrix
/// \param dest Destination matrix, note type is ROMtx.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_MTXReorder       ( MTX_CONST Mtx src, ROMtx dest );

/// \brief Multiplies an array of vectors by a reordered matrix.
///
/// \note It is ok for source == destination.
/// \note Number of vertices transformed cannot be less than 2.
///
/// \param m Reordered matrix
/// \param srcBase Start of source vector array
/// \param dstBase Start of the resulting vector array
/// \param count Number of vectors in srcBase and dstBase arrays. Count must be greater than 2.
///
/// \par Usage
///  - Add "#define MTX_USE_ASM" prior to including the mtx header to use PPC Assembly based version
///  - Add "#define MTX_USE_PS" prior to including the mtx header to use PPC Paired-Single instruction based version
///  - Add "#define MTX_USE_C" prior to including the mtx header to use C implementation (default). This is useful for debugging.
///
/// \donotcall \threadsafe \devonly \enddonotcall
///
void C_MTXROMultVecArray( MTX_CONST ROMtx m,  const Vec *srcBase, Vec *dstBase, u32 count );

/// @}

// Bindings
#define MTXReorder            C_MTXReorder

#define MTXROMultVecArray     C_MTXROMultVecArray

/*---------------------------------------------------------------------------*/

/*===========================================================================*/

