﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#if NN_GFX_IS_TARGET_GX
#include <cafe/gx2.h>

#define DEMO_MAX_SAMPLERS           GX2_MAX_SAMPLERS
#define DEMO_MAX_VS_UNIFORM_VARS    GX2_MAX_VS_UNIFORM_VARS
#define DEMO_MAX_PS_UNIFORM_VARS    GX2_MAX_PS_UNIFORM_VARS
#define DEMO_MAX_UNIFORM_BLOCKS     GX2_MAX_UNIFORM_BLOCKS
#define DEMO_MAX_ATTRIB_BUFFERS     GX2_MAX_ATTRIB_BUFFERS
#else
// FIXME: these numbers are bogus
static const int DEMO_MAX_SAMPLERS =          (32);
static const int DEMO_MAX_VS_UNIFORM_VARS =   (64);
static const int DEMO_MAX_PS_UNIFORM_VARS =   (64);
static const int DEMO_MAX_UNIFORM_BLOCKS  =   (32);
static const int DEMO_MAX_ATTRIB_BUFFERS  =   (32);
#endif

/// @addtogroup demoShader
/// @{

/// \brief The structure for vertex/pixel samplers
///
/// Each member map to each sampler number.
///
struct DEMOGfxSamplers
{
    /// Current number of sampler
    u32             count;

    /// The locations in the shader.
    u32             location[DEMO_MAX_SAMPLERS];

    DEMOGfxSamplers()
    {
        count = 0;
    }
};


/// \brief The structure for vertex uniforms
///
/// Each member map to each uniform number.
///
struct DEMOGfxUniformsVS
{
    /// Current number of uniforms
    u32             count;

    /// The locations in the vertex shader.
    u32             location[DEMO_MAX_VS_UNIFORM_VARS];

    DEMOGfxUniformsVS()
    {
        count = 0;
    }
};


/// \brief The structure for pixel uniforms
///
/// Each member map to each uniform number.
///
struct DEMOGfxUniformsPS
{
    /// Current number of uniforms
    u32             count;

    /// The locations in the pixel shader.
    u32             location[DEMO_MAX_PS_UNIFORM_VARS];

    DEMOGfxUniformsPS()
    {
        count = 0;
    }
};

/// \brief The structure for vertex uniform blocks
///
/// Each member map to each uniform number.
///
struct DEMOGfxUniformBlocks
{
    /// Current number of uniform blocks
    u32             count;

    /// The locations in the shader.
    u32             location[DEMO_MAX_UNIFORM_BLOCKS];

    /// The size of uniform blocks
    /// (not supported by nn::gfx)
    //u32             size[GX2_MAX_UNIFORM_BLOCKS];

    DEMOGfxUniformBlocks()
    {
        count = 0;
    }

};

/// \brief The structure for shader data
///
/// Includes members to setup shader
///
class DEMOGfxShader
{
protected:
    /// nn::gfx shaders
    nn::gfx::Shader*   pShader;
#if !NN_GFX_IS_TARGET_GX
    nn::gfx::ResShaderFile* pResShaderFile;
    nn::gfx::ResShaderContainer* pResShaderContainer;
#endif

public:
    nn::gfx::Shader* GetShader() { return pShader;  }
    void SetShader(nn::gfx::Shader* s) { pShader = s; }

    int GetInterfaceSlot( nn::gfx::ShaderStage shaderStage, nn::gfx::ShaderInterfaceType shaderInterfaceType, const char* pName)
    {
        return pShader->GetInterfaceSlot(shaderStage, shaderInterfaceType, pName);
    }
    /// Vertex Attributes used by the shader
    nn::gfx::VertexAttributeStateInfo   vertexAttributes[ DEMO_MAX_ATTRIB_BUFFERS ];

    /// Vertex Buffers used by the shader
    nn::gfx::VertexBufferStateInfo      vertexBuffers[ DEMO_MAX_ATTRIB_BUFFERS ];

    /// Current number of attributes
    u32               attribCount;

    /// Current number of buffers
    u32               bufferCount;

    /// The structure holds vertex shader samplers locations
    DEMOGfxSamplers   samplersVS;

    /// The structure holds pixel shader samplers locations
    DEMOGfxSamplers   samplersPS;

    /// The structure holds geometry shader samplers locations
    DEMOGfxSamplers   samplersGS;

    /// The structure holds vertex shader uniforms locations
    DEMOGfxUniformsVS uniformsVS;

    /// The structure holds pixel shader uniforms locations
    DEMOGfxUniformsPS uniformsPS;

    // GS is UB only

    /// The structure holds vertex shader uniform blocks location and size
    DEMOGfxUniformBlocks uniformBlocksVS;

    /// The structure holds pixel shader uniform blocks location and size
    DEMOGfxUniformBlocks uniformBlocksPS;

    /// The structure holds geometry shader uniform blocks location and size
    DEMOGfxUniformBlocks uniformBlocksGS;

    void* vertexShaderData;
    void* hullShaderData; ///!< Hull Shader/Tessellation Control Shader
    void* domainShaderData; ///!< Domain Shader/Tessellation Evaluation Shader
    void* geomShaderData;
    void* pixelShaderData;
    void* computeShaderData;

    DEMOGfxShader()
    {
        vertexShaderData = NULL;
        pixelShaderData = NULL;
        geomShaderData = NULL;
        hullShaderData = NULL;
        domainShaderData = NULL;
        computeShaderData = NULL;

        attribCount = 0;
        bufferCount = 0;
    }

    friend void DEMOGfxLoadShadersFromFile( DEMOGfxShader* pShader, u32 index, const char* fileName );
    friend BOOL DEMOGfxFreeShaders( DEMOGfxShader* pShader );
};

//
// Functions
//

/// \brief Checks number of shader location
///
/// \param location number of shader location
/// \return result (TRUE or FALSE)
///
inline BOOL DEMOGfxCheckShaderLocation(u32 location)
{
    if(location == (u32)(-1))
    {
        OSReport("Warning : Couldn't find the correct location.\n");
        return FALSE;
    }

    return TRUE;
}

#if NN_GFX_IS_TARGET_GX
/// \brief Loads shaders from file buffer (.gsh)
///
/// \param pShader Pointer to DEMOGfxShader structure
/// \param index   Index of shader in the file buffer
/// \param pData   Pointer to the file buffer
/// \return result (TRUE or FALSE)
///
BOOL DEMOGfxLoadShaders(DEMOGfxShader *pShader, u32 index, const void *pData);
#endif

/// \brief Loads shaders from a file
///
/// \param pShader Pointer to DEMOGfxShader structure
/// \param index   Index of shader in the file buffer
/// \param fileName Pointer to the file name
///
void DEMOGfxLoadShadersFromFile(DEMOGfxShader* pShader, u32 index, const char* fileName);

/// \brief Sets attribute members to DEMOGfxShader structure and initializes stream
///
/// \param pShader       Pointer to DEMOGfxShader structure
/// \param name          Pointer to name of attribute (e.g a_position, a_color etc)
/// \param bufferIndex   Index of attribute buffer
/// \param offset        Offset value for attribute data
/// \param format        Attribute format
/// \return result (TRUE or FALSE)
///
BOOL DEMOGfxInitShaderAttribute(DEMOGfxShader *pShader, const char *name,
                                u32 bufferIndex, u32 offset, nn::gfx::AttributeFormat format);

/// \brief Sets vertexBuffer members to DEMOGfxShader structure and initializes stream
///
/// \param pShader       Pointer to DEMOGfxShader structure
/// \param bufferIndex   Index of attribute buffer
/// \param stride        Stride of the vertex buffer
/// \param divisor       Instancing divisor of the buffer. Set to 0 for no instancing.
///
void DEMOGfxInitShaderVertexBuffer( DEMOGfxShader *pShader, u32 bufferIndex,
    u32 stride, u32 divisor );

/// \brief Gets vertex shader sampler location and sets to DEMOGfxShader structure
///
/// \param pShader Pointer to DEMOGfxShader structure
/// \param name    Pointer to name of sampler (e.g s_texture etc)
/// \return result (TRUE or FALSE)
///
BOOL DEMOGfxGetVertexShaderSamplerLocation(DEMOGfxShader *pShader, const char *name);

/// \brief Gets pixel shader sampler location and sets to DEMOGfxShader structure
///
/// \param pShader Pointer to DEMOGfxShader structure
/// \param name    Pointer to name of sampler (e.g s_texture etc)
/// \return result (TRUE or FALSE)
///
BOOL DEMOGfxGetPixelShaderSamplerLocation(DEMOGfxShader *pShader, const char *name);

/// \brief Gets geometry shader sampler location and sets to DEMOGfxShader structure
///
/// \param pShader Pointer to DEMOGfxShader structure
/// \param name    Pointer to name of sampler (e.g s_texture etc)
/// \return result (TRUE or FALSE)
///
BOOL DEMOGfxGetGeometryShaderSamplerLocation(DEMOGfxShader *pShader, const char *name);

/// \brief Gets vertex shader uniform block location and sets to DEMOGfxShader structure
///
/// \param pShader Pointer to DEMOGfxShader structure
/// \param name    Pointer to name of uniform block (e.g ub_staticBlock etc)
/// \return result (TRUE or FALSE)
///
BOOL DEMOGfxGetVertexShaderUniformBlockLocation(DEMOGfxShader *pShader, const char *name);

/// \brief Gets pixel shader uniform block location and sets to DEMOGfxShader structure
///
/// \param pShader Pointer to DEMOGfxShader structure
/// \param name    Pointer to name of uniform block (e.g ub_dynamicBlock etc)
/// \return result (TRUE or FALSE)
///
BOOL DEMOGfxGetPixelShaderUniformBlockLocation(DEMOGfxShader *pShader, const char *name);

/// \brief Gets geometry shader uniform block location and sets to DEMOGfxShader structure
///
/// \param pShader Pointer to DEMOGfxShader structure
/// \param name    Pointer to name of uniform block (e.g ub_dynamicBlock etc)
/// \return result (TRUE or FALSE)
///
BOOL DEMOGfxGetGeometryShaderUniformBlockLocation(DEMOGfxShader *pShader, const char *name);

/// \brief Free shaders (Vertex, Pixel, Geometry and Fetch shader )
///
/// \param pShader Pointer to DEMOGfxShader structure
///
BOOL DEMOGfxFreeShaders(DEMOGfxShader *pShader);

/// @}
