﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "testG3d_Main.h"

namespace
{

const uint8_t g_AnswerVertexData[] =
{
    0x35, 0x2e, 0xf0, 0x44, 0x87, 0x35, 0x00, 0x3c,
    0x00, 0x9c, 0x7e, 0x5f, 0x0d, 0x0d, 0x0d, 0x01,
    0xff, 0x00, 0x00, 0xff, 0xc0, 0xa2, 0x69, 0x0a,
    0x35, 0x2e, 0xec, 0x44, 0x76, 0x35, 0x00, 0x3c,
    0x00, 0xa0, 0xfc, 0x5c, 0x0d, 0x0d, 0x0d, 0x01,
    0xff, 0x00, 0x00, 0xff, 0xc0, 0xa2, 0x3f, 0x0d
};

const uint16_t g_AnswerIndexData[] =
{
    0, 1, 2, 0, 3, 1, 0, 4, 3, 0, 2, 5, 0, 5
};

const nn::util::Vector3fType g_AnswerBoundingSphereCenter = NN_UTIL_VECTOR_3F_INITIALIZER(0.085770123f, 2.895209f, 0.019090641f);

const float g_AnswerBoundingSphereRadius = 4.475006f;

nn::g3d::TextureRef TextureBindCallback(const char* name, void* pUserData)
{
    NN_UNUSED(pUserData);
    nn::g3d::TextureRef textureRef;
    if (std::strcmp(name, "human_A") == 0)
    {
        textureRef.SetTextureView(reinterpret_cast<nn::gfx::TextureView*>(0x12345678));
        nn::gfx::DescriptorSlot descriptorSlot;
        memset(&descriptorSlot, 0xa5, sizeof(descriptorSlot));
        textureRef.SetDescriptorSlot(descriptorSlot);
        return textureRef;
    }
    return textureRef;
}

nn::g3d::TextureRef NothingTextureBindCallback(const char* name, void* pUserData)
{
    NN_UNUSED(name);
    NN_UNUSED(pUserData);

    return nn::g3d::TextureRef();
}

} // namespace

TEST_F(G3dTest, ResFile)
{
    EXPECT_TRUE(nn::g3d::ResFile::IsValid(GetResFile()));

    const nn::util::BinaryFileHeader* pHeader = GetResFile()->GetFileHeader();
    EXPECT_TRUE(pHeader->IsValid(nn::g3d::ResFile::Signature, NN_G3D_MODEL_BINARY_VERSION_MAJOR,
                                                              NN_G3D_MODEL_BINARY_VERSION_MINOR,
                                                              NN_G3D_MODEL_BINARY_VERSION_MICRO));
    EXPECT_STREQ("human", GetResFile()->GetName());

    // Model
    EXPECT_EQ(2, GetResFile()->GetModelCount());


    void* ptr = reinterpret_cast<void*>(0x12345678);
    GetResFile()->SetUserPtr(ptr);
    EXPECT_EQ(ptr, GetResFile()->GetUserPtr());

    // External File
    EXPECT_EQ(0, GetResFile()->GetExternalFileCount());

    nn::g3d::BindResult result = GetResFile()->BindTexture(TextureBindCallback, NULL);
    EXPECT_FALSE(result.IsComplete());
    EXPECT_FALSE(result.IsMissed());
    EXPECT_TRUE(result.IsBound());
    EXPECT_TRUE(result.IsNotBound());
    GetResFile()->ReleaseTexture();

    result = GetResFile()->BindTexture(NothingTextureBindCallback, NULL);
    EXPECT_FALSE(result.IsComplete());
    EXPECT_TRUE(result.IsMissed());
    EXPECT_FALSE(result.IsBound());
    EXPECT_TRUE(result.IsNotBound());
    GetResFile()->ReleaseTexture();
}

TEST_F(G3dTest, ResModel)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    EXPECT_TRUE(pResModel != NULL);
    EXPECT_TRUE(pResModel == GetResFile()->GetModel(0));
    EXPECT_STREQ("human", GetResFile()->GetModelName(GetResFile()->FindModelIndex("human")));

    EXPECT_STREQ("human", pResModel->GetName());

    void* ptr = reinterpret_cast<void*>(0x12345678);
    GetResFile()->SetUserPtr(ptr);
    EXPECT_EQ(ptr, GetResFile()->GetUserPtr());

    EXPECT_EQ(1, pResModel->GetVertexCount());
    EXPECT_EQ(1, pResModel->GetMaterialCount());
    EXPECT_EQ(1, pResModel->GetShapeCount());
    EXPECT_EQ(5, pResModel->GetUserDataCount());

    nn::g3d::BindResult result = pResModel->BindTexture(TextureBindCallback, NULL);
    EXPECT_TRUE(result.IsComplete());
    EXPECT_TRUE(result.IsBound());
    EXPECT_FALSE(result.IsNotBound());
    nn::g3d::TextureRef textureRef;
    textureRef.SetTextureView(reinterpret_cast<nn::gfx::TextureView*>(ptr));
    EXPECT_TRUE(pResModel->ForceBindTexture(textureRef, "human_A"));
    pResModel->ReleaseTexture();


    // nn::g3d::ResModel::ResetVertexBuffer() のテスト
    {
        pResModel = GetResFile()->FindModel("MiiSeparateBone");

        // インデックスバッファは Setup() 直後の nn::g3d::ResMesh::pIndexBuffer の値を正解値とします。
        // g3d からオリジナルのポインタの値を取得できる口が無いので 3dBinaryConverter のメモリレイアウト変更時に弾くため。
        int shapeCount = pResModel->GetShapeCount();
        int modelMeshNum = 0;
        for(int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex )
        {
            auto pResShape = pResModel->GetShape( shapeIndex );
            modelMeshNum += pResShape->GetMeshCount();
        }
        nn::gfx::Buffer** goldenIndexBuffPtrArray = new nn::gfx::Buffer*[modelMeshNum];
        int idxModelMesh = 0;
        for(int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex )
        {
            auto pResShape = pResModel->GetShape( shapeIndex );
            for( int meshIndex = 0; meshIndex < pResShape->GetMeshCount(); ++meshIndex, ++idxModelMesh )
            {
                auto pResMesh = pResShape->GetMesh( meshIndex );
                goldenIndexBuffPtrArray[idxModelMesh] = pResMesh->GetIndexBuffer();
            }
        }

        // nn::g3d::ResVertexData::pVertexBufferPtrArray を汚します。
        int vertexCount = pResModel->GetVertexCount();
        for( int idxVertex = 0; idxVertex < vertexCount; ++idxVertex )
        {
            auto pResVertex = pResModel->GetVertex( idxVertex );
            for( int idxVertexBuff = 0; idxVertexBuff < pResVertex->GetVertexBufferCount(); ++idxVertexBuff )
            {
                // ポインタの書き換え
                nn::g3d::ResVertexData& resVertexData = const_cast<nn::g3d::ResVertex*>( pResVertex )->ToData();
                void* dest = &resVertexData.pVertexBufferPtrArray.Get()[idxVertexBuff];
                size_t size = sizeof( void* );
                memset( dest, 0xcd, size );    // ポインタサイズは環境依存
            }
        }
        // nn::g3d::ResMeshData::pIndexBuffer を汚します。
        for(int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex )
        {
            auto pResShape = pResModel->GetShape( shapeIndex );
            for( int meshIndex = 0; meshIndex < pResShape->GetMeshCount(); ++meshIndex, ++idxModelMesh )
            {
                auto pResMesh = pResShape->GetMesh( meshIndex );
                auto& resMeshData = pResMesh->ToData();
                void* dest = &resMeshData.pIndexBuffer;
                size_t size = sizeof( void* );
                memset( dest, 0xcd, size );    // ポインタサイズは環境依存
            }
        }

        // リセット
        const_cast<nn::g3d::ResModel*>( pResModel )->Reset();

        // 正解値比較
        // 頂点バッファ
        for( int idxVertex = 0; idxVertex < vertexCount; ++idxVertex )
        {
            auto pResVertex = pResModel->GetVertex( idxVertex );
            for( int idxVertexBuff = 0; idxVertexBuff < pResVertex->GetVertexBufferCount(); ++idxVertexBuff )
            {
                auto goldenVal = &pResVertex->ToData().pVertexBufferArray.Get()[idxVertexBuff];
                EXPECT_EQ( goldenVal, pResVertex->GetVertexBuffer( idxVertexBuff ) );
            }
        }
        // インデックスバッファ
        idxModelMesh = 0;
        for(int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex )
        {
            auto pResShape = pResModel->GetShape( shapeIndex );
            for( int meshIndex = 0; meshIndex < pResShape->GetMeshCount(); ++meshIndex, ++idxModelMesh )
            {
                auto pResMesh = pResShape->GetMesh( meshIndex );
                EXPECT_EQ( goldenIndexBuffPtrArray[idxModelMesh] , pResMesh->GetIndexBuffer() );
            }
        }

        delete[] goldenIndexBuffPtrArray;
    }
}

TEST_F(G3dTest, ResVertex)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResVertex* pResVertex = pResModel->GetVertex(0);
    EXPECT_TRUE(pResVertex != NULL);

    void* ptr = reinterpret_cast<void*>(0x12345678);
    pResVertex->SetUserPtr(ptr);
    EXPECT_EQ(ptr, pResVertex->GetUserPtr());

    EXPECT_EQ(0, pResVertex->GetIndex());
    EXPECT_EQ(3065, pResVertex->GetCount());
    EXPECT_EQ(5, pResVertex->GetVertexAttrCount());
    EXPECT_EQ(1, pResVertex->GetVertexBufferCount());
    EXPECT_EQ(3, pResVertex->GetVertexInfluenceCount());

    EXPECT_EQ(24, pResVertex->GetVertexBufferStride(0));
    nn::gfx::Buffer* pBuffer = pResVertex->GetVertexBuffer(0);
    EXPECT_TRUE(pBuffer != NULL);
    nn::gfx::Buffer::InfoType* pBufferInfo = pResVertex->GetVertexBufferInfo(0);
    EXPECT_TRUE(pBufferInfo != NULL);
    const float* pMapBuffer = pBuffer->Map<const float>();
    EXPECT_EQ(0, ::std::memcmp(pMapBuffer, g_AnswerVertexData, sizeof(g_AnswerVertexData)));
    pBuffer->Unmap();
    //size_t size = pResVertex->GetVertexBufferSize(0);
    //EXPECT_TRUE(size == pBufferView->ToData()->size);
}

TEST_F(G3dTest, ResVertexAttrib)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResVertex* pResVertex = pResModel->GetVertex(0);
    nn::g3d::ResVertexAttr* pResVertexAttr = pResVertex->FindVertexAttr("_n0");
    EXPECT_TRUE(pResVertexAttr != NULL);
    EXPECT_TRUE(pResVertexAttr == pResVertex->GetVertexAttr(1));
    EXPECT_STREQ("_n0", pResVertex->GetVertexAttrName(pResVertex->FindVertexAttrIndex("_n0")));

    EXPECT_STREQ("_n0", pResVertexAttr->GetName());
    EXPECT_EQ(nn::gfx::AttributeFormat_10_10_10_2_Snorm, pResVertexAttr->GetFormat());
    EXPECT_EQ(0, pResVertexAttr->GetBufferIndex());
    EXPECT_EQ(8, pResVertexAttr->GetOffset());
}

TEST_F(G3dTest, ResShape)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResShape* pResShape = pResModel->FindShape("polygon3");
    EXPECT_TRUE(pResShape != NULL);
    EXPECT_TRUE(pResShape == pResModel->GetShape(0));
    EXPECT_STREQ("polygon3", pResModel->GetShapeName(pResModel->FindShapeIndex("polygon3")));

    void* ptr = reinterpret_cast<void*>(0x12345678);
    pResShape->SetUserPtr(ptr);
    EXPECT_EQ(ptr, pResShape->GetUserPtr());

    EXPECT_STREQ("polygon3", pResShape->GetName());
    EXPECT_EQ(0, pResShape->GetIndex());
    EXPECT_EQ(0, pResShape->GetBoneIndex());
    EXPECT_EQ(0, pResShape->GetMaterialIndex());
    EXPECT_EQ(0, pResShape->GetVertexIndex());
    EXPECT_EQ(53, pResShape->GetSkinBoneIndexCount());
    EXPECT_EQ(3, pResShape->GetVertexSkinCount());

    EXPECT_TRUE(pResShape->IsVertexOwner());
    EXPECT_FALSE(pResShape->IsRigidBody());
    EXPECT_FALSE(pResShape->IsRigidSkinning());
    EXPECT_TRUE(pResShape->IsSmoothSkinning());
    EXPECT_TRUE(pResShape->GetVertex() == pResModel->GetVertex(0));

    EXPECT_EQ(1, pResShape->GetMeshCount());
    EXPECT_EQ(1, pResShape->GetSubMeshCount());
    EXPECT_EQ(0, pResShape->GetKeyShapeCount());
    EXPECT_EQ(0, pResShape->GetTargetAttribCount());

    nn::g3d::Bounding bounding = pResShape->GetBounding();
    EXPECT_FLOAT_EQ(0.55662012f,      bounding.center.x); // x
    EXPECT_FLOAT_EQ(-1.01327896e-006f, bounding.center.y); // y
    EXPECT_FLOAT_EQ(-0.074681848f,    bounding.center.z); // z
    EXPECT_FLOAT_EQ(0.79700816f,      bounding.extent.x); // x
    EXPECT_FLOAT_EQ(0.568395019f,      bounding.extent.y); // y
    EXPECT_FLOAT_EQ(0.51746404f,      bounding.extent.z); // z
    EXPECT_FLOAT_EQ(0.92987555f,      pResShape->GetRadius());
}

TEST_F(G3dTest, ResMesh)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResShape* pResShape = pResModel->FindShape("polygon3");
    nn::g3d::ResMesh* pResMesh = pResShape->GetMesh(0);
    EXPECT_TRUE(pResMesh != NULL);

    EXPECT_EQ(1, pResMesh->GetSubMeshCount());
    EXPECT_EQ(14628, pResMesh->GetCount());
    EXPECT_EQ(nn::gfx::PrimitiveTopology_TriangleList, pResMesh->GetPrimitiveType());
    EXPECT_EQ(nn::gfx::IndexFormat_Uint16, pResMesh->GetIndexFormat());

    nn::gfx::Buffer* pBuffer = pResMesh->GetIndexBuffer();
    EXPECT_TRUE(pBuffer != NULL);
    void* pMapBuffer = pBuffer->Map();
    EXPECT_EQ(0, ::std::memcmp(pMapBuffer, g_AnswerIndexData, sizeof(g_AnswerIndexData)));
    pBuffer->Unmap();
    //size_t size = pResMesh->GetIndexBufferSize();
    //EXPECT_TRUE(size == pBufferView->ToData()->size);
}

TEST_F(G3dTest, ResSubMesh)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResShape* pResShape = pResModel->FindShape("polygon3");
    nn::g3d::ResMesh* pResMesh = pResShape->GetMesh(0);
    nn::g3d::ResSubMesh* pResSubMesh = pResMesh->GetSubMesh(0);
    EXPECT_TRUE(pResSubMesh != NULL);

    EXPECT_EQ(0, pResSubMesh->GetOffset());
    EXPECT_EQ(14628, pResSubMesh->GetCount());
}

TEST_F(G3dTest, ResMaterial)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResMaterial* pResMaterial = pResModel->FindMaterial("human_all");
    EXPECT_TRUE(pResMaterial != NULL);
    EXPECT_TRUE(pResMaterial == pResModel->GetMaterial(0));
    EXPECT_STREQ("human_all", pResModel->GetMaterialName(pResModel->FindMaterialIndex("human_all")));

    EXPECT_STREQ("human_all", pResMaterial->GetName());
    EXPECT_EQ(0, pResMaterial->GetIndex());

    void* ptr = reinterpret_cast<void*>(0x12345678);
    pResMaterial->SetUserPtr(ptr);
    EXPECT_EQ(ptr, pResMaterial->GetUserPtr());

    EXPECT_EQ(11, pResMaterial->GetShaderParamCount());
    EXPECT_FALSE(pResMaterial->HasVolatile());
    EXPECT_FALSE(pResMaterial->IsVolatile(3));
    pResMaterial->SetVolatile(3);
    EXPECT_TRUE(pResMaterial->HasVolatile());
    pResMaterial->ResetVolatile(3);
    EXPECT_FALSE(pResMaterial->HasVolatile());

    EXPECT_EQ(1, pResMaterial->GetSamplerCount());
    EXPECT_EQ(1, pResMaterial->GetTextureCount());
    pResMaterial->SetTextureCount(3);
    EXPECT_EQ(3, pResMaterial->GetTextureCount());
    EXPECT_EQ(1, pResMaterial->GetRenderInfoCount());

    EXPECT_EQ(168, pResMaterial->GetSrcParamSize());
    EXPECT_EQ(0, pResMaterial->GetRawParamSize());

    EXPECT_TRUE(pResMaterial->GetShaderParam(5) != NULL);

    // テクスチャ
    const nn::gfx::TextureView* pTextureView = pResMaterial->GetTexture( 0 );
    EXPECT_TRUE(pTextureView == NULL);
    EXPECT_FALSE(pResMaterial->IsTextureBound(0));
    nn::g3d::BindResult result = pResMaterial->BindTexture(TextureBindCallback, NULL);
    EXPECT_TRUE(result.IsComplete());
    EXPECT_TRUE(result.IsBound());
    EXPECT_FALSE(result.IsNotBound());
    nn::g3d::TextureRef textureRef;
    nn::gfx::DescriptorSlot descriptorSlot;
    memset(&descriptorSlot, 0xa5, sizeof(descriptorSlot));
    textureRef.SetTextureView(reinterpret_cast<nn::gfx::TextureView*>(ptr));
    textureRef.SetDescriptorSlot(descriptorSlot);
    EXPECT_TRUE(pResMaterial->ForceBindTexture(textureRef, "human_A"));
    EXPECT_TRUE(pResMaterial->IsTextureBound( 0 ));
    pResMaterial->ReleaseTexture();
    EXPECT_FALSE(pResMaterial->IsTextureBound( 0 ));
    pResMaterial->SetTexture(0, reinterpret_cast<nn::gfx::TextureView*>(ptr));
    EXPECT_TRUE(pResMaterial->GetTexture(0) == reinterpret_cast<nn::gfx::TextureView*>(ptr));
    pResMaterial->SetTextureDescriptorSlot(0, descriptorSlot);
    nn::gfx::DescriptorSlot getTextureDescriptorSlot = pResMaterial->GetTextureDescriptorSlot(0);
    EXPECT_TRUE(memcmp(&getTextureDescriptorSlot, &descriptorSlot, sizeof(descriptorSlot)) == 0);
    pResMaterial->ReleaseTexture();

    // サンプラー
    const char* samplerName = pResMaterial->GetSamplerName(0);
    EXPECT_EQ(0, pResMaterial->FindSamplerIndex(samplerName));
    nn::gfx::DescriptorSlot getSamplerDescriptorSlot = pResMaterial->GetSamplerDescriptorSlot(0);
    EXPECT_FALSE(getSamplerDescriptorSlot.IsValid());
    pResMaterial->SetSamplerDescriptorSlot(0, descriptorSlot);
    getSamplerDescriptorSlot = pResMaterial->GetSamplerDescriptorSlot(0);
    EXPECT_TRUE(getSamplerDescriptorSlot.IsValid());
    EXPECT_TRUE(memcmp(&getSamplerDescriptorSlot, &descriptorSlot, sizeof(descriptorSlot)) == 0);
    nn::gfx::DescriptorSlot findSamplerDescriptorSlot = pResMaterial->FindSamplerDescriptorSlot(samplerName);
    EXPECT_TRUE(memcmp(&getSamplerDescriptorSlot, &findSamplerDescriptorSlot, sizeof(findSamplerDescriptorSlot)) == 0);
    pResMaterial->ClearSamplerDescriptorSlot();
    getSamplerDescriptorSlot = pResMaterial->GetSamplerDescriptorSlot(0);
    EXPECT_FALSE(getSamplerDescriptorSlot.IsValid());
}

TEST_F(G3dTest, ResRenderInfo)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResMaterial* pResMaterial = pResModel->FindMaterial("human_all");
    nn::g3d::ResRenderInfo* pResRenderInfo = pResMaterial->FindRenderInfo("priority");
    EXPECT_TRUE(pResRenderInfo != NULL);
    EXPECT_TRUE(pResRenderInfo == pResMaterial->GetRenderInfo(0));
    EXPECT_STREQ("priority", pResMaterial->GetRenderInfoName(pResMaterial->FindRenderInfoIndex("priority")));

    EXPECT_STREQ("priority", pResRenderInfo->GetName());
    EXPECT_EQ(nn::g3d::ResRenderInfo::Type_Int, pResRenderInfo->GetType());
    EXPECT_EQ(1, pResRenderInfo->GetArrayLength());
    const int* intArray = pResRenderInfo->GetInt();
    EXPECT_EQ(0, intArray[0]);
    EXPECT_EQ(pResRenderInfo->GetInt(0), intArray[0]);
}

TEST_F(G3dTest, Sampler)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResMaterial* pResMaterial = pResModel->FindMaterial("human_all");
    nn::gfx::Sampler* pSampler = pResMaterial->FindSampler("_a0");
    EXPECT_TRUE(pSampler != NULL);
    EXPECT_TRUE(pSampler == pResMaterial->GetSampler(0));
    EXPECT_STREQ("_a0", pResMaterial->GetSamplerName(pResMaterial->FindSamplerIndex("_a0")));
    EXPECT_EQ(1/*初期化済みステート*/, pSampler->ToData()->state);
}

TEST_F(G3dTest, SamplerInfo)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResMaterial* pResMaterial = pResModel->FindMaterial("human_all");
    nn::gfx::SamplerInfo* pSamplerInfo = pResMaterial->GetSamplerInfo(pResMaterial->FindSamplerIndex("_a0"));

    EXPECT_TRUE(pSamplerInfo != NULL);
    EXPECT_TRUE(pSamplerInfo == pResMaterial->GetSamplerInfo(0));
    EXPECT_EQ(nn::gfx::TextureAddressMode_Repeat, pSamplerInfo->GetAddressU());
    EXPECT_EQ(nn::gfx::TextureAddressMode_Repeat, pSamplerInfo->GetAddressV());
    EXPECT_EQ(nn::gfx::TextureAddressMode_Repeat, pSamplerInfo->GetAddressW());
    EXPECT_EQ(nn::gfx::ComparisonFunction_Never, pSamplerInfo->GetComparisonFunction());
    EXPECT_EQ(nn::gfx::TextureBorderColorType_White, pSamplerInfo->GetBorderColorType());
    EXPECT_EQ(1, pSamplerInfo->GetMaxAnisotropy());
    EXPECT_EQ(40, pSamplerInfo->GetFilterMode()); // (linear, linear, none) = 32 + 8 + 0
    EXPECT_EQ(0, pSamplerInfo->GetMinLod());
    EXPECT_EQ(13, pSamplerInfo->GetMaxLod());
    EXPECT_EQ(0, pSamplerInfo->GetLodBias());
}

TEST_F(G3dTest, ResShaderParam )
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResMaterial* pResMaterial = pResModel->FindMaterial("human_all");
    nn::g3d::ResShaderParam* pResShaderParam = pResMaterial->FindShaderParam("rim_int");
    EXPECT_TRUE(pResShaderParam != NULL);
    EXPECT_TRUE(pResShaderParam == pResMaterial->GetShaderParam(3));
    EXPECT_STREQ("rim_int", pResMaterial->GetShaderParamName(pResMaterial->FindShaderParamIndex("rim_int")));

    EXPECT_STREQ("rim_int", pResShaderParam->GetId());
    EXPECT_EQ(3, pResShaderParam->GetDependedIndex());
    EXPECT_EQ(3, pResShaderParam->GetDependIndex());
    EXPECT_EQ(nn::g3d::ResShaderParam::Type_Float, pResShaderParam->GetType());
    EXPECT_EQ(-1, pResShaderParam->GetOffset());
    EXPECT_EQ(20, pResShaderParam->GetSrcOffset());
    EXPECT_EQ(4, pResShaderParam->GetSize());
    EXPECT_EQ(4, pResShaderParam->GetSrcSize());

    EXPECT_EQ(12, nn::g3d::ResShaderParam::GetSize(nn::g3d::ResShaderParam::Type_Float3));
    EXPECT_EQ(64, nn::g3d::ResShaderParam::GetSize(nn::g3d::ResShaderParam::Type_Float4x3));
    EXPECT_EQ(sizeof(nn::util::FloatColumnMajor4x3), nn::g3d::ResShaderParam::GetSize(nn::g3d::ResShaderParam::Type_Srt3d));

    EXPECT_EQ(12, nn::g3d::ResShaderParam::GetSrcSize(nn::g3d::ResShaderParam::Type_Float3));
    EXPECT_EQ(48, nn::g3d::ResShaderParam::GetSrcSize(nn::g3d::ResShaderParam::Type_Float4x3));
    EXPECT_EQ(sizeof(nn::g3d::Srt3d), nn::g3d::ResShaderParam::GetSrcSize(nn::g3d::ResShaderParam::Type_Srt3d));
}

TEST_F(G3dTest, ResShaderAssign)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResMaterial* pResMaterial = pResModel->FindMaterial("human_all");
    nn::g3d::ResShaderAssign* pResShaderAssign = pResMaterial->GetShaderAssign();
    EXPECT_TRUE(pResShaderAssign != NULL);

    EXPECT_STREQ("demo", pResShaderAssign->GetShaderArchiveName());
    EXPECT_STREQ("basic", pResShaderAssign->GetShadingModelName());

    EXPECT_EQ(8, pResShaderAssign->GetAttrAssignCount());
    EXPECT_STREQ("_i0", pResShaderAssign->GetAttrAssign(2));
    EXPECT_STREQ("_i0", pResShaderAssign->FindAttrAssign("_i0"));
    EXPECT_EQ(2, pResShaderAssign->FindAttrAssignIndex("_i0"));
    EXPECT_STREQ("_i0", pResShaderAssign->GetAttrAssignId(2));

    EXPECT_EQ(6, pResShaderAssign->GetShaderOptionCount());
    EXPECT_STREQ("0", pResShaderAssign->GetShaderOption(2));
    EXPECT_STREQ("0", pResShaderAssign->FindShaderOption("multi_tex"));
    EXPECT_EQ(2, pResShaderAssign->FindShaderOptionIndex("multi_tex"));
    EXPECT_STREQ("multi_tex", pResShaderAssign->GetShaderOptionId(2));

    EXPECT_EQ(1, pResShaderAssign->GetSamplerAssignCount());
    EXPECT_STREQ("_a0", pResShaderAssign->GetSamplerAssign(0));
    EXPECT_STREQ("_a0", pResShaderAssign->FindSamplerAssign("_a0"));
    EXPECT_EQ(0, pResShaderAssign->FindSamplerAssignIndex("_a0"));
    EXPECT_STREQ("_a0", pResShaderAssign->GetSamplerAssignId(0));
}

TEST_F(G3dTest, ResSkeleton)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResSkeleton* pResSkeleton = pResModel->GetSkeleton();

    void* ptr = reinterpret_cast<void*>(0x12345678);
    pResSkeleton->SetUserPtr(ptr);
    EXPECT_EQ(ptr, pResSkeleton->GetUserPtr());

    EXPECT_EQ(nn::g3d::ResSkeleton::ScaleMode_Maya, pResSkeleton->GetScaleMode());
    EXPECT_EQ(nn::g3d::ResSkeleton::RotateMode_Quat, pResSkeleton->GetRotateMode());
    EXPECT_EQ(53, pResSkeleton->GetSmoothMtxCount());
    EXPECT_EQ(0, pResSkeleton->GetRigidMtxCount());
    EXPECT_EQ(53, pResSkeleton->GetMtxCount());
    EXPECT_EQ(53, pResSkeleton->GetBoneCount());
    EXPECT_EQ(10, pResSkeleton->GetBranchEndIndex(1));
}

// Todo : ベクトル取得関連のテストない
TEST_F(G3dTest, ResBone)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    nn::g3d::ResSkeleton* pResSkeleton = pResModel->GetSkeleton();
    nn::g3d::ResBone* pResBone = pResSkeleton->FindBone("leg_l2");
    EXPECT_TRUE(pResBone != NULL);
    EXPECT_TRUE(pResBone == pResSkeleton->GetBone(3));
    EXPECT_STREQ("leg_l2", pResSkeleton->GetBoneName(pResSkeleton->FindBoneIndex("leg_l2")));

    EXPECT_STREQ("leg_l2", pResBone->GetName());
    EXPECT_EQ(3, pResBone->GetIndex());
    EXPECT_EQ(3, pResBone->GetSmoothMtxIndex());
    EXPECT_EQ(-1, pResBone->GetRigidMtxIndex());
    EXPECT_EQ(2, pResBone->GetParentIndex());
    EXPECT_EQ(nn::g3d::ResBone::RotateMode_Quat, pResBone->GetRotateMode());
    EXPECT_EQ(nn::g3d::ResBone::Flag_BillboardChild, pResBone->GetBillboardMode());
}

TEST_F(G3dTest, ResUserData)
{
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");

    // user_int
    auto pResUserData = pResModel->GetUserData( 0 );
    EXPECT_STREQ( "test_int0", pResUserData->GetName() );
    EXPECT_EQ( 3, pResUserData->GetCount() );
    const int* pIntVal = pResUserData->GetInt();
    EXPECT_EQ( 0, pIntVal[0] );
    EXPECT_EQ( 1, pIntVal[1] );
    EXPECT_EQ( 2, pIntVal[2] );
    EXPECT_EQ( pResUserData->GetInt(0), pIntVal[0] );
    EXPECT_EQ( pResUserData->GetInt(1), pIntVal[1] );
    EXPECT_EQ( pResUserData->GetInt(2), pIntVal[2] );

    // user_float
    pResUserData = pResModel->GetUserData( 1 );
    EXPECT_STREQ( "test_float0", pResUserData->GetName() );
    EXPECT_EQ( 3, pResUserData->GetCount() );
    const float* pFloatVal = pResUserData->GetFloat();
    EXPECT_FLOAT_EQ( 0.f, pFloatVal[0] );
    EXPECT_FLOAT_EQ( 1.f, pFloatVal[1] );
    EXPECT_FLOAT_EQ( 2.f, pFloatVal[2] );
    EXPECT_EQ( pResUserData->GetFloat(0), pFloatVal[0] );
    EXPECT_EQ( pResUserData->GetFloat(1), pFloatVal[1] );
    EXPECT_EQ( pResUserData->GetFloat(2), pFloatVal[2] );

    // user_string
    pResUserData = pResModel->GetUserData( 2 );
    EXPECT_STREQ( "test_string0", pResUserData->GetName() );
    EXPECT_EQ( 2, pResUserData->GetCount() );
    EXPECT_STREQ( "abc ABC", pResUserData->GetString( 0 ) );
    EXPECT_STREQ( "xyz XYZ", pResUserData->GetString( 1 ) );

    // user_wstring
    pResUserData = pResModel->GetUserData( 3 );
    EXPECT_STREQ( "test_wstring0", pResUserData->GetName() );
    EXPECT_EQ( 3, pResUserData->GetCount() );
    EXPECT_STREQ( "abc ABC", pResUserData->GetString( 0 ) );
    EXPECT_STREQ( "xyz XYZ", pResUserData->GetString( 1 ) );
    unsigned char Utf8StrGoldenVal[] =    /* ロケール違いによる warning 回避のためバイト列で正解値を用意する */
    {
        0xe3, 0x81, 0x82,    // "あ"
        0xe3, 0x80, 0x80,    // "　" 全角スペース
        0xe3, 0x81, 0x84,    // "い"
        0xe3, 0x80, 0x80,    // "　" 全角スペース
        0x0a,                // LF
        0xe3, 0x81, 0x86,    // "う"
    };
    size_t strSize = sizeof( Utf8StrGoldenVal ) / sizeof( char* );
    EXPECT_TRUE( memcmp( Utf8StrGoldenVal, pResUserData->GetString( 2 ), strSize ) == 0 );

    // user_stream
    pResUserData = pResModel->GetUserData( 4 );
    EXPECT_STREQ( "test_stream0", pResUserData->GetName() );
    EXPECT_EQ( 16, pResUserData->GetStreamSize() );
    uint32_t* pData = reinterpret_cast<uint32_t*>( pResUserData->GetStream() );
    const uint32_t StreamElemGoldenVal = 0xAFBEADDE;    // 正解値は 0xAFBEADDE が 4 つ並んだ straem です。
    EXPECT_EQ( StreamElemGoldenVal, pData[0] );
    EXPECT_EQ( StreamElemGoldenVal, pData[1] );
    EXPECT_EQ( StreamElemGoldenVal, pData[2] );
    EXPECT_EQ( StreamElemGoldenVal, pData[3] );
}

namespace
{

size_t CalculateBuildMaterialObjBufferSize(int bufferCount, nn::g3d::ResMaterial* pResMaterial)
{
    nn::util::MemorySplitter::MemoryBlock memoryBlock;
    size_t defaultAlignment = memoryBlock.GetAlignment();

    size_t size = nn::util::align_up(pResMaterial->GetShaderParamCount(), 32) >> 3;
    size = nn::util::align_up(size, 4);
    size += (nn::util::align_up(pResMaterial->GetShaderParamCount(), 32) >> 3) * bufferCount;
    size = nn::util::align_up(size, defaultAlignment);
    size += nn::util::align_up(pResMaterial->GetSrcParamSize(), 4);
    size = nn::util::align_up(size, defaultAlignment);
    size += sizeof(nn::gfx::TextureView*) * pResMaterial->GetTextureCount();
    size = nn::util::align_up(size, defaultAlignment);
    size += sizeof(nn::gfx::DescriptorSlot) * pResMaterial->GetTextureCount();
    size = nn::util::align_up(size, defaultAlignment);
    size += sizeof(nn::gfx::Buffer) * bufferCount;

    return size;
}

}

TEST_F(G3dTest, MaterialObj)
{
    nn::gfx::Device* pDevice = GetDevice();
    nn::g3d::MaterialObj  materialObj;
    nn::gfx::TextureView resTextureView;
    nn::gfx::DescriptorSlot resDescriptorSlot;
    nn::g3d::ResMaterial* pResMaterial = GetResFile()->FindModel("human")->FindMaterial("human_all");

    // 適当なテクスチャ設定をする
    {
        pResMaterial->SetTexture(0, &resTextureView);
        memset(&resDescriptorSlot, 0x01, sizeof(nn::gfx::DescriptorSlot));
        pResMaterial->SetTextureDescriptorSlot(0, resDescriptorSlot);
    }

    // ユニフォームブロックサイズが0のケース
    {
        nn::g3d::MaterialObj::Builder builder(pResMaterial);
        builder.CalculateMemorySize();
        EXPECT_TRUE(builder.IsMemoryCalculated());
        size_t size = builder.GetWorkMemorySize();
        size_t expectedSize = CalculateBuildMaterialObjBufferSize(builder.GetBufferingCount(), pResMaterial);
        EXPECT_EQ(expectedSize, size);
        SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::MaterialObj::Alignment_Buffer));
        builder.Build(&materialObj, buffer.Get(), size);

        size = materialObj.CalculateBlockBufferSize(pDevice);
        EXPECT_EQ(0, size);

        EXPECT_TRUE(materialObj.SetupBlockBuffer(pDevice, NULL, 0, 0));
        EXPECT_FALSE(materialObj.IsBlockBufferValid());
    }
    // ユニフォームブロックサイズが0でないケース
    {
        pResMaterial->SetRawParamSize(16);
        nn::g3d::MaterialObj::Builder builder(pResMaterial);
        builder.BufferingCount(2);
        builder.CalculateMemorySize();
        EXPECT_TRUE(builder.IsMemoryCalculated());
        size_t size = builder.GetWorkMemorySize();
        size_t expectedSize = CalculateBuildMaterialObjBufferSize(builder.GetBufferingCount(), pResMaterial);
        EXPECT_EQ(expectedSize, size);
        SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::MaterialObj::Alignment_Buffer));
        builder.Build(&materialObj, buffer.Get(), size);

        size = materialObj.CalculateBlockBufferSize(pDevice);
        expectedSize = nn::util::align_up(pResMaterial->GetRawParamSize(), materialObj.GetBlockBufferAlignment(pDevice)) * builder.GetBufferingCount();
        EXPECT_EQ(expectedSize, size);
        ptrdiff_t offset = AllocateMemoryPool(size, materialObj.GetBlockBufferAlignment(pDevice));
        EXPECT_TRUE(materialObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, size));
        EXPECT_TRUE(materialObj.IsBlockBufferValid());

        EXPECT_TRUE(materialObj.GetResource() == pResMaterial);
        EXPECT_TRUE(materialObj.GetBufferPtr() == buffer.Get());
        EXPECT_TRUE(materialObj.GetMemoryPoolPtr() == GetWriteCombineMemoryPool());
        EXPECT_TRUE(materialObj.GetMemoryPoolOffset() == offset);
        EXPECT_TRUE(builder.GetBufferingCount() == materialObj.GetBufferingCount());

        // シェーダーパラメータが取得できるかチェック
        nn::g3d::ResShaderParam* pResShaderParam = const_cast<nn::g3d::ResShaderParam*>(materialObj.FindResShaderParam("rim_int"));
        {
            EXPECT_EQ(materialObj.GetShaderParamCount(), pResMaterial->GetShaderParamCount());
            EXPECT_TRUE(pResShaderParam != NULL);
            EXPECT_TRUE(pResShaderParam == materialObj.GetResShaderParam(3));
            EXPECT_STREQ("rim_int", materialObj.GetShaderParamName(materialObj.FindShaderParamIndex("rim_int")));
        }

        // ダーティフラグの正当性チェック
        {
            EXPECT_FALSE(materialObj.IsMaterialBlockDirty());
        }

        // データ更新の正当性チェック
        {
            // バッファ0をデフォルト値で変更
            pResShaderParam->SetOffset(8);
            int* pParam = materialObj.EditShaderParam<int>(3);
            EXPECT_TRUE(reinterpret_cast<const void*>(pParam) == materialObj.GetShaderParam(3));
            // ダーティフラグの正当性をチェック
            EXPECT_TRUE(materialObj.IsMaterialBlockDirty());
            pResShaderParam->SetOffset(-1);
            materialObj.ResetDirtyFlags();
            EXPECT_FALSE(materialObj.IsMaterialBlockDirty());
            // 再度変更
            pResShaderParam->SetOffset(8);
            pParam = materialObj.EditShaderParam<int>(3);
            materialObj.CalculateMaterial(0);

            int originalValue = *pParam;
            materialObj.CalculateMaterial(0);
            nn::gfx::Buffer* pBuffer = materialObj.GetMaterialBlock(0);
            EXPECT_TRUE(pBuffer != NULL);
            const int* pBlock = pBuffer->Map<int>();
            pBuffer->InvalidateMappedRange(0, materialObj.GetMaterialBlockSize());
            EXPECT_EQ(originalValue, *(pBlock + 2));
            pBuffer->Unmap();

            // バッファ１を変更
            pParam = materialObj.EditShaderParam<int>(3);
            *pParam = 0x1234;
            materialObj.CalculateMaterial(1);
            pBuffer = materialObj.GetMaterialBlock(1);
            EXPECT_TRUE(pBuffer != NULL);
            pBlock = pBuffer->Map<int>();
            pBuffer->InvalidateMappedRange(0, materialObj.GetMaterialBlockSize());
            EXPECT_EQ(0x1234, *(pBlock + 2));
            pBuffer->Unmap();

            // バッファ０は変更されていないことを確認
            pBuffer = materialObj.GetMaterialBlock(0);
            pBlock = pBuffer->Map<int>();
            pBuffer->InvalidateMappedRange(0, materialObj.GetMaterialBlockSize());
            EXPECT_EQ(originalValue, *(pBlock + 2));
            pBuffer->Unmap();

            // データを元に戻す
            materialObj.ClearShaderParam ();
            materialObj.CalculateMaterial(1);
            pBuffer = materialObj.GetMaterialBlock(1);
            pBlock = pBuffer->Map<int>();
            pBuffer->InvalidateMappedRange(0, materialObj.GetMaterialBlockSize());
            EXPECT_EQ(originalValue, *(pBlock + 2));
            pBuffer->Unmap();
        }

        // テクスチャ
        {
            EXPECT_EQ(materialObj.GetTextureCount(), pResMaterial->GetTextureCount());
            EXPECT_STREQ("human_A", materialObj.GetTextureName(0));
            nn::g3d::TextureRef getTextureRef = materialObj.GetTexture(0);
            EXPECT_TRUE(getTextureRef.IsValid());
            nn::g3d::TextureRef setTextureRef;
            nn::gfx::DescriptorSlot descriptorSlot;
            memset(&descriptorSlot, 0xa5, sizeof(descriptorSlot));
            setTextureRef.SetTextureView(reinterpret_cast<nn::gfx::TextureView*>(0x12345678));
            setTextureRef.SetDescriptorSlot(descriptorSlot);
            materialObj.SetTexture(0, setTextureRef);
            getTextureRef = materialObj.GetTexture(0);
            EXPECT_TRUE(getTextureRef.GetTextureView() == reinterpret_cast<nn::gfx::TextureView*>(0x12345678));
            nn::gfx::DescriptorSlot getDescriptorSlot = getTextureRef.GetDescriptorSlot();
            EXPECT_TRUE(memcmp(&getDescriptorSlot, &descriptorSlot, sizeof(descriptorSlot)) == 0);
            materialObj.ClearTexture();
            getTextureRef = materialObj.GetTexture(0);
            EXPECT_TRUE(getTextureRef.GetTextureView() == &resTextureView);
            getDescriptorSlot = getTextureRef.GetDescriptorSlot();
            EXPECT_TRUE(memcmp(&getDescriptorSlot, &resDescriptorSlot, sizeof(nn::gfx::DescriptorSlot)) == 0);
        }

        // サンプラー
        {
            EXPECT_EQ(materialObj.GetSamplerCount(), pResMaterial->GetSamplerCount());
            nn::g3d::SamplerRef samplerRef = materialObj.FindSampler("_a0");
            EXPECT_TRUE(samplerRef.GetSampler() != NULL);
            EXPECT_TRUE(samplerRef.GetSampler() == materialObj.GetSampler(0).GetSampler());
            EXPECT_FALSE(samplerRef.GetDescriptorSlot().IsValid());
            EXPECT_STREQ("_a0", materialObj.GetSamplerName(materialObj.FindSamplerIndex("_a0")));
        }

        // ユーザーポインタ
        {
            void* ptr = reinterpret_cast<void*>(0x12345678);
            materialObj.SetUserPtr(ptr);
            EXPECT_EQ(ptr, materialObj.GetUserPtr());
        }

        materialObj.CleanupBlockBuffer(GetDevice());
        FreeMemoryPool();
    }
} //NOLINT

namespace
{

size_t CalculateBuildShapeObjBufferSize(int viewCount, int bufferCount, size_t userAreaSize, nn::g3d::ResShape* pResShape)
{
    nn::util::MemorySplitter::MemoryBlock memoryBlock;
    size_t defaultAlignment = memoryBlock.GetAlignment();

    size_t size = sizeof(nn::g3d::Aabb) * pResShape->GetSubMeshCount();
    size = nn::util::align_up(size, defaultAlignment);
    size += sizeof(nn::gfx::Buffer) * viewCount * bufferCount;
    size = nn::util::align_up(size, defaultAlignment);
    size += sizeof(float) * pResShape->GetKeyShapeCount();
    size = nn::util::align_up(size, 4);
    size += nn::util::align_up(pResShape->GetKeyShapeCount(), 32) >> 3;
    size = nn::util::align_up(size, defaultAlignment);
    size += sizeof(nn::g3d::Sphere) * 2;
    size = nn::util::align_up(size, defaultAlignment);
    size += nn::util::align_up(userAreaSize, 4);

    return size;
}

}
// Todo : テストにないもの
// CalculateSubMeshBounding humanが剛体ではないから
// MakeSubMeshRange
// GetSubMeshBoundingArray humanが剛体ではないから
TEST_F(G3dTest, ShapeObj)
{
    nn::gfx::Device* pDevice = GetDevice();
    nn::g3d::ShapeObj shapeObj;
    nn::g3d::ResShape* pResShape = GetResFile()->FindModel("human")->FindShape("polygon3");
    nn::g3d::ShapeObj::Builder builder(pResShape);
    builder.ViewCount(2);
    builder.UserAreaSize(nn::g3d::ShapeBlock::Size_UserArea);
    builder.BufferingCount(2);
    builder.ViewDependent();
    builder.SetBoundingEnabled();
    builder.CalculateMemorySize();
    EXPECT_TRUE(builder.IsMemoryCalculated());
    size_t size = builder.GetWorkMemorySize();
    size_t expectedSize = CalculateBuildShapeObjBufferSize(builder.GetViewCount(),
                                                      builder.GetBufferingCount(),
                                                      builder.GetUserAreaSize(),
                                                      pResShape);
    EXPECT_EQ(expectedSize, size);
    SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ShapeObj::Alignment_Buffer));
    builder.Build(&shapeObj, buffer.Get(), size);

    size = shapeObj.CalculateBlockBufferSize(pDevice);
    expectedSize = nn::util::align_up(sizeof(nn::g3d::ShapeBlock), shapeObj.GetBlockBufferAlignment(pDevice)) * shapeObj.GetShapeBlockCount() * shapeObj.GetBufferingCount();
    EXPECT_EQ(expectedSize, size);
    ptrdiff_t offset = AllocateMemoryPool(size, shapeObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(shapeObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, size));
    EXPECT_TRUE(shapeObj.IsBlockBufferValid());

    EXPECT_TRUE(shapeObj.GetResource() == pResShape);
    EXPECT_TRUE(shapeObj.GetBufferPtr() == buffer.Get());
    EXPECT_TRUE(shapeObj.GetMemoryPoolPtr() == GetWriteCombineMemoryPool());
    EXPECT_TRUE(shapeObj.GetMemoryPoolOffset() == offset);
    EXPECT_TRUE(builder.GetBufferingCount() == shapeObj.GetBufferingCount());
    EXPECT_EQ(0, shapeObj.GetMaterialIndex());
    EXPECT_EQ(0, shapeObj.GetBoneIndex());
    EXPECT_EQ(0, shapeObj.GetVertexIndex());
    EXPECT_EQ(3, shapeObj.GetVertexSkinCount());

    EXPECT_FALSE(shapeObj.IsRigidBody());
    EXPECT_FALSE(shapeObj.IsRigidSkinning());
    EXPECT_TRUE(shapeObj.IsSmoothSkinning());

    EXPECT_EQ(builder.GetViewCount(), shapeObj.GetViewCount());
    EXPECT_TRUE(shapeObj.IsViewDependent());

    // メッシュ
    {
        EXPECT_EQ(1, shapeObj.GetMeshCount());
        EXPECT_EQ(1, shapeObj.GetSubMeshCount());
        const nn::g3d::ResMesh* pMesh = shapeObj.GetResMesh();
        EXPECT_TRUE(pMesh != NULL);
        EXPECT_TRUE(pMesh == shapeObj.GetResMesh(0));
    }

    // ユーザーポインタ
    {
        void* ptr = reinterpret_cast<void*>(0x12345678);
        shapeObj.SetUserPtr(ptr);
        EXPECT_EQ(ptr, shapeObj.GetUserPtr());
    }

    // 頂点情報
    {
        const nn::g3d::ResVertex* pVertex = shapeObj.GetResVertex();
        EXPECT_TRUE(pVertex != NULL);
        const nn::gfx::Buffer* pBuffer = shapeObj.GetVertexBuffer(0);
        EXPECT_TRUE(pBuffer != NULL);
        EXPECT_EQ(1, shapeObj.GetVertexBufferCount());
    }

    // 頂点属性
    {
        EXPECT_EQ(5, shapeObj.GetVertexAttrCount());
        const nn::g3d::ResVertexAttr* pResVertexAttr = shapeObj.FindResVertexAttr("_n0");
        EXPECT_TRUE(pResVertexAttr != NULL);
        EXPECT_TRUE(pResVertexAttr == shapeObj.GetResVertexAttr(1));
        EXPECT_STREQ("_n0", shapeObj.GetVertexAttrName(shapeObj.FindVertexAttrIndex("_n0")));
    }

    // ユニフォームブロック & ユーザーエリア & バウンディング
    {
        nn::g3d::SkeletonObj skeletonObj;
        nn::g3d::ResSkeleton* pResSkeleton = GetResFile()->FindModel("human")->GetSkeleton();
        nn::g3d::SkeletonObj::Builder skeletonBuilder(pResSkeleton);
        skeletonBuilder.CalculateMemorySize();
        size_t bufferSize = skeletonBuilder.GetWorkMemorySize();
        SimplePtr skeletonBuffer(nnt::g3d::AlignedAllocate<void>(bufferSize, nn::g3d::SkeletonObj::Alignment_Buffer));
        skeletonBuilder.Build(&skeletonObj, skeletonBuffer.Get(), bufferSize);

        nn::util::Matrix4x3fType baseMtx;
        nn::util::Vector3fType rotateVector = {0.0f, 2 * 3.14f * 2 / 256, 0.0f};
        MatrixSetRotateXyz(&baseMtx, rotateVector);
        nn::util::Vector3fType tranlateVector = {0.0f, 0.0f, 0.0f};
        MatrixSetAxisW(&baseMtx, tranlateVector);
        skeletonObj.CalculateWorldMtx(baseMtx);

        // バウンディング
        EXPECT_TRUE(shapeObj.GetBounding() != NULL);
        EXPECT_TRUE(shapeObj.GetSubMeshBoundingArray() != NULL);
        shapeObj.CalculateBounding(&skeletonObj);
        nn::g3d::Sphere* pSphere = shapeObj.GetBounding();
        EXPECT_TRUE(IsVector3fEqual(g_AnswerBoundingSphereCenter, pSphere->center));
        EXPECT_TRUE(IsFloatEqual(g_AnswerBoundingSphereRadius, pSphere->radius));

        // ユニフォームブロック & ユーザーエリア
        nn::g3d::ShapeBlock shapeBlock;
        VectorSet(&rotateVector, 1.0f, 2.0f, 3.0f);
        nn::util::Matrix4x3fType worldMtx;
        MatrixSetRotateXyz(&worldMtx, rotateVector);
        VectorSet(&tranlateVector, 1.0f, 2.0f, 3.0f);
        MatrixSetAxisW(&worldMtx, tranlateVector);
        MatrixStore(&shapeBlock.worldMtx, worldMtx);

        shapeBlock.vtxSkinCount = shapeObj.GetVertexSkinCount();
        int userIntCount = static_cast<int>(shapeObj.GetUserAreaSize()) >> 2;
        for (int i = 0; i < userIntCount; ++i)
        {
            shapeBlock.userInt[i] = i;
        }

        // ビュー0 バッファ0に書き込み
        void* pUserArea = shapeObj.GetUserArea();
        EXPECT_TRUE(pUserArea != NULL);
        EXPECT_EQ(builder.GetUserAreaSize(), shapeObj.GetUserAreaSize());
        memcpy(pUserArea, shapeBlock.userInt, shapeObj.GetUserAreaSize());

        shapeObj.CalculateShape(0, worldMtx, 0);
        nn::gfx::Buffer* pBuffer = shapeObj.GetShapeBlock(0, 0);
        EXPECT_TRUE(pBuffer != NULL);
        EXPECT_TRUE(pBuffer != shapeObj.GetShapeBlock(0, 1));
        EXPECT_TRUE(pBuffer != shapeObj.GetShapeBlock(1, 0));

        nn::g3d::ShapeBlock* pDstShapeBlock = pBuffer->Map<nn::g3d::ShapeBlock>();
        pBuffer->InvalidateMappedRange(0, sizeof(nn::g3d::ShapeBlock));
        EXPECT_TRUE(std::memcmp(&pDstShapeBlock->worldMtx, &shapeBlock.worldMtx, sizeof(nn::util::FloatColumnMajor4x3)) == 0);
        EXPECT_EQ(pDstShapeBlock->vtxSkinCount, shapeBlock.vtxSkinCount);
        EXPECT_TRUE(std::memcmp(&pDstShapeBlock->userInt, &shapeBlock.userInt, shapeObj.GetUserAreaSize()) == 0);
        pBuffer->Unmap();

        // ビュー0 バッファ1に書き込み
        VectorSet(&rotateVector, 4.0f, 5.0f, 6.0f);
        VectorSet(&tranlateVector, 7.0f, 8.0f, 9.0f);
        MatrixSetRotateXyz(&worldMtx, rotateVector);
        MatrixSetAxisW(&worldMtx, tranlateVector);
        MatrixStore(&shapeBlock.worldMtx, worldMtx);
        for (int i = 0; i < userIntCount; ++i)
        {
            shapeBlock.userInt[i] = i + 1;
        }
        memcpy(pUserArea, shapeBlock.userInt, shapeObj.GetUserAreaSize());

        shapeObj.CalculateShape(0, worldMtx, 1);
        pBuffer = shapeObj.GetShapeBlock(0, 1);
        pDstShapeBlock = pBuffer->Map<nn::g3d::ShapeBlock>();
        pBuffer->InvalidateMappedRange(0, sizeof(nn::g3d::ShapeBlock));
        EXPECT_TRUE(std::memcmp(&pDstShapeBlock->worldMtx, &shapeBlock.worldMtx, sizeof(nn::util::FloatColumnMajor4x3)) == 0);
        EXPECT_EQ(pDstShapeBlock->vtxSkinCount, shapeBlock.vtxSkinCount);
        EXPECT_TRUE(std::memcmp(&pDstShapeBlock->userInt, &shapeBlock.userInt, shapeObj.GetUserAreaSize()) == 0);
        pBuffer->Unmap();

        // ビュー1 バッファ1に書き込み
        VectorSet(&rotateVector, 7.0f, 8.0f, 9.0f);
        VectorSet(&tranlateVector, 10.0f, 11.0f, 12.0f);
        MatrixSetRotateXyz(&worldMtx, rotateVector);
        MatrixSetAxisW(&worldMtx, tranlateVector);
        MatrixStore(&shapeBlock.worldMtx, worldMtx);
        for (int i = 0; i < userIntCount; ++i)
        {
            shapeBlock.userInt[i] = i + 2;
        }
        memcpy(pUserArea, shapeBlock.userInt, shapeObj.GetUserAreaSize());

        shapeObj.CalculateShape(1, worldMtx, 1);
        pBuffer = shapeObj.GetShapeBlock(1, 1);
        pDstShapeBlock = pBuffer->Map<nn::g3d::ShapeBlock>();
        pBuffer->InvalidateMappedRange(0, sizeof(nn::g3d::ShapeBlock));
        EXPECT_TRUE(std::memcmp(&pDstShapeBlock->worldMtx, &shapeBlock.worldMtx, sizeof(nn::util::FloatColumnMajor4x3)) == 0);
        EXPECT_EQ(pDstShapeBlock->vtxSkinCount, shapeBlock.vtxSkinCount);
        EXPECT_TRUE(std::memcmp(&pDstShapeBlock->userInt, &shapeBlock.userInt, shapeObj.GetUserAreaSize()) == 0);
        pBuffer->Unmap();
    }

    shapeObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
} //NOLINT

namespace
{

size_t CalculateBuildSkeletonObjBufferSize(int bufferCount, nn::g3d::ResSkeleton* pResSkeleton)
{
    nn::util::MemorySplitter::MemoryBlock memoryBlock;
    size_t defaultAlignment = memoryBlock.GetAlignment();

    size_t size = sizeof(nn::util::Matrix4x3fType) * pResSkeleton->GetBoneCount();
    size = nn::util::align_up(size, defaultAlignment);
    size += sizeof(nn::g3d::LocalMtx) * pResSkeleton->GetBoneCount();
    size = nn::util::align_up(size, defaultAlignment);
    if (pResSkeleton->GetScaleMode() == nn::g3d::ResSkeleton::ScaleMode_Softimage)
    {
        size += sizeof(nn::util::Vector3fType) * pResSkeleton->GetBoneCount();;
        size = nn::util::align_up(size, defaultAlignment);
    }
    size += sizeof(nn::gfx::Buffer) * bufferCount;
    size = nn::util::align_up(size, defaultAlignment);

    return size;
}

}
// Todo : Billboardのテストない humanにはビルボード設定がないから
TEST_F(G3dTest, SkeletonObj)
{
    nn::gfx::Device* pDevice = GetDevice();
    nn::g3d::SkeletonObj skeletonObj;
    nn::g3d::ResSkeleton* pResSkeleton = GetResFile()->FindModel("human")->GetSkeleton();
    nn::g3d::SkeletonObj::Builder builder(pResSkeleton);
    builder.BufferingCount(2);
    builder.CalculateMemorySize();
    EXPECT_TRUE(builder.IsMemoryCalculated());
    size_t size = builder.GetWorkMemorySize();
    size_t expectedSize = CalculateBuildSkeletonObjBufferSize(builder.GetBufferingCount(), pResSkeleton);
    EXPECT_EQ(expectedSize, size);
    SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::SkeletonObj::Alignment_Buffer));
    builder.Build(&skeletonObj, buffer.Get(), size);

    size = skeletonObj.CalculateBlockBufferSize(pDevice);
    expectedSize = nn::util::align_up(sizeof(nn::util::FloatColumnMajor4x3) * pResSkeleton->GetMtxCount(), skeletonObj.GetBlockBufferAlignment(pDevice))
                 * skeletonObj.GetBufferingCount();
    EXPECT_EQ(expectedSize, size);
    ptrdiff_t offset = AllocateMemoryPool(size, skeletonObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(skeletonObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, size));
    EXPECT_TRUE(skeletonObj.IsBlockBufferValid());

    EXPECT_TRUE(skeletonObj.GetResource() == pResSkeleton);
    EXPECT_TRUE(skeletonObj.GetBufferPtr() == buffer.Get());
    EXPECT_TRUE(skeletonObj.GetMemoryPoolPtr() == GetWriteCombineMemoryPool());
    EXPECT_TRUE(skeletonObj.GetMemoryPoolOffset() == offset);
    EXPECT_TRUE(builder.GetBufferingCount() == skeletonObj.GetBufferingCount());

    EXPECT_EQ(nn::g3d::ResSkeleton::ScaleMode_Maya, skeletonObj.GetScaleMode());
    EXPECT_EQ(nn::g3d::ResSkeleton::RotateMode_Quat, skeletonObj.GetRotateMode());

    // マトリクス
    {
        nn::g3d::LocalMtx* pLocalMtx = skeletonObj.GetLocalMtxArray();
        EXPECT_TRUE(pLocalMtx != NULL);
        nn::util::Matrix4x3fType* pWorldMtx = skeletonObj.GetWorldMtxArray();
        EXPECT_TRUE(pWorldMtx != NULL);

        nn::g3d::LocalMtx originalMtx = *pLocalMtx;
        // マトリクスを適当に変更する
        MatrixIdentity(&pLocalMtx->mtxRT);
        EXPECT_FALSE(std::memcmp(&originalMtx, pLocalMtx, sizeof(nn::g3d::LocalMtx)) == 0);
        skeletonObj.ClearLocalMtx();
        EXPECT_TRUE(std::memcmp(&originalMtx, pLocalMtx, sizeof(nn::g3d::LocalMtx)) == 0);

        nn::util::Matrix4x3fType baseMtx;
        nn::util::Vector3fType rotateVector;
        VectorSet(&rotateVector, 0.0f, 2 * 3.14f * 2 / 256, 0.0f);
        MatrixSetRotateXyz(&baseMtx, rotateVector);
        nn::util::Vector3fType translateVector;
        VectorSet(&translateVector, 0.0f, 0.0f, 0.0f);
        MatrixSetTranslate(&baseMtx, translateVector);
        skeletonObj.CalculateWorldMtx(baseMtx);
        // 比較用期待値計算
        nn::util::Matrix4x3fType expectMtx0;
        nn::util::MatrixMultiply(&expectMtx0, pLocalMtx[0].mtxRT, baseMtx);
        EXPECT_TRUE(IsMtx43Equal(expectMtx0, pWorldMtx[0]));
        // human 以外を使う場合、変更が必要
        nn::util::Matrix4x3fType expectMtx1;
        nn::util::Matrix4x3fType& parentMtx = pWorldMtx[0];
        nn::util::MatrixMultiply(&expectMtx1, pLocalMtx[1].mtxRT, parentMtx);
        EXPECT_TRUE(IsMtx43Equal(expectMtx1, pWorldMtx[1]));
    }

    // ユニフォームブロック
    {
        nn::gfx::Buffer* pBuffer = skeletonObj.GetMtxBlock(0);
        EXPECT_TRUE(pBuffer != NULL);
        EXPECT_TRUE(pBuffer != skeletonObj.GetMtxBlock(1));

        // バッファ0に計算結果が反映されているかチェック
        skeletonObj.CalculateSkeleton(0);
        // 比較用期待値計算
        const int16_t* pMtxToBoneTable = pResSkeleton->ToData().pMtxToBoneTable.Get();
        const nn::util::FloatColumnMajor4x3* pInvModelMatrixArray = pResSkeleton->ToData().pInvModelMatrixArray.Get();
        int boneIndex = pMtxToBoneTable[0];
        nn::util::Matrix4x3fType invMtx;
        nn::util::MatrixLoad(&invMtx, pInvModelMatrixArray[0]);
        nn::util::Matrix4x3fType* pWorldMtx = skeletonObj.GetWorldMtxArray();
        nn::util::Matrix4x3fType expectMtx;
        nn::util::MatrixMultiply(&expectMtx, invMtx, pWorldMtx[boneIndex]);
        nn::util::FloatColumnMajor4x3 expectFloatMtx;
        nn::util::MatrixStore(&expectFloatMtx, expectMtx);

        uint8_t* ptr = pBuffer->Map<uint8_t>();
        pBuffer->InvalidateMappedRange(0, skeletonObj.GetMtxBlockSize());
        EXPECT_TRUE(std::memcmp(ptr, &expectFloatMtx, sizeof(expectFloatMtx)) == 0);
        pBuffer->Unmap();

        // バッファ1に計算結果が反映されているかチェック
        skeletonObj.CalculateSkeleton(1);
        pBuffer = skeletonObj.GetMtxBlock(1);
        ptr = pBuffer->Map<uint8_t>();
        pBuffer->InvalidateMappedRange(0, skeletonObj.GetMtxBlockSize());
        EXPECT_TRUE(std::memcmp(ptr, &expectFloatMtx, sizeof(expectFloatMtx)) == 0);
        pBuffer->Unmap();
    }

    // ボーン
    {
        const nn::g3d::ResBone* pResBone = skeletonObj.FindResBone("leg_l2");
        EXPECT_TRUE(pResBone != NULL);
        EXPECT_TRUE(pResBone == skeletonObj.GetResBone(3));
        EXPECT_STREQ("leg_l2", skeletonObj.GetBoneName(skeletonObj.FindBoneIndex("leg_l2")));
    }

    skeletonObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
}

TEST_F(G3dTest, ModelObj)
{
    nn::gfx::Device* pDevice = GetDevice();
    nn::g3d::ModelObj modelObj;
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    pResModel->Reset();
    pResModel->GetMaterial(0)->SetRawParamSize(4);
    pResModel->GetMaterial(0)->FindShaderParam("rim_int")->SetOffset(0);

    nn::g3d::ModelObj::Builder builder(pResModel);
    builder.SetBoundingEnabled();
    builder.MaterialBufferingCount(2);
    builder.ShapeBufferingCount(2);
    builder.SkeletonBufferingCount(2);
    builder.ShapeUserAreaSize(nn::g3d::ShapeBlock::Size_UserArea);
    builder.ViewCount(2);
    builder.CalculateMemorySize();
    EXPECT_TRUE(builder.IsMemoryCalculated());
    size_t bufferSize = builder.GetWorkMemorySize();

    nn::util::MemorySplitter::MemoryBlock memoryBlock;
    size_t defaultAlignment = memoryBlock.GetAlignment();
    size_t expectedSize = CalculateBuildSkeletonObjBufferSize(builder.GetSkeletonBufferingCount(),
                                                         pResModel->GetSkeleton());
    expectedSize = nn::util::align_up(expectedSize, nn::g3d::SkeletonObj::Alignment_Buffer);
    expectedSize = nn::util::align_up(expectedSize, nn::g3d::ShapeObj::Alignment_Buffer);
    size_t shapeSize = 0;
    for (int i = 0; i < pResModel->GetShapeCount(); ++i)
    {
        int viewCount;
        int boneIndex = pResModel->GetShape(i)->GetBoneIndex();
        nn::g3d::ResBone* pResBone = pResModel->GetSkeleton()->GetBone(boneIndex);
        if (pResBone->GetBillboardMode() == nn::g3d::ResBone::Flag_BillboardNone)
        {
            viewCount = 1;
        }
        else
        {
            viewCount = builder.GetViewCount();
        }

        size_t size = CalculateBuildShapeObjBufferSize(viewCount,
                                                  builder.GetShapeBufferingCount(),
                                                  builder.GetShapeUserAreaSize(),
                                                  pResModel->GetShape(i));
        shapeSize += nn::util::align_up(size, nn::g3d::ShapeObj::Alignment_Buffer);
    }
    expectedSize += shapeSize;
    expectedSize = nn::util::align_up(expectedSize, nn::g3d::MaterialObj::Alignment_Buffer);
    size_t materialSize = 0;
    for (int i = 0; i < pResModel->GetMaterialCount(); ++i)
    {
        size_t size = CalculateBuildMaterialObjBufferSize(builder.GetMaterialBufferingCount(),
                                                     pResModel->GetMaterial(i));
        materialSize += nn::util::align_up(size, nn::g3d::MaterialObj::Alignment_Buffer);
    }
    expectedSize += materialSize;
    expectedSize = nn::util::align_up(expectedSize, defaultAlignment);

    expectedSize += sizeof(nn::g3d::SkeletonObj);
    expectedSize = nn::util::align_up(expectedSize, defaultAlignment);
    expectedSize += sizeof(nn::g3d::ShapeObj) * pResModel->GetShapeCount();
    expectedSize = nn::util::align_up(expectedSize, defaultAlignment);
    expectedSize += sizeof(nn::g3d::MaterialObj) * pResModel->GetMaterialCount();
    expectedSize = nn::util::align_up(expectedSize, defaultAlignment);
    expectedSize += nn::util::align_up(pResModel->GetSkeleton()->GetBoneCount(), 32) >> 3;
    expectedSize = nn::util::align_up(expectedSize, defaultAlignment);
    expectedSize += nn::util::align_up(pResModel->GetMaterialCount(), 32) >> 3;
    expectedSize = nn::util::align_up(expectedSize, defaultAlignment);
    if (builder.IsBoundingEnabled())
    {
        expectedSize = nn::util::align_up(expectedSize, 16);
        expectedSize += sizeof(nn::g3d::Sphere);
    }
    EXPECT_EQ(expectedSize, bufferSize);

    SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(bufferSize, nn::g3d::ModelObj::Alignment_Buffer));
    builder.Build(&modelObj, buffer.Get(), bufferSize);

    bufferSize = modelObj.CalculateBlockBufferSize(pDevice);
    expectedSize = nn::util::align_up(sizeof(nn::util::FloatColumnMajor4x3) * pResModel->GetSkeleton()->GetMtxCount(),
                                      modelObj.GetSkeleton()->GetBlockBufferAlignment(pDevice))
                 * modelObj.GetSkeleton()->GetBufferingCount();

    for (int i = 0; i < pResModel->GetShapeCount(); ++i)
    {
        expectedSize = nn::util::align_up(expectedSize, modelObj.GetShape(i)->GetBlockBufferAlignment(pDevice));
        size_t size = nn::util::align_up(sizeof(nn::g3d::ShapeBlock), modelObj.GetShape(i)->GetBlockBufferAlignment(pDevice))
                    * modelObj.GetShape(i)->GetShapeBlockCount()
                    * modelObj.GetShape(i)->GetBufferingCount();
        expectedSize += size;
    }

    for (int i = 0; i < pResModel->GetMaterialCount(); ++i)
    {
        expectedSize = nn::util::align_up(expectedSize, modelObj.GetMaterial(i)->GetBlockBufferAlignment(pDevice));
        size_t size = nn::util::align_up(pResModel->GetMaterial(i)->GetRawParamSize(), modelObj.GetMaterial(i)->GetBlockBufferAlignment(pDevice))
                    * modelObj.GetMaterial(i)->GetBufferingCount();
        expectedSize += size;
    }

    EXPECT_EQ(expectedSize, bufferSize);
    ptrdiff_t offset = AllocateMemoryPool(bufferSize, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(GetDevice(), GetWriteCombineMemoryPool(), offset, bufferSize));
    EXPECT_TRUE(modelObj.IsBlockBufferValid());

    EXPECT_TRUE(modelObj.GetResource() == pResModel);
    EXPECT_TRUE(modelObj.GetBufferPtr() == buffer.Get());
    EXPECT_TRUE(modelObj.GetMemoryPoolPtr() == GetWriteCombineMemoryPool());
    EXPECT_TRUE(modelObj.GetMemoryPoolOffset() == offset);
    EXPECT_TRUE(modelObj.IsViewDependent());

    // ユーザーポインタ
    {
        void* ptr = reinterpret_cast<void*>(0x12345678);
        modelObj.SetUserPtr(ptr);
        EXPECT_EQ(ptr, modelObj.GetUserPtr());
    }

    // ビジビリティ
    {
        EXPECT_TRUE(modelObj.IsBoneVisible(0));
        EXPECT_TRUE(modelObj.IsMaterialVisible(0));
        modelObj.SetBoneVisible(0, false);
        modelObj.SetMaterialVisible(0, false);
        EXPECT_FALSE(modelObj.IsBoneVisible(0));
        EXPECT_FALSE(modelObj.IsMaterialVisible(0));
        modelObj.ClearBoneVisible();
        modelObj.ClearMaterialVisible();
        EXPECT_TRUE(modelObj.IsBoneVisible(0));
        EXPECT_TRUE(modelObj.IsMaterialVisible(0));
    }

    nn::util::Matrix4x3fType baseMtx;
    nn::util::Vector3fType rotateVector;
    VectorSet(&rotateVector, 0.0f, 2 * 3.14f * 2 / 256, 0.0f);
    MatrixSetRotateXyz(&baseMtx, rotateVector);
    nn::util::Vector3fType translateVector;
    VectorSet(&translateVector, 0.0f, 0.0f, 0.0f);
    MatrixSetTranslate(&baseMtx, translateVector);
    modelObj.CalculateWorld(baseMtx);
    modelObj.CalculateShape(0);
    modelObj.CalculateSkeleton(0);
    int* pRimInt = reinterpret_cast<int*>(modelObj.GetMaterial(0)->EditShaderParam(3));
    *pRimInt = 0x12345678;
    modelObj.CalculateMaterial(0);
    modelObj.CalculateBounding();

    // スケルトン
    {
        EXPECT_TRUE(modelObj.GetSkeleton() != NULL);
        nn::g3d::SkeletonObj* pSkeletonObj = modelObj.GetSkeleton();
        nn::gfx::Buffer* pBuffer = pSkeletonObj->GetMtxBlock(0);
        EXPECT_TRUE(pBuffer != NULL);

         // 比較用期待値計算
        const int16_t* pMtxToBoneTable = pSkeletonObj->GetResource()->ToData().pMtxToBoneTable.Get();
        const nn::util::FloatColumnMajor4x3* pInvModelMatrixArray = pSkeletonObj->GetResource()->ToData().pInvModelMatrixArray.Get();
        int boneIndex = pMtxToBoneTable[0];
        nn::util::Matrix4x3fType invMtx;
        nn::util::MatrixLoad(&invMtx, pInvModelMatrixArray[0]);
        nn::util::Matrix4x3fType* pWorldMtx = pSkeletonObj->GetWorldMtxArray();
        nn::util::Matrix4x3fType expectMtx;
        nn::util::MatrixMultiply(&expectMtx, invMtx, pWorldMtx[boneIndex]);
        nn::util::FloatColumnMajor4x3 expectFloatMtx;
        nn::util::MatrixStore(&expectFloatMtx, expectMtx);

        // バッファ0に計算結果が反映されているかチェック
        uint8_t* ptr = pBuffer->Map<uint8_t>();
        pBuffer->InvalidateMappedRange(0, pSkeletonObj->GetMtxBlockSize());
        EXPECT_TRUE(std::memcmp(ptr, &expectFloatMtx, sizeof(expectFloatMtx)) == 0);
        pBuffer->Unmap();
    }

    // シェイプ
    {
        EXPECT_EQ(1, modelObj.GetShapeCount());
        nn::g3d::ShapeObj* pShapeObj = modelObj.FindShape("polygon3");
        EXPECT_TRUE(pShapeObj != NULL);
        EXPECT_TRUE(pShapeObj == modelObj.GetShape(0));
        EXPECT_STREQ("polygon3", modelObj.GetShapeName(modelObj.FindShapeIndex("polygon3")));

        nn::g3d::Sphere* pSphere = modelObj.GetBounding();
        IsFloatEqual(g_AnswerBoundingSphereCenter._v[0], pSphere->center._v[0]);
        IsFloatEqual(g_AnswerBoundingSphereCenter._v[1], pSphere->center._v[1]);
        IsFloatEqual(g_AnswerBoundingSphereCenter._v[2], pSphere->center._v[2]);
        IsFloatEqual(g_AnswerBoundingSphereRadius, pSphere->radius);

        nn::util::FloatColumnMajor4x3 floatIdentityMtx = NN_UTIL_FLOAT_COLUMN_MAJOR_4X3_INITIALIZER(1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f);
        nn::gfx::Buffer* pBuffer = pShapeObj->GetShapeBlock(0, 0);
        nn::g3d::ShapeBlock* pShapeBlock = pBuffer->Map<nn::g3d::ShapeBlock>();
        pBuffer->InvalidateMappedRange(0, sizeof(nn::g3d::ShapeBlock));
        int originalValue0 = pShapeBlock->userInt[0];
        int originalValue1 = pShapeBlock->userInt[1];
        EXPECT_TRUE(memcmp(&floatIdentityMtx, &pShapeBlock->worldMtx, sizeof(floatIdentityMtx)) == 0);
        pBuffer->Unmap();
        pBuffer = pShapeObj->GetShapeBlock(1, 1);
        pShapeBlock = pBuffer->Map<nn::g3d::ShapeBlock>();
        pBuffer->InvalidateMappedRange(0, sizeof(nn::g3d::ShapeBlock));
        EXPECT_TRUE(memcmp(&floatIdentityMtx, &pShapeBlock->worldMtx, sizeof(floatIdentityMtx)) == 0);
        pBuffer->Unmap();

        {
            modelObj.UpdateViewDependency();
            modelObj.SetShapeUserAreaForceUpdateEnabled();
            EXPECT_TRUE(modelObj.IsShapeUserAreaForceUpdateEnabled());
            modelObj.SetShapeUserAreaForceUpdateDisabled();
            EXPECT_FALSE(modelObj.IsShapeUserAreaForceUpdateEnabled());
            int* pUserArea = pShapeObj->GetUserArea<int>();
            pUserArea[0] = 100;
            pUserArea[1] = 200;
            // 更新されないことを確認
            nn::util::Matrix4x3fType identityMtx;
            nn::util::MatrixIdentity(&identityMtx);
            modelObj.CalculateView(0, identityMtx, 0);
            modelObj.CalculateShape(0);
            pBuffer = pShapeObj->GetShapeBlock(0, 0);
            pShapeBlock = pBuffer->Map<nn::g3d::ShapeBlock>();
            pBuffer->InvalidateMappedRange(0, sizeof(nn::g3d::ShapeBlock));
            EXPECT_EQ(originalValue0, pShapeBlock->userInt[0]);
            EXPECT_EQ(originalValue1, pShapeBlock->userInt[1]);
            pBuffer->Unmap();
            // 更新されることを確認
            modelObj.SetShapeUserAreaForceUpdateEnabled();
            modelObj.CalculateView(0, identityMtx, 0);
            modelObj.CalculateShape(0);
            pBuffer = pShapeObj->GetShapeBlock(0, 0);
            pShapeBlock = pBuffer->Map<nn::g3d::ShapeBlock>();
            pBuffer->InvalidateMappedRange(0, sizeof(nn::g3d::ShapeBlock));
            EXPECT_EQ(100, pShapeBlock->userInt[0]);
            EXPECT_EQ(200, pShapeBlock->userInt[1]);
            pBuffer->Unmap();
            pUserArea[0] = 345;
            pUserArea[1] = 678;
            modelObj.CalculateView(1, identityMtx, 1);
            modelObj.CalculateShape(1);
            pBuffer = pShapeObj->GetShapeBlock(1, 1);
            pShapeBlock = pBuffer->Map<nn::g3d::ShapeBlock>();
            pBuffer->InvalidateMappedRange(0, sizeof(nn::g3d::ShapeBlock));
            EXPECT_EQ(345, pShapeBlock->userInt[0]);
            EXPECT_EQ(678, pShapeBlock->userInt[1]);
            pBuffer->Unmap();
        }
    }

    // マテリアル
    {
        EXPECT_EQ(1, modelObj.GetMaterialCount());
        nn::g3d::MaterialObj* pMaterialObj = modelObj.FindMaterial("human_all");
        EXPECT_TRUE(pMaterialObj != NULL);
        EXPECT_TRUE(pMaterialObj == modelObj.GetMaterial(0));
        EXPECT_STREQ("human_all", modelObj.GetMaterialName(modelObj.FindMaterialIndex("human_all")));

        nn::gfx::Buffer* pBuffer = pMaterialObj->GetMaterialBlock(0);
        const int* pBlock = pBuffer->Map<const int>();
        pBuffer->InvalidateMappedRange(0, pMaterialObj->GetMaterialBlockSize());
        EXPECT_EQ(*pRimInt, *pBlock);
        pBuffer->Unmap();
    }

    modelObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
} //NOLINT

namespace {

nn::g3d::ViewVolume g_AnswerFrustumViewVolume =
{
    // aabb
    {
        NN_UTIL_VECTOR_3F_INITIALIZER(-359218.313f, -904903.688f, -728188.000f),
        NN_UTIL_VECTOR_3F_INITIALIZER( 325096.188f,  910781.563f,  708208.000f),

    },
    // plane
    {
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.830511391f,  -0.168502420f,   0.530902803f), -11.9291172f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.874911904f,  -0.124101788f,   0.468110859f), -12.1892090f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.853553355f,  -0.146446556f,   0.499999970f), -7.07106781f},
        {NN_UTIL_VECTOR_3F_INITIALIZER(-0.853553355f,   0.146446541f,  -0.499999940f), -19987.9063f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.849626780f,  -0.125068620f,   0.512339890f), -12.3689804f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.856946826f,  -0.167732969f,   0.487347692f), -11.7656155f},
    },
    6,
    0
};

nn::g3d::ViewVolume g_AnswerOrthoViewVolume =
{
    // aabb
    {
        NN_UTIL_VECTOR_3F_INITIALIZER(-17146.6055f, -216.228424f, -10169.5479f),
        NN_UTIL_VECTOR_3F_INITIALIZER( 91.2715454f,  3165.89233f,  187.049500f),
    },
    // plane
    {
        {NN_UTIL_VECTOR_3F_INITIALIZER(-0.500000060f, -0.499999076f,  0.707107365f), -109.571289f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.500000060f,  0.499999076f, -0.707107365f), -115.428703f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.853553355f, -0.146446556f,  0.499999970f), -7.07106781f},
        {NN_UTIL_VECTOR_3F_INITIALIZER(-0.853554249f,  0.146446913f, -0.499998540f), -19987.9277f},
        {NN_UTIL_VECTOR_3F_INITIALIZER(-0.146445855f,  0.853553355f,  0.500000238f), -212.071030f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.146446377f, -0.853553712f, -0.499999613f), -187.928864f},
    },
    6,
    0
};

nn::g3d::ViewVolume g_AnswerPerspectiveViewVolume =
{
    // aabb
    {
        NN_UTIL_VECTOR_3F_INITIALIZER(-25638.0664f, -11495.9316f, -24546.1152f),
        NN_UTIL_VECTOR_3F_INITIALIZER( 7.87648296f,  17373.7949f,  4566.11914f),
    },
    // plane
    {
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.103504255f, -0.489453465f,  0.865864933f), -4.79915714f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.908737779f,  0.315780222f, -0.272907376f), -9.51610661f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.853553474f, -0.146446571f,  0.499999940f), -7.07106876f},
        {NN_UTIL_VECTOR_3F_INITIALIZER(-0.853553414f,  0.146446541f, -0.499999970f), -19987.9277f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.191341773f,  0.732537866f,  0.653281331f), -15.7716093f},
        {NN_UTIL_VECTOR_3F_INITIALIZER( 0.461939722f, -0.844623148f, -0.270598054f),  6.53281498f},
    },
    6,
    0
};

}

TEST_F(G3dTest, ViewVolume)
{
    // 3x4の行列
    nn::util::Matrix4x3fType viewToWorld;
    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER(0.5f, 0.5f, 0.5f);
    nn::util::Vector3fType rotate = NN_UTIL_VECTOR_3F_INITIALIZER(nn::util::DegreeToRadian(45.0f), nn::util::DegreeToRadian(45.0f), nn::util::DegreeToRadian(45.0f));
    nn::util::Vector3fType trans = NN_UTIL_VECTOR_3F_INITIALIZER(10.0f, 10.0f, 10.0f);
    nn::util::MatrixSetScaleRotateXyz(&viewToWorld, scale, rotate);
    nn::util::MatrixSetTranslate(&viewToWorld, trans);

    nn::g3d::ViewVolume viewVolume;
    viewVolume.SetFrustum(400.0f, -400.0f, -225.0f, 225.0f, 10.0f, 40000.0f, viewToWorld);
    EXPECT_TRUE(IsVector3fEqual(viewVolume.aabb.min, g_AnswerFrustumViewVolume.aabb.min));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.aabb.max, g_AnswerFrustumViewVolume.aabb.max));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[0].normal, g_AnswerFrustumViewVolume.planes[0].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerFrustumViewVolume.planes[0].dist, viewVolume.planes[0].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[1].normal, g_AnswerFrustumViewVolume.planes[1].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerFrustumViewVolume.planes[1].dist, viewVolume.planes[1].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[2].normal, g_AnswerFrustumViewVolume.planes[2].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerFrustumViewVolume.planes[2].dist, viewVolume.planes[2].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[3].normal, g_AnswerFrustumViewVolume.planes[3].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerFrustumViewVolume.planes[3].dist, viewVolume.planes[3].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[4].normal, g_AnswerFrustumViewVolume.planes[4].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerFrustumViewVolume.planes[4].dist, viewVolume.planes[4].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[5].normal, g_AnswerFrustumViewVolume.planes[5].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerFrustumViewVolume.planes[5].dist, viewVolume.planes[5].dist));


    viewVolume.SetOrtho(400.0f, -400.0f, -225.0f, 225.0f, 10.0f, 40000.0f, viewToWorld);
    EXPECT_TRUE(IsVector3fEqual(viewVolume.aabb.min, g_AnswerOrthoViewVolume.aabb.min));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.aabb.max, g_AnswerOrthoViewVolume.aabb.max));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[0].normal, g_AnswerOrthoViewVolume.planes[0].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerOrthoViewVolume.planes[0].dist, viewVolume.planes[0].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[1].normal, g_AnswerOrthoViewVolume.planes[1].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerOrthoViewVolume.planes[1].dist, viewVolume.planes[1].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[2].normal, g_AnswerOrthoViewVolume.planes[2].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerOrthoViewVolume.planes[2].dist, viewVolume.planes[2].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[3].normal, g_AnswerOrthoViewVolume.planes[3].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerOrthoViewVolume.planes[3].dist, viewVolume.planes[3].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[4].normal, g_AnswerOrthoViewVolume.planes[4].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerOrthoViewVolume.planes[4].dist, viewVolume.planes[4].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[5].normal, g_AnswerOrthoViewVolume.planes[5].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerOrthoViewVolume.planes[5].dist, viewVolume.planes[5].dist));

    viewVolume.SetPerspective(nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, 10.0f, 40000.0f, viewToWorld);
    EXPECT_TRUE(IsVector3fEqual(viewVolume.aabb.min, g_AnswerPerspectiveViewVolume.aabb.min));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.aabb.max, g_AnswerPerspectiveViewVolume.aabb.max));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[0].normal, g_AnswerPerspectiveViewVolume.planes[0].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerPerspectiveViewVolume.planes[0].dist, viewVolume.planes[0].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[1].normal, g_AnswerPerspectiveViewVolume.planes[1].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerPerspectiveViewVolume.planes[1].dist, viewVolume.planes[1].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[2].normal, g_AnswerPerspectiveViewVolume.planes[2].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerPerspectiveViewVolume.planes[2].dist, viewVolume.planes[2].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[3].normal, g_AnswerPerspectiveViewVolume.planes[3].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerPerspectiveViewVolume.planes[3].dist, viewVolume.planes[3].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[4].normal, g_AnswerPerspectiveViewVolume.planes[4].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerPerspectiveViewVolume.planes[4].dist, viewVolume.planes[4].dist));
    EXPECT_TRUE(IsVector3fEqual(viewVolume.planes[5].normal, g_AnswerPerspectiveViewVolume.planes[5].normal));
    EXPECT_TRUE(IsFloatEqual(g_AnswerPerspectiveViewVolume.planes[5].dist, viewVolume.planes[5].dist));
}

namespace {

const nn::util::Matrix4x3fType g_AnswerWorldMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     0.000807728909f, 0.999999523f,    0.000000000f,
    -0.999999523f,    0.000807728909f, 0.000000000f,
     0.000000000f,    0.000000000f,    0.999999940f,
     0.00343403057f,  2.90352345f,     0.139957905f
);

const nn::util::Matrix4x3fType g_AnswerWorldViewpointMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     0.00200247183f,   0.998498380f,    0.0547448434f,
    -0.999997973f,     0.00202497328f, -0.000355554919f,
    -0.000465877820f, -0.0547440164f,   0.998500288f,
    0.00343403057f,    2.90352345f,     0.139957905f
);

const nn::util::Matrix4x3fType g_AnswerScreenMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     0.000807539269f, 0.999999702f,    0.000000000f,
    -0.999999702f,    0.000807539269f, 0.000000000f,
     0.000000000f,    0.000000000f,    0.999999940f,
     0.00343403057f,  2.90352345f,     0.139957905f
);

const nn::util::Matrix4x3fType g_AnswerScreenNocombineMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     0.999759078f,     -0.00304756640f,  0.0217405781f,
    -7.85611337e-006f,  0.990267754f,    0.139175713f,
    -0.0219531413f,    -0.139142349f,    0.990029037f,
     0.00309615280f,    0.0477423668f,  -0.402779043f
);

const nn::util::Matrix4x3fType g_AnswerScreenViewpointMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     0.000806328258f,  0.998500049f,     0.0547443777f,
    -0.999999583f,     0.000830623263f, -0.000421037403f,
    -0.000465877820f, -0.0547440164f,    0.998500288f,
     0.00343403057f,   2.90352345f,      0.139957905f
);

const nn::util::Matrix4x3fType g_AnswerYaxisMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     0.000807728909f,  0.999999523f,     0.000000000f,
    -0.999761164f,     0.000807536417f,  0.0218407698f,
     0.0218407605f,   -1.76414214e-005f, 0.999761403f,
     0.00343403057f,   2.90352345f,      0.139957905f
);

const nn::util::Matrix4x3fType g_AnswerYaxisViewpointMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     0.00200247136f,  0.998498261f,    0.0547448397f,
    -0.999761164f,    0.000807536417f, 0.0218407698f,
     0.0217637643f,  -0.0547755025f,   0.998261452f,
     0.00343403057f,  2.90352345f,     0.139957905f
);

}

TEST_F(G3dTest, Billboard)
{
    nn::gfx::Device* pDevice = GetDevice();
    nn::g3d::ModelObj modelObj;
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    pResModel->Reset();

    nn::g3d::ResSkeletalAnim* pResSkeletalAnim = GetResFile()->FindSkeletalAnim("human_walk");
    nn::g3d::SkeletalAnimObj::Builder skeletalAnimbuilder;
    nn::g3d::SkeletalAnimObj skeletalAnimObj;
    skeletalAnimbuilder.Reserve(pResModel);
    skeletalAnimbuilder.Reserve(pResSkeletalAnim);
    skeletalAnimbuilder.CalculateMemorySize();
    size_t bufferSize = skeletalAnimbuilder.GetWorkMemorySize();
    SimplePtr skeletalAnimBuffer(nnt::g3d::AlignedAllocate<void>(bufferSize, nn::g3d::SkeletalAnimObj::Alignment_Buffer));
    bool success = skeletalAnimbuilder.Build(&skeletalAnimObj, skeletalAnimBuffer.Get(), bufferSize);
    EXPECT_TRUE(success);

    // リソースの設定を行います。再設定可能です。
    skeletalAnimObj.SetResource(pResSkeletalAnim);

    // ビルボードモードを強制的に変更する
    nn::g3d::ResBone* pResBone = const_cast<nn::g3d::ResBone*>(pResModel->GetSkeleton()->GetBone(0));
    pResBone->ToData().flag &= ~nn::g3d::ResBone::Mask_Billboard;
    pResBone->ToData().flag |= nn::g3d::ResBone::Flag_BillboardScreenViewVector;

    nn::g3d::ModelObj::Builder builder(pResModel);
    builder.CalculateMemorySize();
    size_t size = builder.GetWorkMemorySize();

    void * buffer = nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer);
    builder.Build(&modelObj, buffer, size);

    size = modelObj.CalculateBlockBufferSize(pDevice);
    ptrdiff_t offset = AllocateMemoryPool(size, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(GetDevice(), GetWriteCombineMemoryPool(), offset, size));

    // モデルへの関連付けを行います。
    skeletalAnimObj.Bind(&modelObj);

    // カメラ行列。
    nn::util::Vector3fType pos = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 2.5f, 7.5f);
    nn::util::Vector3fType target = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 2.5f, 0.0f);
    nn::util::Vector3fType up = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 1.0f, 0.0f);

    nn::util::Matrix4x3fType cameraMtx;
    MatrixLookAtRightHanded(&cameraMtx, pos, target, up);

    skeletalAnimObj.GetFrameCtrl().SetFrame(1.0f);
    skeletalAnimObj.Calculate();
    skeletalAnimObj.ApplyTo(&modelObj);
    skeletalAnimObj.GetFrameCtrl().UpdateFrame();

    // ベース行列。
    nn::util::Matrix4x3fType baseMtx;
    nn::util::Vector3fType rotate = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 3.14f * 2 / 256, 0.0f);
    MatrixSetRotateXyz(&baseMtx, rotate);
    nn::util::Vector3fType translate = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 0.0f, 0.0f);
    MatrixSetAxisW(&baseMtx, translate);

    modelObj.CalculateWorld(baseMtx);

    // GPU 待ちが必要な計算
    modelObj.CalculateSkeleton(0);
    modelObj.CalculateMaterial(0);
    modelObj.CalculateShape(0);
    modelObj.CalculateView(0, cameraMtx, 0);

    nn::util::Matrix4x3fType mtx;
    modelObj.GetSkeleton()->CalculateBillboardMtx(&mtx, cameraMtx, 0, true);
    EXPECT_TRUE(IsMtx43Equal(mtx, g_AnswerScreenMtx, 4));
    modelObj.GetSkeleton()->CalculateBillboardMtx(&mtx, cameraMtx, 0, false);
    EXPECT_TRUE(IsMtx43Equal(mtx, g_AnswerScreenNocombineMtx, 4));
    modelObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
    nnt::g3d::Deallocate(buffer);

    // ビルボードモードを強制的に変更する
    pResBone->ToData().flag &= ~nn::g3d::ResBone::Mask_Billboard;
    pResBone->ToData().flag |= nn::g3d::ResBone::Flag_BillboardWorldViewVector;

    builder.CalculateMemorySize();
    size = builder.GetWorkMemorySize();

    buffer = nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer);
    builder.Build(&modelObj, buffer, size);

    size = modelObj.CalculateBlockBufferSize(pDevice);
    offset = AllocateMemoryPool(size, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(GetDevice(), GetWriteCombineMemoryPool(), offset, size));

    // モデルへの関連付けを行います。
    skeletalAnimObj.Bind(&modelObj);

    skeletalAnimObj.GetFrameCtrl().SetFrame(1.0f);
    skeletalAnimObj.Calculate();
    skeletalAnimObj.ApplyTo(&modelObj);
    skeletalAnimObj.GetFrameCtrl().UpdateFrame();

    modelObj.CalculateWorld(baseMtx);

    // GPU 待ちが必要な計算
    modelObj.CalculateSkeleton(0);
    modelObj.CalculateMaterial(0);
    modelObj.CalculateShape(0);
    modelObj.CalculateView(0, cameraMtx, 0);

    modelObj.GetSkeleton()->CalculateBillboardMtx(&mtx, cameraMtx, 0, true);
    EXPECT_TRUE(IsMtx43Equal(mtx, g_AnswerWorldMtx, 4));
    modelObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
    nnt::g3d::Deallocate(buffer);

    // ビルボードモードを強制的に変更する
    pResBone->ToData().flag &= ~nn::g3d::ResBone::Mask_Billboard;
    pResBone->ToData().flag |= nn::g3d::ResBone::Flag_BillboardWorldViewPoint;

    builder.CalculateMemorySize();
    size = builder.GetWorkMemorySize();

    buffer = nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer);
    builder.Build(&modelObj, buffer, size);

    size = modelObj.CalculateBlockBufferSize(pDevice);
    offset = AllocateMemoryPool(size, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(GetDevice(), GetWriteCombineMemoryPool(), offset, size));

    // モデルへの関連付けを行います。
    skeletalAnimObj.Bind(&modelObj);

    skeletalAnimObj.GetFrameCtrl().SetFrame(1.0f);
    skeletalAnimObj.Calculate();
    skeletalAnimObj.ApplyTo(&modelObj);
    skeletalAnimObj.GetFrameCtrl().UpdateFrame();

    modelObj.CalculateWorld(baseMtx);

    // GPU 待ちが必要な計算
    modelObj.CalculateSkeleton(0);
    modelObj.CalculateMaterial(0);
    modelObj.CalculateShape(0);
    modelObj.CalculateView(0, cameraMtx, 0);

    modelObj.GetSkeleton()->CalculateBillboardMtx(&mtx, cameraMtx, 0, true);
    EXPECT_TRUE(IsMtx43Equal(mtx, g_AnswerWorldViewpointMtx, 4));
    modelObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
    nnt::g3d::Deallocate(buffer);

    // ビルボードモードを強制的に変更する
    pResBone->ToData().flag &= ~nn::g3d::ResBone::Mask_Billboard;
    pResBone->ToData().flag |= nn::g3d::ResBone::Flag_BillboardScreenViewPoint;

    builder.CalculateMemorySize();
    size = builder.GetWorkMemorySize();

    buffer = nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer);
    builder.Build(&modelObj, buffer, size);

    size = modelObj.CalculateBlockBufferSize(pDevice);
    offset = AllocateMemoryPool(size, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(GetDevice(), GetWriteCombineMemoryPool(), offset, size));

    // モデルへの関連付けを行います。
    skeletalAnimObj.Bind(&modelObj);

    skeletalAnimObj.GetFrameCtrl().SetFrame(1.0f);
    skeletalAnimObj.Calculate();
    skeletalAnimObj.ApplyTo(&modelObj);
    skeletalAnimObj.GetFrameCtrl().UpdateFrame();

    modelObj.CalculateWorld(baseMtx);

    // GPU 待ちが必要な計算
    modelObj.CalculateSkeleton(0);
    modelObj.CalculateMaterial(0);
    modelObj.CalculateShape(0);
    modelObj.CalculateView(0, cameraMtx, 0);

    modelObj.GetSkeleton()->CalculateBillboardMtx(&mtx, cameraMtx, 0, true);
    EXPECT_TRUE(IsMtx43Equal(mtx, g_AnswerScreenViewpointMtx, 4));
    modelObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
    nnt::g3d::Deallocate(buffer);

    // ビルボードモードを強制的に変更する
    pResBone->ToData().flag &= ~nn::g3d::ResBone::Mask_Billboard;
    pResBone->ToData().flag |= nn::g3d::ResBone::Flag_BillboardYaxisViewVector;

    builder.CalculateMemorySize();
    size = builder.GetWorkMemorySize();

    buffer = nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer);
    builder.Build(&modelObj, buffer, size);

    size = modelObj.CalculateBlockBufferSize(pDevice);
    offset = AllocateMemoryPool(size, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(GetDevice(), GetWriteCombineMemoryPool(), offset, size));

    // モデルへの関連付けを行います。
    skeletalAnimObj.Bind(&modelObj);

    skeletalAnimObj.GetFrameCtrl().SetFrame(1.0f);
    skeletalAnimObj.Calculate();
    skeletalAnimObj.ApplyTo(&modelObj);
    skeletalAnimObj.GetFrameCtrl().UpdateFrame();

    modelObj.CalculateWorld(baseMtx);

    // GPU 待ちが必要な計算
    modelObj.CalculateSkeleton(0);
    modelObj.CalculateMaterial(0);
    modelObj.CalculateShape(0);
    modelObj.CalculateView(0, cameraMtx, 0);

    modelObj.GetSkeleton()->CalculateBillboardMtx(&mtx, cameraMtx, 0, true);
    EXPECT_TRUE(IsMtx43Equal(mtx, g_AnswerYaxisMtx, 4));
    modelObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
    nnt::g3d::Deallocate(buffer);

    // ビルボードモードを強制的に変更する
    pResBone->ToData().flag &= ~nn::g3d::ResBone::Mask_Billboard;
    pResBone->ToData().flag |= nn::g3d::ResBone::Flag_BillboardYaxisViewPoint;

    builder.CalculateMemorySize();
    size = builder.GetWorkMemorySize();

    buffer = nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer);
    builder.Build(&modelObj, buffer, size);

    size = modelObj.CalculateBlockBufferSize(pDevice);
    offset = AllocateMemoryPool(size, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(GetDevice(), GetWriteCombineMemoryPool(), offset, size));

    // モデルへの関連付けを行います。
    skeletalAnimObj.Bind(&modelObj);

    skeletalAnimObj.GetFrameCtrl().SetFrame(1.0f);
    skeletalAnimObj.Calculate();
    skeletalAnimObj.ApplyTo(&modelObj);
    skeletalAnimObj.GetFrameCtrl().UpdateFrame();

    modelObj.CalculateWorld(baseMtx);

    // GPU 待ちが必要な計算
    modelObj.CalculateSkeleton(0);
    modelObj.CalculateMaterial(0);
    modelObj.CalculateShape(0);
    modelObj.CalculateView(0, cameraMtx, 0);

    modelObj.GetSkeleton()->CalculateBillboardMtx(&mtx, cameraMtx, 0, true);
    EXPECT_TRUE(IsMtx43Equal(mtx, g_AnswerYaxisViewpointMtx, 4));
    modelObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
    nnt::g3d::Deallocate(buffer);
} //NOLINT
