﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @examplesource{Town.cpp,PageSampleG3dTown}
 *
 * @brief
 * 実際のゲームに近い内容のサンプルプログラム
 */

/**
 * @page PageSampleG3dTown G3dDemo Town
 * @tableofcontents
 *
 * @image html Applications\G3dDemo\town.png
 *
 * @brief
 * サンプルプログラム Town の解説です。実際のゲームに近い内容のサンプルプログラムです。
 *
 * @section PageSampleG3dTown_SectionBrief 概要
 * 遠景や地形、キャラクタなど複数のモデルを表示する、実際のゲームに近い内容の処理を行うサンプルです。
 *
 * @section PageSampleG3dTown_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/G3dDemo
 * Samples/Sources/Applications/G3dDemo @endlink 以下にあります。
 *
 * @section PageSampleG3dTown_SectionNecessaryEnvironment 必要な環境
 * 特になし。
 *
 * @section PageSampleG3dTown_SectionHowToOperate 操作方法
 * Joy-Con、DebugPad、PCの場合キーボード、マウスによる入力を用いて、以下の操作が可能です。
 * <p>
 * 各モードにおいて共通操作
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> Start ボタン、スペースキーまたはタッチスクリーン長押し、Joy-Con の+ボタン </td><td> メニューに戻ります。 </td></tr>
 * <tr><td> 左スティックまたはマウス左ボタン押し </td><td> カメラ回転。マウスの場合、画面の中心点からクリック位置への方向、距離によって回転します。 </td></tr>
 * <tr><td> 右スティックまたはマウス右ボタン押し </td><td> カメラ移動。マウスの場合、画面の中心点からクリック位置への方向、距離によって移動します。 </td></tr>
 * <tr><td> L ボタンまたは L キー </td><td> エディットモードとデバッグモードを切り替えます。 </td></tr>
 * <tr><td> R ボタンまたは R キー </td><td> メニュー表示のオン、オフを切り替えます。 </td></tr>
 * </table>
 * エディットモード
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> 十字ボタン上下またはカーソルキー上下 </td><td> エディット項目の選択を切り替えます。 </td></tr>
 * <tr><td> 十字ボタン左右またはカーソルキー左右 </td><td> 選択されているエディット項目の値を増減します。 </td></tr>
 * <tr><td> Selected Model </td><td> 選択対象のモデルを変更します。 </td></tr>
 * <tr><td> Selected Shape </td><td> 選択中のモデルに含まれる選択対象のシェイプを変更します。 </td></tr>
 * <tr><td> Anim Speed </td><td> エディット対象のモデルのアニメーションスピードを変更します。 </td></tr>
 * <tr><td> A ボタンまたは A キー </td><td> 3dEditor が接続されている場合、モデルをアタッチします。接続されていない場合は何もしません。 </td></tr>
 * <tr><td> B ボタンまたは B キー </td><td> 3dEditor が接続されている場合、画面輝度平均値計算用シェーダーをアタッチします。接続されていない場合は何もしません。 </td></tr>
 * <tr><td> X ボタンまたは X キー </td><td> 3dEditor が接続されている場合、シャドウシェーダーをアタッチします。接続されていない場合は何もしません。 </td></tr>
 * <tr><td> Y ボタンまたは Y キー </td><td> 3dEditor が接続されている場合、G バッファー用シェーダーをアタッチします。接続されていない場合は何もしません。 </td></tr>
 * <tr><td> ZR ボタンまたは Z キー </td><td> 選択しているモデルにカメラが移動します。 </td></tr>
 * </table>
 * デバッグモード 1
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> 十字ボタン上下またはカーソルキー上下 </td><td> デバッグ項目の選択を切り替えます。 </td></tr>
 * <tr><td> 十字ボタン左右またはカーソルキー左右 </td><td> 選択されているデバッグ項目の値を切り替えます。 </td></tr>
 * <tr><td> Selected Model </td><td> 選択対象のモデルを変更します。 </td></tr>
 * <tr><td> Selected Shape </td><td> 選択中のモデルに含まれる選択対象のシェイプを変更します。 </td></tr>
 * <tr><td> Model Animation </td><td> モデルアニメーションのオン、オフを切り替えます。 </td></tr>
 * <tr><td> Light Animation </td><td> ライトアニメーションのオン、オフを切り替えます。 </td></tr>
 * <tr><td> Wireframe </td><td> ワイヤーフレーム表示のオン、オフを切り替えます。 </td></tr>
 * <tr><td> Ice </td><td> 雪シェーダーのオン、オフを切り替えます。 </td></tr>
 * <tr><td> Shadow </td><td> シャドウ処理のオン、オフを切り替えます。 </td></tr>
 * <tr><td> Water </td><td> 水面処理のオン、オフを切り替えます。 </td></tr>
 * <tr><td> Geometry </td><td> G バッファー処理のオン、オフを切り替えます。 </td></tr>
 * <tr><td> Light </td><td> ライトプリパス処理のオン、オフを切り替えます。 </td></tr>
 * <tr><td> Average </td><td> 画面の輝度変化の平滑処理のオン、オフを切り替えます。 </td></tr>
 * </table>
 * デバッグモード 2
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> 十字ボタン上下またはカーソルキー上下 </td><td> デバッグ項目の選択を切り替えます。 </td></tr>
 * <tr><td> 十字ボタン左右またはカーソルキー左右 </td><td> 選択されているデバッグ項目の値を切り替えます。 </td></tr>
 * <tr><td> Selected Model </td><td> 選択対象のモデルを変更します。 </td></tr>
 * <tr><td> Selected Shape </td><td> 選択中のモデルに含まれる選択対象のシェイプを変更します。 </td></tr>
 * <tr><td> Bounding Sphere </td><td> バウンディングスフィア表示のオン、オフを切り替えます。 </td></tr>
 * <tr><td> Bounding Box </td><td> バウンディングボックス表示のオン、オフを切り替えます。 </td></tr>
 * <tr><td> Single Shape </td><td> 選択されているシェイプの単体表示のオン、オフを切り替えます。 </td></tr>
 * <tr><td> Lod Level </td><td> LODレベルの色分け表示のオン、オフを切り替えます。 </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleG3dTown_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleG3dTown_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、メニューから Town を選択し、実行してください。
 *
 * @section PageSampleG3dTown_SectionDetail 解説
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - コマンドバッファーを初期化
 * - 各種レンダーターゲットを初期化
 * - シェーダーアーカイブ読み込み、初期化
 * - モデルリソース読み込み、初期化およびテクスチャーの関連付け
 * - シェーダー、動的テクスチャーとモデルの関連付け
 * - モデルとアニメーションの関連付け
 * - ループ開始
 * - 前フレームに作成したコマンドリストの実行開始
 * - 各行列計算
 * - モデル更新およびシェーダー更新
 * - 水面描画のコマンド生成
 * - デプスマップ生成のコマンド生成
 * - Gバッファー生成のコマンド生成
 * - ライトプリパスのコマンド生成
 * - 最終描画結果のためのコマンド生成（不透明)
 * - 最終描画結果のためのコマンド生成（半透明)
 * - 画面輝度計算のコンピュートシェーダーのためのコマンド生成
 * - コマンドリストの実行完了待ち
 * - GPU待ちが必要な計算処理
 * - スキャンバッファーをスワップ
 * - ループ開始に戻る
 *
 */

#include "SampleSources/g3ddemo_ViewerUtility.h"
#include "SampleSources/TextureAverageShader.h"
#include "SampleSources/Town.h"
#include <nn/g3d/g3d_Viewer.h>
#include <cfloat>
#include <nn/time.h>
#include <nnt/graphics/testGraphics_PerformanceProfileData.h>

namespace g3ddemo = nn::g3d::demo;

g3ddemo::ResourceHolder         g_Holder;

nns::g3d::RenderView            g_RenderView;
nn::util::Vector3fType          g_CameraPosition;
nn::util::Vector3fType          g_CameraTarget;

TownViewer                      g_TownViewer;
nn::g3d::ResShaderArchive*      g_pTownShaderArchive = nullptr;
nn::g3d::ResShaderArchive*      g_pTownDemoShaderArchive = nullptr;
nn::g3d::ResShadingModel*       g_pResShadingModelPtr[ShadingModelType_Count];

extern int                      g_SelectedModelIndex;
extern int                      g_SelectedShapeIndex;
extern int                      g_SelectedSubmeshIndex;
extern int                      g_SelectedViewType;
extern int                      g_SelectedFrameBufferType;
extern MenuFlagSet              g_MenuFlag;
extern int                      g_InvalidForceLodLevel;

namespace {

nn::util::Vector3fType          g_CameraUp;
nn::util::Vector3fType          g_LightPosition;

g3ddemo::LoadMeter              g_LoadMeter;

TownContext                     g_Context;
TextureAverageShader            g_TextureAverageShader;
TownPointLight                  g_PointLight;
g3ddemo::ScreenInfo             g_ScreenInfo;
g3ddemo::BoundingRenderer       g_BoundingRenderer;
nn::g3d::ModelObj*              g_pEnvModelObj = nullptr;

nns::gfx::DepthStencilBuffer    g_DepthBufferShadow;
nns::gfx::FrameBuffer           g_FrameBuffer[FrameBufferType_Count];
nns::gfx::ColorBuffer           g_ColorBuffer;
nn::gfx::DescriptorSlot         g_WaterDescriptorSlot;
nn::gfx::DescriptorSlot         g_DepthDescriptorSlot;
nn::gfx::DescriptorSlot         g_GeometryColorDescriptorSlot;
nn::gfx::DescriptorSlot         g_GeometryDepthDescriptorSlot;
nn::gfx::DescriptorSlot         g_LightDescriptorSlot;
nn::gfx::DescriptorSlot         g_ColorDescriptorSlot;
nn::gfx::DescriptorSlot         g_DebugFrameBufferSamplerDescriptorSlot;

CalculateLod                    g_CalculateLodDefault;
CalculateLod                    g_CalculateLodWater;

ViewVolumeRenderer              g_ViewVolumeRenderer;
nn::util::Matrix4x4fType        g_InvViewProjectionMatrix;

int                             g_FrameCount = 0;
const int                       g_BufferingCount = 2;

const char* g_pShaderArchiveNameArray[ShadingModelType_Count] = {
    "shadow",
    "gbuffer",
    "tex_average",
    "town",
};

const char* g_pShaderingModelNameArray[ShadingModelType_Count] = {
    "shadow",
    "gbuffer",
    "tex_average",
    "town_light",
};


// サブメッシュ LOD 計算
const int g_MaxLodLevel = 8;

enum File
{
    File_TownEnv = 0,
    File_WhiteTownWaterCaustics,
    File_WhiteTownWaterFlat,
    File_BgWhiteTown,
    File_Duck,
    File_FemaleA,
    File_Fish,
    File_MaleA,
    File_Light,
    File_Count
};

#define RESOURCE_PATH "Resource:/"

const char* BfshaPath[] = {
    RESOURCE_PATH "shader/demo.bfsha",  // 3D Editor 用
    RESOURCE_PATH "shader/town.bfsha",  // 3D Editor 用
    RESOURCE_PATH "shader/shadow.bfsha",
    RESOURCE_PATH "shader/gbuffer.bfsha",
    RESOURCE_PATH "shader/tex_average.bfsha",
};
const char* BfresPath[] = {
    RESOURCE_PATH "town_env.bfres",
    RESOURCE_PATH "WhiteTown_Water_Caustics.bfres",
    RESOURCE_PATH "WhiteTown_Water_Flat.bfres",
    RESOURCE_PATH "bg_WhiteTown.bfres",
    RESOURCE_PATH "Duck.bfres",
    RESOURCE_PATH "FemaleA.bfres",
    RESOURCE_PATH "Fish.bfres",
    RESOURCE_PATH "MaleA.bfres",
    RESOURCE_PATH "Light.bfres",
};

//-------------------------------------------------------------------
//  初期化・終了処理
//-------------------------------------------------------------------

// 表示情報の初期化
void InitializeScreenInfo(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // テキストメニューを初期化
    g_ScreenInfo.Initialize(pDevice);
    g_ScreenInfo.SetWindowPosition(16, 16);
    g_ScreenInfo.SetFontSize(20);
}

// パフォーマンスメーターの初期化
void InitializePerformanceMeter(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(1);
    info.SetCpuSectionCountMax(64);
    info.SetGpuSectionCountMax(64);
    // このサンプルでは 1つ前のフレームで作成したコマンドを実行するため
    // 計測結果を参照するバッファーを CPU と GPU で 1つずらす必要があります。
    info.SetCpuBufferCount(2);
    info.SetGpuBufferCount(3);
    g_LoadMeter.Initialize(pDevice, info);
}

// フレームバッファーを初期化
void InitializeFrameBuffer(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // 水面用フレームバッファーをセットアップ
    {
        nns::gfx::FrameBuffer::InfoType info(640, 360);
        info.SetColorBufferFormat(nn::gfx::ImageFormat_R11_G11_B10_Float);
        g3ddemo::InitializeFrameBuffer(&g_FrameBuffer[FrameBufferType_Water], &info);
        g_FrameBuffer[FrameBufferType_Water].SetClearColor(0.3f, 0.3f, 0.3f);
        g_WaterDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_FrameBuffer[FrameBufferType_Water].GetColorBuffer()->GetTextureView());
    }

    // シャドウ用デプスバッファー初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(640);
        info.SetHeight(640);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_Texture);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_D32_Float);
        info.SetMipCount(1);
        info.SetArrayLength(TownContext::cascadeShadowCount);
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
        size_t align = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);
        ptrdiff_t offset = pGfxFramework->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, align);
        nn::gfx::MemoryPool* pMemoryPool = pGfxFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);
        g_DepthBufferShadow.Initialize(pDevice, info, pMemoryPool, offset, size);
        g_DepthDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_DepthBufferShadow.GetTextureView());
    }

    // シャドウ用フレームバッファー初期化
    {
        nns::gfx::FrameBuffer::InfoType info(640, 640);
        info.SetDepthBufferDisabled();
        g3ddemo::InitializeFrameBuffer(&g_FrameBuffer[FrameBufferType_Shadow], &info);
    }

    // ジオメトリ用フレームバッファー初期化
    {
        nns::gfx::FrameBuffer::InfoType info(640, 360);
        info.SetColorBufferFormat(nn::gfx::ImageFormat_R11_G11_B10_Float);
        g3ddemo::InitializeFrameBuffer(&g_FrameBuffer[FrameBufferType_Geometry], &info);
        g_GeometryColorDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_FrameBuffer[FrameBufferType_Geometry].GetColorBuffer()->GetTextureView());
        g_GeometryDepthDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_FrameBuffer[FrameBufferType_Geometry].GetDepthBuffer()->GetTextureView());
        g_FrameBuffer[FrameBufferType_Geometry].SetClearColor(0.3f, 0.3f, 0.3f);
    }

    // ライト用フレームバッファー初期化
    {
        nns::gfx::FrameBuffer::InfoType info(640, 360);
        info.SetColorBufferFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetDepthBufferDisabled();
        g3ddemo::InitializeFrameBuffer(&g_FrameBuffer[FrameBufferType_Light], &info);
        g_FrameBuffer[FrameBufferType_Light].SetClearColor(0.0f, 0.0f, 0.0f, 0.0f);
        g_LightDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_FrameBuffer[FrameBufferType_Light].GetColorBuffer()->GetTextureView());
    }

    // コピー用カラーバッファー初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(pGfxFramework->GetDisplayWidth());
        info.SetHeight(pGfxFramework->GetDisplayHeight());
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R11_G11_B10_Float);
        info.SetMipCount(1);
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
        size_t align = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);
        ptrdiff_t offset = pGfxFramework->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, align);
        nn::gfx::MemoryPool* pMemoryPool = pGfxFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);
        g_ColorBuffer.Initialize(pDevice, info, pMemoryPool, offset, size);
        g_ColorDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_ColorBuffer.GetTextureView());
    }

    // フレームバッファーのデバッグ描画用サンプラーを取得し、登録
    {
        nn::gfx::Sampler* pSampler = pGfxFramework->GetSampler(nns::gfx::GraphicsFramework::SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp);
        int slot = pGfxFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
        pGfxFramework->SetSamplerToDescriptorPool(slot, pSampler);
        int* pSlot = g3ddemo::AllocateMemory<int>(sizeof(int));
        *pSlot = slot;
        pSampler->SetUserPtr(pSlot);

        nn::gfx::DescriptorPool* pSamplerDescriptorPool = pGfxFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler);
        pSamplerDescriptorPool->GetDescriptorSlot(&g_DebugFrameBufferSamplerDescriptorSlot, slot);
    }
}

// フレームバッファの破棄
void FinalizeFrameBuffer(nns::gfx::GraphicsFramework* pGfxFramework) NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // フレームバッファのデバッグ描画用 サンプラーの登録を解除
    {
        nn::gfx::Sampler* pSampler = pGfxFramework->GetSampler(nns::gfx::GraphicsFramework::SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp);
        int* pSlot = reinterpret_cast<int*>(pSampler->GetUserPtr());
        pGfxFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, *pSlot);
        g3ddemo::FreeMemory(pSlot);
        pSampler->SetUserPtr(nullptr);
    }

    // ディスクリプタプールからテクスチャーの登録を解除
    g3ddemo::UnregisterTextureFromDescriptorPool(g_FrameBuffer[FrameBufferType_Water].GetColorBuffer()->GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_DepthBufferShadow.GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_FrameBuffer[FrameBufferType_Light].GetColorBuffer()->GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_FrameBuffer[FrameBufferType_Geometry].GetColorBuffer()->GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_FrameBuffer[FrameBufferType_Geometry].GetDepthBuffer()->GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_ColorBuffer.GetTextureView());

    // フレームバッファーの破棄
    g3ddemo::FinalizeFrameBuffer(&g_FrameBuffer[FrameBufferType_Light]);
    g3ddemo::FinalizeFrameBuffer(&g_FrameBuffer[FrameBufferType_Geometry]);
    g3ddemo::FinalizeFrameBuffer(&g_FrameBuffer[FrameBufferType_Shadow]);
    g3ddemo::FinalizeFrameBuffer(&g_FrameBuffer[FrameBufferType_Water]);

    // シャドウ用デプスバッファーの破棄
    {
        ptrdiff_t offset = g_DepthBufferShadow.GetMemoryPoolOffset();
        g_DepthBufferShadow.Finalize(pDevice);
        pGfxFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, offset);
    }

    // カラーバッファーの破棄
    {
        ptrdiff_t offset = g_ColorBuffer.GetMemoryPoolOffset();
        g_ColorBuffer.Finalize(pDevice);
        pGfxFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, offset);
    }
}

// ビューの初期化
void InitializeRenderView(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_RenderView.Initialize(pDevice, ViewType_Count, g_BufferingCount);

    // 投影行列を設定
    {
        nn::util::Matrix4x4fType matrix;
        MatrixPerspectiveFieldOfViewRightHanded(&matrix, nn::util::DegreeToRadian(45.0f),
            16.0f / 9.0f, 10.0f, 40000.0f);
        g_RenderView.SetProjectionMtx(ViewType_Default, matrix);
    }

    {
        nn::util::Matrix4x4fType matrix;
        MatrixPerspectiveFieldOfViewRightHanded(&matrix, nn::util::DegreeToRadian(45.0f),
            static_cast<float>(g_FrameBuffer[FrameBufferType_Water].GetWidth()) / static_cast<float>(g_FrameBuffer[FrameBufferType_Water].GetHeight()),
            10.0f, 40000.0f);
        g_RenderView.SetProjectionMtx(ViewType_Water, matrix);
    }
}

// リソースの初期化
void InitializeResource(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // リソースホルダの初期化
    g_Holder.Initialize();

    // シェーダーアーカイブの初期化
    for (int pathIndex = 0; pathIndex < NN_ARRAY_SIZE(BfshaPath); ++pathIndex)
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(BfshaPath[pathIndex]);
        nn::g3d::ResShaderArchive* pShaderArchive = pFile->GetResShaderArchive();
        pShaderArchive->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
    }

    for (nn::g3d::ResShaderFile* pShaderFile : g_Holder.shaderFiles)
    {
        nn::g3d::ResShaderArchive* pResShaderArchive = pShaderFile->GetResShaderArchive();

        // Viewer で利用するシェーダーを設定します。
        const char* pResShaderArchiveName = pResShaderArchive->GetName();
        if (strcmp(pResShaderArchiveName, "town") == 0)
        {
            g_pTownShaderArchive = pResShaderArchive;
        }
        if (strcmp(pResShaderArchiveName, "demo") == 0)
        {
            g_pTownDemoShaderArchive = pResShaderArchive;
        }

        // モデル固有では無いシェーディングモデルのポインターを設定します。
        for (int shaderIndex = 0; shaderIndex < ShadingModelType_Count; ++shaderIndex)
        {
            if (strcmp(pResShaderArchiveName, g_pShaderArchiveNameArray[shaderIndex]) != 0)
            {
                continue;
            }
            nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->FindShadingModel(g_pShaderingModelNameArray[shaderIndex]);
            g_pResShadingModelPtr[shaderIndex] = pResShadingModel;
            break;
        }
    }
    NN_ASSERT_NOT_NULL(g_pTownShaderArchive);
    NN_ASSERT_NOT_NULL(g_pTownDemoShaderArchive);

    for (nn::g3d::ResShadingModel* pResShadingModel : g_pResShadingModelPtr)
    {
        NN_ASSERT_NOT_NULL(pResShadingModel);
        NN_UNUSED(pResShadingModel);
    }

    // サンプラーの参照テクスチャーが空の場合に割り当てられるダミーテクスチャーの初期化
    g3ddemo::CreateDummyTextureAndSampler(pDevice);

    // リソースモデルの初期化
    for (int pathIndex = 0; pathIndex < NN_ARRAY_SIZE(BfresPath); ++pathIndex)
    {
        nn::g3d::ResFile* pResFile = g3ddemo::LoadResource<nn::g3d::ResFile>(BfresPath[pathIndex]);
        pResFile->Setup(pDevice);
        g_Holder.files.PushBack(pResFile);

        // テクスチャーおよびサンプラーをディスクリプタプールに登録
        g3ddemo::SetupTexture(pDevice, pResFile, g3ddemo::TextureBindCallback);
        g3ddemo::RegisterSamplerToDescriptorPool(pResFile);
    }

    // モデル固有のシェーダーのセットアップ
    const char* materialShaderArchiveFileName = "town.bfsha";
    for (nn::g3d::ResFile* pResFile : g_Holder.files)
    {
        nn::g3d::ResExternalFile* pResExternalFile = pResFile->FindExternalFile(materialShaderArchiveFileName);
        if (pResExternalFile)
        {
            void* pExternalFile = pResExternalFile->GetData();
            NN_ASSERT(nn::g3d::ResShaderFile::IsValid(pExternalFile));
            nn::g3d::ResShaderArchive* pModelShader = nn::g3d::ResShaderFile::ResCast(pExternalFile)->GetResShaderArchive();
            pModelShader->Setup(pDevice);
            pResFile->SetUserPtr(pModelShader);
        }
    }
}

// リソースの破棄
void FinalizeResource(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // モデルに付加されているシェーダーアーカイブを破棄
    for (nn::g3d::ResFile* pResFile : g_Holder.files)
    {
        nn::g3d::ResExternalFile* pResExternalFile = pResFile->FindExternalFile("town.bfsha");
        if (pResExternalFile)
        {
            void* pExternalFile = pResExternalFile->GetData();
            nn::g3d::ResShaderArchive* pModelShader = nn::g3d::ResShaderFile::ResCast(pExternalFile)->GetResShaderArchive();
            pModelShader->Cleanup(pDevice);
        }
    }

    // 環境モデルの破棄
    nns::g3d::DestroyModelObj(pDevice, g_pEnvModelObj);

    // リソースホルダ内を破棄
    g3ddemo::DestroyAll(pDevice, &g_Holder);
}

// 環境モデルの初期化
void InitializeEnvModel(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    nn::g3d::ResFile* pResFile = g_Holder.files[File_TownEnv];
    nn::g3d::ResModel* pResModel = pResFile->GetModel(0);
    nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(0);
    nn::gfx::Sampler* pSampler = pResMaterial->FindSampler("_d0");

    // サンプラーを作成し直す
    g3ddemo::UnregisterSamplerFromDescriptorPool(pSampler);
    pSampler->Finalize(pDevice);

    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToBorder);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToBorder);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
    samplerInfo.SetBorderColorType(nn::gfx::TextureBorderColorType_White);
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_Comparison_MinLinear_MagLinear_MipLinear);
    samplerInfo.SetComparisonFunction(nn::gfx::ComparisonFunction_LessEqual);

    pSampler->Initialize(pDevice, samplerInfo);
    g3ddemo::RegisterSamplerToDescriptorPool(pSampler);

    // フレームバッファーをテクスチャーに接続
    {
        nn::g3d::TextureRef textureRef(g_FrameBuffer[FrameBufferType_Water].GetColorBuffer()->GetTextureView(), g_WaterDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "water");
    }
    {
        nn::g3d::TextureRef textureRef(g_DepthBufferShadow.GetTextureView(), g_DepthDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "shadow");
    }
    {
        nn::g3d::TextureRef textureRef(g_FrameBuffer[FrameBufferType_Geometry].GetColorBuffer()->GetTextureView(), g_GeometryColorDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "normal");
    }
    {
        nn::g3d::TextureRef textureRef(g_FrameBuffer[FrameBufferType_Geometry].GetDepthBuffer()->GetTextureView(), g_GeometryDepthDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "depth");
    }
    {
        nn::g3d::TextureRef textureRef(g_FrameBuffer[FrameBufferType_Light].GetColorBuffer()->GetTextureView(), g_LightDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "light");
    }
    {
        nn::g3d::TextureRef textureRef(g_ColorBuffer.GetTextureView(), g_ColorDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "color");
    }

    // ユニフォームブロックのサイズを設定
    nn::g3d::ResShaderArchive* pModelShader = pResFile->GetUserPtr<nn::g3d::ResShaderArchive>();
    nn::g3d::ShaderUtility::BindShaderParam(pResMaterial, pModelShader->FindShadingModel("town_env"));

    // ModelObjを作成
    nn::g3d::ModelObj::Builder builder(pResModel);
    builder.MaterialBufferingCount(g_BufferingCount);
    builder.SkeletonBufferingCount(g_BufferingCount);
    builder.ShapeBufferingCount(g_BufferingCount);
    g_pEnvModelObj = nns::g3d::CreateModelObj(pDevice, builder);
}

// コンピュートシェーダー用に SSBO とディスパッチパラメータの設定を行います。
void InitializeTextureAverageShader(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_TextureAverageShader.Initialize(pDevice, g_pResShadingModelPtr[ShadingModelType_TextureAverage]);
    g_TextureAverageShader.SetTileResolution(pDevice, 64, 64);
}

// モデルの描画設定を初期化
void InitializeRenderModel(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    for (int fileIndex = File_WhiteTownWaterCaustics; fileIndex <= File_MaleA; ++fileIndex)
    {
        nn::g3d::ResFile* pResFile = pHolder->files[fileIndex];
        CreateRenderModel(pHolder, pDevice, pResFile);
    }
}

// ShapeObj の ユーザーエリアに LOD 情報を設定します。
void SetupShapeUserArea(nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
{
    for (int shapeIndex = 0, shapeCount = pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);

        // 各LODレベルの頂点バッファー内での開始オフセットを設定する
        for (int meshIndex = 0; meshIndex < g_MaxLodLevel; ++meshIndex)
        {
            if (meshIndex < pShapeObj->GetMeshCount())
            {
                const nn::g3d::ResMesh* pResMesh = pShapeObj->GetResMesh(meshIndex);
                pShapeObj->GetUserArea<int>()[meshIndex] = static_cast<int>(pResMesh->GetOffset());
            }
            else
            {
                // LODレベルが存在しない場合は-1
                pShapeObj->GetUserArea<int>()[meshIndex] = -1;
            }
        }
    }

    // ユーザーエリアの更新を必ず行う
    pModelObj->SetShapeUserAreaForceUpdateEnabled();
}

// モデルインスタンスの生成
void InitializeModelAnimObj(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    for (int fileIndex = File_WhiteTownWaterCaustics; fileIndex <= File_MaleA; ++fileIndex)
    {
        nn::g3d::ResFile* pResFile = pHolder->files[fileIndex];
        CreateModelAnimObj(pHolder, pDevice, pResFile, nullptr);
    }
}

// RenderModelObj の初期化
void InitializeRenderModelObj(g3ddemo::ResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_ASSERT_EQUAL(pHolder->modelAnimObjs.GetCount(), pHolder->renderModels.GetCount());

    // RenderModelObj の構築
    for (int modelIndex = 0, modelCount = pHolder->modelAnimObjs.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        CreateRenderModelObj(pHolder, pHolder->modelAnimObjs[modelIndex], pHolder->renderModels[modelIndex]);
    }
}

// RenderModelObj 内にあるシェーダーセレクターに水面描画用のキーを書き込みます。
void SetupWaterVariation(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    int shapeCount = pRenderModelObj->GetModelObj()->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        nns::g3d::RenderUnitObj* pRenderUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
        nn::g3d::ShaderSelector* pShaderSelector = pRenderUnitObj->GetShaderSelector(DrawPassType_Water);
        int optionIndex = pShaderSelector->FindDynamicOptionIndex("water_world");
        if (optionIndex == nn::util::ResDic::Npos)
        {
            continue;
        }

        const nn::g3d::ResShaderOption* pResShaderOption = pShaderSelector->GetDynamicOption(optionIndex);
        int choiceIndex = pResShaderOption->FindChoiceIndex("1");
        if (choiceIndex == nn::util::ResDic::Npos)
        {
            continue;
        }

        pShaderSelector->WriteDynamicKey(optionIndex, choiceIndex);
    }
}

// モデルに LOD やフラスタムカリングの設定
void SetupRenderModelObj(g3ddemo::ResourceHolder* pHolder) NN_NOEXCEPT
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
    {
        ::SetupRenderModelObj(pRenderModelObj);
    }
}

// アニメーションの登録
void RegisterAnimation(const char* pModelName, const char* pAnimName, float step) NN_NOEXCEPT
{
    for (int objIndex = 0, objCount = g_Holder.modelAnimObjs.GetCount(); objIndex < objCount; ++objIndex)
    {
        nns::g3d::ModelAnimObj* pModelAnimObj = g_Holder.modelAnimObjs[objIndex];
        int* pAnimId = g_Holder.animationIds[objIndex];

        if (strcmp(pModelName, pModelAnimObj->GetModelObj()->GetResource()->GetName()) == 0)
        {
            for (nn::g3d::ResFile* pFile : g_Holder.files)
            {
                nn::g3d::ResSkeletalAnim* pResSkeletalAnim;
                if ((pResSkeletalAnim = pFile->FindSkeletalAnim(pAnimName)) != nullptr)
                {
                    int id = pModelAnimObj->CreateAnim(pResSkeletalAnim);
                    pModelAnimObj->GetAnimObj(id)->GetFrameCtrl().SetStep(step);
                    pAnimId[AnimationType_Skeletal] = id;
                }

                nn::g3d::ResMaterialAnim* pResMaterialAnim;
                if ((pResMaterialAnim = pFile->FindMaterialAnim(pAnimName)) != nullptr)
                {
                    int id = pModelAnimObj->CreateAnim(pResMaterialAnim);
                    pModelAnimObj->GetAnimObj(id)->GetFrameCtrl().SetStep(step);
                    pAnimId[AnimationType_Material] = id;
                }

                nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim;
                if ((pResBoneVisibilityAnim = pFile->FindBoneVisibilityAnim(pAnimName)) != nullptr)
                {
                    int id = pModelAnimObj->CreateAnim(pResBoneVisibilityAnim);
                    pModelAnimObj->GetAnimObj(id)->GetFrameCtrl().SetStep(step);
                    pAnimId[AnimationType_BoneVisibility] = id;
                }

                nn::g3d::ResShapeAnim* pResShapeAnim;
                if ((pResShapeAnim = pFile->FindShapeAnim(pAnimName)) != nullptr)
                {
                    int id = pModelAnimObj->CreateAnim(pResShapeAnim);
                    pModelAnimObj->GetAnimObj(id)->GetFrameCtrl().SetStep(step);
                    pAnimId[AnimationType_Shape] = id;
                }
            }
        }
    }
}

// アニメーションの初期化
void InitializeAnimation() NN_NOEXCEPT
{
    RegisterAnimation("bg_WhiteTown", "bg_WhiteTown", 1.0f);
    RegisterAnimation("bg_WhiteTown_cloth", "bg_WhiteTown_cloth", 1.0f);
    RegisterAnimation("Duck", "Duck", 1.0f);
    RegisterAnimation("FemaleA", "FemaleA_MoAnime", 1.0f);
    RegisterAnimation("Fish", "Fish", 1.0f);
    RegisterAnimation("MaleA", "MaleA_MoAnime", 1.0f);
    RegisterAnimation("WhiteTown_Water_Caustics", "WhiteTown_Water_Caustics", 0.2f);
    RegisterAnimation("bg_WhiteTown_Water_Flat", "bg_WhiteTown_Water_Flat", 0.5f);
}

// モデルの位置を初期化
void InitializeModelPosition() NN_NOEXCEPT
{
    for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        const char* pName = pModelAnimObj->GetModelObj()->GetName();
        if (strcmp(pName, "FemaleA") == 0)
        {
            nn::util::Vector3fType rotate;
            nn::util::Vector3fType translate;
            VectorSet(&rotate, 0.0f, nn::util::DegreeToRadian(60.0f), 0.0f);
            VectorSet(&translate, -300.0f, 12.0f, 1550.0f);
            pModelAnimObj->SetRotate(rotate);
            pModelAnimObj->SetTranslate(translate);
        }
        else if (strcmp(pName, "MaleA") == 0)
        {
            nn::util::Vector3fType rotate;
            nn::util::Vector3fType translate;
            VectorSet(&rotate, 0.0f, 0.0f, 0.0f);
            VectorSet(&translate, 0.0f, 15.0f, 170.0f);
            pModelAnimObj->SetRotate(rotate);
            pModelAnimObj->SetTranslate(translate);
        }
    }
}

// ModelAnimObj を更新
void CalculateModelAnimObjs(int uniformBufferIndex, bool enableAnimation) NN_NOEXCEPT;

// リソースホルダーから指定した名前の RenderModelObj を取得
nns::g3d::RenderModelObj* FindRenderModelObj(const char* pModelName) NN_NOEXCEPT;

// ライトの配置を初期化
void InitializePointLight(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // 配置座標を得るために更新
    CalculateModelAnimObjs(0, true);
    CalculateModelAnimObjs(1, true);

    nn::g3d::ResFile* pLightFile = g_Holder.files[File_Light];
    nn::g3d::ResModel* pLightResModel = pLightFile->FindModel("Point");
    NN_ASSERT_NOT_NULL(pLightResModel);

    // ライトの設定を初期化
    g_PointLight.Initialize(pDevice, pLightResModel, g_pResShadingModelPtr[ShadingModelType_Light]);

    // Town のモデルを取得
    nns::g3d::RenderModelObj* pTownRenderModelObj = FindRenderModelObj("bg_WhiteTown");
    NN_ASSERT_NOT_NULL(pTownRenderModelObj);

    const char* pointLightPrefix = "PointLight_";
    const nn::util::Float3 pointLightColor = NN_UTIL_FLOAT_3_INITIALIZER(1.0f, 0.8f, 0.4f);

    // ポイントライト用のボーンを取得し、ライトを配置
    const nn::g3d::SkeletonObj* pSkeletonObj = pTownRenderModelObj->GetModelObj()->GetSkeleton();
    for (int boneIndex = 0, boneCount = pSkeletonObj->GetBoneCount(); boneIndex < boneCount; ++boneIndex)
    {
        const nn::g3d::ResBone* pResBone = pSkeletonObj->GetResBone(boneIndex);
        if (strncmp(pResBone->GetName(), pointLightPrefix, strlen(pointLightPrefix)) != 0)
        {
            continue;
        }

        // ライト用のボーンから位置を取得
        const nn::util::Matrix4x3fType& matrix = pSkeletonObj->GetWorldMtxArray()[boneIndex];
        nn::util::Vector3fType center;
        MatrixGetAxisW(&center, matrix);

        g_PointLight.AddLight(pDevice, pointLightColor, center);
    }
}

// カメラ配置の初期化
void InitializeCamera() NN_NOEXCEPT
{
    VectorSet(&g_CameraPosition, 380.0f, 140.0f, 2400.0f);
    VectorSet(&g_CameraUp, 0.0f, 1.0f, 0.0f);
    VectorSet(&g_CameraTarget, -400.0f, 0.0f, 0.0f);
    VectorSet(&g_LightPosition, 2000.0f, 2200.0f, 2000.0f);
    nn::util::Matrix4x3fType matrix;
    MatrixLookAtRightHanded(&matrix, g_CameraPosition, g_CameraTarget, g_CameraUp);
    g_RenderView.SetViewMtx(ViewType_Default, matrix);
}

// ビューアーライブラリの初期化
void InitializeViewer() NN_NOEXCEPT
{
    g_TownViewer.Initialize(g_Holder, TownViewer::TownViewerCallback, TownViewer::ViewerTextureBindCallback);
    g_TownViewer.StartPollThread();
    g_TownViewer.Open();
}

void FinalizeViewer() NN_NOEXCEPT
{
    // ビューアーの終了処理
    g_TownViewer.Close();
    g_TownViewer.StopPollThread();
    g_TownViewer.Finalize();
}

// Townの初期化
void InitializeTown(nns::gfx::GraphicsFramework* pGfxFramework) NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // メニュー表示用パラメーターの初期化
    InitializeMenu();

    // 表示情報の初期化
    InitializeScreenInfo(pDevice);

    // バウンディング表示の初期化
    g_BoundingRenderer.Initialize(pDevice);

    // 現在のスレッド(メインスレッド)を 0 番コアに割り当て
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

    // 負荷計測メーターの初期化
    InitializePerformanceMeter(pDevice);

    // フレームバッファーの初期化
    InitializeFrameBuffer(pGfxFramework, pDevice);

    // コンテキストの初期化
    g_Context.Initialize(pDevice);

    // ビューの初期化
    InitializeRenderView(pDevice);

    // リソースの初期化
    InitializeResource(pDevice);

    // 環境モデルの初期化
    InitializeEnvModel(pDevice);

    // フレームバッファの平均輝度を求めるシェーダーを初期化
    InitializeTextureAverageShader(pDevice);

    // モデルの描画設定を初期化
    InitializeRenderModel(&g_Holder, pDevice);

    // モデルインスタンスの生成
    InitializeModelAnimObj(&g_Holder, pDevice);

    // モデル描画設定とモデルインスタンスの関連付け
    InitializeRenderModelObj(&g_Holder);

    // モデルに LOD やフラスタムカリングの設定
    SetupRenderModelObj(&g_Holder);

    // アニメーションの登録
    InitializeAnimation();

    // モデルの位置を初期化
    InitializeModelPosition();

    // ポイントライトを初期化
    InitializePointLight(pDevice);

    // カメラ配置の初期化
    InitializeCamera();

    // ビューアーライブラリの初期化
    InitializeViewer();

    // ビューボリューム表示の初期化
    g_ViewVolumeRenderer.Initialize();

    // シェーダーに必要なスクラッチメモリーを設定
    g3ddemo::SetShaderScratchMemory(g_Holder.shaderFiles);
}

// Townの終了処理
void FinalizeTown(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // ビューボリューム表示の終了処理
    g_ViewVolumeRenderer.Finalize();

    // ライトの終了処理
    g_PointLight.Finalize(pDevice);

    FinalizeViewer();

    // 平均輝度を求めるシェーダーの破棄
    g_TextureAverageShader.Finalize(pDevice);

    // ダミーテクスチャーの破棄
    g3ddemo::DeleteDummyTextureAndSampler(pDevice);

     // リソースの破棄
    FinalizeResource(pDevice);

    // ビューの破棄
    g_RenderView.Finalize(pDevice);

    // コンテキストの破棄
    g_Context.Finalize(pDevice);

    // フレームバッファの終了処理
    FinalizeFrameBuffer(pGfxFramework);

    // 負荷計測メーターの破棄
    g_LoadMeter.Cleanup(pDevice);

     // バウンディング表示の破棄
    g_BoundingRenderer.Finalize(pDevice);

    // スクリーン情報表示の破棄
    g_ScreenInfo.Cleanup(pDevice);

    // フレームワークのコールバック関数を無効化
    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);
}

//-------------------------------------------------------------------
//  更新
//-------------------------------------------------------------------

// カメラの更新処理
void UpdateCamera(const g3ddemo::Pad& pad) NN_NOEXCEPT
{
    nn::util::Vector3fType& pos = g_CameraPosition;
    nn::util::Vector3fType& target = g_CameraTarget;
    nn::util::Vector3fType& up = g_CameraUp;
    const nn::util::Matrix4x3fType& viewMtx = g_RenderView.GetViewMtx(ViewType_Default);

    const float RotSpeed = 0.02f;
    const float AtMoveSpeed = 20.0f;

    nn::util::Vector3fType right;
    nn::util::Vector3fType camUp;
    nn::util::Vector3fType look;
    g3ddemo::MatrixGetColumn0(&right, viewMtx);
    g3ddemo::MatrixGetColumn1(&camUp, viewMtx);
    g3ddemo::MatrixGetColumn2(&look, viewMtx);

    nn::util::Vector3fType diff;
    VectorSubtract(&diff, target, g_CameraPosition);

    float sqrtLength = VectorLengthSquared(diff);
    float dist = sqrtLength / std::sqrt(sqrtLength);
    VectorNormalize(&diff, diff);

    nn::util::Vector3fType dirX;
    nn::util::Vector3fType dirY;
    nn::util::Vector3fType dirZ;

    VectorCross(&dirZ, right, camUp);
    VectorNormalize(&dirZ, dirZ);

    VectorCross(&dirY, dirZ, right);
    VectorNormalize(&dirY, dirY);

    const g3ddemo::Pad::AnalogStick& analogStick = pad.GetAnalogStick();

    VectorMultiply(&dirX, right, AtMoveSpeed * (analogStick.rightX));
    VectorAdd(&pos, dirX, pos);
    VectorMultiply(&dirZ, dirZ, AtMoveSpeed * (-analogStick.rightY));
    VectorAdd(&pos, dirZ, pos);

    nn::util::Vector4fType quatanion;
    nn::util::Vector4fType quatanionX;
    nn::util::Vector4fType quatanionY;
    {
        float angle = analogStick.leftY * RotSpeed * 0.5f;
        float cos = std::cos(angle);
        float sin = std::sin(angle);
        VectorSet(&quatanionX, sin * VectorGetX(right), sin * VectorGetY(right), sin * VectorGetZ(right), cos);
    }
    {
        float angle = analogStick.leftX * -RotSpeed * 0.5f;
        VectorSet(&quatanionY, 0.0f, std::sin(angle), 0.0f, std::cos(angle));
    }

    QuaternionMultiply(&quatanion, quatanionY, quatanionX);

    {
        nn::util::Matrix4x3fType matrix;
        MatrixSetRotate(&matrix, quatanion);

        VectorTransformNormal(&diff, diff, matrix);
        VectorNormalize(&diff, diff);

        VectorMultiply(&target, diff, dist);
        VectorAdd(&target, target, pos);

        VectorCross(&up, right, diff);
        VectorSet(&up, 0.0f, (VectorGetY(up) >= 0) ? 1.0f : -1.0f, 0.0f);
    }

    {
        nn::util::Matrix4x3fType matrix;
        MatrixLookAtRightHanded(&matrix, pos, target, up);
        g_RenderView.SetViewMtx(ViewType_Default, matrix);
    }
}

// モデルの更新処理
void CalculateModelAnimObjs(int uniformBufferIndex, bool enableAnimation) NN_NOEXCEPT
{
    // 各描画パス共通のModelAnimObjを更新
    for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        // アニメーションの実行
        if (enableAnimation)
        {
            pModelAnimObj->CalculateAnimations();

            // 3DEditor でスケルタルア二メーションが再生されていた場合には、
            // CalculateAnimations() で更新されたアニメーションを上書きします。
            // また、キャッシュを活用するためにスケルタルアニメーションの計算から
            // ワールド行列計算までを連続して行います。
            nn::g3d::viewer::ViewerServer::GetInstance().CalculateSkeletalAnimations(pModelAnimObj->GetModelObj());
        }

        pModelAnimObj->CalculateWorld();
        pModelAnimObj->CalculateBounding();
    }

    if (enableAnimation)
    {
        nn::g3d::viewer::ViewerServer::GetInstance().CalculateAnimations();
    }

    for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        pModelAnimObj->CalculateUniformBlock(uniformBufferIndex);
    }

}

// リソースホルダーから指定した名前の RenderModelObj を取得
nns::g3d::RenderModelObj* FindRenderModelObj(const char* pModelName) NN_NOEXCEPT
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        if (strcmp(pRenderModelObj->GetModelObj()->GetName(), pModelName) != 0)
        {
            continue;
        }

        return pRenderModelObj;
    }

    return nullptr;
}

// AABB のサイズを計算
void CalculateAdjustedAabb(
    nn::g3d::Aabb* pAabbObj,
    nns::g3d::RenderUnitObj* pRenderUnitObj,
    const nn::util::Matrix4x3fType* pLightView,
    nn::g3d::ViewVolume* pViewVolume
) NN_NOEXCEPT
{
    const nn::g3d::ShapeObj* pShapeObj = pRenderUnitObj->GetShapeObj();
    const nn::g3d::Sphere* pSphere = pShapeObj->GetBounding();
    if (!pSphere || !pViewVolume->TestIntersection(*pSphere))
    {
        return;
    }

    nn::util::Vector3fType pos;
    VectorTransform(&pos, pSphere->center, *pLightView);
    VectorSet(&pAabbObj->min,
        (std::min)(VectorGetX(pAabbObj->min), VectorGetX(pos) - pSphere->radius),
        (std::min)(VectorGetY(pAabbObj->min), VectorGetY(pos) - pSphere->radius),
        (std::min)(VectorGetZ(pAabbObj->min), VectorGetZ(pos) - pSphere->radius));
    VectorSet(&pAabbObj->max,
        (std::max)(VectorGetX(pAabbObj->max), VectorGetX(pos) + pSphere->radius),
        (std::max)(VectorGetY(pAabbObj->max), VectorGetY(pos) + pSphere->radius),
        (std::max)(VectorGetZ(pAabbObj->max), VectorGetZ(pos) + pSphere->radius));
}

bool CanDrawToShadowMap(const nns::g3d::RenderUnitObj* pRenderUnitObj, int passIndex) NN_NOEXCEPT;

// 光源のプロジェクション行列を計算
void CalculateLightProj(
    nn::util::Matrix4x4fType* pOutLightProj,
    const nn::util::Matrix4x3fType* pLightView,
    const nn::util::Matrix4x3fType* pCameraView,
    const nn::util::Matrix4x4fType* pCameraProj
) NN_NOEXCEPT
{
    nn::util::Matrix4x4fType cameraViewProj;
    nn::util::Matrix4x4fType invCameraViewProj;
    nn::util::Matrix4x4fType screenToLightView;
    MatrixMultiply(&cameraViewProj, *pCameraView, *pCameraProj);
    MatrixInverse(&invCameraViewProj, cameraViewProj);
    MatrixMultiply(&screenToLightView, invCameraViewProj, *pLightView);

    nn::g3d::Aabb aabb;
    nn::g3d::Aabb aabbObj;
    VectorSet(&aabb.min, FLT_MAX, FLT_MAX, FLT_MAX);
    VectorSet(&aabbObj.min, FLT_MAX, FLT_MAX, FLT_MAX);
    VectorSet(&aabb.max, -FLT_MAX, -FLT_MAX, -FLT_MAX);
    VectorSet(&aabbObj.max, -FLT_MAX, -FLT_MAX, -FLT_MAX);

    nn::util::Vector3fType posLightView;
    for (int x = 0; x < 2; ++x)
    {
        for (int y = 0; y < 2; ++y)
        {
            for (int z = 0; z < 2; ++z)
            {
                VectorSet(&posLightView,
                    x == 0 ? -1.0f : 1.0f,
                    y == 0 ? -1.0f : 1.0f,
                    z == 0 ? -1.0f : 1.0f);
                VectorTransformCoord(&posLightView, posLightView, screenToLightView);
                VectorSet(&aabb.min,
                    (std::min)(VectorGetX(aabb.min), VectorGetX(posLightView)),
                    (std::min)(VectorGetY(aabb.min), VectorGetY(posLightView)),
                    (std::min)(VectorGetZ(aabb.min), VectorGetZ(posLightView)));
                VectorSet(&aabb.max,
                    (std::max)(VectorGetX(aabb.max), VectorGetX(posLightView)),
                    (std::max)(VectorGetY(aabb.max), VectorGetY(posLightView)),
                    (std::max)(VectorGetZ(aabb.max), VectorGetZ(posLightView)));
            }
        }
    }

    nn::g3d::ViewVolume viewVolume;
    nn::util::Matrix4x3fType invLightView;
    MatrixInverse(&invLightView, *pLightView);
    viewVolume.SetOrtho(VectorGetY(aabb.max), VectorGetY(aabb.min), VectorGetX(aabb.min), VectorGetX(aabb.max), 0.0f, -VectorGetZ(aabb.min), invLightView);

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        for (int shapeIndex = 0, shapeCount = pRenderModelObj->GetModelObj()->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            nns::g3d::RenderUnitObj* pRenderUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
            if (!CanDrawToShadowMap(pRenderUnitObj, DrawPassType_Shadow))
            {
                continue;
            }
            CalculateAdjustedAabb(&aabbObj, pRenderUnitObj, pLightView, &viewVolume);
        }

    }

    MatrixOrthographicOffCenterRightHanded(pOutLightProj,
        VectorGetX(aabbObj.min), VectorGetX(aabbObj.max), VectorGetY(aabbObj.min),
        VectorGetY(aabbObj.max), -VectorGetZ(aabbObj.max), -VectorGetZ(aabbObj.min));
}

// コンテキストの計算
void CalculateContext(int bufferIndex) NN_NOEXCEPT
{
    const float cascadeDepth[TownContext::cascadeShadowCount + 1] = { 10.0f, 800.0f, 1600.0f, 4500.0f, 40000.0f };

    for (int sliceIndex = 0; sliceIndex < TownContext::cascadeShadowCount; ++sliceIndex)
    {
        nn::util::Matrix4x4fType projMtx;
        MatrixPerspectiveFieldOfViewRightHanded(&projMtx, nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, cascadeDepth[sliceIndex], cascadeDepth[sliceIndex + 1]);

        nn::util::Vector3fType up = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 1.0f, 0.0f);
        nn::util::Vector3fType target = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 0.0f, 0.0f);

        {
            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(&matrix, g_LightPosition, target, up);
            g_RenderView.SetViewMtx(ViewType_Light0 + sliceIndex, matrix);
        }

        {
            nn::util::Matrix4x4fType matrix;
            CalculateLightProj(
                &matrix,
                &g_RenderView.GetViewMtx(ViewType_Light0 + sliceIndex),
                &g_RenderView.GetViewMtx(ViewType_Default),
                &projMtx
            );

            g_RenderView.SetProjectionMtx(ViewType_Light0 + sliceIndex, matrix);
        }
        MatrixMultiply(&projMtx, g_RenderView.GetViewMtx(ViewType_Light0 + sliceIndex), g_RenderView.GetProjectionMtx(ViewType_Light0 + sliceIndex));

        g_Context.SetLightProjection(bufferIndex, sliceIndex, projMtx);
    }

    g_Context.SetCascadeDepth(bufferIndex, &cascadeDepth[1]);
    g_Context.SetNearFar(bufferIndex, 10.0f, 40000.0f);
    float tanFovY = std::tan(nn::util::DegreeToRadian(45.0f * 0.5f));
    g_Context.SetTangentFov(bufferIndex, tanFovY * (16.0f / 9.0f), tanFovY);
    if (g_MenuFlag.Test<MenuFlag::TextureCompute>() && g_FrameCount > 1)
    {
        // 輝度平均を算出
        g_TextureAverageShader.CalculateAverage(bufferIndex);
        nn::util::Float4& texAverage = g_TextureAverageShader.GetTextureAverage(bufferIndex);
        float diff = texAverage.x + texAverage.y + texAverage.z - 0.65f;
        // 前回計算した結果
        float diffIntensity = g_Context.GetDiffIntensity(1 - bufferIndex);
        diffIntensity -= diff * diff * diff * 0.04f;
        if (diffIntensity < 0.3f) diffIntensity = 0.3f;
        if (diffIntensity > 1.7f) diffIntensity = 1.7f;
        g_Context.SetDiffIntensity(bufferIndex, diffIntensity);
    }
    else
    {
        g_Context.SetDiffIntensity(bufferIndex, nn::g3d::Math::Abs(((g_FrameCount + 27000) % 36000 - 18000) / 18000.0f) * 2.0f);
    }

    g_Context.Calculate(bufferIndex);
}

// 計算処理
void Calculate(nns::gfx::GraphicsFramework* pGfxFramework, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pGfxFramework);

    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    // 計算
    NN_PERF_BEGIN_MEASURE_NAME("Calculate 1");
    {
        const g3ddemo::Pad& pad = g3ddemo::GetPad();

        // メニューコントロール
        UpdateMenuParameter(pad);

        // カメラコントロール
        UpdateCamera(pad);

        // 水面用のビュー行列を更新
        {
            nn::util::Vector3fType pos;
            VectorSet(&pos, VectorGetX(g_CameraPosition), -VectorGetY(g_CameraPosition), VectorGetZ(g_CameraPosition));
            nn::util::Vector3fType target;
            VectorSet(&target, VectorGetX(g_CameraTarget), -VectorGetY(g_CameraTarget), VectorGetZ(g_CameraTarget));

            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(&matrix, pos, target, g_CameraUp);
            g_RenderView.SetViewMtx(ViewType_Water, matrix);
        }

        // 環境モデルの更新処理
        g_pEnvModelObj->CalculateMaterial(uniformBufferIndex);

        // ビューアーの更新処理
        g_TownViewer.Update();

        // ModelAnimObjの更新処理
        CalculateModelAnimObjs(uniformBufferIndex, g_MenuFlag.Test<MenuFlag::ModelAnimation>());

        // シェーダーの更新処理
        for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
        {
            pRenderModelObj->UpdateShader(pDevice);
        }

        {
            g_TextureAverageShader.UpdateShader(pDevice);
        }

        // ライトの更新処理
        g_PointLight.Calculate(pDevice, uniformBufferIndex, g_MenuFlag.Test<MenuFlag::LightAnimation>());

        // コンテキストの計算
        CalculateContext(uniformBufferIndex);

        // ビューの更新処理
        g_RenderView.Calculate(uniformBufferIndex);

        // ビューボリューム描画設定の更新
        g_ViewVolumeRenderer.Update(uniformBufferIndex);
        g_ViewVolumeRenderer.SetViewMatrix(g_RenderView.GetViewMtx(ViewType_Default));
        g_ViewVolumeRenderer.SetProjectionMatrix(g_RenderView.GetProjectionMtx(ViewType_Default));

        // フラスタムカリング用のビューボリュームを更新
        if (g_MenuFlag.Test<MenuFlag::FreezeViewVolume>())
        {
            if (g_SelectedViewType != ViewType_Default)
            {
                 g_RenderView.GetViewVolume(ViewType_Default).SetPerspective(nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f,
                10.0f, 40000.0f, g_RenderView.GetInverseViewMtx(ViewType_Default));
            }

            if (g_SelectedViewType != ViewType_Water)
            {
                g_RenderView.GetViewVolume(ViewType_Water).SetPerspective(nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f,
                10.0f, 40000.0f, g_RenderView.GetInverseViewMtx(ViewType_Water));
            }
        }
        else
        {
            g_RenderView.GetViewVolume(ViewType_Default).SetPerspective(nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f,
                10.0f, 40000.0f, g_RenderView.GetInverseViewMtx(ViewType_Default));
            g_RenderView.GetViewVolume(ViewType_Water).SetPerspective(nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f,
                10.0f, 40000.0f, g_RenderView.GetInverseViewMtx(ViewType_Water));

            const nn::util::Matrix4x3fType viewMatrix = g_RenderView.GetViewMtx(g_SelectedViewType);
            const nn::util::Matrix4x4fType projectionMatrix = g_RenderView.GetProjectionMtx(g_SelectedViewType);
            nn::util::Matrix4x4fType viewProjectionMatrix;
            nn::util::MatrixMultiply(&viewProjectionMatrix, viewMatrix, projectionMatrix);

            nn::util::MatrixInverse(&g_InvViewProjectionMatrix, viewProjectionMatrix);
        }
    }
    NN_PERF_END_MEASURE();
}

//-------------------------------------------------------------------
//  描画
//-------------------------------------------------------------------

// 画面のクリア
void ClearScreen(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Black());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Clear");

    // フレームバッファーをクリア
    g3ddemo::ClearFrameBuffer(pCommandBuffer, &g_FrameBuffer[FrameBufferType_Water]);
    g3ddemo::ClearFrameBuffer(pCommandBuffer, &g_FrameBuffer[FrameBufferType_Geometry]);
    g3ddemo::ClearFrameBuffer(pCommandBuffer, &g_FrameBuffer[FrameBufferType_Shadow]);
    g3ddemo::ClearFrameBuffer(pCommandBuffer, &g_FrameBuffer[FrameBufferType_Light]);

    // シャドウ用の深度ステンシルをクリア
    nn::gfx::DepthStencilView* pShadowDepth = g_DepthBufferShadow.GetDepthStencilView(0);
    pCommandBuffer->ClearDepthStencil(pShadowDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

    // カラーターゲットをクリア
    nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
    pCommandBuffer->ClearColor(pColor, 0.0f, 0.0f, 0.0f, 1.0f, nullptr);

    // 深度ステンシルをクリア
    nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
    pCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// シャドウマップに描画するか判定
bool CanDrawToShadowMap(const nns::g3d::RenderUnitObj* pRenderUnitObj, int passIndex) NN_NOEXCEPT
{
    const nn::g3d::ResMaterial* pResMaterial = pRenderUnitObj->GetRenderShape(passIndex)->GetRenderMaterial()->GetResMaterial();
    const nn::g3d::ResRenderInfo* pRenderInfo = pResMaterial->FindRenderInfo("dynamic_shadow");

    if (!pRenderInfo)
    {
        return false;
    }
    return *pRenderInfo->GetInt() == 1;
}

// シャドウの描画
void DrawShadow(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::ShadowRendering>())
    {
        return;
    }

    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Shadow");

    for (int sliceIndex = 0; sliceIndex < TownContext::cascadeShadowCount; ++sliceIndex)
    {
        nn::gfx::ColorTargetView* pColorTargetView = g_FrameBuffer[FrameBufferType_Shadow].GetColorBuffer()->GetColorTargetView();
        pCommandBuffer->SetRenderTargets(1, &pColorTargetView, g_DepthBufferShadow.GetDepthStencilView(sliceIndex + 1));
        pCommandBuffer->SetViewportScissorState(g_FrameBuffer[FrameBufferType_Shadow].GetViewportScissorState());

        // ユニフォームブロックを再バインド
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(ViewType_Light0 + sliceIndex, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(ViewType_Light0 + sliceIndex, 1);

        for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
        {
            pRenderModelObj->BindUniformBlock(DrawPassType_Shadow, "view", buffers, g_BufferingCount);
            pRenderModelObj->Draw(
                pCommandBuffer,
                DrawPassType_Shadow,
                ViewType_Light0 + sliceIndex,
                uniformBufferIndex,
                CanDrawToShadowMap
            );
        }
    }

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// 水面の描画
void DrawWater(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::WaterRendering>())
    {
        return;
    }

    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Blue());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Water");

    g3ddemo::SetFrameBufferToRenderTarget(pCommandBuffer, &g_FrameBuffer[FrameBufferType_Water]);
    pCommandBuffer->SetViewportScissorState(g_FrameBuffer[FrameBufferType_Water].GetViewportScissorState());

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        pRenderModelObj->Draw(pCommandBuffer, DrawPassType_Water, ViewType_Water, uniformBufferIndex);
    }

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// Gバッファーの描画
void DrawGeometry(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::GeometryRendering>())
    {
        return;
    }

    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Geometry");

    g3ddemo::SetFrameBufferToRenderTarget(pCommandBuffer, &g_FrameBuffer[FrameBufferType_Geometry]);
    pCommandBuffer->SetViewportScissorState(g_FrameBuffer[FrameBufferType_Geometry].GetViewportScissorState());

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        pRenderModelObj->Draw(pCommandBuffer, DrawPassType_Geometry, ViewType_Default, uniformBufferIndex);
    }

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// ライトの描画
void DrawLight(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::LightRendering>())
    {
        return;
    }

    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Yellow());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Light");

    g3ddemo::SetFrameBufferToRenderTarget(pCommandBuffer, &g_FrameBuffer[FrameBufferType_Light]);
    pCommandBuffer->SetViewportScissorState(g_FrameBuffer[FrameBufferType_Light].GetViewportScissorState());
    g_PointLight.Draw(pCommandBuffer, uniformBufferIndex);

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// 不透明オブジェクトか判定
bool IsOpaqueObject(const nns::g3d::RenderUnitObj* pRenderUnitObj, int passIndex) NN_NOEXCEPT
{
    const nn::g3d::ResMaterial* pResMaterial = pRenderUnitObj->GetRenderShape(passIndex)->GetRenderMaterial()->GetResMaterial();
    const nn::g3d::ResRenderInfo* pRenderInfo = pResMaterial->FindRenderInfo("blend");
    if (!pRenderInfo)
    {
        return true;
    }
    return *pRenderInfo->GetInt() == g3ddemo::BlendMode_Opaque;
}

// 不透明オブジェクトの描画
void DrawOpaque(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8(255, 0, 200));
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Opaque");

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        pRenderModelObj->Draw(
            pCommandBuffer,
            DrawPassType_Model,
            ViewType_Default,
            uniformBufferIndex,
            IsOpaqueObject
        );
    }

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// 半透明オブジェクトの描画
void DrawTranslucent(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Blue());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Translucent");

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        pRenderModelObj->Draw(
            pCommandBuffer,
            DrawPassType_Model,
            ViewType_Default,
            uniformBufferIndex,
            [](const nns::g3d::RenderUnitObj* pRenderUnitObj, int passIndex)
            {
                return !IsOpaqueObject(pRenderUnitObj, passIndex);
            }
        );
    }

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// カラーバッファーへコピー
void CopyColorBuffer(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::White());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Copy");

    nn::gfx::TextureSubresource subresource;
    subresource.SetDefault();
    nn::gfx::TextureCopyRegion copyRegion;
    copyRegion.SetDefault();
    copyRegion.SetWidth(pGfxFramework->GetDisplayWidth());
    copyRegion.SetHeight(pGfxFramework->GetDisplayHeight());
    pCommandBuffer->CopyImage(g_ColorBuffer.GetTexture(), subresource, 0, 0, 0, pGfxFramework->GetColorBuffer(), copyRegion);

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// 選択されているオブジェクトのみを描画
void DrawSelectedObj(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8(255, 0, 200));
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "DrawSelectedObj");

    int modelIndex = g_SelectedModelIndex;
    int shapeIndex = g_SelectedShapeIndex;
    int submeshIndex = g_SelectedSubmeshIndex;

    NN_ASSERT_RANGE(modelIndex, 0, g_Holder.modelAnimObjs.GetCount());
    nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[modelIndex];

    nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
    NN_ASSERT_RANGE(shapeIndex, 0, pModelObj->GetShapeCount());
    pRenderModelObj->SetDrawSingleShapeEnabled(shapeIndex);

    if (g_MenuFlag.Test<MenuFlag::SingleSubmesh>())
    {
        NN_ASSERT_RANGE(submeshIndex, 0, pModelObj->GetShape(shapeIndex)->GetSubMeshCount());
        pRenderModelObj->GetRenderUnitObj(shapeIndex)->SetDrawSingleSubmeshEnabled(submeshIndex);
    }

    pRenderModelObj->Draw(pCommandBuffer, DrawPassType_Model, ViewType_Default, uniformBufferIndex);
    pRenderModelObj->SetDrawSingleShapeDisabled();

    if (g_MenuFlag.Test<MenuFlag::SingleSubmesh>())
    {
        pRenderModelObj->GetRenderUnitObj(shapeIndex)->SetDrawSingleSubmeshDisabled();
    }

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// 最終出力を描画
void DrawFinalImage(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    // 指定のLODレベルでの強制表示を設定
    int modelIndex = g_SelectedModelIndex;
    int forceLodLevel = GetForceLodLevel();

    NN_ASSERT_RANGE(modelIndex, 0, g_Holder.modelAnimObjs.GetCount());
    nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[modelIndex];

    if (forceLodLevel == g_InvalidForceLodLevel)
    {
        pRenderModelObj->SetDrawLodDisabled();
    }
    else
    {
        pRenderModelObj->SetDrawLodEnabled(forceLodLevel);
    }

    // レンダーターゲットを設定
    nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
    nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
    pCommandBuffer->SetRenderTargets(1, &pColor, pDepth);
    pCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

    // 選択されたオブジェクトだけ描画
    if (g_MenuFlag.Test<MenuFlag::SingleShape>())
    {
        DrawSelectedObj(pCommandBuffer, uniformBufferIndex);
        return;
    }

    // 不透明オブジェクトの描画
    DrawOpaque(pCommandBuffer, uniformBufferIndex);

    // カラーバッファーへコピー
    CopyColorBuffer(pGfxFramework, pCommandBuffer);

    // 半透明オブジェクトの描画
    DrawTranslucent(pCommandBuffer, uniformBufferIndex);
}

// バウンディング球を描画
void DrawBoundingSphere(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(g_SelectedModelIndex, 0, g_Holder.renderModelObjs.GetCount());

    nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[g_SelectedModelIndex];
    g_BoundingRenderer.DrawSphere(pRenderModelObj->GetModelObj(), pCommandBuffer, GetForceLodLevel());
}

// バウンディングボックスを描画
void DrawBoundingBox(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(g_SelectedModelIndex, 0, g_Holder.renderModelObjs.GetCount());

    nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[g_SelectedModelIndex];
    nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();

    int forceLodLevel = GetForceLodLevel();
    int shapeIndex = g_SelectedShapeIndex;
    if (g_MenuFlag.Test<MenuFlag::SingleSubmesh>())
    {
        g_BoundingRenderer.DrawBox(pModelObj, pCommandBuffer, shapeIndex, forceLodLevel, g_SelectedSubmeshIndex);
        return;
    }

    g_BoundingRenderer.DrawBox(pModelObj, pCommandBuffer, shapeIndex, forceLodLevel);
}

// バウンディングを描画
void DrawBounding(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    nn::util::Matrix4x3fType viewMtx = g_RenderView.GetViewMtx(ViewType_Default);
    nn::util::Matrix4x4fType projMtx = g_RenderView.GetProjectionMtx(ViewType_Default);
    g_BoundingRenderer.Update(pCommandBuffer, &viewMtx, &projMtx);

    if (g_MenuFlag.Test<MenuFlag::BoundingSphere>())
    {
        DrawBoundingSphere(pCommandBuffer);
        return;
    }

    if (g_MenuFlag.Test<MenuFlag::BoundingBox>())
    {
        DrawBoundingBox(pCommandBuffer);
        return;
    }
}

// ビューボリュームを描画
void DrawViewVolume(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::DrawViewVolume>())
    {
        return;
    }

    if (g_SelectedViewType == ViewType_Default && !g_MenuFlag.Test<MenuFlag::FreezeViewVolume>())
    {
        return;
    }

    nn::util::Color4u8Type color;
    color.v[0] = 0;
    color.v[1] = 255;
    color.v[2] = 255;
    color.v[3] = 32;

    g_ViewVolumeRenderer.Draw(pCommandBuffer, g_InvViewProjectionMatrix, color);
}

// デバッグ用フレームバッファーの描画
void CalculateDebugFrameBufferInfo(nns::gfx::GraphicsFramework* pGfxFramework) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::DrawDebugFrameBuffer>())
    {
        return;
    }

    NN_ASSERT_RANGE(g_SelectedFrameBufferType, FrameBufferType_Shadow, FrameBufferType_Count);

    const nn::gfx::DescriptorSlot* descriptorSlots[FrameBufferType_Count] =
    {
      &g_DepthDescriptorSlot,
      &g_WaterDescriptorSlot,
      &g_GeometryColorDescriptorSlot,
      &g_LightDescriptorSlot
    };

    nn::util::Color4u8Type color = { { 255, 255, 255, 255 } };
    if (g_SelectedFrameBufferType != FrameBufferType_Shadow)
    {
        g_ScreenInfo.AddQuadTexture(
            pGfxFramework->GetDisplayWidth() * 0.5f,
            0.0f,
            pGfxFramework->GetDisplayWidth() * 0.5f,
            pGfxFramework->GetDisplayHeight() * 0.5f,
            color,
            *descriptorSlots[g_SelectedFrameBufferType],
            g_DebugFrameBufferSamplerDescriptorSlot
        );

        return;
    }

    for (int depthIndex = 0; depthIndex < TownContext::cascadeShadowCount; ++depthIndex)
    {
        float x = pGfxFramework->GetDisplayWidth() * 0.5f;
        float offsetRate = static_cast<float>(depthIndex) / static_cast<float>(TownContext::cascadeShadowCount);
        x += (pGfxFramework->GetDisplayWidth() * 0.5f) * offsetRate;

        float size = pGfxFramework->GetDisplayWidth() * 0.5f / static_cast<float>(TownContext::cascadeShadowCount);
        g_ScreenInfo.AddQuadTexture(
            x,
            0.0f,
            size,
            size,
            depthIndex,
            color,
            g_DepthDescriptorSlot,
            g_DebugFrameBufferSamplerDescriptorSlot
        );
    }
}

// 情報テキストと負荷メーターの描画
void DrawInfo(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::CommandBuffer* pCommandBuffer, int bufferIndex) NN_NOEXCEPT
{
    if (!IsMenuEnabled())
    {
        return;
    }

    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Cyan());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Info");

    nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
    nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
    pCommandBuffer->SetRenderTargets(1, &pColor, pDepth);
    pCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

    CalculateMenuInfo(g_ScreenInfo, g_TextureAverageShader.GetTextureAverage(bufferIndex));
    CalculateDebugFrameBufferInfo(pGfxFramework);
    g_ScreenInfo.Draw(pCommandBuffer);

    g_LoadMeter.SetScale(2);
    g_LoadMeter.Draw(pCommandBuffer);
    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// コンピュートシェーダー実行
void CalculateTextureCompute(nn::gfx::CommandBuffer* pCommandBuffer, int bufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::TextureCompute>())
    {
        return;
    }

    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "Calculate");

    g_TextureAverageShader.MakeCommand(pCommandBuffer, bufferIndex, g_ColorDescriptorSlot);

    NN_PERF_END_MEASURE_GPU(pCommandBuffer);
}

// 描画コマンドを作成
void MakeCommand(nns::gfx::GraphicsFramework* pGfxFramework, int uniformBufferIndex, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pUserData);

    // コマンド生成
    NN_PERF_BEGIN_MEASURE_NAME("Make Command");
    pGfxFramework->BeginFrame(uniformBufferIndex);
    {
        nn::gfx::CommandBuffer* pCommandBuffer = pGfxFramework->GetRootCommandBuffer(uniformBufferIndex);

        // ディスクリプタとシェーダーコードのGPUキャッシュを無効化
        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor);

        // テクスチャーキャッシュ無効化
        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);

        // 画面のクリア
        ClearScreen(pGfxFramework, pCommandBuffer);

        // シャドウの描画
        DrawShadow(pCommandBuffer, uniformBufferIndex);

        // 水面の描画
        DrawWater(pCommandBuffer, uniformBufferIndex);

        // Gバッファーの描画
        DrawGeometry(pCommandBuffer, uniformBufferIndex);

        // キャッシュをフラッシュ
        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer);

        // ライトの描画
        DrawLight(pCommandBuffer, uniformBufferIndex);

        // キャッシュをフラッシュ
        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_DepthStencil);

        // 最終画面描画
        DrawFinalImage(pGfxFramework, pCommandBuffer, uniformBufferIndex);

        // バウンディング描画
        DrawBounding(pCommandBuffer);

        // ViewVolume 描画
        DrawViewVolume(pCommandBuffer);

        // 情報テキストと負荷メーターの描画
        DrawInfo(pGfxFramework, pCommandBuffer, uniformBufferIndex);

        // GPU演算
        CalculateTextureCompute(pCommandBuffer, uniformBufferIndex);

        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    }
    pGfxFramework->EndFrame(uniformBufferIndex);
    NN_PERF_END_MEASURE();
}



} // anonymous namespace

// ワイヤーフレーム切り替え
void SetWireframe(nns::g3d::RenderModelObj* pRenderModelObj, int fillMode, int materialIndex) NN_NOEXCEPT
{
    nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
    const nn::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndex);
    const nn::g3d::ResMaterial* pResMaterial = pMaterialObj->GetResource();
    nns::g3d::RenderModel* pRenderModel = pRenderModelObj->GetRenderModel();
    nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(materialIndex);

    // マテリアルにラスタライザーステートを設定
    const nn::g3d::ResRenderInfo* pRenderInfo = pResMaterial->FindRenderInfo("culling");
    if (pRenderInfo)
    {
        const nn::gfx::RasterizerState* pRasterizerState;
        NN_ASSERT_GREATER(pRenderInfo->GetArrayLength(), 0);

        g3ddemo::CullMode cullMode = static_cast<g3ddemo::CullMode>(pRenderInfo->GetInt()[0]);
        NN_ASSERT_RANGE(cullMode, g3ddemo::CullMode_Back, g3ddemo::CullMode_Max);

        pRasterizerState = g3ddemo::GetRasterizerState(cullMode, fillMode);
        pRenderMaterial->SetRasterizerState(pRasterizerState);
    }
}

// レンダーステートを設定
void SetupRenderState(nns::g3d::RenderModel* pRenderModel) NN_NOEXCEPT
{
    for (int passIndex = 0, passCount = pRenderModel->GetPassCount(); passIndex < passCount; ++passIndex)
    {
        for (int materialIndex = 0, materialCount = pRenderModel->GetResModel()->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
        {
            const nn::g3d::ResMaterial* pResMaterial = pRenderModel->GetResModel()->GetMaterial(materialIndex);
            nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(passIndex, materialIndex);

            // ブレンドステートを設定
            const nn::g3d::ResRenderInfo* pRenderInfo = pResMaterial->FindRenderInfo("blend");
            if (pRenderInfo)
            {
                int blendType = *pRenderInfo->GetInt();
                NN_ASSERT_RANGE(blendType, g3ddemo::BlendMode_Opaque, g3ddemo::BlendMode_Max);
                const nn::gfx::BlendState* pBlendState = g3ddemo::GetBlendState(blendType);
                pRenderMaterial->SetBlendState(pBlendState);
            }

            // ラスタライザーステートを設定
            if (passIndex == DrawPassType_Shadow)
            {
                const nn::gfx::RasterizerState* pRasterizerState = g3ddemo::GetRasterizerState(g3ddemo::CullMode_Front);
                pRenderMaterial->SetRasterizerState(pRasterizerState);
                continue;
            }

            pRenderInfo = pResMaterial->FindRenderInfo("culling");
            if (pRenderInfo)
            {
                int cullType = *pRenderInfo->GetInt();
                NN_ASSERT_RANGE(cullType, g3ddemo::CullMode_Back, g3ddemo::CullMode_Max);
                const nn::gfx::RasterizerState* pRasterizerState = g3ddemo::GetRasterizerState(cullType);
                pRenderMaterial->SetRasterizerState(pRasterizerState);
            }

            // デプスステンシルステートは変更しない
        }
    }
}

// ユニフォームブロックの設定を行います。
void BindUniformBlockAndSampler(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    // envモデルのユニフォームブロックとサンプラーをバインド
    {
        NN_ASSERT_NOT_NULL(g_pEnvModelObj);
        const nn::g3d::MaterialObj* pEnvMaterialObj = g_pEnvModelObj->GetMaterial(0);

        // ユニフォームを設定
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = pEnvMaterialObj->GetMaterialBlock(0);
        buffers[1] = pEnvMaterialObj->GetMaterialBlock(1);
        pRenderModelObj->BindUniformBlock("env", buffers, g_BufferingCount);

        // サンプラーを設定
        const nn::g3d::ResMaterial* pResMaterial = pEnvMaterialObj->GetResource();
        for (int samplerIndex = 0, samplerCount = pResMaterial->GetSamplerCount(); samplerIndex < samplerCount; ++samplerIndex)
        {
            const char* pSamplerName = pResMaterial->GetSamplerName(samplerIndex);
            nn::g3d::SamplerRef samplerRef = pEnvMaterialObj->GetSampler(samplerIndex);
            nn::g3d::TextureRef textureRef = pEnvMaterialObj->GetTexture(samplerIndex);
            pRenderModelObj->BindSampler(pSamplerName, textureRef.GetDescriptorSlot(), samplerRef.GetDescriptorSlot());
        }
    }

    // コンテキストのユニフォームブロックを設定
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_Context.GetContextBlock(0);
        buffers[1] = g_Context.GetContextBlock(1);
        pRenderModelObj->BindUniformBlock("context", buffers, g_BufferingCount);
    }

    // ビューのユニフォームブロックを設定
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(ViewType_Default, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(ViewType_Default, 1);
        pRenderModelObj->BindUniformBlock("view", buffers, g_BufferingCount);
    }

    // フラスタムカリング用のビューボリュームを設定
    pRenderModelObj->SetViewVolume(ViewType_Default, &g_RenderView.GetViewVolume(ViewType_Default));
}

// RenderModelObj にユニフォームブロックや LOD に関する設定を行います。
void SetupRenderModelObj(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    // LODレベルの判定に使用される関数を初期化
    static const float lodThresholds[] = { 1.0f, 0.4f, 0.2f, 0.1f };
    g_CalculateLodDefault = CalculateLod(&g_RenderView.GetViewMtx(ViewType_Default),
        1.0f / nn::util::TanEst(nn::util::DegreeToRadian(45.0f) * 0.5f), lodThresholds,
        sizeof(lodThresholds) / sizeof(float));

    static const float lodThresholdsWater[] = { 1.0f, 0.8f, 0.4f, 0.2f };
    g_CalculateLodWater = CalculateLod(&g_RenderView.GetViewMtx(ViewType_Water),
        1.0f / nn::util::TanEst(nn::util::DegreeToRadian(45.0f) * 0.5f), lodThresholdsWater,
        sizeof(lodThresholdsWater) / sizeof(float));

    // LOD レベル判定関数の設定
    {
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Default, &g_CalculateLodDefault);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Water, &g_CalculateLodWater);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Light0, &g_CalculateLodDefault);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Light1, &g_CalculateLodDefault);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Light2, &g_CalculateLodDefault);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Light3, &g_CalculateLodDefault);
    }

    // 複数パス共通の設定
    BindUniformBlockAndSampler(pRenderModelObj);

    // 水面描画パスの設定
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(ViewType_Water, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(ViewType_Water, 1);

        pRenderModelObj->BindUniformBlock(DrawPassType_Water, "view", buffers, g_BufferingCount);
        SetupWaterVariation(pRenderModelObj);

        // フラスタムカリング用のビューボリュームを設定
        pRenderModelObj->SetViewVolume(ViewType_Water, &g_RenderView.GetViewVolume(ViewType_Water));
    }
}

// RenderModel の初期化
void CreateRenderModel(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    nn::g3d::ResShaderArchive* pModelResShaderArchive = pResFile->GetUserPtr<nn::g3d::ResShaderArchive>();
    NN_ASSERT_NOT_NULL(pModelResShaderArchive);

    for (int modelIndex = 0, modelCount = pResFile->GetModelCount(); modelIndex < modelCount; ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);
        nns::g3d::RenderModel* pRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel, DrawPassType_Count);
        pRenderModel->CreateDrawPass(pDevice, DrawPassType_Model, pModelResShaderArchive);
        pRenderModel->CreateDrawPass(pDevice, DrawPassType_Water, pModelResShaderArchive);
        pRenderModel->CreateDrawPass(pDevice, DrawPassType_Geometry, g_pResShadingModelPtr[ShadingModelType_Geometry]);
        pRenderModel->CreateDrawPass(pDevice, DrawPassType_Shadow, g_pResShadingModelPtr[ShadingModelType_Shadow]);
        SetupRenderState(pRenderModel);
        pHolder->renderModels.PushBack(pRenderModel);
    }
}

// ModelAnimObj の初期化
void CreateModelAnimObj(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile, CreateModelInstanceCallback pCallback) NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = pResFile->GetModelCount(); modelIndex < modelCount; ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);

        // ModelAnimObj の初期化
        nns::g3d::ModelAnimObj* pModelAnimObj;
        {
            nn::g3d::ModelObj::Builder builder(pResModel);
            builder.ViewCount(ViewType_Count);
            builder.ShapeBufferingCount(g_BufferingCount);
            builder.SkeletonBufferingCount(g_BufferingCount);
            builder.MaterialBufferingCount(g_BufferingCount);
            builder.SetBoundingEnabled();
            builder.ShapeUserAreaSize(sizeof(int) * g_MaxLodLevel);
            pModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);
        }
        NN_ASSERT_NOT_NULL(pModelAnimObj);
        pHolder->modelAnimObjs.PushBack(pModelAnimObj);

        // LOD 情報を設定
        SetupShapeUserArea(pModelAnimObj->GetModelObj());

        // アニメーションIDの記録用バッファーを作成
        {
            size_t bufferSize = sizeof(int) * AnimationType_Count;
            int* pAnimId = g3ddemo::AllocateMemory<int>(bufferSize);
            for (int animationIndex = 0; animationIndex < AnimationType_Count; ++animationIndex)
            {
                pAnimId[animationIndex] = nns::g3d::ModelAnimObj::InvalidAnimId;
            }
            pHolder->animationIds.PushBack(pAnimId);
        }

        // コールバック関数が設定されていた場合、呼び出します。
        if (pCallback)
        {
            pCallback(pModelAnimObj->GetModelObj());
        }
    }
}

// RenderModelObj の初期化
void CreateRenderModelObj(g3ddemo::ResourceHolder* pHolder, nns::g3d::ModelAnimObj* pModelAnimObj, nns::g3d::RenderModel* pRenderModel) NN_NOEXCEPT
{
    NN_ASSERT(pModelAnimObj->GetModelObj()->GetResource() == pRenderModel->GetResModel());

    nns::g3d::RenderModelObj* pRenderModelObj = nns::g3d::CreateRenderModelObj(pModelAnimObj->GetModelObj(), pRenderModel);
    g3ddemo::WriteSkinningOption(pRenderModelObj);

    pRenderModelObj->BindDummySampler(g3ddemo::GetDummyTextureRef().GetDescriptorSlot(), g3ddemo::GetDummySamplerDescriptorSlot());
    pHolder->renderModelObjs.PushBack(pRenderModelObj);
}

int TownMain(bool isUseRankTurn) NN_NOEXCEPT
{
    nn::os::Tick beginTick = nn::os::GetSystemTick();
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_DeferredSubmission);
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // Townの初期化
    InitializeTown(pGfxFramework);

    g3ddemo::Pad& pad = g3ddemo::GetPad();
    pad.Reset();

    int uniformBufferIndex = 0;
    pGfxFramework->SetCalculateCallback(Calculate, &uniformBufferIndex);
    pGfxFramework->SetMakeCommandCallback(MakeCommand, &uniformBufferIndex);
    pGfxFramework->ResetFrame();
    g_FrameCount = 0;

    // 結果出力ファイルの初期化
    nnt::graphics::PerformanceProfileData  profile;
    profile.Initialize(1024, g3ddemo::AllocateMemory, g3ddemo::FreeMemory);
    profile.SetName("Town");
    nn::time::PosixTime posixTime;
    nn::time::CalendarTime  calendarTime;
    nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    nn::time::ToCalendarTime(&calendarTime, NULL, posixTime);
    char timeString[32];
    nn::util::SNPrintf(timeString, 32,
        "%04d-%02d-%02d %02d:%02d:%02d",
        calendarTime.year, calendarTime.month, calendarTime.day,
        calendarTime.hour, calendarTime.minute, calendarTime.second);
    profile.SetDate(timeString);

    // 5分間計測する
    int profiledFrameCount = 0;
    int profileFrameCount = 60 * 15;
    if (isUseRankTurn)
    {
        profileFrameCount = 60 * 240;
    }
    nn::os::Tick endTick = nn::os::GetSystemTick();
    NN_LOG("InitialTownCost:%f(ms)\n", (endTick.ToTimeSpan().GetMicroSeconds() - beginTick.ToTimeSpan().GetMicroSeconds()) / 1000.f);

    // メインループ
    while (NN_STATIC_CONDITION(1))
    {
        //if (GetMenuMode() != MenuType_Debug1)
        //{
        //    NN_PERF_SET_ENABLED(false);
        //}
        //else
        //{
            NN_PERF_SET_ENABLED(true);
        //}

        NN_PERF_BEGIN_FRAME();

        // タッチ長押しで終了
        g3ddemo::GetControllerManager()->Update();
        if (g3ddemo::isExitDemo())
        {
            break;
        }

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            break;
        }

        // フレーム処理実行
        pGfxFramework->ProcessFrame();


        // システムコアが落ち着くまで開始後 5 秒間は計測しない
        if (g_FrameCount > 300)
        {
            NN_PERF_END_FRAME_NO_CLEAR_PROFILE();
            profiledFrameCount++;
            // profileFrameCount 分計測したら計測を終了する
            if (profiledFrameCount == profileFrameCount)
            {
                break;
            }
        }
        else
        {
            NN_PERF_END_FRAME();
        }
        uniformBufferIndex ^= 1;
        ++g_FrameCount;
    }
    pGfxFramework->QueueFinish();

    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);

    uint64_t cpuElapsedMicroSeconds = NN_PERF_GET_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetMicroSeconds();
    uint64_t gpuElapsedMicroSeconds = NN_PERF_GET_ELAPSED_TIME_GPU(NULL, 0).GetMicroSeconds();

    float cpuAverage = cpuElapsedMicroSeconds / static_cast<float>(profiledFrameCount);
    float gpuAverage = gpuElapsedMicroSeconds / static_cast<float>(profiledFrameCount);
    NN_LOG("%f %f\n", cpuAverage, gpuAverage);
    // 結果をファイルに出力
    profile.SetCpuValue(0, static_cast<int64_t>(cpuAverage));
    profile.SetGpuValue(0, static_cast<int64_t>(gpuAverage));

    nn::util::SNPrintf(timeString, 32,
        "%04d%02d%02d_%02d%02d%02d",
        calendarTime.year, calendarTime.month, calendarTime.day,
        calendarTime.hour, calendarTime.minute, calendarTime.second);
    char    filePath[1024];
    nn::util::SNPrintf(filePath, 1024, "Outputs:/Outputs/TownVideo_%s.json", timeString);
    if (!isUseRankTurn)
    {
        profile.Write(filePath);
    }
    profile.Finalize();

    // Townの解放処理
    FinalizeTown(pGfxFramework, pDevice);

    return EXIT_SUCCESS;
}
