﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/gfx.h>
#include <nn/g3d.h>
#include <nn/fs.h>
#include <nn/hid.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/util/util_Arithmetic.h>
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif
#include <nns/hid.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/settings/settings_DebugPad.h>
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#endif

namespace nn { namespace g3d { namespace demo {

inline bool IsPowerOfTwo(size_t size) NN_NOEXCEPT
{
    return ((size - 1) & size) == 0 && size != 0;
}

inline void* AddOffset(void* ptr, ptrdiff_t offset) NN_NOEXCEPT
{
    return reinterpret_cast<char*>(ptr) + offset;
}

inline const void* AddOffset(const void* ptr, ptrdiff_t offset) NN_NOEXCEPT
{
    return reinterpret_cast<const char*>(ptr) + offset;
}

inline void MatrixGetColumn0(nn::util::Vector3fType* pDstVector, const nn::util::Matrix4x3fType& srcMtx) NN_NOEXCEPT
{
    nn::util::Float4x3 mtx;
    MatrixStore(&mtx, srcMtx);
    VectorSet(pDstVector, mtx.m00, mtx.m10, mtx.m20);
}

inline void MatrixGetColumn1(nn::util::Vector3fType* pDstVector, const nn::util::Matrix4x3fType& srcMtx) NN_NOEXCEPT
{
    nn::util::Float4x3 mtx;
    MatrixStore(&mtx, srcMtx);
    VectorSet(pDstVector, mtx.m01, mtx.m11, mtx.m21);
}

inline void MatrixGetColumn2(nn::util::Vector3fType* pDstVector, const nn::util::Matrix4x3fType& srcMtx) NN_NOEXCEPT
{
    nn::util::Float4x3 mtx;
    MatrixStore(&mtx, srcMtx);
    VectorSet(pDstVector, mtx.m02, mtx.m12, mtx.m22);
}

inline void MatrixSetColumn0(nn::util::Matrix4x3fType* pDstMtx, const nn::util::Vector3fType& srcVector) NN_NOEXCEPT
{
    nn::util::Float4x3 mtx;
    MatrixStore(&mtx, *pDstMtx);
    mtx.m00 = VectorGetX(srcVector);
    mtx.m10 = VectorGetY(srcVector);
    mtx.m20 = VectorGetZ(srcVector);
    MatrixLoad(pDstMtx, mtx);
}

inline void MatrixSetColumn1(nn::util::Matrix4x3fType* pDstMtx, const nn::util::Vector3fType& srcVector) NN_NOEXCEPT
{
    nn::util::Float4x3 mtx;
    mtx.m01 = VectorGetX(srcVector);
    mtx.m11 = VectorGetY(srcVector);
    mtx.m21 = VectorGetZ(srcVector);
    MatrixLoad(pDstMtx, mtx);
}

inline void MatrixSetColumn2(nn::util::Matrix4x3fType* pDstMtx, const nn::util::Vector3fType& srcVector) NN_NOEXCEPT
{
    nn::util::Float4x3 mtx;
    mtx.m02 = VectorGetX(srcVector);
    mtx.m12 = VectorGetY(srcVector);
    mtx.m22 = VectorGetZ(srcVector);
    MatrixLoad(pDstMtx, mtx);
}

void Initialize() NN_NOEXCEPT;
void Finalize() NN_NOEXCEPT;
void InitializeDemo() NN_NOEXCEPT;
void FinalizeDemo() NN_NOEXCEPT;

//--------------------------------------------------------------------------------------------------
// Memory

enum
{
    DEFAULT_ALIGNMENT = 8
};

void* AllocateMemory(size_t size, size_t alignment = DEFAULT_ALIGNMENT) NN_NOEXCEPT;

template <typename T>
NN_FORCEINLINE
T* AllocateMemory(size_t size, size_t alignment) NN_NOEXCEPT
{
    return reinterpret_cast<T*>(AllocateMemory(size, alignment));
}

template <typename T>
NN_FORCEINLINE
T* AllocateMemory(size_t size) NN_NOEXCEPT
{
    return AllocateMemory<T>(size, DEFAULT_ALIGNMENT);
}

template <typename T>
NN_FORCEINLINE
T* AllocateMemory(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pUserData);
    return AllocateMemory<T>(size, alignment);
}

void FreeMemory(void* ptr) NN_NOEXCEPT;

NN_FORCEINLINE
void FreeMemory(void* ptr, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    FreeMemory(ptr);
}

NN_FORCEINLINE
void FreeMemory(void* ptr, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pUserData);
    FreeMemory(ptr);
}

nn::mem::StandardAllocator* GetAllocator() NN_NOEXCEPT;

//--------------------------------------------------------------------------------------------------
// Pad

class Pad
{
public:
    enum
    {
        BUTTON_LEFT             = 0x00000001,
        BUTTON_RIGHT            = 0x00000002,
        BUTTON_DOWN             = 0x00000004,
        BUTTON_UP               = 0x00000008,
        TRIGGER_Z               = 0x00000010,
        TRIGGER_R               = 0x00000020,
        TRIGGER_L               = 0x00000040,
        BUTTON_A                = 0x00000100,
        BUTTON_B                = 0x00000200,
        BUTTON_X                = 0x00000400,
        BUTTON_Y                = 0x00000800,
        BUTTON_START            = 0x00001000,
        STICK_R_LEFT            = 0x00010000,
        STICK_R_RIGHT           = 0x00020000,
        STICK_R_DOWN            = 0x00040000,
        STICK_R_UP              = 0x00080000,
        STICK_L_LEFT            = 0x00100000,
        STICK_L_RIGHT           = 0x00200000,
        STICK_L_DOWN            = 0x00400000,
        STICK_L_UP              = 0x00800000,
        BUTTON_STICK_L_LEFT     = BUTTON_LEFT | STICK_L_LEFT,
        BUTTON_STICK_L_RIGHT    = BUTTON_RIGHT | STICK_L_RIGHT,
        BUTTON_STICK_L_DOWN     = BUTTON_DOWN | STICK_L_DOWN,
        BUTTON_STICK_L_UP       = BUTTON_UP | STICK_L_UP
    };

    struct Stick
    {
        int8_t leftX, leftY, rightX, rightY;
    };
    struct AnalogStick
    {
        float leftX, leftY, rightX, rightY;
    };

    Pad() NN_NOEXCEPT
    : m_InitNpad(false)
    {}

    void Initialize() NN_NOEXCEPT;
    void InitializeNpad() NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;
    bool Read() NN_NOEXCEPT;

    bool IsHold(Bit32 mask) const NN_NOEXCEPT
    {
        return 0 != (m_Button & mask);
    }
    bool IsTriggered(Bit32 mask) const NN_NOEXCEPT
    {
        return 0 != (m_Triggered & mask);
    }
    bool IsReleased(Bit32 mask) const NN_NOEXCEPT
    {
        return 0 != (m_Released & mask);
    }
    bool IsHoldAll(Bit32 mask) const NN_NOEXCEPT
    {
        return mask == (m_Button & mask);
    }
    bool IsHoldAllAndTriggered(Bit32 mask) const NN_NOEXCEPT
    {
        return IsHoldAll(mask) && IsTriggered(mask);
    }

    Bit32 GetButton() const NN_NOEXCEPT
    {
        return m_Button;
    }
    Bit32 GetLastButton() const NN_NOEXCEPT
    {
        return m_LastButton;
    }
    const Stick& GetStick() const NN_NOEXCEPT
    {
        return m_Stick;
    }
    const Stick& GetLastStick() const NN_NOEXCEPT
    {
        return m_LastStick;
    }
    const AnalogStick& GetAnalogStick() const NN_NOEXCEPT
    {
        return m_AnalogStick;
    }

private:
    enum NpadId
    {
        NPADID_NUM = 2,
    };

    enum NpadState
    {
        NpadState_Buttons = (1 << 0),
        NpadState_AnalogStickL = (1 << 1),
        NpadState_AnalogStickR = (1 << 2),
    };

    nn::hid::NpadIdType m_NpadIds[NPADID_NUM];
    bool m_InitNpad;
    void ReadNpad(Bit32* pButton, Stick* pStick) NN_NOEXCEPT;
    Bit32 GetNpadState(nn::hid::NpadButtonSet* pNpadButton,
                      nn::hid::AnalogStickState* pAnalogStickL,
                      nn::hid::AnalogStickState* pAnalogStickR) NN_NOEXCEPT;

private:
    Bit32 m_Button;
    Bit32 m_LastButton;
    Stick m_Stick;
    Stick m_LastStick;
    AnalogStick m_AnalogStick;
    Bit32 m_Triggered;
    Bit32 m_Released;
    nn::settings::DebugPadKeyboardMap m_DebugPadKeyboardMap;

    NN_DISALLOW_COPY(Pad);
};

Pad& GetPad() NN_NOEXCEPT;

//--------------------------------------------------------------------------------------------------
// CameraController

void ControlCamera(
    nn::util::Vector3fType* pCameraPos,
    nn::util::Vector3fType* pCameraUp,
    nn::util::Vector3fType* pCameraTarget,
    const nn::util::Matrix4x3fType* pViewMtx,
    const Pad* pPad) NN_NOEXCEPT;

void LookAtModel(nn::util::Vector3fType* pCameraPos, nn::util::Vector3fType* pCameraTarget, const nn::g3d::ModelObj* pTarget) NN_NOEXCEPT;

void LookAtShape(nn::util::Vector3fType* pCameraPos, nn::util::Vector3fType* pCameraTarget, const nn::g3d::ShapeObj* pTarget) NN_NOEXCEPT;

void LookAtSubMesh(
    nn::util::Vector3fType* pCameraPos,
    nn::util::Vector3fType* pCameraTarget,
    const nn::g3d::ShapeObj* pTarget,
    int meshIndex,
    int submeshIndex) NN_NOEXCEPT;

//--------------------------------------------------------------------------------------------------
// File

typedef nn::fs::FileHandle FileHandle;

int OpenFile(FileHandle* pHandle, const char* path, const char* mode) NN_NOEXCEPT;
int CloseFile(FileHandle handle) NN_NOEXCEPT;
int GetFileSize(FileHandle handle, size_t* pSize) NN_NOEXCEPT;
int ReadFile(FileHandle handle, void* pDst, size_t length) NN_NOEXCEPT;
int WriteFile(FileHandle handle, const void* pSrc, size_t length) NN_NOEXCEPT;

void* LoadFile(const char* path, size_t* pSize = nullptr, size_t alignment = DEFAULT_ALIGNMENT) NN_NOEXCEPT;
void SaveFile(const char* path, const void* pData, size_t size) NN_NOEXCEPT;
size_t GetFileAlignment(const char* path) NN_NOEXCEPT;

template<typename T>

T* LoadResource(const char* path) NN_NOEXCEPT
{
    // ファイルのアライメントを取得。
    size_t alignment = GetFileAlignment(path);
    void* pFile = LoadFile(path, nullptr, alignment);
    NN_ASSERT(T::IsValid(pFile));
    // ロードしたファイルをリソースに変換。
    T* pResFile = T::ResCast(pFile);
    return pResFile;
}

//--------------------------------------------------------------------------------------------------
// Container

template <typename T>
class Vector
{
public:
    Vector()  NN_NOEXCEPT : m_pBuffer(nullptr), m_Size(0), m_Capacity(0)
    {
    }
    Vector(void* pBuffer, size_t size) NN_NOEXCEPT
    {
        SetBuffer(pBuffer, size);
    }

    bool Empty() const NN_NOEXCEPT
    {
        return m_Size == 0;
    }
    int GetCount() const NN_NOEXCEPT
    {
        return static_cast<int>(m_Size);
    }
    int Capacity() const NN_NOEXCEPT
    {
        return static_cast<int>(m_Capacity);
    }

    void SetBuffer(void* pBuffer, size_t size) NN_NOEXCEPT
    {
        NN_ASSERT(pBuffer != nullptr);
        NN_ASSERT(size > 0);
        Clear();
        m_pBuffer = reinterpret_cast<T*>(pBuffer);
        m_Size = 0;
        m_Capacity = size / sizeof(T);
    }
    void* GetBuffer() NN_NOEXCEPT
    {
        return m_pBuffer;
    }

    void ResetBuffer() NN_NOEXCEPT
    {
        Clear();
        m_pBuffer = nullptr;
    }

    T& operator[](int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, static_cast<int>(m_Size));
        return m_pBuffer[index];
    }

    const T& operator[](int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, static_cast<int>(m_Size));
        return m_pBuffer[index];
    }

    void Clear() NN_NOEXCEPT
    {
        for (int idx = 0, num = static_cast<int>(m_Size); idx < num; ++idx)
        {
            m_pBuffer[idx].~T();
        }
        m_Size = 0;
    }

    T& Front() NN_NOEXCEPT
    {
        NN_ASSERT(!Empty());
        return m_pBuffer[0];
    }

    const T& Front() const NN_NOEXCEPT
    {
        NN_ASSERT(!Empty());
        return m_pBuffer[0];
    }

    T& Back() NN_NOEXCEPT
    {
        NN_ASSERT(!Empty());
        return m_pBuffer[m_Size - 1];
    }

    const T& Back() const NN_NOEXCEPT
    {
        NN_ASSERT(!Empty());
        return m_pBuffer[m_Size - 1];
    }

    void PushBack(const T& x) NN_NOEXCEPT
    {
        NN_ASSERT(m_Size < m_Capacity);
        new (&m_pBuffer[m_Size++]) T(x);
    }

    void PopBack() NN_NOEXCEPT
    {
        NN_ASSERT(m_Size > 0);
        m_pBuffer[--m_Size].~T();
    }

    void Insert(int index, const T& x) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, static_cast<int>(m_Size + 1));
        NN_ASSERT(m_Size < m_Capacity);
        for (int rindex = static_cast<int>(m_Size); rindex > index; --rindex)
        {
            new(&m_pBuffer[rindex]) T(m_pBuffer[rindex - 1]);
            m_pBuffer[rindex - 1].~T();
        }
        new(&m_pBuffer[index]) T(x);
        ++m_Size;
    }

    void Erase(int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, static_cast<int>(m_Size));
        m_pBuffer[index].~T();
        for (int findex = index; findex < static_cast<int>(m_Size - 1); ++findex)
        {
            new(&m_pBuffer[findex]) T(m_pBuffer[findex + 1]);
            m_pBuffer[findex + 1].~T();
        }
        --m_Size;
    }

    int FindIndex(T elem) const NN_NOEXCEPT
    {
        for (int elemIndex = 0; elemIndex < static_cast<int>(m_Size); ++elemIndex)
        {
            if (elem == m_pBuffer[elemIndex])
            {
                return elemIndex;
            }
        }

        return -1;
    }

    void Swap(Vector<T>& other) NN_NOEXCEPT
    {
        std::swap(m_pBuffer, other.m_pBuffer);
        std::swap(m_Size, other.m_Size);
        std::swap(m_Capacity, other.m_Capacity);
    }

    T* begin() NN_NOEXCEPT
    {
        return m_pBuffer;
    }

    const T* begin() const NN_NOEXCEPT
    {
        return m_pBuffer;
    }

    T* end() NN_NOEXCEPT
    {
        return &m_pBuffer[m_Size];
    }

    const T* end() const NN_NOEXCEPT
    {
        return &m_pBuffer[m_Size];
    }

private:
    T* m_pBuffer;
    size_t m_Size;
    size_t m_Capacity;

    NN_DISALLOW_COPY(Vector);
};

template<typename T>
NN_FORCEINLINE
void CreateVector(
    Vector<T>* pOutList, int count) NN_NOEXCEPT
{
    size_t bufferSize = sizeof(T) * count;
    void* buffer = AllocateMemory(bufferSize, DEFAULT_ALIGNMENT);
    pOutList->SetBuffer(buffer, bufferSize);
}

template<typename T>
NN_FORCEINLINE
void CopyVector(
    Vector<T>* pOutList,
    const Vector<T>& inputList) NN_NOEXCEPT
{
    pOutList->Clear();
    for (int index = 0, count = inputList.GetCount(); index < count; ++index)
    {
        T pInput = inputList[index];
        pOutList->PushBack(pInput);
    }
}

template<typename T>
NN_FORCEINLINE
void DestroyVector(Vector<T>* pInOutList) NN_NOEXCEPT
{
    FreeMemory(pInOutList->GetBuffer());
}

//--------------------------------------------------------------------------------------------------
// Menu

class CmdMenu
{
public:
    struct Item
    {
        const char* title;
        size_t len;
        int (*pFunc)();
    };

    explicit CmdMenu(const Item* pItems) NN_NOEXCEPT
        : m_NumItem(0), m_SelectMenu(0), m_pItems(pItems)
    {
        if (pItems)
        {
            for (const Item* pItem = pItems; pItem->pFunc; ++pItem)
            {
                ++m_NumItem;
            }
        }
    }

    void PrintMenu() const NN_NOEXCEPT;
    int Run(int item) const NN_NOEXCEPT;
    int Select() NN_NOEXCEPT;

    int Loop() NN_NOEXCEPT;

private:
    int m_NumItem;
    int m_SelectMenu;
    const Item* m_pItems;
};

bool isExitDemo() NN_NOEXCEPT;
nns::hid::ControllerManager* GetControllerManager() NN_NOEXCEPT;

}}} // namespace nn::g3d::demo

