﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testG3d_Main.h"
#include <nn/fs.h>
#include <nn/os.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
#include <nv/nv_MemoryManagement.h>
#endif

#include <nnt/base/testBase_Exit.h>
#include "nnt/g3d/testG3d_TestUtility.h"

namespace
{
nn::gfx::Device g_Device;

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif
}

float GetAllowableError(float value, int significantDigits) NN_NOEXCEPT
{
    float absValue = abs(value);
    float maxDigitLog;
    if (absValue == 0.0f)
    {
        maxDigitLog = static_cast<float>(-(significantDigits - 1));
    }
    else
    {
        maxDigitLog = floor(log10(absValue)) - (significantDigits - 1);
    }
    return powf(10, maxDigitLog);
}

bool IsFloatEqual(float value1, float value2, int significantDigits) NN_NOEXCEPT
{
    // 誤差を許さない場合
    if (significantDigits == -1)
    {
        if (value1 == value2)
        {
            return true;
        }
        return false;
    }
    else
    {
        float allowableError = GetAllowableError(value1, significantDigits);
        if (abs(value1 - value2) <= allowableError)
        {
            return true;
        }
        return false;
    }
}

bool IsVector3fEqual(const nn::util::Vector3fType& vector1, const nn::util::Vector3fType& vector2, int significantDigits) NN_NOEXCEPT
{
    if (IsFloatEqual(VectorGetX(vector1), VectorGetX(vector2), significantDigits))
    {
        if (IsFloatEqual(VectorGetY(vector1), VectorGetY(vector2), significantDigits))
        {
            if (IsFloatEqual(VectorGetZ(vector1), VectorGetZ(vector2), significantDigits))
            {
                return true;
            }
        }
    }

    return false;
}

bool IsVector4fEqual(const nn::util::Vector4fType& vector1, const nn::util::Vector4fType& vector2, int significantDigits) NN_NOEXCEPT
{
    if (IsFloatEqual(VectorGetX(vector1), VectorGetX(vector2), significantDigits))
    {
        if (IsFloatEqual(VectorGetY(vector1), VectorGetY(vector2), significantDigits))
        {
            if (IsFloatEqual(VectorGetZ(vector1), VectorGetZ(vector2), significantDigits))
            {
                if (IsFloatEqual(VectorGetW(vector1), VectorGetW(vector2), significantDigits))
                {
                    return true;
                }
            }
        }
    }

    return false;
}

bool IsMtx43Equal(const nn::util::Matrix4x3fType& mtx1, const nn::util::Matrix4x3fType& mtx2, int significantDigits) NN_NOEXCEPT
{
    nn::util::Vector3fType vector1;
    nn::util::Vector3fType vector2;

    MatrixGetAxisX(&vector1, mtx1);
    MatrixGetAxisX(&vector2, mtx2);
    if (IsVector3fEqual(vector1, vector2, significantDigits) == false)
    {
        return false;
    }
    MatrixGetAxisY(&vector1, mtx1);
    MatrixGetAxisY(&vector2, mtx2);
    if (IsVector3fEqual(vector1, vector2, significantDigits) == false)
    {
        return false;
    }
    MatrixGetAxisZ(&vector1, mtx1);
    MatrixGetAxisZ(&vector2, mtx2);
    if (IsVector3fEqual(vector1, vector2, significantDigits) == false)
    {
        return false;
    }
    MatrixGetAxisW(&vector1, mtx1);
MatrixGetAxisW(&vector2, mtx2);
if (IsVector3fEqual(vector1, vector2, significantDigits) == false)
{
    return false;
}

return true;
}

bool IsMtx44Equal(const nn::util::Matrix4x4fType& mtx1, const nn::util::Matrix4x4fType& mtx2, int significantDigits) NN_NOEXCEPT
{
    nn::util::FloatRowMajor4x4 floatMtx1;
    nn::util::FloatRowMajor4x4 floatMtx2;

    MatrixStore(&floatMtx1, mtx1);
    MatrixStore(&floatMtx2, mtx2);

    if (!IsFloatEqual(floatMtx1.m00, floatMtx2.m00, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m01, floatMtx2.m01, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m02, floatMtx2.m02, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m03, floatMtx2.m03, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m10, floatMtx2.m10, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m11, floatMtx2.m11, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m12, floatMtx2.m12, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m13, floatMtx2.m13, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m20, floatMtx2.m20, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m21, floatMtx2.m21, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m22, floatMtx2.m22, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m23, floatMtx2.m23, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m30, floatMtx2.m30, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m31, floatMtx2.m31, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m32, floatMtx2.m32, significantDigits)) return false;
    if (!IsFloatEqual(floatMtx1.m33, floatMtx2.m33, significantDigits)) return false;

    return true;
}

// ResFileのアライメントはBinaryFileHeaderから取得
size_t GetFileAlignment(const char* path)
{
    nn::fs::FileHandle fileHandle;

    nn::Result result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    void* buffer;
    buffer = nnt::g3d::AlignedAllocate<void>(sizeof(nn::util::BinaryFileHeader), 8);

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, buffer, sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT(result.IsSuccess());

    nn::fs::CloseFile(fileHandle);

    size_t alignment = reinterpret_cast<nn::util::BinaryFileHeader*>(buffer)->GetAlignment();
    nnt::g3d::Deallocate(buffer);

    return alignment;
}

size_t GetFileSize(const char* path)
{
    nn::fs::FileHandle fileHandle;

    nn::Result result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    void* buffer;
    buffer = nnt::g3d::AlignedAllocate<void>(sizeof(nn::util::BinaryFileHeader), 8);

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, buffer, sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT(result.IsSuccess());

    nn::fs::CloseFile(fileHandle);

    size_t size = reinterpret_cast<nn::util::BinaryFileHeader*>(buffer)->GetFileSize();
    nnt::g3d::Deallocate(buffer);

    return size;
}

TEST_F(G3dTest, BfresBasic)
{
    const char* bfresPath = "Resource:/Basic.bfres";
    const char* referenceBfresPath = "Resource:/ReferenceBasic.bfres";

    size_t size;
    void* pFile = nnt::g3d::LoadFile(&size, bfresPath, GetFileAlignment(bfresPath));
    void* pReferenceFile = nnt::g3d::LoadFile(&size, referenceBfresPath, GetFileAlignment(referenceBfresPath));

    if (memcmp(pFile, pReferenceFile, GetFileSize(bfresPath)) == 0)
    {
        SUCCEED();
    }
    else
    {
        nn::g3d::ResFile* pResFile = nn::g3d::ResFile::ResCast(pFile);
        pResFile->Setup(&g_Device);
        nn::g3d::ResFile*  pReferenceResFile = nn::g3d::ResFile::ResCast(pReferenceFile);
        pReferenceResFile->Setup(&g_Device);

        CheckResFileDataEqual(pResFile->ToData(), pReferenceResFile->ToData());

        pResFile->Cleanup(&g_Device);
        pReferenceResFile->Cleanup(&g_Device);
        FAIL();
    }
    nnt::g3d::UnloadFile(pFile);
    nnt::g3d::UnloadFile(pReferenceFile);
}

// --ignore-assign のテスト
TEST_F(G3dTest, BfresIgnoreAssign)
{
    const char* bfresPath = "Resource:/IgnoreAssign.bfres";
    const char* referenceBfresPath = "Resource:/ReferenceIgnoreAssign.bfres";

    size_t size;
    void* pFile = nnt::g3d::LoadFile(&size, bfresPath, GetFileAlignment(bfresPath));
    void* pReferenceFile = nnt::g3d::LoadFile(&size, referenceBfresPath, GetFileAlignment(referenceBfresPath));

    if (memcmp(pFile, pReferenceFile, GetFileSize(bfresPath)) == 0)
    {
        SUCCEED();
    }
    else
    {
        nn::g3d::ResFile* pResFile = nn::g3d::ResFile::ResCast(pFile);
        pResFile->Setup(&g_Device);
        nn::g3d::ResFile*  pReferenceResFile = nn::g3d::ResFile::ResCast(pReferenceFile);
        pReferenceResFile->Setup(&g_Device);

        CheckResFileDataEqual(pResFile->ToData(), pReferenceResFile->ToData());

        pResFile->Cleanup(&g_Device);
        pReferenceResFile->Cleanup(&g_Device);
        FAIL();
    }
    nnt::g3d::UnloadFile(pFile);
    nnt::g3d::UnloadFile(pReferenceFile);
}

// --disable-attribute-alignment-sort のテスト
TEST_F(G3dTest, BfresDisableAttributeSort)
{
    const char* bfresPath = "Resource:/DisableAttributeSort.bfres";
    const char* referenceBfresPath = "Resource:/ReferenceDisableAttributeSort.bfres";

    size_t size;
    void* pFile = nnt::g3d::LoadFile(&size, bfresPath, GetFileAlignment(bfresPath));
    void* pReferenceFile = nnt::g3d::LoadFile(&size, referenceBfresPath, GetFileAlignment(referenceBfresPath));

    if (memcmp(pFile, pReferenceFile, GetFileSize(bfresPath)) == 0)
    {
        SUCCEED();
    }
    else
    {
        nn::g3d::ResFile* pResFile = nn::g3d::ResFile::ResCast(pFile);
        pResFile->Setup(&g_Device);
        nn::g3d::ResFile*  pReferenceResFile = nn::g3d::ResFile::ResCast(pReferenceFile);
        pReferenceResFile->Setup(&g_Device);

        CheckResFileDataEqual(pResFile->ToData(), pReferenceResFile->ToData());

        pResFile->Cleanup(&g_Device);
        pReferenceResFile->Cleanup(&g_Device);
        FAIL();
    }
    nnt::g3d::UnloadFile(pFile);
    nnt::g3d::UnloadFile(pReferenceFile);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    void* pGraphicMemory = malloc(GraphicsSystemMemorySize);
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(pGraphicMemory, GraphicsSystemMemorySize);
#endif

    nnt::g3d::InitializeG3dTest(g_Device);

    int result = RUN_ALL_TESTS();

    nnt::g3d::FinalizeG3dTest(g_Device);
    nnt::Exit(result);
}
