﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "g3ddemo_ViewerUtility.h"
#include "TextureAverageShader.h"
#include "Town.h"
#include <nn/g3d/g3d_Viewer.h>
#include <cfloat>

#include "testG3d_CaptureSetting.h"
#include "testG3d_FrameCapture.h"

nnt::g3d::FrameCapture* GetFrameCapture();
void InitializeScreenCapture();
void GetScreenCaptureCamera(const int idxCamera, nn::util::Vector3fType* pCameraPosition, nn::util::Vector3fType* pCameraTarget);
void ScreenCapture(const int idxFrame);
void FinalizeScreenCapture();
uint64_t GetFrameCaptureCountMax();
int g_ScreenCaptureCount = 0;
nnt::g3d::CaptureSetting* g_pCaptureSetting = nullptr;

namespace
{
using namespace nnt::g3d;

MenuFlagSet g_MenuFlag;
int g_SelectedModelIndex = 0;
int g_SelectedShapeIndex = 0;
int g_SelectedSubmeshIndex = 0;

void InitializeMenu() NN_NOEXCEPT
{
    g_MenuFlag.Set<MenuFlag::TextureCompute>(true);
    g_MenuFlag.Set<MenuFlag::ModelAnimation>(true);
    g_MenuFlag.Set<MenuFlag::LightAnimation>(false);
    g_MenuFlag.Set<MenuFlag::ShadowRendering>(true);
    g_MenuFlag.Set<MenuFlag::WaterRendering>(true);
    g_MenuFlag.Set<MenuFlag::GeometryRendering>(true);
    g_MenuFlag.Set<MenuFlag::LightRendering>(true);
    g_MenuFlag.Set<MenuFlag::SingleShape>(false);
    g_MenuFlag.Set<MenuFlag::SingleSubmesh>(false);
    g_MenuFlag.Set<MenuFlag::BoundingSphere>(false);
    g_MenuFlag.Set<MenuFlag::BoundingBox>(false);
}

nn::gfx::Fence g_CaptureFence;

void InitializeCaptureFence(nn::gfx::Device* pDevice)
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();
    g_CaptureFence.Initialize(pDevice, info);
}

void FinalizeCaptureFence(nn::gfx::Device* pDevice)
{
    g_CaptureFence.Finalize(pDevice);
}

void MakeCaptureCommand(nns::gfx::GraphicsFramework* pGfxFramework, int bufferIndex)
{
    nn::gfx::CommandBuffer* pCommandBuffer = pGfxFramework->GetRootCommandBuffer(bufferIndex);

    // テクスチャーキャッシュ無効化
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor);
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);

    GetFrameCapture()->PushCaptureCommand(pCommandBuffer, pGfxFramework->GetScanBuffer(bufferIndex));
    pCommandBuffer->End();

    // キャプチャ完了待ちのフェンスを挟む
    pGfxFramework->ExecuteCommand(bufferIndex, &g_CaptureFence);

    // コマンドのレコードを再開させる
    pCommandBuffer->Begin();
}

// キャプチャ結果を取得し png ファイルの作成
void ExecuteScreenCapture(int captureFrame) NN_NOEXCEPT
{
    if (captureFrame < 0)
    {
        return;
    }

    // GPU のキャプチャ処理が終了するまで待機します。
    {
        const int64_t oneFrame = 16;
        nn::gfx::SyncResult result = g_CaptureFence.Sync(nn::TimeSpan::FromMilliSeconds(oneFrame));
        if (result != nn::gfx::SyncResult_Success)
        {
            return;
        }
    }

    // キャプチャ処理 (GPU バッファの結果から png ファイルを生成)
    const nnt::g3d::CaptureSetting::Camera* pCameraSetting = g_pCaptureSetting->GetCamera(g_ScreenCaptureCount);
    if (pCameraSetting->frame < 0)
    {
        if (captureFrame > 0 && captureFrame % 30 == 0)
        {
            // キャプチャ実行
            NN_LOG("Captured camera %d on frame %d\n", g_ScreenCaptureCount, captureFrame);
            ScreenCapture(++g_ScreenCaptureCount);
        }
    }
    else if (static_cast<int>(pCameraSetting->frame) == captureFrame)
    {
        // キャプチャ実行
        NN_LOG("Captured camera %d on frame %d\n", g_ScreenCaptureCount, captureFrame);
        ScreenCapture(++g_ScreenCaptureCount);
    }
    else
    {
        NN_ASSERT(static_cast<int>(pCameraSetting->frame) > captureFrame, "%d, %d",
            static_cast<int>(pCameraSetting->frame), captureFrame);
    }
}

extern g3ddemo::ResourceHolder g_Holder;
void RegisterAnimation(const char* pModelName, const char* pAnimName, float step, bool isMirroringEnabled, const char* retargetingHostName) NN_NOEXCEPT;
void RegisterAnimation(const char* pModelName, const char* pAnimName, float step) NN_NOEXCEPT;

// テスト用アニメーション登録
void RegisterAnimationToModelAutomatically(const CaptureSetting* pCaptureSetting, float step)
{
    for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        const char* pModelName = pModelAnimObj->GetModelObj()->GetName();
        const CaptureSetting::Model* pModelSetting = pCaptureSetting->GetModel(pModelName);

        if (pModelSetting && pModelSetting->isAutoAnimBindEnabled)
        {
            // 先頭の文字がモデル名で始まるアニメーションを追加する
            for (nn::g3d::ResFile* pFile : g_Holder.files)
            {
                // スケルタルアニメーション
                {
                    int animCount = pFile->GetSkeletalAnimCount();
                    for (int animIndex = 0; animIndex < animCount; ++animIndex)
                    {
                        nn::g3d::ResSkeletalAnim* pAnim = pFile->GetSkeletalAnim(animIndex);
                        const char* pAnimName = pAnim->GetName();
                        if (pModelSetting->autoBindAnimName)
                        {
                            if (strcmp(pAnimName, pModelSetting->autoBindAnimName) == 0)
                            {
                                RegisterAnimation(pModelName, pAnimName, step, pModelSetting->isMirroringEnabled, pModelSetting->retargetingHostName);
                            }
                        }
                        else if (strncmp(pModelName, pAnimName, strlen(pModelName)) == 0)
                        {
                            RegisterAnimation(pModelName, pAnimName, step);
                        }
                    }
                }

                // マテリアルアニメーション
                {
                    int animCount = pFile->GetMaterialAnimCount();
                    for (int animIndex = 0; animIndex < animCount; ++animIndex)
                    {
                        nn::g3d::ResMaterialAnim* pAnim = pFile->GetMaterialAnim(animIndex);
                        const char* pAnimName = pAnim->GetName();
                        if (pModelSetting->autoBindAnimName)
                        {
                            if (strncmp(pModelSetting->autoBindAnimName, pAnimName, strlen(pModelSetting->autoBindAnimName)) == 0)
                            {
                                RegisterAnimation(pModelName, pAnimName, step);
                            }
                        }
                        else if (strncmp(pModelName, pAnimName, strlen(pModelName)) == 0)
                        {
                            RegisterAnimation(pModelName, pAnimName, step);
                        }
                    }
                }
            }
        }
    }
}

nn::g3d::ResFile* FindResFile(g3ddemo::ResourceHolder* pHolder, const char* name)
{
    for (nn::g3d::ResFile* pResFile : pHolder->files)
    {
        if (strcmp(name, pResFile->GetName()) != 0)
        {
            continue;
        }
        return pResFile;
    }
    return nullptr;
}

} // tests

namespace g3ddemo = nn::g3d::demo;
namespace {

g3ddemo::ResourceHolder         g_Holder;

nns::g3d::RenderView            g_RenderView;
nn::util::Vector3fType          g_CameraPosition;
nn::util::Vector3fType          g_CameraTarget;

nn::g3d::ResShadingModel*       g_pResShadingModelPtr[ShadingModelType_Count];

nn::util::Vector3fType          g_CameraUp;
nn::util::Vector3fType          g_LightPosition;

TownContext                     g_Context;
TextureAverageShader            g_TextureAverageShader;
TownPointLight                  g_PointLight;
g3ddemo::ScreenInfo             g_ScreenInfo;
g3ddemo::BoundingRenderer       g_BoundingRenderer;
nn::g3d::ModelObj*              g_pEnvModelObj = nullptr;

nns::gfx::DepthStencilBuffer    g_DepthBufferShadow;
nns::gfx::FrameBuffer           g_FrameBufferWater;
nns::gfx::FrameBuffer           g_FrameBufferShadow;
nns::gfx::FrameBuffer           g_FrameBufferGeometry;
nns::gfx::FrameBuffer           g_FrameBufferLight;
nns::gfx::ColorBuffer           g_ColorBuffer;
nn::gfx::DescriptorSlot         g_WaterDescriptorSlot;
nn::gfx::DescriptorSlot         g_DepthDescriptorSlot;
nn::gfx::DescriptorSlot         g_GeometryColorDescriptorSlot;
nn::gfx::DescriptorSlot         g_GeometryDepthDescriptorSlot;
nn::gfx::DescriptorSlot         g_LightDescriptorSlot;
nn::gfx::DescriptorSlot         g_ColorDescriptorSlot;

CalculateLod                    g_CalculateLodDefault;
CalculateLod                    g_CalculateLodWater;

int                             g_FrameCount = 0;
const int                       g_BufferingCount = 2;

const char* g_pShaderArchiveNameArray[ShadingModelType_Count] = {
    "shadow",
    "gbuffer",
    "tex_average",
    "town",
};

const char* g_pShaderingModelNameArray[ShadingModelType_Count] = {
    "shadow",
    "gbuffer",
    "tex_average",
    "town_light",
};


// サブメッシュ LOD 計算
const int g_MaxLodLevel = 8;

enum File
{
    File_TownEnv = 0,
    File_WhiteTownWaterCaustics,
    File_WhiteTownWaterFlat,
    File_BgWhiteTown,
    File_Duck,
    File_FemaleA,
    File_Fish,
    File_MaleA,
    File_Light,
    File_Count
};

#define RESOURCE_PATH "Resource:/"

const char* BfshaPath[] = {
    RESOURCE_PATH "shader/demo.bfsha",  // 3D Editor 用
    RESOURCE_PATH "shader/town.bfsha",  // 3D Editor 用
    RESOURCE_PATH "shader/shadow.bfsha",
    RESOURCE_PATH "shader/gbuffer.bfsha",
    RESOURCE_PATH "shader/tex_average.bfsha",
};

//-------------------------------------------------------------------
//  初期化・終了処理
//-------------------------------------------------------------------

// 表示情報の初期化
void InitializeScreenInfo(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // テキストメニューを初期化
    g_ScreenInfo.Initialize(pDevice);
    g_ScreenInfo.SetWindowPosition(16, 16);
    g_ScreenInfo.SetFontSize(20);
}

// フレームバッファーを初期化
void InitializeFrameBuffer(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // 水面用フレームバッファーをセットアップ
    {
        nns::gfx::FrameBuffer::InfoType info(640, 360);
        info.SetColorBufferFormat(nn::gfx::ImageFormat_R11_G11_B10_Float);
        g3ddemo::InitializeFrameBuffer(&g_FrameBufferWater, &info);
        g_FrameBufferWater.SetClearColor(0.3f, 0.3f, 0.3f);
        g_WaterDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_FrameBufferWater.GetColorBuffer()->GetTextureView());
    }

    // シャドウ用デプスバッファー初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(640);
        info.SetHeight(640);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_Texture);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_D32_Float);
        info.SetMipCount(1);
        info.SetArrayLength(TownContext::cascadeShadowCount + 1);
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
        size_t align = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);
        ptrdiff_t offset = pGfxFramework->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, align);
        nn::gfx::MemoryPool* pMemoryPool = pGfxFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);
        g_DepthBufferShadow.Initialize(pDevice, info, pMemoryPool, offset, size);
        g_DepthDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_DepthBufferShadow.GetTextureView());
    }

    // シャドウ用フレームバッファー初期化
    {
        nns::gfx::FrameBuffer::InfoType info(640, 640);
        info.SetDepthBufferDisabled();
        g3ddemo::InitializeFrameBuffer(&g_FrameBufferShadow, &info);
    }

    // ジオメトリ用フレームバッファー初期化
    {
        nns::gfx::FrameBuffer::InfoType info(640, 360);
        info.SetColorBufferFormat(nn::gfx::ImageFormat_R11_G11_B10_Float);
        g3ddemo::InitializeFrameBuffer(&g_FrameBufferGeometry, &info);
        g_GeometryColorDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_FrameBufferGeometry.GetColorBuffer()->GetTextureView());
        g_GeometryDepthDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_FrameBufferGeometry.GetDepthBuffer()->GetTextureView());
        g_FrameBufferGeometry.SetClearColor(0.3f, 0.3f, 0.3f);
    }

    // ライト用フレームバッファー初期化
    {
        nns::gfx::FrameBuffer::InfoType info(640, 360);
        info.SetColorBufferFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetDepthBufferDisabled();
        g3ddemo::InitializeFrameBuffer(&g_FrameBufferLight, &info);
        g_FrameBufferLight.SetClearColor(0.0f, 0.0f, 0.0f, 0.0f);
        g_LightDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_FrameBufferLight.GetColorBuffer()->GetTextureView());
    }

    // コピー用カラーバッファー初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(pGfxFramework->GetDisplayWidth());
        info.SetHeight(pGfxFramework->GetDisplayHeight());
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R11_G11_B10_Float);
        info.SetMipCount(1);
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
        size_t align = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);
        ptrdiff_t offset = pGfxFramework->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, align);
        nn::gfx::MemoryPool* pMemoryPool = pGfxFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);
        g_ColorBuffer.Initialize(pDevice, info, pMemoryPool, offset, size);
        g_ColorDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_ColorBuffer.GetTextureView());
    }
}

// フレームバッファの破棄
void FinalizeFrameBuffer(nns::gfx::GraphicsFramework* pGfxFramework) NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // ディスクリプタプールからテクスチャーの登録を解除
    g3ddemo::UnregisterTextureFromDescriptorPool(g_FrameBufferWater.GetColorBuffer()->GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_DepthBufferShadow.GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_FrameBufferLight.GetColorBuffer()->GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_FrameBufferGeometry.GetColorBuffer()->GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_FrameBufferGeometry.GetDepthBuffer()->GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_ColorBuffer.GetTextureView());

    // フレームバッファーの破棄
    g3ddemo::FinalizeFrameBuffer(&g_FrameBufferLight);
    g3ddemo::FinalizeFrameBuffer(&g_FrameBufferGeometry);
    g3ddemo::FinalizeFrameBuffer(&g_FrameBufferShadow);
    g3ddemo::FinalizeFrameBuffer(&g_FrameBufferWater);

    // シャドウ用デプスバッファーの破棄
    {
        ptrdiff_t offset = g_DepthBufferShadow.GetMemoryPoolOffset();
        g_DepthBufferShadow.Finalize(pDevice);
        pGfxFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, offset);
    }

    // カラーバッファーの破棄
    {
        ptrdiff_t offset = g_ColorBuffer.GetMemoryPoolOffset();
        g_ColorBuffer.Finalize(pDevice);
        pGfxFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, offset);
    }
}

// ビューの初期化
void InitializeRenderView(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_RenderView.Initialize(pDevice, ViewType_Count, g_BufferingCount);

    // 投影行列を設定
    {
        nn::util::Matrix4x4fType matrix;
        MatrixPerspectiveFieldOfViewRightHanded(&matrix, nn::util::DegreeToRadian(45.0f),
            16.0f / 9.0f, 10.0f, 40000.0f);
        g_RenderView.SetProjectionMtx(ViewType_Default, matrix);
    }

    {
        nn::util::Matrix4x4fType matrix;
        MatrixPerspectiveFieldOfViewRightHanded(&matrix, nn::util::DegreeToRadian(45.0f),
            static_cast<float>(g_FrameBufferWater.GetWidth()) / static_cast<float>(g_FrameBufferWater.GetHeight()),
            10.0f, 40000.0f);
        g_RenderView.SetProjectionMtx(ViewType_Water, matrix);
    }
}

// リソースの初期化
void InitializeResource(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // リソースホルダの初期化
    g_Holder.Initialize();

    // シェーダーアーカイブの初期化
    for (int pathIndex = 0; pathIndex < NN_ARRAY_SIZE(BfshaPath); ++pathIndex)
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(BfshaPath[pathIndex]);
        nn::g3d::ResShaderArchive* pShaderArchive = pFile->GetResShaderArchive();
        pShaderArchive->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
    }

    for (nn::g3d::ResShaderFile* pShaderFile : g_Holder.shaderFiles)
    {
        nn::g3d::ResShaderArchive* pResShaderArchive = pShaderFile->GetResShaderArchive();

        const char* pResShaderArchiveName = pResShaderArchive->GetName();

        // モデル固有では無いシェーディングモデルのポインターを設定します。
        for (int shaderIndex = 0; shaderIndex < ShadingModelType_Count; ++shaderIndex)
        {
            if (strcmp(pResShaderArchiveName, g_pShaderArchiveNameArray[shaderIndex]) != 0)
            {
                continue;
            }
            nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->FindShadingModel(g_pShaderingModelNameArray[shaderIndex]);
            g_pResShadingModelPtr[shaderIndex] = pResShadingModel;
            break;
        }
    }

    for (nn::g3d::ResShadingModel* pResShadingModel : g_pResShadingModelPtr)
    {
        NN_ASSERT_NOT_NULL(pResShadingModel);
        NN_UNUSED(pResShadingModel);
    }

    // サンプラーの参照テクスチャーが空の場合に割り当てられるダミーテクスチャーの初期化
    g3ddemo::CreateDummyTextureAndSampler(pDevice);

    // リソースモデルの初期化
    for (int resourceIndex = 0; resourceIndex < g_pCaptureSetting->GetResourceCount(); ++resourceIndex)
    {
        const nnt::g3d::CaptureSetting::Resource* pResourceSettings = g_pCaptureSetting->GetResource(resourceIndex);
        nn::g3d::ResFile* pResFile = g3ddemo::LoadResource<nn::g3d::ResFile>(pResourceSettings->path);
        pResFile->Setup(pDevice);
        g_Holder.files.PushBack(pResFile);

        // テクスチャーおよびサンプラーをディスクリプタプールに登録
        g3ddemo::SetupTexture(pDevice, pResFile, g3ddemo::TextureBindCallback);
        g3ddemo::RegisterSamplerToDescriptorPool(pResFile);
    }

    // モデル固有のシェーダーのセットアップ
    const char* materialShaderArchiveFileName = "town.bfsha";
    for (nn::g3d::ResFile* pResFile : g_Holder.files)
    {
        nn::g3d::ResExternalFile* pResExternalFile = pResFile->FindExternalFile(materialShaderArchiveFileName);
        if (pResExternalFile)
        {
            void* pExternalFile = pResExternalFile->GetData();
            NN_ASSERT(nn::g3d::ResShaderFile::IsValid(pExternalFile));
            nn::g3d::ResShaderArchive* pModelShader = nn::g3d::ResShaderFile::ResCast(pExternalFile)->GetResShaderArchive();
            pModelShader->Setup(pDevice);
            pResFile->SetUserPtr(pModelShader);
        }
    }
}

// リソースの破棄
void FinalizeResource(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // モデルに付加されているシェーダーアーカイブを破棄
    for (nn::g3d::ResFile* pResFile : g_Holder.files)
    {
        nn::g3d::ResExternalFile* pResExternalFile = pResFile->FindExternalFile("town.bfsha");
        if (pResExternalFile)
        {
            void* pExternalFile = pResExternalFile->GetData();
            nn::g3d::ResShaderArchive* pModelShader = nn::g3d::ResShaderFile::ResCast(pExternalFile)->GetResShaderArchive();
            pModelShader->Cleanup(pDevice);
        }
    }

    // 環境モデルの破棄
    nns::g3d::DestroyModelObj(pDevice, g_pEnvModelObj);

    // リソースホルダ内を破棄
    g3ddemo::DestroyAll(pDevice, &g_Holder);
}

// 環境モデルの初期化
void InitializeEnvModel(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    nn::g3d::ResFile* pResFile = g_Holder.files[File_TownEnv];
    nn::g3d::ResModel* pResModel = pResFile->GetModel(0);
    nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(0);
    nn::gfx::Sampler* pSampler = pResMaterial->FindSampler("_d0");

    // サンプラーを作成し直す
    g3ddemo::UnregisterSamplerFromDescriptorPool(pSampler);
    pSampler->Finalize(pDevice);

    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToBorder);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToBorder);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
    samplerInfo.SetBorderColorType(nn::gfx::TextureBorderColorType_White);
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_Comparison_MinLinear_MagLinear_MipLinear);
    samplerInfo.SetComparisonFunction(nn::gfx::ComparisonFunction_LessEqual);

    pSampler->Initialize(pDevice, samplerInfo);
    g3ddemo::RegisterSamplerToDescriptorPool(pSampler);

    // フレームバッファーをテクスチャーに接続
    {
        nn::g3d::TextureRef textureRef(g_FrameBufferWater.GetColorBuffer()->GetTextureView(), g_WaterDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "water");
    }
    {
        nn::g3d::TextureRef textureRef(g_DepthBufferShadow.GetTextureView(), g_DepthDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "shadow");
    }
    {
        nn::g3d::TextureRef textureRef(g_FrameBufferGeometry.GetColorBuffer()->GetTextureView(), g_GeometryColorDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "normal");
    }
    {
        nn::g3d::TextureRef textureRef(g_FrameBufferGeometry.GetDepthBuffer()->GetTextureView(), g_GeometryDepthDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "depth");
    }
    {
        nn::g3d::TextureRef textureRef(g_FrameBufferLight.GetColorBuffer()->GetTextureView(), g_LightDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "light");
    }
    {
        nn::g3d::TextureRef textureRef(g_ColorBuffer.GetTextureView(), g_ColorDescriptorSlot);
        pResModel->ForceBindTexture(textureRef, "color");
    }

    // ユニフォームブロックのサイズを設定
    nn::g3d::ResShaderArchive* pModelShader = pResFile->GetUserPtr<nn::g3d::ResShaderArchive>();
    nn::g3d::ShaderUtility::BindShaderParam(pResMaterial, pModelShader->FindShadingModel("town_env"));

    // ModelObjを作成
    nn::g3d::ModelObj::Builder builder(pResModel);
    builder.MaterialBufferingCount(g_BufferingCount);
    builder.SkeletonBufferingCount(g_BufferingCount);
    builder.ShapeBufferingCount(g_BufferingCount);
    g_pEnvModelObj = nns::g3d::CreateModelObj(pDevice, builder);
}

// コンピュートシェーダー用に SSBO とディスパッチパラメータの設定を行います。
void InitializeTextureAverageShader(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_TextureAverageShader.Initialize(pDevice, g_pResShadingModelPtr[ShadingModelType_TextureAverage]);
    g_TextureAverageShader.SetTileResolution(pDevice, 64, 64);
}

// モデルの描画設定を初期化
void InitializeRenderModel(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    for (int fileIndex = 0; fileIndex < g_Holder.files.GetCount(); ++fileIndex)
    {
        nn::g3d::ResFile* pResFile = pHolder->files[fileIndex];
        if (strcmp("town_env", pResFile->GetName()) == 0)
        {
            continue;
        }
        if (strcmp("Light", pResFile->GetName()) == 0)
        {
            continue;
        }

        CreateRenderModel(pHolder, pDevice, pResFile);
    }
}

// ShapeObj の ユーザーエリアに LOD 情報を設定します。
void SetupShapeUserArea(nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
{
    for (int shapeIndex = 0, shapeCount = pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);

        // 各LODレベルの頂点バッファー内での開始オフセットを設定する
        for (int meshIndex = 0; meshIndex < g_MaxLodLevel; ++meshIndex)
        {
            if (meshIndex < pShapeObj->GetMeshCount())
            {
                const nn::g3d::ResMesh* pResMesh = pShapeObj->GetResMesh(meshIndex);
                pShapeObj->GetUserArea<int>()[meshIndex] = static_cast<int>(pResMesh->GetOffset());
            }
            else
            {
                // LODレベルが存在しない場合は-1
                pShapeObj->GetUserArea<int>()[meshIndex] = -1;
            }
        }
    }

    // ユーザーエリアの更新を必ず行う
    pModelObj->SetShapeUserAreaForceUpdateEnabled();
}

// モデルインスタンスの生成
void InitializeModelAnimObj(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    for (int fileIndex = 0; fileIndex < g_Holder.files.GetCount(); ++fileIndex)
    {
        nn::g3d::ResFile* pResFile = pHolder->files[fileIndex];
        if (strcmp("town_env", pResFile->GetName()) == 0)
        {
            continue;
        }
        if (strcmp("Light", pResFile->GetName()) == 0)
        {
            continue;
        }

        CreateModelAnimObj(pHolder, pDevice, pResFile, nullptr);
    }
}

// RenderModelObj の初期化
void InitializeRenderModelObj(g3ddemo::ResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_ASSERT_EQUAL(pHolder->modelAnimObjs.GetCount(), pHolder->renderModels.GetCount());

    // RenderModelObj の構築
    for (int modelIndex = 0, modelCount = pHolder->modelAnimObjs.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        CreateRenderModelObj(pHolder, pHolder->modelAnimObjs[modelIndex], pHolder->renderModels[modelIndex]);
    }
}

// RenderModelObj 内にあるシェーダーセレクターに水面描画用のキーを書き込みます。
void SetupWaterVariation(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    int shapeCount = pRenderModelObj->GetModelObj()->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        nns::g3d::RenderUnitObj* pRenderUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
        nn::g3d::ShaderSelector* pShaderSelector = pRenderUnitObj->GetShaderSelector(DrawPassType_Water);
        int optionIndex = pShaderSelector->FindDynamicOptionIndex("water_world");
        if (optionIndex == nn::util::ResDic::Npos)
        {
            continue;
        }

        const nn::g3d::ResShaderOption* pResShaderOption = pShaderSelector->GetDynamicOption(optionIndex);
        int choiceIndex = pResShaderOption->FindChoiceIndex("1");
        if (choiceIndex == nn::util::ResDic::Npos)
        {
            continue;
        }

        pShaderSelector->WriteDynamicKey(optionIndex, choiceIndex);
    }
}

// モデルに LOD やフラスタムカリングの設定
void SetupRenderModelObj(g3ddemo::ResourceHolder* pHolder) NN_NOEXCEPT
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
    {
        ::SetupRenderModelObj(pRenderModelObj);
    }
}

void RegisterAnimation(const char* pModelName, const char* pAnimName, float step, bool isMirroringEnabled, const char* retargetingHostName) NN_NOEXCEPT
{
    for (int objIndex = 0, objCount = g_Holder.modelAnimObjs.GetCount(); objIndex < objCount; ++objIndex)
    {
        nns::g3d::ModelAnimObj* pModelAnimObj = g_Holder.modelAnimObjs[objIndex];
        int* pAnimId = g_Holder.animationIds[objIndex];

        if (strcmp(pModelName, pModelAnimObj->GetModelObj()->GetResource()->GetName()) == 0)
        {
            for (nn::g3d::ResFile* pFile : g_Holder.files)
            {
                nn::g3d::ResSkeletalAnim* pResSkeletalAnim;
                if ((pResSkeletalAnim = pFile->FindSkeletalAnim(pAnimName)) != nullptr)
                {
                    nn::g3d::SkeletalAnimObj::BindArgument bindArg;
                    bindArg.SetResource(pModelAnimObj->GetModelObj()->GetSkeleton()->GetResource());
                    if (retargetingHostName)
                    {
                        for (int index = 0; index < g_Holder.modelAnimObjs.GetCount(); ++index)
                        {
                            nns::g3d::ModelAnimObj* pHostModelAnimObj = g_Holder.modelAnimObjs[index];
                            if (strcmp(retargetingHostName, pHostModelAnimObj->GetModelObj()->GetResource()->GetName()) == 0)
                            {
                                bindArg.SetHostResource(pHostModelAnimObj->GetModelObj()->GetSkeleton()->GetResource());
                                bindArg.SetRetargetingEnabled();
                            }
                        }
                    }
                    if (isMirroringEnabled)
                    {
                        bindArg.SetMirroringEnabled();
                    }
                    int id = pModelAnimObj->CreateAnim(pResSkeletalAnim, bindArg);
                    pModelAnimObj->GetAnimObj(id)->GetFrameCtrl().SetStep(step);
                    pAnimId[AnimationType_Skeletal] = id;
                }

                nn::g3d::ResMaterialAnim* pResMaterialAnim;
                if ((pResMaterialAnim = pFile->FindMaterialAnim(pAnimName)) != nullptr)
                {
                    int id = pModelAnimObj->CreateAnim(pResMaterialAnim);
                    pModelAnimObj->GetAnimObj(id)->GetFrameCtrl().SetStep(step);
                    pAnimId[AnimationType_Material] = id;
                }

                nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim;
                if ((pResBoneVisibilityAnim = pFile->FindBoneVisibilityAnim(pAnimName)) != nullptr)
                {
                    int id = pModelAnimObj->CreateAnim(pResBoneVisibilityAnim);
                    pModelAnimObj->GetAnimObj(id)->GetFrameCtrl().SetStep(step);
                    pAnimId[AnimationType_BoneVisibility] = id;
                }

                nn::g3d::ResShapeAnim* pResShapeAnim;
                if ((pResShapeAnim = pFile->FindShapeAnim(pAnimName)) != nullptr)
                {
                    int id = pModelAnimObj->CreateAnim(pResShapeAnim);
                    pModelAnimObj->GetAnimObj(id)->GetFrameCtrl().SetStep(step);
                    pAnimId[AnimationType_Shape] = id;
                }
            }
        }
    }
}

// アニメーションの登録
void RegisterAnimation(const char* pModelName, const char* pAnimName, float step) NN_NOEXCEPT
{
    RegisterAnimation(pModelName, pAnimName, step, false, NULL);
}

// アニメーションの初期化
void InitializeAnimation() NN_NOEXCEPT
{
    RegisterAnimation("bg_WhiteTown", "bg_WhiteTown", 1.0f);
    RegisterAnimation("bg_WhiteTown_cloth", "bg_WhiteTown_cloth", 1.0f);
    RegisterAnimation("Duck", "Duck", 1.0f);
    RegisterAnimation("FemaleA", "FemaleA_MoAnime", 1.0f);
    RegisterAnimation("Fish", "Fish", 1.0f);
    RegisterAnimation("MaleA", "MaleA_MoAnime", 1.0f);
    RegisterAnimation("WhiteTown_Water_Caustics", "WhiteTown_Water_Caustics", 0.2f);
    RegisterAnimation("bg_WhiteTown_Water_Flat", "bg_WhiteTown_Water_Flat", 0.5f);

    // テスト用アニメーションを追加
    RegisterAnimationToModelAutomatically(g_pCaptureSetting, 1.0f);
}

// モデルの位置を初期化
void InitializeModelPosition() NN_NOEXCEPT
{
    for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        const char* pName = pModelAnimObj->GetModelObj()->GetName();
        if (strcmp(pName, "FemaleA") == 0)
        {
            nn::util::Vector3fType rotate;
            nn::util::Vector3fType translate;
            VectorSet(&rotate, 0.0f, nn::util::DegreeToRadian(60.0f), 0.0f);
            VectorSet(&translate, -300.0f, 12.0f, 1550.0f);
            pModelAnimObj->SetRotate(rotate);
            pModelAnimObj->SetTranslate(translate);
        }
        else if (strcmp(pName, "MaleA") == 0)
        {
            nn::util::Vector3fType rotate;
            nn::util::Vector3fType translate;
            VectorSet(&rotate, 0.0f, 0.0f, 0.0f);
            VectorSet(&translate, 0.0f, 15.0f, 170.0f);
            pModelAnimObj->SetRotate(rotate);
            pModelAnimObj->SetTranslate(translate);
        }
    }

    //  外部ファイルのモデル設定を反映
    for (int modelSettingIndex = 0; modelSettingIndex < g_pCaptureSetting->GetModelCount(); ++modelSettingIndex)
    {
        const CaptureSetting::Model* pModelSetting = g_pCaptureSetting->GetModel(modelSettingIndex);
        nn::util::Vector3fType translate;
        VectorSet(&translate, pModelSetting->posX, pModelSetting->posY, pModelSetting->posZ);

        for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
        {
            const char* pName = pModelAnimObj->GetModelObj()->GetName();
            if (strcmp(pName, pModelSetting->name) == 0)
            {
                pModelAnimObj->SetTranslate(translate);
            }
        }
    }
}

// ModelAnimObj を更新
void CalculateModelAnimObjs(int uniformBufferIndex, bool enableAnimation) NN_NOEXCEPT;

// リソースホルダーから指定した名前の RenderModelObj を取得
nns::g3d::RenderModelObj* FindRenderModelObj(const char* pModelName) NN_NOEXCEPT;

// ライトの配置を初期化
void InitializePointLight(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // 配置座標を得るために更新
    CalculateModelAnimObjs(0, true);
    CalculateModelAnimObjs(1, true);

    nn::g3d::ResFile* pLightFile = FindResFile(&g_Holder, "Light");
    if (!pLightFile)
    {
        return;
    }

    nn::g3d::ResModel* pLightResModel = pLightFile->FindModel("Point");
    NN_ASSERT_NOT_NULL(pLightResModel);

    // ライトの設定を初期化
    g_PointLight.Initialize(pDevice, pLightResModel, g_pResShadingModelPtr[ShadingModelType_Light]);

    // Town のモデルを取得
    nns::g3d::RenderModelObj* pTownRenderModelObj = FindRenderModelObj("bg_WhiteTown");
    NN_ASSERT_NOT_NULL(pTownRenderModelObj);

    const char* pointLightPrefix = "PointLight_";
    const nn::util::Float3 pointLightColor = NN_UTIL_FLOAT_3_INITIALIZER(1.0f, 0.8f, 0.4f);

    // ポイントライト用のボーンを取得し、ライトを配置
    const nn::g3d::SkeletonObj* pSkeletonObj = pTownRenderModelObj->GetModelObj()->GetSkeleton();
    for (int boneIndex = 0, boneCount = pSkeletonObj->GetBoneCount(); boneIndex < boneCount; ++boneIndex)
    {
        const nn::g3d::ResBone* pResBone = pSkeletonObj->GetResBone(boneIndex);
        if (strncmp(pResBone->GetName(), pointLightPrefix, strlen(pointLightPrefix)) != 0)
        {
            continue;
        }

        // ライト用のボーンから位置を取得
        const nn::util::Matrix4x3fType& matrix = pSkeletonObj->GetWorldMtxArray()[boneIndex];
        nn::util::Vector3fType center;
        MatrixGetAxisW(&center, matrix);

        g_PointLight.AddLight(pDevice, pointLightColor, center);
    }
}

// カメラ配置の初期化
void InitializeCamera() NN_NOEXCEPT
{
    VectorSet(&g_CameraPosition, 380.0f, 140.0f, 2400.0f);
    VectorSet(&g_CameraUp, 0.0f, 1.0f, 0.0f);
    VectorSet(&g_CameraTarget, -400.0f, 0.0f, 0.0f);
    VectorSet(&g_LightPosition, 2000.0f, 2200.0f, 2000.0f);
    nn::util::Matrix4x3fType matrix;
    MatrixLookAtRightHanded(&matrix, g_CameraPosition, g_CameraTarget, g_CameraUp);
    g_RenderView.SetViewMtx(ViewType_Default, matrix);
}

// Townの初期化
void InitializeTown(nns::gfx::GraphicsFramework* pGfxFramework) NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // メニュー項目の初期化
    InitializeMenu();

    // 表示情報の初期化
    InitializeScreenInfo(pDevice);

    // バウンディング表示の初期化
    g_BoundingRenderer.Initialize(pDevice);

    // 現在のスレッド(メインスレッド)を 0 番コアに割り当て
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

    // フレームバッファーの初期化
    InitializeFrameBuffer(pGfxFramework, pDevice);

    // コンテキストの初期化
    g_Context.Initialize(pDevice);

    // ビューの初期化
    InitializeRenderView(pDevice);

    // リソースの初期化
    InitializeResource(pDevice);

    // 環境モデルの初期化
    InitializeEnvModel(pDevice);

    // フレームバッファの平均輝度を求めるシェーダーを初期化
    InitializeTextureAverageShader(pDevice);

    // モデルの描画設定を初期化
    InitializeRenderModel(&g_Holder, pDevice);

    // モデルインスタンスの生成
    InitializeModelAnimObj(&g_Holder, pDevice);

    // モデル描画設定とモデルインスタンスの関連付け
    InitializeRenderModelObj(&g_Holder);

    // モデルに LOD やフラスタムカリングの設定
    SetupRenderModelObj(&g_Holder);

    // アニメーションの登録
    InitializeAnimation();

    // モデルの位置を初期化
    InitializeModelPosition();

    // ポイントライトを初期化
    InitializePointLight(pDevice);

    // カメラ配置の初期化
    InitializeCamera();

    // シェーダーに必要なスクラッチメモリーを設定
    g3ddemo::SetShaderScratchMemory(g_Holder.shaderFiles);

    // キャプチャテストを初期化
    InitializeScreenCapture();

    // キャプチャ用フェンスを初期化
    InitializeCaptureFence(pDevice);
}

// Townの終了処理
void FinalizeTown(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // キャプチャ用フェンスを破棄
    FinalizeCaptureFence(pDevice);

    // ライトの終了処理
    if (g_PointLight.IsInitialized())
    {
        g_PointLight.Finalize(pDevice);
    }

    // 平均輝度を求めるシェーダーの破棄
    g_TextureAverageShader.Finalize(pDevice);

    // ダミーテクスチャーの破棄
    g3ddemo::DeleteDummyTextureAndSampler(pDevice);

     // リソースの破棄
    FinalizeResource(pDevice);

    // ビューの破棄
    g_RenderView.Finalize(pDevice);

    // コンテキストの破棄
    g_Context.Finalize(pDevice);

    // フレームバッファの終了処理
    FinalizeFrameBuffer(pGfxFramework);

     // バウンディング表示の破棄
    g_BoundingRenderer.Finalize(pDevice);

    // スクリーン情報表示の破棄
    g_ScreenInfo.Cleanup(pDevice);

    // フレームワークのコールバック関数を無効化
    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);
}

//-------------------------------------------------------------------
//  更新
//-------------------------------------------------------------------

// モデルの更新処理
void CalculateModelAnimObjs(int uniformBufferIndex, bool enableAnimation) NN_NOEXCEPT
{
    // 各描画パス共通のModelAnimObjを更新
    for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        // アニメーションの実行
        if (enableAnimation)
        {
            pModelAnimObj->Calculate();
        }

        pModelAnimObj->CalculateUniformBlock(uniformBufferIndex);
    }
}

// リソースホルダーから指定した名前の RenderModelObj を取得
nns::g3d::RenderModelObj* FindRenderModelObj(const char* pModelName) NN_NOEXCEPT
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        if (strcmp(pRenderModelObj->GetModelObj()->GetName(), pModelName) != 0)
        {
            continue;
        }

        return pRenderModelObj;
    }

    return nullptr;
}

// AABB のサイズを計算
void CalculateAdjustedAabb(
    nn::g3d::Aabb* pAabbObj,
    nns::g3d::RenderUnitObj* pRenderUnitObj,
    const nn::util::Matrix4x3fType* pLightView,
    nn::g3d::ViewVolume* pViewVolume
) NN_NOEXCEPT
{
    const nn::g3d::ShapeObj* pShapeObj = pRenderUnitObj->GetShapeObj();
    const nn::g3d::Sphere* pSphere = pShapeObj->GetBounding();
    if (!pSphere || !pViewVolume->TestIntersection(*pSphere))
    {
        return;
    }

    nn::util::Vector3fType pos;
    VectorTransform(&pos, pSphere->center, *pLightView);
    VectorSet(&pAabbObj->min,
        (std::min)(VectorGetX(pAabbObj->min), VectorGetX(pos) - pSphere->radius),
        (std::min)(VectorGetY(pAabbObj->min), VectorGetY(pos) - pSphere->radius),
        (std::min)(VectorGetZ(pAabbObj->min), VectorGetZ(pos) - pSphere->radius));
    VectorSet(&pAabbObj->max,
        (std::max)(VectorGetX(pAabbObj->max), VectorGetX(pos) + pSphere->radius),
        (std::max)(VectorGetY(pAabbObj->max), VectorGetY(pos) + pSphere->radius),
        (std::max)(VectorGetZ(pAabbObj->max), VectorGetZ(pos) + pSphere->radius));
}

bool CanDrawToShadowMap(const nns::g3d::RenderUnitObj* pRenderUnitObj, int passIndex) NN_NOEXCEPT;

// 光源のプロジェクション行列を計算
void CalculateLightProj(
    nn::util::Matrix4x4fType* pOutLightProj,
    const nn::util::Matrix4x3fType* pLightView,
    const nn::util::Matrix4x3fType* pCameraView,
    const nn::util::Matrix4x4fType* pCameraProj
) NN_NOEXCEPT
{
    nn::util::Matrix4x4fType cameraViewProj;
    nn::util::Matrix4x4fType invCameraViewProj;
    nn::util::Matrix4x4fType screenToLightView;
    MatrixMultiply(&cameraViewProj, *pCameraView, *pCameraProj);
    MatrixInverse(&invCameraViewProj, cameraViewProj);
    MatrixMultiply(&screenToLightView, invCameraViewProj, *pLightView);

    nn::g3d::Aabb aabb;
    nn::g3d::Aabb aabbObj;
    VectorSet(&aabb.min, FLT_MAX, FLT_MAX, FLT_MAX);
    VectorSet(&aabbObj.min, FLT_MAX, FLT_MAX, FLT_MAX);
    VectorSet(&aabb.max, -FLT_MAX, -FLT_MAX, -FLT_MAX);
    VectorSet(&aabbObj.max, -FLT_MAX, -FLT_MAX, -FLT_MAX);

    nn::util::Vector3fType posLightView;
    for (int x = 0; x < 2; ++x)
    {
        for (int y = 0; y < 2; ++y)
        {
            for (int z = 0; z < 2; ++z)
            {
                VectorSet(&posLightView,
                    x == 0 ? -1.0f : 1.0f,
                    y == 0 ? -1.0f : 1.0f,
                    z == 0 ? -1.0f : 1.0f);
                VectorTransformCoord(&posLightView, posLightView, screenToLightView);
                VectorSet(&aabb.min,
                    (std::min)(VectorGetX(aabb.min), VectorGetX(posLightView)),
                    (std::min)(VectorGetY(aabb.min), VectorGetY(posLightView)),
                    (std::min)(VectorGetZ(aabb.min), VectorGetZ(posLightView)));
                VectorSet(&aabb.max,
                    (std::max)(VectorGetX(aabb.max), VectorGetX(posLightView)),
                    (std::max)(VectorGetY(aabb.max), VectorGetY(posLightView)),
                    (std::max)(VectorGetZ(aabb.max), VectorGetZ(posLightView)));
            }
        }
    }

    nn::g3d::ViewVolume viewVolume;
    nn::util::Matrix4x3fType invLightView;
    MatrixInverse(&invLightView, *pLightView);
    viewVolume.SetOrtho(VectorGetY(aabb.max), VectorGetY(aabb.min), VectorGetX(aabb.min), VectorGetX(aabb.max), 0.0f, -VectorGetZ(aabb.min), invLightView);

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        for (int shapeIndex = 0, shapeCount = pRenderModelObj->GetModelObj()->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            nns::g3d::RenderUnitObj* pRenderUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
            if (!CanDrawToShadowMap(pRenderUnitObj, DrawPassType_Shadow))
            {
                continue;
            }
            CalculateAdjustedAabb(&aabbObj, pRenderUnitObj, pLightView, &viewVolume);
        }

    }

    MatrixOrthographicOffCenterRightHanded(pOutLightProj,
        VectorGetX(aabbObj.min), VectorGetX(aabbObj.max), VectorGetY(aabbObj.min),
        VectorGetY(aabbObj.max), -VectorGetZ(aabbObj.max), -VectorGetZ(aabbObj.min));
}

// コンテキストの計算
void CalculateContext(int bufferIndex) NN_NOEXCEPT
{
    const float cascadeDepth[TownContext::cascadeShadowCount + 1] = { 10.0f, 800.0f, 1600.0f, 4500.0f, 40000.0f };

    for (int sliceIndex = 0; sliceIndex < TownContext::cascadeShadowCount; ++sliceIndex)
    {
        nn::util::Matrix4x4fType projMtx;
        MatrixPerspectiveFieldOfViewRightHanded(&projMtx, nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, cascadeDepth[sliceIndex], cascadeDepth[sliceIndex + 1]);

        nn::util::Vector3fType up = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 1.0f, 0.0f);
        nn::util::Vector3fType target = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 0.0f, 0.0f);

        {
            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(&matrix, g_LightPosition, target, up);
            g_RenderView.SetViewMtx(ViewType_Light0 + sliceIndex, matrix);
        }

        {
            nn::util::Matrix4x4fType matrix;
            CalculateLightProj(
                &matrix,
                &g_RenderView.GetViewMtx(ViewType_Light0 + sliceIndex),
                &g_RenderView.GetViewMtx(ViewType_Default),
                &projMtx
            );

            g_RenderView.SetProjectionMtx(ViewType_Light0 + sliceIndex, matrix);
        }
        MatrixMultiply(&projMtx, g_RenderView.GetViewMtx(ViewType_Light0 + sliceIndex), g_RenderView.GetProjectionMtx(ViewType_Light0 + sliceIndex));

        g_Context.SetLightProjection(bufferIndex, sliceIndex, projMtx);
    }

    g_Context.SetCascadeDepth(bufferIndex, &cascadeDepth[1]);
    g_Context.SetNearFar(bufferIndex, 10.0f, 40000.0f);
    float tanFovY = std::tan(nn::util::DegreeToRadian(45.0f * 0.5f));
    g_Context.SetTangentFov(bufferIndex, tanFovY * (16.0f / 9.0f), tanFovY);
    if (g_MenuFlag.Test<MenuFlag::TextureCompute>() && g_FrameCount > 1)
    {
        // 輝度平均を算出
        g_TextureAverageShader.CalculateAverage(bufferIndex);
        nn::util::Float4& texAverage = g_TextureAverageShader.GetTextureAverage(bufferIndex);
        float diff = texAverage.x + texAverage.y + texAverage.z - 0.65f;
        // 前回計算した結果
        float diffIntensity = g_Context.GetDiffIntensity(1 - bufferIndex);
        diffIntensity -= diff * diff * diff * 0.04f;
        if (diffIntensity < 0.3f) diffIntensity = 0.3f;
        if (diffIntensity > 1.7f) diffIntensity = 1.7f;
        g_Context.SetDiffIntensity(bufferIndex, diffIntensity);
    }
    else
    {
        g_Context.SetDiffIntensity(bufferIndex, nn::g3d::Math::Abs(((g_FrameCount + 27000) % 36000 - 18000) / 18000.0f) * 2.0f);
    }

    g_Context.Calculate(bufferIndex);
}

// 計算処理
void Calculate(nns::gfx::GraphicsFramework* pGfxFramework, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pGfxFramework);

    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    // 計算
    {
        // カメラポジションを外部ファイルを元に設定
        {
            GetScreenCaptureCamera(g_ScreenCaptureCount, &g_CameraPosition, &g_CameraTarget);
            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(&matrix, g_CameraPosition, g_CameraTarget, g_CameraUp);
            g_RenderView.SetViewMtx(ViewType_Default, matrix);
        }

        // 水面用のビュー行列を更新
        {
            nn::util::Vector3fType pos;
            VectorSet(&pos, VectorGetX(g_CameraPosition), -VectorGetY(g_CameraPosition), VectorGetZ(g_CameraPosition));
            nn::util::Vector3fType target;
            VectorSet(&target, VectorGetX(g_CameraTarget), -VectorGetY(g_CameraTarget), VectorGetZ(g_CameraTarget));

            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(&matrix, pos, target, g_CameraUp);
            g_RenderView.SetViewMtx(ViewType_Water, matrix);
        }

        // 環境モデルの更新処理
        g_pEnvModelObj->CalculateMaterial(uniformBufferIndex);
        g_pEnvModelObj->CalculateSkeleton(uniformBufferIndex);
        g_pEnvModelObj->CalculateShape(uniformBufferIndex);

        // ModelAnimObjの更新処理
        CalculateModelAnimObjs(uniformBufferIndex, g_MenuFlag.Test<MenuFlag::ModelAnimation>());

        // シェーダーの更新処理
        for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
        {
            pRenderModelObj->UpdateShader(pDevice);
        }

        // ライトの更新処理
        if (g_PointLight.IsInitialized())
        {
            g_PointLight.Calculate(pDevice, uniformBufferIndex, g_MenuFlag.Test<MenuFlag::LightAnimation>());
        }

        // コンテキストの計算
        CalculateContext(uniformBufferIndex);

        // ビューの更新処理
        g_RenderView.Calculate(uniformBufferIndex);

        // フラスタムカリング用のビューボリュームを更新
        {
            g_RenderView.GetViewVolume(ViewType_Default).SetPerspective(nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f,
                10.0f, 40000.0f, g_RenderView.GetInverseViewMtx(ViewType_Default));
            g_RenderView.GetViewVolume(ViewType_Water).SetPerspective(nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f,
                10.0f, 40000.0f, g_RenderView.GetInverseViewMtx(ViewType_Water));
        }

        // キャプチャ処理
        // MakeCommand() で描画を始める前にコピーをとっており、スキャンバッファはダブルバッファリングされているため
        // キャプチャバッファに格納されているデータは 数 フレーム前のスキャンバッファです。
        //  |       F0      |               F1              |               F2              |       F3      |
        //  |   MakeCommand |   Calculate   |   MakeCommand |   Calculate   |   MakeCommand |   Calculate   |
        //  |   cap|app0    |   app1|cap(x) |   cap|app1    |   app0|cap(x) |   cap|app0    |   app1|cap(F0)|
        const int bufferingCount = pGfxFramework->GetBufferCount();
        int screenCaptureFrame = g_FrameCount - (bufferingCount + 1);
        ExecuteScreenCapture(screenCaptureFrame);
    }
}

//-------------------------------------------------------------------
//  描画
//-------------------------------------------------------------------

// 画面のクリア
void ClearScreen(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{

    // フレームバッファーをクリア
    g3ddemo::ClearFrameBuffer(pCommandBuffer, &g_FrameBufferWater);
    g3ddemo::ClearFrameBuffer(pCommandBuffer, &g_FrameBufferGeometry);
    g3ddemo::ClearFrameBuffer(pCommandBuffer, &g_FrameBufferShadow);
    g3ddemo::ClearFrameBuffer(pCommandBuffer, &g_FrameBufferLight);

    // シャドウ用の深度ステンシルをクリア
    nn::gfx::DepthStencilView* pShadowDepth = g_DepthBufferShadow.GetDepthStencilView(0);
    pCommandBuffer->ClearDepthStencil(pShadowDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

    // カラーターゲットをクリア
    nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
    pCommandBuffer->ClearColor(pColor, 0.0f, 0.0f, 0.0f, 1.0f, nullptr);

    // 深度ステンシルをクリア
    nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
    pCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

}

// シャドウマップに描画するか判定
bool CanDrawToShadowMap(const nns::g3d::RenderUnitObj* pRenderUnitObj, int passIndex) NN_NOEXCEPT
{
    const nn::g3d::ResMaterial* pResMaterial = pRenderUnitObj->GetRenderShape(passIndex)->GetRenderMaterial()->GetResMaterial();
    const nn::g3d::ResRenderInfo* pRenderInfo = pResMaterial->FindRenderInfo("dynamic_shadow");

    if (!pRenderInfo)
    {
        return false;
    }
    return pRenderInfo->GetInt(0) == 1;
}

// シャドウの描画
void DrawShadow(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::ShadowRendering>())
    {
        return;
    }

    for (int sliceIndex = 0; sliceIndex < TownContext::cascadeShadowCount; ++sliceIndex)
    {
        nn::gfx::ColorTargetView* pColorTargetView = g_FrameBufferShadow.GetColorBuffer()->GetColorTargetView();
        pCommandBuffer->SetRenderTargets(1, &pColorTargetView, g_DepthBufferShadow.GetDepthStencilView(sliceIndex + 1));
        pCommandBuffer->SetViewportScissorState(g_FrameBufferShadow.GetViewportScissorState());

        // ユニフォームブロックを再バインド
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(ViewType_Light0 + sliceIndex, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(ViewType_Light0 + sliceIndex, 1);

        for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
        {
            pRenderModelObj->BindUniformBlock(DrawPassType_Shadow, "view", buffers, g_BufferingCount);
            pRenderModelObj->Draw(
                pCommandBuffer,
                DrawPassType_Shadow,
                ViewType_Light0 + sliceIndex,
                uniformBufferIndex,
                CanDrawToShadowMap
            );
        }
    }

}

// 水面の描画
void DrawWater(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::WaterRendering>())
    {
        return;
    }

    g3ddemo::SetFrameBufferToRenderTarget(pCommandBuffer, &g_FrameBufferWater);
    pCommandBuffer->SetViewportScissorState(g_FrameBufferWater.GetViewportScissorState());

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        pRenderModelObj->Draw(pCommandBuffer, DrawPassType_Water, ViewType_Water, uniformBufferIndex);
    }
}

// Gバッファーの描画
void DrawGeometry(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::GeometryRendering>())
    {
        return;
    }

    g3ddemo::SetFrameBufferToRenderTarget(pCommandBuffer, &g_FrameBufferGeometry);
    pCommandBuffer->SetViewportScissorState(g_FrameBufferGeometry.GetViewportScissorState());

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        pRenderModelObj->Draw(pCommandBuffer, DrawPassType_Geometry, ViewType_Default, uniformBufferIndex);
    }
}

// ライトの描画
void DrawLight(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::LightRendering>() || !g_PointLight.IsInitialized())
    {
        return;
    }

    g3ddemo::SetFrameBufferToRenderTarget(pCommandBuffer, &g_FrameBufferLight);
    pCommandBuffer->SetViewportScissorState(g_FrameBufferLight.GetViewportScissorState());

    g_PointLight.Draw(pCommandBuffer, uniformBufferIndex);
}

// 不透明オブジェクトか判定
bool IsOpaqueObject(const nns::g3d::RenderUnitObj* pRenderUnitObj, int passIndex) NN_NOEXCEPT
{
    const nn::g3d::ResMaterial* pResMaterial = pRenderUnitObj->GetRenderShape(passIndex)->GetRenderMaterial()->GetResMaterial();
    const nn::g3d::ResRenderInfo* pRenderInfo = pResMaterial->FindRenderInfo("blend");
    if (!pRenderInfo)
    {
        return true;
    }
    return pRenderInfo->GetInt(0) == g3ddemo::BlendMode_Opaque;
}

// 不透明オブジェクトの描画
void DrawOpaque(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        pRenderModelObj->Draw(
            pCommandBuffer,
            DrawPassType_Model,
            ViewType_Default,
            uniformBufferIndex,
            IsOpaqueObject
        );
    }
}

// 半透明オブジェクトの描画
void DrawTranslucent(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        pRenderModelObj->Draw(
            pCommandBuffer,
            DrawPassType_Model,
            ViewType_Default,
            uniformBufferIndex,
            [](const nns::g3d::RenderUnitObj* pRenderUnitObj, int passIndex)
            {
                return !IsOpaqueObject(pRenderUnitObj, passIndex);
            }
        );
    }
}

// カラーバッファーへコピー
void CopyColorBuffer(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    nn::gfx::TextureSubresource subresource;
    subresource.SetDefault();
    nn::gfx::TextureCopyRegion copyRegion;
    copyRegion.SetDefault();
    copyRegion.SetWidth(pGfxFramework->GetDisplayWidth());
    copyRegion.SetHeight(pGfxFramework->GetDisplayHeight());
    pCommandBuffer->CopyImage(g_ColorBuffer.GetTexture(), subresource, 0, 0, 0, pGfxFramework->GetColorBuffer(), copyRegion);
}

// 選択されているオブジェクトのみを描画
void DrawSelectedObj(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    int modelIndex = g_SelectedModelIndex;
    int shapeIndex = g_SelectedShapeIndex;
    int submeshIndex = g_SelectedSubmeshIndex;

    NN_ASSERT_RANGE(modelIndex, 0, g_Holder.modelAnimObjs.GetCount());
    nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[modelIndex];

    nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
    NN_ASSERT_RANGE(shapeIndex, 0, pModelObj->GetShapeCount());
    pRenderModelObj->SetDrawSingleShapeEnabled(shapeIndex);

    if (g_MenuFlag.Test<MenuFlag::SingleSubmesh>())
    {
        NN_ASSERT_RANGE(submeshIndex, 0, pModelObj->GetShape(shapeIndex)->GetSubMeshCount());
        pRenderModelObj->GetRenderUnitObj(shapeIndex)->SetDrawSingleSubmeshEnabled(submeshIndex);
    }

    pRenderModelObj->Draw(pCommandBuffer, DrawPassType_Model, ViewType_Default, uniformBufferIndex);
    pRenderModelObj->SetDrawSingleShapeDisabled();

    if (g_MenuFlag.Test<MenuFlag::SingleSubmesh>())
    {
        pRenderModelObj->GetRenderUnitObj(shapeIndex)->SetDrawSingleSubmeshDisabled();
    }
}

// 最終出力を描画
void DrawFinalImage(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) NN_NOEXCEPT
{
    // 指定のLODレベルでの強制表示を設定
    int modelIndex = g_SelectedModelIndex;
    int forceLodLevel = g_pCaptureSetting->GetLodLevel();

    NN_ASSERT_RANGE(modelIndex, 0, g_Holder.modelAnimObjs.GetCount());
    nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[modelIndex];
    if (forceLodLevel > -1)
    {
        pRenderModelObj->SetDrawLodEnabled(forceLodLevel);
    }
    else
    {
        pRenderModelObj->SetDrawLodDisabled();
    }

    // レンダーターゲットを設定
    nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
    nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
    pCommandBuffer->SetRenderTargets(1, &pColor, pDepth);
    pCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

    // 選択されたオブジェクトだけ描画
    if (g_MenuFlag.Test<MenuFlag::SingleShape>())
    {
        DrawSelectedObj(pCommandBuffer, uniformBufferIndex);
        return;
    }

    // 不透明オブジェクトの描画
    DrawOpaque(pCommandBuffer, uniformBufferIndex);

    // カラーバッファーへコピー
    CopyColorBuffer(pGfxFramework, pCommandBuffer);

    // 半透明オブジェクトの描画
    DrawTranslucent(pCommandBuffer, uniformBufferIndex);
}

// コンピュートシェーダー実行
void CalculateTextureCompute(nn::gfx::CommandBuffer* pCommandBuffer, int bufferIndex) NN_NOEXCEPT
{
    if (!g_MenuFlag.Test<MenuFlag::TextureCompute>())
    {
        return;
    }

    g_TextureAverageShader.MakeCommand(pCommandBuffer, bufferIndex, g_ColorDescriptorSlot);
}

// 描画コマンドを作成
void MakeCommand(nns::gfx::GraphicsFramework* pGfxFramework, int uniformBufferIndex, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pUserData);

    // コマンド生成
    pGfxFramework->BeginFrame(uniformBufferIndex);
    {
         // キャプチャコマンドを作成
        MakeCaptureCommand(pGfxFramework, uniformBufferIndex);

        nn::gfx::CommandBuffer* pCommandBuffer = pGfxFramework->GetRootCommandBuffer(uniformBufferIndex);

        // ディスクリプタとシェーダーコードのGPUキャッシュを無効化 は実は BeginFrame で設定されているので不要
        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor);

        // テクスチャーキャッシュ無効化
        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);

        // 画面のクリア
        ClearScreen(pGfxFramework, pCommandBuffer);

        // シャドウの描画
        DrawShadow(pCommandBuffer, uniformBufferIndex);

        // 水面の描画
        DrawWater(pCommandBuffer, uniformBufferIndex);

        // Gバッファーの描画
        DrawGeometry(pCommandBuffer, uniformBufferIndex);

        // キャッシュをフラッシュ
        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer);

        // ライトの描画
        DrawLight(pCommandBuffer, uniformBufferIndex);

        // キャッシュをフラッシュ
        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_DepthStencil);

        // 最終画面描画
        DrawFinalImage(pGfxFramework, pCommandBuffer, uniformBufferIndex);

        // GPU演算
        CalculateTextureCompute(pCommandBuffer, uniformBufferIndex);

        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);
    }
    pGfxFramework->EndFrame(uniformBufferIndex);
}



} // anonymous namespace

// ワイヤーフレーム切り替え
void SetWireframe(nns::g3d::RenderModelObj* pRenderModelObj, int fillMode, int materialIndex) NN_NOEXCEPT
{
    nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
    const nn::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndex);
    const nn::g3d::ResMaterial* pResMaterial = pMaterialObj->GetResource();
    nns::g3d::RenderModel* pRenderModel = pRenderModelObj->GetRenderModel();
    nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(materialIndex);

    // マテリアルにラスタライザーステートを設定
    const nn::g3d::ResRenderInfo* pRenderInfo = pResMaterial->FindRenderInfo("culling");
    if (pRenderInfo)
    {
        const nn::gfx::RasterizerState* pRasterizerState;
        NN_ASSERT_GREATER(pRenderInfo->GetArrayLength(), 0);

        g3ddemo::CullMode cullMode = static_cast<g3ddemo::CullMode>(pRenderInfo->GetInt(0));

        bool isVisible = cullMode >= 0 && cullMode < g3ddemo::CullMode_Max;
        pModelObj->SetMaterialVisible(materialIndex, isVisible);

        if (!isVisible)
        {
            return;
        }

        pRasterizerState = g3ddemo::GetRasterizerState(cullMode, fillMode);
        pRenderMaterial->SetRasterizerState(pRasterizerState);
    }
}

// レンダーステートを設定
void SetupRenderState(nns::g3d::RenderModel* pRenderModel) NN_NOEXCEPT
{
    for (int passIndex = 0, passCount = pRenderModel->GetPassCount(); passIndex < passCount; ++passIndex)
    {
        for (int materialIndex = 0, materialCount = pRenderModel->GetResModel()->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
        {
            const nn::g3d::ResMaterial* pResMaterial = pRenderModel->GetResModel()->GetMaterial(materialIndex);
            nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(passIndex, materialIndex);

            // ブレンドステートを設定
            const nn::g3d::ResRenderInfo* pRenderInfo = pResMaterial->FindRenderInfo("blend");
            if (pRenderInfo)
            {
                int blendType = pRenderInfo->GetInt(0);
                NN_ASSERT_RANGE(blendType, g3ddemo::BlendMode_Opaque, g3ddemo::BlendMode_Max);
                const nn::gfx::BlendState* pBlendState = g3ddemo::GetBlendState(blendType);
                pRenderMaterial->SetBlendState(pBlendState);
            }

            // ラスタライザーステートを設定
            if (passIndex == DrawPassType_Shadow)
            {
                const nn::gfx::RasterizerState* pRasterizerState = g3ddemo::GetRasterizerState(nn::gfx::CullMode_Front);
                pRenderMaterial->SetRasterizerState(pRasterizerState);
                continue;
            }

            pRenderInfo = pResMaterial->FindRenderInfo("culling");
            if (pRenderInfo)
            {
                int cullType = pRenderInfo->GetInt(0);
                NN_ASSERT_RANGE(cullType, g3ddemo::CullMode_Back, g3ddemo::CullMode_Max);
                const nn::gfx::RasterizerState* pRasterizerState = g3ddemo::GetRasterizerState(cullType);
                pRenderMaterial->SetRasterizerState(pRasterizerState);
            }

            // デプスステンシルステートは変更しない
        }
    }
}

// ユニフォームブロックの設定を行います。
void BindUniformBlockAndSampler(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    // envモデルのユニフォームブロックとサンプラーをバインド
    {
        NN_ASSERT_NOT_NULL(g_pEnvModelObj);
        const nn::g3d::MaterialObj* pEnvMaterialObj = g_pEnvModelObj->GetMaterial(0);

        // ユニフォームを設定
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = pEnvMaterialObj->GetMaterialBlock(0);
        buffers[1] = pEnvMaterialObj->GetMaterialBlock(1);
        pRenderModelObj->BindUniformBlock("env", buffers, g_BufferingCount);

        // サンプラーを設定
        const nn::g3d::ResMaterial* pResMaterial = pEnvMaterialObj->GetResource();
        for (int samplerIndex = 0, samplerCount = pResMaterial->GetSamplerCount(); samplerIndex < samplerCount; ++samplerIndex)
        {
            const char* pSamplerName = pResMaterial->GetSamplerName(samplerIndex);
            nn::g3d::SamplerRef samplerRef = pEnvMaterialObj->GetSampler(samplerIndex);
            nn::g3d::TextureRef textureRef = pEnvMaterialObj->GetTexture(samplerIndex);
            pRenderModelObj->BindSampler(pSamplerName, textureRef.GetDescriptorSlot(), samplerRef.GetDescriptorSlot());
        }
    }

    // コンテキストのユニフォームブロックを設定
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_Context.GetContextBlock(0);
        buffers[1] = g_Context.GetContextBlock(1);
        pRenderModelObj->BindUniformBlock("context", buffers, g_BufferingCount);
    }

    // ビューのユニフォームブロックを設定
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(ViewType_Default, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(ViewType_Default, 1);
        pRenderModelObj->BindUniformBlock("view", buffers, g_BufferingCount);
    }

    // フラスタムカリング用のビューボリュームを設定
    pRenderModelObj->SetViewVolume(ViewType_Default, &g_RenderView.GetViewVolume(ViewType_Default));
}

// RenderModelObj にユニフォームブロックや LOD に関する設定を行います。
void SetupRenderModelObj(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    // LODレベルの判定に使用される関数を初期化
    static const float lodThresholds[] = { 1.0f, 0.4f, 0.2f, 0.1f };
    g_CalculateLodDefault = CalculateLod(&g_RenderView.GetViewMtx(ViewType_Default),
        1.0f / nn::util::TanEst(nn::util::DegreeToRadian(45.0f) * 0.5f), lodThresholds,
        sizeof(lodThresholds) / sizeof(float));

    static const float lodThresholdsWater[] = { 1.0f, 0.8f, 0.4f, 0.2f };
    g_CalculateLodWater = CalculateLod(&g_RenderView.GetViewMtx(ViewType_Water),
        1.0f / nn::util::TanEst(nn::util::DegreeToRadian(45.0f) * 0.5f), lodThresholdsWater,
        sizeof(lodThresholdsWater) / sizeof(float));

    // LOD レベル判定関数の設定
    {
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Default, &g_CalculateLodDefault);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Water, &g_CalculateLodWater);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Light0, &g_CalculateLodDefault);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Light1, &g_CalculateLodDefault);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Light2, &g_CalculateLodDefault);
        pRenderModelObj->SetCalculateLodLevelFunctor(ViewType_Light3, &g_CalculateLodDefault);
    }

    // 複数パス共通の設定
    BindUniformBlockAndSampler(pRenderModelObj);

    // 水面描画パスの設定
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(ViewType_Water, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(ViewType_Water, 1);

        pRenderModelObj->BindUniformBlock(DrawPassType_Water, "view", buffers, g_BufferingCount);
        SetupWaterVariation(pRenderModelObj);

        // フラスタムカリング用のビューボリュームを設定
        pRenderModelObj->SetViewVolume(ViewType_Water, &g_RenderView.GetViewVolume(ViewType_Water));
    }
}

// RenderModel の初期化
void CreateRenderModel(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    nn::g3d::ResShaderArchive* pModelResShaderArchive = pResFile->GetUserPtr<nn::g3d::ResShaderArchive>();
    if (!pModelResShaderArchive)
    {
        // 指定が無い場合はデモ用シェーダを使用する
        pModelResShaderArchive = g_Holder.shaderFiles[0]->GetResShaderArchive();
    }
    NN_ASSERT_NOT_NULL(pModelResShaderArchive);

    for (int modelIndex = 0, modelCount = pResFile->GetModelCount(); modelIndex < modelCount; ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);
        nns::g3d::RenderModel* pRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel, DrawPassType_Count);
        pRenderModel->CreateDrawPass(pDevice, DrawPassType_Model, pModelResShaderArchive);
        pRenderModel->CreateDrawPass(pDevice, DrawPassType_Water, pModelResShaderArchive);
        pRenderModel->CreateDrawPass(pDevice, DrawPassType_Geometry, g_pResShadingModelPtr[ShadingModelType_Geometry]);
        pRenderModel->CreateDrawPass(pDevice, DrawPassType_Shadow, g_pResShadingModelPtr[ShadingModelType_Shadow]);
        SetupRenderState(pRenderModel);
        pHolder->renderModels.PushBack(pRenderModel);
    }
}

// ModelAnimObj の初期化
void CreateModelAnimObj(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile, CreateModelInstanceCallback pCallback) NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = pResFile->GetModelCount(); modelIndex < modelCount; ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);

        // ModelAnimObj の初期化
        nns::g3d::ModelAnimObj* pModelAnimObj;
        {
            nn::g3d::ModelObj::Builder builder(pResModel);
            builder.ViewCount(ViewType_Count);
            builder.ShapeBufferingCount(g_BufferingCount);
            builder.SkeletonBufferingCount(g_BufferingCount);
            builder.MaterialBufferingCount(g_BufferingCount);
            builder.SetBoundingEnabled();
            builder.ShapeUserAreaSize(sizeof(int) * g_MaxLodLevel);
            pModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);
        }
        NN_ASSERT_NOT_NULL(pModelAnimObj);
        pHolder->modelAnimObjs.PushBack(pModelAnimObj);

        // LOD 情報を設定
        SetupShapeUserArea(pModelAnimObj->GetModelObj());

        // アニメーションIDの記録用バッファーを作成
        {
            size_t bufferSize = sizeof(int) * AnimationType_Count;
            int* pAnimId = g3ddemo::AllocateMemory<int>(bufferSize);
            for (int animationIndex = 0; animationIndex < AnimationType_Count; ++animationIndex)
            {
                pAnimId[animationIndex] = nns::g3d::ModelAnimObj::InvalidAnimId;
            }
            pHolder->animationIds.PushBack(pAnimId);
        }

        // コールバック関数が設定されていた場合、呼び出します。
        if (pCallback)
        {
            pCallback(pModelAnimObj->GetModelObj());
        }
    }
}

// RenderModelObj の初期化
void CreateRenderModelObj(g3ddemo::ResourceHolder* pHolder, nns::g3d::ModelAnimObj* pModelAnimObj, nns::g3d::RenderModel* pRenderModel) NN_NOEXCEPT
{
    NN_ASSERT(pModelAnimObj->GetModelObj()->GetResource() == pRenderModel->GetResModel());

    nns::g3d::RenderModelObj* pRenderModelObj = nns::g3d::CreateRenderModelObj(pModelAnimObj->GetModelObj(), pRenderModel);
    g3ddemo::WriteSkinningOption(pRenderModelObj);

    pRenderModelObj->BindDummySampler(g3ddemo::GetDummyTextureRef().GetDescriptorSlot(), g3ddemo::GetDummySamplerDescriptorSlot());
    pHolder->renderModelObjs.PushBack(pRenderModelObj);
}

int TownMain(const nnt::g3d::CaptureSetting* pCaptureSetting) NN_NOEXCEPT
{
    g_pCaptureSetting = const_cast<nnt::g3d::CaptureSetting*>(pCaptureSetting);
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_DeferredSubmission);
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // Townの初期化
    InitializeTown(pGfxFramework);

    int uniformBufferIndex = 0;
    pGfxFramework->SetCalculateCallback(Calculate, &uniformBufferIndex);
    pGfxFramework->SetMakeCommandCallback(MakeCommand, &uniformBufferIndex);
    pGfxFramework->ResetFrame();

    // メインループ
    while (NN_STATIC_CONDITION(1))
    {
        // 一定回数キャプチャしたら終了
        if (g_ScreenCaptureCount >= GetFrameCaptureCountMax())
        {
            break;
        }

        // フレーム処理実行
        pGfxFramework->ProcessFrame();

        uniformBufferIndex ^= 1;
        ++g_FrameCount;
    }
    pGfxFramework->QueueFinish();

    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);

    // キャプチャテストを破棄
    FinalizeScreenCapture();

    // Townの解放処理
    FinalizeTown(pGfxFramework, pDevice);

    return EXIT_SUCCESS;
}
