﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nnt/g3d/testG3d_TestUtility.h"
#include <nn/fs.h>
#include <nn/mem.h>
#include <cstdlib>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
#include <nv/nv_MemoryManagement.h>
#endif

namespace nnt {
    namespace g3d {
        namespace detail {
            namespace {
                const size_t MemoryHeapSize = 300 * 1024 * 1024;
                void* g_pHeap;
                nn::mem::StandardAllocator g_Allocator;
                void* g_MountRomCacheBuffer = nullptr;

                // Gfx の初期化を行う
                void InitializeGfx(nn::gfx::Device& device) NN_NOEXCEPT;
                // Gfx の終了時処理を行う
                void FinalizeGfx(nn::gfx::Device& device) NN_NOEXCEPT;

                // ファイルシステムの初期化を行い、ホスト PC 上のファイルシステムをマウントする
                void InitializeFileSystem() NN_NOEXCEPT;
                // ファイルシステムの終了時処理を行い、ホスト PC 上のファイルシステムをアンマウントする
                void FinalizeFileSystem() NN_NOEXCEPT;
            }
        }

        void InitializeG3dTest(nn::gfx::Device& device) NN_NOEXCEPT {
            // テスト用ヒープ領域確保
            detail::g_pHeap = malloc(detail::MemoryHeapSize);
            // nn::mem::StandardAllocator の初期化
            detail::g_Allocator.Initialize(detail::g_pHeap, detail::MemoryHeapSize);
            // ファイルシステム の初期化
            detail::InitializeFileSystem();
            // Gfx の初期化
            detail::InitializeGfx(device);
        }

        void FinalizeG3dTest(nn::gfx::Device& device) NN_NOEXCEPT {
            // ファイルシステム の終了時処理
            detail::FinalizeGfx(device);
            // nn::mem::Gfx の終了時処理
            detail::FinalizeFileSystem();
            // nn::mem::StandardAllocator の終了時処理
            detail::g_Allocator.Finalize();
            // テスト用ヒープ領域解放
            free(detail::g_pHeap);
        }

        void* LoadFile(size_t* pOutReadSize, const char* filePath, size_t alignment) NN_NOEXCEPT {
            nn::fs::FileHandle fileHandle;
            int64_t fileSize = 0;

            nn::Result result = nn::fs::OpenFile(&fileHandle, filePath, nn::fs::OpenMode_Read);
            NN_ASSERT(result.IsSuccess(), "File: %s\n", filePath);

            result = nn::fs::GetFileSize(&fileSize, fileHandle);
            NN_ASSERT(result.IsSuccess(), "File: %s\n", filePath);

            void* buffer;
            buffer = AlignedAllocate<void>(static_cast<size_t>(fileSize), alignment);

            result = nn::fs::ReadFile(pOutReadSize, fileHandle, 0, buffer, static_cast<size_t>(fileSize));
            NN_ASSERT(result.IsSuccess(), "File: %s\n", filePath);

            nn::fs::CloseFile(fileHandle);

            return buffer;
        }
        void UnloadFile(void* ptr) NN_NOEXCEPT {
            Deallocate(ptr);
        }

        void* Allocate(size_t allocateSize) NN_NOEXCEPT {
            void* buffer = detail::g_Allocator.Allocate(allocateSize);
            NN_ASSERT_NOT_NULL(buffer);
            return buffer;
        }

        void* AlignedAllocate(size_t allocateSize, size_t alignment) NN_NOEXCEPT {
            return detail::g_Allocator.Allocate(allocateSize, alignment);
        }

        void Deallocate(void* ptr) NN_NOEXCEPT {
            detail::g_Allocator.Free(ptr);
        }

        namespace detail {
            namespace {
                //------------------------------------------------------------------------------
                //  nn::fs用メモリ取得用関数
                //------------------------------------------------------------------------------
                void* AllocateForFS(size_t size) {
                    return g_Allocator.Allocate(size, 4);
                }
                //------------------------------------------------------------------------------
                //  nn::fs用メモリ解放用関数
                //------------------------------------------------------------------------------
                void DeallocateForFS(void* ptr, size_t size) {
                    NN_UNUSED(size);
                    g_Allocator.Free(ptr);
                }

                void InitializeGfx(nn::gfx::Device& device) NN_NOEXCEPT
                {
                    // gfxライブラリを初期化
                    nn::gfx::Initialize();

                    // デバイスを初期化
                    {
                        nn::gfx::Device::InfoType deviceInfoType;
                        deviceInfoType.SetDefault();
#ifdef NN_SDK_BUILD_DEBUG
                        deviceInfoType.SetDebugMode( nn::gfx::DebugMode_Enable );
#endif
                        device.Initialize(deviceInfoType);
                    }
                }

                void FinalizeGfx(nn::gfx::Device& device) NN_NOEXCEPT
                {
                    device.Finalize();

                    nn::gfx::Finalize();
                }

                void InitializeFileSystem() NN_NOEXCEPT {
                    // ファイルシステムの初期化
                    nn::fs::SetAllocator(detail::AllocateForFS, detail::DeallocateForFS);

                    // リソースをマウント
                    size_t mountRomCacheBufferSize = 0;
                    nn::Result fsResult = nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize);
                    NN_ASSERT(fsResult.IsSuccess());
                    g_MountRomCacheBuffer = Allocate(mountRomCacheBufferSize);
                    fsResult = nn::fs::MountRom("Resource", g_MountRomCacheBuffer, mountRomCacheBufferSize);
                    NN_ASSERT(fsResult.IsSuccess());

                    nn::fs::MountHostRoot();
                }

                void FinalizeFileSystem() NN_NOEXCEPT {
                    nn::fs::UnmountHostRoot();
                    nn::fs::Unmount("Resource");
                    Deallocate(g_MountRomCacheBuffer);
                    g_MountRomCacheBuffer = nullptr;
                }
            }
        }
    }
}
