﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/graphics/testGraphics_Path.h>
#include <nnt/graphics/testGraphics_Png.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <GL/glew.h>
#endif

#include "testEft_Capture.h"
#include "testEft_FrameCapture.h"
#include "testEft_CaptureParameter.h"

namespace nnt{
namespace eft{

//------------------------------------------------------------------------------
//  エフェクトビューアのコールバック関数の定義
//------------------------------------------------------------------------------
void _eftDrawPostCallback( const void* pParam )
{
    NN_UNUSED(pParam);

#if defined( NN_BUILD_CONFIG_OS_WIN )
    glBindVertexArray( 0 );
#endif
}

//------------------------------------------------------------------------------
//  コンストラクタ
//------------------------------------------------------------------------------
TestEftCapture::TestEftCapture()
{
}

//------------------------------------------------------------------------------
//  デストラクタ
//------------------------------------------------------------------------------
TestEftCapture::~TestEftCapture()
{
}

//------------------------------------------------------------------------------
//  初期化
//------------------------------------------------------------------------------
bool TestEftCapture::Initialize(
    EftRenderSystem* pRenderSystem,
    EffectSystem* pEffectSystem,
    FrameCapture* pFrameCapture )
{
    m_pRenderSystem = pRenderSystem;
    m_pEffectSystem = pEffectSystem;
    m_pFrameCapture = pFrameCapture;

    return true;
}

//------------------------------------------------------------------------------
//  解放処理
//------------------------------------------------------------------------------
void TestEftCapture::Finalize()
{
}

//------------------------------------------------------------------------------
//  実行
//------------------------------------------------------------------------------
void TestEftCapture::Run( const char* outPath, int captureCount, float captureFrame[] )
{
    nn::util::Vector3fType              gCameraPosition;            // カメラ位置
    nn::util::Vector3fType              gCameraLookAt;              // カメラ視点
    nn::util::Matrix4x3fType            gView;                      // モデルビュー
    nn::util::Vector3fType              gCamUp;
    nn::util::Vector3fType              gTarget;
    nn::util::Matrix4x4fType            gProjctionMatrix;           // プロジェクション
    uint32_t gEsetIndex = 0;

    TestCustomBehaviorParam             gTestCustomBehaviorParam;   // テストのカスタム動作パラメータ

    // カメラを初期化
    nn::util::VectorSet( &gCameraPosition, 0.0f, 30.0f, 50.0f );
    nn::util::VectorSet( &gCameraLookAt, 0.0f, 0.0f, 0.0f );

    nn::util::VectorSet( &gCamUp,  0.0f, 1.0f, 0.0f );
    nn::util::VectorSet( &gTarget,  0.0f, 0.0f, 0.0f );

    nn::util::MatrixIdentity( &gView );
    nn::util::MatrixLookAtRightHanded( &gView, gCameraPosition, gTarget, gCamUp );

    gTestCustomBehaviorParam.SetCamera( &gView, gCameraPosition, gTarget, gCamUp );

    // プロジェクションの初期化
    float fovy = nn::util::FloatPi / 3.0f;
    float width = static_cast<float>(m_pRenderSystem->GetScreenWidth());
    float height = static_cast<float>(m_pRenderSystem->GetScreenHeight());
    float aspect = width / height;
    float zNear = 0.1f;
    float zFar = 1000.0f;
    nn::util::MatrixIdentity(&gProjctionMatrix);
    nn::util::MatrixPerspectiveFieldOfViewRightHanded( &gProjctionMatrix, fovy, aspect, zNear, zFar );

    gTestCustomBehaviorParam.SetProjection( &gProjctionMatrix, fovy, aspect, zNear, zFar );

    // エミッタセットのユーザデータからテストのカスタム動作のパラメータを取得
    gTestCustomBehaviorParam.PreparCustomParam( m_pEffectSystem->GetHandle(), captureFrame[captureCount - 1] );

    int frame = 0;
    int writeCount = 0;

    if (captureCount == 0)
    {
        return;
    }

    bool first = true;
    for(;;)
    {
        gTestCustomBehaviorParam.Update( frame );

        // このフレームの描画結果をキャプチャするか
        bool capture = false;
        if ( captureFrame[writeCount] == frame )
        {
            capture = true;
        }
        // コマンドの生成
        m_pRenderSystem->ResetCommandBuffer();
        m_pRenderSystem->GetCommandBuffer()->Begin();
        {
            m_pRenderSystem->GetCommandBuffer()->InvalidateMemory( nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode );

            // ディスクリプタプールをセットする。；
            m_pRenderSystem->GetCommandBuffer()->SetDescriptorPool( m_pRenderSystem->GetTextureDescriptorPool() );
            m_pRenderSystem->GetCommandBuffer()->SetDescriptorPool( m_pRenderSystem->GetSamplerDescriptorPool() );

            nn::gfx::ColorTargetView* pTarget = m_pRenderSystem->GetColorTargetView();
            m_pRenderSystem->GetCommandBuffer()->ClearColor( pTarget, 0.0f, 0.0f, 0.0f, 1.0f, NULL );
            m_pRenderSystem->GetCommandBuffer()->ClearDepthStencil( m_pRenderSystem->GetDepthStencilView(), 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );
            m_pRenderSystem->GetCommandBuffer()->SetRenderTargets( 1, &pTarget, m_pRenderSystem->GetDepthStencilView() );
            m_pRenderSystem->GetCommandBuffer()->SetViewportScissorState( m_pRenderSystem->GetViewportScissorState() );

            // エフェクトの計算
            m_pEffectSystem->Calc();

            // エフェクト描画処理
            m_pEffectSystem->DrawEffectSystem( gView, gProjctionMatrix, *gTestCustomBehaviorParam.GetCurrentCameraPos(), 0.1f, 10000.f );


#if NN_GFX_IS_TARGET_GL
            // ※設定していないと切り替え時にエラー
            m_pRenderSystem->GetCommandBuffer()->Gl4SetUserCommand( _eftDrawPostCallback, NULL );
#endif
        }
        if(capture)
        {
            m_pFrameCapture->PushCaptureCommand(m_pRenderSystem->GetCommandBuffer(), m_pRenderSystem->GetColorBuffer(), EftRenderSystem::ColorBufferImageFormat);
        }

        // カラーバッファからスキャンバッファへコピー
        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(m_pRenderSystem->GetScreenWidth());
        region.SetHeight(m_pRenderSystem->GetScreenHeight());
        m_pRenderSystem->GetCommandBuffer()->BlitImage(m_pRenderSystem->GetCurrentSwapChain()->AcquireNextScanBuffer(),
                                                       region,
                                                       m_pRenderSystem->GetColorBuffer(),
                                                       region, 0);

        m_pRenderSystem->GetCommandBuffer()->End();

        if (!first)
        {
            // コマンドの実行
            m_pRenderSystem->GetQueue()->ExecuteCommand( m_pRenderSystem->GetCommandBuffer(), m_pRenderSystem->GetFence() );
        }

        if (first)
        {
            first = false;
        }
        else
        {
            // 結果の表示
            m_pRenderSystem->GetQueue()->Flush();
#ifndef NN_BUILD_CONFIG_OS_HORIZON
            // NOTE: 実機で垂直同期数を '0' に設定すると返ってこなくなる為、Present() を行わないようにします。
            //       ローカル環境で描画を直接確かめたい場合は '1' を設定してください。
            //       又、nvn Win 環境で Present() を行わないようにした場合は、Queue の Finalize() から返ってこなくなります。
            m_pRenderSystem->GetQueue()->Present( m_pRenderSystem->GetCurrentSwapChain(), 0 );
#endif
            m_pRenderSystem->GetFence()->Sync( nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 ) ); // きっちり描画されるまで待つために 1 秒。

#if !defined(NN_PLATFORM_CAFE)
            if(capture)
            {
                // キャプチャ画像のファイル名を設定
                nn::vfx::Resource* resource = m_pEffectSystem->GetSystem()->GetResource( m_pEffectSystem->GetResourceId() );
                nnt::graphics::Path filePath( "%s/%s_%03d.png", outPath, resource->GetEmitterSetName( gEsetIndex ), frame );
                // 書き出し
                m_pFrameCapture->FetchCaptureResult();
                m_pFrameCapture->SaveToPng( filePath.GetString() );
                writeCount++;
            }
#endif
            if (writeCount == captureCount)
            {
                m_pEffectSystem->DestroyEmitterSet();
                gEsetIndex++;
                if ( gEsetIndex == m_pEffectSystem->GetEmitterSetNum(m_pEffectSystem->GetResourceId()) )
                {
                    break;
                }
                m_pEffectSystem->CreateEmitterSet( gEsetIndex, m_pEffectSystem->GetResourceId(), 0 );
                writeCount = 0;
                frame = 0;

                first = true;

                // テストのカスタム動作の設定
                nn::util::VectorSet( &gCameraPosition, 0.0f, 30.0f, 50.0f );
                nn::util::VectorSet( &gCamUp,  0.0f, 1.0f, 0.0f );
                nn::util::VectorSet( &gTarget,  0.0f, 0.0f, 0.0f );
                gTestCustomBehaviorParam.SetCamera( &gView, gCameraPosition, gTarget, gCamUp );

                gTestCustomBehaviorParam.SetProjection( &gProjctionMatrix, fovy, aspect, zNear, zFar );

                // エミッタセットのユーザデータからカスタム動作のパラメータを取得
                gTestCustomBehaviorParam.PreparCustomParam( m_pEffectSystem->GetHandle(), captureFrame[captureCount - 1] );
            }
        }
        if ( !first ) frame++;
    }
}// NOLINT(impl/function_size)

}
}
