﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cassert>

#include <nw/eft/typeDef2.h>

#if EFT_IS_WIN
    #include <fstream>
#endif
#if EFT_IS_CAFE
    #include <cafe/fs.h>
#endif

#include "testEft_FileManager.h"

using namespace std;

#if EFT_IS_CAFE
    enum{
        MaxPathLength = 256,
    };

    #define sprintf_s sprintf
#endif


//------------------------------------------------------------------------------
//  唯一のインスタンスを取得します。
//------------------------------------------------------------------------------
FileManager& FileManager::GetInstance()
{
    static FileManager s_instance;
    return s_instance;
}


//---------------------------------------------------------------------------
//  ヒープを設定します。
//---------------------------------------------------------------------------
void FileManager::SetHeap( nw::eft2::Heap* pHeap )
{
    m_pHeap = pHeap;
}


#if EFT_IS_WIN

//------------------------------------------------------------------------------
//  ファイルからデータを読み込みます。
//------------------------------------------------------------------------------
bool FileManager::ReadFile ( const char* pPath, u8** ppData, u32* pSize )
{
    *ppData = NULL;
    *pSize = 0;

    // ファイルを開く
    std::ifstream fp( pPath, std::ios::binary );
    if ( fp.fail() ) return false;

    u32 size;

    // ファイルサイズを取得
    fp.seekg( 0L, std::ios::end );
    size = static_cast<u32>( fp.tellg() );

    void* pData;

    // ファイルサイズのメモリを確保
    pData = m_pHeap->Alloc( size );

    // ファイルを先頭から読み込む
    fp.seekg( 0L, std::ios::beg );
    fp.read( static_cast<char *>( pData ), size );

    if ( fp.bad() )
    {
        m_pHeap->Free( pData );
        return false;
    }

    *ppData = static_cast<u8*>( pData );
    *pSize = size;

    return true;
}

#endif
#if EFT_IS_CAFE

//------------------------------------------------------------------------------
//  ファイルからデータを読み込みます。
//------------------------------------------------------------------------------
bool FileManager::ReadFile ( const char* pPath, u8** ppData, u32* pSize )
{
    *ppData = NULL;
    *pSize = 0;

    // ファイルパスをHFIOパスに変換
    char hfioPath[MaxPathLength];
    sprintf( hfioPath, "/vol/hfio01/%s", pPath );
    hfioPath[13] = '\\';  // ':' を '\\' に置換

    // FSクライアントを作成
    FSClient fsClient;
    FSAddClient( &fsClient, FS_RET_NO_ERROR );

    FSStateChangeParams params;
    params.userCallback = NULL;
    params.userContext  = NULL;
    params.ioMsgQueue   = NULL;

    FSSetStateChangeNotification( &fsClient, &params );

    // コマンドブロックを初期化
    FSCmdBlock cmd;
    FSInitCmdBlock( &cmd );

    // マウントソースを検索
    FSMountSource mountSrc;
    FSStatus st = FSGetMountSource( &fsClient, &cmd, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR );
//    OSReport("!! status : %d\n", st);

    // HFIOをマウント
    char mountPath[FS_MAX_MOUNTPATH_SIZE];
    FSMount( &fsClient, &cmd, &mountSrc, mountPath, sizeof(mountPath), FS_RET_NO_ERROR );
//    OSReport("!! mount path : %s\n", mountPath);

    // ファイル情報を取得
    FSStat fileStat;
    FSGetStat( &fsClient, &cmd, hfioPath, &fileStat, FS_RET_NO_ERROR );

    // IO Buffer用にアライメントを調整したバッファを作成
    u32 size = fileStat.size;
    u8* pBuffer = static_cast<u8*>( m_pHeap->Alloc( size, FS_IO_BUFFER_ALIGN ) );

    // ファイルを開く
    FSFileHandle fileHandle;
    FSOpenFile( &fsClient, &cmd, hfioPath, "r", &fileHandle, FS_RET_NO_ERROR );

    // ファイルからデータを読み込む
    FSReadFile( &fsClient, &cmd, pBuffer, size, 1, fileHandle, 0, FS_RET_NO_ERROR );

    // ファイルを閉じる
    FSCloseFile( &fsClient, &cmd, fileHandle, FS_RET_NO_ERROR );

    // HFIOをアンマウント
    FSUnmount( &fsClient, &cmd, mountPath, FS_RET_NO_ERROR );

    // FSクライアントを破棄
    FSDelClient( &fsClient, FS_RET_NO_ERROR );

    *ppData = pBuffer;
    *pSize = size;

    return true;
}

#endif


//------------------------------------------------------------------------------
//  ファイルから読み込んだデータを破棄します。
//------------------------------------------------------------------------------
void FileManager::ReleaseData ( u8* pData )
{
    m_pHeap->Free( pData );
}


#if EFT_IS_WIN

//------------------------------------------------------------------------------
//  データをファイルに保存します。
//------------------------------------------------------------------------------
bool FileManager::WriteFile( const u8* pData, u32 size, const char* pPath )
{
    // ファイルを開く
    ofstream ofs;
    ofs.open( pPath, ios::out | ios::binary );

    // データを保存する
    ofs.write( reinterpret_cast<const char*>( pData ), size );

    // ファイルを閉じる
    ofs.close();

    return true;
}

#endif
#if EFT_IS_CAFE

//------------------------------------------------------------------------------
//  データをファイルに保存します。
//------------------------------------------------------------------------------
bool FileManager::WriteFile( const u8* pData, u32 size, const char* pPath)
{
    // ファイルパスをHFIOパスに変換
    char hfioPath[MaxPathLength];
    sprintf(hfioPath, "/vol/hfio01/%s", pPath);
    hfioPath[13] = '\\';  // ':' を '\\' に置換

    // IO Buffer用にアライメントを調整したバッファを作成
    u8* pAlignedBuffer = static_cast<u8*>( m_pHeap->Alloc( size, FS_IO_BUFFER_ALIGN ) );

    memcpy( pAlignedBuffer, pData, size );

    // FSクライアントを作成
    FSClient fsClient;
    FSAddClient(&fsClient, FS_RET_NO_ERROR);

    FSStateChangeParams params;
    params.userCallback = NULL;
    params.userContext  = NULL;
    params.ioMsgQueue   = NULL;

    FSSetStateChangeNotification(&fsClient, &params);

    // コマンドブロックを初期化
    FSCmdBlock cmd;
    FSInitCmdBlock(&cmd);

    // マウントソースを検索
    FSMountSource mountSrc;
    FSStatus st = FSGetMountSource(&fsClient, &cmd, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR);
//    OSReport("!! status : %d\n", st);

    // HFIOをマウント
    char mountPath[FS_MAX_MOUNTPATH_SIZE];
    FSMount(&fsClient, &cmd, &mountSrc, mountPath, sizeof(mountPath), FS_RET_NO_ERROR);
//    OSReport("!! mount path : %s\n", mountPath);

    // ファイルを開く
    FSFileHandle fileHandle;
    FSOpenFile(&fsClient, &cmd, hfioPath, "w", &fileHandle, FS_RET_NO_ERROR);

    // ファイルにデータを書き込む
    FSWriteFile(&fsClient, &cmd, pAlignedBuffer, size, 1, fileHandle, 0, FS_RET_NO_ERROR);

    // ファイルを閉じる
    FSCloseFile(&fsClient, &cmd, fileHandle, FS_RET_NO_ERROR);

    // HFIOをアンマウント
    FSUnmount(&fsClient, &cmd, mountPath, FS_RET_NO_ERROR);

    // FSクライアントを破棄
    FSDelClient(&fsClient, FS_RET_NO_ERROR);

    // バッファを破棄
    m_pHeap->Free( pAlignedBuffer );

    return true;
}

#endif
