﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEft_HandlingCommand.h"
#include "testEft_HandlingTestSet.h"


//------------------------------------------------------------------------------
// HandlingCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
HandlingCommand::HandlingCommand( int exeFrame )
    : m_exeFrame( exeFrame )
{
}


//------------------------------------------------------------------------------
//  コマンド開始フレーム番号を取得します。
//------------------------------------------------------------------------------
int HandlingCommand::GetEndFrame() const
{
    return m_exeFrame;
}


//------------------------------------------------------------------------------
//  コマンド終了フレーム番号を取得します。
//------------------------------------------------------------------------------
int HandlingCommand::GetBeginFrame() const
{
    return m_exeFrame;
}


//------------------------------------------------------------------------------
//  コマンドを実行するかどうか取得します。
//------------------------------------------------------------------------------
bool HandlingCommand::DoExecute( int numFrame ) const
{
    return ( numFrame == m_exeFrame );
}


//------------------------------------------------------------------------------
// CreateEsetCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
CreateEsetCommand::CreateEsetCommand( int exeFrame, uint32_t esetId )
    : HandlingCommand( exeFrame ), m_esetId( esetId )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void CreateEsetCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::System* pEffectSystem = pTest->GetEffectSystem();
    nn::vfx::Handle* pEsetHandle = pTest->GetEsetHandle();

    bool resCreate = pEffectSystem->CreateEmitterSetId( pEsetHandle, m_esetId, 0, 0 );
    NN_SDK_ASSERT(resCreate == true);

#ifndef _DEBUG
    NN_UNUSED(resCreate);
#endif
}


//------------------------------------------------------------------------------
// KillEsetCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
KillEsetCommand::KillEsetCommand( int exeFrame, nn::vfx::EmitterSet* pEmitterSet )
    : HandlingCommand( exeFrame ), m_pEmitterSet( pEmitterSet ), m_pEmitterSetName(), m_resId()
{
}


//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
KillEsetCommand::KillEsetCommand( int exeFrame, const char* pEmitterSetName, uint32_t resId )
    : HandlingCommand( exeFrame ), m_pEmitterSet(), m_pEmitterSetName( pEmitterSetName ), m_resId( resId )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void KillEsetCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::System* pEffectSystem = pTest->GetEffectSystem();

    if (m_pEmitterSet != NULL)
    {
        pEffectSystem->KillEmitterSet( m_pEmitterSet );
    }
    else
    {
        pEffectSystem->KillEmitterSet( m_pEmitterSetName, m_resId );
    }
}


//------------------------------------------------------------------------------
// KillAllEsetCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
KillAllEsetCommand::KillAllEsetCommand( int exeFrame )
    : HandlingCommand( exeFrame )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void KillAllEsetCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::System* pEffectSystem = pTest->GetEffectSystem();

    pEffectSystem->KillAllEmitterSet();
}


//------------------------------------------------------------------------------
// FadeCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
FadeCommand::FadeCommand( int exeFrame )
    : HandlingCommand( exeFrame )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void FadeCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->Fade();
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetAllDirectionalVelCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetAddVelCommand::SetAddVelCommand( int exeFrame, const nn::util::Vector3fType& velocity )
    : HandlingCommand( exeFrame ), m_velocity( velocity )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetAddVelCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetAddVel( m_velocity );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetAllDirectionalVelCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetAllDirectionalVelCommand::SetAllDirectionalVelCommand( int exeFrame, float scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetAllDirectionalVelCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetAllDirectionalVel( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetAlphaCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetAlphaCommand::SetAlphaCommand( int exeFrame, float alpha )
    : HandlingCommand( exeFrame ), m_alpha( alpha )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetAlphaCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetAlpha( m_alpha );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetCalcEnableCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetCalcEnableCommand::SetCalcEnableCommand( int exeFrame, bool enabled )
    : HandlingCommand( exeFrame ), m_enabled( enabled )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetCalcEnableCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetCalculationEnable( m_enabled );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetColorCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetColorCommand::SetColorCommand( int exeFrame, const nn::util::Vector4fType& color )
    : HandlingCommand( exeFrame ), m_color( color )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetColorCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetColor( m_color );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetDirectionalCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetDirectionalCommand::SetDirectionalCommand( int exeFrame, const nn::util::Vector3fType& direction )
    : HandlingCommand( exeFrame ), m_direction( direction )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetDirectionalCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetDirectional( m_direction );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetDirectionalVelCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetDirectionalVelCommand::SetDirectionalVelCommand( int exeFrame, float scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetDirectionalVelCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetDirectionalVel( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetDrawPriorityCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetDrawPriorityCommand::SetDrawPriorityCommand( int exeFrame, unsigned char priority )
    : HandlingCommand( exeFrame ), m_priority( priority )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetDrawPriorityCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetDrawPriority( m_priority );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmissionInternalScaleCommand

//------------------------------------------------------------------------------
//  コストラクタです。
//------------------------------------------------------------------------------
SetEmissionIntervalScaleCommand::SetEmissionIntervalScaleCommand( int exeFrame, float scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmissionIntervalScaleCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmissionIntervalScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmissionParticleScaleCommand

//------------------------------------------------------------------------------
//  コストラクタです。
//------------------------------------------------------------------------------
SetEmissionParticleScaleCommand::SetEmissionParticleScaleCommand( int exeFrame, const nn::util::Float2& scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmissionParticleScaleCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmissionParticleScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmissionRatioScaleCommand

//------------------------------------------------------------------------------
//  コストラクタです。
//------------------------------------------------------------------------------
SetEmissionRatioScaleCommand::SetEmissionRatioScaleCommand( int exeFrame, float scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmissionRatioScaleCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmissionRatioScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmitterColorCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetEmitterColorCommand::SetEmitterColorCommand( int exeFrame, const nn::util::Vector4fType& color0, const nn::util::Vector4fType& color1 )
    : HandlingCommand( exeFrame ), m_color0( color0 ), m_color1( color1 )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmitterColorCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmitterColor( nn::util::Vector4fType( m_color0 ), nn::util::Vector4fType( m_color1 ) );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmitterScaleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetEmitterScaleCommand::SetEmitterScaleCommand( int exeFrame, const nn::util::Vector3fType& scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmitterScaleCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmitterVolumeScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmitterVisibleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetEmitterVisibleCommand::SetEmitterVisibleCommand( int exeFrame, uint32_t emitterIndex, bool visible )
    : HandlingCommand( exeFrame ), m_emitterIndex( emitterIndex ), m_visible( visible )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmitterVisibleCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        const nn::vfx::Emitter* pEmitter = pEset->GetAliveEmitter( m_emitterIndex );
        if ( pEmitter != NULL )
        {
            const char* pEmitterName = pEmitter->GetName();
            if ( pEmitterName != NULL )
            {
                pEset->SetEmitterVisible( pEmitterName, m_visible );
            }
        }
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetInitRotateCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetInitRotateCommand::SetInitRotateCommand( int exeFrame, const nn::util::Vector3fType& rotate )
    : HandlingCommand( exeFrame ), m_rotate( rotate )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetInitRotateCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetInitRotate( m_rotate );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetMatrixCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetMatrixCommand::SetMatrixCommand( int exeFrame, const nn::util::Matrix4x3fType& matrix )
    : HandlingCommand( exeFrame ), m_matrix( matrix )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetMatrixCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetMatrix( m_matrix );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetMatrixFastCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetMatrixFastCommand::SetMatrixFastCommand( int exeFrame, const nn::util::Matrix4x3fType& matrix, const nn::util::Vector3fType& scale )
    : HandlingCommand( exeFrame ), m_matrix( matrix ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetMatrixFastCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetMatrixFast( m_matrix, m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetParticleLifeScaleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetParticleLifeScaleCommand::SetParticleLifeScaleCommand( int exeFrame, float scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetParticleLifeScaleCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetParticleLifeScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetParticleScaleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetParticleScaleCommand::SetParticleScaleCommand( int exeFrame, const nn::util::Float2& scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetParticleScaleCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetParticleScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetRandomVelCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetRandomVelCommand::SetRandomVelCommand( int exeFrame, float scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetRandomVelCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetRandomVel( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetRuntimeUserDataCommand

//------------------------------------------------------------------------------
//  コストラクタです。
//------------------------------------------------------------------------------
SetRuntimeUserDataCommand::SetRuntimeUserDataCommand( int exeFrame, void* data )
    : HandlingCommand( exeFrame ), m_data( data )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetRuntimeUserDataCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetRuntimeUserPtr( m_data );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetStartFrameCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetStartFrameCommand::SetStartFrameCommand( int exeFrame, int frame )
    : HandlingCommand( exeFrame ), m_frame( frame )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetStartFrameCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetStartFrame( m_frame );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetVisibleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetVisibleCommand::SetVisibleCommand( int exeFrame, bool visible )
    : HandlingCommand( exeFrame ), m_visible( visible )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetVisibleCommand::Execute( HandlingTestSet* pTest )
{
    nn::vfx::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetVisible( m_visible );
        pEset = pEset->GetNext();
    }
}
