﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ScreenCapture.h"


//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
ScreenCapture::ScreenCapture()
    : m_pHeap(), m_pFrameBuffer(), m_width(), m_height(), m_pRgbPixels()
{
#if EFT_IS_CAFE
    m_pColorConvertPixels = NULL;
    m_pColorConvertBuffer = NULL;

    m_pTilingConvertPixels = NULL;
    m_pTilingConvertBuffer = NULL;
#endif
}


//------------------------------------------------------------------------------
//  デストラクタです。
//------------------------------------------------------------------------------
ScreenCapture::~ScreenCapture()
{
#if EFT_IS_CAFE
    assert( m_pColorConvertPixels == NULL );
    assert( m_pColorConvertBuffer == NULL );

    assert( m_pTilingConvertPixels == NULL );
    assert( m_pTilingConvertBuffer == NULL );

    assert( m_pRgbPixels == NULL );
#endif
}


//------------------------------------------------------------------------------
//  初期化処理です。
//------------------------------------------------------------------------------
void ScreenCapture::Initialize( TestHeap* pHeap, nw::eftdemo::FrameBuffer* pFrameBuffer )
{
    m_pHeap = pHeap;
    m_pFrameBuffer = pFrameBuffer;

    m_width  = static_cast<int>( m_pFrameBuffer->GetSize().x );
    m_height = static_cast<int>( m_pFrameBuffer->GetSize().y );

    // RGBデータを作成
    m_pRgbPixels = static_cast<u8*>( pHeap->Alloc( m_width * m_height * 3 ) );

#if EFT_IS_CAFE
    // 色変換バッファを作成
    m_pColorConvertBuffer = static_cast<GX2ColorBuffer*>( pHeap->Alloc( sizeof( GX2ColorBuffer ) ) );
    GX2InitColorBuffer( m_pColorConvertBuffer, m_width, m_height, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_AA_MODE_1X );

    m_pColorConvertPixels = static_cast<u8*>( pHeap->Alloc( m_pColorConvertBuffer->surface.imageSize, m_pColorConvertBuffer->surface.alignment ) );
    GX2InitColorBufferPtr( m_pColorConvertBuffer, m_pColorConvertPixels );

    // タイリング変換バッファを作成
    m_pTilingConvertBuffer = static_cast<GX2ColorBuffer*>( pHeap->Alloc( sizeof( GX2ColorBuffer ) ) );
    GX2InitColorBuffer( m_pTilingConvertBuffer, m_width, m_height, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_AA_MODE_1X );

    m_pTilingConvertPixels = static_cast<u8*>( pHeap->Alloc( m_pTilingConvertBuffer->surface.imageSize, m_pTilingConvertBuffer->surface.alignment ) );
    GX2InitColorBufferPtr( m_pTilingConvertBuffer, m_pTilingConvertPixels );

    m_pTilingConvertBuffer->surface.tileMode = GX2_TILE_MODE_LINEAR_SPECIAL;
#endif
}


//------------------------------------------------------------------------------
//  破棄処理です。
//------------------------------------------------------------------------------
void ScreenCapture::Release()
{
    // RGBデータを破棄
    m_pHeap->Free( m_pRgbPixels );
    m_pRgbPixels = NULL;

#if EFT_IS_CAFE
    // 色変換バッファを破棄
    m_pHeap->Free( m_pColorConvertBuffer );
    m_pColorConvertBuffer = NULL;

    m_pHeap->Free( m_pColorConvertPixels );
    m_pColorConvertPixels = NULL;

    // タイリング変換バッファを破棄
    m_pHeap->Free( m_pTilingConvertBuffer );
    m_pTilingConvertBuffer = NULL;

    m_pHeap->Free( m_pTilingConvertPixels );
    m_pTilingConvertPixels = NULL;
#endif
}


//---------------------------------------------------------------------------
//  キャプチャを行います。
//---------------------------------------------------------------------------
void ScreenCapture::Capture()
{
#if EFT_IS_WIN
    glPixelStorei( GL_PACK_ALIGNMENT, 1 );
    glReadPixels( 0, 0, (GLsizei)m_width, (GLsizei)m_height, GL_RGB, GL_UNSIGNED_BYTE, m_pRgbPixels );
#endif
#if EFT_IS_CAFE
    // レンダーステートの設定
    GX2SetAlphaTest(GX2_DISABLE, GX2_COMPARE_LESS, 0.0f);
    GX2SetColorControl( GX2_LOGIC_OP_COPY, 0, GX2_DISABLE, GX2_ENABLE );

    // うまくキャプチャされないときはレンダーステートをデフォルトにして試してください
    // GX2SetDefaultState();

    // キャッシュをフラッシュ
    // フラッシュしないとキャプチャ画像にノイズがのる場合がある
    GX2InvalidateType type = GX2InvalidateType(GX2_INVALIDATE_COLOR_BUFFER | GX2_INVALIDATE_CPU);
    GX2Invalidate(type, m_pColorConvertBuffer->surface.imagePtr, m_pColorConvertBuffer->surface.imageSize);
    GX2Invalidate(type, m_pTilingConvertBuffer->surface.imagePtr, m_pTilingConvertBuffer->surface.imageSize);

    // フレームバッファのカラーフォーマットを変換
    GX2ColorBuffer* pFrameColorBuffer = m_pFrameBuffer->GetGX2ColorBuffer();
    GX2UTCopySurface( &pFrameColorBuffer->surface, 0, 0, &m_pColorConvertBuffer->surface, 0, 0 );

    GX2DrawDone();

    // フレームバッファのタイリングを解除
    GX2CopySurface( &m_pColorConvertBuffer->surface, 0, 0, &m_pTilingConvertBuffer->surface, 0, 0 );

    // RGBAデータをRGBデータに変換
    for (int y = 0; y < m_height; ++y)
    {
        for (int x = 0; x < m_width; ++x)
        {
            int srcIndex = y * m_width * 4 + x * 4;
            int dstIndex = y * m_width * 3 + x * 3;

            m_pRgbPixels[dstIndex + 0] = m_pTilingConvertPixels[srcIndex + 0];
            m_pRgbPixels[dstIndex + 1] = m_pTilingConvertPixels[srcIndex + 1];
            m_pRgbPixels[dstIndex + 2] = m_pTilingConvertPixels[srcIndex + 2];
        }
    }
#endif
}


//---------------------------------------------------------------------------
//  キャプチャ画像の横幅を取得します。
//---------------------------------------------------------------------------
int ScreenCapture::GetWidth() const
{
    return m_width;
}


//---------------------------------------------------------------------------
//  キャプチャ画像の縦幅を取得します。
//---------------------------------------------------------------------------
int ScreenCapture::GetHeight() const
{
    return m_height;
}


//---------------------------------------------------------------------------
//  ピクセルデータを取得します。
//---------------------------------------------------------------------------
const u8* ScreenCapture::GetPixels() const
{
    return m_pRgbPixels;
}
