﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <algorithm>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>
#include <nnt/nnt_Argument.h>
#include <nnt/graphics/testGraphics_GetHostExecutableFilepath.h>

#include <nn/time/time_Api.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_LocationName.h>


// Siglo 環境での警告抑止用
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif

#include <rapidjson/document.h>
#include <rapidjson/writer.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(pop)
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN32
#include <nn/nn_Windows.h>
#endif // NN_BUILD_CONFIG_OS_WIN32

#include "testGraphics_Path.h"
#include "testGraphics_ProfileDataComparer.h"

struct ProgramArgs
{
    enum UpdateTimeStampTarget
    {
        UpdateTimeStampTarget_Lhs = 1 << 0,
        UpdateTimeStampTarget_Rhs = 1 << 1,
        UpdateTimeStampTarget_Max
    };

    void SetDefault()
    {
        InputCount = 0;
        CpuLoadTolerance = 0.05f;
        GpuLoadTolerance = 0.05f;
        MemoryUsageTolerance = 0.02f;
        InitializationLoadTolerance = 0.05f;
        CompareFrameValueMode = false;
        ResultAverageMode = false;
    }

    nnt::graphics::Path     Inputs[2];
    int                     InputCount;
    nnt::graphics::Path     OutputPath;
    nnt::graphics::Path     OutputHtmlTemplatePath;
    nnt::graphics::Path     SearchProfileHistoryPath;
    char    FilePrefixForSearchAndCompare[_MAX_PATH];
    float   CpuLoadTolerance;
    float   GpuLoadTolerance;
    float   MemoryUsageTolerance;
    float   InitializationLoadTolerance;
    bool    CompareFrameValueMode;
    bool    ResultAverageMode;
    uint32_t    UpdateTimeStampTargetFlag;
};

#ifdef NN_BUILD_CONFIG_OS_WIN32
// タイムゾーン情報をレジストリから読み込むための構造体。
//  https://msdn.microsoft.com/en-us/library/ms724253.aspx
typedef struct _REG_TZI_FORMAT
{
    LONG Bias;
    LONG StandardBias;
    LONG DaylightBias;
    SYSTEMTIME StandardDate;
    SYSTEMTIME DaylightDate;
} REG_TZI_FORMAT;
#endif // NN_BUILD_CONFIG_OS_WIN32

//----------------------------------------------------------------------------------------------------
void CreateDirectories(const nnt::graphics::Path* path)
{
    if(path->IsRoot())
    {
        return;
    }
    nn::Result result = nn::fs::CreateDirectory(path->GetString());
    if(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result))
    {
        return;
    }
    else if(nn::fs::ResultPathNotFound::Includes(result))
    {
        nnt::graphics::Path parent;
        path->GetParent(&parent);
        CreateDirectories(&parent);
        result = nn::fs::CreateDirectory(path->GetString());
        NN_ASSERT(result.IsSuccess());
    }
    else
    {
        NN_ASSERT(result.IsSuccess());
    }
}

ProgramArgs g_Args;

//----------------------------------------------------------------------------------------------------
static void* FsAllocate(size_t s)
{
    return malloc(s);
}

//----------------------------------------------------------------------------------------------------
static void FsDeallocate(void* p, size_t)
{
    free(p);
}

//----------------------------------------------------------------------------------------------------
static void* JsonAllocator(size_t size)
{
    return malloc(size);
}

//----------------------------------------------------------------------------------------------------
static void JsonDeallocator(void* ptr, size_t size)
{
    NN_UNUSED(size);
    free(ptr);
}

//----------------------------------------------------------------------------------------------------
//  指定されたパスがディレクトリかどうか判定します。
bool IsDirectory(nnt::graphics::Path&   path)
{
    nn::fs::DirectoryEntryType  type;
    if (nn::fs::GetEntryType(&type, path.GetString()).IsSuccess())
    {
        return type == nn::fs::DirectoryEntryType::DirectoryEntryType_Directory;
    }

    return false;
}

//----------------------------------------------------------------------------------------------------
// 必要なサイズのメモリを malloc で確保して Json データを読み込みます。
void JsonFileReadMalloc(char** ppOutFileImage, size_t& size, const char* pFilePath)
{
    nn::Result result;
    nn::fs::FileHandle file = {};
    int64_t fileSize = 0;
    result = nn::fs::OpenFile(&file, pFilePath, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());
    result= nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(result.IsSuccess());
    size = static_cast<uint32_t>(fileSize);
    *ppOutFileImage = reinterpret_cast<char*>(malloc(size + 1));
    NN_ASSERT_NOT_NULL(*ppOutFileImage);
    result = nn::fs::ReadFile(file, 0, *ppOutFileImage, size);
    NN_ASSERT(result.IsSuccess());
    // テキストとして扱えるように終端文字を詰めておく。
    static_cast<char*>(*ppOutFileImage)[size] = '\0';
    nn::fs::CloseFile(file);
}

//----------------------------------------------------------------------------------------------------
// 指定されたディレクトリ内の要素を取得します。
bool ReadDirectory(nn::fs::DirectoryEntry** ppOutEntries, int64_t& entryCount, nnt::graphics::Path& inputPath)
{
    if (IsDirectory(inputPath))
    {
        nn::Result result;
        nn::fs::DirectoryHandle dirHandle;
        result = nn::fs::OpenDirectory(&dirHandle, inputPath.GetString(), nn::fs::OpenDirectoryMode_File);

        result = nn::fs::GetDirectoryEntryCount(&entryCount, dirHandle);

        *ppOutEntries = static_cast<nn::fs::DirectoryEntry*>(malloc(sizeof(nn::fs::DirectoryEntry) * static_cast<size_t>(entryCount)));

        result = nn::fs::ReadDirectory(&entryCount, *ppOutEntries, dirHandle, entryCount);

        nn::fs::CloseDirectory(dirHandle);

        return true;
    }

    return false;
}

//----------------------------------------------------------------------------------------------------
// ディレクトリ内で指定された Prefix を持つファイルを検索して
// ファイル名に埋め込まれている日付から最新のものを取得します。
void GetNewestJsonFilePath(nnt::graphics::Path& decidedPath, nnt::graphics::Path& inputPath, const char* pFilePrefix)
{
    // ディレクトリなので最新のデータを検索する。
    nn::fs::DirectoryEntry* pDirEntries = NULL;
    int64_t entryCount = 0;

    if (ReadDirectory(&pDirEntries, entryCount, inputPath))
    {
        int64_t newestIndex = 0;

        // ファイル名にエンコードされた日付から最新のファイルを特定する。
        for(int64_t i = 0; i < entryCount; i++)
        {
            int newestDate = 0;
            int newestTime = 0;
            nnt::graphics::Path filePath(pDirEntries[i].name);
            char*   pFileName = filePath.GetFilename().GetString();

            size_t  fileNameLength = strlen(pFileName);
            size_t  prefixLength = strlen(pFilePrefix);
            // Prefix + YYYYMMDD_MMHHSS のフォーマットになっているかチェックする。
            // YYYYMMDD_MMHHSS.json で 20 文字。
            if (fileNameLength == (prefixLength + 20) &&
                strncmp(pFileName, pFilePrefix, prefixLength) == 0)
            {
                const char*   pDate = pFileName + prefixLength;
                const char*   pTime = pFileName + prefixLength + 8 + 1;

                // YYYYMMDD の部分を数値に変換
                int date = atoi(pDate);
                // MMHHSS の部分を数値に変換
                int time = atoi(pTime);

                if (date > newestDate ||
                    (date == newestDate &&
                     time > newestTime))
                {
                    newestDate = date;
                    newestTime = time;
                    newestIndex = i;
                }
            }
        }

        decidedPath.SetString(inputPath.Combine(nnt::graphics::Path(pDirEntries[newestIndex].name)).GetString());
        free(pDirEntries);
    }
    // ファイル名指定なので読み込み。
    else
    {
        decidedPath.SetString(inputPath.GetString());
    }
}

//----------------------------------------------------------------------------------------------------
// ディレクトリ内で指定された Prefix を持つ全てのファイルの平均を取った 「Prefix日付_Average.json」 ファイルを作成して取得します。
void GetAveragedJsonFilePath(nnt::graphics::Path& decidedPath, nnt::graphics::Path& inputPath, const char* pFilePrefix)
{
    // ディレクトリなので最新のデータを検索する。
    nn::fs::DirectoryEntry* pDirEntries = NULL;
    int64_t entryCount = 0;
    int profiledCount = 0;

    // 結果出力ファイルの初期化
    nnt::graphics::PerformanceProfileData  averageProfile;
    averageProfile.Initialize(1024, FsAllocate, FsDeallocate);
    char    filename[_MAX_PATH];
    sprintf(filename, "%sAverage", g_Args.FilePrefixForSearchAndCompare);
    averageProfile.SetName(filename);

    nn::time::PosixTime posixTime;
    nn::time::CalendarTime  calendarTime;
    nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    nn::time::ToCalendarTime(&calendarTime, NULL, posixTime);
    char timeString[32];
    nn::util::SNPrintf(timeString, 32,
        "%04d-%02d-%02d %02d:%02d:%02d",
        calendarTime.year, calendarTime.month, calendarTime.day,
        calendarTime.hour, calendarTime.minute, calendarTime.second);

    // 計測した日付を設定。
    averageProfile.SetDate(timeString);

    int64_t cpuLoadSum[1024] = { 0 };
    int64_t gpuLoadSum[1024] = { 0 };
    int64_t memoryUsageSum[1024] = { 0 };
    int64_t initializationLoadSum = 0;

    int64_t cpuLoadMax = 0;
    int64_t cpuLoadMin = LLONG_MAX;
    int64_t gpuLoadMax = 0;
    int64_t gpuLoadMin = LLONG_MAX;
    int64_t memoryUsageMax = 0;
    int64_t memoryUsageMin = LLONG_MAX;
    int64_t initializationLoadMax = 0;
    int64_t initializationLoadMin = LLONG_MAX;

    int cpuValueCountMax = 0;
    int gpuValueCountMax = 0;
    int memoryUsageValueCountMax = 0;

    if (ReadDirectory(&pDirEntries, entryCount, inputPath))
    {
        for (int64_t i = 0; i < entryCount; i++)
        {
            nnt::graphics::Path filePath(pDirEntries[i].name);
            char*   pFileName = filePath.GetFilename().GetString();

            size_t  fileNameLength = strlen(pFileName);
            size_t  prefixLength = strlen(pFilePrefix);
            // Prefix + YYYYMMDD_MMHHSS のフォーマットになっているかチェックする。
            // YYYYMMDD_MMHHSS.json で 20 文字。
            if (fileNameLength == (prefixLength + 20) &&
                strncmp(pFileName, pFilePrefix, prefixLength) == 0)
            {
                char* pJsonFile = NULL;
                size_t  fileSize = 0;

                JsonFileReadMalloc(&pJsonFile, fileSize, inputPath.Combine(nnt::graphics::Path(filePath)).GetString());

                if (pJsonFile != NULL)
                {
                    nnt::graphics::PerformanceProfileData  profile;
                    if (profile.Initialize(pJsonFile, JsonAllocator, JsonDeallocator))
                    {
                        cpuValueCountMax = std::max(cpuValueCountMax, profile.GetFrameValueCount(nnt::graphics::PerformanceProfileData::FrameValueType_CpuLoad));
                        gpuValueCountMax = std::max(gpuValueCountMax, profile.GetFrameValueCount(nnt::graphics::PerformanceProfileData::FrameValueType_GpuLoad));
                        memoryUsageValueCountMax = std::max(memoryUsageValueCountMax, profile.GetFrameValueCount(nnt::graphics::PerformanceProfileData::FrameValueType_MemoryUsage));
                        cpuLoadMax = std::max(cpuLoadMax, profile.GetAverageValue(nnt::graphics::PerformanceProfileData::FrameValueType_CpuLoad));
                        cpuLoadMin = std::min(cpuLoadMin, profile.GetAverageValue(nnt::graphics::PerformanceProfileData::FrameValueType_CpuLoad));
                        gpuLoadMax = std::max(gpuLoadMax, profile.GetAverageValue(nnt::graphics::PerformanceProfileData::FrameValueType_GpuLoad));
                        gpuLoadMin = std::min(gpuLoadMin, profile.GetAverageValue(nnt::graphics::PerformanceProfileData::FrameValueType_GpuLoad));
                        memoryUsageMax = std::max(memoryUsageMax, profile.GetAverageValue(nnt::graphics::PerformanceProfileData::FrameValueType_MemoryUsage));
                        memoryUsageMin = std::min(memoryUsageMin, profile.GetAverageValue(nnt::graphics::PerformanceProfileData::FrameValueType_MemoryUsage));
                        initializationLoadMax = std::max(initializationLoadMax, profile.GetInitializationLoadValue());
                        initializationLoadMin = std::min(initializationLoadMin, profile.GetInitializationLoadValue());

                        // 全てのファイルの値を加算していく
                        for (int frameCount = 0; frameCount < profile.GetFrameValueCount(nnt::graphics::PerformanceProfileData::FrameValueType_CpuLoad); ++frameCount)
                        {
                            cpuLoadSum[frameCount] += profile.GetFrameValues(nnt::graphics::PerformanceProfileData::FrameValueType_CpuLoad)[frameCount].GetValue();
                            NN_LOG("Cpu = %lld\n", profile.GetFrameValues(nnt::graphics::PerformanceProfileData::FrameValueType_CpuLoad)[frameCount].GetValue());
                        }
                        for (int frameCount = 0; frameCount < profile.GetFrameValueCount(nnt::graphics::PerformanceProfileData::FrameValueType_GpuLoad); ++frameCount)
                        {
                            gpuLoadSum[frameCount] += profile.GetFrameValues(nnt::graphics::PerformanceProfileData::FrameValueType_GpuLoad)[frameCount].GetValue();
                            NN_LOG("Gpu = %lld\n", profile.GetFrameValues(nnt::graphics::PerformanceProfileData::FrameValueType_GpuLoad)[frameCount].GetValue());
                        }
                        for (int frameCount = 0; frameCount < profile.GetFrameValueCount(nnt::graphics::PerformanceProfileData::FrameValueType_MemoryUsage); ++frameCount)
                        {
                            memoryUsageSum[frameCount] += profile.GetFrameValues(nnt::graphics::PerformanceProfileData::FrameValueType_MemoryUsage)[frameCount].GetValue();
                            NN_LOG("MemoryUsage = %lld\n", profile.GetFrameValues(nnt::graphics::PerformanceProfileData::FrameValueType_MemoryUsage)[frameCount].GetValue());
                        }
                        initializationLoadSum += profile.GetInitializationLoadValue();
                        NN_LOG("InitializationLoad = %lld\n", profile.GetInitializationLoadValue());

                        profiledCount++;
                    }
                    profile.Finalize();
                }
                if (pJsonFile != NULL)
                {
                    free(pJsonFile);
                }
            }
        }

        if (profiledCount > 0)
        {
            // 合計をサンプル数で割って平均値を格納する
            for (int frameCount = 0; frameCount < cpuValueCountMax; ++frameCount)
            {
                averageProfile.SetCpuValue(frameCount, cpuLoadSum[frameCount] / profiledCount);
                NN_LOG("CpuAverage = %lld\n", cpuLoadSum[frameCount] / profiledCount);
            }
            for (int frameCount = 0; frameCount < gpuValueCountMax; ++frameCount)
            {
                averageProfile.SetGpuValue(frameCount, gpuLoadSum[frameCount] / profiledCount);
                NN_LOG("GpuAverage = %lld\n", gpuLoadSum[frameCount] / profiledCount);
            }
            for (int frameCount = 0; frameCount < memoryUsageValueCountMax; ++frameCount)
            {
                averageProfile.SetMemoryUsageValue(frameCount, memoryUsageSum[frameCount] / profiledCount);
                NN_LOG("MemoryUsageAverage = %lld\n", memoryUsageSum[frameCount] / profiledCount);
            }
            averageProfile.SetInitializationLoadValue(initializationLoadSum / profiledCount);
        }

        // TeamCity 向けのログ。TeamCity はこのログを拾って以下の情報を統計情報として記録する。
        // ・サンプル数
        // ・全サンプルの GetAverageValue( CPU ) の最大値と最小値
        // ・全サンプルの GetAverageValue( GPU ) の最大値と最小値
        // ・全サンプルの GetAverageValue( MemoryUsage ) の最大値と最小値
        NN_LOG("\n");
        NN_LOG("##teamcity[buildStatisticValue key='%sSampleCount' value='%d']\n", g_Args.FilePrefixForSearchAndCompare, profiledCount);
        if (cpuValueCountMax > 0)
        {
            NN_LOG("##teamcity[buildStatisticValue key='%sCpuMax' value='%lld']\n", g_Args.FilePrefixForSearchAndCompare, cpuLoadMax);
            NN_LOG("##teamcity[buildStatisticValue key='%sCpuMin' value='%lld']\n", g_Args.FilePrefixForSearchAndCompare, cpuLoadMin);
        }
        if (gpuValueCountMax > 0)
        {
            NN_LOG("##teamcity[buildStatisticValue key='%sGpuMax' value='%lld']\n", g_Args.FilePrefixForSearchAndCompare, gpuLoadMax);
            NN_LOG("##teamcity[buildStatisticValue key='%sGpuMin' value='%lld']\n", g_Args.FilePrefixForSearchAndCompare, gpuLoadMin);
        }
        if (memoryUsageValueCountMax > 0)
        {
            NN_LOG("##teamcity[buildStatisticValue key='%sMemoryUsageMax' value='%lld']\n", g_Args.FilePrefixForSearchAndCompare, memoryUsageMax);
            NN_LOG("##teamcity[buildStatisticValue key='%sMemoryUsageMin' value='%lld']\n", g_Args.FilePrefixForSearchAndCompare, memoryUsageMin);
        }
        if (initializationLoadMax > 0 && initializationLoadMin < LLONG_MAX)
        {
            NN_LOG("##teamcity[buildStatisticValue key='%sInitializationLoadMax' value='%lld']\n", g_Args.FilePrefixForSearchAndCompare, initializationLoadMax);
            NN_LOG("##teamcity[buildStatisticValue key='%sInitializationLoadMin' value='%lld']\n", g_Args.FilePrefixForSearchAndCompare, initializationLoadMin);
        }

        // 全サンプルの平均を格納した Json ファイルを出力
        nn::util::SNPrintf(timeString, 32,
            "%04d%02d%02d_%02d%02d%02d",
            calendarTime.year, calendarTime.month, calendarTime.day,
            calendarTime.hour, calendarTime.minute, calendarTime.second);
        char filePath[1024];
        nn::util::SNPrintf(filePath, 1024, "%s/%s%s.json", inputPath.GetString(), g_Args.FilePrefixForSearchAndCompare, timeString);
        averageProfile.Write(filePath);
        averageProfile.Finalize();

        decidedPath.SetString(filePath);

        free(pDirEntries);
    }
    else
    {
        decidedPath.SetString(inputPath.GetString());
    }
} // NOLINT(impl/function_size)

//----------------------------------------------------------------------------------------------------
void WriteFile(const char* pFilePath, const void* pWriteData, size_t size)
{
    nn::Result result;
    nn::fs::FileHandle file = {};
    result = nn::fs::DeleteFile(pFilePath);
    NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result));
    result = nn::fs::CreateFile(pFilePath, size);
    NN_ASSERT(result.IsSuccess());
    result = nn::fs::OpenFile(&file, pFilePath, nn::fs::OpenMode_Write);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_NOT_NULL(pWriteData);
    result = nn::fs::WriteFile(file, 0, pWriteData, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    NN_ASSERT(result.IsSuccess());
    nn::fs::CloseFile(file);
}

#ifdef NN_BUILD_CONFIG_OS_WIN32
//----------------------------------------------------------------------------------------------------
void GetJstTimeZoneInfo(TIME_ZONE_INFORMATION& timeZoneInfo)
{
    //  https://msdn.microsoft.com/en-us/library/ms724253.aspx
    HKEY key;
    REG_TZI_FORMAT regTzi;
    DWORD   type = 0;
    DWORD   size = 0;

    // タイムゾーンの情報はレジストリからしか取得できない。
    RegOpenKeyEx(HKEY_LOCAL_MACHINE,
                    L"SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\Time Zones\\Tokyo Standard Time",
                    0, KEY_QUERY_VALUE, &key);

    size = sizeof(REG_TZI_FORMAT);
    RegQueryValueEx(key, L"TZI", NULL, &type, (LPBYTE)&regTzi, &size);

    timeZoneInfo.Bias = regTzi.Bias;
    timeZoneInfo.StandardBias = regTzi.StandardBias;
    timeZoneInfo.DaylightDate = regTzi.StandardDate;
    timeZoneInfo.DaylightBias = regTzi.DaylightBias;
    timeZoneInfo.DaylightDate = regTzi.DaylightDate;
}
#endif

//----------------------------------------------------------------------------------------------------
// 渡された日付つきのファイル名の日時を更新して新しい文字列を作成します
void UpdateFilenameTimestamp(char* pName, size_t bufferSize, const char* pOrgName)
{
    size_t  nameLength = strlen(pOrgName);
    size_t  suffixLength = strlen("YYYYMMDD_HHMMSS.json");

    if (nameLength < bufferSize &&
        nameLength > suffixLength)
    {
        strncpy(pName, pOrgName, nameLength - suffixLength);

#ifdef NN_BUILD_CONFIG_OS_WIN32
        TIME_ZONE_INFORMATION   timeZoneInfo;

        // Jst タイムゾーンの情報を取得
        GetJstTimeZoneInfo(timeZoneInfo);

        SYSTEMTIME  localTime;
        SYSTEMTIME  utcTime;
        GetSystemTime(&utcTime);

        SystemTimeToTzSpecificLocalTime(&timeZoneInfo, &utcTime, &localTime);

        sprintf(&pName[nameLength - suffixLength], "%04d%02d%02d_%02d%02d%02d.json",
            localTime.wYear, localTime.wMonth, localTime.wDay,
            localTime.wHour, localTime.wMinute, localTime.wSecond);
        pName[nameLength] = '\0';
#else
        nn::time::PosixTime posixTime;
        nn::time::CalendarTime  calendarTime;

        nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
        nn::time::ToCalendarTime(&calendarTime, NULL, posixTime);

        sprintf(&pName[nameLength - suffixLength], "%04d%02d%02d_%02d%02d%02d.json",
            calendarTime.year, calendarTime.month, calendarTime.day,
            calendarTime.hour, calendarTime.minute, calendarTime.second);
        pName[nameLength] = '\0';
#endif  // NN_BUILD_CONFIG_OS_WIN32
    }
}

//----------------------------------------------------------------------------------------------------
// 比較した計測結果を出力ディレクトリに書き出します。
void OutputResultData(nnt::graphics::Path& inputFilePath, nnt::graphics::Path& outputDirectoryPath, bool updateTimestamp)
{
    char*   pFileName = inputFilePath.GetFilename().GetString();
    char*   pJsonFileImage = NULL;
    size_t  jsonFileSize = 0;

    JsonFileReadMalloc(&pJsonFileImage, jsonFileSize, inputFilePath.GetString());

    // 実機でタイムスタンプが正しく取得できない件の対応
    // 特定の日付よりも前の設定がされていたらコピーするタイミングの時刻で改名します。
    char    name[_MAX_PATH];
    if (updateTimestamp)
    {
        UpdateFilenameTimestamp(name, _MAX_PATH, inputFilePath.GetFilename().GetString());

        // ファイル名のタイムスタンプが置き換わった場合は、json ファイル内部のタイムスタンプも更新する。
        pFileName = name;

        rapidjson::Document doc;
        rapidjson::StringBuffer buffer;
        rapidjson::Writer<rapidjson::StringBuffer>  writer(buffer);

        if (!doc.Parse<0>(pJsonFileImage).HasParseError())
        {
            // ヘッダを取得
            char    date[32];

#ifdef NN_BUILD_CONFIG_OS_WIN32
            TIME_ZONE_INFORMATION   timeZoneInfo;

            // Jst タイムゾーンの情報を取得
            GetJstTimeZoneInfo(timeZoneInfo);

            SYSTEMTIME  localTime;
            SYSTEMTIME  utcTime;
            GetSystemTime(&utcTime);

            SystemTimeToTzSpecificLocalTime(&timeZoneInfo, &utcTime, &localTime);

            sprintf(date, "%04d-%02d-%02d %02d:%02d:%02d",
                localTime.wYear, localTime.wMonth, localTime.wDay,
                localTime.wHour, localTime.wMinute, localTime.wSecond);
#else
            nn::time::PosixTime posixTime;
            nn::time::CalendarTime  calendarTime;

            nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
            nn::time::ToCalendarTime(&calendarTime, NULL, posixTime);

            sprintf(date, "%04d-%02d-%02d %02d:%02d:%02d",
                calendarTime.year, calendarTime.month, calendarTime.day,
                calendarTime.hour, calendarTime.minute, calendarTime.second);
#endif
            doc["header"]["date"].SetString(date, static_cast<rapidjson::SizeType>(strlen(date)), doc.GetAllocator());
        }
        doc.Accept(writer);

        WriteFile(outputDirectoryPath.Combine(nnt::graphics::Path(pFileName)).GetString(), buffer.GetString(), buffer.GetSize());
    }
    else
    {
        WriteFile(outputDirectoryPath.Combine(nnt::graphics::Path(pFileName)).GetString(), pJsonFileImage, jsonFileSize);
    }
}

//----------------------------------------------------------------------------------------------------
size_t WritePartsTextToFile(nn::fs::FileHandle& file, const char* pWriteText, size_t offset)
{
    size_t writeSize = strlen(pWriteText);
    nn::fs::WriteFile(file, offset, pWriteText, writeSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    return writeSize;
}

//----------------------------------------------------------------------------------------------------
void OutputProfileHistory(nn::fs::FileHandle& file, size_t& writtenSize)
{
    //  var oldJsonText = [
    //      @OldProfileResultList
    //  ];
    writtenSize += WritePartsTextToFile(file, "var oldJsonText = [\r\n", writtenSize);

    nn::fs::DirectoryEntry* pDirEntries = NULL;
    int64_t entryCount = 0;

    // これまでの計測結果を出力。
    if (g_Args.SearchProfileHistoryPath.GetLength() > 0 &&
        ReadDirectory(&pDirEntries, entryCount, g_Args.SearchProfileHistoryPath))
    {
        size_t  prefixLength = strlen(g_Args.FilePrefixForSearchAndCompare);
        for(int64_t i = 0; i < entryCount; i++)
        {
            nnt::graphics::Path filePath(pDirEntries[i].name);
            char*   pFileName = filePath.GetFilename().GetString();

            size_t  fileNameLength = strlen(pFileName);
            // Prefix + YYYYMMDD_MMHHSS のフォーマットになっているかチェックする。
            // YYYYMMDD_MMHHSS.json で 20 文字。
            if (fileNameLength == (prefixLength + 20) &&
                strncmp(pFileName, g_Args.FilePrefixForSearchAndCompare, prefixLength) == 0)
            {
                char* pJsonFileImage = NULL;
                size_t  jsonFileSize = 0;
                JsonFileReadMalloc(&pJsonFileImage, jsonFileSize, g_Args.SearchProfileHistoryPath.Combine(filePath).GetString());

                rapidjson::Document doc;
                rapidjson::StringBuffer buffer;
                rapidjson::Writer<rapidjson::StringBuffer>  writer(buffer);

                if (!doc.Parse<0>(pJsonFileImage).HasParseError())
                {
                    doc.EraseMember("cpu_load");
                    doc.EraseMember("gpu_load");
                    doc.EraseMember("memory_max_usage");
                }
                doc.Accept(writer);

                writtenSize += WritePartsTextToFile(file, "'", writtenSize);
                writtenSize += WritePartsTextToFile(file, buffer.GetString(), writtenSize);
                writtenSize += WritePartsTextToFile(file, "',\r\n", writtenSize);

                free(pJsonFileImage);
            }
        }

        // 最新の計測結果を出力。
        {
            nnt::graphics::Path     outputSearchPath = g_Args.OutputPath.Combine(nnt::graphics::Path("rhs"));
            nnt::graphics::Path     outputCurrentProfileDataPath;
            GetNewestJsonFilePath(outputCurrentProfileDataPath, outputSearchPath, g_Args.FilePrefixForSearchAndCompare);

            char*   pOutputFileImage = NULL;
            size_t  outputFileSize = 0;
            JsonFileReadMalloc(&pOutputFileImage, outputFileSize, outputCurrentProfileDataPath.GetString());

            rapidjson::Document doc;
            rapidjson::StringBuffer buffer;
            rapidjson::Writer<rapidjson::StringBuffer>  writer(buffer);

            if (!doc.Parse<0>(pOutputFileImage).HasParseError())
            {
                doc.EraseMember("cpu_load");
                doc.EraseMember("gpu_load");
                doc.EraseMember("memory_max_usage");
            }
            doc.Accept(writer);

            writtenSize += WritePartsTextToFile(file, "'", writtenSize);
            writtenSize += WritePartsTextToFile(file, buffer.GetString(), writtenSize);
            writtenSize += WritePartsTextToFile(file, "'\r\n", writtenSize);

            free(pOutputFileImage);
        }
    }

    writtenSize += WritePartsTextToFile(file, "]\r\n", writtenSize);
}


//----------------------------------------------------------------------------------------------------
void OutputCompareResult(nnt::graphics::Path templateHtmlPath, const char* pLhsJsonImage, const char* pRhsJsonImage)
{
    char*   pTemplateFileImage = NULL;
    size_t  templateFileSize = 0;

    JsonFileReadMalloc(&pTemplateFileImage, templateFileSize, templateHtmlPath.GetString());

    if (templateFileSize > 0)
    {
        char    filename[_MAX_PATH];
        sprintf(filename, "%sCompareResult.html", g_Args.FilePrefixForSearchAndCompare);

        const char* pWriteFilePath = g_Args.OutputPath.Combine(nnt::graphics::Path(filename)).GetString();
        nn::Result result;
        nn::fs::FileHandle file = {};
        result = nn::fs::DeleteFile(pWriteFilePath);
        NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result));
        result = nn::fs::CreateFile(pWriteFilePath, 0);
        NN_ASSERT(result.IsSuccess());
        result = nn::fs::OpenFile(&file, pWriteFilePath, nn::fs::OpenMode_Write|nn::fs::OpenMode_AllowAppend);
        NN_ASSERT(result.IsSuccess());

        //  テンプレート中の「@InsertJsonData@」となっている行に以下のテキストを埋め込む。
        //
        //  var lhs_original = JSON.parse('@LhsJsonText');
        //  var rhs_original = JSON.parse('@RhsJsonText');
        //  var oldJsonText = [
        //      @OldProfileResultList
        //  ];
        //
        //  @LhsJsonText    Lhs で指定された JSON データ。
        //  @RhsJsonText    Rhs で指定された JSON データ。
        //  @OldProfileResultList   指定された Prefix に合致する過去の計測結果のリスト。
        //                          header だけ含まれていればよい。

        const char* pTagText = "@InsertJsonData@";
        size_t  tagTextSize = strlen(pTagText);

        size_t  offset = 0;
        size_t  writeStart = 0;
        size_t  writtenSize = 0;

        while(offset < templateFileSize)
        {
            if (pTemplateFileImage[offset] == '@' &&
                templateFileSize - offset >= tagTextSize &&
                strncmp(&pTemplateFileImage[offset], pTagText, tagTextSize) == 0)
            {
                // これまで解析したテキストを書き出す。
                size_t  writeSize = offset - writeStart;
                result = nn::fs::WriteFile(file, writtenSize, &pTemplateFileImage[writeStart], writeSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                writtenSize += writeSize;
                // タグを見つけたので必要なテキストを書き出す。

                //  var lhs_original = JSON.parse('@LhsJsonText');
                writtenSize += WritePartsTextToFile(file, "var lhs_original = JSON.parse('", writtenSize);
                writtenSize += WritePartsTextToFile(file, pLhsJsonImage, writtenSize);
                writtenSize += WritePartsTextToFile(file, "');\r\n", writtenSize);

                //  ar rhs_original = JSON.parse('@RhsJsonText');
                writtenSize += WritePartsTextToFile(file, "var rhs_original = JSON.parse('", writtenSize);
                writtenSize += WritePartsTextToFile(file, pRhsJsonImage, writtenSize);
                writtenSize += WritePartsTextToFile(file, "');\r\n", writtenSize);

                OutputProfileHistory(file, writtenSize);

                writeStart = offset + tagTextSize;
            }
            offset++;
        }

        // 最後に残りを書き込む。
        result = nn::fs::WriteFile(file, writtenSize, &pTemplateFileImage[writeStart], offset - writeStart, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        NN_ASSERT(result.IsSuccess());

        nn::fs::CloseFile(file);

        free(pTemplateFileImage);
    }
}

//----------------------------------------------------------------------------------------------------
TEST(ComparePerformanceProfileData, ComparePerformanceProfileData)
{
    bool succeed = false;

    nn::fs::MountHostRoot();

    // ファイルからデータを入力
    char* pLhsJsonFile = NULL;
    size_t  lhsFileSize = 0;
    nnt::graphics::Path lhsInputPath;
    char* pRhsJsonFile = NULL;
    size_t  rhsFileSize = 0;
    nnt::graphics::Path rhsInputPath;

    GetNewestJsonFilePath(lhsInputPath, g_Args.Inputs[0], g_Args.FilePrefixForSearchAndCompare);
    JsonFileReadMalloc(&pLhsJsonFile, lhsFileSize, lhsInputPath.GetString());

    // ResultAverage オプションが有効の場合は、複数の計測結果(Json)を読み込み、平均値が格納された json ファイルを新たに生成して使用する。
    if (g_Args.ResultAverageMode)
    {
        GetAveragedJsonFilePath(rhsInputPath, g_Args.Inputs[1], g_Args.FilePrefixForSearchAndCompare);
    }
    else
    {
        GetNewestJsonFilePath(rhsInputPath, g_Args.Inputs[1], g_Args.FilePrefixForSearchAndCompare);
    }
    JsonFileReadMalloc(&pRhsJsonFile, rhsFileSize, rhsInputPath.GetString());

    if (pLhsJsonFile != NULL &&
        pRhsJsonFile != NULL)
    {
        nnt::graphics::PerformanceProfileData  lhs, rhs;

        if (lhs.Initialize(pLhsJsonFile, JsonAllocator, JsonDeallocator) &&
            rhs.Initialize(pRhsJsonFile, JsonAllocator, JsonDeallocator))
        {
            if (g_Args.CompareFrameValueMode)
            {
                succeed = CompareCpuLoadFrameValue(lhs, rhs, g_Args.CpuLoadTolerance) &&
                            CompareGpuLoadFrameValue(lhs, rhs, g_Args.GpuLoadTolerance) &&
                            CompareMemoryUsageFrameValue(lhs, rhs, g_Args.MemoryUsageTolerance);
            }
            else
            {
                succeed = CompareCpuAverageLoad(lhs, rhs, g_Args.CpuLoadTolerance) &&
                            CompareGpuAverageLoad(lhs, rhs, g_Args.GpuLoadTolerance) &&
                            CompareMemoryUsageAverage(lhs, rhs, g_Args.MemoryUsageTolerance) &&
                            CompareInitializationLoad(lhs, rhs, g_Args.InitializationLoadTolerance);
            }

            // TeamCity 向けのログ。rhs(テスト対象の計測結果) の平均値を TeamCity に通知する。
            // ・全サンプル/全フレームの CPU (GetCpuValue で取れる値) の平均値。
            // ・全サンプル/全フレームの GPU (GetGpuValue で取れる値) の平均値。
            // ・全サンプル/全フレームの MemoryUsage (GetMemoryUsageValue で取れる値) の平均値。
            // ・全サンプル/全フレームの InitializationLoad (GetInitializationLoadValue で取れる値) の平均値。
            NN_LOG("\n");
            NN_LOG("##teamcity[buildStatisticValue key='%sCpuAverage' value='%lld']\n",
                g_Args.FilePrefixForSearchAndCompare, rhs.GetAverageValue(nnt::graphics::PerformanceProfileData::FrameValueType_CpuLoad));
            NN_LOG("##teamcity[buildStatisticValue key='%sGpuAverage' value='%lld']\n",
                g_Args.FilePrefixForSearchAndCompare, rhs.GetAverageValue(nnt::graphics::PerformanceProfileData::FrameValueType_GpuLoad));
            NN_LOG("##teamcity[buildStatisticValue key='%sMemoryUsageAverage' value='%lld']\n",
                g_Args.FilePrefixForSearchAndCompare, rhs.GetAverageValue(nnt::graphics::PerformanceProfileData::FrameValueType_MemoryUsage));
            NN_LOG("##teamcity[buildStatisticValue key='%sInitializationLoadAverage' value='%lld']\n",
                g_Args.FilePrefixForSearchAndCompare, rhs.GetInitializationLoadValue());
        }

        if (g_Args.OutputPath.GetLength() > 0)
        {
            // 出力先ディレクトリを作成する
            nnt::graphics::Path     lhsOutputPath = g_Args.OutputPath.Combine(nnt::graphics::Path("lhs"));
            nnt::graphics::Path     rhsOutputPath = g_Args.OutputPath.Combine(nnt::graphics::Path("rhs"));
            CreateDirectories(&lhsOutputPath);
            CreateDirectories(&rhsOutputPath);

            OutputResultData(lhsInputPath, lhsOutputPath, (g_Args.UpdateTimeStampTargetFlag & ProgramArgs::UpdateTimeStampTarget_Lhs) != 0);
            OutputResultData(rhsInputPath, rhsOutputPath, (g_Args.UpdateTimeStampTargetFlag & ProgramArgs::UpdateTimeStampTarget_Rhs) != 0);
        }

        lhs.Finalize();
        rhs.Finalize();
    }

    // 計測結果の html を出力する。
    if (g_Args.OutputHtmlTemplatePath.GetLength() > 0 &&
        g_Args.OutputPath.GetLength() > 0)
    {
        OutputCompareResult(g_Args.OutputHtmlTemplatePath, pLhsJsonFile, pRhsJsonFile);
    }

    if (pLhsJsonFile != NULL)
    {
        free(pLhsJsonFile);
    }
    if (pRhsJsonFile != NULL)
    {
        free(pRhsJsonFile);
    }

    nn::fs::UnmountHostRoot();

    EXPECT_TRUE(succeed);
}

void AnalyseArguments(int argc, char** argv)
{
    for(int i = 1; i < argc; i++)
    {
        NN_LOG("argv[%d] = %s\n", i, argv[i]);
        const char* arg = argv[i];
        if(std::strcmp(arg, "--cpu-load-tolerance") == 0)
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            g_Args.CpuLoadTolerance = static_cast<float>(atof(argv[i + 1]) / 100.0f);
            i += 1;
        }
        else if(std::strcmp(arg, "--gpu-load-tolerance") == 0)
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            g_Args.GpuLoadTolerance = static_cast<float>(atof(argv[i + 1]) / 100.0f);
            i += 1;
        }
        else if(std::strcmp(arg, "--memory-usage-tolerance") == 0)
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            g_Args.MemoryUsageTolerance = static_cast<float>(atof(argv[i + 1]) / 100.0f);
            i += 1;
        }
        else if(std::strcmp(arg, "--initialization-load-tolerance") == 0)
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            g_Args.InitializationLoadTolerance = static_cast<float>(atof(argv[i + 1]) / 100.0f);
            i += 1;
        }
        else if(std::strcmp(arg, "--result-root-path") == 0)
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            g_Args.OutputPath = nnt::graphics::Path(argv[i + 1]).Normalize();
            i += 1;
        }
        else if(std::strcmp(arg, "--search-profile-history-path") == 0)
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            g_Args.SearchProfileHistoryPath = nnt::graphics::Path(argv[i + 1]).Normalize();
            i += 1;
        }
        else if(std::strcmp(arg, "--directory-search-file-prefix") == 0)
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            size_t length = strlen(argv[i + 1]);
            strncpy(g_Args.FilePrefixForSearchAndCompare, argv[i + 1], length);
            g_Args.FilePrefixForSearchAndCompare[length] = '\0';
            i += 1;
        }
        else if(std::strcmp(arg, "--enable-frame-value-check") == 0)
        {
            g_Args.CompareFrameValueMode = true;
        }
        else if (std::strcmp(arg, "--enable-result-average") == 0)
        {
            g_Args.ResultAverageMode = true;
        }
        else if(std::strcmp(arg, "--result-html-template") == 0)
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            g_Args.OutputHtmlTemplatePath = nnt::graphics::Path(argv[i + 1]).Normalize();
            i += 1;
        }
        else if(std::strcmp(arg, "--update-timestamp") == 0)
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            if (strcmp(argv[i + 1], "both") == 0)
            {
                g_Args.UpdateTimeStampTargetFlag = ProgramArgs::UpdateTimeStampTarget_Lhs | ProgramArgs::UpdateTimeStampTarget_Rhs;
            }
            else
            {
                if (strcmp(argv[i + 1], "lhs") == 0)
                {
                    g_Args.UpdateTimeStampTargetFlag = ProgramArgs::UpdateTimeStampTarget_Lhs;
                }
                if (strcmp(argv[i + 1], "rhs") == 0)
                {
                    g_Args.UpdateTimeStampTargetFlag = ProgramArgs::UpdateTimeStampTarget_Rhs;
                }
            }
            i += 1;
        }
        else
        {
            if (g_Args.InputCount < 2)
            {
                g_Args.Inputs[g_Args.InputCount++] = nnt::graphics::Path(arg).Normalize();
            }
            else
            {
                NN_LOG("unknown option: %s\n", arg);
            }
        }
    }
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    nn::time::Initialize();

    nn::fs::SetAllocator(FsAllocate, FsDeallocate);

    NN_LOG("Running nnMain() from testGraphics_TestMain.cpp\n");
    ::testing::InitGoogleTest(&argc, argv);

    ::testing::TestEventListeners& listeners =
        ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter =
        listeners.Release(listeners.default_result_printer());
#if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    listeners.Append(new ::nnt::teamcity::ServiceMessageLogger());
#endif
    listeners.Append(defaultResultPrinter);

    g_Args.SetDefault();

    // コマンドライン引数の解析
    AnalyseArguments(argc, argv);

    int result = RUN_ALL_TESTS();

    nn::time::Finalize();

    ::nnt::Exit(result);
}
