﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nnt/graphics/testGraphics_PerformanceProfileData.h>

// Siglo 環境での警告抑止用
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif

#include <rapidjson/document.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(pop)
#endif

namespace nnt{ namespace graphics{

//!
//! @brief  CPU 平均負荷を比較します。
//!
//! @param[in]  lhs 比較対象の計測結果データです。
//! @param[in]  rhs 比較対象の計測結果データです。
//! @param[in]  fTolerance  誤差許容量を指定します。 1 を 100% として扱い、-x% から x% の誤差を許容します。
//!
//! @return 指定された誤差許容量以上の差分がある場合は false を返します。
//!
bool CompareCpuAverageLoad(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance);

//!
//! @brief  GPU 平均負荷を比較します。
//!
//! @param[in]  lhs 比較対象の計測結果データです。
//! @param[in]  rhs 比較対象の計測結果データです。
//! @param[in]  fTolerance  誤差許容量を指定します。 1 を 100% として扱い、-x% から x% の誤差を許容します。
//!
//! @return 指定された誤差許容量以上の差分がある場合は false を返します。
//!
bool CompareGpuAverageLoad(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance);

//!
//! @brief  平均メモリ使用量を比較します。
//!
//! @param[in]  lhs 比較対象の計測結果データです。
//! @param[in]  rhs 比較対象の計測結果データです。
//! @param[in]  fTolerance  誤差許容量を指定します。 1 を 100% として扱い、-x% から x% の誤差を許容します。
//!
//! @return 指定された誤差許容量以上の差分がある場合は false を返します。
//!
bool CompareMemoryUsageAverage(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance);

//!
//! @brief  初期化時の負荷を比較します。
//!
//! @param[in]  lhs 比較対象の計測結果データです。
//! @param[in]  rhs 比較対象の計測結果データです。
//! @param[in]  fTolerance  誤差許容量を指定します。 1 を 100% として扱い、-x% から x% の誤差を許容します。
//!
//! @return 指定された誤差許容量以上の差分がある場合は false を返します。
//!
bool CompareInitializationLoad(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance);

//!
//! @brief  フレームごとの CPU 負荷を比較します。
//!
//! @param[in]  lhs 比較対象の計測結果データです。
//! @param[in]  rhs 比較対象の計測結果データです。
//! @param[in]  fTolerance  誤差許容量を指定します。 1 を 100% として扱い、-x% から x% の誤差を許容します。
//!
//! @return 指定された誤差許容量以上の差分がある場合は false を返します。
//!
bool CompareCpuLoadFrameValue(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance);

//!
//! @brief  フレームごとの GPU 負荷を比較します。
//!
//! @param[in]  lhs 比較対象の計測結果データです。
//! @param[in]  rhs 比較対象の計測結果データです。
//! @param[in]  fTolerance  誤差許容量を指定します。 1 を 100% として扱い、-x% から x% の誤差を許容します。
//!
//! @return 指定された誤差許容量以上の差分がある場合は false を返します。
//!
bool CompareGpuLoadFrameValue(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance);

//!
//! @brief  フレームごとのメモリ使用量を比較します。
//!
//! @param[in]  lhs 比較対象の計測結果データです。
//! @param[in]  rhs 比較対象の計測結果データです。
//! @param[in]  fTolerance  誤差許容量を指定します。 1 を 100% として扱い、-x% から x% の誤差を許容します。
//!
//! @return 指定された誤差許容量以上の差分がある場合は false を返します。
//!
bool CompareMemoryUsageFrameValue(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance);

//!
//! @brief  許容誤差付きで値を比較します。
//!
//! @param[in]  lhs 比較対象の計測結果データです。
//! @param[in]  rhs 比較対象の計測結果データです。
//! @param[in]  fTolerance  誤差許容量を指定します。 1 を 100% として扱い、-x% から x% の誤差を許容します。
//!
//! @return 指定された誤差許容量以上の差分がある場合は false を返します。
//!
bool CompareValue(int64_t lhs, int64_t rhs, float fTolerance, const char *msg = "CompareValue");

//!
//! @brief  フレームごとのメモリ使用量を比較します。
//!
//! @param[in]  pLhs    比較対象の計測データです。
//! @param[in]  lhsCount    pLhs に格納されているデータの数です。
//! @param[in]  pRhs    比較対象の計測データです。
//! @param[in]  rhsCount    pLhs に格納されているデータの数です。
//! @param[in]  fTolerance  誤差許容量を指定します。 1 を 100% として扱い、-x% から x% の誤差を許容します。
//!
//! @return 指定された誤差許容量以上の差分がある場合は false を返します。
//!
bool CompareFrameValue(const PerformanceProfileData::FrameValue* pLhs, int lhsCount, const PerformanceProfileData::FrameValue* pRhs, int rhsCount, float fTolerance, const char *msg = "CompareFrameValue");

}}
