﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testGraphics_PixelwiseImageComparison.h"

#include <cmath>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#define NNT_GRAPHICS_ASSERT_IMAGEINFO_IS_VALID(img)                           \
    NN_ASSERT((img).width > 0);                                           \
    NN_ASSERT((img).height > 0);                                          \
    NN_ASSERT((img).nChannels > 0);                                       \
    NN_ASSERT((img).channelSize > 0);                                     \
    NN_ASSERT((img).pixelPitch >= (img).channelSize * (img).nChannels);   \
    NN_ASSERT((img).stride >= (img).pixelPitch * (img).width);            \
    NN_ASSERT_NOT_NULL((img).pData);                                      \
    NN_ASSERT((img).dataSize >= (img).stride * (img).height);

namespace nnt{ namespace graphics{

    static uint8_t* GetPixelPtr(ImageInformation* img, int x, int y)
    {
        unsigned char* p = static_cast<uint8_t*>(img->pData);
        return p + img->stride * y + img->pixelPitch * x;
    }
    static const uint8_t* GetPixelPtr(const ImageInformation* img, int x, int y)
    {
        const unsigned char* p = static_cast<const uint8_t*>(img->pData);
        return p + img->stride * y + img->pixelPitch * x;
    }

    static void CalculateRange(int* pOutStart0, int* pOutStart1, int* pOutLength, int size, int displacement)
    {
        if(displacement == 0)
        {
            *pOutStart0 = 0;
            *pOutStart1 = 0;
            *pOutLength = size;
        }
        else if(displacement > 0)
        {
            *pOutStart0 = displacement;
            *pOutStart1 = 0;
            *pOutLength = size - displacement;
        }
        else
        {
            *pOutStart0 = 0;
            *pOutStart1 = -displacement;
            *pOutLength = size + displacement;
        }
    }
    static void PixelwiseCompareImpl(
        ImageInformation* resultImg,
        const ImageInformation* img0,
        const ImageInformation* img1,
        int width,
        int height,
        int nChannels,
        int displacementX,
        int displacementY,
        int colorTolerance,
        bool isDifferenceFilledWithMaxValue)
    {
        int beginX0;
        int beginX1;
        int beginY0;
        int beginY1;
        int regionWidth;
        int regionHeight;
        CalculateRange(&beginX0, &beginX1, &regionWidth, width, displacementX);
        CalculateRange(&beginY0, &beginY1, &regionHeight, height, displacementY);
        NN_ASSERT(beginX0 >= 0);
        NN_ASSERT(beginY0 >= 0);
        NN_ASSERT(beginX1 >= 0);
        NN_ASSERT(beginY1 >= 0);
        NN_ASSERT(beginX0 + regionWidth <= width);
        NN_ASSERT(beginY0 + regionHeight <= height);
        NN_ASSERT(beginX1 + regionWidth <= width);
        NN_ASSERT(beginY1 + regionHeight <= height);

        for(int dy = 0; dy < regionHeight; dy++)
        {
            int y0 = beginY0 + dy;
            int y1 = beginY1 + dy;
            for(int dx = 0; dx < regionWidth; dx++)
            {
                int x0 = beginX0 + dx;
                int x1 = beginX1 + dx;
                const uint8_t* pixel0 = GetPixelPtr(img0, x0, y0);
                const uint8_t* pixel1 = GetPixelPtr(img1, x1, y1);
                uint8_t* pixelDst = GetPixelPtr(resultImg, x0, y0);
                for(int c = 0; c < nChannels; c++)
                {
                    int diff = std::abs(static_cast<int>(pixel0[c]) - static_cast<int>(pixel1[c]));
                    // 差がない場合
                    if(diff <= colorTolerance)
                    {
                        ( c == 3 ) ? pixelDst[c] = 0xFF : pixelDst[c] = 0; // 可視化できないためアルファの場合は 0xFF
                    }
                    else
                    {
#if 1
                        if (isDifferenceFilledWithMaxValue)
                        {
                            ( c == 3 ) ? pixelDst[c] = 0 : pixelDst[c] = 255; // 可視化できないためアルファの場合は 0x0
                        }
                        else
                        {
                            pixelDst[c] = static_cast<uint8_t>(diff);
                        }
#else
                        // 一番少ない誤差にする
                        uint8_t value = static_cast<uint8_t>(diff);
                        if(value < pixelDst[c])
                        {
                            pixelDst[c] = value;
                        }
#endif
                    }
                } // for c
            }// for dx
        } // for dy
    }


    int PixelwiseImageComparison::Compare(
        ImageInformation* resultImg,
        const ImageInformation* img0,
        const ImageInformation* img1,
        const Parameter* param) NN_NOEXCEPT
    {
        NNT_GRAPHICS_ASSERT_IMAGEINFO_IS_VALID(*resultImg);
        NNT_GRAPHICS_ASSERT_IMAGEINFO_IS_VALID(*img0);
        NNT_GRAPHICS_ASSERT_IMAGEINFO_IS_VALID(*img1);
        NN_ASSERT(img0->width == img1->width);
        NN_ASSERT(img0->height == img1->height);
        NN_ASSERT(img0->nChannels == img1->nChannels);
        NN_ASSERT(img0->channelSize == img1->channelSize);
        NN_ASSERT(img0->width == resultImg->width);
        NN_ASSERT(img0->height == resultImg->height);
        NN_ASSERT(img0->nChannels == resultImg->nChannels);
        NN_ASSERT(img0->channelSize == resultImg->channelSize);
        NN_ASSERT(img0->channelSize == 1);

        int width = img0->width;
        int height = img0->height;
        int nChannels = img0->nChannels;

        int colorTol = param->GetColorTolerance();
        int displacementTol = param->GetDisplacementTolerance();

        // 結果を最大値で埋める
        for(int y = 0; y < height; y++)
        {
            for(int x = 0; x < width; x++)
            {
                uint8_t* pixelDst = GetPixelPtr(resultImg, x, y);
                for(int c = 0; c < nChannels; c++)
                {
                    pixelDst[c] = 255;
                }
            }
        }

        // ずらしながら比較。一致するところは 0 になる。
        for(int dy = -displacementTol; dy <= displacementTol; dy++)
        {
            for(int dx = -displacementTol; dx <= displacementTol; dx++)
            {
                PixelwiseCompareImpl(resultImg, img0, img1, width, height, nChannels, dx, dy, colorTol, param->IsResultFilledWithMaximum());
            }
        }

        // 全チャンネルが 0 になっていないピクセルを数える
        int nDifferentPixels = 0;
        for(int y = 0; y < height; y++)
        {
            for(int x = 0; x < width; x++)
            {
                const uint8_t* pixelDst = GetPixelPtr(resultImg, x, y);
                for(int c = 0; c < nChannels; c++)
                {
                    if( ( c != 3 ) && pixelDst[c] != 0 )
                    {
                        nDifferentPixels++;
                        break;
                    }
                    if( ( c == 3 ) && (pixelDst[c] != 0xFF) )
                    {
                        nDifferentPixels++;
                        break;
                    }
                }
            }
        }
        return nDifferentPixels;
    }


    //====================================
    // Parameter
    //====================================

    PixelwiseImageComparison::Parameter::Parameter() NN_NOEXCEPT
        : m_ColorTolerance(0)
        , m_DisplacementTolerance(0)
        , m_IsResultFilledWithMaximum(false)
    {
    }

    void PixelwiseImageComparison::Parameter::SetDefault() NN_NOEXCEPT
    {
        m_ColorTolerance = 0;
        m_DisplacementTolerance = 0;
    }

    int PixelwiseImageComparison::Parameter::GetColorTolerance() const NN_NOEXCEPT
    {
        return m_ColorTolerance;
    }
    void PixelwiseImageComparison::Parameter::SetColorTolerance(int value) NN_NOEXCEPT
    {
        m_ColorTolerance = value;
    }

    int PixelwiseImageComparison::Parameter::GetDisplacementTolerance() const NN_NOEXCEPT
    {
        return m_DisplacementTolerance;
    }
    void PixelwiseImageComparison::Parameter::SetDisplacementTolerance(int value) NN_NOEXCEPT
    {
        m_DisplacementTolerance = value;
    }

    bool PixelwiseImageComparison::Parameter::IsResultFilledWithMaximum() const NN_NOEXCEPT
    {
        return m_IsResultFilledWithMaximum;
    }
    void PixelwiseImageComparison::Parameter::SetResultFilledWithMaximum(bool value) NN_NOEXCEPT
    {
        m_IsResultFilledWithMaximum = value;
    }
}}
