﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/dd.h>

#include <nn/gpio/gpio.h>

#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
#include <nn/nn_DeviceCode.h>
#include <nn/devicecode/gpio/devicecode_Predefined.gpio-hardware.nx.h>
#endif

namespace
{

// PINMUX レジスタの Field Name の列挙体
enum PinmuxFieldName
{
    PinmuxFieldName_PINMUX0  = 0,
    PinmuxFieldName_PINMUX1,
    PinmuxFieldName_PUPD2,
    PinmuxFieldName_PUPD3,
    PinmuxFieldName_TRISTATE,
    PinmuxFieldName_PARKED,
    PinmuxFieldName_E_INPUT,
    PinmuxFieldName_LOCK,
    PinmuxFieldName_E_LPDR,
    PinmuxFieldName_E_OD = 11,
    PinmuxFieldName_E_SCHMT = 12,
};

// APB_MISC IO マッピング
const nn::dd::PhysicalAddress   ApbMiscPhysicalAddress   = 0x070000000ull;
const size_t                    ApbMiscAddressSize       = 0x4000;

}


namespace nnt { namespace gpio {

#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    typedef nn::DeviceCode PadName;
    const PadName PadNameList[] =
    {
        NN_DEVICECODE_GPIO_NX_SD_WP, // = NN_DEVICECODE_GPIO_TEGRA_PAD(Z,3)
    };
#else // gen1
    // テスト用の GPIO の Pad 番号
    enum PadName
    {
        PadName_GPIO_PZ3 = 203,
    };

    // テスト対象の Pad の配列
    const PadName PadNameList[] =
    {
        PadName_GPIO_PZ3,
    };
#endif

// テスト対象の Pad の数
const int NumberOfTestPad = sizeof(PadNameList) / sizeof(PadNameList[0]);

// テスト用 pinmux クラス
class Pinmux
{

public:
    Pinmux() NN_NOEXCEPT
    {
        m_ApbMiscVirtualAddress = nn::dd::QueryIoMappingAddress(ApbMiscPhysicalAddress, ApbMiscAddressSize);
        if (m_ApbMiscVirtualAddress == 0)
        {
            // physicalAddress が指す I/O アドレスがマッピングされていない
            NN_ABORT("I/O registers for 0x%llx are not mapped. Make sure the capability setting is properly set for this process.\n", ApbMiscPhysicalAddress);
        }
    }

    // テスト用　： pinmux で Pullup する関数(pinmux ライブラリ等が出来たら置き換える)
    void PullUp(nnt::gpio::PadName padName)
    {
        volatile nn::Bit32* address;

        // TORIAEZU : PZ3 べた書き
        address = reinterpret_cast<volatile nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3288);

        *address |= (1 << PinmuxFieldName_PUPD3);
        *address &= ~(1 << PinmuxFieldName_PUPD2);
    }

    // テスト用　： pinmux で Pulldown する関数(pinmux ライブラリ等が出来たら置き換える)
    void PullDown(nnt::gpio::PadName padName)
    {
        volatile nn::Bit32* address;

        // TORIAEZU : PZ3 べた書き
        address = reinterpret_cast<volatile nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3288);

        *address &= ~(1 << PinmuxFieldName_PUPD3);
        *address |= (1 << PinmuxFieldName_PUPD2);
    }

    // テスト用 ： OutPut が Input にループバックするように設定する(pinmux ライブラリ等が出来たら置き換える)
    void SetDirectionForLoopBack(nnt::gpio::PadName padName)
    {
        volatile nn::Bit32* address;

        // TORIAEZU : PZ3 べた書き
        address = reinterpret_cast<volatile nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3288);

        *address |= (1 << PinmuxFieldName_E_INPUT);
        *address &= ~(1 << PinmuxFieldName_TRISTATE);
    }


private:
    uintptr_t m_ApbMiscVirtualAddress;


};




} } //nnt::gpio
