﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gc/detail/gc_GcCrypto.h>

#include "testGc_Unit_GcLibrary_Util.h"

using namespace nn::gc;
using namespace nn::gc::detail;

GcCrypto& g_Crypto = GcCrypto::GetInstance();

void PrintCounter(Counter128Bit& ctr)
{
    char buf[16];
    ctr.OutputCounterValue(buf, sizeof(buf));

    NN_LOG("    ");
    for(char i=0; i<16; i++)
    {
        NN_LOG("%02X ", buf[15 - i]);
    }
    NN_LOG("\n");
}

bool CounterComp(Counter128Bit& a, Counter128Bit& b)
{
    char buf1[16];
    char buf2[16];

    a.OutputCounterValue(buf1, sizeof(buf1));
    b.OutputCounterValue(buf2, sizeof(buf2));
    return memcmp(buf1, buf2, sizeof(buf1)) == 0;
}

TEST(GcUnitTest, CounterTest)
{
    NN_LOG("\nInfo: Counter Test\n");
    Counter128Bit a;
    Counter128Bit b;

    {
        char buf[16];
        memset(buf, 0, sizeof(buf));
        buf[1] = 1;
        a.ResetCounterWithInitialValue(buf, sizeof(buf));

        memset(buf, 0, sizeof(buf));
        buf[0] = 1;
        b.ResetCounterWithInitialValue(buf, sizeof(buf));
    }

    TESTOUT(CounterComp(a, b) == false, "different counter");

    // カウントアップすれば一致するはず
    b.CountUp(255);
    TESTOUT(CounterComp(a, b) == true, "countup same");

    // Increment と CountUp の挙動は同じ
    a.Increment();
    a.Increment();
    b.CountUp(2);
    TESTOUT(CounterComp(a, b) == true, "increment countup same");

    // リセットして比較
    a.ResetCounter();
    {
        char buf[16];
        memset(buf, 0, sizeof(buf));
        b.ResetCounterWithInitialValue(buf, sizeof(buf));
    }
    TESTOUT(CounterComp(a, b) == true, "reset same");

    // ループはしていないはず
    TESTOUT(a.IsCounterValueOver64bit() == false, "no loop");

    // ループさせると true になる
    a.ResetCounter();
    {
        // 16bit ループを回してみる
        char buf[16];
        memset(buf, 0xFF, sizeof(buf));
        memset(buf, 0, 2);
        a.ResetCounterWithInitialValue(buf, sizeof(buf));

        memset(buf, 0xFF, sizeof(buf));
        b.ResetCounterWithInitialValue(buf, sizeof(buf));
    }
    for(u16 i = 0; i < UINT16_MAX ; i++)
    {
        a.Increment();
    }
    // 今の値は FF FF FF FF ... となるべき
    TESTOUT(CounterComp(a, b) == true, "same value");
    PrintCounter(a);

    // ちょうど 0 に戻るべき
    a.Increment();
    {
        char buf[16];
        memset(buf, 0, sizeof(buf));
        b.ResetCounterWithInitialValue(buf, sizeof(buf));
    }
    TESTOUT(CounterComp(a, b) == true, "increment to 0 same");


    // TODO: 返り値
    //// ちょうど 0 に戻るべき
    //bool flag = a.Increment();
    //TESTOUT(flag == false, "counter goes to 0");
    //PrintCounter(a);
    //// 1になっているべき
    //flag = a.Increment();
    //TESTOUT(flag == false, "no carry flag");
    //PrintCounter(a);

    // NOTE: 内部の deltaCounter のセットはできないので、 Increment の返り値の bool 値を true にするのは至難の業（計算量的に）
}

TEST(GcUnitTest, RandomValueTest)
{
    NN_LOG("\nInfo: Random Value Test\n");
    char buf[GcRandomValueSize];
    {
        buf[23] = 0;
        buf[12] = 0;
    }
    char buf2[GcRandomValueSize];
    memset(buf2, 0x77, sizeof(buf2));

    g_Crypto.GenerateFixedValue(buf, sizeof(buf));
    g_Crypto.GenerateFixedValue(buf2, sizeof(buf2));

    // TEMP: 本当は違う値になることを確認したい
    TESTCMP(buf, buf2, sizeof(buf2), "same random value");
    PrintArray(buf, sizeof(buf));
    PrintArray(buf2, sizeof(buf2));
}

TEST(GcUnitTest, HashTest)
{
    NN_LOG("\nInfo: Hash Test\n");
    char test[GcCrypto::GcSha256Length];
    memset(test, 0, sizeof(test));

    char out[GcCrypto::GcSha256Length];
    g_Crypto.CalcSha256(out, sizeof(out), test, sizeof(test));

    g_Crypto.CalcSha256(test, sizeof(test), test, sizeof(test));

    // 同じ結果になるはず
    ASSERT_EQ(memcmp(out, test, sizeof(out)), 0);
    PrintArray(test, sizeof(test));
    PrintArray(out, sizeof(out));

    // 固定値との比較
    u8 ans[] = {
        0x66, 0x68, 0x7A, 0xAD, 0xF8, 0x62, 0xBD, 0x77,
        0x6C, 0x8F, 0xC1, 0x8B, 0x8E, 0x9F, 0x8E, 0x20,
        0x08, 0x97, 0x14, 0x85, 0x6E, 0xE2, 0x33, 0xB3,
        0x90, 0x2A, 0x59, 0x1D, 0x0D, 0x5F, 0x29, 0x25
    };
    ASSERT_EQ(memcmp(out, ans, sizeof(out)), 0);
}

// AES, SHA256, 乱数 のテストを含む
TEST(GcUnitTest, KeyGenTest)
{
    NN_LOG("\nInfo: Key Generate Test\n");
    char ra[GcRandomValueSize];
    char rb[GcRandomValueSize];

    g_Crypto.GenerateFixedValue(ra, sizeof(ra));
    g_Crypto.GenerateFixedValue(rb, sizeof(rb));
    g_Crypto.GenerateCommonKeyFromRandomValues(ra, sizeof(ra), rb, sizeof(rb));

    // ある時点の開発版鍵の結果のスナップショット
    char ans_aes_key[GcCrypto::GcAesKeyLength] = {
        0x2f,0x21,0xbe,0x55,0x6f,0x37,0x17,0x19,0xff,0x4c,0x54,0xd4,0x7e,0x5a,0xf6,0xc0
    };
    char ans_aes_iv[GcCrypto::GcAesCbcIvLength] = {
        0x81,0x4c,0x5e,0xbd,0x46,0xf2,0xac,0x85,0x26,0x5d,0xce,0x1b,0xae,0x53,0x34,0xff
    };

    // 共通鍵、IVが正しく生成できているか？
    TESTCMP(ans_aes_key, g_Crypto.m_CommonKey, GcCrypto::GcAesKeyLength, "common key");
    TESTCMP(ans_aes_iv, g_Crypto.m_AesInitialVector, GcCrypto::GcAesCbcIvLength, "iv");

    char ta[GcRandomValueSize];
    char tb[GcRandomValueSize];
    char tc[GcRandomValueSize];
    char td[GcRandomValueSize];

    {
        // ** 別に生成した AES CBC 128bit で正しく処理ができるかどうか
        NN_LOG("\nInfo: AES CBC 128 test\n");
        g_Crypto.GenerateFixedValue(ta, sizeof(ta));
        g_Crypto.GenerateFixedValue(tb, sizeof(tb));
        g_Crypto.GenerateFixedValue(tc, sizeof(tc));
        g_Crypto.GenerateFixedValue(td, sizeof(td));

        // 同じバッファを使うかそうでないかのテストも含む
        nn::crypto::Aes128CbcEncryptor enc;
        enc.Initialize(ans_aes_key, sizeof(ans_aes_key), ans_aes_iv, sizeof(ans_aes_iv));
        enc.Update(tb, sizeof(tb), tc, sizeof(tc));
        g_Crypto.EncryptWithAesCbc(ta, sizeof(ta), ta, sizeof(ta));
        TESTCMP(ta, tb, sizeof(ta), "same encrypted value");
        PrintArray(ta, sizeof(ta));
        PrintArray(tb, sizeof(tb));
        PrintArray(tc, sizeof(tc));

        // 復号化も同じになるか、また復号化して同じ値になるか
        nn::crypto::Aes128CbcDecryptor dec;
        dec.Initialize(ans_aes_key, sizeof(ans_aes_key), ans_aes_iv, sizeof(ans_aes_iv));
        dec.Update(tc, sizeof(tc), tb, sizeof(tb));
        dec.Initialize(ans_aes_key, sizeof(ans_aes_key), ans_aes_iv, sizeof(ans_aes_iv));
        dec.Update(tb, sizeof(tb), tb, sizeof(tb));
        g_Crypto.DecryptWithAesCbc(ta, sizeof(ta), ta, sizeof(ta));
        TESTCMP(ta, tc, sizeof(ta), "same decrypted value");
        TESTCMP(tb, tc, sizeof(tb), "same decrypted value with the same buffer (nn::crypto)");
        TESTCMP(ta, td, sizeof(ta), "same encrypt-decrypted value (GcCrypto)");

        PrintArray(ta, sizeof(ta));
        PrintArray(tb, sizeof(tb));
        PrintArray(tc, sizeof(tc));
        PrintArray(td, sizeof(td));
    }

    {
        // ** AES CTR 128bit で正しく処理ができるか
        NN_LOG("\nInfo: AES CTR 128 test\n");
        g_Crypto.GenerateFixedValue(ta, sizeof(ta));
        g_Crypto.GenerateFixedValue(tb, sizeof(tb));
        g_Crypto.GenerateFixedValue(tc, sizeof(tc));
        g_Crypto.GenerateFixedValue(td, sizeof(td));

        // 同じバッファを使うかそうでないかのテストも含む
        nn::crypto::Aes128CtrEncryptor enc;
        enc.Initialize(ans_aes_key, sizeof(ans_aes_key), ans_aes_iv, sizeof(ans_aes_iv));
        enc.Update(tb, sizeof(tb), tc, sizeof(tc));
        g_Crypto.EncryptWithAesCtr(ta, sizeof(ta), ta, sizeof(ta));
        TESTCMP(ta, tb, sizeof(ta), "same encrypted value");
        PrintArray(ta, sizeof(ta));
        PrintArray(tb, sizeof(tb));
        PrintArray(tc, sizeof(tc));

        // 復号化も同じになるか、また復号化して同じ値になるか
        nn::crypto::Aes128CtrDecryptor dec;
        dec.Initialize(ans_aes_key, sizeof(ans_aes_key), ans_aes_iv, sizeof(ans_aes_iv));
        dec.Update(tc, sizeof(tc), tb, sizeof(tb));
        dec.Initialize(ans_aes_key, sizeof(ans_aes_key), ans_aes_iv, sizeof(ans_aes_iv));
        dec.Update(tb, sizeof(tb), tb, sizeof(tb));
        g_Crypto.DecryptWithAesCtr(ta, sizeof(ta), ta, sizeof(ta));
        TESTCMP_NE(ta, tc, sizeof(ta), "not (temp) same decrypted value; NOTE: because of shared CTR");
        TESTCMP(tb, tc, sizeof(tb), "same decrypted value with the same buffer (nn::crypto)");

        char td[GcRandomValueSize];
        g_Crypto.GenerateFixedValue(td, sizeof(td));
        TESTCMP_NE(ta, td, sizeof(ta), "not (temp) same encrypt-decrypted value (GcCrypto); NOTE: because of shared CTR");

        PrintArray(ta, sizeof(ta));
        PrintArray(tb, sizeof(tb));
        PrintArray(tc, sizeof(tc));
        PrintArray(td, sizeof(td));
    }
}
