﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "GcCommon.h"
#include <nn/sdmmc/sdmmc_GcAsic.h>
#include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_IStorage.h>

#include <nn/gc/gc.h>

void SetupDeviceAddressSpace(nn::dd::DeviceAddressSpaceType* pDas, nn::dd::DeviceName deviceName) NN_NOEXCEPT
{
    // デバイスアドレス空間を 32bit アドレス空間で構築する
    nn::Result result = nn::dd::CreateDeviceAddressSpace(pDas, 0x100000000ull);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot create DeviceAddressSpace.");

    // 指定したデバイスに、構築したデバイスアドレス空間を関連付ける
    result = nn::dd::AttachDeviceAddressSpace(pDas, deviceName);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot attach DeviceAddressSpace to device.");
}

void CleanDeviceAddressSpace(nn::dd::DeviceAddressSpaceType* pDas, nn::dd::DeviceName deviceName) NN_NOEXCEPT
{
    // 指定されたデバイスへの関連付けを解除する
    nn::dd::DetachDeviceAddressSpace(pDas, deviceName);

    // デバイスアドレス空間オブジェクトを破棄する
    nn::dd::DestroyDeviceAddressSpace(pDas);
}

nn::dd::DeviceVirtualAddress MapDeviceAddressSpaceAligned(nn::dd::DeviceAddressSpaceType* pDas,
    uintptr_t alignedBufferAddress, size_t alignedBufferSize, nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset) NN_NOEXCEPT
{
    // AllocateMemoryBlock() で確保した大きなデータバッファは下位 22bit 一致でデバイスアドレス空間にメモリをマップする
    static_assert(nn::os::MemoryBlockUnitSize >= nn::sdmmc::BufferDeviceVirtualAddressAlignment, "");
    const size_t DeviceAddressSpaceAligned = (0x1U << 22);
    static_assert(DeviceAddressSpaceAligned >= nn::os::MemoryBlockUnitSize, "");
    const uint64_t DeviceAddressSpaceAlignedMask = DeviceAddressSpaceAligned - 1;
    nn::dd::DeviceVirtualAddress deviceVirtualAddress;
    if (deviceVirtualAddressOffset == 0)
    {
        // NULL を避ける
        deviceVirtualAddress = DeviceAddressSpaceAligned;
    }
    else
    {
        // deviceVirtualAddressOffset までを避ける
        deviceVirtualAddress = nn::util::align_up(deviceVirtualAddressOffset, DeviceAddressSpaceAligned);
    }
    deviceVirtualAddress += (alignedBufferAddress & DeviceAddressSpaceAlignedMask);

    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    NN_ABORT_UNLESS(currentProcessHandle != nn::os::InvalidNativeHandle);
    nn::Result result = nn::dd::MapDeviceAddressSpaceAligned(pDas, currentProcessHandle,
        alignedBufferAddress, alignedBufferSize,
        deviceVirtualAddress,
        nn::dd::MemoryPermission_ReadWrite);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot map memory to DeviceAddressSpace.");

    return deviceVirtualAddress;
}

void UnmapDeviceAddressSpaceAligned(nn::dd::DeviceAddressSpaceType* pDas,
    uintptr_t alignedBufferAddress, size_t alignedBufferSize, nn::dd::DeviceVirtualAddress deviceVirtualAddress) NN_NOEXCEPT
{
    // デバイスアドレス空間からメモリを切り離す
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    NN_ABORT_UNLESS(currentProcessHandle != nn::os::InvalidNativeHandle);
    nn::dd::UnmapDeviceAddressSpace(pDas, currentProcessHandle, alignedBufferAddress, alignedBufferSize, deviceVirtualAddress);
}

void LogBusStatus(nn::sdmmc::Port port) NN_NOEXCEPT
{
    nn::sdmmc::HostBusStatus hostBusStatus;
    nn::sdmmc::GetHostBusStatus(&hostBusStatus, port);

    NN_LOG("Host Bus Power: ");
    switch (hostBusStatus.busPower)
    {
    case nn::sdmmc::BusPower_Off:
        NN_LOG("OFF\n");
        break;
    case nn::sdmmc::BusPower_1_8V:
        NN_LOG("1.8V\n");
        break;
    case nn::sdmmc::BusPower_3_3V:
        NN_LOG("3.3V\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_LOG("Host Bus Width: ");
    switch (hostBusStatus.busWidth)
    {
    case nn::sdmmc::BusWidth_1Bit:
        NN_LOG("1 bit\n");
        break;
    case nn::sdmmc::BusWidth_4Bit:
        NN_LOG("4 bit\n");
        break;
    case nn::sdmmc::BusWidth_8Bit:
        NN_LOG("8 bit\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_LOG("Host Clock: %u (kHz)\n", hostBusStatus.deviceClockFrequencyKHz);

    nn::sdmmc::SpeedMode deviceSpeedMode;
    nn::Result result = nn::sdmmc::GetDeviceSpeedMode(&deviceSpeedMode, port);
    if (result.IsFailure())
    {
        NN_LOG("nn::sdmmc::GetGcAsicSpeedMode is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        return;
    }

    NN_LOG("Device Speed Mode: ");
    switch (deviceSpeedMode)
    {
    case nn::sdmmc::SpeedMode_MmcIdentification:
        NN_LOG("MMC Identification\n");
        break;
    case nn::sdmmc::SpeedMode_MmcLegacySpeed:
        NN_LOG("MMC Legacy Speed\n");
        break;
    case nn::sdmmc::SpeedMode_MmcHighSpeed:
        NN_LOG("MMC High Speed\n");
        break;
    case nn::sdmmc::SpeedMode_MmcHs200:
        NN_LOG("MMC HS200\n");
        break;
    case nn::sdmmc::SpeedMode_MmcHs400:
        NN_LOG("MMC HS400\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardIdentification:
        NN_LOG("SD Card Identification\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardDefaultSpeed:
        NN_LOG("SD Card Default Speed\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardHighSpeed:
        NN_LOG("SD Card High Speed\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr12:
        NN_LOG("SD Card SDR12\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr25:
        NN_LOG("SD Card SDR25\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr50:
        NN_LOG("SD Card SDR50\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr104:
        NN_LOG("SD Card SDR104\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardDdr50:
        NN_LOG("SD Card DDR50\n");
        break;
    case nn::sdmmc::SpeedMode_GcAsicFpgaSpeed:
        NN_LOG("GC ASIC FPGA\n");
        break;
    case nn::sdmmc::SpeedMode_GcAsicSpeed:
        NN_LOG("GC ASIC\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

// #define NN_DETAIL_GC_USE_FIXED_CALIBRATION_DATA
#ifdef NN_DETAIL_GC_USE_FIXED_CALIBRATION_DATA
namespace {
    static const char g_FixedCalibrationDataKeyForDev[] = {
        0x3b,0xb0,0x45,0xa2,0xd0,0xc5,0xcf,0x03,0xc5,0x61,0x04,0xef,0x5c,0xd8,0x29,0x17,
        0x77,0x7b,0xd1,0xa3,0x37,0x48,0xcd,0x26,0xf2,0x3e,0x6d,0x31,0x8f,0x4f,0xad,0xc5,0xca,0xe2,0x47,0xf2,0x45,0xe4,0xa6,0x14,0x3e,0xe5,0x82,0x66,0x77,0xae,0x3a,0x9b,
        0x51,0xb9,0x5a,0x1b,0x8f,0xfb,0x0f,0x64,0xab,0x4e,0xba,0x4f,0x0c,0x11,0xdb,0x72,0xcc,0xbb,0xa6,0xb2,0xac,0x71,0x57,0xbd,0xbe,0x2a,0x24,0x08,0xcf,0xf6,0x92,0x86,
        0x74,0x8d,0x13,0x97,0xe3,0x44,0x20,0x6b,0x20,0xc6,0xf6,0x9b,0x14,0xda,0x64,0xf7,0x9a,0xd5,0xdd,0x15,0x60,0x6d,0x59,0x86,0x59,0x6b,0x8b,0x98,0xc4,0x93,0x1c,0x4a,
        0xcb,0x55,0x3b,0xc1,0x82,0x6f,0x0f,0x32,0x31,0x5b,0x9a,0x6b,0x67,0x68,0x32,0x6b,0x31,0x4f,0x0b,0x05,0x6b,0x5b,0x40,0xb0,0xcb,0x40,0x99,0x9c,0x14,0xd2,0xdd,0x39,
        0xf2,0x05,0x02,0x2e,0x61,0x2b,0xd1,0x7c,0xf8,0x91,0xd9,0x6d,0x15,0x0f,0x8b,0xb1,0x6f,0x35,0xaf,0x87,0x15,0x11,0x72,0xd8,0xce,0xd3,0x47,0xe2,0xb0,0xa3,0xd9,0x29,
        0x21,0x2a,0x1a,0x11,0x13,0x77,0x18,0xb0,0xf6,0x12,0x33,0x58,0xe6,0xee,0x31,0xc6,0x94,0xbe,0x81,0xd6,0xce,0xdd,0x12,0x26,0xff,0x9a,0x1c,0x23,0xb6,0x3c,0xe6,0xd7,
        0x8d,0xf6,0x56,0xcc,0xf2,0xbe,0x41,0xed,0x41,0x32,0x7e,0xf7,0x20,0xae,0xb0,0x2c,0x5a,0x9a,0x06,0x3b,0xdb,0x3c,0xd0,0x29,0x69,0x89,0x7a,0x59,0xac,0xc6,0x7e,0xbc,
        0x9f,0xf4,0xc2,0xf3,0xa7,0x6a,0x37,0x55,0x4f,0xc2,0xb3,0x70,0x20,0x01,0x17,0x2e,0x63,0x8e,0xe9,0x94,0xb7,0x61,0xee,0x58,0x3f,0xe0,0xf7,0x31,0x99,0x09,0xfd,0x72
    };
    static const char g_FixedCalibrationDataCertForDev[] = {
        0x0d,0xa3,0x4d,0x33,0x6a,0xdf,0x11,0xfa,0xf0,0xe2,0xba,0xf3,0xdb,0x8f,0x97,0xc0,0x01,0x1b,0xc5,0x67,0x5b,0xf0,0x75,0x25,0x6b,0x59,0x4b,0x28,0x8c,0x92,0x2d,0x15,
        0xe4,0x5d,0xd4,0xef,0x9a,0xd3,0xa1,0x8d,0xd9,0xf8,0x25,0x2a,0xbf,0xb2,0x3c,0x08,0x15,0x6f,0xe7,0x32,0x9d,0xa3,0xdb,0x59,0xf6,0xe6,0x12,0x8e,0x32,0x65,0x42,0x42,
        0xb8,0x71,0xfc,0xf4,0x76,0x02,0xc2,0xea,0x5e,0x4c,0xc3,0x59,0xdc,0x52,0xfb,0x70,0x50,0xb4,0xec,0x69,0x74,0xb1,0x08,0x8d,0x44,0xd5,0x35,0x9f,0x28,0xcc,0x27,0xee,
        0xd2,0x24,0x14,0xc4,0x02,0x49,0xe1,0xe6,0xc7,0x0d,0x8e,0x57,0x12,0xfb,0x57,0x4e,0xfe,0x3b,0xef,0xc3,0x31,0xfe,0x1c,0x2e,0x98,0x65,0xed,0x44,0x0d,0x58,0x2b,0x46,
        0x5d,0x33,0x94,0x70,0x9b,0xd9,0x2e,0x78,0xc5,0x29,0x9c,0xc2,0x43,0xdf,0xf4,0x45,0xeb,0x11,0x51,0xeb,0x36,0xba,0xea,0x28,0xc9,0x8d,0xa5,0x49,0x66,0x20,0x46,0x39,
        0xfa,0x26,0xac,0x1d,0x6b,0xcc,0x44,0xe5,0xe6,0x35,0x86,0x13,0x95,0xda,0x1e,0x55,0x98,0x85,0xb9,0x06,0x97,0x22,0xef,0x46,0x9f,0xff,0x06,0x40,0x95,0x00,0x4e,0xeb,
        0x95,0xfc,0x06,0x76,0x2c,0x8a,0x75,0x6b,0x4a,0x2e,0x51,0x69,0x9e,0xe8,0xce,0xa9,0x31,0x54,0xc3,0xba,0x86,0x03,0xff,0x42,0x6e,0xbd,0x20,0x81,0xa8,0x7f,0x34,0x8a,
        0xfd,0xf9,0x30,0x31,0x73,0xd7,0x18,0x99,0x9a,0xce,0x98,0x24,0x86,0x6d,0xaf,0x08,0x59,0xa0,0x3f,0x77,0xb1,0xbb,0x64,0xca,0x3e,0xf8,0x7f,0x85,0xe1,0x65,0x63,0x82,
        0x53,0x4f,0x43,0x43,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb5,0x9f,0x3f,0xd5,0xfd,0x12,0x5a,0x34,0xa8,0xcd,0x27,0xc1,0x3e,0x9b,0xed,0x3d,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb8,0x65,0x2f,0x87,0x7a,0x5c,0xb4,0x93,0xc2,0xf0,0x80,0x8b,0xc2,0xb3,0x01,0x79,
        0x0e,0x2c,0xe8,0x82,0xc7,0x4c,0x2c,0x52,0x54,0xb2,0x4b,0xbe,0xa9,0x46,0x72,0xbd,0x7d,0xf2,0xe3,0x1f,0x3f,0x65,0x3e,0xb1,0xb7,0xe4,0x22,0x91,0xe7,0xc2,0xa4,0xec,
        0x9e,0x8e,0x6d,0xfe,0xb8,0x1c,0x8c,0x2e,0x04,0x5b,0x78,0x9d,0x78,0x19,0xad,0x73,0x8f,0xa8,0x12,0x82,0x69,0x36,0xe1,0xaf,0x0b,0x1d,0x19,0xe3,0x20,0x00,0xda,0x6c,
        0x8c,0x3d,0xed,0x95,0xa7,0xf2,0x6c,0x9e,0x0a,0x02,0x67,0x05,0x8d,0xe6,0xe8,0x8e,0x09,0xda,0xd4,0xe3,0x33,0xf0,0xd4,0xe4,0xc1,0x29,0xd9,0xf0,0xdf,0x59,0xaa,0xf5,
        0xb4,0xd0,0x4f,0xb9,0xf1,0x17,0x8b,0xc9,0xd9,0xa7,0xbc,0x6c,0xdf,0x1a,0x00,0x67,0xd4,0x59,0x69,0x03,0xa9,0x8b,0x1c,0x54,0xd9,0xd7,0x02,0x8f,0xb8,0xbf,0xe8,0xe2,
        0x12,0xe1,0xcd,0xb9,0xf0,0x3d,0x3c,0x4f,0x4f,0xc9,0x30,0x6d,0xeb,0xa3,0x0f,0x80,0x4f,0x3a,0x44,0x5e,0xe9,0xc4,0x9c,0x0c,0x08,0x92,0x46,0x10,0xad,0x75,0xf8,0xad,
        0x2d,0xcd,0x46,0xf6,0x2f,0x60,0x5e,0x4c,0x65,0xa2,0xf2,0xb9,0x62,0xbc,0x96,0x44,0x45,0xdf,0xab,0x29,0x9d,0x2d,0xbe,0x9e,0x66,0x60,0x4f,0xdb,0x08,0x4d,0x4c,0x7c,
        0x58,0x79,0x65,0x13,0xc6,0x75,0xe5,0x7a,0x5c,0x34,0xfa,0x7d,0x62,0xc1,0x3f,0xc1,0xf2,0x78,0xb6,0x38,0x8a,0xb6,0x7f,0xb3,0xbc,0x10,0x7c,0x4a,0xd8,0xda,0x1f,0xf6,
        0x85,0xc0,0x53,0x69,0x5c,0xf1,0x0f,0x4c,0x9d,0x22,0x9c,0xfe,0x56,0x9f,0x45,0x87,0x01,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };
}
#endif

nn::Result ReadGcCalibrationPartitionToSetInternalKeys()
{
    NN_LOG("Read From Partition\n");

#ifdef NN_DETAIL_GC_USE_FIXED_CALIBRATION_DATA
    NN_LOG("Use fixed calibrastion data\n");
    nn::gc::PresetInternalKeys(g_FixedCalibrationDataKeyForDev, sizeof(g_FixedCalibrationDataKeyForDev),
                            g_FixedCalibrationDataCertForDev, sizeof(g_FixedCalibrationDataCertForDev));
#else
    const uint32_t BoundaryVersion   = 0x7;
    const size_t VersionAddress      = 0x4;
    const size_t KeyAddress          = 0x3C20;
    const size_t KeySize             = 0x134;
    const size_t OldKeyAddress       = 0x2320;
    const size_t OldKeySize          = 0x110;
    const size_t CertAddress         = 0x2440;
    const size_t CertSize            = 0x400;

    std::unique_ptr<nn::fs::IStorage> storage;
    nn::Result result = nn::fs::OpenBisPartition(&storage, nn::fs::BisPartitionId::CalibrationBinary);
    if(result.IsFailure())
    {
        NN_LOG("Test Failure\n");
        NN_LOG("  OpenBisPartition Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
    if(storage == nullptr)
    {
        NN_LOG("PTR NULL\n");
    }

    static char keyBuffer[KeySize] = {0};
    static char certBuffer[CertSize] = {0};

    uint32_t version = 0;
    NN_RESULT_DO(storage->Read(VersionAddress, &version, sizeof(version)));
    size_t targetKeyAddress = (version >= BoundaryVersion) ? KeyAddress : OldKeyAddress;
    size_t targetKeySize = (version >= BoundaryVersion) ? KeySize : OldKeySize;

    NN_RESULT_DO(storage->Read(targetKeyAddress, keyBuffer, targetKeySize));
    NN_RESULT_DO(storage->Read(CertAddress, certBuffer, CertSize));

    nn::gc::PresetInternalKeys(keyBuffer, targetKeySize, certBuffer, CertSize);
    memset(keyBuffer, 0, sizeof(keyBuffer));
#endif

    NN_RESULT_SUCCESS;
}
