﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "../../Common/testGamePad_Common.h"
#include "../../Common/testGamePad_IrSensor.h"

namespace nnt { namespace irsensor {
    nn::irsensor::IrCameraHandle IrSensorTest::s_Handles[nnt::gamepad::NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}} // namaspace nnt::irsensor

namespace
{

const int VerifyStatesTestCountMax = 5;                                         // ステータスのテストを実施する回数
const nn::TimeSpan VerifyStatesTestLimitTime = nn::TimeSpan::FromSeconds(5);    // ステータスが更新されない状態の許容時間
const int NotReadyWaitLoopCountMax = 1000;                                      // ステータス取得の準備が完了するまでに待つ最大のループ回数
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);        // ステータス取得の準備が完了していない時のポーリング間隔

class MomentProcessorTest : public ::nnt::irsensor::IrSensorTest
{
protected:
    void DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestVerifyMomentState(const nn::irsensor::MomentProcessorState& state, const nn::irsensor::MomentProcessorConfig& config) NN_NOEXCEPT;
    void DoTestReadingMomentStates(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::MomentProcessorConfig& config, const int testCount) NN_NOEXCEPT;
    void DoTestPreprocessBoundary(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::MomentProcessorPreprocess& preprocess) NN_NOEXCEPT;
    void DoTestPreprocessIntensityThresholdBoundary(const nn::irsensor::IrCameraHandle& handle, int threshold) NN_NOEXCEPT;
    void DoTestWoiBoundary(const nn::irsensor::IrCameraHandle& handle, int x, int y, int width, int height) NN_NOEXCEPT;
    void ReadingStateTest(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
};

//==================================================
// プロセッサの起動までの時間を計測
//==================================================
void MomentProcessorTest::DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    int counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();      // 計測を開始した時間を記憶

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Stopped)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            NN_LOG("#     ImageProcessorStatus [ Stopped -> Running ] (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// プロセッサの停止までの時間を計測
//==================================================
void MomentProcessorTest::DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessorAsync(handle);

    auto counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("#     StopImageProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの検証
//==================================================
void MomentProcessorTest::DoTestVerifyMomentState(const nn::irsensor::MomentProcessorState& state, const nn::irsensor::MomentProcessorConfig& config) NN_NOEXCEPT
{
    ASSERT_GE(state.samplingNumber, 0);
    ASSERT_GE(state.timeStamp.GetNanoSeconds(), 0);

    ASSERT_GE(state.ambientNoiseLevel, nn::irsensor::IrCameraAmbientNoiseLevel_Low);
    ASSERT_LE(state.ambientNoiseLevel, nn::irsensor::IrCameraAmbientNoiseLevel_Unknown);

    int index = 0;
    auto width = config.windowOfInterest.width / nn::irsensor::MomentProcessorBlockColumnCount;
    auto height = config.windowOfInterest.height / nn::irsensor::MomentProcessorBlockRowCount;

    for (const auto& block : state.blocks)
    {
        auto x = config.windowOfInterest.x + (index % nn::irsensor::MomentProcessorBlockColumnCount) * width;
        auto y = config.windowOfInterest.y + (index / nn::irsensor::MomentProcessorBlockColumnCount) * height;

        ASSERT_LE(0, block.averageIntensity);
        ASSERT_LE(block.averageIntensity, nn::irsensor::IrCameraIntensityMax);

        if (block.averageIntensity == 0)
        {
            EXPECT_EQ(0, block.centroid.x);
            EXPECT_EQ(0, block.centroid.y);
        }
        else
        {
            EXPECT_LE(x, config.windowOfInterest.x + block.centroid.x);
            EXPECT_LE(config.windowOfInterest.x + block.centroid.x, x + width - 1);
            EXPECT_LE(y, config.windowOfInterest.y + block.centroid.y);
            EXPECT_LE(config.windowOfInterest.y + block.centroid.y, y + height - 1);
        }

        ++index;
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの取得と更新時間の計測
//==================================================
void MomentProcessorTest::DoTestReadingMomentStates(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::MomentProcessorConfig& config, const int testCount) NN_NOEXCEPT
{
    NN_LOG("#     ReadingMomentProcessorStatesTest\n");
    int64_t prevSamplingNumber = -1;

    //-------------------------------------------------------------------------------
    // プロセッサが起動するまでの時間の計測します
    //-------------------------------------------------------------------------------
    nn::irsensor::RunMomentProcessor(handle, config);
    DoTestRunningProcessor(handle);

    //-------------------------------------------------------------------------------
    // ステートが更新されるまでにかかる時間を計測します
    //-------------------------------------------------------------------------------
    for (int i = 0; i < testCount; ++i)
    {
        nn::irsensor::MomentProcessorState state[nn::irsensor::MomentProcessorStateCountMax];
        auto samplingNumber = prevSamplingNumber;
        auto stateCount = 0;
        auto startTime = nn::os::GetSystemTick().ToTimeSpan();                       // 計測を開始した時間を記憶

        NN_LOG("#     Test : [%2d/%2d]\n", i + 1, testCount);
        NN_LOG("#         CallGetMomentProcessorStates\n");

        while (NN_STATIC_CONDITION(true))
        {
            nn::Result result = nn::irsensor::GetMomentProcessorStates(state, &stateCount, nn::irsensor::MomentProcessorStateCountMax, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result) == false)
            {
                if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorDeviceError #############\n");
                    ADD_FAILURE();
                    return;
                }
                samplingNumber = state[0].samplingNumber;
            }
            if (prevSamplingNumber != samplingNumber)
            {
                break;
            }
            ASSERT_LT(nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds(), VerifyStatesTestLimitTime.GetMilliSeconds());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
        NN_LOG("#             UpdateMomentProcessorStates (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
        NN_LOG("#             SamplingNumber %lld -> %lld\n", prevSamplingNumber, samplingNumber);
        prevSamplingNumber = samplingNumber;
        //-------------------------------------------------------------------------------
        // ステータスの検証を行います
        //-------------------------------------------------------------------------------
        NN_LOG("#         VerifyMomentState (StateCount : %d)\n", stateCount);
        for (auto j = 0; j < stateCount; ++j)
        {
            DoTestVerifyMomentState(state[j], config);
        }
    }
    NN_LOG("#     SUCCESS\n");
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// 処理対象の矩形領域を変更
//==================================================
void MomentProcessorTest::DoTestWoiBoundary(const nn::irsensor::IrCameraHandle& handle, int x, int y, int width, int height) NN_NOEXCEPT
{
    NN_LOG("#     WoiBoundaryTest - Pos (%d, %d) : Size (%d, %d)\n", x, y, width, height);
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::MomentProcessorConfig config;
    nn::irsensor::GetMomentProcessorDefaultConfig(&config);

    config.windowOfInterest = nn::irsensor::MakeRect(x, y, width, height);

    DoTestReadingMomentStates(handle, config, 1);
}

//==================================================
// モーメントの算出前に画像処理を変更
//==================================================
void MomentProcessorTest::DoTestPreprocessBoundary(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::MomentProcessorPreprocess& preprocess) NN_NOEXCEPT
{
    NN_LOG("#     PreprocessBoundaryTest\n");
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::MomentProcessorConfig config;
    nn::irsensor::GetMomentProcessorDefaultConfig(&config);

    config.preprocess = preprocess;

    DoTestReadingMomentStates(handle, config, 1);
}

//==================================================
// モーメントの算出前に画像処理の閾値を変更
//==================================================
void MomentProcessorTest::DoTestPreprocessIntensityThresholdBoundary(const nn::irsensor::IrCameraHandle& handle, int threshold) NN_NOEXCEPT
{
    NN_LOG("#     PreprocessIntensityThresholdBoundaryTest\n");
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::MomentProcessorConfig config;
    nn::irsensor::GetMomentProcessorDefaultConfig(&config);

    config.preprocessIntensityThreshold = threshold;

    DoTestReadingMomentStates(handle, config, 1);
}


//==================================================
// ステータス取得のテスト
//==================================================
void MomentProcessorTest::ReadingStateTest(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::MomentProcessorConfig config;

    nn::irsensor::GetMomentProcessorDefaultConfig(&config);

    // 5回ステータスを取得し値の確認を行います
    DoTestReadingMomentStates(handle, config, VerifyStatesTestCountMax);

    // 前処理の種類を変更しステータス取得を行います
    DoTestPreprocessBoundary(handle, nn::irsensor::MomentProcessorPreprocess_Binarize);
    DoTestPreprocessBoundary(handle, nn::irsensor::MomentProcessorPreprocess_Cutoff);

    // 前処理に利用する閾値を変更しステータス取得を行います
    DoTestPreprocessIntensityThresholdBoundary(handle, 0);
    DoTestPreprocessIntensityThresholdBoundary(handle, nn::irsensor::IrCameraIntensityMax);

    // 処理対象の矩形範囲を変更しステータス取得を行います
    DoTestWoiBoundary(handle, 1, 0, nn::irsensor::IrCameraImageWidth - 1, nn::irsensor::IrCameraImageHeight);
    DoTestWoiBoundary(handle, 0, 1, nn::irsensor::IrCameraImageWidth, nn::irsensor::IrCameraImageHeight - 1);
    DoTestWoiBoundary(handle, nn::irsensor::IrCameraImageWidth - nn::irsensor::MomentProcessorBlockColumnCount, 0, nn::irsensor::MomentProcessorBlockColumnCount, nn::irsensor::IrCameraImageHeight);
    DoTestWoiBoundary(handle, 0, nn::irsensor::IrCameraImageHeight - nn::irsensor::MomentProcessorBlockRowCount, nn::irsensor::IrCameraImageWidth, nn::irsensor::MomentProcessorBlockRowCount);
    DoTestWoiBoundary(handle, nn::irsensor::IrCameraImageWidth - nn::irsensor::MomentProcessorBlockColumnCount, nn::irsensor::IrCameraImageHeight - nn::irsensor::MomentProcessorBlockRowCount, nn::irsensor::MomentProcessorBlockColumnCount, nn::irsensor::MomentProcessorBlockRowCount);

    // テスト終了後にプロセッサを停止します
    DoTestStopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータス取得のテスト
//==================================================
TEST_F(MomentProcessorTest, StateReading)
{
    nnt::gamepad::Initialize();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start Moment Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        ReadingStateTest(handle);
    }
    nnt::gamepad::DisconnectAll();
}

// USB有線接続が有効時にステータスが正常に取得できるか
TEST_F(MomentProcessorTest, StateReadingForUsb)
{
    nnt::gamepad::Initialize();

    // USB 有線接続を有効にする
    ::nnt::gamepad::EnableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start Moment Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        ReadingStateTest(handle);
    }

    nnt::gamepad::DisconnectAll();
}

} // namespace
