﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "../../Common/testGamePad_Common.h"
#include "../../Common/testGamePad_IrSensor.h"

namespace nnt { namespace irsensor {
    nn::irsensor::IrCameraHandle IrSensorTest::s_Handles[nnt::gamepad::NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}} // namaspace nnt::irsensor

namespace
{

const int VerifyStatesTestCountMax = 5;                                         // ステータスのテストを実施する回数
const nn::TimeSpan VerifyStatesTestLimitTime = nn::TimeSpan::FromSeconds(5);    // ステータスが更新されない状態の許容時間
const int NotReadyWaitLoopCountMax = 1000;                                      // ステータス取得の準備が完了するまでに待つ最大のループ回数
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);        // ステータス取得の準備が完了していない時のポーリング間隔

class ClusteringProcessorTest : public nnt::irsensor::IrSensorTest
{
protected:
    void DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestVerifyClusteringStates(const nn::irsensor::ClusteringProcessorState& state, nn::irsensor::ClusteringProcessorConfig config) NN_NOEXCEPT;
    void DoTestReadingClusteringStates(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::ClusteringProcessorConfig config, const int testCount) NN_NOEXCEPT;
    void DoTestWoiBoundary(const nn::irsensor::IrCameraHandle& handle, int x, int y, int width, int height) NN_NOEXCEPT;
    void DoTestObjectIntensityMinBoundary(const nn::irsensor::IrCameraHandle& handle, int threshold) NN_NOEXCEPT;
    void DoTestObjectPixelCountBoundary(const nn::irsensor::IrCameraHandle& handle, int min, int max) NN_NOEXCEPT;
    void DoTestClusteringData(const nn::irsensor::ClusteringData& object, const nn::irsensor::ClusteringProcessorConfig& config) NN_NOEXCEPT;
    void DoTestReadingState(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
};

//==================================================
// プロセッサの起動までの時間を計測
//==================================================
void ClusteringProcessorTest::DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    int counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();      // 計測を開始した時間を記憶

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Stopped)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            NN_LOG("#     ImageProcessorStatus [ Stopped -> Running ] (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// プロセッサの停止までの時間を計測
//==================================================
void ClusteringProcessorTest::DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessorAsync(handle);

    auto counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("#     StopImageProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの検証
//==================================================
void ClusteringProcessorTest::DoTestVerifyClusteringStates(const nn::irsensor::ClusteringProcessorState& state, nn::irsensor::ClusteringProcessorConfig config) NN_NOEXCEPT
{
    ASSERT_GE(state.samplingNumber, 0);
    ASSERT_GE(state.timeStamp.GetNanoSeconds(), 0);

    ASSERT_GE(state.ambientNoiseLevel, nn::irsensor::IrCameraAmbientNoiseLevel_Low);
    ASSERT_LE(state.ambientNoiseLevel, nn::irsensor::IrCameraAmbientNoiseLevel_Unknown);

    NN_LOG("#             ObjectCount : %d\n", state.objectCount);

    for (int i = 0; i < state.objectCount; ++i)
    {
        auto& obj = state.objects[i];
        DoTestClusteringData(obj, config);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの取得と更新時間の計測
//==================================================
void ClusteringProcessorTest::DoTestReadingClusteringStates(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::ClusteringProcessorConfig config, const int testCount) NN_NOEXCEPT
{
    NN_LOG("#     ReadingClusteringStatesTest\n");
    int64_t prevSamplingNumber = -1;

    //-------------------------------------------------------------------------------
    // プロセッサが起動するまでの時間の計測します
    //-------------------------------------------------------------------------------
    nn::irsensor::RunClusteringProcessor(handle, config);
    DoTestRunningProcessor(handle);

    //-------------------------------------------------------------------------------
    // ステートが更新されるまでにかかる時間を計測します
    //-------------------------------------------------------------------------------
    for (int i = 0; i < testCount; ++i)
    {
        nn::irsensor::ClusteringProcessorState state[nn::irsensor::ClusteringProcessorStateCountMax];
        auto samplingNumber = prevSamplingNumber;
        auto stateCount = 0;
        auto startTime = nn::os::GetSystemTick().ToTimeSpan();                       // 計測を開始した時間を記憶

        NN_LOG("#     Test : [%2d/%2d]\n", i + 1, testCount);
        NN_LOG("#         CallGetClusteringProcessorStates\n");

        while (NN_STATIC_CONDITION(true))
        {
            nn::Result result = nn::irsensor::GetClusteringProcessorStates(state, &stateCount, nn::irsensor::ClusteringProcessorStateCountMax, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result) == false)
            {
                if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorDeviceError #############\n");
                    ADD_FAILURE();
                    return;
                }
                samplingNumber = state[0].samplingNumber;
            }
            if (prevSamplingNumber != samplingNumber)
            {
                break;
            }
            ASSERT_LT(nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds(), VerifyStatesTestLimitTime.GetMilliSeconds());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
        NN_LOG("#             UpdateClusteringProcessorStates (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
        NN_LOG("#             SamplingNumber %lld -> %lld\n", prevSamplingNumber, samplingNumber);
        prevSamplingNumber = samplingNumber;
        //-------------------------------------------------------------------------------
        // ステータスの検証を行います
        //-------------------------------------------------------------------------------
        NN_LOG("#         VerifyClusteringState (StateCount : %d)\n", stateCount);
        for (auto j = 0; j < stateCount; ++j)
        {
            DoTestVerifyClusteringStates(state[j], config);
        }
    }
    NN_LOG("#     SUCCESS\n");
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// 処理対象となる矩形領域を設定したテスト
//==================================================
void ClusteringProcessorTest::DoTestWoiBoundary(const nn::irsensor::IrCameraHandle& handle, int x, int y, int width, int height) NN_NOEXCEPT
{
    NN_LOG("# ************************************************************\n");
    NN_LOG("#     WoiBoundaryTest Pos: (%d, %d) - Size: (%d, %d)\n", x, y, width, height);
    NN_LOG("# ************************************************************\n");
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::ClusteringProcessorConfig config;
    nn::irsensor::GetClusteringProcessorDefaultConfig(&config);

    config.windowOfInterest = nn::irsensor::MakeRect(x, y, width, height);
    DoTestReadingClusteringStates(handle, config, 1);

    DoTestStopImageProcessor(handle);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// 検出する最小の輝度値を設定したテスト
//==================================================
void ClusteringProcessorTest::DoTestObjectIntensityMinBoundary(const nn::irsensor::IrCameraHandle& handle, int threshold) NN_NOEXCEPT
{
    NN_LOG("# ************************************************************\n");
    NN_LOG("#     ObjectIntensityMinBoundaryTest (ObjectIntensityMin: %d)\n", threshold);
    NN_LOG("# ************************************************************\n");
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::ClusteringProcessorConfig config;
    nn::irsensor::GetClusteringProcessorDefaultConfig(&config);

    config.objectIntensityMin = threshold;
    DoTestReadingClusteringStates(handle, config, 1);

    DoTestStopImageProcessor(handle);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// オブジェクトの最大・最小サイズを設定したテスト
//==================================================
void ClusteringProcessorTest::DoTestObjectPixelCountBoundary(const nn::irsensor::IrCameraHandle& handle, int min, int max) NN_NOEXCEPT
{
    NN_LOG("# ************************************************************\n");
    NN_LOG("#     ObjectPixelCountBoundaryTest (Min: %d - Max: %d)\n", min, max);
    NN_LOG("# ************************************************************\n");
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::ClusteringProcessorConfig config;
    nn::irsensor::GetClusteringProcessorDefaultConfig(&config);

    config.objectPixelCountMin = min;
    config.objectPixelCountMax = max;
    DoTestReadingClusteringStates(handle, config, 1);

    DoTestStopImageProcessor(handle);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ClusteringData の検証
//==================================================
void ClusteringProcessorTest::DoTestClusteringData(const nn::irsensor::ClusteringData& object, const nn::irsensor::ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    const auto& woi = config.windowOfInterest;

    EXPECT_LE(0, object.averageIntensity);
    EXPECT_LE(object.averageIntensity, nn::irsensor::IrCameraIntensityMax);

    EXPECT_LE(0, object.pixelCount);
    EXPECT_LE(object.pixelCount, nn::irsensor::ClusteringProcessorOutObjectPixelCountMax);
    EXPECT_LE(object.pixelCount, woi.width * woi.height);

    EXPECT_LE(0, object.bound.x);
    EXPECT_LE(object.bound.x, woi.width - 1);
    EXPECT_LE(0, object.bound.y);
    EXPECT_LE(object.bound.y, woi.height - 1);

    EXPECT_LE(0, object.bound.width);
    EXPECT_LE(object.bound.width, woi.width);
    EXPECT_LE(0, object.bound.height);
    EXPECT_LE(object.bound.height, woi.height);

    EXPECT_LE(object.bound.x, object.centroid.x);
    EXPECT_LE(object.centroid.x, object.bound.x + object.bound.width - 1);
    EXPECT_LE(object.bound.y, object.centroid.y);
    EXPECT_LE(object.centroid.y, object.bound.y + object.bound.height - 1);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータス取得のテスト
//==================================================
void ClusteringProcessorTest::DoTestReadingState(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::ClusteringProcessorConfig config;
    nn::irsensor::GetClusteringProcessorDefaultConfig(&config);

    // 5回ステータスを取得し値の確認を行います
    DoTestReadingClusteringStates(handle, config, VerifyStatesTestCountMax);

    // テスト終了後にプロセッサを停止します
    DoTestStopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// IRセンサの入力状態が正しく取得できるか
//==================================================
TEST_F(ClusteringProcessorTest, StateReading)
{
    nnt::gamepad::Initialize();

    // ProCon USB 無効
    ::nnt::gamepad::DisableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start Clustering Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");

        NNT_IRSENSOR_EXPECT_EXIT(DoTestReadingState(handle), NNT_IRSENSOR_EXIT_0, "");
        // WoiXBoundary
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, 0, nn::irsensor::IrCameraImageWidth, nn::irsensor::IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 1, 0, nn::irsensor::IrCameraImageWidth - 1, nn::irsensor::IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, nn::irsensor::IrCameraImageWidth - 1, 0, 1, nn::irsensor::IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
        // WoiYBoundary
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, 1, nn::irsensor::IrCameraImageWidth, nn::irsensor::IrCameraImageHeight - 1), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, nn::irsensor::IrCameraImageHeight - 2, nn::irsensor::IrCameraImageWidth, 2), NNT_IRSENSOR_EXIT_0, "");
        // WoiWidthBoundary
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, 0, 1, nn::irsensor::IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
        // WoiHeightBoundary
        // 高さを 1px 以下に設定するとクラスタリングデータを取得できなくなるため 2px 以上を設定します
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, 0, nn::irsensor::IrCameraImageWidth, 2), NNT_IRSENSOR_EXIT_0, "");
        // ObjectPixelCountBoundary
        NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(handle, 0, 0), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(handle, 0, nn::irsensor::ClusteringProcessorObjectPixelCountMax), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(handle, nn::irsensor::ClusteringProcessorObjectPixelCountMax, 0), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(handle, nn::irsensor::ClusteringProcessorObjectPixelCountMax, nn::irsensor::ClusteringProcessorObjectPixelCountMax), NNT_IRSENSOR_EXIT_0, "");
    }
    nnt::gamepad::DisconnectAll();
}

//==================================================
// ProコントローラのUSB有線接続が有効時にIRセンサの入力状態が正しく取得できるか
//==================================================
TEST_F(ClusteringProcessorTest, StateReadingForUsb)
{
    nnt::gamepad::Initialize();

    // ProCon USB 有効
    ::nnt::gamepad::EnableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start Clustering Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");

        NNT_IRSENSOR_EXPECT_EXIT(DoTestReadingState(handle), NNT_IRSENSOR_EXIT_0, "");
        // WoiXBoundary
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, 0, nn::irsensor::IrCameraImageWidth, nn::irsensor::IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 1, 0, nn::irsensor::IrCameraImageWidth - 1, nn::irsensor::IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, nn::irsensor::IrCameraImageWidth - 1, 0, 1, nn::irsensor::IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
        // WoiYBoundary
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, 1, nn::irsensor::IrCameraImageWidth, nn::irsensor::IrCameraImageHeight - 1), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, nn::irsensor::IrCameraImageHeight - 2, nn::irsensor::IrCameraImageWidth, 2), NNT_IRSENSOR_EXIT_0, "");
        // WoiWidthBoundary
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, 0, 1, nn::irsensor::IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
        // WoiHeightBoundary
        // 高さを 1px 以下に設定するとクラスタリングデータを取得できなくなるため 2px 以上を設定します
        NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(handle, 0, 0, nn::irsensor::IrCameraImageWidth, 2), NNT_IRSENSOR_EXIT_0, "");
        // ObjectPixelCountBoundary
        NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(handle, 0, 0), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(handle, 0, nn::irsensor::ClusteringProcessorObjectPixelCountMax), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(handle, nn::irsensor::ClusteringProcessorObjectPixelCountMax, 0), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(handle, nn::irsensor::ClusteringProcessorObjectPixelCountMax, nn::irsensor::ClusteringProcessorObjectPixelCountMax), NNT_IRSENSOR_EXIT_0, "");
    }
    nnt::gamepad::DisconnectAll();
}

} // namespace
