﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "../../Common/testGamePad_Common.h"
#include "../../Common/testGamePad_IrSensor.h"

namespace nnt { namespace irsensor {
    nn::irsensor::IrCameraHandle IrSensorTest::s_Handles[nnt::gamepad::NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}} // namaspace nnt::irsensor

namespace
{

const int VerifyStatesTestCountMax = 5;                                         // ステータスのテストを実施する回数
const nn::TimeSpan VerifyStatesTestLimitTime = nn::TimeSpan::FromSeconds(5);    // ステータスが更新されない状態の許容時間
const int NotReadyWaitLoopCountMax = 1000;                                      // ステータス取得の準備が完了するまでに待つ最大のループ回数
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);        // ステータス取得の準備が完了していない時のポーリング間隔

class AdaptiveClusteringProcessorTest : public nnt::irsensor::IrSensorTest
{
protected:

    void DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;

    void DoTestReadingAdaptiveClusteringState(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::AdaptiveClusteringProcessorConfig config) NN_NOEXCEPT;

    void DoTestVerifyAdaptiveClusteringState(const nn::irsensor::AdaptiveClusteringProcessorState& state, const nn::irsensor::AdaptiveClusteringProcessorConfig config) NN_NOEXCEPT;

    void DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;

    void ReadingStateTest(const nn::irsensor::IrCameraHandle& handle,
        const nn::irsensor::AdaptiveClusteringMode& mode,
        const nn::irsensor::AdaptiveClusteringTargetDistance& distance) NN_NOEXCEPT;
};

//==================================================
// プロセッサの起動までの時間を計測
//==================================================

void AdaptiveClusteringProcessorTest::DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    int counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();      // 計測を開始した時間を記憶

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Stopped)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            NN_LOG("#         Image processor started running. (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// 状態取得
//==================================================

void AdaptiveClusteringProcessorTest::DoTestReadingAdaptiveClusteringState(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::AdaptiveClusteringProcessorConfig config) NN_NOEXCEPT
{
    NN_LOG("#     <GetAdaptiveClusteringState>\n");
    int64_t prevSamplingNumber = -1;
    nn::TimeSpanType prevTimeStamp = nn::TimeSpanType::FromMilliSeconds(0);

    for (int i = 0; i < VerifyStatesTestCountMax; ++i)
    {
        nn::irsensor::AdaptiveClusteringProcessorState state[nn::irsensor::AdaptiveClusteringProcessorStateCountMax];

        NN_LOG("#     Test : [%2d/%2d]\n", i + 1, VerifyStatesTestCountMax);
        //-------------------------------------------------------------------------------
        // プロセッサが起動するまでの時間の計測します
        //-------------------------------------------------------------------------------
        if (i == 0)
        {
            // 初回のみ実行
            nn::irsensor::RunAdaptiveClusteringProcessor(handle, config);
            DoTestRunningProcessor(handle);
        }
        //-------------------------------------------------------------------------------
        // ステートが更新されるまでにかかる時間を計測します
        //-------------------------------------------------------------------------------
        auto samplingNumber = prevSamplingNumber;
        auto timeStamp = prevTimeStamp;
        auto stateCount = 0;
        auto startTime = nn::os::GetSystemTick().ToTimeSpan();                       // 計測を開始した時間を記憶

        while(NN_STATIC_CONDITION(true))
        {
            nn::Result result = nn::irsensor::GetAdaptiveClusteringProcessorStates(state, &stateCount, nn::irsensor::AdaptiveClusteringProcessorStateCountMax, 0, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result) == false)
            {
                if (nn::irsensor::ResultIrsensorUnavailable::Includes(result))
                {
                    NN_LOG("#             ResultIrsensorUnavailable\n");
                    NNT_IRSENSOR_EXIT_SUCCESS;
                    return;
                }
                if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorDeviceError #############\n");
                    return;
                }
                samplingNumber = state[0].samplingNumber;
                timeStamp = state[0].timeStamp;
            }
            if ((prevSamplingNumber != samplingNumber)
                && (prevTimeStamp != timeStamp))
            {
                break;
            }
            ASSERT_LT(nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds(), VerifyStatesTestLimitTime.GetMilliSeconds());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }

        NN_LOG("#             UpdateAdaptiveClusteringState (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
        NN_LOG("#             SamplingNumber %lld -> %lld\n", prevSamplingNumber, samplingNumber);
        NN_LOG("#             Timestamp %lld -> %lld[ms]\n", prevTimeStamp.GetMilliSeconds(), timeStamp.GetMilliSeconds());
        prevSamplingNumber = samplingNumber;
        prevTimeStamp = timeStamp;
        //-------------------------------------------------------------------------------
        // ステートの妥当性を検証します
        //-------------------------------------------------------------------------------
        NN_LOG("#         VerifyHandAnalysisSilhouetteStates (StateCount : %d)\n", stateCount);
        for (auto j = 0; j < stateCount; ++j)
        {
            DoTestVerifyAdaptiveClusteringState(state[j], config);
        }
    }

    NN_LOG("#     SUCCESS\n");

    NNT_IRSENSOR_EXIT_SUCCESS;
}

void AdaptiveClusteringProcessorTest::DoTestVerifyAdaptiveClusteringState(const nn::irsensor::AdaptiveClusteringProcessorState& state, const nn::irsensor::AdaptiveClusteringProcessorConfig config) NN_NOEXCEPT
{
    ASSERT_GE(state.samplingNumber, 0);
    ASSERT_GT(state.timeStamp.GetMicroSeconds(), 0);

    ASSERT_GE(state.ambientNoiseLevel, static_cast<nn::irsensor::IrCameraAmbientNoiseLevel>(0));
    ASSERT_LT(state.ambientNoiseLevel, static_cast<nn::irsensor::IrCameraAmbientNoiseLevel>(3));

    ASSERT_GE(state.objectCount, 0);
    ASSERT_LE(state.objectCount, nn::irsensor::AdaptiveClusteringProcessorObjectCountMax);

    ASSERT_GE(state.backgroundIntensity, 0);
    ASSERT_LE(state.backgroundIntensity, nn::irsensor::IrCameraIntensityMax);

    if (config.mode == nn::irsensor::AdaptiveClusteringMode_StaticFov)
    {
        ASSERT_EQ(state.accuracyLevel, static_cast<nn::irsensor::AdaptiveClusteringAccuracyLevel>(1));
    }
    else if (config.mode == nn::irsensor::AdaptiveClusteringMode_DynamicFov)
    {
        ASSERT_GE(state.accuracyLevel, static_cast<nn::irsensor::AdaptiveClusteringAccuracyLevel>(0));
        ASSERT_LE(state.accuracyLevel, static_cast<nn::irsensor::AdaptiveClusteringAccuracyLevel>(2));
    }

    // 浮動小数型にキャストして比較
    for (auto i = 0; i < state.objectCount; i++)
    {
        auto obj = state.objects[i];
        ASSERT_GE(obj.centroid.x, 0.0f);
        ASSERT_LE(obj.centroid.x, (static_cast<float>(nn::irsensor::IrCameraImageWidth)));
        ASSERT_GE(obj.centroid.y, 0.0f);
        ASSERT_LE(obj.centroid.y, (static_cast<float>(nn::irsensor::IrCameraImageHeight)));

        ASSERT_GE(obj.area, 0.0f);
        ASSERT_LE(obj.area, static_cast<float>(nn::irsensor::IrCameraImageWidth) * static_cast<float>(nn::irsensor::IrCameraImageHeight));
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

// 停止処理
void AdaptiveClusteringProcessorTest::DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessorAsync(handle);

    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("#     StopImageProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

void AdaptiveClusteringProcessorTest::ReadingStateTest(
    const nn::irsensor::IrCameraHandle& handle,
    const nn::irsensor::AdaptiveClusteringMode& mode,
    const nn::irsensor::AdaptiveClusteringTargetDistance& distance) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::AdaptiveClusteringProcessorConfig config;
    config.mode = mode;
    config.distance = distance;

    NN_LOG("# [AdaptiveClusteringMode : %s]\n",
        mode == nn::irsensor::AdaptiveClusteringMode::AdaptiveClusteringMode_StaticFov ?      "StaticFov" :
        mode == nn::irsensor::AdaptiveClusteringMode::AdaptiveClusteringMode_DynamicFov ?     "DynamicFov" :
        "UnKnown"
        );

    NN_LOG("# [AdaptiveClusteringTargetDistance : %s]\n",
        distance == nn::irsensor::AdaptiveClusteringTargetDistance::AdaptiveClusteringTargetDistance_Near ?    "Near" :
        distance == nn::irsensor::AdaptiveClusteringTargetDistance::AdaptiveClusteringTargetDistance_Middle ?  "Middle" :
        distance == nn::irsensor::AdaptiveClusteringTargetDistance::AdaptiveClusteringTargetDistance_Far ?     "Far" :
        "UnKnown"
        );

    DoTestReadingAdaptiveClusteringState(handle, config);
    DoTestStopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

// IRセンサの入力状態が正しく取得できるか
TEST_F(AdaptiveClusteringProcessorTest, StateReading)
{
    nnt::gamepad::Initialize();

    // ProCon USB 無効
    ::nnt::gamepad::DisableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    // IR センサの状態取得
    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];
        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        ReadingStateTest(handle, nn::irsensor::AdaptiveClusteringMode::AdaptiveClusteringMode_StaticFov, nn::irsensor::AdaptiveClusteringTargetDistance_Near);
        ReadingStateTest(handle, nn::irsensor::AdaptiveClusteringMode::AdaptiveClusteringMode_StaticFov, nn::irsensor::AdaptiveClusteringTargetDistance_Middle);
        ReadingStateTest(handle, nn::irsensor::AdaptiveClusteringMode::AdaptiveClusteringMode_StaticFov, nn::irsensor::AdaptiveClusteringTargetDistance_Far);
        ReadingStateTest(handle, nn::irsensor::AdaptiveClusteringMode::AdaptiveClusteringMode_DynamicFov, nn::irsensor::AdaptiveClusteringTargetDistance_Near);
        ReadingStateTest(handle, nn::irsensor::AdaptiveClusteringMode::AdaptiveClusteringMode_DynamicFov, nn::irsensor::AdaptiveClusteringTargetDistance_Middle);
        ReadingStateTest(handle, nn::irsensor::AdaptiveClusteringMode::AdaptiveClusteringMode_DynamicFov, nn::irsensor::AdaptiveClusteringTargetDistance_Far);
    }

    nnt::gamepad::DisconnectAll();
}

} // namespace
