﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <nn/os/os_Thread.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#if !defined(NN_SDK_BUILD_RELEASE)
// death テストをします。
TEST(FsVirtualMemoryStorageDeathTest, Death)
{
    nnt::fs::util::VirtualMemoryStorage storage;

    // 引数不正
    EXPECT_DEATH_IF_SUPPORTED(storage.Initialize(-1), "");

    // １回目の初期化は成功する
    storage.Initialize(0);

    // ２回目の初期化は失敗する
    EXPECT_DEATH_IF_SUPPORTED(storage.Initialize(256), "");
}
#endif

// 各関数の境界値分析をします。
TEST(FsVirtualMemoryStorageTest, Boundary)
{
    static const int64_t StorageSize = nnt::fs::util::VirtualMemoryStorage::BlockSize * 128;

    nnt::fs::util::VirtualMemoryStorage storage(StorageSize);
    nnt::fs::util::Vector<char> buf(128, 0);

    // Read のテスト
    {
        // offset の下限
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Read(0, buf.data(), buf.size())
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.Read(-1, buf.data(), buf.size())
        );

        // offset の上限
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Read(StorageSize, buf.data(), 0)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.Read(StorageSize + 1, buf.data(), 0)
        );

        // buffer の指定
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Read(0, nullptr, 0)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            storage.Read(0, nullptr, 1)
        );

        const int64_t offset = StorageSize - buf.size();

        // size の上限
#if defined(NN_BUILD_TARGET_PLATFORM_ADDRESS_64)
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            storage.Read(0, buf.data(), 0x8000000000000000)
        );
#endif
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Read(offset, buf.data(), buf.size())
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            storage.Read(offset + 1, buf.data(), buf.size())
        );
    }

    // Write のテスト
    {
        // offset の下限
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Write(0, buf.data(), buf.size())
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.Write(-1, buf.data(), buf.size())
        );

        // offset の上限
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Write(StorageSize, buf.data(), 0)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.Write(StorageSize + 1, buf.data(), 0)
        );

        // buffer の指定
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Write(0, nullptr, 0)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            storage.Write(0, nullptr, 1)
        );

        const int64_t offset = StorageSize - buf.size();

        // size の上限
#if defined(NN_BUILD_TARGET_PLATFORM_ADDRESS_64)
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            storage.Write(0, buf.data(), 0x8000000000000000)
        );
#endif
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Write(offset, buf.data(), buf.size())
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            storage.Write(offset + 1, buf.data(), buf.size())
        );
    }

    // OperateRange のテスト
    {
        typedef nn::fs::OperationId OperationId;

        // OperationId のテスト
        NNT_ASSERT_RESULT_SUCCESS(
            storage.OperateRange(OperationId::FillZero, 32, 32)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultUnsupportedOperation,
            storage.OperateRange(OperationId::DestroySignature, 32, 32)
        );

        // offset の下限
        NNT_ASSERT_RESULT_SUCCESS(
            storage.OperateRange(OperationId::FillZero, 0, 32)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.OperateRange(OperationId::FillZero, -1, 32)
        );

        // offset の上限
        NNT_ASSERT_RESULT_SUCCESS(
            storage.OperateRange(OperationId::FillZero, StorageSize, 0)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.OperateRange(OperationId::FillZero, StorageSize + 1, 0)
        );

        const int64_t offset = StorageSize - 32;

        // size の上限
        NNT_ASSERT_RESULT_SUCCESS(
            storage.OperateRange(OperationId::FillZero, offset, 32)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            storage.OperateRange(OperationId::FillZero, offset + 1, 32)
        );
    }
} // NOLINT(impl/function_size)

// ブロックの使用状況を考慮して書き込みのテストをします。
TEST(FsVirtualMemoryStorageTest, Write)
{
    static const size_t BlockSize = nnt::fs::util::VirtualMemoryStorage::BlockSize;
    static const size_t BufferSize = BlockSize * 4;
    static const int64_t StorageSize = BlockSize * 128;

    nnt::fs::util::VirtualMemoryStorage storage(StorageSize);
    nnt::fs::util::Vector<char> buf(BufferSize, 0);

    // ブロックサイズと一致する書き込みをして、ブロック数が１増えることを確認
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 0, buf.data(), BlockSize));
    EXPECT_EQ(1, storage.GetBlockCount());

    // ブロックサイズより小さい書き込みをして、ブロック数が１増えることを確認
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 1, buf.data(), 256));
    EXPECT_EQ(2, storage.GetBlockCount());

    // 1倍 < ブロックサイズ <= ２倍の書き込みをして、ブロック数が２増えることを確認
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 2, buf.data(), BlockSize + 256));
    EXPECT_EQ(4, storage.GetBlockCount());

    // 既存のブロックに収まる書き込みをしても、ブロックが増えないことを確認
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(256, buf.data(), 256));
    EXPECT_EQ(4, storage.GetBlockCount());

    // 既存のブロックに一部またがる書き込みをして、ブロック数が１増えることを確認
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 3 + 256, buf.data(), BlockSize));
    EXPECT_EQ(5, storage.GetBlockCount());

    // ↓のチェックの前準備
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 8, buf.data(), BlockSize));
    EXPECT_EQ(6, storage.GetBlockCount());

    // 既存のブロックとその前後を含む範囲に書き込みをして、ブロック数が２増えることを確認
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 7, buf.data(), BlockSize * 3));
    EXPECT_EQ(8, storage.GetBlockCount());

    // 連続した既存のブロックにまたがる書き込みをしても、ブロックが増えないことを確認
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 0, buf.data(), BufferSize));
    EXPECT_EQ(8, storage.GetBlockCount());
}

// ブロックの使用状況を考慮して読み込みのテストをします。
TEST(FsVirtualMemoryStorageTest, Read)
{
    static const size_t BlockSize = nnt::fs::util::VirtualMemoryStorage::BlockSize;
    static const size_t DataSize = BlockSize / 4;
    static const int64_t StorageSize = BlockSize * 128;

    nnt::fs::util::VirtualMemoryStorage storage(StorageSize);

    nnt::fs::util::Vector<char> zeroBuf(BlockSize, 0);
    nnt::fs::util::Vector<char> baseBuf(BlockSize, 'A');
    nnt::fs::util::Vector<char> readBuf(BlockSize);

    // ブロックのないところを読み込むと、データがすべて 0 であることを確認
    std::fill(readBuf.begin(), readBuf.end(), '0');
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, readBuf.data(), BlockSize));
    EXPECT_EQ(0, std::memcmp(readBuf.data(), zeroBuf.data(), BlockSize));

    NNT_ASSERT_RESULT_SUCCESS(storage.Write(DataSize, baseBuf.data(), DataSize));

    // 既存のブロック上の書き込んだ部分を読み込むと、書き込んだデータと一致することを確認
    std::fill(readBuf.begin(), readBuf.end(), '0');
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(DataSize, readBuf.data(), DataSize));
    EXPECT_EQ(0, std::memcmp(readBuf.data(), baseBuf.data(), DataSize));

    // 既存のブロック上の書き込んでいない部分を読み込むと、データがすべて 0 であることを確認
    std::fill(readBuf.begin(), readBuf.end(), '0');
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(DataSize * 2, readBuf.data(), DataSize));
    EXPECT_EQ(0, std::memcmp(readBuf.data(), zeroBuf.data(), DataSize));

    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize, baseBuf.data(), DataSize));

    // 既存のブロックをまたがる読み込みをして、ブロック上の書き込んだ部分は一致し、それ以外がすべて 0 であることを確認
    std::fill(readBuf.begin(), readBuf.end(), '0');
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(DataSize, readBuf.data(), BlockSize));
    EXPECT_EQ(0, std::memcmp(readBuf.data(), baseBuf.data(), DataSize));
    EXPECT_EQ(0, std::memcmp(readBuf.data() + DataSize, zeroBuf.data(), DataSize * 2));
    EXPECT_EQ(0, std::memcmp(readBuf.data() + DataSize * 3, baseBuf.data(), DataSize));
}

// ランダムなオフセットとサイズで読み書きするテストをします。
TEST(FsVirtualMemoryStorageTest, RandomAccess)
{
    typedef std::pair<int64_t, size_t> DataRange;

    static const int DataCount = 200;
    static const size_t DataSizeMax = nnt::fs::util::VirtualMemoryStorage::BlockSize * 8;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t StorageSize = 4 * 1024 * Giga; // 4TB

    std::mt19937 mt(1);
    nnt::fs::util::Vector<char> buf(DataSizeMax, 0);
    nnt::fs::util::Vector<DataRange> dataRange(DataCount);
    nnt::fs::util::VirtualMemoryStorage storage(StorageSize);

    NN_LOG("Write");

    // データの書き込み
    for( int i = 0; i < DataCount; ++i )
    {
        auto offset = std::uniform_int_distribution<int64_t>(4, StorageSize - 8)(mt);
        auto size = std::uniform_int_distribution<size_t>(0, DataSizeMax - 8)(mt);

        if( StorageSize - 4 < offset + static_cast<int64_t>(size) )
        {
            size = static_cast<size_t>(StorageSize - 4 - offset);
        }

        std::memset(buf.data(), i + 1, size);

        NNT_ASSERT_RESULT_SUCCESS(storage.Write(offset, buf.data(), size));

        dataRange[i] = DataRange(offset, size);

        if( i % 10 == 9 )
        {
            NN_LOG(".");
        }
    }

    NN_LOG("\nRead");

    const char* const data = buf.data();

    // データの読み込み・チェック
    for( int i = 0; i < DataCount; ++i )
    {
        DataRange range = dataRange[i];

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(range.first - 4, buf.data(), range.second + 8));

        // 該当データが書き込まれていないことを確認
        for( size_t j = 0; j < 4; ++j )
        {
            EXPECT_NE(i + 1, static_cast<uint8_t>(data[j]));
        }

        for( size_t j = 4; j < range.second + 4; ++j )
        {
            // 次の書き込みでの上書きを考慮して、i+1 <= data[j] でチェック
            EXPECT_LE(i + 1, static_cast<uint8_t>(data[j]));
        }

        // 該当データが書き込まれていないことを確認
        for( size_t j = range.second + 4; j < range.second + 8; ++j )
        {
            EXPECT_NE(i + 1, static_cast<uint8_t>(data[j]));
        }

        if( i % 10 == 9 )
        {
            NN_LOG(".");
        }
    }

    NN_LOG("\n");
}

// ブロックの使用状況を考慮して OperateRange（0 埋め）のテストをします。
TEST(FsVirtualMemoryStorageTest, OperateRange)
{
    typedef nn::fs::OperationId OperationId;

    static const size_t BlockSize = nnt::fs::util::VirtualMemoryStorage::BlockSize;
    static const size_t DataSize = BlockSize / 4;
    static const int64_t StorageSize = BlockSize * 128;

    nnt::fs::util::VirtualMemoryStorage storage(StorageSize);

    nnt::fs::util::Vector<char> zeroBuf(BlockSize, 0);
    nnt::fs::util::Vector<char> baseBuf(BlockSize * 2, 'A');
    nnt::fs::util::Vector<char> readBuf(BlockSize * 2);

    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize, baseBuf.data(), BlockSize));
    EXPECT_EQ(1, storage.GetBlockCount());

    // ブロックのないところを 0 埋めしても、何も起きないことを確認
    NNT_ASSERT_RESULT_SUCCESS(storage.OperateRange(OperationId::FillZero, 0, BlockSize));
    EXPECT_EQ(1, storage.GetBlockCount());

    // 既存のブロックの一部を 0 埋めしても、ブロックが減らないことを確認
    // （前後の 0 埋めチェック付）
    {
        std::fill(readBuf.begin(), readBuf.end(), '0');
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(BlockSize, readBuf.data(), DataSize * 2));
        EXPECT_EQ(0, std::memcmp(readBuf.data(), baseBuf.data(), DataSize * 2));

        NNT_ASSERT_RESULT_SUCCESS(
            storage.OperateRange(OperationId::FillZero, BlockSize, DataSize));
        EXPECT_EQ(1, storage.GetBlockCount());

        std::fill(readBuf.begin(), readBuf.end(), '0');
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(BlockSize, readBuf.data(), DataSize * 2));
        EXPECT_EQ(0, std::memcmp(readBuf.data(), zeroBuf.data(), DataSize));
        EXPECT_EQ(0, std::memcmp(readBuf.data() + DataSize, baseBuf.data(), DataSize));
    }

    // 既存のブロック全体を 0 埋めして、ブロック数が１減ることを確認
    NNT_ASSERT_RESULT_SUCCESS(
        storage.OperateRange(OperationId::FillZero, BlockSize, BlockSize));
    EXPECT_EQ(0, storage.GetBlockCount());

    NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, baseBuf.data(), BlockSize));
    EXPECT_EQ(1, storage.GetBlockCount());

    // 既存のブロックを一部またぐ 0 埋めをしても、ブロックが減らないことを確認
    // （前後の 0 埋めチェック付）
    {
        std::fill(readBuf.begin(), readBuf.end(), '0');
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, readBuf.data(), BlockSize));
        EXPECT_EQ(0, std::memcmp(readBuf.data(), baseBuf.data(), BlockSize));

        NNT_ASSERT_RESULT_SUCCESS(
            storage.OperateRange(OperationId::FillZero, DataSize, BlockSize));
        EXPECT_EQ(1, storage.GetBlockCount());

        std::fill(readBuf.begin(), readBuf.end(), '0');
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, readBuf.data(), BlockSize));
        EXPECT_EQ(0, std::memcmp(readBuf.data(), baseBuf.data(), DataSize));
        EXPECT_EQ(0, std::memcmp(readBuf.data() + DataSize, zeroBuf.data(), DataSize * 3));
    }

    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 2, baseBuf.data(), BlockSize));
    EXPECT_EQ(2, storage.GetBlockCount());

    // 既存のブロックとその前後を含む範囲を 0 埋めをして、ブロック数が１減ることを確認
    NNT_ASSERT_RESULT_SUCCESS(
        storage.OperateRange(OperationId::FillZero, BlockSize + DataSize, BlockSize * 2));
    EXPECT_EQ(1, storage.GetBlockCount());

    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 0, baseBuf.data(), BlockSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BlockSize * 1, baseBuf.data(), BlockSize));
    EXPECT_EQ(2, storage.GetBlockCount());

    // 連続した既存のブロックをまたぐ 0 埋めをしても、ブロックが減らないことを確認
    // （前後の 0 埋めチェック付）
    {
        std::fill(readBuf.begin(), readBuf.end(), '0');
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, readBuf.data(), BlockSize * 2));
        EXPECT_EQ(0, std::memcmp(readBuf.data(), baseBuf.data(), BlockSize * 2));

        NNT_ASSERT_RESULT_SUCCESS(
            storage.OperateRange(OperationId::FillZero, DataSize, BlockSize));
        EXPECT_EQ(2, storage.GetBlockCount());

        std::fill(readBuf.begin(), readBuf.end(), '0');
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, readBuf.data(), BlockSize * 2));
        EXPECT_EQ(0, std::memcmp(
            readBuf.data(), baseBuf.data(), DataSize));
        EXPECT_EQ(0, std::memcmp(
            readBuf.data() + DataSize, zeroBuf.data(), BlockSize));
        EXPECT_EQ(0, std::memcmp(
            readBuf.data() + DataSize + BlockSize, baseBuf.data(), BlockSize - DataSize));
    }

    // 連続した既存のブロックをまたぎ一方を全て 0 埋めをして、ブロック数が１減ることを確認
    NNT_ASSERT_RESULT_SUCCESS(
        storage.OperateRange(OperationId::FillZero, 0, BlockSize + DataSize));
    EXPECT_EQ(1, storage.GetBlockCount());
}

// メモリ操作以外の関数のテストをします。
TEST(FsVirtualMemoryStorageTest, Utility)
{
    static const int64_t StorageSize = 1024 * 1024 * 1024;

    nnt::fs::util::VirtualMemoryStorage storage;

    // 初期化前でも使えるが何もしない
    storage.Finalize();

    // 初期化前でも使えるが必ず 0 を返す
    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(storage.GetSize(&size));
    EXPECT_EQ(0, size);
    EXPECT_EQ(0, storage.GetSize());

    storage.Initialize(StorageSize);

    // 初期化後は StorageSize を返す
    NNT_ASSERT_RESULT_SUCCESS(storage.GetSize(&size));
    EXPECT_EQ(StorageSize, size);
    EXPECT_EQ(StorageSize, storage.GetSize());

    // 必ず成功するが何もしない
    NNT_ASSERT_RESULT_SUCCESS(storage.Flush());

    storage.Finalize();

    // 終了処理後でも使えるが必ず 0 を返す
    NNT_ASSERT_RESULT_SUCCESS(storage.GetSize(&size));
    EXPECT_EQ(0, size);
    EXPECT_EQ(0, storage.GetSize());
}

// マルチスレッドのテストをします。
TEST(FsVirtualMemoryStorageTest, MultiThread)
{
    typedef nn::fs::OperationId OperationId;

    static const auto StackSize = 32 * 1024;
    static const auto ThreadCount = 8;
    static const auto StorageSize = 2 * 1024 * 1024;
    static const auto WorkSize = 64 * 1024;

    NN_OS_ALIGNAS_THREAD_STACK static char s_ThreadStack[StackSize * ThreadCount] = {};

    struct Argument
    {
        nnt::fs::util::VirtualMemoryStorage* pStorage;
        std::mt19937::result_type seed;
    };

    nnt::fs::util::VirtualMemoryStorage storage(StorageSize);

    const auto threadFunc = [](void* pArg) NN_NOEXCEPT
    {
        Argument& arg = *reinterpret_cast<Argument*>(pArg);
        nnt::fs::util::Vector<char> buf(WorkSize, 'A');
        std::mt19937 mt(arg.seed);

        for( int i = 0; i < 4000; ++i )
        {
            const auto offset = std::uniform_int_distribution<>(0, WorkSize)(mt);
            const auto size = std::uniform_int_distribution<>(0, WorkSize - offset)(mt);

            switch( std::uniform_int_distribution<>(0, 2)(mt) )
            {
            // 読み込み
            case 0:
                NNT_ASSERT_RESULT_SUCCESS(arg.pStorage->Read(offset, buf.data(), size));
                break;

            // 書き込み
            case 1:
                NNT_ASSERT_RESULT_SUCCESS(arg.pStorage->Write(offset, buf.data(), size));
                break;

            // 0 埋め
            default:
                NNT_ASSERT_RESULT_SUCCESS(
                    arg.pStorage->OperateRange(OperationId::FillZero, offset, size));
                break;
            }
        }
    };

    nn::os::ThreadType threads[ThreadCount];
    Argument args[ThreadCount];

    // スレッドの実行
    for( auto i = 0; i < ThreadCount; ++i )
    {
        args[i].pStorage = &storage;
        args[i].seed = i + 1;

        NNT_ASSERT_RESULT_SUCCESS(
            nn::os::CreateThread(
                &threads[i],
                threadFunc,
                &args[i],
                s_ThreadStack + StackSize * i,
                StackSize,
                nn::os::DefaultThreadPriority
            )
        );
        nn::os::StartThread(&threads[i]);
    }

    // スレッドの後始末
    for( auto& thread : threads )
    {
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
    }
}

// SetSize のテストをします。
TEST(FsVirtualMemoryStorageTest, SetSize)
{
    static const size_t BlockSize = nnt::fs::util::VirtualMemoryStorage::BlockSize;
    static const size_t BufferSize = BlockSize * 4;
    static const int64_t StorageSize = BlockSize * 128;

    auto readBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
    auto writeBuffer = nnt::fs::util::AllocateBuffer(BufferSize);

    nnt::fs::util::VirtualMemoryStorage storage(StorageSize);

    auto readWriteOutOfRange = [&](int64_t size) NN_NOEXCEPT
    {
        // 範囲外への読み込みが失敗する
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.Read(size + 1, readBuffer.get(), BufferSize));
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            storage.Read(size - 1, readBuffer.get(), BufferSize));

        // 範囲外への書き込みが失敗する
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.Write(size + 1, writeBuffer.get(), BufferSize));
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            storage.Write(size - 1, writeBuffer.get(), BufferSize));
    };

    // ストレージの範囲外への読み書き
    readWriteOutOfRange(StorageSize);

    // 拡張前に書き込む
    nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), BufferSize);
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(StorageSize - BufferSize, writeBuffer.get(), BufferSize));

    // 拡張前のブロック数を確認
    ASSERT_EQ((BufferSize + BlockSize - 1) / BlockSize, storage.GetBlockCount());

    // サイズを拡張する
    const int64_t expandedSize = StorageSize + BufferSize * 2;
    NNT_ASSERT_RESULT_SUCCESS(storage.SetSize(expandedSize));
    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(storage.GetSize(&size));
    ASSERT_EQ(expandedSize, size);

    // ブロック数は拡張前と同じ
    ASSERT_EQ((BufferSize + BlockSize - 1) / BlockSize, storage.GetBlockCount());

    // 拡張前に書き込んだデータが読み込める
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(StorageSize - BufferSize, readBuffer.get(), BufferSize));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), BufferSize);

    // 拡張した範囲外への読み書き
    readWriteOutOfRange(expandedSize);

    // 拡張した範囲へ読み書き
    nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), BufferSize);
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(expandedSize - BufferSize, writeBuffer.get(), BufferSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(expandedSize - BufferSize, readBuffer.get(), BufferSize));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), BufferSize);

    // 縮小前に書き込む
    nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), BufferSize);
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(StorageSize - BufferSize * 2, writeBuffer.get(), BufferSize));

    // 縮小前のブロック数を確認
    ASSERT_EQ((BufferSize * 3 + BlockSize - 1) / BlockSize, storage.GetBlockCount());

    // サイズを縮小する
    const int64_t shrinkedSize = StorageSize - BufferSize;
    NNT_ASSERT_RESULT_SUCCESS(storage.SetSize(shrinkedSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.GetSize(&size));
    ASSERT_EQ(shrinkedSize, size);

    // 縮小した範囲に書き込んでいた分ブロック数が減っている
    ASSERT_EQ((BufferSize + BlockSize - 1) / BlockSize, storage.GetBlockCount());

    // 縮小前に書き込んだデータが読み込める
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(StorageSize - BufferSize * 2, readBuffer.get(), BufferSize));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), BufferSize);

    // 縮小後の範囲外への読み書き
    readWriteOutOfRange(shrinkedSize);

    // 縮小後の範囲へ読み書き
    nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), BufferSize);
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(shrinkedSize - BufferSize, writeBuffer.get(), BufferSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(shrinkedSize - BufferSize, readBuffer.get(), BufferSize));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), BufferSize);

    // サイズを元に戻す
    NNT_ASSERT_RESULT_SUCCESS(storage.SetSize(StorageSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.GetSize(&size));
    ASSERT_EQ(StorageSize, size);

    // 一度縮小した範囲は 0 フィルされている
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(StorageSize - BufferSize, readBuffer.get(), BufferSize));
    std::memset(writeBuffer.get(), 0, BufferSize);
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), BufferSize);
}

// 4 GB を超えるオフセットのテスト
TEST(FsVirtualMemoryStorageTest, ReadWriteLargeOffset)
{
    static const size_t AccessSize = 1024;
    nnt::fs::util::VirtualMemoryStorage storage(nnt::fs::util::LargeOffsetMax + AccessSize);

    nnt::fs::util::TestStorageAccessWithLargeOffset(&storage, AccessSize);
}
