﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

static const int LoopCount = 10;

static const size_t SIZE_TABLE_SMALL[] = {
    1, 15, 1023, 1024, 16 * 1024
};

static const size_t SIZE_TABLE_LARGE[] = {
    1, 15, 1023, 1024, 512 * 1024
};

class SafeMemoryStorageTest : public ::testing::TestWithParam< size_t >
{
protected:
    std::mt19937 m_Random;

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Random.seed(1);
    }
};

// 初期化テスト
class SafeMemoryStorageInitializeTest : public SafeMemoryStorageTest { };

TEST_P(SafeMemoryStorageInitializeTest, Test)
{
    const size_t BufferSize = GetParam();
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> storage(
        new nnt::fs::util::SafeMemoryStorage(BufferSize)
    );
    EXPECT_EQ(true, storage->CheckValid());
}

INSTANTIATE_TEST_CASE_P(
    SafeMemoryStorageTest,
    SafeMemoryStorageInitializeTest,
    ::testing::ValuesIn(SIZE_TABLE_LARGE)
);

// 範囲内を読み込んでエラーが発生しないことを確認します。
class SafeMemoryStorageReadStandardTest : public SafeMemoryStorageTest { };

TEST_P(SafeMemoryStorageReadStandardTest, Test)
{
    const size_t BufferSize = GetParam();
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> storage(
        new nnt::fs::util::SafeMemoryStorage(BufferSize)
    );
    for( int i = 0; i < LoopCount; ++i )
    {
        std::unique_ptr<char[]> buf(new char [BufferSize]);
        int64_t offset = std::uniform_int_distribution<int64_t>(0, BufferSize - 1)(m_Random);
        size_t sizeRead
            = std::uniform_int_distribution<size_t>(
                1,
                static_cast<size_t>(BufferSize - offset)
              )(m_Random);
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(offset, buf.get(), sizeRead));
    }
    EXPECT_EQ(true, storage->CheckValid());
}

INSTANTIATE_TEST_CASE_P(
    SafeMemoryStorageTest,
    SafeMemoryStorageReadStandardTest,
    ::testing::ValuesIn(SIZE_TABLE_LARGE)
);

// 範囲外を読み込んでエラーが発生することを確認します。
class SafeMemoryStorageReadOutOfRangeTest : public SafeMemoryStorageTest { };

TEST_P(SafeMemoryStorageReadOutOfRangeTest, Test)
{
    const size_t BufferSize = GetParam();
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> storage(
        new nnt::fs::util::SafeMemoryStorage(BufferSize)
    );
    for( int i = 0; i < LoopCount; ++i )
    {
        std::unique_ptr<char[]> buf(new char [BufferSize]);
        // オフセットの時点で範囲外になるよう乱数を生成します。
        // TODO : SafeMemoryStorage では
        //        offset == BufferSize && sizeRead > 0 のときに
        //        失敗すべきだが失敗しない
        //        正しくは (BufferSize, BufferSize + 10000) とあるべき
        int64_t offset
            = std::uniform_int_distribution<int64_t>(BufferSize + 1, BufferSize + 10000)(m_Random);
        size_t sizeRead = std::uniform_int_distribution<size_t>(1, 10000)(m_Random);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage->Read(offset, buf.get(), sizeRead)
        );
    }
    EXPECT_EQ(true, storage->CheckValid());
}

INSTANTIATE_TEST_CASE_P(
    SafeMemoryStorageTest,
    SafeMemoryStorageReadOutOfRangeTest,
    ::testing::ValuesIn(SIZE_TABLE_LARGE)
);

// 範囲外を含む読み込みでもオフセットが範囲内であればエラーが発生しないことを確認します。
class SafeMemoryStorageReadPartialTest : public SafeMemoryStorageTest { };

TEST_P(SafeMemoryStorageReadPartialTest, Test)
{
    const size_t BufferSize = GetParam();
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> storage(
        new nnt::fs::util::SafeMemoryStorage(BufferSize)
    );
    for( int i = 0; i < LoopCount; ++i )
    {
        std::unique_ptr<char[]> buf(new char [BufferSize]);
        size_t offset = std::uniform_int_distribution<size_t>(0, BufferSize - 1)(m_Random);
        size_t sizeRead
            = std::uniform_int_distribution<size_t>(BufferSize - offset + 1, BufferSize + 1)(m_Random);
        // オフセット+サイズが範囲外でもオフセットが範囲内ならエラーにならないことを確認します。
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(offset, buf.get(), sizeRead));
    }
    EXPECT_EQ(true, storage->CheckValid());
}

INSTANTIATE_TEST_CASE_P(
    SafeMemoryStorageTest,
    SafeMemoryStorageReadPartialTest,
    ::testing::ValuesIn(SIZE_TABLE_LARGE)
);

// 範囲内に書き込んでエラーが発生しないことを確認します。
class SafeMemoryStorageWriteStandardTest : public SafeMemoryStorageTest { };

TEST_P(SafeMemoryStorageWriteStandardTest, Test)
{
    const size_t BufferSize = GetParam();
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> storage(
        new nnt::fs::util::SafeMemoryStorage(BufferSize)
    );
    for( int i = 0; i < LoopCount; ++i )
    {
        std::unique_ptr<char[]> buf(new char [BufferSize]);
        int64_t offset = std::uniform_int_distribution<int64_t>(0, BufferSize - 1)(m_Random);
        size_t sizeWrite
            = std::uniform_int_distribution<size_t>(
                1,
                static_cast<size_t>(BufferSize - offset)
              )(m_Random);
        NNT_ASSERT_RESULT_SUCCESS(storage->Write(offset, buf.get(), sizeWrite));
    }
    EXPECT_EQ(true, storage->CheckValid());
}

INSTANTIATE_TEST_CASE_P(
    SafeMemoryStorageTest,
    SafeMemoryStorageWriteStandardTest,
    ::testing::ValuesIn(SIZE_TABLE_LARGE)
);

// 範囲外に書き込んでエラーが発生することを確認します。
class SafeMemoryStorageWriteOutOfRangeTest : public SafeMemoryStorageTest { };

TEST_P(SafeMemoryStorageWriteOutOfRangeTest, Test)
{
    const size_t BufferSize = GetParam();
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> storage(
        new nnt::fs::util::SafeMemoryStorage(BufferSize)
    );
    for( int i = 0; i < LoopCount; ++i )
    {
        std::unique_ptr<char[]> buf(new char [BufferSize]);
        // オフセットの時点で範囲外になるよう乱数を生成します。
        // TODO : SafeMemoryStorage では
        //        offset == BufferSize && sizeWrite > 0 のときに
        //        失敗すべきだが失敗しない
        //        正しくは (BufferSize, BufferSize + 10000) とあるべき
        int64_t offset
            = std::uniform_int_distribution<int64_t>(BufferSize + 1, BufferSize + 10000)(m_Random);
        size_t sizeWrite = std::uniform_int_distribution<size_t>(1, 10000)(m_Random);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage->Write(offset, buf.get(), sizeWrite)
        );
    }
    EXPECT_EQ(true, storage->CheckValid());
}

INSTANTIATE_TEST_CASE_P(
    SafeMemoryStorageTest,
    SafeMemoryStorageWriteOutOfRangeTest,
    ::testing::ValuesIn(SIZE_TABLE_LARGE)
);

// 範囲外を含む書き込みでもオフセットが範囲内であればエラーが発生しないことを確認します。
class SafeMemoryStorageWritePartialTest : public SafeMemoryStorageTest { };

TEST_P(SafeMemoryStorageWritePartialTest, Test)
{
    const size_t BufferSize = GetParam();
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> storage(
        new nnt::fs::util::SafeMemoryStorage(BufferSize)
    );
    for( int i = 0; i < LoopCount; ++i )
    {
        // offset が 0 でも範囲外まで書き込むサイズのバッファを作成
        std::unique_ptr<char[]> buf(new char [BufferSize + 1]);
        size_t offset = std::uniform_int_distribution<size_t>(0, BufferSize - 1)(m_Random);
        size_t sizeWrite
            = std::uniform_int_distribution<size_t>(BufferSize - offset + 1, BufferSize + 1)(m_Random);
        NNT_ASSERT_RESULT_SUCCESS(storage->Write(offset, buf.get(), sizeWrite));
    }
    EXPECT_EQ(true, storage->CheckValid());
}

INSTANTIATE_TEST_CASE_P(
    SafeMemoryStorageTest,
    SafeMemoryStorageWritePartialTest,
    ::testing::ValuesIn(SIZE_TABLE_LARGE)
);

// 全領域リードライトしてデータが化けないことを確認します。
class SafeMemoryStorageReadWriteAllTest : public SafeMemoryStorageTest { };

TEST_P(SafeMemoryStorageReadWriteAllTest, Test)
{
    const size_t BufferSize = GetParam();

    auto seed = 1;
    std::mt19937 mtSrc(seed);
    std::mt19937 mtDst(seed);

    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> storage(
        new nnt::fs::util::SafeMemoryStorage(BufferSize)
    );
    for( int i = 0; i < LoopCount; ++i )
    {
        std::unique_ptr<uint8_t[]> bufWrite(new uint8_t [BufferSize]);
        std::unique_ptr<uint8_t[]> bufRead(new uint8_t [BufferSize]);
        for( size_t indexBuf = 0; indexBuf < BufferSize; ++indexBuf )
        {
            bufWrite[indexBuf]
                = static_cast<uint8_t>(std::uniform_int_distribution<int>(0, 0xFF)(mtSrc));
        }
        NNT_EXPECT_RESULT_SUCCESS(storage->Write(0, bufWrite.get(), BufferSize));
        NNT_EXPECT_RESULT_SUCCESS(storage->Read(0, bufRead.get(), BufferSize));
        for( size_t indexBuf = 0; indexBuf < BufferSize; ++indexBuf )
        {
            uint8_t x = static_cast<uint8_t>(std::uniform_int_distribution<int>(0, 0xFF)(mtDst));
            ASSERT_EQ(x, bufRead[indexBuf]);
        }
    }
    EXPECT_EQ(true, storage->CheckValid());
}

INSTANTIATE_TEST_CASE_P(
    SafeMemoryStorageTest,
    SafeMemoryStorageReadWriteAllTest,
    ::testing::ValuesIn(SIZE_TABLE_SMALL)
);

// ランダムリードライトしてデータが化けないことを確認します。
class SafeMemoryStorageReadWriteRandomTest : public SafeMemoryStorageTest { };

TEST_P(SafeMemoryStorageReadWriteRandomTest, Test)
{
    const size_t BufferSize = GetParam();

    auto seed = 1;
    std::mt19937 mtSrc(seed);
    std::mt19937 mtDst(seed);

    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> storage(
        new nnt::fs::util::SafeMemoryStorage(BufferSize)
    );
    for( int i = 0; i < LoopCount; ++i )
    {
        int64_t offset = std::uniform_int_distribution<int64_t>(0, BufferSize - 1)(m_Random);
        size_t sizeAccess
            = std::uniform_int_distribution<size_t>(
                1,
                static_cast<size_t>(BufferSize - offset)
              )(m_Random);
        std::unique_ptr<uint8_t[]> bufWrite(new uint8_t [BufferSize]);
        std::unique_ptr<uint8_t[]> bufRead(new uint8_t [BufferSize]);
        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            bufWrite[indexBuf]
                = static_cast<uint8_t>(std::uniform_int_distribution<>(0, 0xFF)(mtSrc));
        }
        NNT_EXPECT_RESULT_SUCCESS(storage->Write(offset, bufWrite.get(), sizeAccess));
        NNT_EXPECT_RESULT_SUCCESS(storage->Read(offset, bufRead.get(), sizeAccess));
        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            uint8_t x = static_cast<uint8_t>(std::uniform_int_distribution<int>()(mtDst));
            ASSERT_EQ(x, bufRead[indexBuf]);
        }
    }
    EXPECT_EQ(true, storage->CheckValid());
}

INSTANTIATE_TEST_CASE_P(
    SafeMemoryStorageTest,
    SafeMemoryStorageReadWriteRandomTest,
    ::testing::ValuesIn(SIZE_TABLE_LARGE)
);
