﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <array>
#include <algorithm>
#include <random>
#include <cstring>
#include <thread>

#include <nn/fs.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fssystem/fs_AesXtsFileSystem.h>
#include <nn/fssystem/fs_HostFileSystem.h>
#include <nn/fssystem/fs_StorageFile.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nnt_Argument.h>

#include "testFs_Unit_StorageLayerTestCase.h"


using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nn::fssystem;
using namespace nnt::fs::util;

using nn::Bit8;

namespace {

nnt::fs::util::TemporaryHostDirectory g_HostDirectory;

}

// 書いた内容が読めること
void TestWriteReadStorage(IStorage* pStorage, size_t bufferSize)
{
    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&size));

    ASSERT_GE(size, static_cast<int64_t>(bufferSize * 4));

    std::unique_ptr<char[]> writeBuffer[3];
    for(auto& buffer : writeBuffer)
    {
        buffer.reset(new char[bufferSize]);
        FillBufferWithRandomValue(buffer.get(), bufferSize);
    }

    // 頭、真ん中、最後のみ
    NNT_EXPECT_RESULT_SUCCESS(pStorage->Write(0,                 writeBuffer[0].get(), bufferSize));
    NNT_EXPECT_RESULT_SUCCESS(pStorage->Write(size / 2,          writeBuffer[1].get(), bufferSize));
    NNT_EXPECT_RESULT_SUCCESS(pStorage->Write(size - bufferSize, writeBuffer[2].get(), bufferSize));

    std::unique_ptr<char[]> readBuffer[3];
    for(auto& buffer : readBuffer)
    {
        buffer.reset(new char[bufferSize]);
        memset(buffer.get(), 0x00, bufferSize);
    }

    NNT_EXPECT_RESULT_SUCCESS(pStorage->Read(0,                 readBuffer[0].get(), bufferSize));
    NNT_EXPECT_RESULT_SUCCESS(pStorage->Read(size / 2,          readBuffer[1].get(), bufferSize));
    NNT_EXPECT_RESULT_SUCCESS(pStorage->Read(size - bufferSize, readBuffer[2].get(), bufferSize));

    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer[0].get(), readBuffer[0].get(), bufferSize);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer[1].get(), readBuffer[1].get(), bufferSize);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer[2].get(), readBuffer[2].get(), bufferSize);

}

// 4 GB を超えるオフセットの読み書き
void TestWriteReadStorageWithLargeOffset(nn::fs::IStorage* pStorage, size_t bufferSize, char* writeBuffer) NN_NOEXCEPT
{
    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&size));
    ASSERT_GE(size, static_cast<int64_t>(64) * 1024 * 1024 * 1024 + static_cast<int64_t>(bufferSize));

    std::unique_ptr<char> readBuffer(new char[bufferSize]);
    std::unique_ptr<char> writeBufferArray[LargeOffsetListLength];

    for( int i = 0; i < LargeOffsetListLength; ++i )
    {
        const int64_t offset = LargeOffsetList[i];
        writeBufferArray[i].reset(new char[bufferSize]);
        FillBufferWithRandomValue(writeBufferArray[i].get(), bufferSize);
        if( writeBuffer != nullptr )
        {
            std::memcpy(writeBuffer, writeBufferArray[i].get(), bufferSize);
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(offset, writeBuffer, bufferSize));
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(offset, writeBufferArray[i].get(), bufferSize));
        }
    }

    for( int i = 0; i < LargeOffsetListLength; ++i )
    {
        const int64_t offset = LargeOffsetList[i];
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(offset, readBuffer.get(), bufferSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferArray[i].get(), readBuffer.get(), bufferSize);
    }
}

void TestWriteReadStorageWithLargeOffset(nn::fs::IStorage* pStorage, size_t bufferSize) NN_NOEXCEPT
{
    TestWriteReadStorageWithLargeOffset(pStorage, bufferSize, nullptr);
}

void TestWriteReadStorageWithLargeOffset(nn::fs::fsa::IFile* pFile, size_t bufferSize) NN_NOEXCEPT
{
    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
    ASSERT_GE(size, static_cast<int64_t>(64) * 1024 * 1024 * 1024 + static_cast<int64_t>(bufferSize));

    std::unique_ptr<char> readBuffer(new char[bufferSize]);
    std::unique_ptr<char> writeBufferArray[LargeOffsetListLength];
    for( auto& writeBuffer : writeBufferArray )
    {
        writeBuffer.reset(new char[bufferSize]);
        FillBufferWithRandomValue(writeBuffer.get(), bufferSize);
    }

    // 書き込み
    for( int i = 0; i < LargeOffsetListLength; ++i )
    {
        const int64_t offset = LargeOffsetList[i];

        NNT_ASSERT_RESULT_SUCCESS(pFile->Write(offset, writeBufferArray[i].get(), bufferSize, nn::fs::WriteOption()));
    }

    // 書き込んだデータを読み込む
    size_t readSize = 0;
    for( int i = 0; i < LargeOffsetListLength; ++i )
    {
        const int64_t offset = LargeOffsetList[i];
        NNT_ASSERT_RESULT_SUCCESS(pFile->Read(&readSize, offset, readBuffer.get(), bufferSize, nn::fs::ReadOption()));
        ASSERT_EQ(bufferSize, readSize);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferArray[i].get(), readBuffer.get(), bufferSize);
    }

    // 範囲外への書き込み
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        pFile->Write(size, writeBufferArray[0].get(), bufferSize, nn::fs::WriteOption()));

    // 範囲外の読み込み
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        pFile->Read(&readSize, size + 1, readBuffer.get(), bufferSize, nn::fs::ReadOption()));
}

void TestWriteReadStorageWithLargeOffsetAllowAppend(nn::fs::fsa::IFile* pFile, size_t bufferSize) NN_NOEXCEPT
{
    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));

    std::unique_ptr<char> readBuffer(new char[bufferSize]);
    std::unique_ptr<char> writeBuffer(new char[bufferSize]);
    nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), bufferSize);

    NNT_ASSERT_RESULT_SUCCESS(pFile->Write(size, writeBuffer.get(), bufferSize, nn::fs::WriteOption()));

    size_t readSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->Read(&readSize, size, readBuffer.get(), bufferSize, nn::fs::ReadOption()));
    ASSERT_EQ(bufferSize, readSize);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), bufferSize);

    int64_t appendedSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&appendedSize));
    ASSERT_GE(size + static_cast<int64_t>(bufferSize), appendedSize);
}

// 読み書き領域の境界チェック
void TestBoundaryOffsetSize(IStorage* pStorage, size_t accessSizeMin)
{
    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&size));

    size_t bufferSize = static_cast<size_t>(RoundUp(32, accessSizeMin));
    std::unique_ptr<char[]> buffer(new char[bufferSize]);

    NNT_EXPECT_RESULT_SUCCESS(pStorage->Write(0, buffer.get(), accessSizeMin));
    NNT_EXPECT_RESULT_SUCCESS(pStorage->Read (0, buffer.get(), accessSizeMin));
    NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange, pStorage->Write(-static_cast<int>(accessSizeMin), buffer.get(), accessSizeMin));
    NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange, pStorage->Read (-static_cast<int>(accessSizeMin), buffer.get(), accessSizeMin));

    NNT_EXPECT_RESULT_SUCCESS(pStorage->Write(size - accessSizeMin, buffer.get(), accessSizeMin));
    NNT_EXPECT_RESULT_SUCCESS(pStorage->Read (size - accessSizeMin, buffer.get(), accessSizeMin));
    NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange, pStorage->Write(size, buffer.get(), accessSizeMin));
    NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange, pStorage->Read (size, buffer.get(), accessSizeMin));
}

void TestBoundaryOffsetSize(IStorage* pStorage)
{
    TestBoundaryOffsetSize(pStorage, 1);
}

void TestGetSize(IStorage* pStorage, int64_t expectedSize)
{
    int64_t size;
    NNT_EXPECT_RESULT_SUCCESS(pStorage->GetSize(&size));
    EXPECT_EQ(expectedSize, size);
}

// 0x00000000, 0x00000001, ... という内容であることをチェック
void TestStorageContent(IStorage* pStorage, size_t bufferSize)
{
    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&size));

    std::unique_ptr<char[]> actualBuffer(new char[bufferSize]);
    std::unique_ptr<char[]> expectedBuffer(new char[bufferSize]);

    for(int64_t i = 0; i < size; i += bufferSize)
    {
        FillBufferWith32BitCount(expectedBuffer.get(), bufferSize, i);

        size_t readSize = std::min<size_t>(static_cast<size_t>(size - i), bufferSize);

        NNT_EXPECT_RESULT_SUCCESS(pStorage->Read(i, actualBuffer.get(), readSize));

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(expectedBuffer.get(), actualBuffer.get(), readSize);
    }

}


// 領域の重複がない範囲での、マルチスレッドからの read/write
void TestConcurrentWriteRead(IStorage* pStorage)
{
    const int threadCount = 4;

    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&size));

    std::thread threads[threadCount];

    for(int threadIdx = 0; threadIdx < threadCount; ++threadIdx)
    {
        int sizePerThread = int(size / threadCount);
        int offset = sizePerThread * threadIdx;

        threads[threadIdx] = std::thread([threadIdx, offset, sizePerThread, pStorage]
        {
            const int loopCount = 20;
            std::unique_ptr<char[]> readBuffer(new char[sizePerThread]);
            std::unique_ptr<char[]> writeBuffer(new char[sizePerThread]);

            for(int i = 0; i < loopCount; i++)
            {
                //FillBufferWithRandomValue(writeBuffer.get(), sizePerThread);
                memset(writeBuffer.get(), i + threadIdx, sizePerThread);
                pStorage->Write(offset, writeBuffer.get(), sizePerThread);

                memset(readBuffer.get(), 0x00, sizePerThread);
                pStorage->Read(offset, readBuffer.get(), sizePerThread);

                // TORIAEZU: gtest がスレッドアンセーフなので暫定回避
                if( std::memcmp(writeBuffer.get(), readBuffer.get(), sizePerThread) != 0 )
                {
                    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), sizePerThread);
                }
            }
        });
    }

    for(int threadIdx = 0; threadIdx < threadCount; ++threadIdx)
    {
        threads[threadIdx].join();
    }

}


// pStorage 内のランダムな範囲の read を testCount 回行う
// testCount < 0 の場合無限にテスト
void TestRandomRangeRead(IStorage* pStorage, int testCount)
{
    int64_t storageSize64;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&storageSize64));

    int storageSize = static_cast<int>(storageSize64);

    std::unique_ptr<char[]> readBuffer(new char[storageSize]);
    std::unique_ptr<char[]> expectedBuffer(new char[storageSize]);

    NN_ASSERT_NOT_NULL(readBuffer.get());
    NN_ASSERT_NOT_NULL(expectedBuffer.get());


    FillBufferWith32BitCount(expectedBuffer.get(), storageSize, 0);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    for(int i = 0; testCount < 0 || i < testCount; ++i)
    {
        auto offset = std::uniform_int_distribution<>(0, storageSize - 1)(mt);
        auto size   = std::uniform_int_distribution<>(0, storageSize - offset)(mt);
        //NN_LOG("%08x %08x\n", offset, size);

        nnt::fs::util::InvalidateVariable(readBuffer.get(), storageSize);

        NNT_EXPECT_RESULT_SUCCESS(pStorage->Read(offset, readBuffer.get() + offset, size));

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(expectedBuffer.get() + offset, readBuffer.get() + offset, size);
    }
}

// サイズを 1 バイトずつ増やして半端なサイズを書き込み・読み込み
void TestFractionWriteRead(
    IStorage* pStorage, int64_t offset, size_t sizeBegin, size_t sizeEnd) NN_NOEXCEPT
{
    int64_t storageSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&storageSize));

    std::unique_ptr<char[]> readBuffer(new char[sizeEnd]);
    std::unique_ptr<char[]> expectedBuffer(new char[sizeEnd]);

    NN_ASSERT_NOT_NULL(readBuffer.get());
    NN_ASSERT_NOT_NULL(expectedBuffer.get());

    FillBufferWith32BitCount(expectedBuffer.get(), sizeEnd, offset);

    for( auto size = sizeBegin; size <= sizeEnd; ++size )
    {
        nnt::fs::util::InvalidateVariable(readBuffer.get(), static_cast<int>(size));

        NNT_EXPECT_RESULT_SUCCESS(pStorage->Write(offset, expectedBuffer.get(), size));
        NNT_EXPECT_RESULT_SUCCESS(pStorage->Read(offset, readBuffer.get(), size));

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(expectedBuffer.get(), readBuffer.get(), size);
    }
}

class MemoryStorageTest : public ::testing::Test
{
protected:
    static const int StorageSize = 1024 * 1024;
    std::unique_ptr<Bit8> storageBuffer;
    std::unique_ptr<MemoryStorage> pStorage;

    virtual void SetUp()
    {
        storageBuffer.reset(new Bit8[StorageSize]);
        FillBufferWith32BitCount(storageBuffer.get(), StorageSize, 0);
        pStorage.reset(new MemoryStorage(storageBuffer.get(), StorageSize));
    }
};

TEST_F(MemoryStorageTest, GetSize)
{
    TestGetSize(pStorage.get(), StorageSize);
}
TEST_F(MemoryStorageTest, StorageContent)
{
    TestStorageContent(pStorage.get());
}
TEST_F(MemoryStorageTest, BoundaryOffsetSize)
{
    TestBoundaryOffsetSize(pStorage.get());
}
TEST_F(MemoryStorageTest, WriteRead)
{
    TestWriteReadStorage(pStorage.get());
}
TEST_F(MemoryStorageTest, ConcurrentWriteRead)
{
    TestConcurrentWriteRead(pStorage.get());
}



class SubStorageTest : public ::testing::Test
{
protected:
    static const int StorageSize = 1024 * 1024;
    static const int SubStorageSize = 512 * 1024;
    static const int SubStorageOffset = 256 * 1024;

    std::unique_ptr<Bit8> storageBuffer;
    std::unique_ptr<MemoryStorage> pBaseStorage;
    std::unique_ptr<SubStorage>    pStorage;

    virtual void SetUp()
    {
        storageBuffer.reset(new Bit8[StorageSize]);
        memset(storageBuffer.get(), 0xAB, StorageSize);
        FillBufferWith32BitCount(storageBuffer.get() + SubStorageOffset, SubStorageSize, 0);

        pBaseStorage.reset(new MemoryStorage(storageBuffer.get(), StorageSize));
        pStorage.reset(new SubStorage(pBaseStorage.get(), SubStorageOffset, SubStorageSize));
    }
};

TEST_F(SubStorageTest, GetSize)
{
    TestGetSize(pStorage.get(), SubStorageSize);
}
TEST_F(SubStorageTest, StorageContent)
{
    TestStorageContent(pStorage.get());
}
TEST_F(SubStorageTest, BoundaryOffsetSize)
{
    TestBoundaryOffsetSize(pStorage.get());
}
TEST_F(SubStorageTest, WriteRead)
{
    TestWriteReadStorage(pStorage.get());
}
TEST_F(SubStorageTest, ConcurrentWriteRead)
{
    TestConcurrentWriteRead(pStorage.get());
}

TEST(SubStorageLargeTest, WriteRead)
{
    static const int SubStorageOffset = 256 * 1024;
    static const size_t BufferSize = 1024;
    static const int64_t StorageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + BufferSize;
    nnt::fs::util::VirtualMemoryStorage baseStorage(StorageSize + SubStorageOffset);
    nn::fs::SubStorage subStorage(&baseStorage, SubStorageOffset, StorageSize);
    TestWriteReadStorageWithLargeOffset(&subStorage, BufferSize);
}


class FileHandleStorageTest : public ::testing::Test
{
protected:
    static const int StorageSize = 128 * 1024;
    FileHandle m_Handle;
    String m_FilePath;

    std::unique_ptr<FileHandleStorage> pStorage;

    FileHandleStorageTest()
    {
    }

    virtual void SetUp()
    {
        m_Handle.handle = nullptr;

        m_FilePath = g_HostDirectory.GetPath() + "/FileHandleStorageTest.bin";
        NNT_ASSERT_RESULT_SUCCESS(CreateFile(m_FilePath.c_str(), StorageSize));
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&m_Handle, m_FilePath.c_str(), static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));

        std::unique_ptr<char> buffer(new char[StorageSize]);
        ASSERT_NE(nullptr, buffer);

        FillBufferWith32BitCount(buffer.get(), StorageSize, 0);
        NNT_ASSERT_RESULT_SUCCESS(WriteFile(m_Handle, 0, buffer.get(), StorageSize, WriteOption()));
        NNT_ASSERT_RESULT_SUCCESS(FlushFile(m_Handle));

        pStorage.reset(new FileHandleStorage(m_Handle));
    }

    virtual void TearDown()
    {
        if (m_Handle.handle != nullptr)
        {
            NNT_EXPECT_RESULT_SUCCESS(FlushFile(m_Handle));
            CloseFile(m_Handle);
            NNT_ASSERT_RESULT_SUCCESS(DeleteFile(m_FilePath.c_str()));
        }
    }
};

TEST_F(FileHandleStorageTest, GetSize)
{
    TestGetSize(pStorage.get(), StorageSize);
}
TEST_F(FileHandleStorageTest, StorageContent)
{
    TestStorageContent(pStorage.get());
}
TEST_F(FileHandleStorageTest, BoundaryOffsetSize)
{
    TestBoundaryOffsetSize(pStorage.get());
}
TEST_F(FileHandleStorageTest, WriteRead)
{
    TestWriteReadStorage(pStorage.get());
}
TEST_F(FileHandleStorageTest, ConcurrentWriteRead)
{
    TestConcurrentWriteRead(pStorage.get());
}



class FileStorageTest : public ::testing::Test
{
protected:
    static const int StorageSize = 128 * 1024;

    std::unique_ptr<HostFileSystem> m_pHostFs;
    std::unique_ptr<fsa::IFile> m_pBaseFile;
    std::unique_ptr<FileStorage> pStorage;
    String m_FilePath;

    FileStorageTest()
    {
    }

    virtual void SetUp()
    {
        m_pHostFs.reset(new HostFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(m_pHostFs->Initialize(""));

        m_FilePath = g_HostDirectory.GetPath() + "/FileStorageTest.bin";
        NNT_ASSERT_RESULT_SUCCESS(m_pHostFs->CreateFile(m_FilePath.c_str(), StorageSize, 0));
        NNT_ASSERT_RESULT_SUCCESS(m_pHostFs->OpenFile(&m_pBaseFile, m_FilePath.c_str(), static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));

        std::unique_ptr<char> buffer(new char[StorageSize]);
        ASSERT_NE(nullptr, buffer);

        FillBufferWith32BitCount(buffer.get(), StorageSize, 0);
        NNT_ASSERT_RESULT_SUCCESS(m_pBaseFile->Write(0, buffer.get(), StorageSize, WriteOption()));
        NNT_ASSERT_RESULT_SUCCESS(m_pBaseFile->Flush());

        pStorage.reset(new FileStorage(m_pBaseFile.get()));
    }

    virtual void TearDown()
    {
        if( m_pBaseFile != nullptr )
        {
            m_pBaseFile->Flush();
            m_pBaseFile.reset(nullptr); // close file
            NNT_ASSERT_RESULT_SUCCESS(DeleteFile(m_FilePath.c_str()));
        }
    }
};

TEST_F(FileStorageTest, GetSize)
{
    TestGetSize(pStorage.get(), StorageSize);
}
TEST_F(FileStorageTest, StorageContent)
{
    TestStorageContent(pStorage.get());
}
TEST_F(FileStorageTest, BoundaryOffsetSize)
{
    TestBoundaryOffsetSize(pStorage.get());
}
TEST_F(FileStorageTest, WriteRead)
{
    TestWriteReadStorage(pStorage.get());
}
TEST_F(FileStorageTest, ConcurrentWriteRead)
{
    TestConcurrentWriteRead(pStorage.get());
}

TEST(FileStorageLargeTest, WriteRead)
{
    static const size_t BufferSize = 1024;
    static const int64_t StorageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + BufferSize;
    nnt::fs::util::VirtualMemoryStorage baseStorage(StorageSize);
    nn::fssystem::StorageFile storageFile(&baseStorage, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    nn::fs::FileStorage fileStorage(&storageFile);

    TestWriteReadStorageWithLargeOffset(&fileStorage, BufferSize);
}


namespace {

void* Allocate(size_t size)
{
    return malloc(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    free(p);
}

}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    g_HostDirectory.Create();
    MountHostRoot();

    auto ret = RUN_ALL_TESTS();

    UnmountHostRoot();
    g_HostDirectory.Delete();

    nnt::Exit(ret);
}
