﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_BitUtil.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_SubStorage.h>

#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include "testFs_Unit_StorageLayerTestCase.h"


using namespace nn::fssystem;
using namespace nnt::fs::util;
using nn::Result;
using nn::Bit8;

namespace {

template<typename T>
bool IsAligned(T* target, size_t alignment)
{
    return nn::util::is_aligned(reinterpret_cast<uintptr_t>(target), alignment);
}

bool IsAligned(int64_t target, size_t alignment)
{
    return nn::util::is_aligned(target, alignment);
}

template<typename T>
T* RoundUp(T* target, size_t alignment)
{
    NN_SDK_REQUIRES((alignment & (alignment - 1)) == 0);
    return reinterpret_cast<T*>((reinterpret_cast<uintptr_t>(target) + (alignment - 1)) & ~(static_cast<uintptr_t>(alignment) - 1));
}


// 指定したアライン以外のアクセスをはじくレイヤ
template<int DataAlign, int BufferAlign>
class AlignmentConstraintStorage : public nn::fs::IStorage
{
public:
    explicit AlignmentConstraintStorage(IStorage* pBaseStorage)
        : m_pBaseStorage(pBaseStorage)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ASSERT(IsAligned(offset, DataAlign));
        NN_ASSERT(IsAligned(size,   DataAlign));
        NN_ASSERT(IsAligned(buffer, BufferAlign));
        return m_pBaseStorage->Read(offset, buffer, size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ASSERT(IsAligned(offset, DataAlign));
        NN_ASSERT(IsAligned(size,   DataAlign));
        NN_ASSERT(IsAligned(buffer, BufferAlign));
        return m_pBaseStorage->Write(offset, buffer, size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->Flush();
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->GetSize(outValue);
    }

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->OperateRange(
            outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
    }

private:
    IStorage* m_pBaseStorage;
};


// AlignmentConstraintStorage 上に AlignmentMatchingStorage をかぶせるヘルパクラス
template <int DataAlign, int BufferAlign>
class AlignmentMatchingOnConstraintStorage
{
    std::unique_ptr<nnt::fs::util::AccessCountedMemoryStorage> pMemoryStorage;
    std::unique_ptr<nn::fs::IStorage> pAlignmentConstraintStorage;
    std::unique_ptr<nn::fs::IStorage> pStorage;

public:
    static const int DataAlign = DataAlign;
    static const int StorageSize = 16 * 1024 * 1024;

    AlignmentMatchingOnConstraintStorage()
    {
        pMemoryStorage.reset(new nnt::fs::util::AccessCountedMemoryStorage);
        pMemoryStorage->Initialize(StorageSize);
        FillBufferWith32BitCount(pMemoryStorage->GetBuffer(), StorageSize, 0);
        pAlignmentConstraintStorage.reset(new AlignmentConstraintStorage<DataAlign, BufferAlign>(pMemoryStorage.get()));
        pStorage.reset(new AlignmentMatchingStorage<DataAlign, BufferAlign>(pAlignmentConstraintStorage.get()));
    }

    nn::fs::IStorage* GetStorage()
    {
        return pStorage.get();
    }

    nnt::fs::util::AccessCountedMemoryStorage* GetBaseStorage() NN_NOEXCEPT
    {
        return pMemoryStorage.get();
    }
};

// AlignmentConstraintStorage 上に AlignmentMatchingStoragePolledBuffer をかぶせるヘルパクラス
template <int DataAlignment, int BufferAlignment>
class AlignmentMatchingOnConstraintStoragePooledBuffer
{
public:
    static const int DataAlign = DataAlignment;
    static const int StorageSize = 16 * 1024 * 1024;

public:
    AlignmentMatchingOnConstraintStoragePooledBuffer() NN_NOEXCEPT
    {
        m_pMemoryStorage.reset(new nnt::fs::util::AccessCountedMemoryStorage);
        m_pMemoryStorage->Initialize(StorageSize);
        FillBufferWith32BitCount(m_pMemoryStorage->GetBuffer(), StorageSize, 0);
        m_pAlignmentConstraintStorage.reset(
            new AlignmentConstraintStorage<DataAlignment, BufferAlignment>(
                m_pMemoryStorage.get()));
        m_pStorage.reset(
            new AlignmentMatchingStoragePooledBuffer<BufferAlignment>(
                m_pAlignmentConstraintStorage.get(),
                DataAlignment));
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return m_pStorage.get();
    }

    nnt::fs::util::AccessCountedMemoryStorage* GetBaseStorage() NN_NOEXCEPT
    {
        return m_pMemoryStorage.get();
    }

private:
    std::unique_ptr<nnt::fs::util::AccessCountedMemoryStorage> m_pMemoryStorage;
    std::unique_ptr<nn::fs::IStorage> m_pAlignmentConstraintStorage;
    std::unique_ptr<nn::fs::IStorage> m_pStorage;
};

// AlignmentConstraintStorage 上に AlignmentMatchingStorageInBulkRead をかぶせるヘルパクラス
template <int DataAlignment, int BufferAlignment>
class AlignmentMatchingOnConstraintStorageInBulkRead
{
public:
    static const int DataAlign = DataAlignment;
    static const int StorageSize = 16 * 1024 * 1024;

public:
    AlignmentMatchingOnConstraintStorageInBulkRead() NN_NOEXCEPT
    {
        m_pMemoryStorage.reset(new nnt::fs::util::AccessCountedMemoryStorage);
        m_pMemoryStorage->Initialize(StorageSize);
        FillBufferWith32BitCount(m_pMemoryStorage->GetBuffer(), StorageSize, 0);
        m_pAlignmentConstraintStorage.reset(
            new AlignmentConstraintStorage<DataAlignment, BufferAlignment>(
                m_pMemoryStorage.get()));
        m_pStorage.reset(
            new AlignmentMatchingStorageInBulkRead<BufferAlignment>(
                m_pAlignmentConstraintStorage.get(),
                DataAlignment));
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return m_pStorage.get();
    }

    nnt::fs::util::AccessCountedMemoryStorage* GetBaseStorage() NN_NOEXCEPT
    {
        return m_pMemoryStorage.get();
    }

private:
    std::unique_ptr<nnt::fs::util::AccessCountedMemoryStorage> m_pMemoryStorage;
    std::unique_ptr<nn::fs::IStorage> m_pAlignmentConstraintStorage;
    std::unique_ptr<nn::fs::IStorage> m_pStorage;
};


// offset に下駄をはかせるレイヤ
class JackStorage : public nn::fs::IStorage
{
public:
    JackStorage(IStorage* pBaseStorage, int64_t size)
        : m_pBaseStorage(pBaseStorage)
        , m_JackSize(size)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(offset >= m_JackSize);
        return m_pBaseStorage->Read(offset - m_JackSize, buffer, size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(offset >= m_JackSize);
        return m_pBaseStorage->Write(offset - m_JackSize, buffer, size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        int64_t size;
        NN_RESULT_DO(m_pBaseStorage->GetSize(&size));
        *outValue = size + m_JackSize;

        NN_RESULT_SUCCESS;
    }

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->OperateRange(
            outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
    }

private:
    IStorage* m_pBaseStorage;
    int64_t   m_JackSize;
};

// 4GB 近辺のオフセットのテストのためのヘルパクラス
template <int DataAlign, int BufferAlign>
class JackedAlignmentMatchingOnConstraintStorage
{
    std::unique_ptr<nnt::fs::util::AccessCountedMemoryStorage> pMemoryStorage;
    std::unique_ptr<nn::fs::IStorage> pJackStorage;
    std::unique_ptr<nn::fs::IStorage> pAlignmentConstraintStorage;
    std::unique_ptr<nn::fs::IStorage> pAlignmentMatchingStorage;
    std::unique_ptr<nn::fs::IStorage> pStorage; // subStorage

public:
    static const int DataAlign = DataAlign;
    static const int StorageSize = 16 * 1024 * 1024;

    static const long long int jackSize = 0xFFFFFE00; // 4GB - 512B

    JackedAlignmentMatchingOnConstraintStorage()
    {
        pMemoryStorage.reset(new nnt::fs::util::AccessCountedMemoryStorage);
        pMemoryStorage->Initialize(StorageSize);
        FillBufferWith32BitCount(pMemoryStorage->GetBuffer(), StorageSize, 0);
        pJackStorage.reset(new JackStorage(pMemoryStorage.get(), jackSize));
        pAlignmentConstraintStorage.reset(new AlignmentConstraintStorage<DataAlign, BufferAlign>(pJackStorage.get()));
        pAlignmentMatchingStorage.reset(new AlignmentMatchingStorage<DataAlign, BufferAlign>(pAlignmentConstraintStorage.get()));
        pStorage.reset(new nn::fs::SubStorage(pAlignmentMatchingStorage.get(), jackSize, StorageSize));
    }

    nn::fs::IStorage* GetStorage()
    {
        return pStorage.get();
    }

    nnt::fs::util::AccessCountedMemoryStorage* GetBaseStorage() NN_NOEXCEPT
    {
        return pMemoryStorage.get();
    }
};





const size_t Values[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 15, 16, 17,
    510,  511,  512,  513,  514,
    1022, 1023, 1024, 1025, 1026,
    2046, 2047, 2048, 2049, 2050
};

const int BufferSize = 5 * 1024;
const int BufferAlignSize = 1024;

} // namespace


template <typename T>
void TestUnalignedRead(T* pStorage)
{
    std::unique_ptr<char[]> buffer(new char[BufferSize + BufferAlignSize]);
    char* pAlignedBuffer = RoundUp(buffer.get(), BufferAlignSize);
    std::unique_ptr<char[]> verifyBuffer1(new char[BufferSize]);
    std::unique_ptr<char[]> verifyBuffer2(new char[BufferSize]);

    FillBufferWith32BitCount(verifyBuffer1.get(), BufferSize, 0);
    memset(verifyBuffer2.get(), 0xAA, BufferSize);

    // 各境界値な領域で Read
    for(auto offset : Values)
    {
        for(auto size : Values)
        {
            for(auto bufferAlign : Values)
            {
                if( bufferAlign + size > BufferSize )
                {
                    continue;
                }

                NNT_FS_SCOPED_TRACE_SAFE("offset: %d, size: %d, bufferAlign: %d", offset, size, bufferAlign);

                memset(pAlignedBuffer, 0xAA, BufferSize);
                NNT_EXPECT_RESULT_SUCCESS(pStorage->Read(offset, pAlignedBuffer + bufferAlign, size));

                NNT_FS_UTIL_EXPECT_MEMCMPEQ(verifyBuffer2.get(),                      pAlignedBuffer,                      bufferAlign);
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(verifyBuffer1.get() + offset,             pAlignedBuffer + bufferAlign,        size);
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(verifyBuffer2.get() + bufferAlign + size, pAlignedBuffer + bufferAlign + size, BufferSize - (bufferAlign + size));

            }
        }
    }
}

template <typename T>
void TestUnalignedWrite(T* pStorage)
{
    std::unique_ptr<char[]> buffer(new char[BufferSize + BufferAlignSize]);
    char* pAlignedBuffer = RoundUp(buffer.get(), BufferAlignSize);
    std::unique_ptr<char[]> verifyBuffer1(new char[BufferSize]);
    std::unique_ptr<char[]> verifyBuffer2(new char[BufferSize]);
    std::unique_ptr<char[]> tmpBuffer(new char[BufferSize]);

    FillBufferWith32BitCount(pAlignedBuffer, BufferSize, 0x12345678);
    FillBufferWith32BitCount(verifyBuffer1.get(), BufferSize, 0x12345678);
    FillBufferWith32BitCount(verifyBuffer2.get(), BufferSize, 0);

    // 各境界値な領域で Write
    for(auto offset : Values)
    {
        for(auto size : Values)
        {
            for(auto bufferAlign : Values)
            {
                if( bufferAlign + size > BufferSize )
                {
                    continue;
                }

                NNT_FS_SCOPED_TRACE_SAFE("offset: %d, size: %d, bufferAlign: %d", offset, size, bufferAlign);

                // 状態を復元
                NNT_EXPECT_RESULT_SUCCESS(pStorage->Write(0, verifyBuffer2.get(), BufferSize));

                // 書き込み
                NNT_EXPECT_RESULT_SUCCESS(pStorage->Write(offset, pAlignedBuffer + bufferAlign, size));

                // 内容を確認
                NNT_EXPECT_RESULT_SUCCESS(pStorage->Read(0, tmpBuffer.get(), BufferSize));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(verifyBuffer2.get(),                 tmpBuffer.get(),                 offset);
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(verifyBuffer1.get() + bufferAlign,   tmpBuffer.get() + offset,        size);
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(verifyBuffer2.get() + offset + size, tmpBuffer.get() + offset + size, BufferSize - (offset + size));

            }
        }
    }
}


template <typename T>
class VariableAlignAlignmentMatchingStorage : public ::testing::Test
{
};


TYPED_TEST_CASE_P(VariableAlignAlignmentMatchingStorage);


TYPED_TEST_P(VariableAlignAlignmentMatchingStorage, UnalignedRead)
{
    TypeParam storage;
    TestUnalignedRead(storage.GetStorage());
}

TYPED_TEST_P(VariableAlignAlignmentMatchingStorage, UnalignedWrite)
{
    TypeParam storage;
    TestUnalignedWrite(storage.GetStorage());
}


TYPED_TEST_P(VariableAlignAlignmentMatchingStorage, GetSize)
{
    TypeParam storage;
    TestGetSize(storage.GetStorage(), storage.StorageSize);
}
TYPED_TEST_P(VariableAlignAlignmentMatchingStorage, StorageContent)
{
    TypeParam storage;
    TestStorageContent(storage.GetStorage());
}
TYPED_TEST_P(VariableAlignAlignmentMatchingStorage, BoundaryOffsetSize)
{
    TypeParam storage;
    TestBoundaryOffsetSize(storage.GetStorage(), TypeParam::DataAlign);
}
TYPED_TEST_P(VariableAlignAlignmentMatchingStorage, WriteRead)
{
    TypeParam storage;
    TestWriteReadStorage(storage.GetStorage());
}
TYPED_TEST_P(VariableAlignAlignmentMatchingStorage, ConcurrentWriteRead)
{
    TypeParam storage;
    TestConcurrentWriteRead(storage.GetStorage());
}

TYPED_TEST_P(VariableAlignAlignmentMatchingStorage, Invalidate)
{
    TypeParam storage;

    storage.GetBaseStorage()->ResetAccessCounter();
    NNT_EXPECT_RESULT_SUCCESS(storage.GetStorage()->OperateRange(
        nn::fs::OperationId::Invalidate,
        0,
        1));
    EXPECT_GT(storage.GetBaseStorage()->GetInvalidateTimes(), 0);
}


REGISTER_TYPED_TEST_CASE_P(VariableAlignAlignmentMatchingStorage,
                           UnalignedRead, UnalignedWrite,
                           GetSize, StorageContent, BoundaryOffsetSize, WriteRead, ConcurrentWriteRead, Invalidate
                           );

typedef ::testing::Types<
    JackedAlignmentMatchingOnConstraintStorage<4, 512>,

    AlignmentMatchingOnConstraintStorage<1,   1>,
    AlignmentMatchingOnConstraintStorage<4,   512>,
    AlignmentMatchingOnConstraintStorage<512, 512>,

    AlignmentMatchingOnConstraintStoragePooledBuffer<4,   1>,
    AlignmentMatchingOnConstraintStoragePooledBuffer<512, 1>,

    AlignmentMatchingOnConstraintStorageInBulkRead<4,   1>,
    AlignmentMatchingOnConstraintStorageInBulkRead<512, 1>
> BasicTypes;

typedef ::testing::Types<
    JackedAlignmentMatchingOnConstraintStorage<1, 512>,
    JackedAlignmentMatchingOnConstraintStorage<512, 512>,

    //AlignmentMatchingOnConstraintStorage<1,   1>,
    AlignmentMatchingOnConstraintStorage<1,   4>,
    AlignmentMatchingOnConstraintStorage<1,   8>,
    AlignmentMatchingOnConstraintStorage<1,   512>,

    AlignmentMatchingOnConstraintStorage<4,   1>,
    AlignmentMatchingOnConstraintStorage<4,   4>,
    AlignmentMatchingOnConstraintStorage<4,   8>,
    //AlignmentMatchingOnConstraintStorage<4,   512>,

    AlignmentMatchingOnConstraintStorage<8,   1>,
    AlignmentMatchingOnConstraintStorage<8,   4>,
    AlignmentMatchingOnConstraintStorage<8,   8>,
    AlignmentMatchingOnConstraintStorage<8,   512>,

    AlignmentMatchingOnConstraintStorage<512, 1>,
    AlignmentMatchingOnConstraintStorage<512, 4>,
    AlignmentMatchingOnConstraintStorage<512, 8>
    //AlignmentMatchingOnConstraintStorage<512, 512>
> FullTypes;


INSTANTIATE_TYPED_TEST_CASE_P(Basic, VariableAlignAlignmentMatchingStorage, BasicTypes);
//INSTANTIATE_TYPED_TEST_CASE_P(Full, VariableAlignAlignmentMatchingStorage, FullTypes);



