﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <nn/nn_Macro.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/sdmmc/sdmmc_Common.h>
#include "detail/fssrv_SdmmcStorageService.cpp"
#include "detail/fssrv_SdmmcDummy.cpp"

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

NN_DEFINE_ERROR_RANGE_RESULT(ResultAlignmentError, 500, 1, 2);

// 読み書き時にバッファがアラインされているか確かめるストレージ
class AlignmentCheckStorage : public nnt::fs::util::SafeMemoryStorage
{
public:
    explicit AlignmentCheckStorage(size_t size) NN_NOEXCEPT
        : nnt::fs::util::SafeMemoryStorage(size)
    {
    }

    virtual nn::Result Read(
                int64_t offset,
                void* buffer,
                size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( reinterpret_cast<const uintptr_t>(buffer) % nn::fssrv::detail::DataCacheLineSize != 0 )
        {
            NN_RESULT_THROW(ResultAlignmentError());
        }
        NN_RESULT_DO(nnt::fs::util::SafeMemoryStorage::Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result Write(
                int64_t offset,
                const void* buffer,
                size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( reinterpret_cast<const uintptr_t>(buffer) % nn::fssrv::detail::DataCacheLineSize != 0 )
        {
            NN_RESULT_THROW(ResultAlignmentError());
        }
        NN_RESULT_DO(nnt::fs::util::SafeMemoryStorage::Write(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }
};


class SdmmcStorageTest : public nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture
{
public:
    static const size_t DefaultDataSize = 128 * 1024 * 1024;

public:
    SdmmcStorageTest() NN_NOEXCEPT
      : m_Storage(nn::sdmmc::Port_Mmc0),
        m_DefaultDataBuffer(new char[DefaultDataSize]),
        m_BaseStorage(DefaultDataSize),
        m_Mt(nnt::fs::util::GetRandomSeed())
    {
        nn::sdmmc::SetStorage(&m_BaseStorage);

        std::iota(m_DefaultDataBuffer.get(), m_DefaultDataBuffer.get() + DefaultDataSize, '\x0');
        m_Storage.Write(0, m_DefaultDataBuffer.get(), DefaultDataSize);
    }

    nn::Result ExecuteTest(char* readBuffer, char* writeBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            // 書き換えたものを元に戻す
            NNT_EXPECT_RESULT_SUCCESS(
                m_Storage.Write(0, m_DefaultDataBuffer.get(), DefaultDataSize)
            );
        };

        // SdmmcStorage::Read
        int64_t readOffset = CreateOffset(bufferSize);
        NN_RESULT_DO(m_Storage.Read(readOffset, readBuffer, bufferSize));
        EXPECT_EQ(0, std::memcmp(m_DefaultDataBuffer.get() + readOffset, readBuffer, bufferSize));

        // SdmmcStorage::Write
        int64_t writeOffset = CreateOffset(bufferSize);
        nnt::fs::util::FillBufferWithRandomValue(writeBuffer, bufferSize);
        NN_RESULT_DO(m_Storage.Write(writeOffset, writeBuffer, bufferSize));
        NN_RESULT_DO(m_Storage.Read(writeOffset, readBuffer, bufferSize));

        NN_RESULT_SUCCESS;
    }

protected:
    size_t GetRandomSectorSize(size_t min, size_t max) NN_NOEXCEPT
    {
        size_t minSectorCount = nn::util::DivideUp(min, nn::sdmmc::SectorSize);
        size_t maxSectorCount = max / nn::sdmmc::SectorSize;
        EXPECT_LE(minSectorCount, maxSectorCount);
        return static_cast<size_t>(
                   GetRandom(minSectorCount, maxSectorCount) * nn::sdmmc::SectorSize
               );
    }

    int64_t GetRandom(size_t min, size_t max) NN_NOEXCEPT
    {
        EXPECT_LE(min, max);
        return std::uniform_int_distribution<int64_t>(min, max)(m_Mt);
    }

private:
    int64_t CreateOffset(size_t bufferSize) NN_NOEXCEPT
    {
        return GetRandomSectorSize(0, DefaultDataSize - bufferSize);
    }

private:
    nn::fssrv::detail::SdmmcStorage m_Storage;
    std::unique_ptr<char[]> m_DefaultDataBuffer;
    AlignmentCheckStorage m_BaseStorage;
    std::mt19937 m_Mt;
};

// 通常の読み書きテスト
TEST_F(SdmmcStorageTest, ReadWrite)
{
    const int TestLoopCount = 8;
    const size_t MinBufferSize = 16 * 1024;
    const size_t MaxBufferSize = 64 * 1024 * 1024;
    std::unique_ptr<char[]> readBuffer(new char[MaxBufferSize]);
    std::unique_ptr<char[]> writeBuffer(new char[MaxBufferSize]);

    for( int i = 0; i < TestLoopCount; ++i )
    {
        size_t readWiteBufferSize = GetRandomSectorSize(MinBufferSize, MaxBufferSize);
        NNT_EXPECT_RESULT_SUCCESS(
            ExecuteTest(readBuffer.get(), writeBuffer.get(), readWiteBufferSize)
        );
    }
}

// アラインされていないバッファを渡したときの挙動をテスト
TEST_F(SdmmcStorageTest, ReadWriteWithNotAlignBuffer)
{
    const int TestLoopCount = 8;
    const int64_t MaxBufferAlignShiftSize = 16;
    const size_t MinBufferSize = 16 * 1024;
    const size_t MaxBufferSize = 64 * 1024 * 1024;

    std::unique_ptr<char[]> readBuffer(
                                new char[MaxBufferSize + MaxBufferAlignShiftSize]);
    std::unique_ptr<char[]> writeBuffer(
                                new char[MaxBufferSize + MaxBufferAlignShiftSize]);

    for( int i = 0; i < TestLoopCount; ++i )
    {
        size_t readWiteBufferSize = GetRandomSectorSize(MinBufferSize, MaxBufferSize);

        char* notAlignedReadBuffer = readBuffer.get() + GetRandom(1, MaxBufferAlignShiftSize);
        char* notAlignedWriteBuffer = writeBuffer.get() + GetRandom(1, MaxBufferAlignShiftSize);
        NNT_EXPECT_RESULT_SUCCESS(
            ExecuteTest(notAlignedReadBuffer, notAlignedWriteBuffer, readWiteBufferSize)
        );
    }
}

// PooledBuffer から取得したバッファを渡したときの挙動をテスト
TEST_F(SdmmcStorageTest, ReadWriteWithPooledBuffer)
{
    const size_t PooledBufferAllocateSize
        = nn::fssystem::PooledBuffer::GetAllocatableParticularlyLargeSizeMax() / 4;
    nn::fssystem::PooledBuffer pooledReadBuffer;
    pooledReadBuffer.AllocateParticularlyLarge(PooledBufferAllocateSize, PooledBufferAllocateSize);
    nn::fssystem::PooledBuffer pooledWriteBuffer;
    pooledWriteBuffer.AllocateParticularlyLarge(PooledBufferAllocateSize, PooledBufferAllocateSize);
    const int64_t MaxPooledBufferShiftSectorCount = 16;

    size_t pooledBufferAllocatedSize
               = std::min(pooledReadBuffer.GetSize(), pooledWriteBuffer.GetSize());
    ASSERT_LE(
        16 * 1024,
        pooledBufferAllocatedSize - MaxPooledBufferShiftSectorCount * nn::sdmmc::SectorSize
    );

    const int TestLoopCount = 32;

    for( int i = 0; i < TestLoopCount; ++i )
    {
        size_t readWiteBufferSize = GetRandomSectorSize(
                16 * 1024,
                pooledBufferAllocatedSize - MaxPooledBufferShiftSectorCount * nn::sdmmc::SectorSize
            );
        char* readBuffer = pooledReadBuffer.GetBuffer()
                         + nn::sdmmc::SectorSize * GetRandom(0, MaxPooledBufferShiftSectorCount);
        char* writeBuffer = pooledWriteBuffer.GetBuffer()
                          + nn::sdmmc::SectorSize * GetRandom(0, MaxPooledBufferShiftSectorCount);
        NNT_EXPECT_RESULT_SUCCESS(
            ExecuteTest(readBuffer, writeBuffer, readWiteBufferSize)
        );
    }
}

// PooledBuffer から取得したバッファを、アライメントをずらして渡したときの挙動をテスト
TEST_F(SdmmcStorageTest, ReadWriteWithNotAlignPooledBuffer)
{
    const size_t PooledBufferAllocateSize
        = nn::fssystem::PooledBuffer::GetAllocatableParticularlyLargeSizeMax() / 4;
    nn::fssystem::PooledBuffer pooledReadBuffer;
    pooledReadBuffer.AllocateParticularlyLarge(PooledBufferAllocateSize, PooledBufferAllocateSize);
    nn::fssystem::PooledBuffer pooledWriteBuffer;
    pooledWriteBuffer.AllocateParticularlyLarge(PooledBufferAllocateSize, PooledBufferAllocateSize);
    const int64_t MaxPooledBufferShiftSectorCount = 16;
    const int64_t MaxBufferAlignShiftSize = 16;

    size_t pooledBufferAllocatedSize
               = std::min(pooledReadBuffer.GetSize(), pooledWriteBuffer.GetSize());
    ASSERT_LE(
        16 * 1024,
        pooledBufferAllocatedSize
        - MaxPooledBufferShiftSectorCount * nn::sdmmc::SectorSize
        - MaxBufferAlignShiftSize
    );

    const int TestLoopCount = 32;

    for( int i = 0; i < TestLoopCount; ++i )
    {
        size_t readWiteBufferSize = GetRandomSectorSize(
                16 * 1024,
                pooledBufferAllocatedSize
                - MaxPooledBufferShiftSectorCount * nn::sdmmc::SectorSize
                - MaxBufferAlignShiftSize
            );
        char* readBuffer = pooledReadBuffer.GetBuffer()
                         + nn::sdmmc::SectorSize * GetRandom(0, MaxPooledBufferShiftSectorCount)
                         + GetRandom(1, MaxBufferAlignShiftSize);
        char* writeBuffer = pooledWriteBuffer.GetBuffer()
                          + nn::sdmmc::SectorSize * GetRandom(0, MaxPooledBufferShiftSectorCount)
                         + GetRandom(1, MaxBufferAlignShiftSize);
        NNT_EXPECT_RESULT_SUCCESS(
            ExecuteTest(readBuffer, writeBuffer, readWiteBufferSize)
        );
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    static const size_t BufferPoolSize = 8 * 1024 * 1024;
    static NN_ALIGNAS(4096) char s_BufferPool[BufferPoolSize];
    nn::fssystem::InitializeBufferPool(s_BufferPool, BufferPoolSize);

    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
