﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <random>
#include <nn/os/os_Thread.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_Unit_SaveDataFileSystemDriverLayerTestCase.h"

static const size_t MaxBlockSize = 32 * 1024;

//! ブロックサイズ(2のべき乗)から、その指数を取得する
inline int GetSizeExp(size_t blockSize)
{
    NN_SDK_REQUIRES(nn::util::ispow2(blockSize));
    return nn::util::cntt0(blockSize);
}

//! ランダムなブロックサイズを取得します。2のべき乗が出力されます。
inline size_t GenerateRandomBlockSize(size_t blockSizeMin, size_t blockSizeMax)
{
    static std::mt19937 s_Mt(nnt::fs::util::GetRandomSeed());
    int min = GetSizeExp(blockSizeMin);
    int max = GetSizeExp(blockSizeMax);
    int random = std::uniform_int_distribution<>(0, max - min)(s_Mt) + min;
    return static_cast<size_t>(1) << random;
}

// 4 GB を超えるオフセットでストレージにアクセスする共通テスト
void TestLargeOffsetAccess(nn::fs::IStorage* pStorage, size_t bufferSize) NN_NOEXCEPT;

// 4 GB を超えるオフセットでストレージにアクセスするテスト（読み込みテストを独自関数で行う）
void TestLargeOffsetAccess(
    nn::fs::IStorage* pStorage,
    size_t bufferSize,
    std::function<void(int64_t offset, char* readBuffer, const char* writeBuffer, size_t bufferSize)> readTestFunc) NN_NOEXCEPT;

//! 乱数生成クラス
class Random
{
public:
    Random() NN_NOEXCEPT
        : m_Mt(nnt::fs::util::GetRandomSeed())
    {
    }

    explicit Random(std::mt19937::result_type seed) NN_NOEXCEPT
        : m_Mt(seed)
    {
    }

    template< typename T >
    T Get(T max) const NN_NOEXCEPT
    {
        return std::uniform_int_distribution<T>(0, max)(m_Mt);
    }

    template< typename T >
    T Get(T min, T max) const NN_NOEXCEPT
    {
        return std::uniform_int_distribution<T>(min, max)(m_Mt);
    }

private:
    mutable std::mt19937 m_Mt;
};

class TestStorageSetup
{
public:
    //! コンストラクタです。
    TestStorageSetup() NN_NOEXCEPT
        : m_Random()
    {
    }

    //! ストレージオブジェクトを取得します。
    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        return m_pStorage;
    }

    //! ストレージオブジェクトを設定します。
    void SetStorage(nn::fs::IStorage* pStorage) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pStorage);
        m_pStorage = pStorage;
    }

    //! ストレージオブジェクトをフォーマットします。
    virtual void Format(int64_t sizeFile, int64_t offset, size_t sizeBlock) = 0;

    //! ストレージオブジェクトを初期化します。
    virtual void Initialize(int64_t sizeFile, int64_t offset, size_t sizeBlock) = 0;

    //! ストレージオブジェクトをアンマウントします。
    virtual void Unmount() = 0;

    //! ストレージオブジェクトを破棄します。
    virtual void Finalize() = 0;

    //! 読み込み専用かどうか
    virtual bool IsReadOnly() const = 0;

    //! ストレージへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const = 0;

    //! ストレージのサイズ変更が可能かどうか
    virtual bool IsEnableResize() const = 0;

    //! サイズ 0 のストレージを作成可能かどうか
    virtual bool IsAcceptsSizeZero() const = 0;

protected:
    Random m_Random;

private:
    nn::fs::IStorage* m_pStorage;
};

/**
* @brief IStorage に対する共通テストを実行します。
*
*        tests::StorageTest::RunTest<MyStorageTestSetup>();
*        のように指定することで、ストレージに対する要求仕様が
*        満たされていることを確認する共通テストが行われます。
*
*        テンプレート引数に使用するクラスには @class TestStorageSetup を
*        継承したクラスを指定してください。
*/
class StorageTest
{
private:
    //! 0 バイトのストレージに対するテストを行います。
    static void TestSizeZero(
        nn::fs::IStorage* pStorage,
        bool isAutoExtendSizeOnWrite,
        bool isReadOnly
    ) NN_NOEXCEPT;

    //! 読み込み可能なストレージに対するリード、ベリファイを行うテストです。
    static void TestReadVerify(
        nn::fs::IStorage* pStorage
    ) NN_NOEXCEPT;

    //! 読み書き可能なストレージに対する読み書き時の範囲チェックテストを行います。
    static void TestBounds(
        nn::fs::IStorage* pStorage,
        bool isEnableResize,
        bool isAutoExtendSizeOnWrite
    ) NN_NOEXCEPT;

    //! 読み書き可能なストレージに対する単純な読み書き、ベリファイテストを行います。
    static void TestReadWrite1(
        nn::fs::IStorage* pStorage,
        bool bAutoExtendSizeOnWrite
    ) NN_NOEXCEPT;

    //! 読み書き可能なストレージに対するランダム読み書き、ベリファイテストを行います。
    static void TestReadWrite2(
        nn::fs::IStorage* pStorage
    ) NN_NOEXCEPT;

    //! テストデータを書き込んでおきます。
    static void TestWriteTestData(
        int64_t* pOutSizeFile,
        nn::fs::IStorage* pStorage
    ) NN_NOEXCEPT;

    //! テストデータが適切に書き込まれているかを比較します。
    static void TestVerifyTestData(
        nn::fs::IStorage* pStorage,
        int64_t sizeFile
    ) NN_NOEXCEPT;

public:
    //! ストレージテストを実行します。
    template <class TestSetup>
    static void RunTest()
    {
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        // テストを生成します
        TestSetup tests;

        if( tests.IsAcceptsSizeZero() )
        {
            // ストレージサイズ0を許容できるか、確認します。
            for( size_t sizeBlock = 64; sizeBlock < MaxBlockSize; sizeBlock *= 2 )
            {
                tests.Format(0, 0, sizeBlock);
                tests.Initialize(0, 0, sizeBlock);
                TestSizeZero(tests.GetStorage(), tests.IsAutoExtendSizeOnWrite(), tests.IsReadOnly());
                tests.Finalize();
            }
        }

        for( uint32_t loop = 0; loop < TestLoop; ++loop )
        {
            NN_SDK_LOG(".");

            size_t sizeBlock = GenerateRandomBlockSize(64, MaxBlockSize);
            int64_t sizeFile =
                std::uniform_int_distribution<int64_t>(TestSetup::MinFileSize, TestSize)(mt);
            int64_t offsetFile =
                std::uniform_int_distribution<>(0, 127)(mt);

            // フォーマット
            // sizeFile 以上のサイズを持つストレージが作成されます
            tests.Format(sizeFile, offsetFile, sizeBlock);

            // 初回マウント
            tests.Initialize(sizeFile, offsetFile, sizeBlock);

            // ストレージサイズが sizeFile 以上か確認します
            {
                int64_t sizeActual = 0;
                ASSERT_NE(tests.GetStorage(), nullptr);
                NNT_ASSERT_RESULT_SUCCESS(tests.GetStorage()->GetSize(&sizeActual));
                ASSERT_GE(sizeActual, sizeFile);
            }

            if ( !tests.IsReadOnly() )
            {
                // 境界へアクセスしたときの挙動を確認します
                TestBounds(
                    tests.GetStorage(),
                    tests.IsEnableResize(),
                    tests.IsAutoExtendSizeOnWrite()
                );

                // 読み書きベリファイをします
                TestReadWrite1(tests.GetStorage(), tests.IsAutoExtendSizeOnWrite());
                TestReadWrite2(tests.GetStorage());

                // テストデータを書き込んでおきます
                TestWriteTestData(&sizeFile, tests.GetStorage());
            }

            // アンマウントします
            tests.Unmount();

            // 再度マウントしてテストを行います。
            tests.Initialize(sizeFile, offsetFile, sizeBlock);

            if( !tests.IsReadOnly() )
            {
                // テストデータが適切に書き込まれているかを比較します。
                TestVerifyTestData(tests.GetStorage(), sizeFile);

                // 境界へアクセスしたときの挙動を確認します
                TestBounds(
                    tests.GetStorage(),
                    tests.IsEnableResize(),
                    tests.IsAutoExtendSizeOnWrite()
                );

                // 読み書きベリファイをします
                TestReadWrite1(tests.GetStorage(), tests.IsAutoExtendSizeOnWrite());
                TestReadWrite2(tests.GetStorage());
            }

            tests.Finalize();
        }
        NN_SDK_LOG("\n");
    }

private:
    static const size_t TestSize = 1024 * 1024;
    static const uint32_t TestLoop = 10;
};
