﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <nn/fssystem/save/fs_TruncatedSubStorage.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

// ストレージ範囲内の読み書きが正しくできるかどうか、
// 範囲外を含む範囲を指定したとしても範囲内のみの読み書きをするかどうかテストします。
TEST(FsTruncatedSubStorageTest, Simple)
{
    static const auto SizeStorage = 0x100000100;
    static const auto SizeBuffer = 0x1000;

    static const auto InitialValue = static_cast<char>(0x55);
    static const auto IsInitial = [](char value) NN_NOEXCEPT
    {
        return value == InitialValue;
    };

    nnt::fs::util::VirtualMemoryStorage storageBase(SizeStorage);
    nn::fssystem::TruncatedSubStorage storageTruncated;
    nnt::fs::util::Vector<char> bufferRead(SizeBuffer, InitialValue);
    nnt::fs::util::Vector<char> bufferWrite(SizeBuffer);

    nn::Result result;

    storageTruncated = nn::fs::SubStorage(&storageBase, 0, SizeStorage);
    std::iota(bufferWrite.begin(), bufferWrite.end(), '\x0');

    result = storageTruncated.Write(-1, bufferWrite.data(), bufferWrite.size());
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);

    result = storageTruncated.Write(0, bufferWrite.data(), bufferWrite.size());
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = storageTruncated.Read(0, bufferRead.data(), bufferRead.size());
    NNT_EXPECT_RESULT_SUCCESS(result);
    EXPECT_TRUE(std::equal(bufferWrite.begin(), bufferWrite.end(), bufferRead.begin()));

    result = storageTruncated.Write(SizeStorage - SizeBuffer, bufferWrite.data(), bufferWrite.size());
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = storageTruncated.Read(SizeStorage - SizeBuffer, bufferRead.data(), bufferRead.size());
    NNT_EXPECT_RESULT_SUCCESS(result);
    EXPECT_TRUE(std::equal(bufferWrite.begin(), bufferWrite.end(), bufferRead.begin()));

    std::fill(bufferRead.begin(), bufferRead.end(), InitialValue);
    result = storageTruncated.Write(SizeStorage - 1, bufferWrite.data(), bufferWrite.size());
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = storageTruncated.Read(SizeStorage - 1, bufferRead.data(), bufferRead.size());
    NNT_EXPECT_RESULT_SUCCESS(result);
    EXPECT_EQ(bufferWrite.front(), bufferRead.front());
    EXPECT_TRUE(std::all_of(std::next(bufferRead.begin()), bufferRead.end(), IsInitial));

    std::fill(bufferRead.begin(), bufferRead.end(), InitialValue);
    result = storageTruncated.Write(SizeStorage, bufferWrite.data(), bufferWrite.size());
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = storageTruncated.Read(SizeStorage, bufferRead.data(), bufferRead.size());
    NNT_EXPECT_RESULT_SUCCESS(result);
    EXPECT_TRUE(std::all_of(bufferRead.begin(), bufferRead.end(), IsInitial));
}
