﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataStorage.h>
#include <nn/fssystem/save/fs_JournalStorage.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

#include "testFs_util_CommonStorageTests.h"

namespace nn { namespace fssystem { namespace save {

namespace
{
struct TestParam
{
    uint32_t journalCountMapEntry;  // ジャーナルのデータ本体のブロック数
    uint32_t journalCountReserved;  // ジャーナルの予約領域のブロック数
    int64_t journalSizeBlock;       // ジャーナルのブロックサイズ
    size_t verificationInputParam0; // 完全性検証のハッシュストレージのサイズレベル0
    size_t verificationInputParam1; // 完全性検証のハッシュストレージのサイズレベル1
    size_t verificationInputParam2; // 完全性検証のハッシュストレージのサイズレベル2
    size_t verificationInputParam3; // 完全性検証のハッシュストレージのサイズレベル3
    int maxCacheCount;              // 完全性検証用キャッシュバッファの最大キャッシュ数
    size_t cacheBufferPageSize;     // 完全性検証用キャッシュバッファのページサイズ
    size_t cacheBufferMaxOrder;     // 完全性検証用キャッシュバッファの最大キャッシュサイズのオーダー

    int64_t GetMapEntrySize() const NN_NOEXCEPT
    {
        return journalCountMapEntry * journalSizeBlock;
    }

    int64_t GetReservedAreaSize() const NN_NOEXCEPT
    {
        return journalCountReserved * journalSizeBlock;
    }

    int64_t GetStorageSize() const NN_NOEXCEPT
    {
        return GetMapEntrySize() + GetReservedAreaSize();
    }

    size_t GetCacheSize() const NN_NOEXCEPT
    {
        return cacheBufferPageSize * maxCacheCount;
    }
};

nnt::fs::util::Vector<TestParam> MakeTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<TestParam> testParams;

    {
        TestParam data;
        data.journalCountMapEntry = 128;
        data.journalCountReserved = 32;
        data.journalSizeBlock = 128;
        data.verificationInputParam0 = 1 << 10;
        data.verificationInputParam1 = 1 << 10;
        data.verificationInputParam2 = 1 << 10;
        data.verificationInputParam3 = 1 << 11;
        data.maxCacheCount = 1024;
        data.cacheBufferPageSize = 4096;
        data.cacheBufferMaxOrder = 4;
        testParams.push_back(data);
    }

    {
        TestParam data;
        data.journalCountMapEntry = 512;
        data.journalCountReserved = 256;
        data.journalSizeBlock = 512;
        data.verificationInputParam0 = 1 << 12;
        data.verificationInputParam1 = 1 << 12;
        data.verificationInputParam2 = 1 << 12;
        data.verificationInputParam3 = 1 << 12;
        data.maxCacheCount = 1024;
        data.cacheBufferPageSize = 4096;
        data.cacheBufferMaxOrder = 4;
        testParams.push_back(data);
    }

    {
        TestParam data;
        data.journalCountMapEntry = 256;
        data.journalCountReserved = 16;
        data.journalSizeBlock = 1024;
        data.verificationInputParam0 = 1 << 10;
        data.verificationInputParam1 = 1 << 11;
        data.verificationInputParam2 = 1 << 12;
        data.verificationInputParam3 = 1 << 13;
        data.maxCacheCount = 1024;
        data.cacheBufferPageSize = 4096;
        data.cacheBufferMaxOrder = 8;
        testParams.push_back(data);
    }
#ifdef NN_BUILD_CONFIG_OS_WIN
    {
        TestParam data;
        data.journalCountMapEntry = 1024;
        data.journalCountReserved = 64;
        data.journalSizeBlock = 1024;
        data.verificationInputParam0 = 1 << 10;
        data.verificationInputParam1 = 1 << 10;
        data.verificationInputParam2 = 1 << 10;
        data.verificationInputParam3 = 1 << 10;
        data.maxCacheCount = 2048;
        data.cacheBufferPageSize = 4096;
        data.cacheBufferMaxOrder = 12;
        testParams.push_back(data);
    }
#endif
    return testParams;
}

nnt::fs::util::Vector<TestParam> MakeLargeTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<TestParam> testParams;

    {
        TestParam data;
        data.journalCountMapEntry = 8 * 1024 * 1024;
        data.journalCountReserved = 32;
        data.journalSizeBlock = 16 * 1024;
        data.verificationInputParam0 = 64 * 1024;
        data.verificationInputParam1 = 32 * 1024;
        data.verificationInputParam2 = 4 * 1024;
        data.verificationInputParam3 = 4 * 1024;
        data.maxCacheCount = 4 * 1024;
        data.cacheBufferPageSize = 16 * 1024;
        data.cacheBufferMaxOrder = 8;
        testParams.push_back(data);
    }

    return testParams;
}

}

template<typename TStorage>
class FsJournalIntegritySaveDataStorageTestTemplate : public ::testing::TestWithParam<TestParam>
{
public:
    FsJournalIntegritySaveDataStorageTestTemplate() NN_NOEXCEPT
        : m_BufferManagerSet(),
        m_VerificationInfo(),
        m_Locker(true)
    {
    }

    // テストのセットアップ
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        const TestParam& param = GetParam();
        NNT_ASSERT_RESULT_SUCCESS(
            FormatStorage(param)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            MountStorage()
        );
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        m_SaveDataStorage.Finalize();
    }

    Result FormatStorage(const TestParam& param) NN_NOEXCEPT
    {
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam inputParamIntegrity;
        inputParamIntegrity.sizeBlockLevel[0] = param.verificationInputParam0;
        inputParamIntegrity.sizeBlockLevel[1] = param.verificationInputParam1;
        inputParamIntegrity.sizeBlockLevel[2] = param.verificationInputParam2;
        inputParamIntegrity.sizeBlockLevel[3] = param.verificationInputParam3;

        const int64_t entrySize = param.GetMapEntrySize();
        const int64_t reservedAreaSize = param.GetReservedAreaSize();

        // 初期化に必要な値を QuerySize で取得します
        int64_t sizeTableMax;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::HierarchicalIntegrityVerificationSizeSet sizeSet = { 0 };
        JournalIntegritySaveDataStorage::QuerySize(
            &sizeTableMax,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            &sizeSet,
            inputParamIntegrity,
            reservedAreaSize,
            param.journalSizeBlock,
            entrySize
        );

        // 署名検証のためのメタデータを設定します
        nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation metaInfo = { 0 };
        nn::fs::SubStorage verificationControlAreaStorage;
        {
            const auto bodySize = param.GetStorageSize() + sizeSet.layeredHashSizes[0] + sizeSet.layeredHashSizes[1] + sizeSet.layeredHashSizes[2];
            m_StorageVerificationControlArea.Initialize(static_cast<size_t>(sizeSet.controlSize));
            {
                int64_t offset = 0;
                metaInfo.Format();
                metaInfo.sizeMasterHash = static_cast<uint32_t>(sizeSet.masterHashSize);
                metaInfo.infoLevelHash.maxLayers = 5;
                metaInfo.infoLevelHash.info[3].size = bodySize;
                for( auto level = 0; level < 3; ++level )
                {
                    const auto size = sizeSet.layeredHashSizes[level];
                    metaInfo.infoLevelHash.info[level].offset = offset;
                    metaInfo.infoLevelHash.info[level].size = size;
                    metaInfo.infoLevelHash.info[level].orderBlock
                        = nn::fssystem::save::ILog2(
                            static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[level]));
                    metaInfo.infoLevelHash.info[3].size
                        = metaInfo.infoLevelHash.info[3].size - size;
                    offset += size;
                }
                metaInfo.infoLevelHash.info[3].offset = offset;
                metaInfo.infoLevelHash.info[3].orderBlock
                    = nn::fssystem::save::ILog2(
                        static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[3]));
            }
            verificationControlAreaStorage = fs::SubStorage(&m_StorageVerificationControlArea, 0, sizeSet.controlSize);
            NN_RESULT_DO(
                nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::Format(
                    verificationControlAreaStorage,
                    metaInfo
                )
            );
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea controlArea;
            NN_RESULT_DO(controlArea.Initialize(verificationControlAreaStorage));
            controlArea.GetLevelHashInfo(&m_VerificationInfo);
        }

        // 各ストレージの初期化します
        const int64_t sizeControlArea = sizeof(nn::fssystem::save::JournalStorage::ControlArea);
        m_StorageJournalControlArea.Initialize(sizeControlArea);
        m_StorageBitmapUpdatedPhysical.Initialize(sizeBitmapUpdatedPhysical);
        m_StorageBitmapUpdatedVirtual.Initialize(sizeBitmapUpdatedVirtual);
        m_StorageBitmapUnassigned.Initialize(sizeBitmapUnassigned);
        m_StorageTable.Initialize(sizeTableMax);
        m_StorageJournalBase.Initialize(entrySize + reservedAreaSize);
        m_StorageMasterHash.Initialize(sizeSet.masterHashSize);
        m_StorageHash.Initialize(m_VerificationInfo.info[0].size + m_VerificationInfo.info[1].size + m_VerificationInfo.info[2].size + m_VerificationInfo.info[3].size);

        // 各ストレージをサブストレージ化します
        nn::fs::SubStorage storageMasterHash(&m_StorageMasterHash, 0, sizeSet.masterHashSize);
        nn::fs::SubStorage subStorageJournalControlArea(&m_StorageJournalControlArea, 0, sizeControlArea);
        nn::fs::SubStorage subStorageTable(&m_StorageTable, 0, sizeTableMax);
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(&m_StorageBitmapUpdatedPhysical, 0, sizeBitmapUpdatedPhysical);
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(&m_StorageBitmapUpdatedVirtual, 0, sizeBitmapUpdatedVirtual);
        nn::fs::SubStorage subStorageBitmapUnassigned(&m_StorageBitmapUnassigned, 0, sizeBitmapUnassigned);
        nn::fs::SubStorage subStorageData(&m_StorageJournalBase, 0, entrySize + reservedAreaSize);
        nn::fs::SubStorage storageLayeredHashL1(&m_StorageHash, m_VerificationInfo.info[0].offset, m_VerificationInfo.info[0].size);
        nn::fs::SubStorage storageLayeredHashL2(&m_StorageHash, m_VerificationInfo.info[1].offset, m_VerificationInfo.info[1].size);
        nn::fs::SubStorage storageLayeredHashL3(&m_StorageHash, m_VerificationInfo.info[2].offset, m_VerificationInfo.info[2].size);

        // 初期化用にサブストレージをコピーします
        m_SubStorageJournalControlArea = subStorageJournalControlArea;
        m_SubStorageJournalTable = subStorageTable;
        m_SubStorageJournalBitmapUpdatedPhysical = subStorageBitmapUpdatedPhysical;
        m_SubStorageJournalBitmapUpdatedVirtual = subStorageBitmapUpdatedVirtual;
        m_SubStorageJournalBitmapUnasiggned = subStorageBitmapUnassigned;
        m_SubStorageVerificationMasterHash = storageMasterHash;
        m_SubStorageVerificationLayeredHashL1 = storageLayeredHashL1;
        m_SubStorageVerificationLayeredHashL2 = storageLayeredHashL2;
        m_SubStorageVerificationLayeredHashL3 = storageLayeredHashL3;
        m_SubStorageData = subStorageData;

        InitBuffer(param);

        // フォーマット
        return m_SaveDataStorage.Format(
            subStorageJournalControlArea,
            subStorageTable,
            subStorageBitmapUpdatedPhysical,
            subStorageBitmapUpdatedVirtual,
            subStorageBitmapUnassigned,
            verificationControlAreaStorage,
            storageMasterHash,
            reservedAreaSize,
            param.journalSizeBlock,
            metaInfo,
            entrySize
        );
    } // NOLINT(impl/function_size)

    // 完全性検証ハッシュに割り当てるキャッシュバッファを設定
    void InitBuffer(const TestParam& param) NN_NOEXCEPT
    {
        size_t cacheBufferMax = static_cast<size_t>(1) << (param.cacheBufferMaxOrder - 1);
        const size_t sizeUnitBlock = cacheBufferMax * param.cacheBufferPageSize;
        size_t sizeCacheBuffer = sizeUnitBlock * nn::fssystem::save::IntegrityLayerCountSave;
        m_CacheBuffer.reset(new char[sizeCacheBuffer]);
        {
            m_BufferManager.Initialize(
                param.maxCacheCount,
                reinterpret_cast<uintptr_t>(m_CacheBuffer.get()),
                sizeCacheBuffer,
                param.cacheBufferPageSize
            );
        }
        for( size_t i = 0; i < IntegrityLayerCountSave; ++i )
        {
            m_BufferManagerSet.pBuffer[i] = &m_BufferManager;
        }
    }

    // 初期化
    Result MountStorage() NN_NOEXCEPT
    {
        return m_SaveDataStorage.Initialize(
            m_SubStorageJournalControlArea,
            m_SubStorageJournalTable,
            m_SubStorageJournalBitmapUpdatedPhysical,
            m_SubStorageJournalBitmapUpdatedVirtual,
            m_SubStorageJournalBitmapUnasiggned,
            m_SubStorageVerificationMasterHash,
            m_SubStorageVerificationLayeredHashL1,
            m_SubStorageVerificationLayeredHashL2,
            m_SubStorageVerificationLayeredHashL3,
            m_SubStorageData,
            m_VerificationInfo,
            &m_BufferManagerSet,
            &m_Locker
        );
    }

    void UnmountStorage() NN_NOEXCEPT
    {
        m_SaveDataStorage.Finalize();
    }

    Result ReadStorage(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        return m_SaveDataStorage.Read(offset, buffer, size);
    }
    Result WriteStorage(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        return m_SaveDataStorage.Write(offset, buffer, size);
    }
    Result OperateStorage(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size) NN_NOEXCEPT
    {
        return m_SaveDataStorage.OperateRange(outBuffer, outBufferSize, operationId, offset, size, nullptr, 0);
    }
    Result CommitStorage() NN_NOEXCEPT
    {
        return m_SaveDataStorage.Commit();
    }
    void ResetBuffer() const NN_NOEXCEPT
    {
        nn::fssystem::FileSystemBufferManager* bufferManager
            = reinterpret_cast<nn::fssystem::FileSystemBufferManager*>(m_BufferManagerSet.pBuffer[0]);
        bufferManager->Finalize();
    }
    fs::SubStorage* GetSubStorageMasterHash() NN_NOEXCEPT
    {
        return &m_SubStorageVerificationMasterHash;
    }
    void ResetAccessCounter() NN_NOEXCEPT
    {
        m_StorageMasterHash.ResetAccessCounter();
        m_StorageHash.ResetAccessCounter();
    }
    bool CheckWriteCounterZero() const NN_NOEXCEPT
    {
        int writeCount = m_StorageMasterHash.GetWriteTimes() + m_StorageHash.GetWriteTimes();
        return writeCount == 0;
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return &m_SaveDataStorage;
    }

private:
    // ストレージ本体
    JournalIntegritySaveDataStorage m_SaveDataStorage;

    // 下に敷くストレージ
    TStorage m_StorageJournalControlArea;         // ジャーナルのコントロールエリア
    TStorage m_StorageJournalBase;                // ジャーナルストレージ本体
    TStorage m_StorageTable;                      // ジャーナルのマッピングテーブル
    TStorage m_StorageBitmapUpdatedPhysical;      // ジャーナルの上書き済みビットマップ(物理インデックスベース)
    TStorage m_StorageBitmapUpdatedVirtual;       // ジャーナルの上書き済みビットマップ(論理インデックスベース)
    TStorage m_StorageBitmapUnassigned;           // ジャーナルの未割当ビットマップ(物理インデックスベース)
    TStorage m_StorageMasterHash;                 // 署名検証のマスターハッシュ
    TStorage m_StorageHash;                       // 署名検証のハッシュレイヤ x 3
    TStorage m_StorageVerificationControlArea;    // 署名検証のコントロール情報

    // バッファマネージャ
    FilesystemBufferManagerSet m_BufferManagerSet;
    nn::fssystem::FileSystemBufferManager m_BufferManager;
    std::unique_ptr<char[]> m_CacheBuffer;

    // マウント用の値
    fs::SubStorage m_SubStorageJournalControlArea;
    fs::SubStorage m_SubStorageJournalTable;
    fs::SubStorage m_SubStorageJournalBitmapUpdatedPhysical;
    fs::SubStorage m_SubStorageJournalBitmapUpdatedVirtual;
    fs::SubStorage m_SubStorageJournalBitmapUnasiggned;
    fs::SubStorage m_SubStorageVerificationMasterHash;
    fs::SubStorage m_SubStorageVerificationLayeredHashL1;
    fs::SubStorage m_SubStorageVerificationLayeredHashL2;
    fs::SubStorage m_SubStorageVerificationLayeredHashL3;
    fs::SubStorage m_SubStorageData;
    HierarchicalIntegrityVerificationInformation m_VerificationInfo;
    nn::os::Mutex m_Locker;
};

typedef FsJournalIntegritySaveDataStorageTestTemplate<nnt::fs::util::AccessCountedMemoryStorage> FsJournalIntegritySaveDataStorageTest;
typedef FsJournalIntegritySaveDataStorageTestTemplate<nnt::fs::util::VirtualMemoryStorage> FsJournalIntegritySaveDataStorageLargeTest;

/**
* 成功パターンテスト
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestSimple)
{
    int offset = 20;
    int size = 40;

    std::unique_ptr<char[]> writeBuffer(new char[size]);
    std::unique_ptr<char[]> readBuffer(new char[size]);
    char* pWriteBuffer = writeBuffer.get();
    char* pReadBuffer = readBuffer.get();
    std::memset(pWriteBuffer, 0xFF, size);

    // 書き込んでコミット
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset + 10, pWriteBuffer, size - 10));
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset + 5, pWriteBuffer, size - 5));
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, pWriteBuffer, size));
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, pWriteBuffer, size));
    NNT_ASSERT_RESULT_SUCCESS(CommitStorage());

    // 書き込んだものが読み込めるかテスト
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(offset + 8, pReadBuffer, size - 8));
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(offset, pReadBuffer, size));
    ASSERT_EQ(0, std::memcmp(pReadBuffer, pWriteBuffer, size));

    // 書き込んだ後、コミットしません
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset + 1, pWriteBuffer, size - 1));
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset + 5, pWriteBuffer, size - 5));
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(offset, pReadBuffer, size));

    // 最初に書き込んだものが読めるかテスト
    ASSERT_EQ(0, std::memcmp(pReadBuffer, pWriteBuffer, size));
}

/**
* コミットするまで下位ストレージにアクセスしないことを確かめます
* 予約領域のサイズで 1 度だけ書き込みを行います
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestAccessCountSingleWrite)
{
    const TestParam& param = GetParam();
    int64_t entrySize = param.GetMapEntrySize();
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());
    std::unique_ptr<char[]> buffer(new char[reservedSize]);
    std::iota(buffer.get(), buffer.get() + reservedSize, '\x1');
    int64_t offsetMax = entrySize - reservedSize;
    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;

        // Write を行ったときのアクセスの有無をチェック
        {
            ResetAccessCounter();
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(
                    offset,
                    buffer.get(),
                    reservedSize
                )
            );
            ASSERT_TRUE(CheckWriteCounterZero());
        }

        // コミットすることで下位ストレージを書き換えます
        {
            ResetAccessCounter();
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
            ASSERT_FALSE(CheckWriteCounterZero());
        }

        // 初期化し直します
        UnmountStorage();
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }
}

/**
* コミットするまで下位ストレージにアクセスしないことを確かめます
* 書き込みは 予約領域のサイズの 1/4 を 4回
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestAccessCountMultipleWrite)
{
    const TestParam& param = GetParam();
    int64_t entrySize = param.GetMapEntrySize();
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());
    std::unique_ptr<char[]> buffer(new char[reservedSize]);
    std::iota(buffer.get(), buffer.get() + reservedSize, '\x1');
    int64_t offsetMax = entrySize - reservedSize;
    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;

        // Write を行ったときのアクセスの有無をチェック
        {
            ResetAccessCounter();
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(offset, buffer.get(), reservedSize / 4)
            );
            offset += reservedSize / 4;
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(offset, buffer.get(), reservedSize / 4)
            );
            offset += reservedSize / 4;
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(offset, buffer.get(), reservedSize / 4)
            );
            offset += reservedSize / 4;
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(offset, buffer.get(), reservedSize / 4)
            );
            ASSERT_TRUE(CheckWriteCounterZero());
        }

        // コミットすることで下位ストレージを書き換えます
        {
            ResetAccessCounter();
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
            ASSERT_FALSE(CheckWriteCounterZero());
        }

        // 初期化し直す
        UnmountStorage();
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }
}

/**
* 0 バイト書き込みでは、コミットしても下位ストレージにアクセスしないことを確かめます
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestAccessCountZeroWrite)
{
    const TestParam& param = GetParam();
    int64_t storageSize = param.GetStorageSize();
    std::unique_ptr<char[]> buffer(new char[1]);
    int64_t offsetMax = storageSize;
    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;
        ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(
                offset,
                buffer.get(),
                0
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        ASSERT_TRUE(CheckWriteCounterZero());
    }
}

/**
* マウント直後のコミット等で、下位ストレージにアクセスしないことを確かめます
* 予約領域以上のサイズは書き込みません
* 書き込みは 予約領域のサイズの 1/4 を 4回
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestUnmount)
{
    const TestParam& param = GetParam();
    int64_t entrySize = param.GetMapEntrySize();
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    std::unique_ptr<char[]> buffer(new char[reservedSize]);
    std::iota(buffer.get(), buffer.get() + reservedSize, '\x1');

    // Write を行うだけでは、下位ストレージにアクセスしません
    int64_t offsetMax = entrySize - reservedSize;
    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;

        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(offset, buffer.get(), reservedSize / 4)
        );
        offset += reservedSize / 4;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(offset, buffer.get(), reservedSize / 4)
        );
        offset += reservedSize / 4;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(offset, buffer.get(), reservedSize / 4)
        );
        offset += reservedSize / 4;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(offset, buffer.get(), reservedSize / 4)
        );

        // 書き込みを行ってからアンマウントすると、
        // Journal 復元処理の際にアクセスが発生します
        ResetAccessCounter();
        UnmountStorage();
        ASSERT_FALSE(CheckWriteCounterZero());

        // マウント時に下位ストレージにアクセスが発生しません
        ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
        ASSERT_TRUE(CheckWriteCounterZero());

        // マウント直後にコミットしても下位ストレージにアクセスが発生しません
        ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        ASSERT_TRUE(CheckWriteCounterZero());

        // コミット直後のアンマウント時はアクセスが発生しません
        ResetAccessCounter();
        UnmountStorage();
        ASSERT_TRUE(CheckWriteCounterZero());

        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }
}

/**
* Write した値を Commit した後 Read で読み取ります
* 書き込みは予約領域のサイズ
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestReadWrite)
{
    const TestParam& param = GetParam();

    int64_t entrySize = param.GetMapEntrySize();
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());
    int64_t offsetMax = entrySize - reservedSize;

    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;

        // 書き込み
        std::unique_ptr<char[]> writeBuffer(new char[reservedSize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + reservedSize, static_cast<char>(i));
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), reservedSize));

        // コミット
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());

        // 読み込み
        std::unique_ptr<char[]> readBuffer(new char[reservedSize]);
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(offset, readBuffer.get(), reservedSize));
        ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), reservedSize));
    }
}

/**
* ストレージ全体に書き込みを行い、コミットし、それが Read できるか確かめます
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestReadWriteEntire)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');

    // ストレージ全体に書き込みを行います
    {
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - reservedSize; writtenSize += reservedSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(
                    writtenSize,
                    writeBuffer.get() + writtenSize,
                    static_cast<size_t>(reservedSize)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    // 読み込み
    std::unique_ptr<char[]> readBuffer(new char[entrySize]);
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
    ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), entrySize));
}

/**
* ストレージ全体に書き込みを行い、コミットし、再マウントしてから Read できるか確かめます
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestReadWriteEntireRemount)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');

    // ストレージ全体に書き込みを行います
    {
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - reservedSize; writtenSize += reservedSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, reservedSize));
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    // 再マウント
    UnmountStorage();
    NNT_ASSERT_RESULT_SUCCESS(MountStorage());

    // 読み込み
    std::unique_ptr<char[]> readBuffer(new char[entrySize]);
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
    ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), entrySize));
}

/**
* 書き込み直後に読み込みを行ったときの値を確かめます
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestReadWriteWithoutCommit)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    // ストレージ全体に事前に書き込みを行います
    {
        std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - reservedSize; writtenSize += reservedSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, reservedSize));
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    int64_t offsetMax = entrySize - reservedSize;
    for( int i = 0; i < 3; ++i )
    {
        // 書き込み
        int64_t offset = offsetMax * i / 2;
        std::unique_ptr<char[]> writeBuffer(new char[reservedSize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + reservedSize, '\xA');
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), reservedSize));

        // 読み込み
        std::unique_ptr<char[]> readBuffer(new char[entrySize]);
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
        ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get() + offset, reservedSize));

        // コミットして再マウント
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        UnmountStorage();
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }
}

/**
* とびとびで Write した後 Commit してから Read を行います
* Write するサイズの合計は予約領域サイズまでとします
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestReadWriteSkipping)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    // いったん全体に書き込んでおきます
    {
        std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - reservedSize; writtenSize += reservedSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, reservedSize));
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    // skipSize ずつとばしながら writeSize ずつ書き込みます
    const int64_t skipSize = param.verificationInputParam3;
    const size_t writeSize = param.verificationInputParam3;

    // 比較用のデータも作成
    std::unique_ptr<char[]> compareBuffer(new char[entrySize]);
    std::iota(compareBuffer.get(), compareBuffer.get() + entrySize, '\x1');

    int64_t offset = skipSize;
    int64_t writtenSize = 0;
    std::unique_ptr<char[]> writeBuffer(new char[writeSize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + writeSize, '\xA');
    for( ;; )
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), writeSize));
        memcpy(compareBuffer.get() + offset, writeBuffer.get(), writeSize);
        offset += writeSize + skipSize;
        writtenSize += writeSize;
        if( writtenSize + writeSize > reservedSize
            || offset + writeSize > entrySize )
        {
            break;
        }
    }
    NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    UnmountStorage();
    NNT_ASSERT_RESULT_SUCCESS(MountStorage());

    // 読み込まれる値は書き込んだ値と等しくなります
    {
        std::unique_ptr<char[]> readBuffer(new char[entrySize]);
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
        ASSERT_EQ(0, memcmp(compareBuffer.get(), readBuffer.get(), entrySize));
    }
}

/**
* 重複させながら Write した後 commit してから Read を行います
* Write するサイズの合計は予約領域サイズより大きくなります
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestReadWriteOverlap)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    // いったん全体に書き込んでおきます
    {
        std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - reservedSize; writtenSize += reservedSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, reservedSize));
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    const size_t overlapSize = reservedSize / 8;
    const size_t writeSize = reservedSize / 4 + overlapSize;

    // 比較用のデータも作成
    std::unique_ptr<char[]> compareBuffer(new char[entrySize]);
    std::iota(compareBuffer.get(), compareBuffer.get() + entrySize, '\x1');

    // 重複させながら Write します
    {
        std::unique_ptr<char[]> writeBuffer(new char[writeSize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + writeSize, '\xA');
        int64_t offset = 0;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), writeSize));
        memcpy(compareBuffer.get() + offset, writeBuffer.get(), writeSize);
        offset += writeSize - overlapSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), writeSize));
        memcpy(compareBuffer.get() + offset, writeBuffer.get(), writeSize);
        offset += writeSize - overlapSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), writeSize));
        memcpy(compareBuffer.get() + offset, writeBuffer.get(), writeSize);
    }
    NNT_ASSERT_RESULT_SUCCESS(CommitStorage());

    // 読み込まれる値は書き込んだ値と等しくなります
    {
        std::unique_ptr<char[]> readBuffer(new char[entrySize]);
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
        ASSERT_EQ(0, memcmp(compareBuffer.get(), readBuffer.get(), entrySize));
    }
}

/**
* 読み込み時と書き込み時のオフセットが違っていても正しく読み込めるか確かめます
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestReadWriteOffset)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    // 全体に書き込み
    {
        std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - reservedSize; writtenSize += reservedSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(writtenSize, writeBuffer.get() + writtenSize, reservedSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    // offset を 0～ブロックサイズ でずらしながら読み込みます
    std::unique_ptr<char[]> compareBuffer(new char[reservedSize]);
    std::unique_ptr<char[]> readBuffer(new char[reservedSize]);
    for( int i = 0; i <= param.journalSizeBlock; ++i )
    {
        std::iota(
            compareBuffer.get(),
            compareBuffer.get() + reservedSize,
            static_cast<char>(i + 1)
        );
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(i, readBuffer.get(), reservedSize));
        ASSERT_EQ(0, memcmp(compareBuffer.get(), readBuffer.get(), reservedSize));
    }
}

/**
* 境界周辺に書き込み、正しく書き込めているか確認します
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestBoundaryValue)
{
    const TestParam& param = GetParam();

    const int checkSizeMax = 10;
    if( param.GetReservedAreaSize() < checkSizeMax )
    {
        return;
    }

    int64_t entrySize = param.GetMapEntrySize();

    std::unique_ptr<char[]> writeBuffer(new char[checkSizeMax]);
    std::unique_ptr<char[]> readBuffer(new char[checkSizeMax]);

    for( int size = 10; size > 0; --size )
    {
        std::iota(writeBuffer.get(), writeBuffer.get() + checkSizeMax, static_cast<char>(size * 2));
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(entrySize - size, writeBuffer.get(), size));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(entrySize - size, readBuffer.get(), size));
        ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), size));
    }
}

/**
* 予約領域のサイズぴったりで 1 度だけ書き込みます
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestReservedAreaJust)
{
    const TestParam& param = GetParam();
    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t writableSize = static_cast<size_t>(param.GetReservedAreaSize());

    std::unique_ptr<char[]> writeBuffer(new char[writableSize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + writableSize, '\x1');

    std::unique_ptr<char[]> readBuffer(new char[entrySize]);

    int64_t offsetMax = entrySize - writableSize;
    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(offset, writeBuffer.get(), writableSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }
}

/**
* 予約領域のサイズより 1 バイトだけ大きいサイズで 1 度だけ書き込んだ場合に
* エラーが発生することを確認します
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestReservedAreaOverflow)
{
    const TestParam& param = GetParam();
    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    std::unique_ptr<char[]> writeBuffer(new char[reservedSize + 1]);
    std::iota(writeBuffer.get(), writeBuffer.get() + reservedSize + 1, '\x1');

    std::unique_ptr<char[]> readBuffer(new char[entrySize]);

    int64_t offsetMax = entrySize - (reservedSize + 1);

    for( int i = 0; i < 3; ++i )
    {
        // このテストで破壊されたデータを復元する機能は
        // IntegritySaveDataStorage にはないので、毎回初期化する
        {
            UnmountStorage();
            ResetBuffer();
            NNT_ASSERT_RESULT_SUCCESS(
                FormatStorage(param)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                MountStorage()
            );
        }

        int64_t offset = offsetMax * i / 2;

        // Write 時はエラーが発生するときとしないことがある
        Result resultWrite = WriteStorage(offset, writeBuffer.get(), reservedSize + 1);
        if( resultWrite.IsFailure() )
        {
            // Write のエラーの内容確認
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultMappingTableFull,
                resultWrite
            );

            // セーブデータの状態が更新不能になる
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultMappingTableFull,
                WriteStorage(offset, writeBuffer.get(), 1)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultMappingTableFull,
                CommitStorage()
            );
        }
        else
        {
            // 書き込みが成功してもコミットは失敗する
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultMappingTableFull,
                CommitStorage()
            );
        }
    }
}

/**
* 予約領域ちょうどのサイズで書き込みを行います
* オフセットを変えながら、予約領域を 1～journalSizeBlock で割った数ずつ書き込みます
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestWriteJustSize)
{
    const TestParam& param = GetParam();
    int64_t entrySize = param.GetMapEntrySize();
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    std::unique_ptr<char[]> writeBuffer(new char[reservedSize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + reservedSize, '\x1');

    int64_t offsetMax = entrySize - reservedSize;
    for( int countOffsetLoop = 0; countOffsetLoop < 3; ++countOffsetLoop )
    {
        int64_t offsetBase = offsetMax * countOffsetLoop / 2;
        for( int countSplit = 1; countSplit <= param.journalSizeBlock; countSplit <<= 1 )
        {
            // 予約領域を割った数値を書き込みサイズとします
            size_t writeSize = reservedSize / countSplit;
            int64_t offset = offsetBase + reservedSize;

            // 書き込みます
            for( int i = 0; i < countSplit; ++i )
            {
                offset -= writeSize;
                NNT_ASSERT_RESULT_SUCCESS(
                    WriteStorage(
                        offset,
                        writeBuffer.get(),
                        writeSize
                    )
                );
            }

            // コミット
            NNT_ASSERT_RESULT_SUCCESS(
                CommitStorage()
            );

            // 書き込んだ値が読み取れるか調べます
            {
                std::unique_ptr<char[]> readBuffer(new char[reservedSize]);
                NNT_ASSERT_RESULT_SUCCESS(
                    ReadStorage(
                        offsetBase,
                        readBuffer.get(),
                        reservedSize
                    )
                );
                for( int i = 0; i < countSplit; ++i )
                {
                    ASSERT_EQ(
                        0,
                        memcmp(readBuffer.get() + writeSize * i, writeBuffer.get(), writeSize)
                    );
                }
            }
            UnmountStorage();
            MountStorage();
        }
    }
}

/**
* 同じ箇所に何度も書き込むテスト
* 予約領域のサイズで 100 回書き込みと読み込みを行います
* 10 回に 1 回、コミットします
* 10 回に 1 回、再マウントします
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestWriteOverlap)
{
    const TestParam& param = GetParam();
    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
    std::unique_ptr<char[]> compareBuffer(new char[entrySize]);
    std::unique_ptr<char[]> readBuffer(new char[entrySize]);

    // いったん全体に書き込んでおきます
    {
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        std::iota(compareBuffer.get(), compareBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - reservedSize; writtenSize += reservedSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, reservedSize));
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    for( int i = 0; i < 100; ++i )
    {
        int64_t offset = i / 10;

        // 書き込む値は毎回変えます
        std::iota(writeBuffer.get(), writeBuffer.get() + reservedSize, static_cast<char>(i));
        ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(
                offset,
                writeBuffer.get(),
                reservedSize - 10
            )
        );
        ASSERT_TRUE(CheckWriteCounterZero());
        memcpy(compareBuffer.get() + offset, writeBuffer.get(), reservedSize - 10);

        // 10 回に 1 回コミットします
        if( i % 10 == 9 )
        {
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }

        // 10 回に 1 回再マウントします
        if( i % 10 == 4 )
        {
            UnmountStorage();
            NNT_ASSERT_RESULT_SUCCESS(MountStorage());
        }

        // 読み込み
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
        ASSERT_EQ(0, memcmp(compareBuffer.get(), readBuffer.get(), entrySize));
    }
}

/**
* 下位ストレージを書き換えた場合に署名検証エラーが発生するか確かめます
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestSignatureVerificationFailed)
{
    const TestParam& param = GetParam();
    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    std::unique_ptr<char[]> fillBuffer(new char[entrySize]);

    // いったん全体に書き込んでおきます
    {
        std::iota(fillBuffer.get(), fillBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - reservedSize; writtenSize += reservedSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, fillBuffer.get() + writtenSize, reservedSize));
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, fillBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    int64_t bufferSize = 512;
    std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(buffer.get(), buffer.get() + bufferSize, '\x5');

    // 署名データがキャッシュされているので、
    // キャッシュをリセットしておきます
    {
        UnmountStorage();
        ResetBuffer();
        InitBuffer(param);
        MountStorage();
    }

    // マスターハッシュの値を書き換えます
    GetSubStorageMasterHash()->Write(0, buffer.get(), 32);

    // 検証に失敗します
    int64_t readOffset = 256;
    size_t readSize = 256;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIntegrityVerificationStorageCorrupted,
        ReadStorage(readOffset, buffer.get(), readSize)
    );
}

/**
* 範囲操作をテストします
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, TestOperateRange)
{
    // 書き込み可能ストレージではキャッシュ無効化は非対応
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultUnsupportedOperation,
        OperateStorage(nullptr, 0, nn::fs::OperationId::Invalidate, 0, 1));

    // QueryRange
    nn::fs::QueryRangeInfo info;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        OperateStorage(nullptr, sizeof(info), nn::fs::OperationId::QueryRange, 0, 1));
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidSize,
        OperateStorage(&info, 0, nn::fs::OperationId::QueryRange, 0, 1));

    NNT_ASSERT_RESULT_SUCCESS(
        OperateStorage(&info, sizeof(info), nn::fs::OperationId::QueryRange, 0, 1));
    EXPECT_EQ(0, info.aesCtrKeyTypeFlag);
    EXPECT_EQ(0, info.speedEmulationTypeFlag);
}

/**
* Write のオフセットがストレージサイズ以上の場合エラーが発生する
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, DISABLED_BoundaryWriteOffsetTest)
{
    // debug や develop の場合、IntegrityVerificationStorage::Read で
    // サイズが大きすぎると判定され Assert で落ちる
    const TestParam& param = GetParam();
    int64_t entrySize = param.GetMapEntrySize();
    static const int bufferSize = 31;
    if( param.GetReservedAreaSize() <= bufferSize )
    {
        return;
    }
    std::unique_ptr<char[]> writeBuffer(new char[bufferSize * 10]);
    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize * 10, '\x1');

    // エラーが発生しないパターン
    for( int64_t offset = entrySize - bufferSize; offset < entrySize; ++offset )
    {
        for( int size = 1; offset + size < entrySize; ++size )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(offset, writeBuffer.get(), size)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
    }
    // エラーが発生するパターン
    for( int64_t offset = entrySize; offset <= entrySize + bufferSize; ++offset )
    {
        for( size_t size = 1; size <= bufferSize; ++size )
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultOutOfRange,
                WriteStorage(offset, writeBuffer.get(), size)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
    }
}

/**
* Read の オフセット+読み込みサイズがストレージサイズより大きければエラーが発生する
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, DISABLED_BoundaryReadOffsetTest)
{
    // debug や develop の場合、IntegrityVerificationStorage::Read で
    // サイズが大きすぎると判定され Assert で落ちる

    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetMapEntrySize());
    size_t reservedSize = static_cast<size_t>(param.GetReservedAreaSize());

    // いったん全体に書き込んでおく
    {
        std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - reservedSize; writtenSize += reservedSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, reservedSize));
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    static const int bufferSize = 20;
    std::unique_ptr<char[]> readBuffer(new char[bufferSize]);
    // エラーが発生しないパターン
    for( int i = 0; i < bufferSize - 1; ++i )
    {
        int64_t offset = entrySize - (bufferSize - i);
        size_t size = bufferSize - i;
        NNT_ASSERT_RESULT_SUCCESS(
            ReadStorage(offset, readBuffer.get(), size)
        );
    }
    // オフセットを 1 大きくする、Release でないと ASSERT に引っかかる
    for( int i = 0; i < bufferSize - 1; ++i )
    {
        int64_t offset = entrySize - (bufferSize - i) + 1;
        size_t size = bufferSize - i;
        NNT_ASSERT_RESULT_SUCCESS(
            ReadStorage(offset, readBuffer.get(), size)
        );
    }
    // サイズを 1 大きくする、Release でないと ASSERT に引っかかる
    for( int i = 0; i < bufferSize - 1; ++i )
    {
        int64_t offset = entrySize - (bufferSize - i);
        size_t size = bufferSize - i + 1;
        NNT_ASSERT_RESULT_SUCCESS(
            ReadStorage(offset, readBuffer.get(), size)
        );
    }

}

/**
* 境界外まで書き込む処理であっても、境界内の部分は書き込めていることを確かめる
*/
TEST_P(FsJournalIntegritySaveDataStorageTest, DISABLED_BoundaryWriteTest)
{
    // debug や develop の場合、IntegrityVerificationStorage::Read で
    // サイズが大きすぎると判定され Assert で落ちる
    const TestParam& param = GetParam();
    const int checkSizeMax = 10;
    if( param.GetReservedAreaSize() < checkSizeMax )
    {
        return;
    }
    int64_t entrySize = param.GetMapEntrySize();
    std::unique_ptr<char[]> writeBuffer(new char[checkSizeMax]);
    std::unique_ptr<char[]> readBuffer(new char[checkSizeMax]);

    for( int size = checkSizeMax - 1; size > 0; --size )
    {
        std::iota(
            writeBuffer.get(),
            writeBuffer.get() + checkSizeMax,
            static_cast<char>(size * 2)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            WriteStorage(entrySize - size, writeBuffer.get(), checkSizeMax)
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        NNT_ASSERT_RESULT_SUCCESS(
            ReadStorage(entrySize - size, readBuffer.get(), size)
        );
        ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), size));
    }
}

/**
* 4 GB を超えるオフセットのテスト
*/
TEST_P(FsJournalIntegritySaveDataStorageLargeTest, TestSimple)
{
    TestLargeOffsetAccess(GetStorage(), 1024);
}

INSTANTIATE_TEST_CASE_P(
    TestJournalIntegritySaveDataStorage,
    FsJournalIntegritySaveDataStorageTest,
    ::testing::ValuesIn(MakeTestParam())
);

INSTANTIATE_TEST_CASE_P(
    TestJournalIntegritySaveDataStorage,
    FsJournalIntegritySaveDataStorageLargeTest,
    ::testing::ValuesIn(MakeLargeTestParam())
);

}}}
