﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fssystem/fs_ISaveDataExtraDataAccessor.h>

#include "detail/fssrv_SaveDataExtraDataAccessorCacheManager.h"

using namespace nn::fs;
using namespace nn::fssystem;
using namespace nn::fssrv::detail;

namespace {

    class MySaveDataExtraDataAccessor : public ISaveDataExtraDataAccessor
    {
    public:
        MySaveDataExtraDataAccessor() NN_NOEXCEPT : m_Observer(nullptr)
        {
        }
        ~MySaveDataExtraDataAccessor() NN_NOEXCEPT
        {
            if (m_Observer)
            {
                m_Observer->Unregister(m_SpaceId, m_SaveDataId, this);
            }
        }
        virtual nn::Result WriteExtraData(const nn::fs::SaveDataExtraData& extraData) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(extraData);
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result CommitExtraData(bool updateTimeStamp) NN_NOEXCEPT  NN_OVERRIDE
        {
            NN_UNUSED(updateTimeStamp);
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result ReadExtraData(nn::fs::SaveDataExtraData* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            outValue->ownerId = m_SaveDataId;
            outValue->timeStamp = static_cast<int64_t>(m_SpaceId);
            NN_RESULT_SUCCESS;
        }
        virtual void RegisterCacheObserver(ISaveDataExtraDataAccessorCacheObserver* observer, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT NN_OVERRIDE
        {
            m_Observer = observer;
            m_SaveDataId = saveDataId;
            m_SpaceId = spaceId;
        }
    private:
        ISaveDataExtraDataAccessorCacheObserver* m_Observer;
        nn::fs::SaveDataSpaceId m_SpaceId;
        nn::fs::SaveDataId m_SaveDataId;
    };

    void ValidateCache(ISaveDataExtraDataAccessor* accessor, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId id)
    {
        nn::fs::SaveDataExtraData extraData;
        NNT_EXPECT_RESULT_SUCCESS(accessor->ReadExtraData(&extraData));
        EXPECT_EQ(extraData.timeStamp, static_cast<int64_t>(spaceId));
        EXPECT_EQ(extraData.ownerId, id);
    }
}

TEST(SaveDataExtraDataAccessorCacheManager, Basic)
{
    SaveDataExtraDataAccessorCacheManager manager;

    auto acc1 = std::shared_ptr<ISaveDataExtraDataAccessor>(new MySaveDataExtraDataAccessor());
    auto acc2 = std::shared_ptr<ISaveDataExtraDataAccessor>(new MySaveDataExtraDataAccessor());
    auto acc3 = std::shared_ptr<ISaveDataExtraDataAccessor>(new MySaveDataExtraDataAccessor());
    auto acc4 = std::shared_ptr<ISaveDataExtraDataAccessor>(new MySaveDataExtraDataAccessor());

    // キャッシュに登録
    acc1->RegisterCacheObserver(reinterpret_cast<ISaveDataExtraDataAccessorCacheObserver*>(&manager), SaveDataSpaceId::User, 1);
    acc2->RegisterCacheObserver(reinterpret_cast<ISaveDataExtraDataAccessorCacheObserver*>(&manager), SaveDataSpaceId::User, 2);
    acc3->RegisterCacheObserver(reinterpret_cast<ISaveDataExtraDataAccessorCacheObserver*>(&manager), SaveDataSpaceId::User, 3);
    acc4->RegisterCacheObserver(reinterpret_cast<ISaveDataExtraDataAccessorCacheObserver*>(&manager), SaveDataSpaceId::System, 3);

    NNT_EXPECT_RESULT_SUCCESS(manager.Register(acc1, SaveDataSpaceId::User, 1));
    NNT_EXPECT_RESULT_SUCCESS(manager.Register(acc2, SaveDataSpaceId::User, 2));
    NNT_EXPECT_RESULT_SUCCESS(manager.Register(acc3, SaveDataSpaceId::User, 3));
    NNT_EXPECT_RESULT_SUCCESS(manager.Register(acc4, SaveDataSpaceId::System, 3));

    // キャッシュされていることの確認
    {
        std::shared_ptr<ISaveDataExtraDataAccessor> cache;
        NNT_EXPECT_RESULT_SUCCESS(manager.GetCache(&cache, SaveDataSpaceId::User, 1));
        ValidateCache(cache.get(), SaveDataSpaceId::User, 1);
        NNT_EXPECT_RESULT_SUCCESS(manager.GetCache(&cache, SaveDataSpaceId::User, 2));
        ValidateCache(cache.get(), SaveDataSpaceId::User, 2);
        NNT_EXPECT_RESULT_SUCCESS(manager.GetCache(&cache, SaveDataSpaceId::User, 3));
        ValidateCache(cache.get(), SaveDataSpaceId::User, 3);
        NNT_EXPECT_RESULT_SUCCESS(manager.GetCache(&cache, SaveDataSpaceId::System, 3));
        ValidateCache(cache.get(), SaveDataSpaceId::System, 3);
    }

    // キャッシュしていない ID は見つからない
    {
        std::shared_ptr<ISaveDataExtraDataAccessor> cache;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, manager.GetCache(&cache, SaveDataSpaceId::User, 4));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, manager.GetCache(&cache, SaveDataSpaceId::User, 5));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, manager.GetCache(&cache, SaveDataSpaceId::User, 6));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, manager.GetCache(&cache, SaveDataSpaceId::System, 1));
    }

    // 参照がなくなるとキャッシュも削除される
    acc3.reset();
    {
        std::shared_ptr<ISaveDataExtraDataAccessor> cache;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, manager.GetCache(&cache, SaveDataSpaceId::User, 3));
    }

    // キャッシュを持ったまま参照がなくなってもキャッシュは有効
    {
        std::shared_ptr<ISaveDataExtraDataAccessor> cache;
        NNT_EXPECT_RESULT_SUCCESS(manager.GetCache(&cache, SaveDataSpaceId::User, 1));
        acc1.reset();
        ValidateCache(cache.get(), SaveDataSpaceId::User, 1);
        {
            // 新たに取得もできる
            std::shared_ptr<ISaveDataExtraDataAccessor> tmpCache;
            NNT_EXPECT_RESULT_SUCCESS(manager.GetCache(&tmpCache, SaveDataSpaceId::User, 1));
            ValidateCache(tmpCache.get(), SaveDataSpaceId::User, 1);
        }
    }
    // 参照がコピーされたものでもキャッシュは有効
    {
        auto acc2Copy = acc2;
        acc2.reset();

        std::shared_ptr<ISaveDataExtraDataAccessor> cache;
        NNT_EXPECT_RESULT_SUCCESS(manager.GetCache(&cache, SaveDataSpaceId::User, 2));
        ValidateCache(cache.get(), SaveDataSpaceId::User, 2);

        acc2Copy.reset();
        ValidateCache(cache.get(), SaveDataSpaceId::User, 2);
        {
            // 新たに取得もできる
            std::shared_ptr<ISaveDataExtraDataAccessor> tmpCache;
            NNT_EXPECT_RESULT_SUCCESS(manager.GetCache(&tmpCache, SaveDataSpaceId::User, 2));
            ValidateCache(tmpCache.get(), SaveDataSpaceId::User, 2);
        }
    }

    // その後キャッシュを破棄すると次からはキャッシュが無効
    {
        std::shared_ptr<ISaveDataExtraDataAccessor> cache;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, manager.GetCache(&cache, SaveDataSpaceId::User, 1));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, manager.GetCache(&cache, SaveDataSpaceId::User, 2));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, manager.GetCache(&cache, SaveDataSpaceId::User, 3));
    }
}

TEST(SaveDataExtraDataAccessorCacheManager, Destruct)
{
    SaveDataExtraDataAccessorCacheManager manager;

    {
        auto acc1 = std::shared_ptr<ISaveDataExtraDataAccessor>(new MySaveDataExtraDataAccessor());
        auto acc2 = std::shared_ptr<ISaveDataExtraDataAccessor>(new MySaveDataExtraDataAccessor());
        auto acc3 = std::shared_ptr<ISaveDataExtraDataAccessor>(new MySaveDataExtraDataAccessor());
        auto acc4 = std::shared_ptr<ISaveDataExtraDataAccessor>(new MySaveDataExtraDataAccessor());

        acc1->RegisterCacheObserver(reinterpret_cast<ISaveDataExtraDataAccessorCacheObserver*>(&manager), SaveDataSpaceId::User, 1);
        acc2->RegisterCacheObserver(reinterpret_cast<ISaveDataExtraDataAccessorCacheObserver*>(&manager), SaveDataSpaceId::User, 2);
        acc3->RegisterCacheObserver(reinterpret_cast<ISaveDataExtraDataAccessorCacheObserver*>(&manager), SaveDataSpaceId::User, 3);
        acc4->RegisterCacheObserver(reinterpret_cast<ISaveDataExtraDataAccessorCacheObserver*>(&manager), SaveDataSpaceId::System, 3);

        NNT_EXPECT_RESULT_SUCCESS(manager.Register(acc1, SaveDataSpaceId::User, 1));
        NNT_EXPECT_RESULT_SUCCESS(manager.Register(acc2, SaveDataSpaceId::User, 2));
        NNT_EXPECT_RESULT_SUCCESS(manager.Register(acc3, SaveDataSpaceId::User, 3));
        NNT_EXPECT_RESULT_SUCCESS(manager.Register(acc4, SaveDataSpaceId::System, 3));
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
