﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fsa/fs_Registrar.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_RomFsFileSystem.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fssystem/fs_RomFsFileSystem.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>

#include <nnt/fsUtil/testFs_util.h>

namespace {

using namespace nnt::fs::util;
using namespace nn;

struct Entry
{
    char path[nn::fs::EntryNameLengthMax];
    nnt::fs::util::Hash hash;
};

bool operator<(const Entry& lhs, const Entry& rhs)
{
    return strncmp(lhs.path, rhs.path, sizeof(lhs.path)) < 0;
}

nn::Result ListDirectoryRecursive(nnt::fs::util::Vector<Entry>* outValue, const char* path)
{
    return IterateDirectoryRecursive(path,
        [&](const char* path, const fs::DirectoryEntry& entry){
            NN_UNUSED(entry);
            Entry newEntry = {0};
            strncpy(newEntry.path, path, nn::fs::EntryNameLengthMax);
            memset(newEntry.hash.value, 0, sizeof(newEntry.hash.value));
            outValue->push_back(newEntry);
            NN_RESULT_SUCCESS;
        },
        nullptr,
        [&](const char* path, const fs::DirectoryEntry& entry){
            NN_UNUSED(entry);
            Hash hash = {{0}};
            CalculateFileHash(&hash, path);
            Entry newEntry = {0};
            strncpy(newEntry.path, path, nn::fs::EntryNameLengthMax);
            newEntry.hash = hash;
            outValue->push_back(newEntry);
            NN_RESULT_SUCCESS;
        }
        );
}

const char* g_SourceDirectoryPath;
const char* g_ArchiveFilePath;

class FsRomFsFileSystemTestTraits
{
public:
    typedef fs::RomFsFileSystem RomFsFileSystemType;

public:
    static const bool IsDeathOnLackOfCacheBuffer = true;
};

class FsSystemRomFsFileSystemTestTraits
{
public:
    typedef fssystem::RomFsFileSystem RomFsFileSystemType;

public:
    static const bool IsDeathOnLackOfCacheBuffer = false;
};

template <typename T>
class RomFsFileSystemTest : public ::testing::Test
{
public:
    typedef typename T::RomFsFileSystemType RomFsFileSystem;

public:
    static const bool IsDeathOnLackOfCacheBuffer = T::IsDeathOnLackOfCacheBuffer;
};

template <typename T>
class RomFsFileSystemDeathTest : public RomFsFileSystemTest<T>
{
};

template <typename T>
class RomFsFileSystemWriteDeathTest : public RomFsFileSystemDeathTest<T>
{
};

typedef ::testing::Types<FsRomFsFileSystemTestTraits, FsSystemRomFsFileSystemTestTraits> RomFsFileSystems;
TYPED_TEST_CASE(RomFsFileSystemTest, RomFsFileSystems);
TYPED_TEST_CASE(RomFsFileSystemDeathTest, RomFsFileSystems);
TYPED_TEST_CASE(RomFsFileSystemWriteDeathTest, RomFsFileSystems);

//! 初期化関連の関数失敗をテスト
TYPED_TEST(RomFsFileSystemDeathTest, InitializeFailure)
{
    fs::MountHostRoot();
    NN_UTIL_SCOPE_EXIT
    {
        fs::UnmountHostRoot();
    };

    {
        fs::FileHandle baseHandle;
        NNT_ASSERT_RESULT_SUCCESS(fs::OpenFile(&baseHandle, g_ArchiveFilePath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(baseHandle);
        };

        fs::FileHandleStorage baseStorage(baseHandle);

        std::unique_ptr<TestFixture::RomFsFileSystem> romFs(new TestFixture::RomFsFileSystem());
        ASSERT_NE(romFs, nullptr);

        // 成功する GetRequiredWorkingMemorySize
        size_t cacheBufferSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            TestFixture::RomFsFileSystem::GetRequiredWorkingMemorySize(&cacheBufferSize, &baseStorage)
        );

        // メタデータキャッシュのサイズ不足
        std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
        ASSERT_NE(cacheBuffer, nullptr);
        if( NN_STATIC_CONDITION(TestFixture::IsDeathOnLackOfCacheBuffer) )
        {
            EXPECT_DEATH_IF_SUPPORTED(
                romFs->Initialize(&baseStorage, cacheBuffer.get(), cacheBufferSize - 1, true),
                ""
            );
        }
        else
        {
            NNT_EXPECT_RESULT_FAILURE(
                fs::ResultAllocationMemoryFailedInRomFsFileSystemA,
                romFs->Initialize(&baseStorage, cacheBuffer.get(), cacheBufferSize - 1, true)
            );
        }
    }
}

//! 初期化関連の事前条件をテスト
TYPED_TEST(RomFsFileSystemDeathTest, InitializePreCondition)
{
    fs::MountHostRoot();
    NN_UTIL_SCOPE_EXIT
    {
        fs::UnmountHostRoot();
    };

    {
        fs::FileHandle baseHandle;
        NNT_ASSERT_RESULT_SUCCESS(fs::OpenFile(&baseHandle, g_ArchiveFilePath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(baseHandle);
        };

        fs::FileHandleStorage baseStorage(baseHandle);

        std::unique_ptr<TestFixture::RomFsFileSystem> romFs(new TestFixture::RomFsFileSystem());
        ASSERT_NE(romFs, nullptr);

        // GetRequiredWorkingMemorySize の事前条件テスト
        size_t cacheBufferSize = 0;
        EXPECT_DEATH_IF_SUPPORTED(
            TestFixture::RomFsFileSystem::GetRequiredWorkingMemorySize(nullptr, &baseStorage),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            TestFixture::RomFsFileSystem::GetRequiredWorkingMemorySize(&cacheBufferSize, nullptr),
            ""
        );

        // 成功する GetRequiredWorkingMemorySize
        NNT_ASSERT_RESULT_SUCCESS(
            TestFixture::RomFsFileSystem::GetRequiredWorkingMemorySize(&cacheBufferSize, &baseStorage)
        );

        // Initialize の事前条件テスト
        std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
        ASSERT_NE(cacheBuffer, nullptr);
        EXPECT_DEATH_IF_SUPPORTED(
            romFs->Initialize(nullptr, cacheBuffer.get(), cacheBufferSize, true),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(romFs->Initialize(&baseStorage, nullptr, cacheBufferSize, true), "");
    }
}

//! アーカイブとアーカイブ元ディレクトリの内容が一致するかテスト
TYPED_TEST(RomFsFileSystemTest, VerifyWithSourceDirectory)
{
    NNT_ASSERT_RESULT_SUCCESS(fs::MountHostRoot());
    NN_UTIL_SCOPE_EXIT
    {
        fs::UnmountHostRoot();
    };

    fs::FileHandle baseHandle;
    NNT_ASSERT_RESULT_SUCCESS(fs::OpenFile(&baseHandle, g_ArchiveFilePath, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(baseHandle);
    };

    fs::FileHandleStorage baseStorage(baseHandle);

    size_t cacheBufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(
        TestFixture::RomFsFileSystem::GetRequiredWorkingMemorySize(&cacheBufferSize, &baseStorage)
    );
    auto cacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);
    NN_ASSERT_NOT_NULL(cacheBuffer);

    bool verifyResult = false;
    NNT_ASSERT_RESULT_SUCCESS(
        VerifyMount(
            &verifyResult,
            [&](const char* mountName) NN_NOEXCEPT -> nn::Result
            {
                std::unique_ptr<TestFixture::RomFsFileSystem> romFs(new TestFixture::RomFsFileSystem());
                NN_ASSERT_NOT_NULL(romFs);

                NN_RESULT_DO(
                    romFs->Initialize(&baseStorage, cacheBuffer.get(), cacheBufferSize, true)
                );

                NN_RESULT_DO(fs::fsa::Register(mountName, std::move(romFs)));

                NN_RESULT_SUCCESS;
            },
            g_SourceDirectoryPath,
            "rom",
            "rom:/"
        )
    );

    ASSERT_TRUE(verifyResult);
}

//! アーカイブをマウントして各ファイルに対して関数を実行
template<typename RomFsFileSystem>
void TestEachFile(
    const char* path,
    std::function<void (const char* path, const nn::fs::DirectoryEntry& entry)> findFile
) NN_NOEXCEPT
{
    fs::MountHostRoot();
    NN_UTIL_SCOPE_EXIT
    {
        fs::UnmountHostRoot();
    };

    {
        fs::FileHandle baseHandle;
        NNT_ASSERT_RESULT_SUCCESS(fs::OpenFile(&baseHandle, g_ArchiveFilePath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(baseHandle);
        };

        fs::FileHandleStorage baseStorage(baseHandle);

        std::unique_ptr<RomFsFileSystem> romFs(new RomFsFileSystem());
        ASSERT_NE(romFs, nullptr);

        size_t cacheBufferSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            RomFsFileSystem::GetRequiredWorkingMemorySize(&cacheBufferSize, &baseStorage)
        );
        std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
        ASSERT_NE(cacheBuffer, nullptr);
        NNT_ASSERT_RESULT_SUCCESS(
            romFs->Initialize(&baseStorage, cacheBuffer.get(), cacheBufferSize, true)
        );
        NNT_ASSERT_RESULT_SUCCESS(fs::fsa::Register("rom", std::move(romFs)));
        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount("rom");
        };

        NNT_ASSERT_RESULT_SUCCESS(
            IterateDirectoryRecursive(
                path, nullptr, nullptr,
                [&](const char* path, const nn::fs::DirectoryEntry& entry) NN_NOEXCEPT -> nn::Result
                {
                    findFile(path, entry);
                    if( ::testing::Test::HasFatalFailure() )
                    {
                        NN_RESULT_THROW(nn::fs::ResultInternal());
                    }
                    NN_RESULT_SUCCESS;
                }
            )
        );
    }
}

//! ファイルを書き込みモードで開こうとするとアボートすることをテスト
TYPED_TEST(RomFsFileSystemWriteDeathTest, OpenFileWriteMode)
{
    TestEachFile<TestFixture::RomFsFileSystem>(
        "rom:/",
        [](const char* path, const nn::fs::DirectoryEntry &entry) NN_NOEXCEPT
        {
            NN_UNUSED(entry);

            nn::fs::FileHandle file;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                nn::fs::OpenFile(
                    &file, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend
                ),
                ""
            );
        }
    );
}

//! ファイルへの書き込みがアボートすることをテスト
TYPED_TEST(RomFsFileSystemWriteDeathTest, WriteFile)
{
    TestEachFile<TestFixture::RomFsFileSystem>(
        "rom:/",
        [](const char* path, const nn::fs::DirectoryEntry &entry) NN_NOEXCEPT
        {
            NN_UNUSED(entry);

            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read)
            );
            NN_UTIL_SCOPE_EXIT
            {
                CloseFile(file);
            };

            // 書き込みできるサイズがあるか
            int64_t fileSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));
            char writeData = 123;
            ASSERT_GE(fileSize, static_cast<int64_t>(sizeof(writeData)));

            // 書き込み
            nn::fs::WriteOption writeOption = {};
            EXPECT_DEATH_IF_SUPPORTED(
                nn::fs::WriteFile(file, 0, &writeData, sizeof(writeData), writeOption),
                ""
            );
        }
    );
}

//! ファイルのフラッシュがアボートすることをテスト
TYPED_TEST(RomFsFileSystemWriteDeathTest, FlushFile)
{
    TestEachFile<TestFixture::RomFsFileSystem>(
        "rom:/",
        [](const char* path, const nn::fs::DirectoryEntry &entry) NN_NOEXCEPT
        {
            NN_UNUSED(entry);

            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read)
            );
            NN_UTIL_SCOPE_EXIT
            {
                CloseFile(file);
            };

            EXPECT_DEATH_IF_SUPPORTED(nn::fs::FlushFile(file), "");
        }
    );
}

//! ファイルのサイズ変更がアボートすることをテスト
TYPED_TEST(RomFsFileSystemWriteDeathTest, SetFileSize)
{
    TestEachFile<TestFixture::RomFsFileSystem>(
        "rom:/",
        [](const char* path, const nn::fs::DirectoryEntry &entry) NN_NOEXCEPT
        {
            NN_UNUSED(entry);

            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read)
            );
            NN_UTIL_SCOPE_EXIT
            {
                CloseFile(file);
            };

            // 元のサイズを取得
            int64_t fileSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));

            // サイズ変更
            EXPECT_DEATH_IF_SUPPORTED(nn::fs::SetFileSize(file, fileSize + 1), "");
        }
    );
}

//! 書き込みを伴う操作でアボートすることをテスト
TYPED_TEST(RomFsFileSystemWriteDeathTest, WriteOperation)
{
    fs::MountHostRoot();
    NN_UTIL_SCOPE_EXIT
    {
        fs::UnmountHostRoot();
    };

    {
        fs::FileHandle baseHandle;
        NNT_ASSERT_RESULT_SUCCESS(fs::OpenFile(&baseHandle, g_ArchiveFilePath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(baseHandle);
        };

        fs::FileHandleStorage baseStorage(baseHandle);

        std::unique_ptr<TestFixture::RomFsFileSystem> romFs(new TestFixture::RomFsFileSystem());
        ASSERT_NE(romFs, nullptr);

        size_t cacheBufferSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            TestFixture::RomFsFileSystem::GetRequiredWorkingMemorySize(&cacheBufferSize, &baseStorage)
        );
        std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
        ASSERT_NE(cacheBuffer, nullptr);
        NNT_ASSERT_RESULT_SUCCESS(
            romFs->Initialize(&baseStorage, cacheBuffer.get(), cacheBufferSize, true)
        );
        NNT_ASSERT_RESULT_SUCCESS(fs::fsa::Register("rom", std::move(romFs)));
        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount("rom");
        };

        EXPECT_DEATH_IF_SUPPORTED(nn::fs::CreateFile("rom:/file.dat", 100), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::fs::DeleteFile("rom:/file.dat"), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::fs::CreateDirectory("rom:/dir"), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::fs::DeleteDirectory("rom:/dir"), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::fs::DeleteDirectoryRecursively("rom:/dir"), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::fs::CleanDirectoryRecursively("rom:/dir"), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::fs::RenameFile("rom:/file1.dat", "rom:/file2.dat"), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::fs::RenameDirectory("rom:/dir1", "rom:/dir2"), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::fs::CommitSaveData("rom"), "");
    }
}

//! QueryRange をテスト
TYPED_TEST(RomFsFileSystemTest, QueryRange)
{
    nn::fs::SetEnabledAutoAbort(false);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::SetEnabledAutoAbort(true);
    };

    TestEachFile<TestFixture::RomFsFileSystem>(
        "rom:/",
        [](const char* path, const nn::fs::DirectoryEntry &entry) NN_NOEXCEPT
        {
            NN_UNUSED(entry);

            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read)
            );
            NN_UTIL_SCOPE_EXIT
            {
                CloseFile(file);
            };

            // 元のサイズを取得
            int64_t fileSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));

            nn::fs::QueryRangeInfo info;

            const auto result = nn::fs::QueryRange(nullptr, file, 0, fileSize);
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNullptrArgument, result);

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryRange(&info, file, 0, fileSize));
            EXPECT_EQ(0, info.aesCtrKeyTypeFlag);
            EXPECT_EQ(0, info.speedEmulationTypeFlag);
        }
    );
}

//! バッファ不足を返す Storage に対して適切にリトライをするかどうか確認
TEST(FsSystemRomFsFileSystem, RandomAllocationFailure)
{
    static const char* MountName = "rom";

    NNT_ASSERT_RESULT_SUCCESS(fs::MountHostRoot());
    NN_UTIL_SCOPE_EXIT
    {
        fs::UnmountHostRoot();
    };

    fs::FileHandle baseHandle;
    NNT_ASSERT_RESULT_SUCCESS(fs::OpenFile(&baseHandle, g_ArchiveFilePath, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(baseHandle);
    };

    fs::FileHandleStorage baseStorage(baseHandle);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    RandomAllocationFailureStorage randomFailureStorage(&baseStorage, &mt);

    auto doTest = [&](const char* name, int divider, std::function<void()> action) NN_NOEXCEPT
    {
        randomFailureStorage.SetDivider(divider);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(action());
        NN_LOG("Operation Count %s %d\n", name, randomFailureStorage.GetOperationCount());
        ASSERT_EQ(nullptr, fssystem::buffers::GetBufferManagerContext());
    };

    size_t cacheBufferSize = 0;
    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
        "GetRequiredWorkingMemorySize", 2,
        [&]() NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(
                fssystem::RomFsFileSystem::GetRequiredWorkingMemorySize(&cacheBufferSize, &randomFailureStorage)
            );
        }
    ));

    std::unique_ptr<fssystem::RomFsFileSystem> romFs(new fssystem::RomFsFileSystem());
    ASSERT_NE(nullptr, romFs.get());

    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
        "Mount", 2,
        [&]() NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(
                romFs->Initialize(&randomFailureStorage, nullptr, 0, false)
            );
        }
    ));

    // ファイル・ディレクトリ以外
    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
        "GetEntryType", 10,
        [&]() NN_NOEXCEPT
        {
            fs::DirectoryEntryType type;
            NNT_ASSERT_RESULT_SUCCESS(romFs->GetEntryType(&type, "/file0.dat"));
        }
    ));

    // ファイル読み込み
    {
        std::unique_ptr<fs::fsa::IFile> pFile;
        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
            "OpenFile", 10,
            [&]() NN_NOEXCEPT
            {
                NNT_ASSERT_RESULT_SUCCESS(romFs->OpenFile(&pFile, "/file0.dat", fs::OpenMode_Read));
            }
        ));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                "CloseFile", 0,
                [&]() NN_NOEXCEPT
                {
                    pFile.reset(nullptr);
                }
            ));
        };

        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
            "File::GetSize", 0,
            [&]() NN_NOEXCEPT
            {
                int64_t size;
                NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            }
        ));

        char inBuffer[32];
        char outBuffer[32];

        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
            "File::Read", 2,
            [&]() NN_NOEXCEPT
            {
                size_t size;
                NNT_ASSERT_RESULT_SUCCESS(pFile->Read(&size, 0, outBuffer, sizeof(outBuffer), fs::ReadOption::MakeValue(0)));
            }
        ));

        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
            "File::OperateRange", 2,
            [&]() NN_NOEXCEPT
            {
                NNT_ASSERT_RESULT_FAILURE(
                    fs::ResultUnsupportedOperation,
                    pFile->OperateRange(
                        outBuffer, sizeof(outBuffer),
                        fs::OperationId::Invalidate,
                        0, sizeof(inBuffer),
                        inBuffer, sizeof(inBuffer)
                    )
                );
            }
        ));
    }

    // ディレクトリ読み込み
    {
        std::unique_ptr<fs::fsa::IDirectory> pDirectory;
        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
            "OpenDirectory", 10,
            [&]() NN_NOEXCEPT
            {
                NNT_ASSERT_RESULT_SUCCESS(romFs->OpenDirectory(&pDirectory, "/", fs::OpenDirectoryMode_All));
            }
        ));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                "CloseDirectory", 0,
                [&]() NN_NOEXCEPT
                {
                    pDirectory.reset(nullptr);
                }
            ));
        };

        int64_t count;
        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
            "Diretory::GetEntryCount", 4,
            [&]() NN_NOEXCEPT
            {
                NNT_ASSERT_RESULT_SUCCESS(pDirectory->GetEntryCount(&count));
            }
        ));

        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
            "Directory::Read", 4,
            [&]() NN_NOEXCEPT
            {
                nnt::fs::util::Vector<fs::DirectoryEntry> pEntries(3);

                for( int i = 0; i < count; ++i )
                {
                    int64_t currentCount;
                    NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&currentCount, pEntries.data(), pEntries.size()));
                }
            }
        ));
    }
} // NOLINT(impl/function_size)

}

namespace{

void* Allocate(size_t size)
{
    return malloc(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    return free(p);
}

} // namespace

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);

    if( argc < 3 )
    {
        NN_LOG("Usage: *.exe <path of romfs archive file> <source directory path of archive file>\n");
        nnt::Exit(1);
    }

    g_ArchiveFilePath = argv[1];
    g_SourceDirectoryPath = argv[2];

    nn::fs::SetAllocator(::Allocate, ::Deallocate);

    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
