﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <numeric>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/utilTool/fs_RelocatedBinaryMatch.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageUtils.h>
#include <nn/util/util_TinyMt.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)

namespace {

template< typename T, int Size = sizeof(T) >
struct Random
{
    static T Get(nn::util::TinyMt* pMt) NN_NOEXCEPT
    {
        return static_cast<T>(pMt->GenerateRandomU32());
    }
};

template< typename T >
struct Random<T, sizeof(int64_t)>
{
    static T Get(nn::util::TinyMt* pMt) NN_NOEXCEPT
    {
        return static_cast<T>(pMt->GenerateRandomU64());
    }
};

template< typename T >
inline T GetRandomValue(nn::util::TinyMt* pMt, T min, T max) NN_NOEXCEPT
{
    NN_ASSERT_LESS_EQUAL(min, max);
    if( min == max )
    {
        return min;
    }

    auto range = max - min + 1;
    auto value = Random<T>::Get(pMt);

    return min + (value % range);
}


typedef nn::fssystem::utilTool::RelocationTable RelocationTable;

class RelocationTableBuilder
{
public:
    enum class Type
    {
        Segment,
        Repeat,
    };

public:
    // コンストラクタです。
    RelocationTableBuilder(
        int64_t binarySize,
        size_t segmentSizeMin,
        size_t segmentSizeMax,
        size_t blockSize,
        size_t regionSize
    ) NN_NOEXCEPT
        : m_BinarySize(binarySize)
        , m_SegmentSizeMin(segmentSizeMin)
        , m_SegmentSizeMax(segmentSizeMax)
        , m_BlockSize(blockSize)
        , m_RegionSize(regionSize)
        , m_SegmentCountVariation(10)
        , m_NewSegmentPreference(30)
        , m_OldStorage(blockSize)
        , m_NewStorage(blockSize)
        , m_RelocationTable()
    {
    }

    // 初期化をします。
    nn::Result Initialize(Type type) NN_NOEXCEPT
    {
        if (type == Type::Segment)
        {
            NN_RESULT_DO(
                CreateFiles(
                    [](char* pData, size_t dataSize, int seed) NN_NOEXCEPT
                    {
                        nn::util::TinyMt mt;
                        mt.Initialize(seed);
                        for( size_t index = 0; index < dataSize; ++index )
                        {
                            pData[index] = static_cast<char>(mt.GenerateRandomU32());
                        }
                    }
                    ));
        }
        else if(type == Type::Repeat)
        {
            bool firstSeedInited = false;
            int firstSeed = 0;
            NN_RESULT_DO(
                CreateFiles(
                    [&firstSeedInited, &firstSeed](char* pData, size_t dataSize, int seed) NN_NOEXCEPT
                    {
                        // 繰り返しデータを得るため固定のシードを使い続ける
                        if( firstSeedInited == false )
                        {
                            firstSeedInited = true;
                            firstSeed = seed;
                        }

                        nn::util::TinyMt mt;
                        mt.Initialize(firstSeed);
                        for( size_t index = 0; index < dataSize; ++index )
                        {
                            pData[index] = static_cast<char>(mt.GenerateRandomU32());
                        }
                    }
                    ));
        }
        else
        {
            NN_ASSERT(false);
        }

        return m_NewStorage.GetSize(&m_BinarySize);
    }

    // テーブルを作成します。
    nn::Result Build(int64_t oldStorageOffset, int64_t outputStorageOffset) NN_NOEXCEPT
    {
        RelocationTable::BuildInfo info(
            oldStorageOffset, outputStorageOffset, m_BlockSize, m_BlockSize, m_RegionSize, 0);

        NN_RESULT_DO(info.Initialize(
            nn::fs::SubStorage(&m_OldStorage, 0, m_OldStorage.GetSize()),
            nn::fs::SubStorage(&m_NewStorage, 0, m_NewStorage.GetSize())
        ));

        std::shared_ptr<nn::fssystem::utilTool::BinaryMatchProgress> pProgress;
        return m_RelocationTable.Build(&pProgress, info);
    }

    // 古いデータ用のストレージを取得します。
    nnt::fs::util::SafeMemoryStorage& GetOldStorage() NN_NOEXCEPT
    {
        return m_OldStorage;
    }

    // 新しいデータ用のストレージを取得します。
    nnt::fs::util::SafeMemoryStorage& GetNewStorage() NN_NOEXCEPT
    {
        return m_NewStorage;
    }

    // テーブルを取得します。
    RelocationTable&  GetTable() NN_NOEXCEPT
    {
        return m_RelocationTable;
    }

    // テーブルを取得します。
    const RelocationTable& GetTable() const NN_NOEXCEPT
    {
        return m_RelocationTable;
    }

private:
    struct Segment
    {
        int64_t offset;
        size_t size;
        int seed;
    };

    struct Match
    {
        int64_t oldOffset;
        int64_t newOffset;
        size_t size;
        int seed;
    };

    class SeedGenerator
    {
    public:
        explicit SeedGenerator(nn::util::TinyMt* pRandom) NN_NOEXCEPT
            : m_Random(*pRandom)
            , m_Seeds()
        {
        }

        int Generate() NN_NOEXCEPT
        {
            for( ; ; )
            {
                int seed = static_cast<int>(m_Random.GenerateRandomU32());
                auto it = std::find(m_Seeds.begin(), m_Seeds.end(), seed);
                if( it == m_Seeds.end() )
                {
                    m_Seeds.push_back(seed);
                    return seed;
                }
            }
        }

    private:
        nn::util::TinyMt& m_Random;
        nnt::fs::util::Vector<int> m_Seeds;
    };

private:
    template< typename Functor >
    nn::Result CreateFiles(Functor generate) NN_NOEXCEPT
    {
        nn::util::TinyMt mt;
        mt.Initialize(nnt::fs::util::GetRandomSeed());

        SeedGenerator seeds(&mt);
        nnt::fs::util::Vector<char> buffer(m_SegmentSizeMax);
        nnt::fs::util::Vector<Segment> segments;
        segments.reserve(static_cast<size_t>(m_BinarySize / m_SegmentSizeMin));

        // 古いデータ用のストレージを作成
        {
            int64_t binarySize = 0;
            while( binarySize < m_BinarySize )
            {
                Segment segment =
                {
                    binarySize,
                    nn::util::align_down(
                        GetRandomValue(&mt, m_SegmentSizeMin, m_SegmentSizeMax),
                        m_BlockSize
                    ),
                    seeds.Generate()
                };

                segments.push_back(segment);

                binarySize += segment.size;
            }

            m_OldStorage.Initialize(binarySize);

            int64_t offset = 0;
            for( const auto& segment : segments )
            {
                generate(buffer.data(), segment.size, segment.seed);

                NN_RESULT_DO(m_OldStorage.Write(offset, buffer.data(), segment.size));

                offset += segment.size;
            }
        }

        // 新しいデータ用のファイルを作成
        {
            const auto segmentCountMax =
                GetRandomValue<size_t>(
                    &mt,
                    segments.size() * (100 - m_SegmentCountVariation) / 100,
                    segments.size() * (100 + m_SegmentCountVariation) / 100
                );

            nnt::fs::util::Vector<Match> matches;
            matches.reserve(segmentCountMax);

            size_t segmentIndex = 0;
            size_t binarySize = 0;

            while( matches.size() < segmentCountMax )
            {
                const auto segmentSelect =
                    GetRandomValue<size_t>(
                        &mt, 0, segments.size() * (100 + m_NewSegmentPreference) / 100
                    );

                Match match;

                if( segmentSelect < segments.size() && segmentIndex < segments.size() )
                {
                    match.oldOffset = segments[segmentIndex].offset;
                    match.newOffset = binarySize;
                    match.size = segments[segmentIndex].size;
                    match.seed = segments[segmentIndex].seed;
                }
                else
                {
                    match.oldOffset = std::numeric_limits<int64_t>::min();
                    match.newOffset = binarySize;
                    match.size =
                        nn::util::align_down(
                            GetRandomValue(&mt, m_SegmentSizeMin, m_SegmentSizeMax),
                            m_BlockSize
                        );
                    match.seed = seeds.Generate();
                }

                matches.push_back(match);
                binarySize += match.size;

                ++segmentIndex;
            }

            m_NewStorage.Initialize(binarySize);

            int64_t offset = 0;
            for( const auto& match : matches )
            {
                generate(buffer.data(), match.size, match.seed);

                NN_RESULT_DO(m_NewStorage.Write(offset, buffer.data(), match.size));

                offset += match.size;
            }
        }

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

private:
    int64_t m_BinarySize;
    size_t m_SegmentSizeMin;
    size_t m_SegmentSizeMax;
    size_t m_BlockSize;
    size_t m_RegionSize;
    int m_SegmentCountVariation;
    int m_NewSegmentPreference;
    nnt::fs::util::SafeMemoryStorage m_OldStorage;
    nnt::fs::util::SafeMemoryStorage m_NewStorage;
    RelocationTable m_RelocationTable;
};

}

#if !defined(NN_SDK_BUILD_RELEASE)
// RelocationTable の事前条件をテストします。
TEST(RelocationTableDeathTest, Precondition)
{
    static const size_t BlockSize = 16;
    nnt::fs::util::SafeMemoryStorage safeMemStorage(BlockSize);
    nn::fs::SubStorage storage(&safeMemStorage, 0, safeMemStorage.GetSize());

    // oldOffset
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(-1, 0, 16, 16, 32, 0), "");

    // newOffset
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(0, -1, 16, 16, 32, 0), "");

    // alignment
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(0, 0, 0, 16, 32, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(0, 0, 3, 16, 32, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(0, 0, 32, 16, 32, 0), "");

    // blockSize
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(0, 0, 4, 4, 16, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(0, 0, 8, 9, 16, 0), "");

    // regionSize
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(0, 0, 16, 16, 16, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(0, 0, 16, 16, 33, 0), "");

    // matchSize
    EXPECT_DEATH_IF_SUPPORTED(RelocationTable::BuildInfo(0, 0, 16, 16, 32, -1), "");
}
#endif

namespace {

void VerifyTableOffset(const RelocationTable* pTable, int64_t newStorageSize) NN_NOEXCEPT
{
    nnt::fs::util::Vector<RelocationTable::Entry> tableArray(pTable->size());
    std::copy(pTable->begin(), pTable->end(), tableArray.begin());

    std::sort(
        tableArray.begin(),
        tableArray.end(),
        [](const RelocationTable::Entry& lhs, const RelocationTable::Entry& rhs) NN_NOEXCEPT
        {
            return lhs.GetSrcOffset() < rhs.GetSrcOffset();
        }
        );

    // 移動元を網羅しているかチェック
    {
        int64_t offset = 0;
        for( const auto& data : tableArray )
        {
            ASSERT_EQ(offset, data.GetSrcOffset());
            offset += data.GetSize();
        }
        ASSERT_EQ(offset, newStorageSize);
    }
}

void VerifyTableData(
    const RelocationTable* pTable,
    nn::fs::IStorage* pOldStorage,
    nn::fs::IStorage* pNewStorage,
    int64_t oldStorageOffset,
    int64_t newStorageOffset,
    int64_t range) NN_NOEXCEPT
{
    static const int64_t BufferSizeMax = 4 * 1024 * 1024;

    nnt::fs::util::Vector<char> oldBuffer;
    nnt::fs::util::Vector<char> newBuffer;

    // 新しいストレージから古いストレージのオフセットに調整する値
    const auto adjustOffset = newStorageOffset - oldStorageOffset;

    // データが一致しているかチェック
    for( const auto& data : *pTable )
    {
        if( data.IsMatched() )
        {
            ASSERT_LE(data.GetDstOffset() + adjustOffset + data.GetSize(), range);

            const size_t size = static_cast<size_t>(std::min(data.GetSize(), BufferSizeMax));

            oldBuffer.resize(size);
            newBuffer.resize(size);

            for( int64_t accessedSize = 0; accessedSize < data.GetSize(); accessedSize += size )
            {
                const size_t accessSize = static_cast<size_t>(std::min(
                    static_cast<int64_t>(size), data.GetSize() - accessedSize));
                NNT_ASSERT_RESULT_SUCCESS(pOldStorage->Read(
                    data.GetDstOffset() + adjustOffset + accessedSize, oldBuffer.data(), accessSize));
                NNT_ASSERT_RESULT_SUCCESS(pNewStorage->Read(
                    data.GetSrcOffset() + accessedSize, newBuffer.data(), accessSize));

                NNT_FS_UTIL_ASSERT_MEMCMPEQ(oldBuffer.data(), newBuffer.data(), accessSize);
            }
        }
    }
}

void MakeRelocationTable(
         nn::util::TinyMt* pRandom,
         int64_t oldStorageOffset,
         int64_t newStorageOffset,
         RelocationTableBuilder::Type builderType
     ) NN_NOEXCEPT
{
    static const size_t BlockSize = 16;

    const int64_t binarySize = (512 * 1024) << pRandom->GenerateRandomN(6);
    const size_t segmentSizeMin = 256 << pRandom->GenerateRandomN(4);
    const size_t segmentSizeMax = segmentSizeMin << (1 + pRandom->GenerateRandomN(3));
    const size_t reagionSize = (2 * 1024) << pRandom->GenerateRandomN(4);

    RelocationTableBuilder builder(
        binarySize, segmentSizeMin, segmentSizeMax, BlockSize, reagionSize);

    NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(builderType));
    NNT_ASSERT_RESULT_SUCCESS(builder.Build(oldStorageOffset, newStorageOffset));

    const auto newStorageSize = builder.GetNewStorage().GetSize();
    const auto& table = builder.GetTable();

    // 移動元を網羅しているかチェック
    VerifyTableOffset(&table, newStorageSize);

    // データが一致しているかチェック
    auto& oldStorage = builder.GetOldStorage();
    auto& newStorage = builder.GetNewStorage();
    const auto range = builder.GetOldStorage().GetSize();
    VerifyTableData(&table, &oldStorage, &newStorage, oldStorageOffset, newStorageOffset, range);
}

void MakeRelocationTableBasicTest(RelocationTableBuilder::Type builderType) NN_NOEXCEPT
{
    nn::util::TinyMt random;
    random.Initialize(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 10; ++i )
    {
        MakeRelocationTable(&random, 0, 0, builderType);
    }

    for( int i = 0; i < 10; ++i )
    {
        const int gap = nn::util::align_up(1 + random.GenerateRandomN(2 * 1024), 16);

        MakeRelocationTable(&random, gap, 0, builderType);
    }

    for( int i = 0; i < 10; ++i )
    {
        const int gap = nn::util::align_up(1 + random.GenerateRandomN(2 * 1024), 16);

        MakeRelocationTable(&random, 0, gap, builderType);
    }

    // 完全にオーバーラップしない場合
    {
        int64_t tooLargeGap = 16LL * 1024 * 1024 * 1024 * 1024; // 16TB
        MakeRelocationTable(&random, 0, tooLargeGap, builderType);
        MakeRelocationTable(&random, tooLargeGap, 0, builderType);
    }
}

}

// 正常にテーブルが作成できるかテストします。
TEST(RelocationTable, MakeRelocationTableSegment)
{
    MakeRelocationTableBasicTest(RelocationTableBuilder::Type::Segment);
}

TEST(RelocationTable, MakeRelocationTableRepeat)
{
    MakeRelocationTableBasicTest(RelocationTableBuilder::Type::Repeat);
}

TEST(RelocationTable, MakeRelocationExcludeBinaryMatchResultOverlap)
{
    typedef nn::fssystem::utilTool::BinaryMatch::Result MatchResult;

    const MatchResult matchResultsArray[] =
    {
        {0, 0, 10, 0},
        {10, 10, 10, 0},
        {20, 20, 10, 0},
        {30, -1, 10, 1}
    };
    std::vector<MatchResult> matchResults(
        matchResultsArray,
        matchResultsArray + sizeof(matchResultsArray) / sizeof(matchResultsArray[0]));

    const MatchResult expectedResultsArray[] =
    {
        {0, 0, 10, 0},
        {10, 10, 10, 0},
        {20, 20, 10, 0},
        {30, -1, 10, 1}
    };
    std::vector<MatchResult> expectedResults(
        expectedResultsArray,
        expectedResultsArray + sizeof(expectedResultsArray) / sizeof(expectedResultsArray[0]));

    auto excluded = matchResults;
    nn::fssystem::utilTool::OptimizeBinaryMatchResult(&excluded);
    std::sort(excluded.begin(), excluded.end());

    auto isSameResults = []
        (const std::vector<MatchResult>& lhs, const std::vector<MatchResult>& rhs) NN_NOEXCEPT
        -> bool
    {
        if (lhs.size() != rhs.size()) return false;

        for (int i=0; i < static_cast<int>(lhs.size()); ++i)
        {
            if (lhs[i].oldOffset != rhs[i].oldOffset ||
                lhs[i].newOffset != lhs[i].newOffset ||
                lhs[i].size != lhs[i].size)
            {
                return false;
            }
        }
        return true;
    };

    EXPECT_TRUE(isSameResults(excluded, expectedResults));
}

// 4 GB を超えるオフセットの動作をテストします。
TEST(RelocationTable, MakeRelocationTableSegmentLargeHeavy)
{
    static const int64_t BlockSize = 16;
    static const int64_t RegionSize = 16 * 1024;

    static const int64_t StorageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024;
    nnt::fs::util::VirtualMemoryStorage oldStorage;
    oldStorage.Initialize(StorageSize);
    nnt::fs::util::VirtualMemoryStorage newStorage;
    newStorage.Initialize(StorageSize);

    static const size_t WriteSize = 1024;
    auto buffer = nnt::fs::util::AllocateBuffer(WriteSize);
    {
        const int64_t offset = 0;

        nnt::fs::util::FillBufferWithRandomValue(buffer.get(), WriteSize);
        NNT_ASSERT_RESULT_SUCCESS(oldStorage.Write(offset, buffer.get(), WriteSize));

        nnt::fs::util::FillBufferWithRandomValue(buffer.get(), WriteSize);
        NNT_ASSERT_RESULT_SUCCESS(newStorage.Write(offset, buffer.get(), WriteSize));
    }

    {
        const int64_t offset = static_cast<int64_t>(4) * 1024 * 1024 * 1024;

        nnt::fs::util::FillBufferWith32BitCount(buffer.get(), WriteSize, 0);
        NNT_ASSERT_RESULT_SUCCESS(oldStorage.Write(offset, buffer.get(), WriteSize));
        NNT_ASSERT_RESULT_SUCCESS(newStorage.Write(offset, buffer.get(), WriteSize));
    }

    {
        const int64_t offset = static_cast<int64_t>(8) * 1024 * 1024 * 1024;

        nnt::fs::util::FillBufferWithRandomValue(buffer.get(), WriteSize);
        NNT_ASSERT_RESULT_SUCCESS(oldStorage.Write(offset, buffer.get(), WriteSize));

        nnt::fs::util::FillBufferWithRandomValue(buffer.get(), WriteSize);
        NNT_ASSERT_RESULT_SUCCESS(newStorage.Write(offset, buffer.get(), WriteSize));
    }

    RelocationTable::BuildInfo info(0, 0, BlockSize, BlockSize, RegionSize, 0);

    NNT_ASSERT_RESULT_SUCCESS(info.Initialize(
        nn::fs::SubStorage(&oldStorage, 0, StorageSize),
        nn::fs::SubStorage(&newStorage, 0, StorageSize)));

    RelocationTable relocationTable;
    {
        std::shared_ptr<nn::fssystem::utilTool::BinaryMatchProgress> pProgress;
        NNT_ASSERT_RESULT_SUCCESS(relocationTable.Build(&pProgress, info));
    }

    // 移動元を網羅しているかチェック
    VerifyTableOffset(&relocationTable, StorageSize);

    // データが一致しているかチェック
    VerifyTableData(&relocationTable, &oldStorage, &newStorage, 0, 0, StorageSize);
}

#endif
