﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>
#include <nn/fssystem/utilTool/fs_BlockSignatureContainer.h>
#include <nn/fs/fs_FileStorage.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

inline size_t QueryUnitBlockSize(size_t order, size_t pageSize) NN_NOEXCEPT
{
    return (static_cast<size_t>(1) << (order - 1)) * pageSize;
}

inline int GetSizeExp(size_t blockSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(nn::util::ispow2(blockSize));
    return nn::util::cntt0(blockSize);
}

inline size_t GenerateRandomBlockSize(size_t blockSizeMin, size_t blockSizeMax) NN_NOEXCEPT
{
    static std::mt19937 s_Mt(nnt::fs::util::GetRandomSeed());
    int min = GetSizeExp(blockSizeMin);
    int max = GetSizeExp(blockSizeMax);
    int random = std::uniform_int_distribution<>(0, max - min)(s_Mt) + min;
    return static_cast<size_t>(1) << random;
}

const int LayerCountMax = nn::fssystem::save::IntegrityMaxLayerCount;

typedef nn::fssystem::utilTool::BlockSignatureContainer BlockSignatureContainer;

class TestStorageCreatorBase
{
public:
    typedef nn::fssystem::save::HierarchicalIntegrityVerificationStorage Storage;

public:
    // コンストラクタです。
    TestStorageCreatorBase() NN_NOEXCEPT
        : m_Lock(true)
    {
    }

    // デストラクタです。
    virtual ~TestStorageCreatorBase() NN_NOEXCEPT
    {
        m_StorageIntegrity.Finalize();
        m_CacheBuffer.Finalize();
    }

    // 初期化します。
    nn::Result Initialize(int layerCount, int64_t storageSize) NN_NOEXCEPT
    {
        static const int CacheCountMax = 1024;
        static const size_t CacheBlockSize = 4 * 1024;
        static const size_t CacheBufferMaxOrder = 12;

        const size_t unitBlockSize = QueryUnitBlockSize(CacheBufferMaxOrder, CacheBlockSize);
        const size_t cacheBufferSize = unitBlockSize * layerCount;

        m_Buffer.resize(cacheBufferSize);
        NN_RESULT_DO(
            m_CacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(m_Buffer.data()),
                m_Buffer.size(),
                CacheBlockSize
            )
        );

        for( int i = 0; i < layerCount; ++i )
        {
            m_CacheBufferSet.pBuffer[i] = &m_CacheBuffer;
        }

        StorageControlArea::InputParam inputParamIntegrity;
        for( int i = 0; i < layerCount - 1; ++i )
        {
            inputParamIntegrity.sizeBlockLevel[i] = GenerateRandomBlockSize(GetBlockSizeMin(), GetBlockSizeMax());
        }

        nn::fssystem::save::HierarchicalIntegrityVerificationSizeSet sizes;
        NN_RESULT_DO(
            StorageControlArea::QuerySize(
                &sizes,
                inputParamIntegrity,
                layerCount,
                storageSize
            )
        );

        int64_t bodySize = storageSize;
        for( int i = 0; i < layerCount - 2; ++i )
        {
            bodySize += sizes.layeredHashSizes[i];
        }

        m_BaseStorageControlArea.Initialize(sizes.controlSize);
        m_BaseStorageMasterHash.Initialize(sizes.masterHashSize);
        InitializeDataStorage(bodySize);

        nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation meta = {};
        meta.Format();
        meta.sizeMasterHash = static_cast<uint32_t>(sizes.masterHashSize);
        meta.infoLevelHash.maxLayers = layerCount;

        int64_t offset = 0;
        for( int level = 0; level < layerCount - 2; ++level )
        {
            const int64_t size = sizes.layeredHashSizes[level];
            meta.infoLevelHash.info[level].size = size;
            meta.infoLevelHash.info[level].offset = offset;
            meta.infoLevelHash.info[level].orderBlock =
                nn::fssystem::save::ILog2(
                    static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[level]));
            offset += size;
        }
        meta.infoLevelHash.info[layerCount - 2].size = storageSize;
        meta.infoLevelHash.info[layerCount - 2].offset = offset;
        meta.infoLevelHash.info[layerCount - 2].orderBlock =
            nn::fssystem::save::ILog2(
                static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[layerCount - 2]));

        nn::fs::SubStorage controlAreaStorage(
            &m_BaseStorageControlArea, 0, sizes.controlSize);
        NN_RESULT_DO(StorageControlArea::Format(controlAreaStorage, meta));

        m_BaseStorageMasterHash.OperateRange(
            nn::fs::OperationId::FillZero, 0, sizes.masterHashSize);

        StorageControlArea controlArea;
        NN_RESULT_DO(controlArea.Initialize(controlAreaStorage));

        controlArea.GetLevelHashInfo(&m_LayerInfo);

        nn::fs::SubStorage storageMasterHash(
            &m_BaseStorageMasterHash, 0, controlArea.GetMasterHashSize());
        nn::fs::SubStorage storageData(
            GetDataStorage(), m_LayerInfo.GetDataOffset(), m_LayerInfo.GetDataSize());
        nn::fs::SubStorage storageLayerHash(
            GetDataStorage(), 0, m_LayerInfo.GetLayeredHashSize());

        Storage::HierarchicalStorageInformation storageInfo;
        storageInfo.SetMasterHashStorage(storageMasterHash);
        storageInfo.SetDataStorage(storageData);

        for( int i = 0; i < layerCount - 2; ++i )
        {
            storageInfo[i + 1] = nn::fs::SubStorage(
                &storageLayerHash, m_LayerInfo.info[i].offset, m_LayerInfo.info[i].size);
        }

        NN_RESULT_DO(
            m_StorageIntegrity.Initialize(
                m_LayerInfo,
                storageInfo,
                &m_CacheBufferSet,
                &m_Lock,
                nn::fs::StorageType::StorageType_SaveData
            )
        );

        controlArea.Finalize();

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    // 生成したストレージを取得します。
    Storage& Get() NN_NOEXCEPT
    {
        return m_StorageIntegrity;
    }

    // ハッシュ・データ部分のストレージを取得します。
    virtual nn::fs::IStorage* GetDataStorage() NN_NOEXCEPT = 0;

    virtual void InitializeDataStorage(int64_t size) NN_NOEXCEPT = 0;

    // レイヤー情報を取得します。
    const BlockSignatureContainer::SaveDataLayerInfo& GetLayerInfo() NN_NOEXCEPT
    {
        return m_LayerInfo;
    }

protected:
    virtual size_t GetBlockSizeMin() const NN_NOEXCEPT = 0;
    virtual size_t GetBlockSizeMax() const NN_NOEXCEPT = 0;

private:
    typedef nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea
        StorageControlArea;

    typedef nn::fssystem::save::IntegrityVerificationStorage LayerStorage;

private:
    nn::os::Mutex m_Lock;
    nnt::fs::util::Vector<char> m_Buffer;
    nn::fssystem::FileSystemBufferManager m_CacheBuffer;
    nn::fssystem::save::FilesystemBufferManagerSet m_CacheBufferSet;
    nn::fssystem::save::HierarchicalIntegrityVerificationInformation m_LayerInfo;
    nnt::fs::util::SafeMemoryStorage m_BaseStorageControlArea;
    nnt::fs::util::SafeMemoryStorage m_BaseStorageMasterHash;
    nn::fssystem::save::HierarchicalIntegrityVerificationStorage m_StorageIntegrity;
};

class TestStorageCreator : public TestStorageCreatorBase
{
public:
    virtual nn::fs::IStorage* GetDataStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_BaseStorageDataBody;
    }

    virtual void InitializeDataStorage(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        m_BaseStorageDataBody.Initialize(size);
    }

protected:
    virtual size_t GetBlockSizeMin() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 256;
    }

    virtual size_t GetBlockSizeMax() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 32 * 1024;
    }

private:
    nnt::fs::util::SafeMemoryStorage m_BaseStorageDataBody;
};

class TestLargeStorageCreator : public TestStorageCreatorBase
{
public:
    virtual ~TestLargeStorageCreator() NN_NOEXCEPT NN_OVERRIDE
    {
        m_DataStorage.Finalize();
    }

    virtual nn::fs::IStorage* GetDataStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_DataStorage;
    }

    virtual void InitializeDataStorage(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        m_DataStorage.Initialize(size);
    }

protected:
    virtual size_t GetBlockSizeMin() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 1024;
    }

    virtual size_t GetBlockSizeMax() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 32 * 1024;
    }

private:
    nnt::fs::util::VirtualMemoryStorage m_DataStorage;
};

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 *  @brief  事前条件をテストします。
 */
TEST(BlockSignatureContainerDeathTest, Precondition)
{
    // Initialize() のチェック
    {
        BlockSignatureContainer::SaveDataLayerInfo info = {};
        info.maxLayers = 2;
        EXPECT_DEATH_IF_SUPPORTED(BlockSignatureContainer().Initialize(info), "");
    }
    {
        BlockSignatureContainer::SaveDataLayerInfo info = {};
        info.maxLayers = 8;
        EXPECT_DEATH_IF_SUPPORTED(BlockSignatureContainer().Initialize(info), "");
    }

    // Collect() のチェック
    EXPECT_DEATH_IF_SUPPORTED(BlockSignatureContainer().Collect(-1, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(BlockSignatureContainer().Collect(0, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(BlockSignatureContainer().Collect(0, 0), "");

    // Optimize() のチェック
    EXPECT_DEATH_IF_SUPPORTED(BlockSignatureContainer().Optimize(), "");

    // GetBlock() のチェック
    EXPECT_DEATH_IF_SUPPORTED(BlockSignatureContainer().GetBlock(), "");

    // Optimize() 後の Collect() のチェック
    {
        BlockSignatureContainer::SaveDataLayerInfo info = {};
        info.maxLayers = 7;

        BlockSignatureContainer container;
        container.Initialize(info);
        container.Optimize();

        EXPECT_DEATH_IF_SUPPORTED(container.Collect(0, 0), "");
    }
}
#endif

/**
 *  @brief  収集したデータが適切かテストします。
 */
TEST(BlockSignatureContainerTest, Collection)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int layerCount = 3; layerCount <= LayerCountMax; ++layerCount )
    {
        for( int i = 0; i < 20; ++i )
        {
            const auto storageSize = nn::util::align_up(
                std::uniform_int_distribution<size_t>(32 * 1024, 1024 * 1024)(mt),
                16
            );

            // ストレージ準備（GetLayerInfo() のため）
            TestStorageCreator storage;
            NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(layerCount, storageSize));

            // コンテナ準備
            BlockSignatureContainer container;
            container.Initialize(storage.GetLayerInfo());

            // ハッシュ領域の収集
            for( int j = 0, count = std::uniform_int_distribution<>(1, 20)(mt); j < count; ++j )
            {
                const auto offset = nn::util::align_up(
                    std::uniform_int_distribution<int64_t>(0, storageSize)(mt),
                    16
                );
                const auto size = nn::util::align_up(
                    std::uniform_int_distribution<int64_t>(0, 4 * 1024)(mt),
                    16
                );

                container.Collect(offset, size);
            }
            container.Optimize();

            const auto& blocks = container.GetBlock();

            if( blocks.empty() )
            {
                continue;
            }

            ASSERT_GT(blocks.front().size, 0);
            ASSERT_GE(blocks.front().layerIndex, 0);
            ASSERT_LE(blocks.front().layerIndex, layerCount - 2);

            // ブロックの正当性チェック
            for( auto pos = blocks.begin() + 1, end = blocks.end(); pos != end; ++pos )
            {
                const auto& block1 = *(pos - 1);
                const auto& block2 = *pos;

                ASSERT_GT(block2.size, 0);
                ASSERT_GE(block2.layerIndex, 0);
                ASSERT_LE(block2.layerIndex, layerCount - 2);

                ASSERT_LT(block1.position, block2.position);
                ASSERT_LE(block1.layerIndex, block2.layerIndex);

                if( block1.layerIndex == block2.layerIndex )
                {
                    ASSERT_LT(block1.layerOffset, block2.layerOffset);
                    ASSERT_LT(block1.layerOffset + block1.size, block2.layerOffset);
                    ASSERT_LT(block1.position + block1.size, block2.position);
                }
                else
                {
                    ASSERT_LE(block1.position + block1.size, block2.position);
                }
            }
        }
    }
}

/**
 *  @brief  収集したデータが適切な領域を指しているかテストします。
 */
TEST(BlockSignatureContainerTest, CompareData)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int layerCount = 3; layerCount <= LayerCountMax; ++layerCount )
    {
        for( int i = 0; i < 20; ++i )
        {
            const auto storageSize = nn::util::align_up(
                std::uniform_int_distribution<size_t>(256 * 1024, 4 * 1024 * 1024)(mt),
                16
            );

            TestStorageCreator storage;
            NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(layerCount, storageSize));

            // テストデータの書き出し
            {
                nnt::fs::util::Vector<char> buffer(storageSize);
                {
                    size_t value = 0;
                    std::generate(
                        buffer.begin(),
                        buffer.end(),
                        [&]() NN_NOEXCEPT
                        {
                            ++value;
                            return static_cast<char>(value);
                        }
                    );
                }

                NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, buffer.data(), buffer.size()));
                NNT_ASSERT_RESULT_SUCCESS(storage.Get().Commit());
            }

            // コンテナ準備
            BlockSignatureContainer container;
            container.Initialize(storage.GetLayerInfo());

            // ハッシュ領域の収集
            for( int j = 0, count = std::uniform_int_distribution<>(1, 10)(mt); j < count; ++j )
            {
                const auto offset = nn::util::align_up(
                    std::uniform_int_distribution<int64_t>(0, storageSize)(mt),
                    16
                );
                const auto size = nn::util::align_up(
                    std::uniform_int_distribution<int64_t>(0, 4 * 1024)(mt),
                    16
                );

                container.Collect(offset, size);
            }
            container.Optimize();

            const auto& blocks = container.GetBlock();
            if( blocks.empty() )
            {
                continue;
            }

            auto pDataStorage = storage.GetDataStorage();

            // 書き換え前のデータを取得
            int64_t dataStorageSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(pDataStorage->GetSize(&dataStorageSize));
            nnt::fs::util::Vector<char> originalData(static_cast<size_t>(dataStorageSize));
            NNT_ASSERT_RESULT_SUCCESS(
                pDataStorage->Read(0, originalData.data(), originalData.size()));

            const auto end = blocks.end();
            {
                // データ領域先頭のイテレータ取得
                auto iter = std::partition_point(
                    blocks.begin(),
                    end,
                    [=](const BlockSignatureContainer::Block& block) NN_NOEXCEPT
                    {
                        return block.layerIndex < layerCount - 2;
                    }
                );
                ASSERT_NE(iter, end);

                // データ領域の書き換え
                nnt::fs::util::Vector<char> zero;
                for( ; iter != end; ++iter )
                {
                    const auto& block = *iter;
                    ASSERT_EQ(block.layerIndex, layerCount - 2);

                    zero.resize(static_cast<size_t>(block.size), 0);

                    NNT_ASSERT_RESULT_SUCCESS(
                        storage.Get().Write(block.layerOffset, zero.data(), zero.size()));
                }
                NNT_ASSERT_RESULT_SUCCESS(storage.Get().Commit());
            }

            // 書き換え後のデータを取得
            nnt::fs::util::Vector<char> modifiedData(originalData.size());
            NNT_ASSERT_RESULT_SUCCESS(
                pDataStorage->Read(0, modifiedData.data(), modifiedData.size()));

            size_t position = 0;

            // 書き換え前後のデータを比較
            for( const auto& block : blocks )
            {
                const auto blockPos = static_cast<size_t>(block.position);
                const auto blockSize = static_cast<size_t>(block.size);

                if( position < blockPos )
                {
                    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                        &originalData[position],
                        &modifiedData[position],
                        blockPos - position
                    );
                }

                ASSERT_FALSE(
                    std::equal(
                        &originalData[blockPos],
                        &originalData[blockPos] + blockSize,
                        &modifiedData[blockPos]
                    )
                );

                position = blockPos + blockSize;
            }
            if( position < originalData.size() )
            {
                NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                    &originalData[position],
                    &modifiedData[position],
                    originalData.size() - position
                );
            }
        }
    }
} // NOLINT(impl/function_size)

/**
 *  @brief  4 GB を超えるストレージからデータを収集します。
 */
TEST(BlockSignatureContainerTest, CollectionLargeSize)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int layerCount = 3; layerCount <= LayerCountMax; ++layerCount )
    {
        for( int i = 0; i < 20; ++i )
        {
            const int64_t storageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + 256;

            // ストレージ準備（GetLayerInfo() のため）
            TestLargeStorageCreator storage;
            NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(layerCount, storageSize));

            // コンテナ準備
            BlockSignatureContainer container;
            container.Initialize(storage.GetLayerInfo());

            // ハッシュ領域の収集
            const int count = std::uniform_int_distribution<>(1, 20)(mt);
            for( int j = 0; j < count; ++j )
            {
                const auto offset = nn::util::align_up(
                    std::uniform_int_distribution<int64_t>(0, storageSize)(mt),
                    16
                );
                const auto size = nn::util::align_up(
                    std::uniform_int_distribution<int64_t>(0, 4 * 1024)(mt),
                    16
                );

                container.Collect(offset, size);
            }
            container.Optimize();

            const auto& blocks = container.GetBlock();

            if( blocks.empty() )
            {
                continue;
            }

            ASSERT_GT(blocks.front().size, 0);
            ASSERT_GE(blocks.front().layerIndex, 0);
            ASSERT_LE(blocks.front().layerIndex, layerCount - 2);

            // ブロックの正当性チェック
            for( auto pos = blocks.begin() + 1; pos != blocks.end(); ++pos )
            {
                const auto& block1 = *(pos - 1);
                const auto& block2 = *pos;

                ASSERT_GT(block2.size, 0);
                ASSERT_GE(block2.layerIndex, 0);
                ASSERT_LE(block2.layerIndex, layerCount - 2);

                ASSERT_LT(block1.position, block2.position);
                ASSERT_LE(block1.layerIndex, block2.layerIndex);

                if( block1.layerIndex == block2.layerIndex )
                {
                    ASSERT_LT(block1.layerOffset, block2.layerOffset);
                    ASSERT_LT(block1.layerOffset + block1.size, block2.layerOffset);
                    ASSERT_LT(block1.position + block1.size, block2.position);
                }
                else
                {
                    ASSERT_LE(block1.position + block1.size, block2.position);
                }
            }
        }
    }
}

